/*
 * ===========================
 * VDK Visual Development Kit
 * Version 0.4
 * October 1998
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-130
 */ 

#ifndef VDKUTILS_H
#define VDKUTILS_H
#include <gdk/gdk.h>
/*
VDKRgb
*/
/*!
  \class VDKRgb
  \brief Provides a simple RGB color structure
 */
class VDKRgb
{
 public:
  int red,green,blue;
  /*!
    Constructor
    \param r red component (0-256)
    \param g green component
    \param b blue component
   */
  VDKRgb(int r = 0, int g = 0, int b = 0): red(r),green(g),blue(b) 
    {
    }
  /*!
    Destructor
   */
  ~VDKRgb() 
    {
    }
  /*!
    Constructor
    \param name names are those provided
    in rgb X database (normally on /usr/lib/X11/rgb.txt )
  */
  VDKRgb(char* name) 
    {
      red = green = blue = -1;
      GdkColor color;
      if(gdk_color_parse(name, &color))
	{
	red = color.red >> 8;
	green = color.green >> 8;
	blue = color.blue >> 8;
	}
    }
  /*!
    Returns true if a valid color was loaded
  */
  bool IsValid() const { return red != -1 && green != -1 && blue != -1 ; }
  /*!
    Equality operator
   */
  bool operator==(VDKRgb& c) const
  {
    return ( (red == c.red) && (green == c.green) && (blue == c.blue) );
  }
};

/*
   ------------
   point class
   ------------
*/
/*!
  \class VDKPoint
  \brief Provides a simple point object
 */
class  VDKPoint {
 public:
  /*!
    x,y coordinates
  */
  int x,y;
 public:
  // Constructors
  /*!
    Constructor, make a point with 0,0 coordinates
  */
  VDKPoint(): x(0),y(0)
    {
    }
  /*!
    Constructor
    \param _x x coordinate
    \param _y y coordinate
  */
  VDKPoint(int _x, int _y) 
    {  
      x = _x; y = _y; 
    }
  /*!
    Destructor
   */
  virtual
    ~VDKPoint() 
    {
    }

  /*!
    Equality operator
  */
  int
    operator ==(VDKPoint& p) const{  return (x == p.x ) && (y == p.y);  }
  /*!
    Disequality operator
  */
  int
    operator !=(VDKPoint& p) const{  return ! (*this == p) ;  }
  /*!
    Returns an offset point
    \param dx x offset
    \param dy y offset
  */
  VDKPoint
    OffsetBy(int dx, int dy) const {  return VDKPoint(x+dx, y+dy);  }
  /*!
    Return simmetric point in plane
   */
  VDKPoint
    operator -() const{  return VDKPoint(-x, -y);  }
  /*!
    Modify this with an offset
    \param dx x offset
    \param dy y offset
  */
  VDKPoint&
    Offset(int dx, int dy);

  int
    X() const {  return x;  }
  int
    Y() const {  return y;  }
};

/*
   ----------
   rect class
   ----------
*/
/*!
  \class VDKRect
  \brief Provides a simple rectangle object
 */
class VDKRect
{

 public:
  /*!
    Corners coordinates
   */
  int left,top,right,bottom;
  /*!
    width and height
  */
  int w,h;
 public:


  /*!
    Constructor makes a null rect
   */
  VDKRect() 
    {  
    left = top = right = bottom = w = h = 0;  
    }
  /*!
    Constructor
    \param x left-upper corner x
    \param left-upper corner y
    \param _w width
    \param _h height
  */ 
  VDKRect(int x, int y, int _w, int _h):w(_w),h(_h)
    {
      left = x; top = y; right = x+_w; bottom = y+_h;
    }
  /*!
    Copy initializer
  */
  VDKRect(VDKRect& r):w(r.w),h(r.h)
    {
      left = r.left; right = r.right; top = r.top; bottom = r.bottom;
    }
  /*!
    Destructor
   */
  ~VDKRect()
    {
    }
  /*!
    Returns rect origin
   */
  VDKPoint
    Origin() const {  return VDKPoint(left,top);  }
  /*!
    Returns rect width
   */
  int
    W() const {  return w;  }
  /*!
    Returns rect height
   */
  int
    H() const {  return h;  }
    /*!
    Returns 1 if this contains a point
    \param point  a point reference
   */
  int
    Contains(const VDKPoint& point) const
    {
      return point.X() >= left && point.X() < right
	&& point.Y() >= top && point.Y() < bottom;
    }
    /*!
    Returns 1 if this intersect a rect
    \param rect a rect reference
   */
  int
    Contains( const VDKRect& r) const {
    return r.left >= left && r.right <= right
      && r.top >= top && r.bottom <= bottom;
  }
};

/*!
  \class VDKNotCopyAble
  \brief Hierarchy root class

  All vdk gui and gui related objects (with few exceptions) 
  belong to this. This object can't be copied or assigned,
  any copy/assign operation is flagged as an error since
  copy-initializer and assignement operator are private methods.
  Most of them are constructed on the heap with new operator
  and a great part can be handled with garbage collection.
 */
class VDKNotCopyAble
{
 private:
     VDKNotCopyAble(VDKNotCopyAble const&);
     VDKNotCopyAble& operator=(VDKNotCopyAble const&);
 protected:
     VDKNotCopyAble(){}
     ~VDKNotCopyAble(){}
};

#endif







