#!/bin/sh
# SPDX-License-Identifier: GPL-2.0-or-later
#   Copyright (c) 1998  Martin Schulze <joey@debian.org>
#   Slightly modified by:
#   Luis Francisco Gonzalez <luisgh@debian.org>
#   Emanuele Rocca <ema@debian.org>
# shellcheck disable=SC2086,SC2059

###########################################################################
# Configurable section
###########################################################################
#
# Any entry in the lists of programs that urlview handler will try out will
# be made of program + ':' + TAG where TAG is one of
# PW: X11 program intended to live on after urlview's caller exits.
# XW: X11 program
# XT: Launch with an xterm if possible or as VT if not
# VT: Launch in the same terminal
#
# Entries are separated by tabs! Arguments may be given with spaces

# The lists of programs to be executed are
http_prgs="sensible-browser:PW	sensible-browser:XT	www-browser:PW	www-browser:XT	galeon:PW	konqueror:PW	firefox:PW	lynx:XT	w3m:XT	links:XT"
mailto_prgs="neomutt:VT	mutt:VT	elm:VT	alpine:VT	pine:VT	thunderbird:PW	mail:VT"
gopher_prgs="gopher:XT	lynx:XT"
gemini_prgs="lagrange:XW	amfora:XT"
ftp_prgs="ncftp:XT	lftp:XT	$http_prgs"
download_prgs="wget:XT	curl -SOL:XT	snarf:XT"
file_prgs="$http_prgs	$download_prgs"

XTERM=/usr/bin/x-terminal-emulator


###########################################################################
# Change below this at your own risk
###########################################################################
command -v gettext > /dev/null || alias gettext='printf %s'

getprg() {
	IFS='	'
	for ele in $1; do
		tag=${ele##*:}
		prog=${ele%:*}
		if command -v "${prog%%' '*}" > /dev/null; then
			case "$tag" in
				PW)
					[ -n "$DISPLAY" ] && echo "PW:$prog" && return 0
					;;
				XW)
					[ -n "$DISPLAY" ] && echo "XW:$prog" && return 0
					;;
				XT)
					[ -n "$DISPLAY" ] && [ -x "$XTERM" ] && echo "XT:$XTERM -e $prog" && return 0
					echo "$prog" && return 0
					;;
				VT)
					echo "$prog" && return 0
					;;
			esac
		fi
	done
}

if [ $# -ne 0 ]; then
	url=$1
elif ( : "${URL?}" ) 2>/dev/null; then
	url=$URL
else
	IFS= read -r url
fi

type=${url%%:*}

if [ "$url" = "$type" ]; then
	type=${url%%.*}
	case "$type" in
		www|web|www[1-9])
			type=http
			;;
	esac
	url=$type://$url
fi

if [ "$type" = "ftp" ]; then
	filename=${url##*/}
	if [ -n "$filename" ]; then
		printf "$(TEXTDOMAIN=urlview-ng gettext 'Is "%s" a file? ')" "$filename"
		read -r x
		printf '%s\n' "$x" | grep -q "$(locale yesexpr 2>/dev/null || echo '^[yY1]')" && type="!download"
	fi
fi

case "$type" in
	http|https)
		prg=$(getprg "$http_prgs")
		;;
	ftp)
		prg=$(getprg "$ftp_prgs")
		;;
	mailto)
		prg=$(getprg "$mailto_prgs")
		;;
	gopher)
		prg=$(getprg "$gopher_prgs")
		;;
	gemini)
		prg=$(getprg "$gemini_prgs")
		;;
	file)
		prg=$(getprg "$file_prgs")
		;;
	'!download')
		prg=$(getprg "$download_prgs")
		;;
	*)
		prg=
		# "viewer" as-in something that can open and potentially show the URL
		printf "$(TEXTDOMAIN=urlview-ng gettext 'Unknown URL protocol "%s".
Please report URL and viewer to <~nabijaczleweli/urlview-ng@lists.sr.ht>.
Press enter to continue... ')" "$type" >&2
		read -r _
		;;
esac

[ -z "$prg" ] && exit 127

case "${prg%%:*}" in
	PW)
		nohup ${prg#*:} "$url" 2>/dev/null 1>&2 &
		;;
	XT)
		${prg#*:} "$url" 2>/dev/null &
		;;
	*)
		exec $prg "$url"
		;;
esac
