/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;

using IndexReader = Lucene.Net.Index.IndexReader;
using Term = Lucene.Net.Index.Term;

namespace Lucene.Net.Search
{
	
	/// <summary>Subclass of FilteredTermEnum for enumerating all terms that are similiar
	/// to the specified filter term.
	/// 
	/// <p/>Term enumerations are always ordered by Term.compareTo().  Each term in
	/// the enumeration is greater than all that precede it.
	/// </summary>
	public sealed class FuzzyTermEnum:FilteredTermEnum
	{
		/* Allows us save time required to create a new array
		* everytime similarity is called.
		*/
	    private int[] p;
		private int[] d;
		
		private float similarity;
		private bool endEnum = false;

	    private bool isDisposed;

        private Term searchTerm = null;
		private System.String field;
		private System.String text;
		private System.String prefix;
		
		private float minimumSimilarity;
		private float scale_factor;
		
		/// <summary> Creates a FuzzyTermEnum with an empty prefix and a minSimilarity of 0.5f.
		/// <p/>
		/// After calling the constructor the enumeration is already pointing to the first 
		/// valid term if such a term exists. 
		/// 
		/// </summary>
		/// <param name="reader">
		/// </param>
		/// <param name="term">
		/// </param>
		/// <throws>  IOException </throws>
		/// <seealso cref="FuzzyTermEnum(IndexReader, Term, float, int)">
		/// </seealso>
		public FuzzyTermEnum(IndexReader reader, Term term):this(reader, term, FuzzyQuery.defaultMinSimilarity, FuzzyQuery.defaultPrefixLength)
		{
		}
		
		/// <summary> Creates a FuzzyTermEnum with an empty prefix.
		/// <p/>
		/// After calling the constructor the enumeration is already pointing to the first 
		/// valid term if such a term exists. 
		/// 
		/// </summary>
		/// <param name="reader">
		/// </param>
		/// <param name="term">
		/// </param>
		/// <param name="minSimilarity">
		/// </param>
		/// <throws>  IOException </throws>
		/// <seealso cref="FuzzyTermEnum(IndexReader, Term, float, int)">
		/// </seealso>
		public FuzzyTermEnum(IndexReader reader, Term term, float minSimilarity):this(reader, term, minSimilarity, FuzzyQuery.defaultPrefixLength)
		{
		}
		
		/// <summary> Constructor for enumeration of all terms from specified <c>reader</c> which share a prefix of
		/// length <c>prefixLength</c> with <c>term</c> and which have a fuzzy similarity &gt;
		/// <c>minSimilarity</c>.
		/// <p/>
		/// After calling the constructor the enumeration is already pointing to the first 
		/// valid term if such a term exists. 
		/// 
		/// </summary>
		/// <param name="reader">Delivers terms.
		/// </param>
		/// <param name="term">Pattern term.
		/// </param>
		/// <param name="minSimilarity">Minimum required similarity for terms from the reader. Default value is 0.5f.
		/// </param>
		/// <param name="prefixLength">Length of required common prefix. Default value is 0.
		/// </param>
		/// <throws>  IOException </throws>
		public FuzzyTermEnum(IndexReader reader, Term term, float minSimilarity, int prefixLength):base()
		{
			
			if (minSimilarity >= 1.0f)
				throw new System.ArgumentException("minimumSimilarity cannot be greater than or equal to 1");
			else if (minSimilarity < 0.0f)
				throw new System.ArgumentException("minimumSimilarity cannot be less than 0");
			if (prefixLength < 0