/** \file pnp_ids.c
 *
 *  Provides a lookup table of 3 character Plug and Play manufacturer codes,
 *  which are used, e.g. in EDIDs.
 *
 *  The list, originally created by Microsoft, is now maintained at the UEFI
 *  web site.
 *
 *  \TODO
 *  Instead of the list hard coded here, consider using file
 *  /usr/share/hwdata/pnp.ids from package hwdata, or a flat file extracted
 *  from the UEFI web site.
 */

// Copyright (C) 2022 Sanford Rockowitz <rockowitz@minsoft.com>
// SPDX-License-Identifier: GPL-2.0-or-later

/** \cond */
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
/** \endcond */

#include "coredefs_base.h"
#include "string_util.h"

#include "pnp_ids.h"


typedef struct    {
   char* mfg_code;
   char* mfg_name;
} Pnp_Id_Table_Entry;


Pnp_Id_Table_Entry pnp_id_table[] =   {
   {"AAA","Avolites Ltd"},
   {"AAE","Anatek Electronics Inc."},
   {"AAT","Ann Arbor Technologies"},
   {"ABA","ABBAHOME  INC."},
   {"ABC","AboCom System Inc"},
   {"ABD","Allen Bradley Company"},
   {"ABE","Alcatel Bell"},
   {"ABO","D-Link Systems Inc"},
   {"ABT","Anchor Bay Technologies, Inc."},
   {"ABV","Advanced Research Technology"},
   {"ACA","Ariel Corporation"},
   {"ACB","Aculab Ltd"},
   {"ACC","Accton Technology Corporation"},
   {"ACD","AWETA BV"},
   {"ACE","Actek Engineering Pty Ltd"},
   {"ACG","A&R Cambridge Ltd"},
   {"ACH","Archtek Telecom Corporation"},
   {"ACI","Ancor Communications Inc"},
   {"ACK","Acksys"},
   {"ACL","Apricot Computers"},
   {"ACM","Acroloop Motion Control Systems Inc"},
   {"ACO","Allion Computer Inc."},
   {"ACP","Aspen Tech Inc"},
   {"ACR","Acer Technologies"},
   {"ACS","Altos Computer Systems"},
   {"ACT","Applied Creative Technology"},
   {"ACU","Acculogic"},
   {"ACV","ActivCard S.A"},
   {"ADA","Addi-Data GmbH"},
   {"ADB","Aldebbaron"},
   {"ADC","Acnhor Datacomm"},
   {"ADD","Advanced Peripheral Devices Inc"},
   {"ADE","Arithmos, Inc."},
   {"ADH","Aerodata Holdings Ltd"},
   {"ADI","ADI Systems Inc"},
   {"ADK","Adtek System Science Company Ltd"},
   {"ADL","ASTRA Security Products Ltd"},
   {"ADM","Ad Lib MultiMedia Inc"},
   {"ADN","Analog & Digital Devices Tel. Inc"},
   {"ADP","Adaptec Inc"},
   {"ADR","Nasa Ames Research Center"},
   {"ADS","Analog Devices Inc"},
   {"ADT","Adtek"},
   {"ADT","Aved Display Technologies"},
   {"ADV","Advanced Micro Devices Inc"},
   {"ADX","Adax Inc"},
   {"AEC","Antex Electronics Corporation"},
   {"AED","Advanced Electronic Designs, Inc."},
   {"AEI","Actiontec Electric Inc"},
   {"AEJ","Alpha Electronics Company"},
   {"AEM","ASEM S.p.A."},
   {"AEN","Avencall"},
   {"AEP","Aetas Peripheral International"},
   {"AET","Aethra Telecomunicazioni S.r.l."},
   {"AFA","Alfa Inc"},
   {"AGC","Beijing Aerospace Golden Card Electronic Engineering Co.,Ltd."},
   {"AGI","Artish Graphics Inc"},
   {"AGL","Argolis"},
   {"AGM","Advan Int'l Corporation"},
   {"AGT","Agilent Technologies"},
   {"AHC","Advantech Co., Ltd."},
   {"AIC","Arnos Insturments & Computer Systems"},
   {"AIE","Altmann Industrieelektronik"},
   {"AII","Amptron International Inc."},
   {"AIL","Altos India Ltd"},
   {"AIM","AIMS Lab Inc"},
   {"AIR","Advanced Integ. Research Inc"},
   {"AIS","Alien Internet Services"},
   {"AIW","Aiwa Company Ltd"},
   {"AIX","ALTINEX, INC."},
   {"AJA","AJA Video Systems, Inc."},
   {"AKB","Akebia Ltd"},
   {"AKE","AKAMI Electric Co.,Ltd"},
   {"AKI","AKIA Corporation"},
   {"AKL","AMiT Ltd"},
   {"AKM","Asahi Kasei Microsystems Company Ltd"},
   {"AKP","Atom Komplex Prylad"},
   {"AKY","Askey Computer Corporation"},
   {"ALA","Alacron Inc"},
   {"ALC","Altec Corporation"},
   {"ALD","In4S Inc"},
   {"ALG","Realtek Semiconductor Corp."},
   {"ALH","AL Systems"},
   {"ALI","Acer Labs"},
   {"ALJ","Altec Lansing"},
   {"ALK","Acrolink Inc"},
   {"ALL","Alliance Semiconductor Corporation"},
   {"ALM","Acutec Ltd."},
   {"ALN","Alana Technologies"},
   {"ALO","Algolith Inc."},
   {"ALP","Alps Electric Company Ltd"},
   {"ALR","Advanced Logic"},
   {"ALS","Avance Logic Inc"},
   {"ALS","Texas Advanced optoelectronics Solutions, Inc"},
   {"ALT","Altra"},
   {"ALV","AlphaView LCD"},
   {"ALX","ALEXON Co.,Ltd."},
   {"AMA","Asia Microelectronic Development Inc"},
   {"AMB","Ambient Technologies, Inc."},
   {"AMC","Attachmate Corporation"},
   {"AMD","Amdek Corporation"},
   {"AMI","American Megatrends Inc"},
   {"AML","Anderson Multimedia Communications   {HK} Limited"},
   {"AMN","Amimon LTD."},
   {"AMO","Amino Technologies PLC and Amino Communications Limited"},
   {"AMP","AMP Inc"},
   {"AMS ","ARMSTEL, Inc."},
   {"AMT","AMT International Industry"},
   {"AMX","AMX LLC"},
   {"ANA","Anakron"},
   {"ANC","Ancot"},
   {"AND","Adtran Inc"},
   {"ANI","Anigma Inc"},
   {"ANK","Anko Electronic Company Ltd"},
   {"ANL","Analogix Semiconductor, Inc"},
   {"ANO","Anorad Corporation"},
   {"ANP","Andrew Network Production"},
   {"ANR","ANR Ltd"},
   {"ANS","Ansel Communication Company"},
   {"ANT","Ace CAD Enterprise Company Ltd"},
   {"ANX","Acer Netxus Inc"},
   {"AOA","AOpen Inc."},
   {"AOE","Advanced Optics Electronics, Inc."},
   {"AOL","America OnLine"},
   {"AOT","Alcatel"},
   {"APC","American Power Conversion"},
   {"APD","AppliAdata"},
   {"APE","Alpine Electronics, Inc."},
   {"APG","Horner Electric Inc"},
   {"API","A Plus Info Corporation"},
   {"APL","Aplicom Oy"},
   {"APM","Applied Memory Tech"},
   {"APN","Appian Tech Inc"},
   {"APP","Apple Computer Inc"},
   {"APR","Aprilia s.p.a."},
   {"APS","Autologic Inc"},
   {"APT","Audio Processing Technology Ltd"},
   {"APV","A+V Link "},
   {"APX","AP Designs Ltd"},
   {"ARC","Alta Research Corporation"},
   {"ARE","ICET S.p.A."},
   {"ARG","Argus Electronics Co., LTD"},
   {"ARI","Argosy Research Inc"},
   {"ARK","Ark Logic Inc"},
   {"ARL","Arlotto Comnet Inc"},
   {"ARM","Arima"},
   {"ARO","Poso International B.V."},
   {"ARS","Arescom Inc"},
   {"ART","Corion Industrial Corporation"},
   {"ASC","Ascom Strategic Technology Unit"},
   {"ASD","USC Information Sciences Institute"},
   {"ASE","AseV Display Labs"},
   {"ASI","Ahead Systems"},
   {"ASK","Ask A/S"},
   {"ASL","AccuScene Corporation Ltd"},
   {"ASM","ASEM S.p.A."},
   {"ASN","Asante Tech Inc"},
   {"ASP","ASP Microelectronics Ltd"},
   {"AST","AST Research Inc"},
   {"ASU","Asuscom Network Inc"},
   {"ASX","AudioScience"},
   {"ASY","Rockwell Collins / Airshow Systems"},
   {"ATA","Allied Telesyn International   {Asia} Pte Ltd"},
   {"ATC","Ably-Tech Corporation"},
   {"ATD","Alpha Telecom Inc"},
   {"ATE","Innovate Ltd"},
   {"ATH","Athena Informatica S.R.L."},
   {"ATI","Allied Telesis KK"},
   {"ATK","Allied Telesyn Int'l"},
   {"ATL","Arcus Technology Ltd"},
   {"ATM","ATM Ltd"},
   {"ATN","Athena Smartcard Solutions Ltd."},
   {"ATO","ASTRO DESIGN, INC."},
   {"ATP","Alpha-Top Corporation"},
   {"ATT","AT&T"},
   {"ATV","Office Depot, Inc."},
   {"ATX","Athenix Corporation"},
   {"AUI","Alps Electric Inc"},
   {"AUR","Aureal Semiconductor"},
   {"AUT","Autotime Corporation"},
   {"AVA","Avaya Communication"},
   {"AVC","Auravision Corporation"},
   {"AVD","Avid Electronics Corporation"},
   {"AVE","Add Value Enterpises   {Asia} Pte Ltd"},
   {"AVI","Nippon Avionics Co.,Ltd"},
   {"AVL","Avalue Technology Inc."},
   {"AVM","AVM GmbH"},
   {"AVN ","Advance Computer Corporation"},
   {"AVO","Avocent Corporation"},
   {"AVR","AVer Information Inc."},
   {"AVT","Avtek   {Electronics} Pty Ltd"},
   {"AVV","SBS Technologies   {Canada}, Inc.   {was Avvida Systems, Inc.}"},
   {"AVX","A/Vaux Electronics"},
   {"AVX","AVerMedia Technologies, Inc."},
   {"AWC","Access Works Comm Inc"},
   {"AWL","Aironet Wireless Communications, Inc"},
   {"AWS","Wave Systems"},
   {"AXB","Adrienne Electronics Corporation"},
   {"AXC","AXIOMTEK CO., LTD."},
   {"AXE","D-Link Systems Inc"},
   {"AXI","American Magnetics"},
   {"AXL","Axel"},
   {"AXO","Axonic Labs LLC"},
   {"AXP","American Express"},
   {"AXT","Axtend Technologies Inc"},
   {"AXX","Axxon Computer Corporation"},
   {"AXY","AXYZ Automation Services, Inc"},
   {"AYD","Aydin Displays"},
   {"AYR","Airlib, Inc"},
   {"AZM","AZ Middelheim - Radiotherapy"},
   {"AZT","Aztech Systems Ltd"},
   {"BAC","Biometric Access Corporation"},
   {"BAN","Banyan"},
   {"BBB","an-najah university"},
   {"BBH","B&Bh"},
   {"BBL","Brain Boxes Limited"},
   {"BCC","Beaver Computer Corporaton"},
   {"BCD","Barco GmbH"},
   {"BCM","Broadcom"},
   {"BCQ","Deutsche Telekom Berkom GmbH"},
   {"BCS","Booria CAD/CAM systems"},
   {"BDO","Brahler ICS"},
   {"BDR","Blonder Tongue Labs, Inc."},
   {"BDS","Barco Display Systems"},
   {"BEC","Elektro Beckhoff GmbH"},
   {"BEI","Beckworth Enterprises Inc"},
   {"BEK","Beko Elektronik A.S."},
   {"BEL","Beltronic Industrieelektronik GmbH"},
   {"BEO","Baug & Olufsen"},
   {"BFE","B.F. Engineering Corporation"},
   {"BGB","Barco Graphics N.V"},
   {"BGT","Budzetron Inc"},
   {"BHZ","BitHeadz, Inc."},
   {"BIC","Big Island Communications"},
   {"BII","Boeckeler Instruments Inc"},
   {"BIL","Billion Electric Company Ltd"},
   {"BIO","BioLink Technologies International, Inc."},
   {"BIT","Bit 3 Computer"},
   {"BLI","Busicom"},
   {"BLN","BioLink Technologies"},
   {"BLP","Bloomberg L.P."},
   {"BMD","Blackmagic Design"},
   {"BMI","Benson Medical Instruments Company"},
   {"BML","BIOMED Lab"},
   {"BMS","BIOMEDISYS"},
   {"BNE","Bull AB"},
   {"BNK","Banksia Tech Pty Ltd"},
   {"BNO","Bang & Olufsen"},
   {"BNS","Boulder Nonlinear Systems"},
   {"BOB","Rainy Orchard"},
   {"BOE","BOE"},
   {"BOI","NINGBO BOIGLE DIGITAL TECHNOLOGY CO.,LTD"},
   {"BOS","BOS"},
   {"BPD","Micro Solutions, Inc."},
   {"BPU","Best Power"},
   {"BRA","Braemac Pty Ltd"},
   {"BRC","BARC"},
   {"BRG","Bridge Information Co., Ltd"},
   {"BRI","Boca Research Inc"},
   {"BRM","Braemar Inc"},
   {"BRO","BROTHER INDUSTRIES,LTD."},
   {"BSE","Bose Corporation"},
   {"BSL","Biomedical Systems Laboratory"},
   {"BSN","BRIGHTSIGN, LLC"},
   {"BST","BodySound Technologies, Inc."},
   {"BTC","Bit 3 Computer"},
   {"BTE","Brilliant Technology"},
   {"BTF","Bitfield Oy"},
   {"BTI","BusTech Inc"},
   {"BTO","BioTao Ltd"},
   {"BUF","Yasuhiko Shirai Melco Inc"},
   {"BUG","B.U.G., Inc."},
   {"BUJ","ATI Tech Inc"},
   {"BUL","Bull"},
   {"BUR","Bernecker & Rainer Ind-Eletronik GmbH"},
   {"BUS","BusTek"},
   {"BUT","21ST CENTURY ENTERTAINMENT"},
   {"BWK","Bitworks Inc."},
   {"BXE","Buxco Electronics"},
   {"BYD","byd:sign corporation"},
   {"CAA","Castles Automation Co., Ltd"},
   {"CAC","CA & F Elettronica"},
   {"CAG","CalComp"},
   {"CAI","Canon Inc."},
   {"CAL","Acon"},
   {"CAM","Cambridge Audio"},
   {"CAN","Canopus Company Ltd"},
   {"CAN","Carrera Computer Inc"},
   {"CAN","CORNEA"},
   {"CAR","Cardinal Company Ltd"},
   {"CAS","CASIO COMPUTER CO.,LTD"},
   {"CAT","Consultancy in Advanced Technology"},
   {"CAV","Cavium Networks, Inc"},
   {"CBI","ComputerBoards Inc"},
   {"CBR","Cebra Tech A/S"},
   {"CBT","Cabletime Ltd"},
   {"CBX","Cybex Computer Products Corporation"},
   {"CCC","C-Cube Microsystems"},
   {"CCI","Cache"},
   {"CCJ","CONTEC CO.,LTD."},
   {"CCL","CCL/ITRI"},
   {"CCP","Capetronic USA Inc"},
   {"CDC","Core Dynamics Corporation"},
   {"CDD","Convergent Data Devices"},
   {"CDE","Colin.de"},
   {"CDG","Christie Digital Systems Inc"},
   {"CDI","Concept Development Inc"},
   {"CDK","Cray Communications"},
   {"CDN","Codenoll Technical Corporation"},
   {"CDP","CalComp"},
   {"CDS","Computer Diagnostic Systems"},
   {"CDT","IBM Corporation"},
   {"CDV","Convergent Design Inc."},
   {"CEA","Consumer Electronics Association"},
   {"CEC","Chicony Electronics Company Ltd"},
   {"CED","Cambridge Electronic Design Ltd"},
   {"CEF","Cefar Digital Vision"},
   {"CEI","Crestron Electronics, Inc."},
   {"CEM","MEC Electronics GmbH"},
   {"CEN","Centurion Technologies P/L"},
   {"CEP","C-DAC"},
   {"CER","Ceronix"},
   {"CET","TEC CORPORATION"},
   {"CFG","Atlantis"},
   {"CGA","Chunghwa Picture Tubes, LTD"},
   {"CGS","Chyron Corp"},
   {"CGT","congatec AG"},
   {"CHA","Chase Research PLC"},
   {"CHC","Chic Technology Corp."},
   {"CHD","ChangHong Electric Co.,Ltd"},
   {"CHE","Acer Inc"},
   {"CHG","Sichuan Changhong Electric CO, LTD."},
   {"CHI","Chrontel Inc"},
   {"CHL","Chloride-R&D"},
   {"CHM","CHIC TECHNOLOGY CORP."},
   {"CHO","Sichuang Changhong Corporation"},
   {"CHP","CH Products"},
   {"CHS","Agentur Chairos"},
   {"CHT","Chunghwa Picture Tubes,LTD."},
   {"CHY","Cherry GmbH"},
   {"CIC","Comm. Intelligence Corporation"},
   {"CII","Cromack Industries Inc"},
   {"CIL","Citicom Infotech Private Limited"},
   {"CIN","Citron GmbH"},
   {"CIP","Ciprico Inc"},
   {"CIR","Cirrus Logic Inc"},
   {"CIS","Cisco Systems Inc"},
   {"CIT","Citifax Limited"},
   {"CKC","The Concept Keyboard Company Ltd"},
   {"CKJ","Carina System Co., Ltd."},
   {"CLA","Clarion Company Ltd"},
   {"CLD","COMMAT L.t.d."},
   {"CLE","Classe Audio"},
   {"CLG","CoreLogic"},
   {"CLI","Cirrus Logic Inc"},
   {"CLM","CrystaLake Multimedia"},
   {"CLO","Clone Computers"},
   {"CLT","automated computer control systems"},
   {"CLV","Clevo Company"},
   {"CLX","CardLogix"},
   {"CMC","CMC Ltd"},
   {"CMD","Colorado MicroDisplay, Inc."},
   {"CMG","Chenming Mold Ind. Corp."},
   {"CMI","C-Media Electronics"},
   {"CMM","Comtime GmbH"},
   {"CMN","Chimei Innolux Corporation"},
   {"CMO","Chi Mei Optoelectronics corp."},
   {"CMR","Cambridge Research Systems Ltd"},
   {"CMS","CompuMaster Srl"},
   {"CMX","Comex Electronics AB"},
   {"CNB","American Power Conversion"},
   {"CNC","Alvedon Computers Ltd"},
   {"CNE","Cine-tal"},
   {"CNI","Connect Int'l A/S"},
   {"CNN","Canon Inc"},
   {"CNT","COINT Multimedia Systems"},
   {"COB","COBY Electronics Co., Ltd"},
   {"COD","CODAN Pty. Ltd."},
   {"COI","Codec Inc."},
   {"COL","Rockwell Collins, Inc."},
   {"COM","Comtrol Corporation"},
   {"CON","Contec Company Ltd"},
   {"COO","coolux GmbH "},
   {"COR","Corollary Inc"},
   {"COS","CoStar Corporation"},
   {"COT","Core Technology Inc"},
   {"COW","Polycow Productions"},
   {"COX","Comrex"},
   {"CPC","Ciprico Inc"},
   {"CPD","CompuAdd"},
   {"CPI","Computer Peripherals Inc"},
   {"CPL","Compal Electronics Inc"},
   {"CPM","Capella Microsystems Inc."},
   {"CPQ","Compaq Computer Company"},
   {"CPT","cPATH"},
   {"CPX","Powermatic Data Systems"},
   {"CRC","CONRAC GmbH"},
   {"CRD","Cardinal Technical Inc"},
   {"CRE","Creative Labs Inc"},
   {"CRI","Crio Inc."},
   {"CRL","Creative Logic  "},
   {"CRN","Cornerstone Imaging"},
   {"CRO","Extraordinary Technologies PTY Limited"},
   {"CRQ","Cirque Corporation"},
   {"CRS","Crescendo Communication Inc"},
   {"CRV","Cerevo Inc."},
   {"CRX","Cyrix Corporation"},
   {"CSB","Transtex SA"},
   {"CSC","Crystal Semiconductor"},
   {"CSD","Cresta Systems Inc"},
   {"CSE","Concept Solutions & Engineering"},
   {"CSI","Cabletron System Inc"},
   {"CSM","Cosmic Engineering Inc."},
   {"CSO","California Institute of Technology"},
   {"CSS","CSS Laboratories"},
   {"CST","CSTI Inc"},
   {"CTA","CoSystems Inc"},
   {"CTC","CTC Communication Development Company Ltd"},
   {"CTE","Chunghwa Telecom Co., Ltd."},
   {"CTL","Creative Technology Ltd"},
   {"CTM","Computerm Corporation"},
   {"CTN","Computone Products"},
   {"CTP","Computer Technology Corporation"},
   {"CTS","Comtec Systems Co., Ltd."},
   {"CTX","Creatix Polymedia GmbH"},
   {"CUB","Cubix Corporation"},
   {"CUK","Calibre UK Ltd"},
   {"CVA","Covia Inc."},
   {"CVI","Colorado Video, Inc."},
   {"CVS","Clarity Visual Systems"},
   {"CWR","Connectware Inc"},
   {"CXT","Conexant Systems"},
   {"CYB","CyberVision"},
   {"CYC","Cylink Corporation"},
   {"CYD","Cyclades Corporation"},
   {"CYL","Cyberlabs"},
   {"CYT","Cytechinfo Inc"},
   {"CYV","Cyviz AS"},
   {"CYW","Cyberware"},
   {"CYX","Cyrix Corporation"},
   {"CZE","Carl Zeiss AG"},
   {"DAC","Digital Acoustics Corporation"},
   {"DAE","Digatron Industrie Elektronik GmbH"},
   {"DAI","DAIS SET Ltd."},
   {"DAK","Daktronics"},
   {"DAL","Digital Audio Labs Inc"},
   {"DAN","Danelec Marine A/S"},
   {"DAS","DAVIS AS"},
   {"DAT","Datel Inc"},
   {"DAU","Daou Tech Inc"},
   {"DAV","Davicom Semiconductor Inc"},
   {"DAW","DA2 Technologies Inc"},
   {"DAX","Data Apex Ltd"},
   {"DBD","Diebold Inc."},
   {"DBI","DigiBoard Inc"},
   {"DBK","Databook Inc"},
   {"DBL","Doble Engineering Company"},
   {"DBN","DB Networks Inc"},
   {"DCA","Digital Communications Association"},
   {"DCC","Dale Computer Corporation"},
   {"DCD","Datacast LLC"},
   {"DCE","dSPACE GmbH"},
   {"DCI","Concepts Inc"},
   {"DCL","Dynamic Controls Ltd"},
   {"DCM","DCM Data Products"},
   {"DCO","Dialogue Technology Corporation"},
   {"DCR","Decros Ltd"},
   {"DCS","Diamond Computer Systems Inc"},
   {"DCT","Dancall Telecom A/S"},
   {"DCV","Datatronics Technology Inc"},
   {"DDA","DA2 Technologies Corporation"},
   {"DDD","Danka Data Devices"},
   {"DDE","Datasat Digital Entertainment"},
   {"DDI","Data Display AG"},
   {"DDS","Barco, n.v."},
   {"DDT","Datadesk Technologies Inc"},
   {"DDV","Delta Information Systems, Inc"},
   {"DEC","Digital Equipment Corporation"},
   {"DEI","Deico Electronics"},
   {"DEL","Dell Inc."},
   {"DEN","Densitron Computers Ltd"},
   {"DEX","idex displays"},
   {"DFI","DFI"},
   {"DFK","SharkTec A/S"},
   {"DFT","DEI Holdings dba Definitive Technology"},
   {"DGA","Digiital Arts Inc"},
   {"DGC","Data General Corporation"},
   {"DGI","DIGI International"},
   {"DGK","DugoTech Co., LTD"},
   {"DGP","Digicorp European sales S.A."},
   {"DGS","Diagsoft Inc"},
   {"DGT","Dearborn Group Technology"},
   {"DGT","The Dearborn Group"},
   {"DHP","DH Print"},
   {"DHQ","Quadram"},
   {"DHT","Projectavision Inc"},
   {"DIA","Diadem"},
   {"DIG","Digicom S.p.A."},
   {"DII","Dataq Instruments Inc"},
   {"DIM","dPict Imaging, Inc."},
   {"DIN","Daintelecom Co., Ltd"},
   {"DIS","Diseda S.A."},
   {"DIT","Dragon Information Technology"},
   {"DJE","Capstone Visua lProduct Development "},
   {"DJP","Maygay Machines, Ltd"},
   {"DKY","Datakey Inc"},
   {"DLB","Dolby Laboratories Inc. "},
   {"DLC","Diamond Lane Comm. Corporation"},
   {"DLG","Digital-Logic GmbH"},
   {"DLK","D-Link Systems Inc"},
   {"DLL","Dell Inc"},
   {"DLT","Digitelec Informatique Park Cadera"},
   {"DMB","Digicom Systems Inc"},
   {"DMC","Dune Microsystems Corporation"},
   {"DMM","Dimond Multimedia Systems Inc"},
   {"DMP","D&M Holdings Inc, Professional Business Company"},
   {"DMS","DOME imaging systems"},
   {"DMT","Distributed Management Task Force, Inc.   {DMTF} "},
   {"DMV","NDS Ltd"},
   {"DNA","DNA Enterprises, Inc."},
   {"DNG","Apache Micro Peripherals Inc"},
   {"DNI","Deterministic Networks Inc."},
   {"DNT","Dr. Neuhous Telekommunikation GmbH"},
   {"DNV","DiCon"},
   {"DOL","Dolman Technologies Group Inc"},
   {"DOM","Dome Imaging Systems"},
   {"DON","DENON, Ltd."},
   {"DOT","Dotronic Mikroelektronik GmbH"},
   {"DPA","DigiTalk Pro AV"},
   {"DPC","Delta Electronics Inc"},
   {"DPI","DocuPoint"},
   {"DPL","Digital Projection Limited"},
   {"DPM","ADPM Synthesis sas"},
   {"DPS","Digital Processing Systems"},
   {"DPT","DPT"},
   {"DPX","DpiX, Inc."},
   {"DQB","Datacube Inc"},
   {"DRB","Dr. Bott KG"},
   {"DRC","Data Ray Corp."},
   {"DRD","DIGITAL REFLECTION INC."},
   {"DRI","Data Race Inc"},
   {"DRS","DRS Defense Solutions, LLC"},
   {"DSD","DS Multimedia Pte Ltd"},
   {"DSI","Digitan Systems Inc"},
   {"DSM","DSM Digital Services GmbH"},
   {"DSP","Domain Technology Inc"},
   {"DTA","DELTATEC "},
   {"DTC","DTC Tech Corporation"},
   {"DTE","Dimension Technologies, Inc."},
   {"DTI","Diversified Technology, Inc."},
   {"DTK","Dynax Electronics   {HK} Ltd"},
   {"DTL","e-Net Inc"},
   {"DTN","Datang Telephone Co"},
   {"DTO","Deutsche Thomson OHG"},
   {"DTT","Design & Test Technology, Inc."},
   {"DTX","Data Translation"},
   {"DUA","Dosch & Amand GmbH & Company KG"},
   {"DUN","NCR Corporation"},
   {"DVD","Dictaphone Corporation"},
   {"DVL","Devolo AG"},
   {"DVS","Digital Video System"},
   {"DVT","Data Video"},
   {"DWE","Daewoo Electronics Company Ltd"},
   {"DXC","Digipronix Control Systems"},
   {"DXD","DECIMATOR DESIGN PTY LTD"},
   {"DXL","Dextera Labs Inc"},
   {"DXP","Data Expert Corporation"},
   {"DXS","Signet"},
   {"DYC","Dycam Inc"},
   {"DYM","Dymo-CoStar Corporation"},
   {"DYN","Askey Computer Corporation"},
   {"DYX","Dynax Electronics   {HK} Ltd"},
   {"EAS","Evans and Sutherland Computer"},
   {"EBH","Data Price Informatica"},
   {"EBT","HUALONG TECHNOLOGY CO., LTD"},
   {"ECA","Electro Cam Corp."},
   {"ECC","ESSential Comm. Corporation"},
   {"ECI","Enciris Technologies"},
   {"ECK","Eugene Chukhlomin Sole Proprietorship, d.b.a. "},
   {"ECL","Excel Company Ltd"},
   {"ECM","E-Cmos Tech Corporation"},
   {"ECO","Echo Speech Corporation"},
   {"ECP","Elecom Company Ltd"},
   {"ECS","Elitegroup Computer Systems Company Ltd"},
   {"ECT","Enciris Technologies "},
   {"EDC","e.Digital Corporation"},
   {"EDG","Electronic-Design GmbH"},
   {"EDI","Edimax Tech. Company Ltd"},
   {"EDM","EDMI"},
   {"EDT","Emerging Display Technologies Corp"},
   {"EEE","ET&T Technology Company Ltd"},
   {"EEH","EEH Datalink GmbH"},
   {"EEP","E.E.P.D. GmbH"},
   {"EES","EE Solutions, Inc."},
   {"EGA","Elgato Systems LLC"},
   {"EGD","EIZO GmbH Display Technologies "},
   {"EGL","Eagle Technology"},
   {"EGN","Egenera, Inc."},
   {"EGO","Ergo Electronics"},
   {"EHJ","Epson Research"},
   {"EHN","Enhansoft"},
   {"EIC","Eicon Technology Corporation"},
   {"EKA","MagTek Inc."},
   {"EKC","Eastman Kodak Company"},
   {"EKS","EKSEN YAZILIM"},
   {"ELA","ELAD srl"},
   {"ELC","Electro Scientific Ind"},
   {"ELE","Elecom Company Ltd"},
   {"ELG","Elmeg GmbH Kommunikationstechnik"},
   {"ELI","Edsun Laboratories"},
   {"ELL","Electrosonic Ltd"},
   {"ELM","Elmic Systems Inc"},
   {"ELO","Elo TouchSystems Inc"},
   {"ELO","Tyco Electronics"},
   {"ELS","ELSA GmbH"},
   {"ELT","Element Labs, Inc."},
   {"ELX","Elonex PLC"},
   {"EMB","Embedded computing inc ltd"},
   {"EMC","eMicro Corporation"},
   {"EME","EMiNE TECHNOLOGY COMPANY, LTD."},
   {"EMG","EMG Consultants Inc"},
   {"EMI","Ex Machina Inc"},
   {"EMK","Emcore Corporation"},
   {"EMO","ELMO COMPANY, LIMITED"},
   {"EMU","Emulex Corporation"},
   {"ENC","Eizo Nanao Corporation"},
   {"END","ENIDAN Technologies Ltd"},
   {"ENE","ENE Technology Inc."},
   {"ENI","Efficient Networks"},
   {"ENS","Ensoniq Corporation"},
   {"ENT","Enterprise Comm. & Computing Inc"},
   {"EPC","Empac"},
   {"EPH ","Epiphan Systems Inc. "},
   {"EPI","Envision Peripherals, Inc"},
   {"EPN","EPiCON Inc."},
   {"EPS","KEPS"},
   {"EQP","Equipe Electronics Ltd."},
   {"EQX","Equinox Systems Inc"},
   {"ERG","Ergo System"},
   {"ERI","Ericsson Mobile Communications AB"},
   {"ERN","Ericsson, Inc."},
   {"ERP","Euraplan GmbH"},
   {"ERT","Escort Insturments Corporation"},
   {"ESA","Elbit Systems of America"},
   {"ESC","Eden Sistemas de Computacao S/A"},
   {"ESD","Ensemble Designs, Inc"},
   {"ESG","ELCON Systemtechnik GmbH"},
   {"ESI","Extended Systems, Inc."},
   {"ESK","ES&S"},
   {"ESL","Esterline Technologies"},
   {"ESN","eSATURNUS"},
   {"ESS","ESS Technology Inc"},
   {"EST","Embedded Solution Technology"},
   {"ESY","E-Systems Inc"},
   {"ETC","Everton Technology Company Ltd"},
   {"ETD","ELAN MICROELECTRONICS CORPORATION"},
   {"ETH","Etherboot Project"},
   {"ETI","Eclipse Tech Inc"},
   {"ETK","eTEK Labs Inc."},
   {"ETL","Evertz Microsystems Ltd."},
   {"ETS","Electronic Trade Solutions Ltd"},
   {"ETT","E-Tech Inc"},
   {"EUT","Ericsson Mobile Networks B.V."},
   {"EVE ","Advanced Micro Peripherals Ltd"},
   {"EVI","eviateg GmbH"},
   {"EVX","Everex"},
   {"EXA","Exabyte"},
   {"EXC","Excession Audio"},
   {"EXI","Exide Electronics"},
   {"EXN","RGB Systems, Inc. dba Extron Electronics"},
   {"EXP","Data Export Corporation"},
   {"EXT","Exatech Computadores & Servicos Ltda"},
   {"EXX","Exxact GmbH"},
   {"EXY","Exterity Ltd"},
   {"EYE","eyevis GmbH"},
   {"EZE","EzE Technologies"},
   {"EZP","Storm Technology"},
   {"FAR","Farallon Computing"},
   {"FBI","Interface Corporation"},
   {"FCB","Furukawa Electric Company Ltd"},
   {"FCG","First International Computer Ltd"},
   {"FCS","Focus Enhancements, Inc."},
   {"FDC","Future Domain"},
   {"FDT","Fujitsu Display Technologies Corp."},
   {"FEC","FURUNO ELECTRIC CO., LTD."},
   {"FEL","Fellowes & Questec"},
   {"FEN","Fen Systems Ltd."},
   {"FER","Ferranti Int'L"},
   {"FFC","FUJIFILM Corporation"},
   {"FFI","Fairfield Industries"},
   {"FGD","Lisa Draexlmaier GmbH"},
   {"FGL","Fujitsu General Limited."},
   {"FHL","FHLP"},
   {"FIC","Formosa Industrial Computing Inc"},
   {"FIL","Forefront Int'l Ltd"},
   {"FIN","Finecom Co., Ltd."},
   {"FIR","Chaplet Systems Inc"},
   {"FIS","FLY-IT Simulators"},
   {"FIT","Feature Integration Technology Inc."},
   {"FJC","Fujitsu Takamisawa Component Limited"},
   {"FJS","Fujitsu Spain"},
   {"FJT","F.J. Tieman BV"},
   {"FLE","ADTI Media, Inc"},
   {"FLI","Faroudja Laboratories"},
   {"FLY","Butterfly Communications"},
   {"FMA","Fast Multimedia AG"},
   {"FMC","Ford Microelectronics Inc"},
   {"FMI","Fellowes, Inc."},
   {"FMI","Fujitsu Microelect Inc"},
   {"FML","Fujitsu Microelect Ltd"},
   {"FMZ","Formoza-Altair"},
   {"FNC","Fanuc LTD"},
   {"FNI","Funai Electric Co., Ltd."},
   {"FOA","FOR-A Company Limited"},
   {"FOS","Foss Tecator"},
   {"FOX","HON HAI PRECISON IND.CO.,LTD."},
   {"FPE","Fujitsu Peripherals Ltd"},
   {"FPS","Deltec Corporation"},
   {"FPX","Cirel Systemes"},
   {"FRC","Force Computers"},
   {"FRD","Freedom Scientific BLV"},
   {"FRE","Forvus Research Inc"},
   {"FRI","Fibernet Research Inc"},
   {"FRO","FARO Technologies"},
   {"FRS","South Mountain Technologies, LTD"},
   {"FSC","Future Systems Consulting KK"},
   {"FSI","Fore Systems Inc"},
   {"FST","Modesto PC Inc"},
   {"FTC","Futuretouch Corporation"},
   {"FTE","Frontline Test Equipment Inc."},
   {"FTG","FTG Data Systems"},
   {"FTI","FastPoint Technologies, Inc."},
   {"FTL","FUJITSU TEN LIMITED"},
   {"FTN","Fountain Technologies Inc"},
   {"FTR","Mediasonic"},
   {"FTW","MindTribe Product Engineering, Inc."},
   {"FUJ","Fujitsu Ltd"},
   {"FUN","sisel muhendislik"},
   {"FUS","Fujitsu Siemens Computers GmbH"},
   {"FVC","First Virtual Corporation"},
   {"FVX","C-C-C Group Plc"},
   {"FWA","Attero Tech, LLC"},
   {"FWR","Flat Connections Inc"},
   {"FXX","Fuji Xerox"},
   {"FZC","Founder Group Shenzhen Co."},
   {"FZI","FZI Forschungszentrum Informatik"},
   {"GAG","Gage Applied Sciences Inc"},
   {"GAL","Galil Motion Control"},
   {"GAU","Gaudi Co., Ltd."},
   {"GCC","GCC Technologies Inc"},
   {"GCI","Gateway Comm. Inc"},
   {"GCS","Grey Cell Systems Ltd"},
   {"GDC","General Datacom"},
   {"GDI","G. Diehl ISDN GmbH"},
   {"GDS","GDS"},
   {"GDT","Vortex Computersysteme GmbH"},
   {"GED","General Dynamics C4 Systems"},
   {"GEF","GE Fanuc Embedded Systems"},
   {"GEH","GE Intelligent Platforms - Huntsville "},
   {"GEM","Gem Plus"},
   {"GEN","Genesys ATE Inc"},
   {"GEO","GEO Sense"},
   {"GER","GERMANEERS GmbH"},
   {"GES","GES Singapore Pte Ltd"},
   {"GET","Getac Technology Corporation"},
   {"GFM","GFMesstechnik GmbH"},
   {"GFN","Gefen Inc."},
   {"GGL","Google Inc."},
   {"GIC","General Inst. Corporation"},
   {"GIM","Guillemont International"},
   {"GIP","GI Provision Ltd"},
   {"GIS","AT&T Global Info Solutions"},
   {"GJN","Grand Junction Networks"},
   {"GLD","Goldmund - Digital Audio SA"},
   {"GLE","AD electronics"},
   {"GLM","Genesys Logic"},
   {"GLS","Gadget Labs LLC"},
   {"GMK","GMK Electronic Design GmbH"},
   {"GML","General Information Systems"},
   {"GMM","GMM Research Inc"},
   {"GMN","GEMINI 2000 Ltd"},
   {"GMX","GMX Inc"},
   {"GND","Gennum Corporation"},
   {"GNN","GN Nettest Inc"},
   {"GNZ","Gunze Ltd"},
   {"GRA","Graphica Computer"},
   {"GRE","GOLD RAIN ENTERPRISES CORP."},
   {"GRH","Granch Ltd"},
   {"GRM","Garmin International"},
   {"GRV","Advanced Gravis"},
   {"GRY","Robert Gray Company"},
   {"GSB","NIPPONDENCHI CO,.LTD"},
   {"GSC","General Standards Corporation"},
   {"GSM","Goldstar Company Ltd (LG)"},
   {"GST","Graphic SystemTechnology"},
   {"GSY","Grossenbacher Systeme AG"},
   {"GTC","Graphtec Corporation"},
   {"GTI","Goldtouch"},
   {"GTK","G-Tech Corporation"},
   {"GTM","Garnet System Company Ltd"},
   {"GTS","Geotest Marvin Test Systems Inc"},
   {"GTT","General Touch Technology Co., Ltd."},
   {"GUD","Guntermann & Drunck GmbH"},
   {"GUZ","Guzik Technical Enterprises"},
   {"GVC","GVC Corporation"},
   {"GVL","Global Village Communication"},
   {"GWI","GW Instruments"},
   {"GWY","Gateway 2000"},
   {"GZE","GUNZE Limited"},
   {"HAE","Haider electronics"},
   {"HAI","Haivision Systems Inc."},
   {"HAL","Halberthal"},
   {"HAN","Hanchang System Corporation"},
   {"HAR","Harris Corporation"},
   {"HAY","Hayes Microcomputer Products Inc"},
   {"HCA","DAT"},
   {"HCE","Hitachi Consumer Electronics Co., Ltd"},
   {"HCL","HCL America Inc"},
   {"HCM","HCL Peripherals"},
   {"HCP","Hitachi Computer Products Inc"},
   {"HCW","Hauppauge Computer Works Inc"},
   {"HDC","HardCom Elektronik & Datateknik"},
   {"HDI","HD-INFO d.o.o."},
   {"HDV","Holografika kft."},
   {"HEC","Hisense Electric Co., Ltd."},
   {"HEC","Hitachi Engineering Company Ltd"},
   {"HEL","Hitachi Micro Systems Europe Ltd"},
   {"HER","Ascom Business Systems"},
   {"HET","HETEC Datensysteme GmbH"},
   {"HHC","HIRAKAWA HEWTECH CORP. "},
   {"HHI","Fraunhofer Heinrich-Hertz-Institute"},
   {"HIB","Hibino Corporation"},
   {"HIC","Hitachi Information Technology Co., Ltd."},
   {"HIK","Hikom Co., Ltd."},
   {"HIL","Hilevel Technology"},
   {"HIQ","Kaohsiung Opto Electronics Americas, Inc."},
   {"HIT","Hitachi America Ltd"},
   {"HJI","Harris & Jeffries Inc"},
   {"HKA","HONKO MFG. CO., LTD."},
   {"HKG","Josef Heim KG"},
   {"HMC","Hualon Microelectric Corporation"},
   {"HMK","hmk Daten-System-Technik BmbH"},
   {"HMX","HUMAX Co., Ltd."},
   {"HNS","Hughes Network Systems"},
   {"HOB","HOB Electronic GmbH"},
   {"HOE","Hosiden Corporation"},
   {"HOL","Holoeye Photonics AG"},
   {"HON","Sonitronix"},
   {"HPA","Zytor Communications"},
   {"HPC","Hewlett Packard Co."},
   {"HPD","Hewlett Packard"},
   {"HPI","Headplay, Inc."},
   {"HPK","HAMAMATSU PHOTONICS K.K."},
   {"HPQ","HP"},
   {"HPR","H.P.R. Electronics GmbH"},
   {"HRC","Hercules"},
   {"HRE","Qingdao Haier Electronics Co., Ltd."},
   {"HRI","Hall Research"},
   {"HRL","Herolab GmbH"},
   {"HRS","Harris Semiconductor"},
   {"HRT","HERCULES"},
   {"HSC","Hagiwara Sys-Com Company Ltd"},
   {"HSD","HannStar Display Corp"},
   {"HSM","AT&T Microelectronics"},
   {"HSP","HannStar Display Corp"},
   {"HTC","Hitachi Ltd"},
   {"HTI","Hampshire Company, Inc."},
   {"HTK","Holtek Microelectronics Inc"},
   {"HTX","Hitex Systementwicklung GmbH"},
   {"HUB","GAI-Tronics, A Hubbell Company"},
   {"HUM","IMP Electronics Ltd."},
   {"HWA","Harris Canada Inc"},
   {"HWC","DBA Hans Wedemeyer"},
   {"HWD","Highwater Designs Ltd"},
   {"HWP","Hewlett Packard"},
   {"HXM","Hexium Ltd."},
   {"HYC","Hypercope Gmbh Aachen"},
   {"HYD","Hydis Technologies.Co.,LTD"},
   {"HYO","HYC CO., LTD."},
   {"HYP","Hyphen Ltd"},
   {"HYR","Hypertec Pty Ltd"},
   {"HYT","Heng Yu Technology   {HK} Limited"},
   {"HYV","Hynix Semiconductor"},
   {"IAF","Institut f r angewandte Funksystemtechnik GmbH"},
   {"IAI","Integration Associates, Inc."},
   {"IAT","IAT Germany GmbH"},
   {"IBC","Integrated Business Systems"},
   {"IBI","INBINE.CO.LTD"},
   {"IBM","IBM Brasil"},
   {"IBM","IBM France"},
   {"IBP","IBP Instruments GmbH"},
   {"IBR","IBR GmbH"},
   {"ICA","ICA Inc"},
   {"ICC","BICC Data Networks Ltd"},
   {"ICD","ICD Inc"},
   {"ICE","IC Ensemble"},
   {"ICI","Infotek Communication Inc"},
   {"ICM","Intracom SA"},
   {"ICN","Sanyo Icon"},
   {"ICO","Intel Corp"},
   {"ICP","ICP Electronics, Inc./iEi Technology Corp."},
   {"ICS","Integrated Circuit Systems"},
   {"ICV","Inside Contactless"},
   {"ICX","ICCC A/S"},
   {"IDC","International Datacasting Corporation"},
   {"IDE","IDE Associates"},
   {"IDK","IDK Corporation"},
   {"IDN","Idneo Technologies"},
   {"IDO","IDEO Product Development"},
   {"IDP","Integrated Device Technology, Inc."},
   {"IDS","Interdigital Sistemas de Informacao"},
   {"IDT","International Display Technology"},
   {"IDX","IDEXX Labs"},
   {"IEC","Interlace Engineering Corporation"},
   {"IEE","IEE"},
   {"IEI","Interlink Electronics"},
   {"IFS","In Focus Systems Inc"},
   {"IFT","Informtech"},
   {"IFX","Infineon Technologies AG"},
   {"IFZ","Infinite Z"},
   {"IGC","Intergate Pty Ltd"},
   {"IGM","IGM Communi"},
   {"IHE","InHand Electronics"},
   {"IIC","ISIC Innoscan Industrial Computers A/S"},
   {"III","Intelligent Instrumentation"},
   {"IIN","IINFRA Co., Ltd"},
   {"IKS","Ikos Systems Inc"},
   {"ILC","Image Logic Corporation"},
   {"ILS","Innotech Corporation"},
   {"IMA","Imagraph"},
   {"IMB","ART s.r.l."},
   {"IMC","IMC Networks"},
   {"IMD","ImasDe Canarias S.A."},
   {"IME","Imagraph"},
   {"IMG","IMAGENICS Co., Ltd."},
   {"IMI","International Microsystems Inc"},
   {"IMM","Immersion Corporation"},
   {"IMN","Impossible Production"},
   {"IMP","Impinj"},
   {"IMP","Impression Products Incorporated"},
   {"IMT","Inmax Technology Corporation"},
   {"INC","Home Row Inc"},
   {"IND","ILC"},
   {"INE","Inventec Electronics   {M} Sdn. Bhd."},
   {"INF","Inframetrics Inc"},
   {"ING","Integraph Corporation"},
   {"INI","Initio Corporation"},
   {"INK","Indtek Co., Ltd."},
   {"INL","InnoLux Display Corporation"},
   {"INM","InnoMedia Inc"},
   {"INN","Innovent Systems, Inc."},
   {"INO","Innolab Pte Ltd"},
   {"INP","Interphase Corporation"},
   {"INS","Ines GmbH"},
   {"INT","Interphase Corporation"},
   {"INU","Inovatec S.p.A."},              // actually "inu"
   {"INV","Inviso, Inc."},
   {"INX","Communications Supply Corporation   {A division of WESCO}"},
   {"INZ","Best Buy"},
   {"IOA","CRE Technology Corporation"},
   {"IOD","I-O Data Device Inc"},
   {"IOM","Iomega"},
   {"ION","Inside Out Networks"},
   {"IOS","i-O Display System"},
   {"IOT","I/OTech Inc"},
   {"IPC","IPC Corporation"},
   {"IPD","Industrial Products Design, Inc."},
   {"IPI","Intelligent Platform Management Interface   {IPMI} forum   {Intel, HP, NEC, Dell}"},
   {"IPM","IPM Industria Politecnica Meridionale SpA"},
   {"IPN","Performance Technologies"},
   {"IPP","IP Power Technologies GmbH "},
   {"IPR","Ithaca Peripherals"},
   {"IPS","IPS, Inc.   {Intellectual Property Solutions, Inc.}"},
   {"IPT","International Power Technologies"},
   {"IPW","IPWireless, Inc"},
   {"IQI","IneoQuest Technologies, Inc"},
   {"IQT","IMAGEQUEST Co., Ltd"},
   {"IRD","IRdata"},
   {"ISA","Symbol Technologies"},
   {"ISC","Id3 Semiconductors"},
   {"ISG","Insignia Solutions Inc"},
   {"ISI","Interface Solutions"},
   {"ISL","Isolation Systems"},
   {"ISM","Image Stream Medical"},
   {"ISP","IntreSource Systems Pte Ltd"},
   {"ISR","INSIS Co., LTD."},
   {"ISS","ISS Inc"},
   {"IST","Intersolve Technologies"},
   {"ISY","International Integrated Systems,Inc.  {IISI}"},
   {"ITA","Itausa Export North America"},
   {"ITC","Intercom Inc"},
   {"ITD","Internet Technology Corporation"},
   {"ITE","Integrated Tech Express Inc"},
   {"ITK","ITK Telekommunikation AG"},
   {"ITL","Inter-Tel"},
   {"ITM","ITM inc."},
   {"ITN","The NTI Group"},
   {"ITP","IT-PRO Consulting und Systemhaus GmbH"},
   {"ITR","Infotronic America, Inc."},
   {"ITS","IDTECH"},
   {"ITT","I&T Telecom."},
   {"ITX","integrated Technology Express Inc"},
   {"IUC","ICSL"},
   {"IVI","Intervoice Inc"},
   {"IVM","Iiyama North America"},
   {"IVS","Intevac Photonics Inc."},
   {"IWR","Icuiti Corporation"},
   {"IWX","Intelliworxx, Inc."},
   {"IXD","Intertex Data AB"},
   {"JAC","Astec Inc"},
   {"JAE","Japan Aviation Electronics Industry, Limited"},
   {"JAS","Janz Automationssysteme AG"},
   {"JAT","Jaton Corporation"},
   {"JAZ","Carrera Computer Inc"},
   {"JCE","Jace Tech Inc"},
   {"JDL","Japan Digital Laboratory Co.,Ltd."},
   {"JEN","N-Vision"},
   {"JET","JET POWER TECHNOLOGY CO., LTD."},
   {"JFX","Jones Futurex Inc"},
   {"JGD","University College"},
   {"JIC","Jaeik Information & Communication Co., Ltd."},
   {"JKC","JVC KENWOOD Corporation"},
   {"JMT","Micro Technical Company Ltd"},
   {"JPC","JPC Technology Limited"},
   {"JPW","Wallis Hamilton Industries"},
   {"JQE","CNet Technical Inc"},
   {"JSD","JS DigiTech, Inc"},
   {"JSI","Jupiter Systems, Inc."},
   {"JSK","SANKEN ELECTRIC CO., LTD"},
   {"JTS","JS Motorsports"},
   {"JTY","jetway security micro,inc"},
   {"JUK","Janich & Klass Computertechnik GmbH"},
   {"JUP","Jupiter Systems"},
   {"JVC","JVC"},
   {"JWD","Video International Inc."},
   {"JWL","Jewell Instruments, LLC"},
   {"JWS","JWSpencer & Co."},
   {"JWY","Jetway Information Co., Ltd"},
   {"KAR","Karna"},
   {"KBI","Kidboard Inc"},
   {"KBL","Kobil Systems GmbH"},
   {"KCD","Chunichi Denshi Co.,LTD."},
   {"KCL","Keycorp Ltd"},
   {"KDE","KDE"},
   {"KDK","Kodiak Tech"},
   {"KDM","Korea Data Systems Co., Ltd."},
   {"KDS","KDS USA"},
   {"KDT","KDDI Technology Corporation"},
   {"KEC","Kyushu Electronics Systems Inc"},
   {"KEM","Kontron Embedded Modules GmbH"},
   {"KES","Kesa Corporation"},
   {"KEY","Key Tech Inc"},
   {"KFC","SCD Tech"},
   {"KFE ","Komatsu Forest "},
   {"KFX","Kofax Image Products"},
   {"KGL","KEISOKU GIKEN Co.,Ltd."},
   {"KIS","KiSS Technology A/S"},
   {"KMC","Mitsumi Company Ltd"},
   {"KME","KIMIN Electronics Co., Ltd."},
   {"KML","Kensington Microware Ltd"},
   {"KNC","Konica corporation"},
   {"KNX","Nutech Marketing PTL"},
   {"KOB","Kobil Systems GmbH"},
   {"KOD","Eastman Kodak Company"},
   {"KOE","KOLTER ELECTRONIC"},
   {"KOL","Kollmorgen Motion Technologies Group"},
   {"KOU","KOUZIRO Co.,Ltd."},
   {"KOW","KOWA Company,LTD. "},
   {"KPC","King Phoenix Company"},
   {"KRL","Krell Industries Inc."},
   {"KRM","Kroma Telecom"},
   {"KRY","Kroy LLC"},
   {"KSC","Kinetic Systems Corporation"},
   {"KSL","Karn Solutions Ltd."},
   {"KSX","King Tester Corporation"},
   {"KTC","Kingston Tech Corporation"},
   {"KTD","Takahata Electronics Co.,Ltd."},
   {"KTE","K-Tech"},
   {"KTG","Kayser-Threde GmbH"},
   {"KTI","Konica Technical Inc"},
   {"KTK","Key Tronic Corporation"},
   {"KTN","Katron Tech Inc"},
   {"KUR","Kurta Corporation"},
   {"KVA","Kvaser AB"},
   {"KVX","KeyView"},
   {"KWD","Kenwood Corporation"},
   {"KYC","Kyocera Corporation"},
   {"KYE","KYE Syst Corporation"},
   {"KYK","Samsung Electronics America Inc"},
   {"KZI","K-Zone International co. Ltd."},
   {"KZN","K-Zone International"},
   {"LAB","ACT Labs Ltd"},
   {"LAC","LaCie"},
   {"LAF","Microline"},
   {"LAG","Laguna Systems"},
   {"LAN","Sodeman Lancom Inc"},
   {"LAS","LASAT Comm. A/S"},
   {"LAV","Lava Computer MFG Inc"},
   {"LBO","Lubosoft"},
   {"LCC","LCI"},
   {"LCD","Toshiba Matsushita Display Technology Co., Ltd"},
   {"LCE","La Commande Electronique"},
   {"LCI","Lite-On Communication Inc"},
   {"LCM","Latitude Comm."},
   {"LCN","LEXICON"},
   {"LCS","Longshine Electronics Company"},
   {"LCT","Labcal Technologies"},
   {"LDT","LogiDataTech Electronic GmbH"},
   {"LEC","Lectron Company Ltd"},
   {"LED","Long Engineering Design Inc"},
   {"LEG","Legerity, Inc"},
   {"LEN","Lenovo Group Limited"},
   {"LEO","First International Computer Inc"},
   {"LEX","Lexical Ltd"},
   {"LGC","Logic Ltd"},
   {"LGI","Logitech Inc"},
   {"LGS","LG Semicom Company Ltd"},
   {"LGX","Lasergraphics, Inc."},
   {"LHA","Lars Haagh ApS"},
   {"LHE","Lung Hwa Electronics Company Ltd"},
   {"LHT","Lighthouse Technologies Limited"},
   {"LIN","Lenovo Beijing Co. Ltd."},
   {"LIP","Linked IP GmbH"},
   {"LIT","Lithics Silicon Technology"},
   {"LJX","Datalogic Corporation"},
   {"LKM","Likom Technology Sdn. Bhd."},
   {"LLL","L-3 Communications"},
   {"LMG","Lucent Technologies"},
   {"LMI","Lexmark Int'l Inc"},
   {"LMP","Leda Media Products"},
   {"LMT","Laser Master"},
   {"LND","Land Computer Company Ltd"},
   {"LNK","Link Tech Inc"},
   {"LNR","Linear Systems Ltd."},
   {"LNT","LANETCO International"},
   {"LNV","Lenovo"},
   {"LOC","Locamation B.V."},
   {"LOE","Loewe Opta GmbH"},
   {"LOG","Logicode Technology Inc"},
   {"LOL","Litelogic Operations Ltd"},
   {"LPE","El-PUSK Co., Ltd."},
   {"LPI","Design Technology"},
   {"LSC","LifeSize Communications"},
   {"LSD","Intersil Corporation"},
   {"LSI","Loughborough Sound Images"},
   {"LSJ","LSI Japan Company Ltd"},
   {"LSL","Logical Solutions"},
   {"LSY","LSI Systems Inc"},
   {"LTC","Labtec Inc"},
   {"LTI","Jongshine Tech Inc"},
   {"LTK","Lucidity Technology Company Ltd"},
   {"LTN","Litronic Inc"},
   {"LTS","LTS Scale LLC"},
   {"LTV","Leitch Technology International Inc."},
   {"LTW","Lightware, Inc"},
   {"LUC","Lucent Technologies"},
   {"LUM","Lumagen, Inc."},
   {"LUX","Luxxell Research Inc"},
   {"LVI","LVI Low Vision International AB"},
   {"LWC","Labway Corporation"},
   {"LWR","Lightware Visual Engineering"},
   {"LWW","Lanier Worldwide"},
   {"LXC","LXCO Technologies AG"},
   {"LXN","Luxeon"},
   {"LXS","ELEA CardWare"},
   {"LZX","Lightwell Company Ltd"},
   {"MAC","MAC System Company Ltd"},
   {"MAD","Xedia Corporation"},
   {"MAE","Maestro Pty Ltd"},
   {"MAG","MAG InnoVision"},
   {"MAI","Mutoh America Inc"},
   {"MAL","Meridian Audio Ltd"},
   {"MAN","LGIC"},
   {"MAS","Mass Inc."},
   {"MAT","Matsushita Electric Ind. Company Ltd"},
   {"MAX","Rogen Tech Distribution Inc"},
   {"MAY","Maynard Electronics"},
   {"MAZ","MAZeT GmbH"},
   {"MBC","MBC"},
   {"MBD","Microbus PLC"},
   {"MBM","Marshall Electronics"},
   {"MBV","Moreton Bay"},
   {"MCA","American Nuclear Systems Inc"},
   {"MCC","Micro Industries"},
   {"MCD","McDATA Corporation"},
   {"MCE","Metz-Werke GmbH & Co KG"},
   {"MCG","Motorola Computer Group"},
   {"MCI","Micronics Computers"},
   {"MCL","Motorola Communications Israel"},
   {"MCM","Metricom Inc"},
   {"MCN","Micron Electronics Inc"},
   {"MCO","Motion Computing Inc."},
   {"MCP","Magni Systems Inc"},
   {"MCQ","Mat's Computers"},
   {"MCR","Marina Communicaitons"},
   {"MCS","Micro Computer Systems"},
   {"MCT","Microtec"},
   {"MDA","Media4 Inc"},
   {"MDC","Midori Electronics"},
   {"MDD","MODIS"},
   {"MDG","Madge Networks"},
   {"MDI","Micro Design Inc"},
   {"MDK","Mediatek Corporation"},
   {"MDO","Panasonic"},
   {"MDR","Medar Inc"},
   {"MDS","Micro Display Systems Inc"},
   {"MDT","Magus Data Tech"},
   {"MDV","MET Development Inc"},
   {"MDX","MicroDatec GmbH"},
   {"MDY","Microdyne Inc"},
   {"MEC","Mega System Technologies Inc"},
   {"MED","Messeltronik Dresden GmbH"},
   {"MEE","Mitsubishi Electric Engineering Co., Ltd."},
   {"MEG","Abeam Tech Ltd"},
   {"MEI","Panasonic Industry Company"},
   {"MEJ","Mac-Eight Co., LTD."},
   {"MEL","Mitsubishi Electric Corporation"},
   {"MEN","MEN Mikroelectronik Nueruberg GmbH"},
   {"MEP","Meld Technology"},
   {"MEQ","Matelect Ltd."},
   {"MET","Metheus Corporation"},
   {"MEX","MSC Vertriebs GmbH"},
   {"MFG","MicroField Graphics Inc"},
   {"MFI","Micro Firmware"},
   {"MFR","MediaFire Corp."},
   {"MGA","Mega System Technologies, Inc."},
   {"MGC","Mentor Graphics Corporation"},
   {"MGE","Schneider Electric S.A."},
   {"MGL","M-G Technology Ltd"},
   {"MGT","Megatech R & D Company"},
   {"MIC","Micom Communications Inc"},
   {"MID","miro Displays"},
   {"MII","Mitec Inc"},
   {"MIL","Marconi Instruments Ltd"},
   {"MIM","Mimio – A Newell Rubbermaid Company"},
   {"MIN","Minicom Digital Signage"},
   {"MIP","micronpc.com"},
   {"MIR","Miro Computer Prod."},
   {"MIS","Modular Industrial Solutions Inc"},
   {"MIT","MCM Industrial Technology GmbH"},
   {"MJI","MARANTZ JAPAN, INC."},
   {"MJS","MJS Designs"},
   {"MKC","Media Tek Inc."},
   {"MKT","MICROTEK Inc."},
   {"MKV","Trtheim Technology"},
   {"MLD","Deep Video Imaging Ltd"},
   {"MLG","Micrologica AG"},
   {"MLI","McIntosh Laboratory Inc."},
   {"MLM","Millennium Engineering Inc"},
   {"MLN","Mark Levinson"},
   {"MLS","Milestone EPE"},
   {"MLX","Mylex Corporation"},
   {"MMA","Micromedia AG"},
   {"MMD","Micromed Biotecnologia Ltd"},
   {"MMF","Minnesota Mining and Manufacturing"},
   {"MMI","Multimax"},
   {"MMM","Electronic Measurements"},
   {"MMN","MiniMan Inc"},
   {"MMS","MMS Electronics"},
   {"MNC","Mini Micro Methods Ltd"},
   {"MNL","Monorail Inc"},
   {"MNP","Microcom"},
   {"MOD","Modular Technology"},
   {"MOM","Momentum Data Systems"},
   {"MOS","Moses Corporation"},
   {"MOT","Motorola UDS"},
   {"MPC","M-Pact Inc"},
   {"MPI","Mediatrix Peripherals Inc"},
   {"MPJ","Microlab"},
   {"MPL","Maple Research Inst. Company Ltd"},
   {"MPN","Mainpine Limited"},
   {"MPS","mps Software GmbH"},
   {"MPX","Micropix Technologies, Ltd."},
   {"MQP","MultiQ Products AB"},
   {"MRA","Miranda Technologies Inc"},
   {"MRC","Marconi Simulation & Ty-Coch Way Training"},
   {"MRD","MicroDisplay Corporation"},
   {"MRK","Maruko & Company Ltd"},
   {"MRL","Miratel"},
   {"MRO","Medikro Oy"},
   {"MRT","Merging Technologies"},
   {"MSA","Micro Systemation AB"},
   {"MSC","Mouse Systems Corporation"},
   {"MSD","Datenerfassungs- und Informationssysteme"},
   {"MSF","M-Systems Flash Disk Pioneers"},
   {"MSG","MSI GmbH"},
   {"MSH","Microsoft"},
   {"MSI","Microstep"},
   {"MSK","Megasoft Inc"},
   {"MSL","MicroSlate Inc."},
   {"MSM","Advanced Digital Systems"},
   {"MSP","Mistral Solutions [P] Ltd."},
   {"MSR","MASPRO DENKOH Corp."},
   {"MST","MS Telematica"},
   {"MSU","motorola"},
   {"MSV","Mosgi Corporation"},
   {"MSX","Micomsoft Co., Ltd."},
   {"MSY","MicroTouch Systems Inc"},
   {"MTB","Media Technologies Ltd."},
   {"MTC","Mars-Tech Corporation"},
   {"MTD","MindTech Display Co. Ltd"},
   {"MTE","MediaTec GmbH"},
   {"MTH","Micro-Tech Hearing Instruments"},
   {"MTI","MaxCom Technical Inc"},
   {"MTI","Motorola Inc."},
   {"MTK","Microtek International Inc."},
   {"MTL","Mitel Corporation"},
   {"MTM","Motium"},
   {"MTN","Mtron Storage Technology Co., Ltd."},
   {"MTR","Mitron computer Inc"},
   {"MTS","Multi-Tech Systems"},
   {"MTU","Mark of the Unicorn Inc"},
   {"MTX","Matrox"},
   {"MUD","Multi-Dimension Institute"},
   {"MUK","mainpine limited"},
   {"MVD","Microvitec PLC"},
   {"MVI","Media Vision Inc"},
   {"MVM","SOBO VISION"},
   {"MVS","Microvision"},
   {"MVX","COM 1"},
   {"MWI","Multiwave Innovation Pte Ltd"},
   {"MWR","mware"},
   {"MWY","Microway Inc"},
   {"MXD","MaxData Computer GmbH & Co.KG"},
   {"MXI","Macronix Inc"},
   {"MXL","Hitachi Maxell, Ltd."},
   {"MXP","Maxpeed Corporation"},
   {"MXT","Maxtech Corporation"},
   {"MXV","MaxVision Corporation"},
   {"MYA","Monydata"},
   {"MYR","Myriad Solutions Ltd"},
   {"MYX","Micronyx Inc"},
   {"NAC","Ncast Corporation"},
   {"NAD","NAD Electronics"},
   {"NAK","Nakano Engineering Co.,Ltd."},
   {"NAL","Network Alchemy"},
   {"NAT","NaturalPoint Inc."},
   {"NAV","Navigation Corporation"},
   {"NAX","Naxos Tecnologia"},
   {"NBL","N*Able Technologies Inc"},
   {"NBS","National Key Lab. on ISN"},
   {"NBT","NingBo Bestwinning Technology CO., Ltd"},
   {"NCA","Nixdorf Company"},
   {"NCC","NCR Corporation"},
   {"NCE","Norcent Technology, Inc."},
   {"NCI","NewCom Inc"},
   {"NCL","NetComm Ltd"},
   {"NCR","NCR Electronics"},
   {"NCS","Northgate Computer Systems"},
   {"NCT","NEC CustomTechnica, Ltd."},
   {"NDC","National DataComm Corporaiton"},
   {"NDI","National Display Systems"},
   {"NDK","Naitoh Densei CO., LTD."},
   {"NDL","Network Designers"},
   {"NDS","Nokia Data"},
   {"NEC","NEC Corporation"},
   {"NEO","NEO TELECOM CO.,LTD."},
   {"NET","Mettler Toledo"},
   {"NEU","NEUROTEC - EMPRESA DE PESQUISA E DESENVOLVIMENTO EM BIOMEDICINA"},
   {"NEX","Nexgen Mediatech Inc.,"},
   {"NFC","BTC Korea Co., Ltd"},
   {"NFS","Number Five Software"},
   {"NGC","Network General"},
   {"NGS","A D S Exports"},
   {"NHT","Vinci Labs"},
   {"NIC","National Instruments Corporation"},
   {"NIS","Nissei Electric Company"},
   {"NIT","Network Info Technology"},
   {"NIX","Seanix Technology Inc"},
   {"NLC","Next Level Communications"},
   {"NME","Navico, Inc."},
   {"NMP","Nokia Mobile Phones"},
   {"NMS","Natural Micro System"},
   {"NMV","NEC-Mitsubishi Electric Visual Systems Corporation"},
   {"NMX","Neomagic"},
   {"NNC","NNC"},
   {"NOE","NordicEye AB"},
   {"NOI","North Invent A/S"},
   {"NOK","Nokia Display Products"},
   {"NOR","Norand Corporation"},
   {"NOT","Not Limited Inc"},
   {"NPI","Network Peripherals Inc"},
   {"NRL","U.S. Naval Research Lab"},
   {"NRT","Beijing Northern Radiantelecom Co."},
   {"NRV","Taugagreining hf"},
   {"NSC","National Semiconductor Corporation"},
   {"NSI","NISSEI ELECTRIC CO.,LTD"},
   {"NSP","Nspire System Inc."},
   {"NSS","Newport Systems Solutions"},
   {"NST","Network Security Technology Co"},
   {"NTC","NeoTech S.R.L"},
   {"NTI","New Tech Int'l Company"},
   {"NTL","National Transcomm. Ltd"},
   {"NTN","Nuvoton Technology Corporation "},
   {"NTR","N-trig Innovative Technologies, Inc."},
   {"NTS","Nits Technology Inc."},
   {"NTT","NTT Advanced Technology Corporation"},
   {"NTW","Networth Inc"},
   {"NTX","Netaccess Inc"},
   {"NUG","NU Technology, Inc."},
   {"NUI","NU Inc."},
   {"NVC","NetVision Corporation"},
   {"NVD","Nvidia"},
   {"NVI","NuVision US, Inc."},
   {"NVL","Novell Inc"},
   {"NVT","Navatek Engineering Corporation"},
   {"NWC","NW Computer Engineering"},
   {"NWP","NovaWeb Technologies Inc"},
   {"NWS","Newisys, Inc."},
   {"NXC","NextCom K.K."},
   {"NXG","Nexgen"},
   {"NXP","NXP Semiconductors bv."},
   {"NXQ","Nexiq Technologies, Inc."},
   {"NXS","Technology Nexus Secure Open Systems AB"},
   {"NYC","nakayo telecommunications,inc."},
   {"OAK","Oak Tech Inc"},
   {"OAS","Oasys Technology Company"},
   {"OBS","Optibase Technologies"},
   {"OCD","Macraigor Systems Inc"},
   {"OCN","Olfan"},
   {"OCS","Open Connect Solutions"},
   {"ODM","ODME Inc."},
   {"ODR","Odrac"},
   {"OEC","ORION ELECTRIC CO.,LTD"},
   {"OEI","Optum Engineering Inc."},
   {"OIC","Option Industrial Computers"},
   {"OIM","Option International"},
   {"OIN","Option International"},
   {"OKI","OKI Electric Industrial Company Ltd"},
   {"OLC","Olicom A/S"},
   {"OLD","Olidata S.p.A."},
   {"OLI","Olivetti"},
   {"OLT","Olitec S.A."},
   {"OLV","Olitec S.A."},
   {"OLY","OLYMPUS CORPORATION"},
   {"OMC","OBJIX Multimedia Corporation"},
   {"OMN","Omnitel"},
   {"OMR","Omron Corporation"},
   {"ONE","Oneac Corporation"},
   {"ONK","ONKYO Corporation"},
   {"ONL","OnLive, Inc"},
   {"ONS","On Systems Inc"},
   {"ONW","OPEN Networks Ltd"},
   {"ONX","SOMELEC Z.I. Du Vert Galanta"},
   {"OOS","OSRAM"},
   {"OPC","Opcode Inc"},
   {"OPI","D.N.S. Corporation"},
   {"OPP","OPPO Digital, Inc."},
   {"OPT","OPTi Inc"},
   {"OPV","Optivision Inc"},
   {"OQI","Oksori Company Ltd"},
   {"ORG","ORGA Kartensysteme GmbH"},
   {"ORI","OSR Open Systems Resources, Inc."},
   {"ORN","ORION ELECTRIC CO., LTD."},
   {"OSA","OSAKA Micro Computer, Inc."},
   {"OSP","OPTI-UPS Corporation"},
   {"OSR","Oksori Company Ltd"},
   {"OTB","outsidetheboxstuff.com"},
   {"OTI","Orchid Technology"},
   {"OTM","Optoma Corporation           "},
   {"OTT","OPTO22, Inc."},
   {"OUK","OUK Company Ltd"},
   {"OVR","Oculus VR, Inc."},
   {"OWL","Mediacom Technologies Pte Ltd"},
   {"OXU","Oxus Research S.A."},
   {"OYO","Shadow Systems"},
   {"OZC","OZ Corporation"},
   {"OZO","Tribe Computer Works Inc"},
   {"PAC","Pacific Avionics Corporation"},
   {"PAD","Promotion and Display Technology Ltd."},
   {"PAK","Many CNC System Co., Ltd."},
   {"PAM","Peter Antesberger Messtechnik"},
   {"PAN","The Panda Project"},
   {"PAR","Parallan Comp Inc"},
   {"PBI","Pitney Bowes"},
   {"PBL","Packard Bell Electronics"},
   {"PBN","Packard Bell NEC"},
   {"PBV","Pitney Bowes"},
   {"PCA","Philips BU Add On Card"},
   {"PCB","OCTAL S.A."},
   {"PCC","PowerCom Technology Company Ltd"},
   {"PCG","First Industrial Computer Inc"},
   {"PCI","Pioneer Computer Inc"},
   {"PCK","PCBANK21"},
   {"PCL","pentel.co.,ltd"},
   {"PCM","PCM Systems Corporation"},
   {"PCO","Performance Concepts Inc.,"},
   {"PCP","Procomp USA Inc"},
   {"PCS","TOSHIBA PERSONAL COMPUTER SYSTEM CORPRATION"},
   {"PCT","PC-Tel Inc"},
   {"PCW","Pacific CommWare Inc"},
   {"PCX","PC Xperten"},
   {"PDM","Psion Dacom Plc."},
   {"PDN","AT&T Paradyne"},
   {"PDR","Pure Data Inc"},
   {"PDS","PD Systems International Ltd"},
   {"PDT","PDTS - Prozessdatentechnik und Systeme"},
   {"PDV","Prodrive B.V."},
   {"PEC","POTRANS Electrical Corp."},
   {"PEI","PEI Electronics Inc"},
   {"PEL","Primax Electric Ltd"},
   {"PEN","Interactive Computer Products Inc"},
   {"PEP","Peppercon AG"},
   {"PER","Perceptive Signal Technologies"},
   {"PET","Practical Electronic Tools"},
   {"PFT","Telia ProSoft AB"},
   {"PGI","PACSGEAR, Inc."},
   {"PGM","Paradigm Advanced Research Centre"},
   {"PGP","propagamma kommunikation"},
   {"PGS","Princeton Graphic Systems"},
   {"PHC","Pijnenburg Beheer N.V."},
   {"PHE","Philips Medical Systems Boeblingen GmbH"},
   {"PHL","Philips Consumer Electronics Company"},
   {"PHO","Photonics Systems Inc."},
   {"PHS","Philips Communication Systems"},
   {"PHY","Phylon Communications"},
   {"PIE","Pacific Image Electronics Company Ltd"},
   {"PIM","Prism, LLC"},
   {"PIO","Pioneer Electronic Corporation"},
   {"PIX","Pixie Tech Inc"},
   {"PJA","Projecta"},
   {"PJD","Projectiondesign AS"},
   {"PJT","Pan Jit International Inc."},
   {"PKA","Acco UK ltd. "},
   {"PLC","Pro-Log Corporation"},
   {"PLF","Panasonic Avionics Corporation"},
   {"PLM","PROLINK Microsystems Corp."},
   {"PLT","PT Hartono Istana Teknologi"},
   {"PLV","PLUS Vision Corp."},
   {"PLX","Parallax Graphics"},
   {"PLY","Polycom Inc."},
   {"PMC","PMC Consumer Electronics Ltd"},
   {"PMD","TDK USA Corporation"},
   {"PMM","Point Multimedia System"},
   {"PMT","Promate Electronic Co., Ltd."},
   {"PMX","Photomatrix"},
   {"PNG","Microsoft"},
   {"PNG","P.I. Engineering Inc"},
   {"PNL","Panelview, Inc."},
   {"PNP","Microsoft"},
   {"PNR","Planar Systems, Inc."},
   {"PNS","PanaScope"},
   {"PNX","Phoenix Technologies, Ltd."},
   {"POL","PolyComp   {PTY} Ltd."},
   {"PON","Perpetual Technologies, LLC"},
   {"POR","Portalis LC"},
   {"PPC","Phoenixtec Power Company Ltd"},
   {"PPD","MEPhI"},
   {"PPI","Practical Peripherals"},
   {"PPM","Clinton Electronics Corp."},
   {"PPP","Purup Prepress AS"},
   {"PPR","PicPro"},
   {"PPX","Perceptive Pixel Inc."},
   {"PQI","Pixel Qi"},
   {"PRA","PRO/AUTOMATION"},
   {"PRC","PerComm"},
   {"PRD","Praim S.R.L."},
   {"PRF","Digital Electronics Corporation"},
   {"PRG","The Phoenix Research Group Inc"},
   {"PRI","Priva Hortimation BV"},
   {"PRM","Prometheus"},
   {"PRO","Proteon"},
   {"PRS","Leutron Vision"},
   {"PRT","Parade Technologies, Ltd."},
   {"PRX","Proxima Corporation"},
   {"PSA","Advanced Signal Processing Technologies"},
   {"PSC","Philips Semiconductors"},
   {"PSD","Peus-Systems GmbH"},
   {"PSE","Practical Solutions Pte., Ltd."},
   {"PSI","PSI-Perceptive Solutions Inc"},
   {"PSL","Perle Systems Limited"},
   {"PSM","Prosum"},
   {"PST","Global Data SA"},
   {"PSY","Prodea Systems Inc."},
   {"PTA","PAR Tech Inc."},
   {"PTC","PS Technology Corporation"},
   {"PTG","Cipher Systems Inc"},
   {"PTH","Pathlight Technology Inc"},
   {"PTI","Promise Technology Inc"},
   {"PTL","Pantel Inc"},
   {"PTS","Plain Tree Systems Inc"},
   {"PUL","Pulse-Eight Ltd"},
   {"PVG","Proview Global Co., Ltd"},
   {"PVI","Prime view international Co., Ltd"},
   {"PVM","Penta Studiotechnik GmbH"},
   {"PVN","Pixel Vision"},
   {"PVP","Klos Technologies, Inc."},
   {"PXC","Phoenix Contact"},
   {"PXE","PIXELA CORPORATION"},
   {"PXL","The Moving Pixel Company"},
   {"PXM","Proxim Inc"},
   {"QCC","QuakeCom Company Ltd"},
   {"QCH","Metronics Inc"},
   {"QCI","Quanta Computer Inc"},
   {"QCK","Quick Corporation"},
   {"QCL","Quadrant Components Inc"},
   {"QCP","Qualcomm Inc"},
   {"QDI","Quantum Data Incorporated"},
   {"QDM","Quadram"},
   {"QDS","Quanta Display Inc."},
   {"QFF","Padix Co., Inc."},
   {"QFI","Quickflex, Inc"},
   {"QLC","Q-Logic"},
   {"QQQ","Chuomusen Co., Ltd. "},
   {"QSI","Quantum Solutions, Inc."},
   {"QTD","Quantum 3D Inc"},
   {"QTH","Questech Ltd"},
   {"QTI","Quicknet Technologies Inc"},
   {"QTM","Quantum"},
   {"QTR","Qtronix Corporation"},
   {"QUA","Quatographic AG"},
   {"QUE","Questra Consulting"},
   {"QVU","Quartics"},
   {"RAC","Racore Computer Products Inc"},
   {"RAD","Radisys Corporation"},
   {"RAI","Rockwell Automation/Intecolor"},
   {"RAN","Rancho Tech Inc"},
   {"RAR","Raritan, Inc."},
   {"RAS","RAScom Inc"},
   {"RAT","Rent-A-Tech"},
   {"RAY","Raylar Design, Inc."},
   {"RCE","Parc d'Activite des Bellevues"},
   {"RCH","Reach Technology Inc"},
   {"RCI","RC International"},
   {"RCN","Radio Consult SRL"},
   {"RCO","Rockwell Collins"},
   {"RDI","Rainbow Displays, Inc."},
   {"RDM","Tremon Enterprises Company Ltd"},
   {"RDN","RADIODATA GmbH"},
   {"RDS","Radius Inc"},
   {"REA","Real D"},
   {"REC","ReCom"},
   {"RED","Research Electronics Development Inc"},
   {"REF","Reflectivity, Inc."},
   {"REH","Rehan Electronics Ltd."},
   {"REL","Reliance Electric Ind Corporation"},
   {"REM","SCI Systems Inc."},
   {"REN","Renesas Technology Corp."},
   {"RES","ResMed Pty Ltd"},
   {"RET","Resonance Technology, Inc."},
   {"REX","RATOC Systems, Inc."},
   {"RGB","RGB Spectrum"},
   {"RGL","Robertson Geologging Ltd"},
   {"RHD","RightHand Technologies"},
   {"RHM","Rohm Company Ltd"},
   {"RHT","Red Hat, Inc."},
   {"RIC","RICOH COMPANY, LTD."},
   {"RII","Racal Interlan Inc"},
   {"RIO","Rios Systems Company Ltd"},
   {"RIT","Ritech Inc"},
   {"RIV","Rivulet Communications"},
   {"RJA","Roland Corporation"},
   {"RJS","Advanced Engineering"},
   {"RKC","Reakin Technolohy Corporation"},
   {"RLD","MEPCO"},
   {"RLN","RadioLAN Inc"},
   {"RMC","Raritan Computer, Inc"},
   {"RMP","Research Machines"},
   {"RMT","Roper Mobile"},
   {"RNB","Rainbow Technologies"},
   {"ROB","Robust Electronics GmbH "},
   {"ROH","Rohm Co., Ltd."},
   {"ROK","Rockwell International"},
   {"ROP","Roper International Ltd"},
   {"ROS","Rohde & Schwarz"},
   {"RPI","RoomPro Technologies"},
   {"RPT","R.P.T.Intergroups"},
   {"RRI","Radicom Research Inc"},
   {"RSC","PhotoTelesis"},
   {"RSH","ADC-Centre"},
   {"RSI","Rampage Systems Inc"},
   {"RSN","Radiospire Networks, Inc."},
   {"RSQ","R Squared"},
   {"RSS","Rockwell Semiconductor Systems"},
   {"RSV","Ross Video Ltd"},
   {"RSX","Rapid Tech Corporation"},
   {"RTC","Relia Technologies"},
   {"RTI","Rancho Tech Inc"},
   {"RTL","Realtek Semiconductor Company Ltd"},
   {"RTS","Raintree Systems"},
   {"RUN","RUNCO International"},
   {"RUP","Ups Manufactoring s.r.l."},
   {"RVC","RSI Systems Inc"},
   {"RVI","Realvision Inc"},
   {"RVL","Reveal Computer Prod"},
   {"RWC","Red Wing Corporation"},
   {"RXT","Tectona SoftSolutions   {P} Ltd.,"},
   {"SAA","Sanritz Automation Co.,Ltd."},
   {"SAE","Saab Aerotech"},
   {"SAG","Sedlbauer"},
   {"SAI","Sage Inc"},
   {"SAK","Saitek Ltd"},
   {"SAM","Samsung Electric Company"},
   {"SAN","Sanyo Electric Co.,Ltd."},
   {"SAS","Stores Automated Systems Inc"},
   {"SAT","Shuttle Tech"},
   {"SBC","Shanghai Bell Telephone Equip Mfg Co"},
   {"SBD","Softbed - Consulting & Development Ltd"},
   {"SBI","SMART Technologies Inc."},
   {"SBS","SBS-or Industrial Computers GmbH"},
   {"SBT","Senseboard Technologies AB"},
   {"SCB","SeeCubic B.V."},
   {"SCC","SORD Computer Corporation"},
   {"SCD","Sanyo Electric Company Ltd"},
   {"SCE","Sun Corporation"},
   {"SCH","Schlumberger Cards"},
   {"SCI","System Craft"},
   {"SCL","Sigmacom Co., Ltd."},
   {"SCM","SCM Microsystems Inc"},
   {"SCN","Scanport, Inc."},
   {"SCO","SORCUS Computer GmbH"},
   {"SCP","Scriptel Corporation"},
   {"SCR","Systran Corporation"},
   {"SCS","Nanomach Anstalt"},
   {"SCT","Smart Card Technology"},
   {"SDA","SAT   {Societe Anonyme}"},
   {"SDD","Intrada-SDD Ltd"},
   {"SDE","Sherwood Digital Electronics Corporation"},
   {"SDF","SODIFF E&T CO., Ltd."},
   {"SDH","Communications Specialies, Inc."},
   {"SDI","Samtron Displays Inc"},
   {"SDK","SAIT-Devlonics"},
   {"SDR","SDR Systems"},
   {"SDS","SunRiver Data System"},
   {"SDT","Siemens AG"},
   {"SDX","SDX Business Systems Ltd"},
   {"SEA","Seanix Technology Inc."},
   {"SEB","system elektronik GmbH"},
   {"SEC","Seiko Epson Corporation"},
   {"SEE","SeeColor Corporation"},
   {"SEI","Seitz & Associates Inc"},
   {"SEL","Way2Call Communications"},
   {"SEM","Samsung Electronics Company Ltd"},
   {"SEN","Sencore"},
   {"SEO","SEOS Ltd"},
   {"SEP","SEP Eletronica Ltda."},
   {"SER","Sony Ericsson Mobile Communications Inc."},
   {"SES","Session Control LLC"},
   {"SET","SendTek Corporation"},
   {"SFM","TORNADO Company"},
   {"SFT","Mikroforum Ring 3"},
   {"SGC","Spectragraphics Corporation"},
   {"SGD","Sigma Designs, Inc."},
   {"SGE","Kansai Electric Company Ltd"},
   {"SGI","Scan Group Ltd"},
   {"SGL","Super Gate Technology Company Ltd"},
   {"SGM","SAGEM"},
   {"SGO","Logos Design A/S"},
   {"SGT","Stargate Technology"},
   {"SGW","Shanghai Guowei Science and Technology Co., Ltd."},
   {"SGX","Silicon Graphics Inc"},
   {"SGZ","Systec Computer GmbH"},
   {"SHC","ShibaSoku Co., Ltd."},
   {"SHG","Soft & Hardware development Goldammer GmbH"},
   {"SHI","Jiangsu Shinco Electronic Group Co., Ltd"},
   {"SHP","Sharp Corporation"},
   {"SHR","Digital Discovery"},
   {"SHT","Shin Ho Tech"},
   {"SIA","SIEMENS AG"},
   {"SIB","Sanyo Electric Company Ltd"},
   {"SIC","Sysmate Corporation"},
   {"SID","Seiko Instruments Information Devices Inc"},
   {"SIE","Siemens"},
   {"SIG","Sigma Designs Inc"},
   {"SII","Silicon Image, Inc."},
   {"SIL","Silicon Laboratories, Inc"},
   {"SIM","S3 Inc"},
   {"SIN","Singular Technology Co., Ltd."},
   {"SIR","Sirius Technologies Pty Ltd"},
   {"SIS","Silicon Integrated Systems Corporation"},
   {"SIT","Sitintel"},
   {"SIU","Seiko Instruments USA Inc"},
   {"SIX","Zuniq Data Corporation"},
   {"SJE","Sejin Electron Inc"},
   {"SKD","Schneider & Koch"},
   {"SKT","Samsung Electro-Mechanics Company Ltd"},
   {"SKY","SKYDATA S.P.A."},
   {"SLA","Systeme Lauer GmbH&Co KG"},
   {"SLB","Shlumberger Ltd"},
   {"SLC","Syslogic Datentechnik AG"},
   {"SLF","StarLeaf"},
   {"SLH","Silicon Library Inc."},
   {"SLI","Symbios Logic Inc"},
   {"SLK","Silitek Corporation"},
   {"SLM","Solomon Technology Corporation"},
   {"SLR","Schlumberger Technology Corporate"},
   {"SLS","Schnick-Schnack-Systems GmbH"},
   {"SLT","Salt Internatioinal Corp."},
   {"SLX","Specialix"},
   {"SMA","SMART Modular Technologies"},
   {"SMB","Schlumberger"},
   {"SMC","Standard Microsystems Corporation"},
   {"SME","Sysmate Company"},
   {"SMI","SpaceLabs Medical Inc"},
   {"SMK","SMK CORPORATION"},
   {"SML","Sumitomo Metal Industries, Ltd."},
   {"SMM","Shark Multimedia Inc"},
   {"SMO","STMicroelectronics"},
   {"SMP","Simple Computing"},
   {"SMR","B.& V. s.r.l."},
   {"SMS","Silicom Multimedia Systems Inc"},
   {"SMT","Silcom Manufacturing Tech Inc"},
   {"SNC","Sentronic International Corp."},
   {"SNI","Siemens Microdesign GmbH"},
   {"SNK","S&K Electronics"},
   {"SNO","SINOSUN TECHNOLOGY CO., LTD"},
   {"SNP","Siemens Nixdorf Info Systems"},
   {"SNS","Cirtech   {UK} Ltd"},
   {"SNT","SuperNet Inc"},
   {"SNW","Snell & Wilcox"},
   {"SNX","Sonix Comm. Ltd"},
   {"SNY","Sony"},
   {"SOI","Silicon Optix Corporation"},
   {"SOL","Solitron Technologies Inc"},
   {"SON","Sony"},
   {"SOR","Sorcus Computer GmbH"},
   {"SOT","Sotec Company Ltd"},
   {"SOY","SOYO Group, Inc"},
   {"SPC","SpinCore Technologies, Inc"},
   {"SPE","SPEA Software AG"},
   {"SPH","G&W Instruments GmbH"},
   {"SPI","SPACE-I Co., Ltd."},
   {"SPK","SpeakerCraft"},
   {"SPL","Smart Silicon Systems Pty Ltd"},
   {"SPN","Sapience Corporation"},
   {"SPR","pmns GmbH"},
   {"SPS","Synopsys Inc"},
   {"SPT","Sceptre Tech Inc"},
   {"SPU","SIM2 Multimedia S.P.A."},
   {"SPX","Simplex Time Recorder Co."},
   {"SQT","Sequent Computer Systems Inc"},
   {"SRC","Integrated Tech Express Inc"},
   {"SRD","Setred"},
   {"SRF","Surf Communication Solutions Ltd"},
   {"SRG","Intuitive Surgical, Inc."},
   {"SRS","SR-Systems e.K."},
   {"SRT","SeeReal Technologies GmbH"},
   {"SSC","Sierra Semiconductor Inc"},
   {"SSD","FlightSafety International"},
   {"SSE","Samsung Electronic Co."},
   {"SSI","S-S Technology Inc"},
   {"SSJ","Sankyo Seiki Mfg.co., Ltd"},
   {"SSP","Spectrum Signal Proecessing Inc"},
   {"SSS","S3 Inc"},
   {"SST","SystemSoft Corporation"},
   {"STA","ST Electronics Systems Assembly Pte Ltd"},
   {"STB","STB Systems Inc"},
   {"STC","STAC Electronics"},
   {"STD","STD Computer Inc"},
   {"STE","SII Ido-Tsushin Inc"},
   {"STF","Starflight Electronics"},
   {"STG","StereoGraphics Corp."},
   {"STH","Semtech Corporation"},
   {"STI","Smart Tech Inc"},
   {"STK","SANTAK CORP."},
   {"STL","SigmaTel Inc"},
   {"STM","SGS Thomson Microelectronics"},
   {"STN","Samsung Electronics America"},
   {"STO","Stollmann E+V GmbH"},
   {"STP","StreamPlay Ltd"},
   {"STR","Starlight Networks Inc"},
   {"STS","SITECSYSTEM CO., LTD."},
   {"STT","Star Paging Telecom Tech   {Shenzhen} Co. Ltd."},
   {"STU","Sentelic Corporation"},
   {"STW","Starwin Inc."},
   {"STX","ST-Ericsson"},
   {"STY","SDS Technologies"},
   {"SUB","Subspace Comm. Inc"},
   {"SUM","Summagraphics Corporation"},
   {"SUN","Sun Electronics Corporation"},
   {"SUP","Supra Corporation"},
   {"SUR","Surenam Computer Corporation"},
   {"SVA","SGEG"},
   {"SVC","Intellix Corp."},
   {"SVD","SVD Computer"},
   {"SVI","Sun Microsystems"},
   {"SVS","SVSI"},
   {"SVT","SEVIT Co., Ltd."},
   {"SWC","Software Café"},
   {"SWI","Sierra Wireless Inc."},
   {"SWL","Sharedware Ltd"},
   {"SWS","Static"},
   {"SWT","Software Technologies Group,Inc."},
   {"SXB","Syntax-Brillian"},
   {"SXD","Silex technology, Inc."},
   {"SXG","SELEX GALILEO"},
   {"SXL","SolutionInside"},
   {"SXT","SHARP TAKAYA ELECTRONIC INDUSTRY CO.,LTD."},
   {"SYC","Sysmic"},
   {"SYE","SY Electronics Ltd"},
   {"SYK","Stryker Communications"},
   {"SYL","Sylvania Computer Products"},
   {"SYM","Symicron Computer Communications Ltd."},
   {"SYN","Synaptics Inc"},
   {"SYP","SYPRO Co Ltd"},
   {"SYS","Sysgration Ltd"},
   {"SYT","Seyeon Tech Company Ltd"},
   {"SYV","SYVAX Inc"},
   {"SYX","Prime Systems, Inc."},
   {"TAA","Tandberg"},
   {"TAB","Todos Data System AB"},
   {"TAG","Teles AG"},
   {"TAI","Toshiba America Info Systems Inc"},
   {"TAM","Tamura Seisakusyo Ltd"},
   {"TAS","Taskit Rechnertechnik GmbH"},
   {"TAT","Teleliaison Inc"},
   {"TAX","Taxan   {Europe} Ltd"},
   {"TBB","Triple S Engineering Inc"},
   {"TBC","Turbo Communication, Inc"},
   {"TBS","Turtle Beach System"},
   {"TCC","Tandon Corporation"},
   {"TCD","Taicom Data Systems Co., Ltd."},
   {"TCE","Century Corporation"},
   {"TCH","Interaction Systems, Inc"},
   {"TCI","Tulip Computers Int'l B.V."},
   {"TCJ","TEAC America Inc"},
   {"TCL","Technical Concepts Ltd"},
   {"TCM","3Com Corporation"},
   {"TCN","Tecnetics   {PTY} Ltd"},
   {"TCO","Thomas-Conrad Corporation"},
   {"TCR","Thomson Consumer Electronics"},
   {"TCS","Tatung Company of America Inc"},
   {"TCT","Telecom Technology Centre Co. Ltd."},
   {"TCX","FREEMARS Heavy Industries"},
   {"TDC","Teradici"},
   {"TDD","Tandberg Data Display AS"},
   {"TDK","TDK USA Corporation"},
   {"TDM","Tandem Computer Europe Inc"},
   {"TDP","3D Perception"},
   {"TDS","Tri-Data Systems Inc"},
   {"TDT","TDT"},
   {"TDV","TDVision Systems, Inc."},
   {"TDY","Tandy Electronics"},
   {"TEA","TEAC System Corporation"},
   {"TEC","Tecmar Inc"},
   {"TEK","Tektronix Inc"},
   {"TEL","Promotion and Display Technology Ltd."},
   {"TER","TerraTec Electronic GmbH"},
   {"TGC","Toshiba Global Commerce Solutions, Inc."},
   {"TGI","TriGem Computer Inc"},
   {"TGM","TriGem Computer,Inc."},
   {"TGS","Torus Systems Ltd"},
   {"TGV","Grass Valley Germany GmbH"},
   {"THN","Thundercom Holdings Sdn. Bhd."},
   {"TIC","Trigem KinfoComm"},
   {"TIP","TIPTEL AG"},
   {"TIV","OOO Technoinvest"},
   {"TIX","Tixi.Com GmbH"},
   {"TKC","Taiko Electric Works.LTD"},
   {"TKN","Teknor Microsystem Inc"},
   {"TKO","TouchKo, Inc."},
   {"TKS","TimeKeeping Systems, Inc."},
   {"TLA","Ferrari Electronic GmbH"},
   {"TLD","Telindus"},
   {"TLF","Teleforce.,co,ltd"},
   {"TLI","TOSHIBA TELI CORPORATION"},
   {"TLK","Telelink AG"},
   {"TLS","Teleste Educational OY"},
   {"TLT","Dai Telecom S.p.A."},
   {"TLV","S3 Inc"},
   {"TLX","Telxon Corporation"},
   {"TMC","Techmedia Computer Systems Corporation"},
   {"TME","AT&T Microelectronics"},
   {"TMI","Texas Microsystem"},
   {"TMM","Time Management, Inc."},
   {"TMR","Taicom International Inc"},
   {"TMS","Trident Microsystems Ltd"},
   {"TMT","T-Metrics Inc."},
   {"TMX","Thermotrex Corporation"},
   {"TNC","TNC Industrial Company Ltd"},
   {"TNM","TECNIMAGEN SA"},
   {"TNY","Tennyson Tech Pty Ltd"},
   {"TOE","TOEI Electronics Co., Ltd."},
   {"TOG","The OPEN Group"},
   {"TON","TONNA"},
   {"TOP","Orion Communications Co., Ltd."},
   {"TOS","Toshiba Corporation"},
   {"TOU","Touchstone Technology"},
   {"TPC","Touch Panel Systems Corporation"},
   {"TPE","Technology Power Enterprises Inc"},
   {"TPJ","Junnila"},
   {"TPK","TOPRE CORPORATION"},
   {"TPR","Topro Technology Inc"},
   {"TPS","Teleprocessing Systeme GmbH"},
   {"TPT","Thruput Ltd"},
   {"TPV","Top Victory Electronics   { Fujian } Company Ltd"},
   {"TPZ","Ypoaz Systems Inc"},
   {"TRA","TriTech Microelectronics International"},
   {"TRC","Trioc AB"},
   {"TRD","Trident Microsystem Inc"},
   {"TRE","Tremetrics"},
   {"TRI","Tricord Systems"},
   {"TRL","Royal Information"},
   {"TRM","Tekram Technology Company Ltd"},
   {"TRN","Datacommunicatie Tron B.V."},
   {"TRS","Torus Systems Ltd"},
   {"TRT","Tritec Electronic AG"},
   {"TRU","Aashima Technology B.V."},
   {"TRV","Trivisio Prototyping GmbH"},
   {"TRX","Trex Enterprises"},
   {"TSB","Toshiba America Info Systems Inc"},
   {"TSC","Sanyo Electric Company Ltd"},
   {"TSD","TechniSat Digital GmbH"},
   {"TSE","Tottori Sanyo Electric"},
   {"TSF","Racal-Airtech Software Forge Ltd"},
   {"TSG","The Software Group Ltd"},
   {"TSI","TeleVideo Systems"},
   {"TSL","Tottori SANYO Electric Co., Ltd."},
   {"TSP","U.S. Navy"},
   {"TST","Transtream Inc"},
   {"TSV","TRANSVIDEO"},
   {"TSY","TouchSystems"},
   {"TTA","Topson Technology Co., Ltd."},
   {"TTB","National Semiconductor Japan Ltd"},
   {"TTC","Telecommunications Techniques Corporation"},
   {"TTE","TTE, Inc."},
   {"TTI","Trenton Terminals Inc"},
   {"TTK","Totoku Electric Company Ltd"},
   {"TTL","2-Tel B.V."},
   {"TTS","TechnoTrend Systemtechnik GmbH"},
   {"TTY","TRIDELITY Display Solutions GmbH"},
   {"TUA ","T+A elektroakustik GmbH"},
   {"TUT","Tut Systems"},
   {"TVD","Tecnovision"},
   {"TVI","Truevision"},
   {"TVM","Taiwan Video & Monitor Corporation"},
   {"TVO","TV One Ltd"},
   {"TVR","TV Interactive Corporation"},
   {"TVS","TVS Electronics Limited"},
   {"TVV","TV1 GmbH"},
   {"TWA","Tidewater Association"},
   {"TWE","Kontron Electronik"},
   {"TWH","Twinhead International Corporation"},
   {"TWI","Easytel oy"},
   {"TWK","TOWITOKO electronics GmbH"},
   {"TWX","TEKWorx Limited"},
   {"TXL","Trixel Ltd"},
   {"TXN","Texas Insturments"},
   {"TXT","Textron Defense System"},
   {"TYN","Tyan Computer Corporation"},
   {"UAS","Ultima Associates Pte Ltd"},
   {"UBI","Ungermann-Bass Inc"},
   {"UBL","Ubinetics Ltd."},
   {"UDN","Uniden Corporation"},
   {"UEC","Ultima Electronics Corporation"},
   {"UEG","Elitegroup Computer Systems Company Ltd"},
   {"UEI","Universal Electronics Inc"},
   {"UET","Universal Empowering Technologies"},
   {"UFG","UNIGRAF-USA"},
   {"UFO","UFO Systems Inc"},
   {"UHB","XOCECO"},
   {"UIC","Uniform Industrial Corporation"},
   {"UJR","Ueda Japan Radio Co., Ltd."},
   {"ULT","Ultra Network Tech"},
   {"UMC","United Microelectr Corporation"},
   {"UMG","Umezawa Giken Co.,Ltd"},
   {"UMM","Universal Multimedia"},
   {"UNA","Unisys DSD"},
   {"UNB","Unisys Corporation"},
   {"UNC","Unisys Corporation"},
   {"UNI","Uniform Industry Corp."},
   {"UNI","Unisys Corporation"},
   {"UNK","Unknown"},
   {"UNM","Unisys Corporation"},
   {"UNO","Unisys Corporation"},
   {"UNP","Unitop"},
   {"UNS","Unisys Corporation"},
   {"UNT","Unisys Corporation"},
   {"UNY","Unicate"},
   {"UPP","UPPI"},
   {"UPS","Systems Enhancement"},
   {"URD","Video Computer S.p.A."},
   {"USA","Utimaco Safeware AG"},
   {"USD","U.S. Digital Corporation"},
   {"USI","Universal Scientific Industrial Co., Ltd."},
   {"USR","U.S. Robotics Inc"},
   {"UTD","Up to Date Tech"},
   {"UWC","Uniwill Computer Corp."},
   {"VAD","Vaddio, LLC"},
   {"VAL","Valence Computing Corporation"},
   {"VAR","Varian Australia Pty Ltd"},
   {"VBR","VBrick Systems Inc."},
   {"VBT","Valley Board Ltda"},
   {"VCC","Virtual Computer Corporation"},
   {"VCI","VistaCom Inc"},
   {"VCJ","Victor Company of Japan, Limited"},
   {"VCM","Vector Magnetics, LLC"},
   {"VCX","VCONEX"},
   {"VDA","Victor Data Systems"},
   {"VDC","VDC Display Systems"},
   {"VDM","Vadem"},
   {"VDO","Video & Display Oriented Corporation"},
   {"VDS","Vidisys GmbH & Company"},
   {"VDT","Viditec, Inc."},
   {"VEC","Vector Informatik GmbH"},
   {"VEK","Vektrex"},
   {"VES","Vestel Elektronik Sanayi ve Ticaret A. S."},
   {"VFI","VeriFone Inc"},
   {"VHI","Macrocad Development Inc."},
   {"VIA","VIA Tech Inc"},
   {"VIB","Tatung UK Ltd"},
   {"VIC","Victron B.V."},
   {"VID","Ingram Macrotron Germany"},
   {"VIK","Viking Connectors"},
   {"VIM","Via Mons Ltd."},
   {"VIN","Vine Micros Ltd"},
   {"VIR","Visual Interface, Inc"},
   {"VIS","Visioneer"},
   {"VIT","Visitech AS"},
   {"VIZ","VIZIO, Inc"},
   {"VLB","ValleyBoard Ltda."},
   {"VLK","Vislink International Ltd"},
   {"VLT","VideoLan Technologies"},
   {"VMI","Vermont Microsystems"},
   {"VML","Vine Micros Limited"},
   {"VMW","VMware Inc.,"},
   {"VNC","Vinca Corporation"},
   {"VOB","MaxData Computer AG"},
   {"VPI","Video Products Inc"},
   {"VPR","Best Buy"},
   {"VQ@","Vision Quest"},
   {"VRC","Virtual Resources Corporation"},
   {"VSC","ViewSonic Corporation"},
   {"VSD","3M"},
   {"VSI","VideoServer"},
   {"VSN","Ingram Macrotron"},
   {"VSP","Vision Systems GmbH"},
   {"VSR","V-Star Electronics Inc."},
   {"VTC","VTel Corporation"},
   {"VTG","Voice Technologies Group Inc"},
   {"VTI","VLSI Tech Inc"},
   {"VTK","Viewteck Co., Ltd."},
   {"VTL","Vivid Technology Pte Ltd"},
   {"VTM","Miltope Corporation"},
   {"VTN","VIDEOTRON CORP."},
   {"VTS","VTech Computers Ltd"},
   {"VTV","VATIV Technologies"},
   {"VTX","Vestax Corporation"},
   {"VUT","Vutrix   {UK} Ltd"},
   {"VWB","Vweb Corp."},
   {"WAC","Wacom Tech"},
   {"WAL","Wave Access"},
   {"WAV","Wavephore"},
   {"WBN","MicroSoftWare"},
   {"WBS","WB Systemtechnik GmbH"},
   {"WCI","Wisecom Inc"},
   {"WCS","Woodwind Communications Systems Inc"},
   {"WDC","Western Digital"},
   {"WDE","Westinghouse Digital Electronics"},
   {"WEB","WebGear Inc"},
   {"WEC","Winbond Electronics Corporation"},
   {"WEL ","W-DEV"},
   {"WEY","WEY Design AG"},
   {"WHI","Whistle Communications"},
   {"WII","Innoware Inc"},
   {"WIL","WIPRO Information Technology Ltd"},
   {"WIN","Wintop Technology Inc"},
   {"WIP","Wipro Infotech"},
   {"WKH","Uni-Take Int'l Inc."},
   {"WLD","Wildfire Communications Inc"},
   {"WML","Wolfson Microelectronics Ltd"},
   {"WMO","Westermo Teleindustri AB"},
   {"WMT","Winmate Communication Inc"},
   {"WNI","WillNet Inc."},
   {"WNV","Winnov L.P."},
   {"WNX","Wincor Nixdorf International GmbH"},
   {"WPA","Matsushita Communication Industrial Co., Ltd."},
   {"WPI","Wearnes Peripherals International   {Pte} Ltd"},
   {"WRC","WiNRADiO Communications"},
   {"WSC","CIS Technology Inc"},
   {"WSP","Wireless And Smart Products Inc."},
   {"WST","Wistron Corporation"},
   {"WTC","ACC Microelectronics"},
   {"WTI","WorkStation Tech"},
   {"WTK","Wearnes Thakral Pte"},
   {"WTS","Restek Electric Company Ltd"},
   {"WVM","Wave Systems Corporation"},
   {"WVV","WolfVision GmbH"},
   {"WWV","World Wide Video, Inc."},
   {"WXT","Woxter Technology Co. Ltd"},
   {"WYS","Myse Technology"},
   {"WYT","Wooyoung Image & Information Co.,Ltd."},
   {"XAC","XAC Automation Corp"},
   {"XAD","Alpha Data"},
   {"XDM","XDM Ltd."},
   {"XFG","Jan Strapko - FOTO"},
   {"XFO","EXFO Electro Optical Engineering"},
   {"XIN","Xinex Networks Inc"},
   {"XIO","Xiotech Corporation"},
   {"XIR","Xirocm Inc"},
   {"XIT","Xitel Pty ltd"},
   {"XLX","Xilinx, Inc."},
   {"XMM","C3PO S.L."},
   {"XNT","XN Technologies, Inc."},
   {"XQU","SHANGHAI SVA-DAV ELECTRONICS CO., LTD"},
   {"XRC","Xircom Inc"},
   {"XRO","XORO ELECTRONICS   {CHENGDU} LIMITED"},
   {"XSN","Xscreen AS"},
   {"XST","XS Technologies Inc"},
   {"XSY","XSYS"},
   {"XTD","Icuiti Corporation"},
   {"XTE","X2E GmbH"},
   {"XTL","Crystal Computer"},
   {"XTN","X-10   {USA} Inc"},
   {"XYC","Xycotec Computer GmbH"},
   {"YED","Y-E Data Inc"},
   {"YHQ","Yokogawa Electric Corporation"},
   {"YHW","Exacom SA"},
   {"YMH","Yamaha Corporation"},
   {"YOW","American Biometric Company"},
   {"ZAN","Zandar Technologies plc"},
   {"ZAX","Zefiro Acoustics"},
   {"ZAZ","Zazzle Technologies"},
   {"ZBR","Zebra Technologies International, LLC"},
   {"ZCT","ZeitControl cardsystems GmbH"},
   {"ZDS","Zenith Data Systems"},
   {"ZGT","Zenith Data Systems"},
   {"ZIC","Nationz Technologies Inc."},
   {"ZMT","Zalman Tech Co., Ltd."},
   {"ZMZ","Z Microsystems"},
   {"ZNI","Zetinet Inc"},
   {"ZNX","Znyx Adv. Systems"},
   {"ZOW","Zowie Intertainment, Inc"},
   {"ZRN","Zoran Corporation"},
   {"ZSE","Zenith Data Systems"},
   {"ZTC","ZyDAS Technology Corporation"},
   {"ZTE","ZTE Corporation"},
   {"ZTI","Zoom Telephonics Inc"},
   {"ZTM","ZT Group Int'l Inc."},
   {"ZTT","Z3 Technology"},
   {"ZYD","Zydacron Inc"},
   {"ZYP","Zypcom Inc"},
   {"ZYT","Zytex Computers"},
   {"ZYX","Zyxel"},
   {"ZZZ","Boca Research Inc"}

};
const int pnp_table_size = sizeof(pnp_id_table) / sizeof(Pnp_Id_Table_Entry); // ARRAY_SIZE(pnp_id_table);


static char * pnp_name0(char * id, int first, int last) {
   bool debug = false;
   if (debug)
      printf("(%s) Starting, id=%s, first=%d, last=%d\n",  __func__, id, first, last);

   char * result = "UNK";
   if (first <= last && first >= 0 && last < pnp_table_size) {
      if (debug)
         printf("(%s) mfg_code[%d] = %s, mfg_code[%d] = %s\n",
               __func__, first, pnp_id_table[first].mfg_name, last, pnp_id_table[last].mfg_name);
      int middle = (last-first)/2 + first;
      int cmprc = strcmp(id,pnp_id_table[middle].mfg_code);
      if (cmprc == 0)
            result = pnp_id_table[middle].mfg_name;
      else if (cmprc < 0)
         result = pnp_name0(id, first, middle-1);
      else
         result = pnp_name0(id, middle+1, last);
   }

   if (debug)
      printf("(%s) Done.    Returning: %s\n", __func__, result);
   return result;
}


/** Returns the name associated with a 3 character manufacturer id.
 *
 * \param id manufacturer id
 * \param manufacturer name, or "UNK" if not found
 */
char * pnp_name(char * id) {
   char * result = NULL;
   strupper(id);      // "inu" is a lower case code
   result = pnp_name0(id, 0, pnp_table_size-1);
   return result;
}


#ifdef TESTS
static void one_pnp_test(char * id, char * expected) {
   printf("(%s) pnp_table_size=%d\n", __func__, pnp_table_size);
   printf("(%s) Starting id = %s\n", __func__, id);
   char * name = pnp_name(id);
   char * expect = (expected) ? expected :  "UNK";
   printf("(%s) Done     Expected: %s,   Returning: %s\n", __func__, expect, name);
}


void pnp_id_tests() {
   one_pnp_test("123", NULL);
   one_pnp_test("abc", NULL);
   one_pnp_test("AAA", "Avolites Ltd");
   one_pnp_test("ZZZ", "Boca Research Inc");
   one_pnp_test("ZYZ", NULL);   // no existant
   one_pnp_test("ZYX", "Zyxel");   // second last
   one_pnp_test("AAB", NULL);   // not exist
   one_pnp_test("AAE", "Anatek Electronics Inc.");
   one_pnp_test("VCX", "VCONEX");
   one_pnp_test("UFH", NULL);      // does not exist
}
#endif
