/* $Cambridge: hermes/src/prayer/session/favourite.c,v 1.2 2008/05/19 15:55:58 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "prayer_session.h"

/* Class to support favourite list of mail folders. Possible that we
 * should use subscribed list from IMAP server instead? */

/* favourite_list_create() ***********************************************
 *
 * Create a fresh favorite list structure. (Uses NIL pool: code will
 * do its own memory allocation...)
 ************************************************************************/

struct favourite_list *favourite_list_create(void)
{
    struct favourite_list *fl;

    fl = pool_alloc(NIL, sizeof(struct favourite_list));
    fl->list = list_create(NIL, T);
    fl->preferred = NIL;

    return (fl);
}

/* favourite_list_free() *************************************************
 *
 * Free favourite list, including each element of list.
 * 
 ************************************************************************/

void favourite_list_free(struct favourite_list *fl)
{
    list_free(fl->list);
    if (fl->preferred)
        free(fl->preferred);

    free(fl);
}

/* ====================================================================== */

/* favourite_preferred() *************************************************
 *
 * Set the preferred mail folder.
 ************************************************************************/

void favourite_preferred(struct favourite_list *fl, char *preferred)
{
    string_strdup(&fl->preferred, preferred);
}

/* favourite_add() *******************************************************
 *
 * Add to favourites list.
 *     fl: Favourite list
 *   name: Name to add (will be duplicated)
 ************************************************************************/

BOOL favourite_add(struct favourite_list *fl, char *name)
{
    if (list_lookup_byname(fl->list, name))
        return (NIL);

    list_push(fl->list, pool_alloc(NIL, sizeof(struct list_item)), name);
    return (T);
}

/* favourite_delete() ****************************************************
 *
 * Remove from favourites list.
 *     fl: Favourite list
 *   name: Name to remove (list_remove_byname() will free())
 ************************************************************************/

BOOL favourite_delete(struct favourite_list * fl, char *name)
{
    return (list_remove_byname(fl->list, name));
}

/* favourite_rename() ****************************************************
 *
 * Rename entry on favourite list
 *      fl: Favourite list
 * oldname: Original name
 * newname: Replacement name.
 ************************************************************************/

BOOL
favourite_rename(struct favourite_list * fl, char *oldname, char *newname)
{
    return (list_rename_item(fl->list, oldname, newname));
}

/* ====================================================================== */

/* Interface to user preferences file */

/* favourite_parse_line() ************************************************
 *
 * Parse a line from the user preferences file
 *      fl: Favourite List
 *    line: Line to parse
 * session: For logging purposes only
 ************************************************************************/

void
favourite_parse_line(struct favourite_list *fl, char *line,
                     struct session *session)
{
    struct config *config = session->config;

    if (config->strip_mail) {
        if (!strncmp(line, "__PREFERRED__ ", strlen("__PREFERRED__ "))) {
            char *s = line + strlen("__PREFERRED__ ");

            favourite_preferred(fl, s);
        } else {
            favourite_add(fl, line);
        }
    } else {
        if (!strncmp(line, "__PREFERRED__ ", strlen("__PREFERRED__ ")))
            favourite_preferred(fl, line + strlen("__PREFERRED__ "));
        else
            favourite_add(fl, line);
    }
}

/* favourite_print_options() *********************************************
 *
 * Convert favourites list to printable format suitable for prefs file
 *     fl: Favourite list
 *      b: Target buffer
 ************************************************************************/

void favourite_print_options(struct favourite_list *fl, struct buffer *b)
{
    struct list_item *li;

    if (fl->preferred)
        bprintf(b, "__PREFERRED__ %s" CRLF, fl->preferred);

    for (li = fl->list->head; li; li = li->next)
        bprintf(b, "%s" CRLF, li->name);
}

/* ====================================================================== */

/* favourite_template_vals() *********************************************
 *
 * Convert favourites list into tval data
 *
 ************************************************************************/

void
favourite_template_vals(struct favourite_list *fl,
                        struct template_vals *tvals,
                        char *array)
{
    struct list_item *li;
    unsigned long offset = 0;

    for (li = fl->list->head; li; li = li->next) {
        template_vals_foreach_init(tvals, array, offset);
        template_vals_foreach_string(tvals, array, offset,
                                     "name", li->name);

        if (offset % 2 == 0)
            template_vals_foreach_ulong(tvals, array, offset,
                                        "even_row", 1);
        offset++;
    }
}
