/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2000, 2010 Oracle and/or its affiliates.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

#ifndef INCLUDED_SLIDESHOW_ACTIVITYPARAMETERS_HXX
#define INCLUDED_SLIDESHOW_ACTIVITYPARAMETERS_HXX

#include "event.hxx"
#include "eventqueue.hxx"
#include "expressionnode.hxx"
#include "wakeupevent.hxx"

#include <boost/optional.hpp>
#include <vector>

namespace slideshow {
namespace internal {

/** Parameter struct for animation activities
    
    This struct contains all common parameters needed to
    initialize the activities generated by the ActivityFactory.
*/
struct ActivityParameters
{            
    /** Create 
        
        @param rEndEvent
        Event to be fired, when the activity ends.
        
        @param rEventQueue
        Queue to add end event to
        
        @param nMinDuration
        Minimal duration of the activity (might actually be
        longer because of nMinNumberOfFrames). Note that this
        duration must always be the <em>simple</em> duration,
        i.e. without any repeat.
        
        @param rRepeats
        Number of repeats. If this parameter is invalid,
        infinite repeat is assumed.
        
        @param nAccelerationFraction
        Value between 0 and 1, denoting the fraction of the
        total simple duration, which the animation should
        accelerate.
        
        @param nDecelerationFraction 
        Value between 0 and 1, denoting the fraction of the
        total simple duration, which the animation should
        decelerate. Note that the ranges
        [0,nAccelerationFraction] and
        [nDecelerationFraction,1] must be non-overlapping!
        
        @param bAutoReverse
        When true, at the end of the simple duration, the
        animation plays reversed to the start value. Note that
        nMinDuration still specifies the simple duration,
        i.e. when bAutoReverse is true, the implicit duration
        doubles.
    */
    ActivityParameters(
        const EventSharedPtr&                       rEndEvent,
        EventQueue&                                 rEventQueue,
        ActivitiesQueue&                            rActivitiesQueue,
        double                                      nMinDuration,
        ::boost::optional<double> const&            rRepeats,
        double                                      nAccelerationFraction,
        double                                      nDecelerationFraction,
        sal_uInt32                                  nMinNumberOfFrames,
        bool                                        bAutoReverse )
        : mrEndEvent( rEndEvent ),
          mpWakeupEvent(),
          mrEventQueue( rEventQueue ),
          mrActivitiesQueue( rActivitiesQueue ),
          mpFormula(),
          maDiscreteTimes(),
          mnMinDuration( nMinDuration ),
          mrRepeats( rRepeats ),
          mnAccelerationFraction( nAccelerationFraction ),
          mnDecelerationFraction( nDecelerationFraction ),
          mnMinNumberOfFrames( nMinNumberOfFrames ),
          mbAutoReverse( bAutoReverse ) {}
    
    /// End event to fire, when activity is over
    const EventSharedPtr&                       mrEndEvent;
    /// Wakeup event to use for discrete activities
    WakeupEventSharedPtr                        mpWakeupEvent;
    
    /// EventQueue to add events to
    EventQueue&                                 mrEventQueue;
    
    /// ActivitiesQueue to add events to
    ActivitiesQueue&                            mrActivitiesQueue;
    
    /// Optional formula
    ExpressionNodeSharedPtr                     mpFormula;
    
    /// Key times, for discrete and key time activities
    ::std::vector< double >                     maDiscreteTimes;
    
    /// Total duration of activity (including all repeats)
    const double                                mnMinDuration;
    ::boost::optional<double> const&            mrRepeats;
    const double                                mnAccelerationFraction;
    const double                                mnDecelerationFraction;
    
    /// Minimal number of frames this activity must render
    const sal_uInt32                            mnMinNumberOfFrames;
    
    /// When true, activity is played reversed after mnDuration.
    const bool                                  mbAutoReverse;
};

} // namespace internal
} // namespace presentation

#endif /* INCLUDED_SLIDESHOW_ACTIVITYPARAMETERS_HXX */

