/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: OOXMLContext.cxx,v $
 *
 *  $Revision: 1.39 $
 *
 *  last change: $Author: vg $ $Date: 2007/10/29 15:29:52 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <iostream>
#include <ooxml/resourceids.hxx>
#include "OOXMLContext.hxx"
#include "OOXMLPropertySetImpl.hxx"
#include "TokenMap.hxx"

namespace ooxml
{

using namespace ::std;

static const sal_uInt8 s0x7[] = { 0x7, 0x0 };
static const sal_uInt8 s0xd[] = { 0xd, 0x0 };

const rtl::OUString sEmpty;

static string resourceToString(OOXMLContext::ResourceEnum_t eResource)
{
    string sResult;

    switch (eResource)
    {
    case OOXMLContext::STREAM:
        sResult = "Stream";
        break;
    case OOXMLContext::PROPERTIES:
        sResult = "Properties";
        break;
    case OOXMLContext::TABLE:
        sResult = "Table";
        break;
    default:
        sResult = "??";
    }

    return sResult;
}

/*
  class OOXMLParserState
*/

OOXMLParserState::OOXMLParserState()
:  mbInSectionGroup(false), 
   mbInParagraphGroup(false), 
   mbInCharacterGroup(false),
   mbLastParagraphInSection(false), 
   mbForwardEvents(true),
   mnContexts(0),
   mpDocument(NULL)
{
}

OOXMLParserState::~OOXMLParserState()
{
}

void OOXMLParserState::setLastParagraphInSection(bool bLastParagraphInSection)
{
    mbLastParagraphInSection = bLastParagraphInSection;
}

bool OOXMLParserState::isLastParagraphInSection() const
{
    return mbLastParagraphInSection;
}

void OOXMLParserState::setInSectionGroup(bool bInSectionGroup)
{
    mbInSectionGroup = bInSectionGroup;
}

bool OOXMLParserState::isInSectionGroup() const
{
    return mbInSectionGroup;
}

void OOXMLParserState::setInParagraphGroup(bool bInParagaphGroup)
{
    mbInParagraphGroup = bInParagaphGroup;
}

bool OOXMLParserState::isInParagraphGroup() const
{
    return mbInParagraphGroup;
}

void OOXMLParserState::setInCharacterGroup(bool bInCharacterGroup)
{
    mbInCharacterGroup = bInCharacterGroup;
}

bool OOXMLParserState::isInCharacterGroup() const
{
    return mbInCharacterGroup;
}

void OOXMLParserState::setForwardEvents(bool bForwardEvents)
{
    mbForwardEvents = bForwardEvents;
}

bool OOXMLParserState::isForwardEvents() const
{
    return mbForwardEvents;
}

void OOXMLParserState::incContextCount()
{
    mnContexts++;
}

const string OOXMLParserState::getHandle() const
{
    char sBuffer[256];

    snprintf(sBuffer, sizeof(sBuffer), "%d", mnContexts);

    return sBuffer;
}

unsigned int OOXMLParserState::getContextCount() const
{
    return mnContexts;
}

void OOXMLParserState::setDocument(OOXMLDocument * pDocument)
{
    mpDocument = pDocument;
}

OOXMLDocument * OOXMLParserState::getDocument() const
{
    return mpDocument;
}

void OOXMLParserState::setXNoteId(const rtl::OUString & rId)
{
    msXNoteId = rId;
}

const rtl::OUString & OOXMLParserState::getXNoteId() const
{
    return msXNoteId;
}

string OOXMLParserState::toString() const
{
    string sResult = "("; 

    if (isInSectionGroup())
        sResult += "s";
    else
        sResult += "-";

    if (isInParagraphGroup())
        sResult += "p";
    else
        sResult += "-";

    if (isInCharacterGroup())
        sResult += "c";
    else
        sResult += "-";
    
    if (isForwardEvents())
        sResult += "f";
    else
        sResult += "-";

    sResult += ", \"";
    sResult += OUStringToOString(getXNoteId(), 
                                 RTL_TEXTENCODING_ASCII_US).getStr();

    sResult += "\")";

    return sResult;
}

/*
  class OOXMLContext
*/


OOXMLContext::OOXMLContext(Stream & rStream)
: mrStream(rStream), 
  mNamespaceMapper(),
  mnTableDepth(0), 
  meToken(OOXML_TOKENS_END),
  mpParserState(new OOXMLParserState())
{
    mpParserState->incContextCount();
}

OOXMLContext::OOXMLContext(const OOXMLContext & rContext)
: mrStream(rContext.mrStream), 
  mNamespaceMapper(rContext.mNamespaceMapper), 
  mnTableDepth(rContext.mnTableDepth), 
  meToken(OOXML_TOKENS_END),
  mpParserState(rContext.mpParserState)
{
    mpParserState->incContextCount();
}

OOXMLContext::~OOXMLContext()
{
}

TokenEnum_t OOXMLContext::getToken(TokenMap::TokenType_t nType,
                                   const rtl::OUString & rTokenString)
{
    return mNamespaceMapper.getToken(nType, rTokenString);
}

void OOXMLContext::addNamespaceMapping(const rtl::OUString & rPrefix,
                                       const rtl::OUString & rUrl)
{
    mNamespaceMapper.addMapping(rPrefix, rUrl);
}

OOXMLContext::Pointer_t OOXMLContext::element(TokenEnum_t nToken)
{
    return elementFromRefs(nToken);
}

void OOXMLContext::endElement(TokenEnum_t /* nToken */,
                              OOXMLContext::Pointer_t /* pContext */)
{
}

bool OOXMLContext::lcl_attribute(TokenEnum_t /*nToken*/, 
                                 const rtl::OUString & /*rStr*/)
{
    return true;
}

bool OOXMLContext::attribute(TokenEnum_t nToken, const rtl::OUString & rStr)
{
    bool bResult = true;

    bResult = lcl_attribute(nToken, rStr);

#ifdef DEBUG_ATTRIBUTES
    clog << "OOXMLContext::attribute: name: " 
         << OUStringToOString
        (TokenMap::getInstance()->getTokenString(TokenMap::ATTRIBUTE, nToken),
         RTL_TEXTENCODING_ASCII_US).getStr()
         << ", value: " 
         << OUStringToOString(rStr, RTL_TEXTENCODING_ASCII_US).getStr()
         << ", return: " << bResult << endl;
#endif

    return bResult;
}

OOXMLValue::Pointer_t OOXMLContext::getValue() const
{
    return OOXMLValue::Pointer_t();
}

void OOXMLContext::lcl_characters(const rtl::OUString & /*rStr*/)
{
}

void OOXMLContext::characters(const rtl::OUString & rStr)     
{
    if (isForwardEvents())
        lcl_characters(rStr);
}

doctok::Id OOXMLContext::getId(TokenEnum_t nToken)
{
    return getIdFromRefs(nToken);
}

void OOXMLContext::processToken(TokenEnum_t /* nToken */)
{
}

string OOXMLContext::toString() const
{
    string sResult = "(";
    sResult += getType();
    sResult += ", ";
    sResult += getResourceString();
    sResult += ",";
    
    OOXMLValue::Pointer_t pVal(getValue());

    if (pVal.get() != NULL)
        sResult += pVal->toString();
    else
        sResult += "(null)";

    sResult += mpParserState->toString();

    sResult += ")";

    return sResult;
}

string OOXMLContext::getResourceString() const
{
    return resourceToString(getResource());
}

void OOXMLContext::setParentResource(ResourceEnum_t /*eResource*/)
{
}

void OOXMLContext::startAction()
{
}

void OOXMLContext::endAction()
{
}

void OOXMLContext::sendTableDepth() const
{
    if (mnTableDepth > 0)
    {
        OOXMLPropertySet * pProps = new OOXMLPropertySetImpl();
        {
            OOXMLValue::Pointer_t pVal
                (new OOXMLIntegerValue(mnTableDepth));
            OOXMLProperty::Pointer_t pProp
                (new OOXMLPropertyImpl(0x6649, pVal, OOXMLPropertyImpl::SPRM));
            pProps->add(pProp);
        }
        {
            OOXMLValue::Pointer_t pVal
                (new OOXMLIntegerValue(1));
            OOXMLProperty::Pointer_t pProp
                (new OOXMLPropertyImpl(0x2416, pVal, OOXMLPropertyImpl::SPRM));
            pProps->add(pProp);
        }
        
        mrStream.props(doctok::Reference<Properties>::Pointer_t(pProps));
    }
}

void OOXMLContext::startCharacterGroup()
{
    if (isForwardEvents() && ! mpParserState->isInCharacterGroup())
    {
        mrStream.startCharacterGroup();    
        mpParserState->setInCharacterGroup(true);
    }
}

void OOXMLContext::endCharacterGroup()
{
    if (mpParserState->isInCharacterGroup() && isForwardEvents())
    {
        mrStream.endCharacterGroup();
        mpParserState->setInCharacterGroup(false);
    }
}

void OOXMLContext::startParagraphGroup()
{
    endCharacterGroup();

    if (isForwardEvents() && ! mpParserState->isInParagraphGroup())
    {
        mrStream.info(mpParserState->getHandle());
        mrStream.startParagraphGroup();
        mpParserState->setInParagraphGroup(true);    
    }

    if (mpParserState->isLastParagraphInSection())
    {
        startSectionGroup();
        mpParserState->setLastParagraphInSection(false);
    }
}

void OOXMLContext::endParagraphGroup()
{
    endCharacterGroup();

    if (mpParserState->isInParagraphGroup() && isForwardEvents())
    {
        mrStream.endParagraphGroup();    
        mpParserState->setInParagraphGroup(false);
    }
/*    
    if (mpParserState->isLastParagraphInSection())
        endSectionGroup(); */
}

void OOXMLContext::startSectionGroup()
{
    endParagraphGroup();

    if (isForwardEvents() && ! mpParserState->isInSectionGroup())
    {
        mrStream.info(mpParserState->getHandle());
        mrStream.startSectionGroup();
        mpParserState->setInSectionGroup(true);
    }
}

void OOXMLContext::endSectionGroup()
{
    endParagraphGroup();

    if (mpParserState->isInSectionGroup() && isForwardEvents())
    {
        mrStream.endSectionGroup();    
        mpParserState->setInSectionGroup(false);
    }
}

void OOXMLContext::setLastParagraphInSection()
{
    mpParserState->setLastParagraphInSection(true);
}

void OOXMLContext::setDocument(OOXMLDocument * pDocument)
{
    mpParserState->setDocument(pDocument);
}

void OOXMLContext::setForwardEvents(bool bForwardEvents)
{
    mpParserState->setForwardEvents(bForwardEvents);
}

bool OOXMLContext::isForwardEvents() const
{
    return mpParserState->isForwardEvents();
}

void OOXMLContext::setXNoteId(const rtl::OUString & rId)
{
    mpParserState->setXNoteId(rId);
}

const rtl::OUString & OOXMLContext::getXNoteId() const
{
    return mpParserState->getXNoteId();
}

void OOXMLContext::resolveFootnote(const rtl::OUString & rId)
{
    mpParserState->getDocument()->resolveFootnote
        (mrStream, rId);
}

void OOXMLContext::resolveEndnote(const rtl::OUString & rId)
{
    mpParserState->getDocument()->resolveEndnote
        (mrStream, rId);
}

void OOXMLContext::resolveComment(const rtl::OUString & rId)
{
    mpParserState->getDocument()->resolveComment(mrStream, rId);
}

void OOXMLContext::resolveHeader(const sal_Int32 type, const rtl::OUString & rId)
{
    mpParserState->getDocument()->resolveHeader(mrStream, type, rId);
}

void OOXMLContext::resolveFooter(const sal_Int32 type, const rtl::OUString & rId)
{
    mpParserState->getDocument()->resolveFooter(mrStream, type, rId);
}

void OOXMLContext::setToken(TokenEnum_t eToken)
{
    meToken = eToken;
}

void OOXMLContext::resolvePropertySetAttrs()
{
}

/*
  class OOXMLConstextStream
 */

OOXMLContextStream::OOXMLContextStream(Stream & rStream)
: OOXMLContext(rStream),
  mpPropertySetAttrs(new OOXMLPropertySetImpl())
{
}

OOXMLContextStream::OOXMLContextStream(const OOXMLContext & rContext)
: OOXMLContext(rContext),
  mpPropertySetAttrs(new OOXMLPropertySetImpl())
{
}

OOXMLContextStream::~OOXMLContextStream()
{
}

void OOXMLContextStream::newProperty(TokenEnum_t nToken, 
                                         OOXMLValue::Pointer_t pVal)
{
    doctok::Id nId = this->getId(nToken);
    
    OOXMLPropertyImpl::Pointer_t pProperty
        (new OOXMLPropertyImpl(nId, pVal, OOXMLPropertyImpl::ATTRIBUTE));
    mpPropertySetAttrs->add(pProperty);    
}

void OOXMLContextStream::lcl_characters(const rtl::OUString & sText) 
{    
    mrStream.utext(reinterpret_cast < const sal_uInt8 * > 
                   (sText.getStr()), 
                   sText.getLength());
}

void OOXMLContextStream::setPropertySetAttrs
(OOXMLPropertySet::Pointer_t pPropertySetAttrs)
{
    mpPropertySetAttrs = pPropertySetAttrs;
}

OOXMLPropertySet::Pointer_t OOXMLContextStream::getPropertySetAttrs() const
{
    return mpPropertySetAttrs;
}

void OOXMLContextStream::resolvePropertySetAttrs()
{
    doctok::Reference<Properties>::Pointer_t pProps(mpPropertySetAttrs->clone());
    mrStream.props(pProps);
}

/*
  class OOXMLContextProperties
 */
OOXMLContextProperties::OOXMLContextProperties(Stream & rStream)
: OOXMLContext(rStream), mbResolve(false)
{    
}

OOXMLContextProperties::OOXMLContextProperties(const OOXMLContext & rContext)
: OOXMLContext(rContext), mpPropertySet(new OOXMLPropertySetImpl()), 
  mbResolve(false)
{
}

OOXMLContextProperties::~OOXMLContextProperties()
{
}

OOXMLValue::Pointer_t OOXMLContextProperties::getValue() const
{
    return OOXMLValue::Pointer_t(new OOXMLPropertySetValue(mpPropertySet));
}

void OOXMLContextProperties::processToken(TokenEnum_t /* nToken */)
{    
    if (mbResolve)
    {
        doctok::Reference<Properties>::Pointer_t pProps(mpPropertySet->clone());
        if (isForwardEvents())
            mrStream.props(pProps);
    }
}

void OOXMLContextProperties::endElement(TokenEnum_t nToken,
                                        OOXMLContext::Pointer_t pContext)
{
    const doctok::Id nId = this->getId(nToken);

    OOXMLPropertyImpl::Pointer_t pProperty
     (new OOXMLPropertyImpl(nId, pContext->getValue(), 
                            OOXMLPropertyImpl::SPRM));
    mpPropertySet->add(pProperty);
}

void OOXMLContextProperties::newProperty(TokenEnum_t nToken, 
                                         OOXMLValue::Pointer_t pVal)
{
    doctok::Id nId = this->getId(nToken);
    
    OOXMLPropertyImpl::Pointer_t pProperty
        (new OOXMLPropertyImpl(nId, pVal, OOXMLPropertyImpl::ATTRIBUTE));
    mpPropertySet->add(pProperty);    
}

void OOXMLContextProperties::setPropertySet(OOXMLPropertySet::Pointer_t 
                                            pPropertySet)
{
    mpPropertySet = pPropertySet;
}

string OOXMLContextProperties::toString() const
{
    string sResult = "(";
    sResult += getType();
    sResult += ", ";
    sResult += getResourceString();
    sResult += ", ";
    sResult += mbResolve ? "resolve" : "noResolve";
    sResult += ")";

    return sResult;
}

void OOXMLContextProperties::setParentResource(ResourceEnum_t eResource)
{
    if (eResource == STREAM)
        mbResolve = true;
}

/*
  class OOXMLContextProperty
*/
OOXMLContextProperty::OOXMLContextProperty(Stream & rStream)
: OOXMLContext(rStream)
{
}

OOXMLContextProperty::OOXMLContextProperty(const OOXMLContext & rContext)
: OOXMLContext(rContext)
{
}

OOXMLContextProperty::~OOXMLContextProperty()
{
}

/*
  class OOXMLContextSingleElement
*/
OOXMLContextSingleElement::OOXMLContextSingleElement(Stream & rStream)
: OOXMLContext(rStream), mpPropertySet(new OOXMLPropertySetImpl())
{
}

OOXMLContextSingleElement::OOXMLContextSingleElement(const OOXMLContext & rContext)
: OOXMLContext(rContext), mpPropertySet(new OOXMLPropertySetImpl())
{
}

OOXMLContextSingleElement::~OOXMLContextSingleElement()
{
}

void OOXMLContextSingleElement::newProperty(TokenEnum_t nToken, 
                                               OOXMLValue::Pointer_t pVal)
{
    doctok::Id nId = this->getId(nToken);
    
    OOXMLPropertyImpl::Pointer_t pProperty
        (new OOXMLPropertyImpl(nId, pVal, OOXMLPropertyImpl::ATTRIBUTE));
    mpPropertySet->add(pProperty);    
}

void OOXMLContextSingleElement::processToken(TokenEnum_t nToken)
{
    rtl::OUString tempChar;
    switch (nToken)
    {
    case OOXML_ELEMENT_wordprocessingml_noBreakHyphen:
        tempChar = rtl::OUString(0x001E);
        break;
    case OOXML_ELEMENT_wordprocessingml_softHyphen:
        tempChar = rtl::OUString(0x001F);
        break;
    case OOXML_ELEMENT_wordprocessingml_cr:
        tempChar = rtl::OUString(0x000D);
        break;
    case OOXML_ELEMENT_wordprocessingml_tab:
        tempChar = rtl::OUString(0x0009);
        break;
    case OOXML_ELEMENT_wordprocessingml_br:
    	{
            OOXMLBreakHandler aBreakHandler(mrStream, this);
            mpPropertySet->resolve(aBreakHandler);
        }
        break;
    default:
        break;
    }
    
    if (tempChar.getLength() && isForwardEvents())
            mrStream.utext(reinterpret_cast < const sal_uInt8 * >(tempChar.getStr()), tempChar.getLength());
}

/*
  class OOXMLContextBooleanValue
 */

OOXMLContextBooleanValue::OOXMLContextBooleanValue
(const OOXMLContextBooleanValue & rSrc)
: OOXMLContext(rSrc), mbValue(true)
{
}

OOXMLContextBooleanValue::OOXMLContextBooleanValue(Stream & rStream)
: OOXMLContext(rStream), mbValue(true)
{
}

OOXMLContextBooleanValue::OOXMLContextBooleanValue
(const OOXMLContext & rContext)
: OOXMLContext(rContext), mbValue(true)
{
}

OOXMLContextBooleanValue::~OOXMLContextBooleanValue()
{
}

bool OOXMLContextBooleanValue::lcl_attribute(TokenEnum_t /*nToken*/, 
                                             const ::rtl::OUString & /*rStr*/)
{
    return false;
}

OOXMLValue::Pointer_t OOXMLContextBooleanValue::getValue() const
{
    return OOXMLValue::Pointer_t(new OOXMLBooleanValue(mbValue));
}

/*
  class OOXMLContextIntegerValue
 */

OOXMLContextIntegerValue::OOXMLContextIntegerValue
(const OOXMLContext & rSrc)
: OOXMLContext(rSrc), mnValue(0)
{
}

OOXMLContextIntegerValue::OOXMLContextIntegerValue
(Stream & rStream)
: OOXMLContext(rStream), mnValue(0)
{
}

OOXMLContextIntegerValue::~OOXMLContextIntegerValue()
{
}

bool OOXMLContextIntegerValue::lcl_attribute(TokenEnum_t /*nToken*/, 
                                             const ::rtl::OUString & /*rStr*/)
{
    return false;
}

OOXMLValue::Pointer_t OOXMLContextIntegerValue::getValue() const
{
    return OOXMLValue::Pointer_t(new OOXMLIntegerValue(mnValue));
}

/*
  class OOXMLContextHexValue
 */

OOXMLContextHexValue::OOXMLContextHexValue
(const OOXMLContext & rSrc)
: OOXMLContext(rSrc), mnValue(0)
{
}

OOXMLContextHexValue::OOXMLContextHexValue
(Stream & rStream)
: OOXMLContext(rStream), mnValue(0)
{
}

OOXMLContextHexValue::~OOXMLContextHexValue()
{
}

bool OOXMLContextHexValue::lcl_attribute(TokenEnum_t /*nToken*/, 
                                         const ::rtl::OUString & /*rStr*/)
{
    return false;
}

OOXMLValue::Pointer_t OOXMLContextHexValue::getValue() const
{
    return OOXMLValue::Pointer_t(new OOXMLIntegerValue(mnValue));
}

/*
  class OOXMLContextListValue
*/

OOXMLContextListValue::OOXMLContextListValue(const OOXMLContext & rSrc)
: OOXMLContext(rSrc), mnValue(0)
{
}

OOXMLContextListValue::OOXMLContextListValue(Stream & rStream)
: OOXMLContext(rStream), mnValue(0)
{
}

OOXMLContextListValue::~OOXMLContextListValue()
{
}

bool OOXMLContextListValue::lcl_attribute(TokenEnum_t /*nToken*/, 
                                          const ::rtl::OUString & /*rString*/)
{
    return false;
}

OOXMLValue::Pointer_t OOXMLContextListValue::getValue() const
{
    return OOXMLValue::Pointer_t(new OOXMLListValue(mnValue));
}

/*
  class OOXMLContextStringValue
*/

OOXMLContextStringValue::OOXMLContextStringValue(const OOXMLContext & rSrc)
: OOXMLContext(rSrc), msValue()
{
}

OOXMLContextStringValue::OOXMLContextStringValue(Stream & rStream)
: OOXMLContext(rStream), msValue()
{
}

OOXMLContextStringValue::~OOXMLContextStringValue()
{
}

bool OOXMLContextStringValue::lcl_attribute(TokenEnum_t /*nToken*/, 
                                            const ::rtl::OUString & /*rString*/)
{
    return false;
}

OOXMLValue::Pointer_t OOXMLContextStringValue::getValue() const
{
    return OOXMLValue::Pointer_t(new OOXMLStringValue(msValue));
}

/*
  class OOXMLContextTable
*/

OOXMLContextTable::OOXMLContextTable(Stream & rStream)
: OOXMLContext(rStream)
{
}

OOXMLContextTable::OOXMLContextTable(const OOXMLContext & rContext)
: OOXMLContext(rContext)
{
}

OOXMLContextTable::~OOXMLContextTable()
{
    doctok::Reference<Table>::Pointer_t pTable(mTable.clone());
    if (isForwardEvents())
        mrStream.table(mId, pTable);
}

void OOXMLContextTable::setId(doctok::Id nId)
{
    mId = nId;
}

void OOXMLContextTable::endElement(TokenEnum_t /*nToken*/,
                                   OOXMLContext::Pointer_t pContext)
{
    OOXMLValue::Pointer_t pValue(pContext->getValue());
    
    if (pValue.get() != NULL)
    {
        OOXMLTableImpl::ValuePointer_t pTmpVal(pValue->clone());
        mTable.add(pTmpVal);        
    }
}

void OOXMLContextTable::newPropertySet(OOXMLPropertySet::Pointer_t /*pPropertySet*/)
{
}

/*
  class OOXMLContextTextTable
*/
OOXMLContextTextTable::OOXMLContextTextTable(Stream & rStream)
: OOXMLContextStream(rStream)
{
}

OOXMLContextTextTable::OOXMLContextTextTable(const OOXMLContext & rContext)
: OOXMLContextStream(rContext)
{
    mnTableDepth++;
}

OOXMLContextTextTable::~OOXMLContextTextTable()
{
}

/*
  class OOXMLContextTextTableRow
*/
OOXMLContextTextTableRow::OOXMLContextTextTableRow(Stream & rStream)
: OOXMLContextStream(rStream)
{
    startRow();
}

OOXMLContextTextTableRow::OOXMLContextTextTableRow
(const OOXMLContext & rContext)
: OOXMLContextStream(rContext)
{
    startRow();
}

OOXMLContextTextTableRow::~OOXMLContextTextTableRow()
{
    endRow();
}

void OOXMLContextTextTableRow::startRow()
{    
}

void OOXMLContextTextTableRow::endRow()
{
    startParagraphGroup();

    OOXMLPropertySet * pProps = new OOXMLPropertySetImpl();
    {
        OOXMLValue::Pointer_t pVal
            (new OOXMLIntegerValue(mnTableDepth));
        OOXMLProperty::Pointer_t pProp
            (new OOXMLPropertyImpl(0x6649, pVal, OOXMLPropertyImpl::SPRM));
        pProps->add(pProp);
    }
    {
        OOXMLValue::Pointer_t pVal
            (new OOXMLIntegerValue(1));
        OOXMLProperty::Pointer_t pProp
            (new OOXMLPropertyImpl(0x2416, pVal, OOXMLPropertyImpl::SPRM));
        pProps->add(pProp);
    }
    {
        OOXMLValue::Pointer_t pVal
            (new OOXMLIntegerValue(1));
        OOXMLProperty::Pointer_t pProp
            (new OOXMLPropertyImpl(0x244c, pVal, OOXMLPropertyImpl::SPRM));
        pProps->add(pProp);
    }
    
    if (isForwardEvents())
        mrStream.props(doctok::Reference<Properties>::Pointer_t(pProps));

    startCharacterGroup();
    
    if (isForwardEvents())
        mrStream.utext(s0xd, 1);

    endParagraphGroup();
}

/*
  class OOXMLContextTextTableCell
*/
OOXMLContextTextTableCell::OOXMLContextTextTableCell(Stream & rStream)
: OOXMLContextStream(rStream)
{
}

OOXMLContextTextTableCell::OOXMLContextTextTableCell
(const OOXMLContext & rContext)
: OOXMLContextStream(rContext)
{
}

OOXMLContextTextTableCell::~OOXMLContextTextTableCell()
{
}

void OOXMLContextTextTableCell::startCell()
{
    sendTableDepth();
}

void OOXMLContextTextTableCell::endCell()
{
    startCharacterGroup();

    OOXMLPropertySet * pProps = new OOXMLPropertySetImpl();
    {
        OOXMLValue::Pointer_t pVal
            (new OOXMLIntegerValue(mnTableDepth));
        OOXMLProperty::Pointer_t pProp
            (new OOXMLPropertyImpl(0x6649, pVal, OOXMLPropertyImpl::SPRM));
        pProps->add(pProp);
    }
    {
        OOXMLValue::Pointer_t pVal
            (new OOXMLIntegerValue(1));
        OOXMLProperty::Pointer_t pProp
            (new OOXMLPropertyImpl(0x2416, pVal, OOXMLPropertyImpl::SPRM));
        pProps->add(pProp);
    }
    {
        OOXMLValue::Pointer_t pVal
            (new OOXMLIntegerValue(mnTableDepth));
        OOXMLProperty::Pointer_t pProp
            (new OOXMLPropertyImpl(0x244b, pVal, OOXMLPropertyImpl::SPRM));
        pProps->add(pProp);
    }

    if (isForwardEvents())
        mrStream.props(doctok::Reference<Properties>::Pointer_t(pProps));

    // no endCharacterGroup here
}

/*
  class OOXMLContextXNote
 */
OOXMLContextXNote::OOXMLContextXNote(Stream & rStream)
: OOXMLContextStream(rStream)
{
    mbForwardEventsSaved = mpParserState->isForwardEvents();
    setForwardEvents(false);
}

OOXMLContextXNote::OOXMLContextXNote(const OOXMLContext & rContext)
: OOXMLContextStream(rContext)
{
    mbForwardEventsSaved = isForwardEvents();
    setForwardEvents(false);

    clog << "OOXMLContextXNote:" <<  
        OUStringToOString(getXNoteId(), 
                          RTL_TEXTENCODING_ASCII_US).getStr()
         << endl;
}

OOXMLContextXNote::~OOXMLContextXNote()
{
    setForwardEvents(mbForwardEventsSaved);
}

void OOXMLContextXNote::checkId(const rtl::OUString & rId)
{
    if (rId.compareTo(getXNoteId()) == 0)
        setForwardEvents(true);

    clog << "OOXMLContextXNote::checkId: " << isForwardEvents() << endl;
}

/*
  class OOXMLFootnoteHandler
 */
OOXMLFootnoteHandler::OOXMLFootnoteHandler(OOXMLContext * pContext)
: mpContext(pContext)
{
}

OOXMLFootnoteHandler::~OOXMLFootnoteHandler()
{
}

void OOXMLFootnoteHandler::attribute(Id name, Value & val)
{
    switch (name)
    {
    case NS_ooxml::LN_CT_FtnEdnRef_id:
        mpContext->resolveFootnote(val.getString());
        break;
    default:
        break;
    }
}

void OOXMLFootnoteHandler::sprm(Sprm & /*sprm*/)
{
}

/*
  class OOXMLEndnoteHandler
 */
OOXMLEndnoteHandler::OOXMLEndnoteHandler(OOXMLContext * pContext)
: mpContext(pContext)
{
}

OOXMLEndnoteHandler::~OOXMLEndnoteHandler()
{
}

void OOXMLEndnoteHandler::attribute(Id name, Value & val)
{
    switch (name)
    {
    case NS_ooxml::LN_CT_FtnEdnRef_id:
        mpContext->resolveEndnote(val.getString());
        break;
    default:
        break;
    }
}

void OOXMLEndnoteHandler::sprm(Sprm & /*sprm*/)
{
}

/*
  class OOXMLCommentHandler
*/
OOXMLCommentHandler::OOXMLCommentHandler(OOXMLContext * pContext)
: mpContext(pContext)
{    
}

OOXMLCommentHandler::~OOXMLCommentHandler()
{
}

void OOXMLCommentHandler::attribute(Id name, Value & val)
{
    switch (name)
    {
    case NS_ooxml::LN_CT_Markup_id:
        mpContext->resolveComment(val.getString());
        break;
    default:
        ;
    }
}

void OOXMLCommentHandler::sprm(Sprm & /*sprm*/)
{
}

/*
  class OOXMLFooterHandler
 */
OOXMLFooterHandler::OOXMLFooterHandler(OOXMLContext * pContext)
: mpContext(pContext), msStreamId(), mnType(0)
{
}

OOXMLFooterHandler::~OOXMLFooterHandler()
{
    mpContext->resolveFooter(mnType, msStreamId);
}

void OOXMLFooterHandler::attribute(Id name, Value & val)
{
    switch (name)
    {
    case NS_ooxml::LN_CT_HdrFtrRef_id:
        msStreamId = val.getString();
        break;
    case NS_ooxml::LN_CT_HdrFtrRef_type:
        mnType = val.getInt();
        break;
    default:
        break;
    }
}

void OOXMLFooterHandler::sprm(Sprm & /*sprm*/)
{
}

/*
  class OOXMLHeaderHandler
 */
OOXMLHeaderHandler::OOXMLHeaderHandler(OOXMLContext * pContext)
: mpContext(pContext), msStreamId(), mnType(0)
{
}

OOXMLHeaderHandler::~OOXMLHeaderHandler()
{
    mpContext->resolveHeader(mnType, msStreamId);
}

void OOXMLHeaderHandler::attribute(Id name, Value & val)
{
    switch (name)
    {
    case NS_ooxml::LN_CT_HdrFtrRef_id:
        msStreamId = val.getString();
        break;
    case NS_ooxml::LN_CT_HdrFtrRef_type:
        mnType = val.getInt();
        break;
    default:
        break;
    }
}

void OOXMLHeaderHandler::sprm(Sprm & /*sprm*/)
{
}

/*
  class OOXMLBreakHandler
 */
OOXMLBreakHandler::OOXMLBreakHandler(Stream &rStream, OOXMLContext * pContext)
: mpContext(pContext), mnType(0), mnClear(0), mrStream(rStream)
{
}

OOXMLBreakHandler::~OOXMLBreakHandler()
{
    ::rtl::OUString tmpBreak;
    switch (mnType)
    {
    case NS_ooxml::LN_Value_ST_BrType_column:
        tmpBreak = ::rtl::OUString(0x000E);
        break;
    case NS_ooxml::LN_Value_ST_BrType_page:
        tmpBreak = ::rtl::OUString(0x000C);
        break;
    case NS_ooxml::LN_Value_ST_BrType_textWrapping:
    default: // when no attribute type is present, the spec assume textWrapping
        tmpBreak = ::rtl::OUString(0x000A);
        break;
    }
    if (tmpBreak.getLength())
        mrStream.utext(reinterpret_cast < const sal_uInt8 * > 
                      (tmpBreak.getStr()), tmpBreak.getLength());
}

void OOXMLBreakHandler::attribute(Id name, Value & val)
{
    switch (name)
    {
    case NS_ooxml::LN_CT_Br_type:
        mnType = val.getInt();
        break;
    case NS_ooxml::LN_CT_Br_clear:
    	mnClear = val.getInt();
    	break;
    default:
        break;
    }
}

void OOXMLBreakHandler::sprm(Sprm & /*sprm*/)
{
}

} // namespace ooxml
