// -*- C++ -*-
/**
 * \file filename.h
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS.
 */

#ifndef FILENAME_H
#define FILENAME_H

#include <string>


namespace lyx {
namespace support {


class FileName {
public:
	FileName();
	/** \param abs_filename the file in question. Must have an absolute path.
	 *  \param save_abs_path how is the file to be output to file?
	 */
	FileName(std::string const & abs_filename, bool save_abs_path = true);

	/** \param filename the file in question. May have either a relative
	 *  or an absolute path.
	 *  \param buffer_path if \c filename has a relative path, generate
	 *  the absolute path using this.
	 */
	void set(std::string const & filename, std::string const & buffer_path);

	void erase();
	bool empty() const { return name_.empty(); }

	bool saveAbsPath() const { return save_abs_path_; }
	std::string const absFilename() const { return name_; }
	/// \param buffer_path if empty, uses `pwd`
	std::string const relFilename(std::string const & buffer_path = std::string()) const;
	/// \param buf_path if empty, uses `pwd`
	std::string const outputFilename(std::string const & buf_path = std::string()) const;

	/** @returns a mangled representation of the absolute file name
	 *  suitable for use in the temp dir when, for example, converting
	 *  an image file to another format.
	 *
	 *  @param dir the directory that will contain this file with
	 *  its mangled name. This information is used by the mangling
	 *  algorithm when determining the maximum allowable length of
	 *  the mangled name.
	 *
	 *  An example of a mangled name:
	 *  C:/foo bar/baz.eps -> 0C__foo_bar_baz.eps
	 *
	 *  It is guaranteed that
	 *  - two different filenames have different mangled names
	 *  - two FileName instances with the same filename have identical
	 *    mangled names.
	 *
	 *  Only the mangled file name is returned. It is not prepended
	 *  with @c dir.
	 */
	std::string const
	mangledFilename(std::string const & dir = std::string()) const;

	/// \return true if the file is compressed.
	bool isZipped() const;
	/// \return the absolute file name without its .gz, .z, .Z extension
	std::string const unzippedFilename() const;

private:
	std::string name_;
	bool save_abs_path_;
};


bool operator==(FileName const &, FileName const &);
bool operator!=(FileName const &, FileName const &);


} // namespace support
} // namespace lyx

#endif
