/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef NETWORKWINDOW_H
#define NETWORKWINDOW_H

#include <kdialogbase.h>

// Forward declarations
class QCheckBox;
class KTextEdit;


/**
 * A network window, and general event log.
 * This dialog receives signals from other classes regarding events
 * in the system. It allows some basic filtering, so that the end
 * users can see basic 'online/offline' logs, while MSN protocol hackers
 * can get the full joy of the protocol information.
 *
 * @author Michael Jarrett (JudgeBeavis@hotmail.com)
 * @date April, 2004.
 */
class NetworkWindow : public KDialogBase
{
  Q_OBJECT

  public:
    NetworkWindow(QWidget *parent= 0, const char *name= 0);

  public slots:
    /**
     * Sets the number of lines that can be displayed in the log.
     * We will buffer this many log entries as well, though less
     * may be displayed if some entries are multi-line.
     */
    void setLogCapacity(int entries);

    /**
     * Adds an entry to the log.
     * This is a text message, formatted like QTextEdit expects in LogText mode.
     * The log will add date/time info to it, the rest is up to you.
     * Lines will not be broken unless you do so explicitly.
     * @param priority An integer representing the importance of a message.
     *                 any integer is valid, but I recommend -10 to +10, where
     *                 0 is the 'standard' log setting, and 10 are messages that
     *                 you shouldn't be able to hide ever.
     * @param msg  The message. Please i18n it before giving it to us.
     */
    void addLogEntry(int priority, const QString &msg);


    /**
     * Logs incoming messages from the server.
     * A shortcut to display an inbound server message, but with the ability
     * to filter separately. Also, any tag-like things are replaced with
     * entities so they can be displayed.
     * @param msg  Raw message buffer from the connection.
     */
    void addIncomingServerMessage(const QString &msg);


    /**
     * Logs outgoing traffic.
     * Logs outgoing messages to the server.
     * A shortcut to display an inbound server message, but with the ability
     * to filter separately. Also, any tag-like things are replaced with
     * entities so they can be displayed.
     * @param msg  Raw buffer of what you intend to send.
     */
    void addOutgoingServerMessage(const QString &msg);



  protected:
    void addMessage(const QString &msg);


  protected slots:
    void adjustFilters();


  private:
    // Controls
    KTextEdit    *logWidget_; ///< The widget used to display the log
    QCheckBox    *filterIncomingWidget_, *filterOutgoingWidget_;

    // Data
    int          capacity_;
    bool         showIncoming_, showOutgoing_;
};


#endif
