#ifndef KADU_USERBOX_H
#define KADU_USERBOX_H

#include <qlistbox.h>
#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qtimer.h>

#include <vector>

#include "gadu.h"
#include "misc.h"
#include "userlistelement.h"

class QFontMetrics;
class UserBoxSlots;
class ULEComparer;

/**
	Klasa reprezentujca kontakt wizualnie na licie kontaktw. Opisuje ona ikon kontaktu,
	jego wywietlan nazw oraz ewentualny opis.
	\class KaduListBoxPixmap
	\brief Klasa reprezentujca kontakt wizualnie.
**/
class KaduListBoxPixmap : public QListBoxItem
{
	public:
		/**
			\fn KaduListBoxPixmap(const QPixmap &pix, UserListElement user, bool bold = false)
			Konstruktor dodajcy kontakt do listy z podan ikon.
			\param pix ikona wywietlana obok kontaktu.
			\param user kontakt, z ktrego bd pobierane dane do wywietlenia
			\param bold warto logiczna informujca o tym, czy nazwa kontaktu
				ma by wywietlana pogrubion czcionk.
		**/
		KaduListBoxPixmap(const QPixmap &pix, UserListElement user, bool bold = false);

		/**
			\fn bool isBold() const
			Zwraca informacj, czy nazwa kontaktu jest wywietlana pogrubion czcionk, czy nie.
		**/
		bool isBold() const { return bold; }

		/**
			\fn int height(const QListBox *lb) const
			Zwraca wysoko elementu reprezentujcego wywietlany kontakt w pikselach.
			\param lb obiekt reprezentujcy wywietlany kontakt.
		**/
		int height(const QListBox *lb) const;

		/**
			\fn int width(const QListBox *lb) const
			Zwraca szeroko elementu reprezentujcego wywietlany kontakt w pikselach.
			\param lb obiekt reprezentujcy wywietlany kontakt.
		**/
		int width(const QListBox *lb) const;

		const UserListElement User;
	protected:
		/**
			\fn int lineHeight(const QListBox *lb) const
			Zwraca wysoko samej czcionki dla elementu reprezentujcego wywietlany kontakt w pikselach.
			\param lb obiekt reprezentujcy wywietlany kontakt.
		**/
		int lineHeight(const QListBox *lb) const;

		/**
			\fn void paint(QPainter *painter)
			Rysuje wywietlany kontakt na licie.
			\param painter wskanik do obiektu rysujcego.
		**/
		void paint(QPainter *painter);

		/**
			\fn void setBold(bool b)
			Ustawia stan pogrubienia czcionki wywietlanego kontaktu.
			\param b warto logiczna okrelajca, czy czcionka ma by pogrubiona czy nie.
		**/
		void setBold(bool b) { bold = b; }

		/**
			\fn void calculateSize(const QString &text, int width, QStringList &out, int &height) const
			Oblicza odpowiedni wysoko elementu listy kontaktw tak, aby pomieci
			opis podzielony na niezbdn liczb linii.
			\param[in] text wywietlany opis.
			\param[in] width szeroko kolumny listy kontaktw.
			\param[out] out lista acuchw znakw, zawierajca kolejne linie opisu powstae
				w wyniku podzielenia opisu tak, aby zmieci si w danej szerokoci kolumny.
			\param[out] height wysoko elementu listy kontaktw potrzebna aby pomieci opis
				podzielony w liniach.
		**/
		void calculateSize(const QString &text, int width, QStringList &out, int &height) const;

		/**
			\fn void changeText(const QString &text)
			Zmienia nazw wywietlan dla kontaktu na licie na podan.
			\param text nowa nazwa do wywietlenia.
		**/
		void changeText(const QString &text);
		friend class UserBox;
		friend class UserBoxSlots;

		/*funkcje wprowadzone eby zaoszczdzi na odwoaniach do pliku konfiguracyjnego*/
		/**
			\fn static void setFont(const QFont &f)
			Ustawia czcionk dla tego elementu.
			\param f czcionka
		**/
		static void setFont(const QFont &f);

		/**
			\fn static void setMyUIN(UinType u)
			Ustawia UIN lokalnego uytkownika Kadu dla tego elementu, aby byo wiadomo,
			e ten element jego reprezentuje (poniewa do pobierania informacji o statusie i opisie
			uywane s rne metody dla kontaktu i lokalnego uytkownika).
			\param u numer UIN.
		**/
		static void setMyUIN(UinType u);

		/**
			\fn static void setShowDesc(bool sd)
			Ustawia stan wywietlania opisw na licie kontaktw.
			\param sd warto logiczna informujca o tym, czy opisy maj by wywietlane na licie kontatw.
		**/
		static void setShowDesc(bool sd);

		/**
			\fn static void setAlignTop(bool at)
			Ustawia stan wyrwnywania do grnego brzegu elementu.
			\param at warto logiczna informujca o tym, czy wywietlana nazwa kontaktu ma by wyrwnywana
				do grnej krawdzi elementu.
		**/
		static void setAlignTop(bool at);

		/**
			\fn static void setShowMultilineDesc(bool m)
			Ustawia stan wywietlania opisw wieloliniowych.
			\param m warto logiczna informujca o tym, czy opis tego elementu moe by wywietlany
				w wielu liniach, jeli zawiera znaki nowej linii.
		**/
		static void setShowMultilineDesc(bool m);

		/**
			\fn static void setMultiColumn(bool m)
			Ustawia stan wywietlania listy kontaktw w wielu kolumnach.
			\param m warto logiczna informujca o tym, czy lista ma by wywietlana w wielu kolumnach.
		**/
		static void setMultiColumn(bool m);

		/**
			\fn static void setMultiColumnWidth(int w)
			Ustawia szeroko jednej kolumny dla wywietlania listy w trybie wielu kolumn.
			\param w szeroko kolumny w pikselach.
		**/
		static void setMultiColumnWidth(int w);

		/**
			\fn static void setDescriptionColor(const QColor &col)
			Ustawia kolor wywietlanego opisu.
			\param col kolor
		**/
		static void setDescriptionColor(const QColor &col);

	private:
		QPixmap pm;
		bool bold;
		static QFontMetrics *descriptionFontMetrics;

		static UinType myUIN;
		static bool ShowDesc;
		static bool AlignUserboxIconsTop;
		static bool ShowMultilineDesc;
		static bool MultiColumn;
		static int  MultiColumnWidth;
		static QColor descColor;

		mutable QString buf_text;
		mutable int buf_width;
		mutable QStringList buf_out;
		mutable int buf_height;
};

/**
	Klasa reprezentujca menu podrczne listy kontaktw - czyli to, ktre dostajemy klikajc
	na ktrymkolwiek kontakcie prawym przyciskiem myszy.
	\class UserBoxMenu
	\brief Menu podrczne listy kontaktw.
**/
class UserBoxMenu : public QPopupMenu
{
	Q_OBJECT

	private:
		QValueList<QPair<QString, QString> > iconNames;
	private slots:
		void restoreLook();

	public:
		/**
			\fn UserBoxMenu(QWidget *parent=0, const char* name=0)
			Standardowy konstruktor.
			\param parent wskanik na obiekt kontrolki-rodzica.
			\param name nazwa kontrolki.
		**/
		UserBoxMenu(QWidget *parent=0, const char* name=0);

		/**
			\fn int getItem(const QString &caption) const
			Zwraca unikaln liczb identyfikujc element menu z podanym napisem.
			\param caption napis elementu.
		**/
		int getItem(const QString &caption) const;

	public slots:
		/**
			\fn void show(QListBoxItem *item)
			Pokazuje menu dla podanego kontaktu.
			\param item element wywietlanej listy kontaktw reprezentujcy rzdany kontakt.
		**/
		void show(QListBoxItem *item);

		/**
			\fn int addItem(const QString &text, const QObject* receiver, const char* member, const QKeySequence accel= 0, int id= -1)
			Dodaje nowy element do menu.
			\param text napis dla nowego elementu.
			\param receiver obiekt odbierajcy sygna wybrania elementu z menu.
			\param member SLOT obiektu \a receiver ktry zostanie wykonany po wybraniu elementu z menu.
			\param accel skrt klawiaturowy dla tego elementu. Domylnie brak.
			\param id Unikatowa liczba identyfikujca nowy element. Domylnie pierwsza wolna.
		**/
		int addItem(const QString &text, const QObject* receiver, const char* member, const QKeySequence accel= 0, int id= -1);

		/**
			\fn int addItem(const QString &iconname, const QString &text, const QObject* receiver, const char* member, const QKeySequence accel= 0, int id= -1)
			Dodaje nowy element do menu.
			\param iconname nazwa ikony z zestawu lub sciezka do pliku
			\param text napis dla nowego elementu.
			\param receiver obiekt odbierajcy sygna wybrania elementu z menu.
			\param member SLOT obiektu \a receiver ktry zostanie wykonany po wybraniu elementu z menu.
			\param accel skrt klawiaturowy dla tego elementu. Domylnie brak.
			\param id Unikatowa liczba identyfikujca nowy element. Domylnie pierwsza wolna.
		**/
		int addItem(const QString &iconname, const QString &text, const QObject* receiver, const char* member, const QKeySequence accel= 0, int id= -1);

		/**
			\fn int addItemAtPos(int index,const QString &iconname, const QString &text, const QObject* receiver, const char* member, const QKeySequence accel= 0, int id= -1)
			Dodaje nowy element do menu.
			\param index pozycja (liczc od 0) na ktrej znale ma si nowy element.
			\param iconname nazwa ikony z zestawu lub sciezka do pliku
			\param text napis dla nowego elementu.
			\param receiver obiekt odbierajcy sygna wybrania elementu z menu.
			\param member SLOT obiektu \a receiver ktry zostanie wykonany po wybraniu elementu z menu.
			\param accel skrt klawiaturowy dla tego elementu. Domylnie brak.
			\param id Unikatowa liczba identyfikujca nowy element. Domylnie pierwsza wolna.
		**/
		int addItemAtPos(int index,const QString &iconname, const QString &text, const QObject* receiver, const char* member, const QKeySequence accel= 0, int id= -1);

		/**
			\fn void refreshIcons()
			Przeadowuje wszystkie ikony w tym menu dodane przy pomocy powyszych funkcji
		**/
		void refreshIcons();
	signals:
		/**
			\fn void popup()
			Sygna emitowany, gdy menu jest wywoywane.
		**/
		void popup();
};


/**
	Klasa reprezentujca list kontaktw wraz z ikonkami stanw.
	\class UserBox
	\brief Wywietlana lista kontaktw.
**/
class UserBox : public QListBox
{
	Q_OBJECT

	public:
		/**
			\fn UserBox(UserGroup *group = userlist, QWidget* parent = 0, const char* name = 0, WFlags f = 0)
			Standardowy konstruktor tworzcy list kontaktw.
			\param group grupa kontaktw, ktra bdzie wywietlana
			\param parent rodzic kontrolki. Domylnie 0.
			\param name nazwa kontrolki. Domylnie 0.
			\param f flagi kontrolki. Domylnie 0.
		**/
		UserBox(UserGroup *group = userlist, QWidget* parent = 0, const char* name = 0, WFlags f = 0);

		virtual ~UserBox();

		/**
			\var static UserBoxMenu *userboxmenu
			Wskanik do menu kontekstowego listy kontaktw.
		**/
		static UserBoxMenu *userboxmenu;

		static void setColorsOrBackgrounds();

		/**
			\fn static void initModule()
			Inicjalizuje zmienne niezbdne do dziaania UserBox. Funkcja ta jest
			wywoywana przy starcie Kadu przez rdze Kadu.
		**/
		static void initModule();

		static void closeModule();

		/**
			\fn UserListElements getSelectedUsers() const
			Funkcja zwraca list zaznaczonych uzytkownikw.
			\return UserListElements z zaznaczonymi uytkownikami.
		**/
		UserListElements selectedUsers() const;

		/**
			\fn const UserGroup *visibleUsers() const
			zwraca list kontaktw, ktre s obecnie na licie
			\return grupa kontaktw
		**/
		const UserGroup *visibleUsers() const;

		/**
			\fn QValueList<UserGroup *> filters() const
			Zwraca list filtrw "pozytywnych"
			\return lista filtrw
		**/
		QValueList<UserGroup *> filters() const;

		/**
			\fn QValueList<UserGroup *> negativeFilters() const
			Zwraca list filtrw "negatywnych"
			\return lista filtrw
		**/
		QValueList<UserGroup *> negativeFilters() const;

		/**
			\fn bool currentUserExists() const
			\return informacja o istnieniu biecego kontaktu
		**/
		bool currentUserExists() const;

		/**
			\fn UserListElement currentUser() const
			\attention {przed wywoaniem tej metody naley sprawdzi, czy istnieje biecy kontakt!}
			\return biecy kontakt
			\see UserBox::currentUserExists()
		**/
		UserListElement currentUser() const;

		/**
			\fn static UserBox* activeUserBox()
			Funkcja znajdujaca aktywny UserBox.
			\return wskanik do aktywnego UserBox'a, jeli taki nie istnieje zwracana jest warto NULL.
		**/
		static UserBox* activeUserBox();

		class CmpFuncDesc
		{
			public:
 				CmpFuncDesc(QString i, QString d, int (*f)(const UserListElement &, const UserListElement &))
 					: id(i), description(d), func(f) {}
 				CmpFuncDesc() : id(), description(), func(0){}

				QString id;
				QString description;
				int (*func)(const UserListElement &, const UserListElement &);
		};

		/**
			\fn QValueList<UserBox::CmpFuncDesc> compareFunctions() const
			\return lista obiektw opisujcych funkcje porwnujce
		**/
 		QValueList<UserBox::CmpFuncDesc> compareFunctions() const;

		/**
			\fn void addCompareFunction(const QString &id, const QString &trDescription, int (*cmp)(const UserListElement &, const UserListElement &))
			Dodaje funkcj porwnujc do biecego UserBoksa.
			Funkcja cmp powinna zwraca warto:
				< 0 gdy pierwszy kontakt powinien znale si przed drugim
				= 0 gdy kolejno kontaktw nie ma znaczenia
				> 0 gdy pierwszy kontakt powinien znale si za drugim
			\param id krtki identyfikator funkcji
			\param trDescription pzetumaczony opis funkcji (w zamierzeniu do wywietlenia w konfiguracji)
			\param cmp funkcja porwnujca
		**/
		void addCompareFunction(const QString &id, const QString &trDescription,
					int (*cmp)(const UserListElement &, const UserListElement &));

		/**
			\fn static void refreshAll()
			Odwiea wszystkie UserBoksy.
		**/
		static void refreshAll();

		/**
			\fn static void refreshAllLater()
			Odwiea wszystkie UserBoksy, ale dopiero po powrocie do ptli zdarze Qt.
		**/
		static void refreshAllLater();

		static const QValueList<UserBox *> &userboxes() {return UserBoxes;}

		static CreateNotifier createNotifier;
	public slots:

		/**
			\fn void applyFilter(UserGroup *group)
			Nakada na biecy UserBox filtr "pozytywny" - wywietlane bd
			tylko te kontakty, ktre nale do group.
			\param group filtr
		**/
		void applyFilter(UserGroup *group);

		/**
			\fn void removeFilter(UserGroup *group)
			Usuwa z biecego UserBoksa wskazany filtr "pozytywny".
			\param group filtr
		**/
		void removeFilter(UserGroup *group);

		/**
			\fn void applyNegativeFilter(UserGroup *group)
			Nakada na biecy UserBox filtr "negatywny" - wywietlane bd
			tylko te kontakty, ktre nie nale do group.
			\param group filtr
		**/
		void applyNegativeFilter(UserGroup *group);

		/**
			\fn void removeNegativeFilter(UserGroup *group)
			Usuwa z biecego UserBoksa wskazany filtr "negatywny".
			\param group filtr
		**/
		void removeNegativeFilter(UserGroup *group);

		/**
			\fn void removeCompareFunction(const QString &id)
			Usuwa funkcj porwnujc o identyfikatorze id.
			\param id identyfikator funkcji porwnujcej
		**/
		void removeCompareFunction(const QString &id);

		/**
			\fn void moveUpCompareFunction(const QString &id)
			Przesuwa funkcj porwnujc o identyfikatorze id wyej w kolejnoci sprawdzania.
			\param id identyfikator funkcji porwnujcej
		**/
		bool moveUpCompareFunction(const QString &id);

		/**
			\fn void moveDownCompareFunction(const QString &id)
			Przesuwa funkcj porwnujc o identyfikatorze id niej w kolejnoci sprawdzania.
			\param id identyfikator funkcji porwnujcej
		**/
 		bool moveDownCompareFunction(const QString &id);

		/**
			\fn void refresh()
			Odwiea biecy UserBox. Nie naley tej funkcji naduywa,
			bo wikszo (90%) sytuacji jest wykrywanych przez sam klas.
		**/
		void refresh();

		void refreshLater();
	signals:
		/**
			\fn void doubleClicked(UserListElement user)
			Sygna jest emitowany, gdy na ktrym z kontaktw kliknito dwukrotnie.
			\param user kontakt, na ktrym kilknito dwukrotnie
			\warning Uywaj tego sygnau zamiast QListBox::doubleClicked(QListBoxItem *) !!!
			Tamten ze wzgldu na odwieanie listy w jednym ze slotw podczonych
			do tego sygnau czasami przekazuje wskanik do elementu, ktry ju NIE ISTNIEJE.
		**/
		void doubleClicked(UserListElement user);

		/**
			\fn void returnPressed(UserListElement user)
			Sygna jest emitowany, gdy wcinito klawisz enter na wybranym kontakcie.
			\param user kontakt, na ktrym wcinito enter
			\warning Uywaj tego sygnau zamiast QListBox::returnPressed(QListBoxItem *) !!!
			Tamten ze wzgldu na odwieanie listy w jednym ze slotw podczonych
			do tego sygnau czasami przekazuje wskanik do elementu, ktry ju NIE ISTNIEJE.
		**/
		void returnPressed(UserListElement user);

		/**
			\fn void currentChanged(UserListElement user)
			Sygna jest emitowany, gdy zmieni si biecy kontakt.
			\attention {raczej naleu uywa tego sygnau zamiast QListBox::currentChaned(QListBoxItem *)}
			\param user obecnie biecy kontakt
		**/
		void currentChanged(UserListElement user);

		void changeToolTip(const QPoint &point, UserListElement user, bool show);

	private slots:
		void doubleClickedSlot(QListBoxItem *item);
		void returnPressedSlot(QListBoxItem *item);
		void currentChangedSlot(QListBoxItem *item);

		void userAddedToVisible(UserListElement elem, bool massively, bool last);
		void userRemovedFromVisible(UserListElement elem, bool massively, bool last);

		void userAddedToGroup(UserListElement elem, bool massively, bool last);
		void userRemovedFromGroup(UserListElement elem, bool massively, bool last);

		void userDataChanged(UserListElement elem, QString name, QVariant oldValue,
							QVariant currentValue, bool massively, bool last);
		void protocolUserDataChanged(QString protocolName, UserListElement elem,
							QString name, QVariant oldValue, QVariant currentValue,
							bool massively, bool last);
		void removingProtocol(UserListElement elem, QString protocolName, bool massively, bool last);

		void tipTimeout();
		void restartTip(const QPoint &p);
		void hideTip();
		void resetVerticalPosition();
		void rememberVerticalPosition();

		void messageFromUserAdded(UserListElement elem);
		void messageFromUserDeleted(UserListElement elem);

	private:
		static QValueList<UserBox*> UserBoxes;
		static UserBoxSlots *userboxslots;

		UserGroup *VisibleUsers;
		QValueList<UserGroup *> Filters;
		QValueList<UserGroup *> NegativeFilters;
		std::vector<UserListElement> sortHelper;
		std::vector<UserListElement> toRemove;
		QMap<const UserGroup *, UserListElements> AppendProxy;
		QMap<const UserGroup *, UserListElements> RemoveProxy;

		ULEComparer *comparer;
		void sort();
		QTimer refreshTimer;

		UserListElement lastMouseStopUser;
		static UserListElement nullElement;
		QPoint lastMouseStop;
		bool tipAlive;
		QTimer tipTimer;

		QTimer verticalPositionTimer;
		int lastVerticalPosition;

		friend class UserBoxSlots;

	protected:
		virtual void wheelEvent(QWheelEvent *e);
		virtual void enterEvent(QEvent *);
		virtual void leaveEvent(QEvent *);

		/**
			\fn virtual void mousePressEvent(QMouseEvent *e)
			Wcinito ktry z przyciskw myszki na licie kontaktw.
			\param e wskanik obiektu opisujcego to zdarzenie.
		**/
		virtual void mousePressEvent(QMouseEvent *e);

		virtual void mouseReleaseEvent(QMouseEvent *e);

		/**
			\fn virtual void mouseMoveEvent(QMouseEvent* e)
			Poruszono myszk nad list kontaktw.
			\param e wskanik obiektu opisujcego to zdarzenie.
		**/
		virtual void mouseMoveEvent(QMouseEvent* e);

		/**
			\fn virtual void keyPressEvent(QKeyEvent *e)
			Wcinito ktry z klawiszw w aktywnej licie kontaktw.
			\param e wskanik obiektu opisujcego to zdarzenie.
		**/
		virtual void keyPressEvent(QKeyEvent *e);

		/**
			\fn virtual void resizeEvent(QResizeEvent *)
			Lista kontaktw zmienia swj rozmiar.
			\param e wskanik obiektu opisujcego to zdarzenie.
		**/
		virtual void resizeEvent(QResizeEvent *);
};

/**
	Klasa ta pozwala na podgld wprowadzanych zmian konfiguracyjnych wobec UserBox
	w oknie konfiguracji, zanim zaaplikuje si je do Kadu.
	\class UserBoxSlots
	\brief Obsuga UserBox w konfiguracji.
**/
class UserBoxSlots : public QObject
{
	Q_OBJECT
	public slots:
		/**
			\fn void onCreateTabLook()
			Obsuguje sekcj UserBox podczas otwierania okna konfiguracji.
		**/
		void onCreateTabLook();

		/**
			\fn void onDestroyConfigDialog()
			Obsuguje sekcj UserBox podczas zamykania okna konfiguracji.
		**/
		void onApplyTabLook();

		/**
			\fn void chooseColor(const char *name, const QColor& color)
			Odwiea podgld wybranego koloru.
			\param name nazwa elementu, dla ktrego wybrano kolor.
				\arg \c userbox_bg_color oznacza kolor ta.
				\arg \c userbox_font_color oznacz kolor czcionki.
			\param color wybrany kolor.
		**/
		void chooseColor(const char *name, const QColor& color);

		/**
			\fn void chooseFont(const char *name, const QFont& font)
			Odwiea podgld wybranej czcionki.
			\param name nazwa elementu, dla ktrego wybrano czcionk.
				\arg \c userbox_font_box oznacza ogln czcionk listy kontaktw.
			\param font wybrana czcionka.
		**/
		void chooseFont(const char *name, const QFont& font);

		/**
			\fn void onMultiColumnUserbox(bool toggled)
			Wcza bd wycza kontrolk szerokoci kolumny listy kontaktw, w zalenoci od podanego argumentu.
			\param togglet warto logiczna informujca, czy kontrolka ma by wczona czy wyczona.
		**/
		void onMultiColumnUserbox(bool toggled);

		/**
			\fn void updatePreview()
			Odwiea podgld wszystkich elementw UserBox'a.
		**/
		void updatePreview();

		void chooseBackgroundFile();
		void userboxBackgroundMove(bool toggled);
	private slots:
		void backgroundFileChanged(const QString &text);
};

/**
	\fn int compareAltNick(const UserListElement &u1, const UserListElement &u2)
	Funkcja porwnujca AltNicki metod QString::localeAwareCompare()
**/
int compareAltNick(const UserListElement &u1, const UserListElement &u2);

/**
	\fn int compareAltNickCaseInsensitive(const UserListElement &u1, const UserListElement &u2)
	Funkcja porwnujca AltNicki metod QString::localeAwareCompare(), wczeniej robica lower(),
	a wic wolniejsza od compareAltNick
**/
int compareAltNickCaseInsensitive(const UserListElement &u1, const UserListElement &u2);

/**
	\fn int compareStatus(const UserListElement &u1, const UserListElement &u2)
	Funkcja porwnujca statusy w protokole Gadu-Gadu. Uwaa status "dostpny" i "zajty" za rwnowane.
**/
int compareStatus(const UserListElement &u1, const UserListElement &u2);

#endif
