/***************************************************************************
 *   Weather module for Kadu communicator                                  *
 *   2004 Micha Maek (bysk)                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "getcitydialog.h"

#include <qlayout.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qpushbutton.h>
#include <qhbox.h>
#include <qlineedit.h>

#include "icons_manager.h"
#include "config_file.h"
#include "message_box.h"
#include "debug.h"

#include "textprogress.h"
#include "weather_global.h"
#include "show_forecast_dialog.h"

/*
	GetCityDialog
*/

GetCityDialog::GetCityDialog()
:
	QDialog( 0, "GetCityDialog" ),
	currentMode_(NONE)
{
	setWFlags(WDestructiveClose);
	setCaption( tr("City search") );
	
	// 1) Progress layout
	progress_ = new TextProgress( this );
	progress_->setMinimumWidth(300);
	progress_->setTextFormat( Qt::PlainText );
	progress_->setAlignment( Qt::AlignCenter | Qt::SingleLine );
	progress_->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
	cancelButton_ = new QPushButton( icons_manager->loadIcon("CloseWindowButton"), tr("Cancel"), this );	
	cancelButton_->setMaximumWidth(100);
	cancelButton_->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	
	// 2) City edit layout
	ecHLayout_ = new QHBox( this );
	ecHLayout_->setSpacing( 5 );
	new QLabel( tr("City:"), ecHLayout_ );
	cityEdit_ = new QComboBox( ecHLayout_ );
	cityEdit_->setEditable( true );
	cityEdit_->insertStringList( weather_global->recentLocations_ );
	cityEdit_->setCurrentText( QString::null );
	cityEdit_->setMinimumWidth(200);
	cityEdit_->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed));
	findButton_ = new QPushButton( icons_manager->loadIcon("LookupUserInfo"),  tr("Find"), this );
	findButton_->setMaximumWidth(100);
	findButton_->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	
	// 3) Select city
	okNewSearchHLayout_ = new QHBox( this );
	okNewSearchHLayout_->setSpacing( 5 );
	cityLabel_ = new QLabel( tr("Select city:"), this );
	cityList_ = new QListBox( this );
	cityList_->setMinimumHeight(150);
	QPushButton* newSearchButton = new QPushButton( icons_manager->loadIcon("LookupUserInfo"), tr("New search"), okNewSearchHLayout_ );
	newSearchButton->setMaximumWidth(100);
	newSearchButton->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	okButton_ = new QPushButton( icons_manager->loadIcon("OkWindowButton"), tr("OK"), okNewSearchHLayout_ );
	okButton_->setMaximumWidth(100);
	okButton_->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed) );
	
	// Main layout
	mainLayout_ = new QVBoxLayout( this );
	mainLayout_->setSpacing( 5 );
	mainLayout_->setMargin( 5 );
	mainLayout_->addWidget( cityLabel_, 0, Qt::AlignLeft ); cityLabel_->hide();
	mainLayout_->addWidget( progress_, 0 ); progress_->hide();
	mainLayout_->addWidget( ecHLayout_, 0 ); ecHLayout_->hide();
	mainLayout_->addWidget( cityList_, 0 ); cityList_->hide();
	mainLayout_->addWidget( cancelButton_, 0, Qt::AlignCenter ); cancelButton_->hide();
	mainLayout_->addWidget( findButton_, 0, Qt::AlignCenter ); findButton_->hide();
	mainLayout_->addWidget( okNewSearchHLayout_, 0, Qt::AlignCenter ); okNewSearchHLayout_->hide();
	
	connect( cancelButton_, SIGNAL(clicked()), this, SLOT(cancelClicked()) );
	connect( findButton_, SIGNAL(clicked()), this, SLOT(findClicked()) );
	connect( okButton_, SIGNAL(clicked()), this, SLOT(okClicked()) );
	connect( newSearchButton, SIGNAL(clicked()), this, SLOT(newSearchClicked()) );
	connect( &searchId_, SIGNAL(nextServerSearch( const QString&, const QString& )),
			this, SLOT(nextServerSearch( const QString&, const QString& )) );
	//connect( cityEdit_, SIGNAL(activated(int)), this, SLOT(findClicked()) );
	connect( &searchId_, SIGNAL(finished()), this, SLOT(searchFinished()) );
	connect( cityList_, SIGNAL(doubleClicked(QListBoxItem*)), this, SLOT(listDoubleClicked(QListBoxItem*)) );
	connect( cityList_, SIGNAL(returnPressed(QListBoxItem*)), this, SLOT(listDoubleClicked(QListBoxItem*)) );
}

GetCityDialog::~GetCityDialog()
{
}

void GetCityDialog::show()
{
	QString ggUserId;
	getUserData( &ggUserId );
	
	if( !ggUserId.isEmpty() )
	{
		// Activating progressLayout_
		progress_->setText( tr("Retrieving city from public directory") );
		switchLayout( SEARCH );
		
		connect(gadu, SIGNAL(newSearchResults(SearchResults &, int, int)), this,
				SLOT(newSearchResults(SearchResults &, int, int)));
		
		SearchRecord searchRecord;
		searchRecord.reqUin( ggUserId );
		gadu->searchInPubdir( searchRecord );
	}
	else
	{
		switchLayout( ENTER );
	}
	
	QDialog::show();
}

void GetCityDialog::showForecast( const CitySearchResult& result )
{
	kdebugf();
	
	ShowForecastDialog* sfd = new ShowForecastDialog( result );
	sfd->show();
	sfd->setFocus();
	
	close();
	
	kdebugf2();
}

void GetCityDialog::newSearchResults(SearchResults& searchResults, int seq, int fromUin)
{
	disconnect(gadu, SIGNAL(newSearchResults(SearchResults &, int, int)), this,
		SLOT(newSearchResults(SearchResults &, int, int)));
	
	if( !searchResults.isEmpty() )
	{
		const SearchResult& result = searchResults.first();
		if( !result.City.isEmpty() )
		{
			findCity( result.City );
			return;
		}
		else if( !result.FamilyCity.isEmpty() )
		{
			findCity( result.FamilyCity );
			return;
		}
	}
	
	switchLayout( ENTER );
}

void GetCityDialog::cancelClicked()
{
	kdebugf();
	
	disconnect(gadu, SIGNAL(newSearchResults(SearchResults &, int, int)), this,
		SLOT(newSearchResults(SearchResults &, int, int)));
	
	searchId_.cancel();
	
	switchLayout( ENTER );
	
	kdebugf2();
}

void GetCityDialog::findClicked()
{
	QString cityName = cityEdit_->currentText();
	if( cityName.isEmpty() )
		MessageBox::wrn( tr("Enter city name!") );
	else
	{
		weather_global->insertRecentLocation( cityName );
		findCity( cityName );
	}
}

void GetCityDialog::okClicked()
{
	showCityAtPos( cityList_->currentItem() );
}

void GetCityDialog::newSearchClicked()
{
	switchLayout( ENTER );
	cityEdit_->lineEdit()->selectAll();
}

void GetCityDialog::listDoubleClicked( QListBoxItem* item )
{
	showCityAtPos( cityList_->index( item ) );
}

void GetCityDialog::findCity( const QString& name )
{
	// Activating progressLayout_
	cityEdit_->setCurrentText( name );
	cityEdit_->lineEdit()->selectAll();
	
	progress_->clearProgressText();
	switchLayout( SEARCH );
	
	searchId_.findID( name );
}

void GetCityDialog::switchLayout( Mode mode )
{
	if( currentMode_ == SEARCH )
	{
		progress_->hide();
		cancelButton_->hide();
		cancelButton_->setDefault( false );
	}
	else if( currentMode_ == ENTER )
	{
		ecHLayout_->hide();
		cityEdit_->hide();
		findButton_->hide();
		findButton_->setDefault( false );
	}
	else if( currentMode_ == SELECT )
	{
		cityLabel_->hide();
		cityList_->hide();
		okNewSearchHLayout_->hide();
		okButton_->setDefault( false );
	}
	
	if( mode == SEARCH )
	{
		mainLayout_->setResizeMode( QLayout::Auto );
		progress_->show();
		cancelButton_->show();
		cancelButton_->setDefault( true );
	}
	else if( mode == ENTER )
	{
		mainLayout_->setResizeMode( QLayout::Fixed );
		ecHLayout_->show();
		cityEdit_->show();
		cityEdit_->setFocus();
		findButton_->show();
		findButton_->setDefault( true );
	}
	else if( mode == SELECT )
	{
		mainLayout_->setResizeMode( QLayout::Auto );
		cityLabel_->show();
		cityList_->show();
		cityList_->setFocus();
		okNewSearchHLayout_->show();
		okButton_->setDefault( true );
	}
	
	currentMode_ = mode;
}

void GetCityDialog::showCityAtPos( int pos )
{
	if( pos >= 0 )
	{
		const CITYSEARCHRESULTS& results = searchId_.getResult();	
		const CitySearchResult& city = results[ pos ];
		showForecast( city );
	}
}

void GetCityDialog::nextServerSearch( const QString& city, const QString& serverName )
{
	progress_->setText( tr("Searching for %1 in %2").arg(city).arg(serverName) );
}

void GetCityDialog::searchFinished()
{
	cityList_->clear();
	const CITYSEARCHRESULTS& results = searchId_.getResult();
	
	if( results.isEmpty() )
	{
		switchLayout( ENTER );
		MessageBox::wrn( tr("City not found!") );
	}
	else if( results.count() == 1 )
	{
		showForecast( results.first() );
	}
	else
	{
		CITYSEARCHRESULTS::const_iterator it = results.begin();
		while( it != results.end() )
		{
			const CitySearchResult& city = *it;
			
			QString serverName;
			weather_global->getServerName( city.server_, &serverName );
			cityList_->insertItem( city.cityName_ + " - " + serverName );
			++it;
		}
		
		cityList_->setCurrentItem(0);
		switchLayout( SELECT );
	}
}

/*
	GetUserCityDialog
*/

GetUserCityDialog::GetUserCityDialog( UserListElement user )
:
	user_( user )
{

}

void GetUserCityDialog::getUserData( QString* ggUserId )
{
	*ggUserId = user_.ID("Gadu");
}

void GetUserCityDialog::showForecast( const CitySearchResult& result )
{
	kdebugf();
	
	result.writeUserWeatherData( user_ );
	
	ShowUserForecastDialog* sfd = new ShowUserForecastDialog( result, user_ );
	sfd->show();
	sfd->setFocus();
	
	close();
	
	kdebugf2();
}

/*
	GetMyCityDialog
*/

void GetMyCityDialog::getUserData( QString* ggUserId )
{
	*ggUserId = config_file.readEntry("General","UIN");
}

void GetMyCityDialog::showForecast( const CitySearchResult& result )
{
	kdebugf();
	
	result.writeMyWeatherData();
	
	ShowMyForecastDialog* sfd = new ShowMyForecastDialog( result );
	sfd->show();
	sfd->setFocus();
	
	close();
	
	kdebugf2();
}
