#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef __SOLARIS__
# include "../include/os.h"
#endif

#include "../include/fio.h"

#include "guiutils.h"
#include "pie_chart.h"
#include "cdialog.h"
#include "fb.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_archive_stat.h"
#include "edv_archive_cmt.h"
#include "archive_info_dlg.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_browse_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_ok_20x20.xpm"


static gint EDVArchiveInfoDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVArchiveInfoAnyChangedCB(GtkWidget *widget, gpointer data);

static void EDVArchiveInfoPasteCommentCB(
	GtkWidget *widget, gpointer data
);
static void EDVArchiveInfoInsertCommentFileCB(
	GtkWidget *widget, gpointer data
);
static void EDVArchiveInfoSetCommentCB(
	GtkWidget *widget, gpointer data
);

static void EDVArchiveInfoOKCB(GtkWidget *widget, gpointer data);

static void EDVArchiveInfoInsertCommentFile(
	edv_archive_info_dlg_struct *d, const gchar *path
);

edv_archive_info_dlg_struct *EDVArchiveInfoNew(
	edv_core_struct *core,
	const gchar *arch_obj,
	const gchar *password,
	GtkWidget *ref_toplevel
);
void EDVArchiveInfoResetHasChanges(
	edv_archive_info_dlg_struct *d, gboolean has_changes
);
void EDVArchiveInfoUpdateMenus(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoSetBusy(edv_archive_info_dlg_struct *d, gboolean is_busy);
gboolean EDVArchiveInfoIsMapped(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoMap(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoUnmap(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoDelete(edv_archive_info_dlg_struct *d);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Archive Info toplevel GtkWindow "delete_event" signal callback.
 */
static gint EDVArchiveInfoDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVArchiveInfoOKCB(widget, data);
	return(TRUE);
}


/*
 *	Archive Info changed signal callback.
 */
static void EDVArchiveInfoAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	if(!d->has_changes)
	    EDVArchiveInfoResetHasChanges(d, TRUE);
}

/*
 *	Paste comment callback.
 */
static void EDVArchiveInfoPasteCommentCB(
	GtkWidget *widget, gpointer data
)
{
	gchar *s;
	GtkText *text;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	text = (GtkText *)d->comment_text;
	if(text == NULL)
	    return;

	EDVArchiveInfoSetBusy(d, TRUE);

	/* Paste */
	s = GUIDDEGetString(
	    GTK_WIDGET(text),		/* Widget */
	    GDK_SELECTION_PRIMARY,	/* Selection */
	    GDK_CURRENT_TIME		/* Time */
	);
	if(s != NULL)
	{
#if 0
/* The GtkEditable widget already has a handler to receive the paste
 * so we only need to make a request and not actually paste
 */
	    gtk_text_freeze(text);
	    gtk_text_insert(
		text, NULL, NULL, NULL,
		s, -1
	    );
	    gtk_text_thaw(text);
#endif
	    g_free(s);
	}

	EDVArchiveInfoResetHasChanges(d, TRUE);

	EDVArchiveInfoSetBusy(d, FALSE);
}

/*
 *	Insert comment file callback.
 */
static void EDVArchiveInfoInsertCommentFileCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean status;
	gchar *parent_path;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	EDVArchiveInfoSetBusy(d, TRUE);

	toplevel = d->toplevel;

	/* Get startup path from archive path */
	parent_path = (d->arch_obj != NULL) ?
	    g_dirname(d->arch_obj) : NULL;

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user for file to insert */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Insert File",
	    "Insert", "Cancel",
	    parent_path,	/* Startup path */
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status && (total_path_rtns > 0))
	    EDVArchiveInfoInsertCommentFile(d, path_rtn[0]);

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Delete startup path */
	g_free(parent_path);

	EDVArchiveInfoSetBusy(d, FALSE);
}

/*
 *	Set Comment Button callback.
 */
static void EDVArchiveInfoSetCommentCB(GtkWidget *widget, gpointer data)
{
	gchar *s, **strv = NULL;
	gint status, strc = 0;
	GtkEditable *editable;
	edv_core_struct *core;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	editable = (GtkEditable *)d->comment_text;
	core = d->core;
	if((core == NULL) || (editable == NULL) || (d->arch_obj == NULL))
	    return;

	EDVArchiveInfoSetBusy(d, TRUE);

	/* Get the comment and explode into multiple lines */
	s = gtk_editable_get_chars(editable, 0, -1);
	strv = g_strsplit(s, "\n", -1);
	g_free(s);
	for(strc = 0; strv[strc] != NULL; strc++);

	/* Set the archive's comment */
	status = EDVArchCommentSet(
	    core, d->arch_obj, strv, strc
	);

	/* Delete the comment lines */
	g_strfreev(strv);

	/* Reset has changes */
	if(status == 0)
	{
	    EDVArchiveInfoResetHasChanges(d, FALSE);

	    if(!STRISEMPTY(d->arch_obj))
	    {
		struct stat lstat_buf;
		if(!lstat(d->arch_obj, &lstat_buf))
		    EDVObjectModifiedEmit(
			core, d->arch_obj, d->arch_obj, &lstat_buf
		    );
	    }
	}

	EDVArchiveInfoSetBusy(d, FALSE);

	/* Error setting archive comment? */
	if(status != 0)
	{
	    const gchar *mesg;

	    switch(status)
	    {
	      case -3:
		mesg =
#if defined(PROG_LANGUAGE_SPANISH)
"El error de sistemas."
#elif defined(PROG_LANGUAGE_FRENCH)
"Erreur de systmes."
#elif defined(PROG_LANGUAGE_GERMAN)
"Systeme fehler."
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'errore di sistemi."
#elif defined(PROG_LANGUAGE_DUTCH)
"Systemen fout."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O erro de sistemas."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Systemfeil."
#elif defined(PROG_LANGUAGE_POLISH)
"Bd systemu."
#else
"Systems error."
#endif
		;
		break;
	      case -2:
		mesg =
#if defined(PROG_LANGUAGE_SPANISH)
"Este formato del archivo no sostiene los comentarios."
#elif defined(PROG_LANGUAGE_FRENCH)
"Ce format de l'archive pas soutien commente."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das format dieses archivs untersttzt bemerkungen nicht."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Questo formato dell'archivio non sostiene i commenti."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het formaat van deze archief steunt opmerkingen niet."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Este formato do arquivo nao apoia comentrios."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dette arkivs format sttter ikke kommentarer."
#elif defined(PROG_LANGUAGE_POLISH)
"Format tego archiwum nie obsuguje komentarzy."
#else
"This archive's format does not support comments."
#endif
		;
		break;
	      default:
		mesg =
#if defined(PROG_LANGUAGE_SPANISH)
"El comentario del archivo de poner de error."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le commentaire d'archive de montage d'erreur."
#elif defined(PROG_LANGUAGE_GERMAN)
"Fehler setzen archiv bemerkung."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il commento di archivio di montaggio di errore."
#elif defined(PROG_LANGUAGE_DUTCH)
"Fout zetten archief opmerking."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Erro por o comentrio de arquivo."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"En feil forekommet whiles etting arkivkommentaren."
#elif defined(PROG_LANGUAGE_POLISH)
"Bd przy ustawianiu komentarza archiwum."
#else
"Error setting archive comment."
#endif
		;
		break;
	    }
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario Fijo Fall"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rgler Le Commentaire Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Feste Bemerkung Hat Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Regolato Il Commento Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vast Opmerking Verzuimde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comentrio Fixo Fracassou"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fast Comment Failed"
#elif defined(PROG_LANGUAGE_POLISH)
"Bd ustawiania komentarza"
#else
"Set Comment Failed"
#endif
		, mesg,
		NULL,
		d->toplevel
	    );
	}
}

/*
 *	OK Button callback.
 */
static void EDVArchiveInfoOKCB(GtkWidget *widget, gpointer data)
{
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	EDVArchiveInfoUnmap(d);
	EDVArchiveInfoResetHasChanges(d, FALSE);
}


/*
 *	Inserts the contents in the file specified by path into the
 *	comment text.
 */
static void EDVArchiveInfoInsertCommentFile(
	edv_archive_info_dlg_struct *d, const gchar *path
)
{
	FILE *fp;
	GtkText *text;

	if((d == NULL) || STRISEMPTY(path))
	    return;

	text = (GtkText *)d->comment_text;
	if(text == NULL)
	    return;

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp != NULL)
	{
	    gulong block_size;
	    guint8 *buf;
	    struct stat stat_buf;

	    gtk_text_freeze(text);

	    if(fstat(fileno(fp), &stat_buf))
	    {
		block_size = 0l;
	    }
	    else
	    {
		block_size = (gulong)stat_buf.st_blksize;
	    }

	    /* Allocate the read buffer */
	    buf = (block_size > 0l) ?
		(guint8 *)g_malloc(block_size * sizeof(guint8)) : NULL;
	    if(buf != NULL)
	    {
		gint bytes_read;
		gchar *s, *s_end;

		/* Begin reading the file one block at a time */
		while(!feof(fp))
		{
		    /* Read the next block from the file */
		    bytes_read = (gint)fread(
			buf, sizeof(guint8), (size_t)block_size, fp
		    );
		    if(bytes_read <= 0)
			break;

		    /* Convert non-printable characters into spaces */
		    s = (gchar *)buf;
		    s_end = s + bytes_read;
		    while(s < s_end)
		    {
			/* Not a printable character? */
			if(!isprint((int)(*s)))
			{
			    /* Exclude newlines */
			    if(*s != '\n')
				*s = ' ';
			}
			s++;
		    }

		    /* Insert this block to the comment text */
		    gtk_text_insert(
			text, NULL, NULL, NULL,
			(const gchar *)buf, bytes_read
		    );
		}

		/* Delete the read buffer */
		g_free(buf);
	    }
	    else
	    {
		gint c;
		guint8 buf[1];

		while(!feof(fp))
		{
		    /* Read the next character from the file */
		    c = (gint)fgetc(fp);
		    if((int)c == EOF)
			break;

		    /* Convert non-printable characters into spaces */
		    if(!isprint((int)c) && (c != '\n'))
			c = ' ';

		    buf[0] = c;

		    /* Insert this character to the comment text */
		    gtk_text_insert(
			text, NULL, NULL, NULL,
			(const gchar *)buf, 1
		    );
		}
	    }

	    /* Close the file */
	    fclose(fp);

	    gtk_text_thaw(text);

	    EDVArchiveInfoResetHasChanges(d, TRUE);
	}
	else
	{
	    const gint error_code = (gint)errno;
	    gchar *s, *error_msg = STRDUP(g_strerror(error_code));
	    if(error_msg != NULL)
	    {
		*error_msg = (gchar)toupper((int)(*error_msg));
		s = g_strdup_printf(
"%s:\n\
\n\
    %s",
		    error_msg,
		    path
		);
		EDVPlaySoundError(d->core);
		CDialogSetTransientFor(d->toplevel);
		CDialogGetResponse(
		    "Insert File Error",
		    s, NULL,
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK 
		);
		CDialogSetTransientFor(NULL);
		g_free(s);
		g_free(error_msg);
	    }
	}
}


/*
 *	Creates a new Archive Info Dialog.
 *
 *	If arch_obj is not NULL then the archive object information will
 *	be loaded.
 */
edv_archive_info_dlg_struct *EDVArchiveInfoNew(
	edv_core_struct *core,
	const gchar *arch_obj,
	const gchar *password,
	GtkWidget *ref_toplevel
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *s, *s2, *comment = NULL;
	gint	total_objects = 0,
		file_objects = 0,
		directory_objects = 0,
		link_objects = 0,
		other_objects = 0;
	gulong	files_size = 0l,
		directories_size = 0l,
		links_size = 0l,
		others_size = 0l,
		uncompressed_size = 0l;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GtkRcStyle *rcstyle;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *sb, *parent, *parent2, *parent3, *parent4, *parent5;
	GtkEditable *editable;
	GtkText *text;
	pie_chart_struct *pc;
	edv_object_struct *obj;
	struct stat lstat_buf;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(g_malloc0(
	    sizeof(edv_archive_info_dlg_struct)
	));
	if((d == NULL) || (core == NULL))
	{
	    g_free(d);
	    return(NULL);
	}

	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->processing = FALSE;
	d->busy_count = 0;
	d->freeze_count = 0;
	d->core = core;

	d->busy_cur = gdk_cursor_new(GDK_WATCH);
	d->text_cur = gdk_cursor_new(GDK_XTERM);

	d->arch_obj = STRDUP(arch_obj);
	d->has_changes = FALSE;


	/* Get archive object statistics */
	obj = EDVObjectNew();
	EDVObjectSetPath(obj, d->arch_obj);
	if(!lstat((const char *)arch_obj, &lstat_buf))
	    EDVObjectSetStat(obj, &lstat_buf);
	EDVObjectUpdateLinkFlags(obj);
	/* Get medium pixmap and mask */
	EDVMatchObjectIcon(
	    core->device, core->total_devices,
	    core->mimetype, core->total_mimetypes,
	    obj->type,
	    obj->full_path,
	    EDV_OBJECT_IS_LINK_VALID(obj),
	    obj->permissions,
	    1,			/* Medium sized icons */
	    &pixmap, &mask,
	    NULL, NULL,
	    NULL, NULL,
	    NULL, NULL
	);
	/* Get number of objects and total size */
	if(d->arch_obj != NULL)
	{
	    gulong this_obj_size;
	    GList *glist, *obj_list = EDVArchStatList(
		core, d->arch_obj,
		NULL,			/* Get all the objects */
		NULL,			/* No filter */
		password,
		NULL, NULL		/* No progress callback */
	    );
	    edv_archive_object_struct *obj;
	    for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		total_objects++;
		switch(obj->type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
		    break;
		  case EDV_OBJECT_TYPE_FILE:
		    file_objects++;
		    this_obj_size = obj->size + (
			(obj->full_path != NULL) ?
			STRLEN(obj->full_path) : STRLEN(obj->name)
		    );
		    files_size += this_obj_size;
		    uncompressed_size += this_obj_size;
		    break;
		  case EDV_OBJECT_TYPE_DIRECTORY:
		    directory_objects++;
		    this_obj_size = obj->size + (
			(obj->full_path != NULL) ?
			STRLEN(obj->full_path) : STRLEN(obj->name)
		    );
		    directories_size += this_obj_size;
		    uncompressed_size += this_obj_size;
		    break;
		  case EDV_OBJECT_TYPE_LINK:
		    link_objects++;
		    this_obj_size = obj->size + (
			(obj->full_path != NULL) ?
			STRLEN(obj->full_path) : STRLEN(obj->name)
		    ) + STRLEN(obj->link_value);
		    links_size += this_obj_size;
		    uncompressed_size += this_obj_size;
		    break;
		  default:
		    other_objects++;
		    this_obj_size = obj->size + (
			(obj->full_path != NULL) ?
			STRLEN(obj->full_path) : STRLEN(obj->name)
		    );
		    others_size += this_obj_size;
		    uncompressed_size += this_obj_size;
		    break;
		}

		EDVArchObjectDelete(obj);
	    }
	    g_list_free(obj_list);

	    /* Get the comment */
	    comment = EDVArchCommentGet(core, d->arch_obj);
	    uncompressed_size += STRLEN(comment);
	}


	d->freeze_count--;


	/* Begin creating widgets */

	/* Create toplevel */
	d->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	s = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario & La Estadstica"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commenter & La Statistique"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bemerkung & Statistik"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Commento & La Statistica"
#elif defined(PROG_LANGUAGE_DUTCH)
"Opmerking & Statistieken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Comentrio & Estatstica"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommentar & Statistics"
#elif defined(PROG_LANGUAGE_POLISH)
"Komentarz i statystyki archiwum"
#else
"Archive Comment & Statistics"
#endif
	    ": %s", obj->name
	);
	gtk_window_set_title(GTK_WINDOW(w), s);
	g_free(s);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
/*	    GUISetWMIcon(window, (guint8 **)archiver_48x48_xpm); */
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
	    gtk_window_set_transient_for(
		GTK_WINDOW(w), GTK_WINDOW(ref_toplevel)
	    );
	}
	parent = w;

	/* Main vbox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Vbox for information and statistics */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;


	/* Hbox to separate two columns */
	w = gtk_hbox_new(FALSE, 20);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Hbox for icon and name */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Icon */
	if(pixmap != NULL)
	{
	    d->icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}
	/* Label */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
#endif
	);
	s = EDVShortenPath(obj->name, 40);
	d->name_label = w = gtk_label_new(s);
	g_free(s);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle)


	/* Location hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#elif defined(PROG_LANGUAGE_POLISH)
"cieka"
#else
"Location"
#endif
	    ":"
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle)
	/* Location Label */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	s2 = g_dirname(obj->full_path);
	s = EDVShortenPath(s2, 45);
	d->location_label = w = gtk_label_new(s);
	g_free(s);
	g_free(s2);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle)


	/* Hbox to separate columns for object and size break-down */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Table for object and size break-down labels */
	w = gtk_table_new(5, 3, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(w), border_minor);
	gtk_table_set_col_spacings(GTK_TABLE(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

#define CREATE_LABEL(r,n,i,b)	{				\
 GtkAttachOptions	x_attach_opt = 0,			\
			y_attach_opt = 0;			\
 guint	x_pad = 0, y_pad = 0;					\
								\
 /* Alignment for name label */					\
 w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 100, -1);				\
 gtk_table_attach(						\
  GTK_TABLE(parent4), w,					\
  0, 1, (r), (r) + 1,						\
  x_attach_opt, y_attach_opt, x_pad, y_pad			\
 );								\
 gtk_widget_show(w);						\
 parent5 = w;							\
 /* Name label */						\
 w = gtk_label_new((n));					\
 gtk_container_add(GTK_CONTAINER(parent5), w);			\
 gtk_widget_modify_style(w, rcstyle);				\
 gtk_widget_show(w);						\
								\
 /* Alignment for quantity label */				\
 w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 50, -1);				\
 gtk_table_attach(						\
  GTK_TABLE(parent4), w,					\
  1, 2, (r), (r) + 1,						\
  x_attach_opt, y_attach_opt, x_pad, y_pad			\
 );								\
 gtk_widget_show(w);						\
 parent5 = w;							\
 /* Quantity label */						\
 s = g_strdup_printf("%i", (i));				\
 w = gtk_label_new(s);						\
 g_free(s);							\
 gtk_container_add(GTK_CONTAINER(parent5), w);			\
 gtk_widget_modify_style(w, rcstyle);                           \
 gtk_widget_show(w);                                            \
								\
 /* Alignmet for size label */					\
 w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 150, -1);				\
 gtk_table_attach(						\
  GTK_TABLE(parent4), w,					\
  2, 3, (r), (r) + 1,						\
  x_attach_opt, y_attach_opt, x_pad, y_pad			\
 );								\
 gtk_widget_show(w);						\
 parent5 = w;							\
 /* Size label */						\
 s = g_strdup_printf(						\
  "%s byte%s",							\
  EDVGetObjectSizeStr(core, (b)),				\
  ((b) == 1) ? "" : "s"						\
 );								\
 w = gtk_label_new(s);						\
 g_free(s);							\
 gtk_container_add(GTK_CONTAINER(parent5), w);			\
 gtk_widget_modify_style(w, rcstyle);				\
 gtk_widget_show(w);						\
}

	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
#if defined(PROG_LANGUAGE_SPANISH)
	CREATE_LABEL(0, "Los Archivos:", file_objects, files_size);
	CREATE_LABEL(1, "Las Guas:", directory_objects, directories_size);
	CREATE_LABEL(2, "Las Conexiones:", link_objects, links_size);
	CREATE_LABEL(3, "Otros Objetos:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_FRENCH)
	CREATE_LABEL(0, "Fichiers:", file_objects, files_size);
	CREATE_LABEL(1, "Annuaires:", directory_objects, directories_size);
	CREATE_LABEL(2, "Liens:", link_objects, links_size);
	CREATE_LABEL(3, "Autres Objets:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_GERMAN)
	CREATE_LABEL(0, "Akten:", file_objects, files_size);
	CREATE_LABEL(1, "Verzeichnisse:", directory_objects, directories_size);
	CREATE_LABEL(2, "Kettenglieder:", link_objects, links_size);
	CREATE_LABEL(3, "Andere Objekte:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_ITALIAN)
	CREATE_LABEL(0, "I File:", file_objects, files_size);
	CREATE_LABEL(1, "Gli Elenchi:", directory_objects, directories_size);
	CREATE_LABEL(2, "Le Maglie:", link_objects, links_size);
	CREATE_LABEL(3, "L'Altro Obbietta:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_DUTCH)
	CREATE_LABEL(0, "Dossiers:", file_objects, files_size);
	CREATE_LABEL(1, "Gidzen:", directory_objects, directories_size);
	CREATE_LABEL(2, "Schakels:", link_objects, links_size);
	CREATE_LABEL(3, "Anderze Voorwerpen:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	CREATE_LABEL(0, "Os Arquivos:", file_objects, files_size);
	CREATE_LABEL(1, "Os Guias:", directory_objects, directories_size);
	CREATE_LABEL(2, "Os Elos:", link_objects, links_size);
	CREATE_LABEL(3, "Outro Ope-se:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	CREATE_LABEL(0, "Arkiver:", file_objects, files_size);
	CREATE_LABEL(1, "Kataloger:", directory_objects, directories_size);
	CREATE_LABEL(2, "Ledd:", link_objects, links_size);
	CREATE_LABEL(3, "Annen Object:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_POLISH)
	CREATE_LABEL(0, "Pliki:", file_objects, files_size);
	CREATE_LABEL(1, "Katalogi:", directory_objects, directories_size);
	CREATE_LABEL(2, "Odnoniki:", link_objects, links_size);
	CREATE_LABEL(3, "Inne obiekty:", other_objects, others_size);
#else
	CREATE_LABEL(0, "Files:", file_objects, files_size);
	CREATE_LABEL(1, "Directories:", directory_objects, directories_size);
	CREATE_LABEL(2, "Links:", link_objects, links_size);
	CREATE_LABEL(3, "Misc Objects:", other_objects, others_size);
#endif
	GTK_RC_STYLE_UNREF(rcstyle)

	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-bold-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-bold-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
#if defined(PROG_LANGUAGE_SPANISH)
	CREATE_LABEL(4, "El Suma:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_FRENCH)
	CREATE_LABEL(4, "Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_GERMAN)
	CREATE_LABEL(4, "Summe:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_ITALIAN)
	CREATE_LABEL(4, "Il Totale:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_DUTCH)
	CREATE_LABEL(4, "Totaal:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	CREATE_LABEL(4, "O Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	CREATE_LABEL(4, "Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_POLISH)
	CREATE_LABEL(4, "Razem:", total_objects, uncompressed_size);
#else
	CREATE_LABEL(4, "Total:", total_objects, uncompressed_size);
#endif
	GTK_RC_STYLE_UNREF(rcstyle)

#undef CREATE_LABEL



	/* Right column vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Size pie chart */
	if(obj != NULL)
	{
	    gchar *s2;
	    GdkColor *c, uncompressed_size_color, current_size_color;

	    /* Uncompressed Size */
	    GtkAdjustment *adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, (gfloat)MAX(uncompressed_size, obj->size),
		0.0f, 0.0f, 0.0f
	    );
	    c = &uncompressed_size_color;
	    GDK_COLOR_SET_COEFF(c, 0.0f, 0.0f, 1.0f);
	    s = g_strdup_printf(
		"%s bytes",
		EDVGetObjectSizeStr(
		    core, 
		    MAX(uncompressed_size, obj->size)
		)
	    );
	    s2 = g_strdup_printf(
		"%.0f%% Compressed",
		CLIP(100.0f - (((uncompressed_size > 0) ?
		    (gfloat)obj->size / (gfloat)uncompressed_size : 1.0f
		) * 100.0f), 0.0f, 100.0f)
	    );

	    /* Create Pie Chart */
	    d->piechart = pc = PieChartNew(
		adj, c, 110, 70,
#if defined(PROG_LANGUAGE_SPANISH)
		"El Tamao", s2,
		"Ensanchado:", s
#elif defined(PROG_LANGUAGE_FRENCH)
		"Taille", s2,
		"Augment:", s
#elif defined(PROG_LANGUAGE_GERMAN)
		"Gre", s2,
		"Ausgebreitet:", s
#elif defined(PROG_LANGUAGE_ITALIAN)
		"La Misura", s2,
		"Allargato:", s
#elif defined(PROG_LANGUAGE_DUTCH)
		"Maat", s2,
		"Uitgebreide:", s
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"O Tamanho", s2,
		"Expandido:", s
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Strrelse", s2,
		"Utvidet:", s
#elif defined(PROG_LANGUAGE_POLISH)
		"Rozmiar", s2,
		"Niezkompresowany:", s
#else
		"Size", s2,		/* Header & Footer */
		"Uncompressed:", s
#endif
	    );
	    g_free(s);
	    g_free(s2);
	    gtk_object_unref(GTK_OBJECT(adj));
	    w = pc->toplevel;
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Current Size */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, (gfloat)obj->size,
		0.0f, 0.0f, 0.0f
	    );
	    c = &current_size_color;
	    GDK_COLOR_SET_COEFF(c, 1.0f, 0.0f, 1.0f);
	    s = g_strdup_printf(
		"%s bytes",
		EDVGetObjectSizeStr(core, obj->size)
	    );
	    PieChartValueAdd(
		pc, adj, c,
#if defined(PROG_LANGUAGE_SPANISH)
"Comprimido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Serr"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zusammengedrckt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Compresso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Samengedrukte"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comprimido"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Komprimert"
#elif defined(PROG_LANGUAGE_POLISH)
"Zkompresowany"
#else
"Compressed"
#endif
		":", s
	    );
	    g_free(s);
	    gtk_object_unref(GTK_OBJECT(adj));
	}


	/* Comments Frame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commentaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bemerkung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Commento"
#elif defined(PROG_LANGUAGE_DUTCH)
"Opmerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Comentrio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommentar"
#elif defined(PROG_LANGUAGE_POLISH)
"Komentarz"
#else
"Comment"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Table for comments text box and scroll bars */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Comments text box */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	d->comment_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
	gtk_widget_set_usize(w, 520, 150);
	text->default_tab_width = 8;
	gtk_text_set_editable(text, TRUE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
	    GTK_TABLE(parent3), w,
	    0, 1, 0, 1,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoAnyChangedCB), d
	);
	gtk_widget_realize(w);
	gtk_widget_modify_style(w, rcstyle);
	gdk_window_set_cursor(text->text_area, d->text_cur);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);
	/* Vertical GtkScrollBar */
	sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
	gtk_table_attach(
	    GTK_TABLE(parent3), sb,
	    1, 2, 0, 1,
	    GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	gtk_widget_show(sb);
	text = GTK_TEXT(d->comment_text);
	/* Set the comment */
	if(!STRISEMPTY(comment))
	{
	    gtk_text_freeze(text);
	    gtk_text_insert(text, NULL, NULL, NULL, comment, -1);
	    gtk_text_thaw(text);
	}

	/* Hbox for set comment button */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Set Comment Button */
	d->comment_set_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"El Conjunto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Srie"
#elif defined(PROG_LANGUAGE_GERMAN)
"Satz"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Serie"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Jogo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sett"
#elif defined(PROG_LANGUAGE_POLISH)
"Ustaw"
#else
"Set"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoSetCommentCB), d
	);
	gtk_widget_show(w);


	/* Hbox for comment utils buttons */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Paste Comment Button */
	d->comment_paste_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_paste_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoPasteCommentCB), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"La Pasta"
#elif defined(PROG_LANGUAGE_FRENCH)
"Pte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Paste"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Pasta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plakmiddel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Pasta"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Masse"
#elif defined(PROG_LANGUAGE_POLISH)
"Wklej"
#else
"Paste"
#endif
	);
	gtk_widget_show(w);
	/* Insert Comment File Button */
	d->comment_insert_file_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoInsertCommentFileCB), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"El Archivo De La Adicin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier D'Insertion"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen Sie Akte Ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il File Di Inserzione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Voeg Dossier In"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Insira Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy Arkiv"
#elif defined(PROG_LANGUAGE_POLISH)
"Wstaw plik"
#else
"Insert File"
#endif
	);
	gtk_widget_show(w);


	/* Delete archive object statistics (no longer needed) */
	EDVObjectDelete(obj);

	/* Delete the comment */
	g_free(comment);
	comment = NULL;


	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);


	d->freeze_count--;


	EDVArchiveInfoUpdateMenus(d);


	return(d);
}

/*
 *	Resets the Archive Info Dialog has_changes marker.
 */
void EDVArchiveInfoResetHasChanges(
	edv_archive_info_dlg_struct *d, gboolean has_changes
)
{
	if(d == NULL)
	    return;

	if(d->has_changes == has_changes)
	    return;

	d->has_changes = has_changes;
	EDVArchiveInfoUpdateMenus(d);
}

/*
 *	Updates the Archive Info Dialog's widgets to reflect current
 *	values.
 */
void EDVArchiveInfoUpdateMenus(edv_archive_info_dlg_struct *d)
{
	gboolean	sensitive,
			has_changes;

	if(d == NULL)
	    return;

	has_changes = d->has_changes;

	/* Set Comment Button */
	sensitive = has_changes;
	GTK_WIDGET_SET_SENSITIVE(d->comment_set_btn, sensitive);
}

/*
 *	Sets the Archive Info Dialog as busy or ready.
 */
void EDVArchiveInfoSetBusy(edv_archive_info_dlg_struct *d, gboolean is_busy)
{
	GdkCursor *cur, *text_cur;
	GtkWidget *w;

	if(d == NULL)
	    return;

	if(is_busy)
	{
	    d->busy_count++;
	    if(d->busy_count > 1)
		return;

	    text_cur = cur = d->busy_cur;
	}
	else
	{
	    d->busy_count--;
	    if(d->busy_count < 0)
		d->busy_count = 0;

	    if(d->busy_count > 0)
		return;

	    cur = NULL;
	    text_cur = d->text_cur;
	}

	/* Set the toplevel's cursor */
	w = d->toplevel;
	if((w != NULL) ? (w->window != NULL) : FALSE)
	    gdk_window_set_cursor(w->window, cur);

	/* Set the comment GtkText's cursor */
	w = d->comment_text;
	if(w != NULL)
	{
	    GtkText *text = GTK_TEXT(w);
	    if(text->text_area != NULL)
		gdk_window_set_cursor(text->text_area, text_cur);
	}

	gdk_flush();
}

/*
 *	Checks if the Archive Info Dialog is mapped.
 */
gboolean EDVArchiveInfoIsMapped(edv_archive_info_dlg_struct *d)
{
	GtkWidget *w = (d != NULL) ? d->toplevel : NULL;
	return((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE);
}

/*
 *	Maps the Archive Info Dialog.
 */
void EDVArchiveInfoMap(edv_archive_info_dlg_struct *d)
{
	GtkWidget *w = (d != NULL) ? d->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_show_raise(w);

	/* Grab focus and default on the OK button */
	w = d->ok_btn;
	if(w != NULL)
	{
	    gtk_widget_grab_focus(w);
	    gtk_widget_grab_default(w);
	}
}

/*
 *	Unmap the Archive Info Dialog.
 */
void EDVArchiveInfoUnmap(edv_archive_info_dlg_struct *d)
{
	GtkWidget *w = (d != NULL) ? d->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}

/*
 *	Deletes the Archive Info Dialog.
 */
void EDVArchiveInfoDelete(edv_archive_info_dlg_struct *d)
{
	if(d == NULL)
	    return;

	g_free(d->arch_obj);
	d->arch_obj = NULL;

	PieChartDelete(d->piechart);
	d->piechart = NULL;

	GTK_WIDGET_DESTROY(d->icon_pm);
	GTK_WIDGET_DESTROY(d->name_label);
	GTK_WIDGET_DESTROY(d->location_label);
	GTK_WIDGET_DESTROY(d->comment_text);
	GTK_WIDGET_DESTROY(d->comment_set_btn);
	GTK_WIDGET_DESTROY(d->comment_paste_btn);
	GTK_WIDGET_DESTROY(d->comment_insert_file_btn);
	GTK_WIDGET_DESTROY(d->ok_btn);

	GTK_WIDGET_DESTROY(d->main_vbox);

	gtk_window_set_modal(GTK_WINDOW(d->toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(d->toplevel), NULL);
	GTK_WIDGET_DESTROY(d->toplevel);

	GTK_ACCEL_GROUP_UNREF(d->accelgrp);

	GDK_CURSOR_DESTROY(d->text_cur);
	GDK_CURSOR_DESTROY(d->busy_cur);

	g_free(d);
}
