/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import java.util.Enumeration;

/**
 *
 * @author brownrigg
 * @version $Id: WMSGetMapRequest.java 5011 2008-04-10 16:53:54Z rick $
 */
public class WMSGetMapRequest  extends WMSRequest {

    public WMSGetMapRequest(WMSHttpServletRequest req) throws WMSServiceException {
        super(req);
        Enumeration params = req.getParameterNames();
        isValid = true;  // assume the best until proven otherwise...
        
        while(params.hasMoreElements()) {
            String param = (String)params.nextElement();
            String value = req.getParameter(param);
            if (value == null) continue;
            
            // nothing fancy here; just a rather tedious enumeration...
            if (param.equalsIgnoreCase(REQUEST)) {
                if (value.equals(GETMAP))
                    hasRequestParam = true;
                continue;
            }
            
            if (param.equalsIgnoreCase(VERSION)) {
                version = value;
                continue;
            }
            
            if (param.equalsIgnoreCase(LAYERS)) {
                layers = req.getParameterValues(param);
                continue;
            }
            
            if (param.equalsIgnoreCase(STYLES)) {
                styles = req.getParameterValues(param);
                continue;
            }
            
            if (param.equalsIgnoreCase(CRS)) {
                crs = value;
                continue;
            }
            
            if (param.equalsIgnoreCase(BBOX)) {
                String[] boxStr = req.getParameterValues(param);
                if (boxStr.length != 4) {
                    errorMsg.append("Invalid BBOX specification; ");
                    isValid = false;
                    continue;
                }
                bbox = new double[4];
                try {
                    for (int i=0; i<4; i++) {
                        bbox[i] = Double.parseDouble(boxStr[i]);
                    }
                } catch (NumberFormatException ex) {
                    errorMsg.append("invalid numeric values in BBOX specification; ");
                    isValid = false;
                    continue;
                }
                if (bbox[XMIN] >= bbox[XMAX] || bbox[YMIN] >= bbox[YMAX]) {
                    errorMsg.append("BBOX min(s) greater than max(s); ");
                    isValid = false;
                }
                
                continue;
            }
            
            if (param.equalsIgnoreCase(WIDTH)) {
                try {
                    width = Integer.parseInt(value);
                    if (width <= 0) {
                        errorMsg.append("WIDTH param must be greater than 0; ");
                        isValid = false;
                    }
                } catch (NumberFormatException ex) {
                    errorMsg.append("Invalid numerical value for WIDTH; ");
                    isValid = false;
                }
                continue;
            }
            
            if (param.equalsIgnoreCase(HEIGHT)) {
                try {
                    height = Integer.parseInt(value);
                    if (height <= 0) {
                        errorMsg.append("HEIGHT param must be greater than 0; ");
                        isValid = false;
                    }
                } catch (NumberFormatException ex) {
                    errorMsg.append("Invalid numerical value for WIDTH; ");
                    isValid = false;
                }
                continue;
            }
            
            if (param.equalsIgnoreCase(FORMAT)) {
                format = value;
                continue;
            }
            
            if (param.equalsIgnoreCase(TRANSPARENT)) {
                if (value.equals("FALSE")) transparent = false;
                else if (value.equals("TRUE")) transparent = true;
                else
                    // bogus value, but don't consider it invalid request...
                    errorMsg.append("invalid TRANSPARENT value: " + value + ", defaulting to false; ");
                continue;
            }
            
            if (param.equalsIgnoreCase(BGCOLOR)) {
                bgColor = value;
                continue;
            }
            
            if (param.equalsIgnoreCase(EXCEPTIONS)) {
                exceptions = value;
                continue;
            }
            
            if (param.equalsIgnoreCase(TIME)) {
                time = value;
                continue;
            }
            
            if (param.equalsIgnoreCase(ELEVATION)) {
                try {
                    elevation = Double.parseDouble(value);
                } catch (NumberFormatException ex) {
                    errorMsg.append("Invalid numerical value for elevation: " + value + ", ignored; ");
                }
                continue;
            }
            
            //******************************************************************
            // Note the spec allows for optional "other sample dimensions"
            // Not implemented here.
            // *****************************************************************
        }
        
        // Check that all the mandatory parameters are present...
        if (version == null)
            missingParam(VERSION);
        
        if (!hasRequestParam)
            missingParam(REQUEST);
        
        if (layers == null)
            missingParam(LAYERS);
        
        if (styles == null)
            missingParam(STYLES);

        /***************
         *  TODO:  We'll relax this requirement for now, as i) its not used by this server
         * at present, and ii) its causing some of our client-apps grief.
         *
        if (crs == null)
            missingParam(CRS);
         **************/

        if (format == null)
            missingParam(FORMAT);
        
        if (bbox == null)
            missingParam(BBOX);
        
        if (width <= 0)
            missingParam(WIDTH);
        
        if (height <= 0)
            missingParam(HEIGHT);
        
        // A final integrity check...
        if (styles != null && layers != null)
            if (styles.length != layers.length) {
                errorMsg.append("STYLES and LAYERS specs not of same arity; ");
                isValid = false;
            }        
    }
    
    public boolean isValid() {  return isValid;   }
    
    public String getParseReport() {
        return errorMsg.toString();
    }
    
    public String toString() {
        StringBuffer str = new StringBuffer();
        str.append("GetMap: ");
        str.append(" layers(");
        for (int i=0; i<layers.length; i++) {
            if (i > 0) str.append(", ");
            str.append(layers[i]);
        }
        str.append(") styles(");
        for (int i=0; i<styles.length; i++) {
            if (i > 0) str.append(", ");
            str.append(styles[i]);
        }
        str.append(") crs("); str.append(crs);
        str.append(") bbox(");
        for (int i=0; i<bbox.length; i++) {
            if (i > 0) str.append(", ");
            str.append(bbox[i]);
        }
        str.append(") WxH("); str.append(width); str.append("x"); str.append(height);
        str.append(") format("); str.append(format);
        str.append(") transp("); str.append(transparent);
        str.append(")");
        return str.toString();
    }
    
    // Convenience method; this pattern of error reporting is repeated a number of times.
    private void missingParam(String param) {
        errorMsg.append("Missing required ");
        errorMsg.append(param);
        errorMsg.append(" parameter; ");
        isValid = false;
    }
    
    // gettors...
    public String getVersion() { return version; }
    public String[] getLayers() { return layers; }
    public String[] getStyles() { return styles; }
    public String getCRS() { return crs; }
    public double[] getBBox() { return bbox; }
    public double getBBoxXMin() { return bbox[XMIN]; }
    public double getBBoxXMax() { return bbox[XMAX]; }
    public double getBBoxYMin() { return bbox[YMIN]; }
    public double getBBoxYMax() { return bbox[YMAX]; }
    public int getWidth() { return width; }
    public int getHeight() { return height; }
    public String getFormat() { return format; }
    public boolean isTransparent() { return transparent; }
    public String getBGColor() { return bgColor; }
    public String getExceptions() { return exceptions; }
    public String getTime() { return time; }
    public double getElevation() { return elevation; }
    
    // data members...
    private boolean isValid;
    private boolean hasRequestParam = false;
    private String version = null;
    private String[] layers = null;
    private String[] styles = null;
    private String crs = null;
    private double[] bbox = null;
    private int width = -1;
    private int height = -1;
    private String format = null;
    private boolean transparent = false;
    private String bgColor = null;
    private String exceptions = null;
    private String time = null;
    private double elevation = -Double.MAX_VALUE;
    private StringBuffer errorMsg = new StringBuffer();

    // request-specific parameter names...
    private static final String LAYERS = "LAYERS";
    private static final String STYLES = "STYLES";
    private static final String CRS = "CRS";
    private static final String BBOX = "BBOX";
    private static final String WIDTH = "WIDTH";
    private static final String HEIGHT = "HEIGHT";
    private static final String TRANSPARENT = "TRANSPARENT";
    private static final String BGCOLOR = "BGCOLOR";
    private static final String TIME = "TIME";
    private static final String ELEVATION = "ELEVATION";

    // indices into bbox
    private static int XMIN = 0;
    private static int YMIN = 1;
    private static int XMAX = 2;
    private static int YMAX = 3;
}
