/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.theme;

import java.awt.Color;
import java.util.HashMap;
import java.util.Map;

import org.jvnet.substance.color.ShiftColorScheme;

/**
 * Shift theme. A shift theme is based on some original theme that is shifted
 * towards some color.
 * 
 * @author Kirill Grouchnikov
 * @see ShiftColorScheme
 */
public class SubstanceShiftTheme extends SubstanceWrapperTheme {
	/**
	 * The background shift factor for <code>this</code> theme.
	 */
	private double backgroundShiftFactor;

	/**
	 * The background shift color for <code>this</code> theme.
	 */
	private Color backgroundShiftColor;

	/**
	 * The foreground shift factor for <code>this</code> theme.
	 */
	private double foregroundShiftFactor;

	/**
	 * The foreground shift color for <code>this</code> theme.
	 */
	private Color foregroundShiftColor;

	/**
	 * Creates a new shift theme. <b>Do not</b> use this constructor directly,
	 * use {@link SubstanceTheme#shift(Color, double)} instead.
	 * 
	 * @param substanceTheme
	 *            The original theme.
	 * @param backgroundShiftColor
	 *            Shift color for background colors.
	 * @param backgroundShiftFactor
	 *            Shift factor for background colors.
	 * @param foregroundShiftColor
	 *            Shift color for foreground colors.
	 * @param foregroundShiftFactor
	 *            Shift factor for foreground colors.
	 */
	SubstanceShiftTheme(SubstanceTheme substanceTheme,
			Color backgroundShiftColor, double backgroundShiftFactor,
			Color foregroundShiftColor, double foregroundShiftFactor) {
		super(substanceTheme, new ShiftColorScheme(substanceTheme
				.getColorScheme(), backgroundShiftColor, backgroundShiftFactor,
				foregroundShiftColor, foregroundShiftFactor, true), "Shift "
				+ substanceTheme.getDisplayName() + " to ["
				+ backgroundShiftColor + "] "
				+ (int) (100 * backgroundShiftFactor) + "% * ["
				+ foregroundShiftColor + "]"
				+ (int) (100 * foregroundShiftFactor) + "%", substanceTheme
				.getKind());
		this.backgroundShiftColor = backgroundShiftColor;
		this.backgroundShiftFactor = backgroundShiftFactor;
		this.foregroundShiftColor = foregroundShiftColor;
		this.foregroundShiftFactor = foregroundShiftFactor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDefaultTheme()
	 */
	@Override
	public SubstanceTheme getDefaultTheme() {
		if (this.defaultTheme == null)
			this.defaultTheme = this.originalTheme.getDefaultTheme().shift(
					this.backgroundShiftColor, this.backgroundShiftFactor,
					this.foregroundShiftColor, this.foregroundShiftFactor);
		return this.defaultTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDisabledTheme()
	 */
	@Override
	public SubstanceTheme getDisabledTheme() {
		if (this.disabledTheme == null)
			this.disabledTheme = this.originalTheme.getDisabledTheme()
					.shift(this.backgroundShiftColor,
							this.backgroundShiftFactor / 2.0,
							this.foregroundShiftColor,
							this.foregroundShiftFactor / 2.0);
		return this.disabledTheme;
	}

	/**
	 * Cache of shifted themes.
	 */
	protected static Map<String, SubstanceTheme> shiftedCache = new HashMap<String, SubstanceTheme>();

	/**
	 * Returns a shifted theme. This method is for internal use only.
	 * 
	 * @param orig
	 *            The original theme.
	 * @param backgroundShiftColor
	 *            Shift color for the background theme colors. May be
	 *            <code>null</code> - in this case, the background theme
	 *            colors will not be shifted.
	 * @param backgroundShiftFactor
	 *            Shift factor for the background theme colors. If the shift
	 *            color for the background theme colors is <code>null</code>,
	 *            this value is ignored.
	 * @param foregroundShiftColor
	 *            Shift color for the foreground theme colors. May be
	 *            <code>null</code> - in this case, the foreground theme
	 *            colors will not be shifted.
	 * @param foregroundShiftFactor
	 *            Shift factor for the foreground theme colors. If the shift
	 *            color for the foreground theme colors is <code>null</code>,
	 *            this value is ignored.
	 * @return Shifted theme.
	 */
	public static synchronized SubstanceTheme getShiftedTheme(
			SubstanceTheme orig, Color backgroundShiftColor,
			double backgroundShiftFactor, Color foregroundShiftColor,
			double foregroundShiftFactor) {
		String key = orig.getDisplayName() + ":" + backgroundShiftColor + ":"
				+ backgroundShiftFactor + ":" + foregroundShiftColor + ":"
				+ foregroundShiftFactor;
		SubstanceTheme result = shiftedCache.get(key);
		if (result == null) {
			result = orig.shift(backgroundShiftColor, backgroundShiftFactor,
					foregroundShiftColor, foregroundShiftFactor);
			shiftedCache.put(key, result);
		}
		return result;
	}
}
