/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.BasicBorders;
import javax.swing.plaf.basic.BasicPasswordFieldUI;
import javax.swing.text.*;

import org.jvnet.lafwidget.animation.FadeStateListener;
import org.jvnet.substance.painter.text.SubstanceTextPainter;
import org.jvnet.substance.text.SubstanceFieldView;
import org.jvnet.substance.text.SubstanceHighlighter;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.*;

/**
 * UI for password fields in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstancePasswordFieldUI extends BasicPasswordFieldUI {
	/**
	 * Listener for fade animations.
	 */
	protected FadeStateListener substanceFadeStateListener;

	/**
	 * The associated password field.
	 */
	protected JPasswordField passwordField;

	/**
	 * Property change listener.
	 */
	protected PropertyChangeListener substancePropertyChangeListener;

	/**
	 * Custom password view.
	 * 
	 * @author Kirill Grouchnikov
	 */
	private static class SubstancePasswordView extends SubstanceFieldView {
		/**
		 * The associated password field.
		 */
		private JPasswordField field;

		/**
		 * Simple constructor.
		 * 
		 * @param field
		 *            The associated password field.
		 * @param element
		 *            The element
		 */
		public SubstancePasswordView(JPasswordField field, Element element) {
			super(element);
			this.field = field;
		}

		protected int drawEchoCharacter(Graphics g, int x, int y, char c,
				boolean isSelected) {
			Container container = this.getContainer();

			Graphics2D graphics = (Graphics2D) g;
			graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
					RenderingHints.VALUE_ANTIALIAS_ON);

			JPasswordField field = (JPasswordField) container;

			int fontSize = SubstanceSizeUtils.getComponentFontSize(this.field);
			int dotDiameter = SubstanceSizeUtils
					.getPasswordDotDiameter(fontSize);
			int dotGap = SubstanceSizeUtils.getPasswordDotGap(fontSize);
			ComponentState state = // isSelected ? ComponentState.SELECTED
			(field.isEnabled() ? ComponentState.DEFAULT
					: ComponentState.DISABLED_UNSELECTED);
			SubstanceTheme theme = SubstanceThemeUtilities
					.getTheme(field, state);
			Color topColor = isSelected ? theme.getSelectionForegroundColor()
					: theme.getForegroundColor();
			Color bottomColor = topColor.brighter();
			graphics.setPaint(new GradientPaint(x, y - dotDiameter, topColor,
					x, y, bottomColor));
			int echoPerChar = SubstanceCoreUtilities.getEchoPerChar(field);
			for (int i = 0; i < echoPerChar; i++) {
				graphics.fillOval(x + dotGap / 2, y - dotDiameter, dotDiameter,
						dotDiameter);
				x += (dotDiameter + dotGap);
			}
			return x;
		}

		protected int getEchoCharAdvance() {
			int fontSize = SubstanceSizeUtils.getComponentFontSize(this.field);
			int dotDiameter = SubstanceSizeUtils
					.getPasswordDotDiameter(fontSize);
			int dotGap = SubstanceSizeUtils.getPasswordDotGap(fontSize);
			int echoPerChar = SubstanceCoreUtilities.getEchoPerChar(field);
			return echoPerChar * (dotDiameter + dotGap);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.text.PlainView#drawSelectedText(java.awt.Graphics,
		 *      int, int, int, int)
		 */
		@Override
		protected int drawSelectedText(Graphics g, final int x, final int y,
				int p0, int p1) throws BadLocationException {
			Container c = getContainer();
			if (c instanceof JPasswordField) {
				JPasswordField f = (JPasswordField) c;
				if (!f.echoCharIsSet()) {
					return super.drawSelectedText(g, x, y, p0, p1);
				}
				final int n = p1 - p0;
				final char echoChar = f.getEchoChar();
				SubstanceTextPainter.BackgroundPaintingCallback echoCharsCallback = new SubstanceTextPainter.BackgroundPaintingCallback() {
					public void paintBackground(Graphics g) {
						int currPos = x;
						for (int i = 0; i < n; i++) {
							currPos = drawEchoCharacter(g, currPos, y,
									echoChar, true);
						}
					}
				};
				SubstanceTextPainter textPainter = SubstanceLookAndFeel
						.getCurrentTextPainter();
				if (textPainter.needsBackgroundImage()) {
					textPainter.attachCallback(echoCharsCallback);
				} else {
					echoCharsCallback.paintBackground(g);
				}
				return x + n * getEchoCharAdvance();
			}
			return x;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.text.PlainView#drawUnselectedText(java.awt.Graphics,
		 *      int, int, int, int)
		 */
		@Override
		protected int drawUnselectedText(Graphics g, final int x, final int y,
				int p0, int p1) throws BadLocationException {
			Container c = getContainer();
			if (c instanceof JPasswordField) {
				JPasswordField f = (JPasswordField) c;
				if (!f.echoCharIsSet()) {
					return super.drawUnselectedText(g, x, y, p0, p1);
				}
				final int n = p1 - p0;
				final char echoChar = f.getEchoChar();
				SubstanceTextPainter.BackgroundPaintingCallback echoCharsCallback = new SubstanceTextPainter.BackgroundPaintingCallback() {
					public void paintBackground(Graphics g) {
						int currPos = x;
						for (int i = 0; i < n; i++) {
							currPos = drawEchoCharacter(g, currPos, y,
									echoChar, false);
						}
					}
				};
				SubstanceTextPainter textPainter = SubstanceLookAndFeel
						.getCurrentTextPainter();
				if (textPainter.needsBackgroundImage()) {
					textPainter.attachCallback(echoCharsCallback);
				} else {
					echoCharsCallback.paintBackground(g);
				}
				return x + n * getEchoCharAdvance();
			}
			return x;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.text.View#modelToView(int, java.awt.Shape,
		 *      javax.swing.text.Position.Bias)
		 */
		@Override
		public Shape modelToView(int pos, Shape a, Position.Bias b)
				throws BadLocationException {
			Container c = this.getContainer();
			if (c instanceof JPasswordField) {
				JPasswordField f = (JPasswordField) c;
				if (!f.echoCharIsSet()) {
					return super.modelToView(pos, a, b);
				}

				Rectangle alloc = this.adjustAllocation(a).getBounds();
				int echoPerChar = SubstanceCoreUtilities.getEchoPerChar(f);
				int fontSize = SubstanceSizeUtils
						.getComponentFontSize(this.field);
				int dotWidth = SubstanceSizeUtils
						.getPasswordDotDiameter(fontSize)
						+ SubstanceSizeUtils.getPasswordDotGap(fontSize);

				int dx = (pos - this.getStartOffset()) * echoPerChar * dotWidth;
				alloc.x += dx;
				alloc.width = 1;
				return alloc;
			}
			return null;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.text.View#viewToModel(float, float, java.awt.Shape,
		 *      javax.swing.text.Position.Bias[])
		 */
		@Override
		public int viewToModel(float fx, float fy, Shape a, Position.Bias[] bias) {
			bias[0] = Position.Bias.Forward;
			int n = 0;
			Container c = this.getContainer();
			if (c instanceof JPasswordField) {
				JPasswordField f = (JPasswordField) c;
				if (!f.echoCharIsSet()) {
					return super.viewToModel(fx, fy, a, bias);
				}
				a = this.adjustAllocation(a);
				Rectangle alloc = (a instanceof Rectangle) ? (Rectangle) a : a
						.getBounds();
				int echoPerChar = SubstanceCoreUtilities.getEchoPerChar(f);
				int fontSize = SubstanceSizeUtils
						.getComponentFontSize(this.field);
				int dotWidth = SubstanceSizeUtils
						.getPasswordDotDiameter(fontSize)
						+ SubstanceSizeUtils.getPasswordDotGap(fontSize);
				n = ((int) fx - alloc.x) / (echoPerChar * dotWidth);
				if (n < 0) {
					n = 0;
				} else {
					if (n > (this.getStartOffset() + this.getDocument()
							.getLength())) {
						n = this.getDocument().getLength()
								- this.getStartOffset();
					}
				}
			}
			return this.getStartOffset() + n;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.text.View#getPreferredSpan(int)
		 */
		@Override
		public float getPreferredSpan(int axis) {
			switch (axis) {
			case View.X_AXIS:
				Container c = this.getContainer();
				if (c instanceof JPasswordField) {
					JPasswordField f = (JPasswordField) c;
					if (f.echoCharIsSet()) {
						int echoPerChar = SubstanceCoreUtilities
								.getEchoPerChar(f);
						int fontSize = SubstanceSizeUtils
								.getComponentFontSize(this.field);
						int dotWidth = SubstanceSizeUtils
								.getPasswordDotDiameter(fontSize)
								+ SubstanceSizeUtils
										.getPasswordDotGap(fontSize);
						return echoPerChar * dotWidth
								* this.getDocument().getLength();
					}
				}
			}
			return super.getPreferredSpan(axis);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent c) {
		return new SubstancePasswordFieldUI(c);
	}

	/**
	 * Creates the UI delegate for the specified component (password field).
	 * 
	 * @param c
	 *            Component.
	 */
	public SubstancePasswordFieldUI(JComponent c) {
		super();
		this.passwordField = (JPasswordField) c;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.text.ViewFactory#create(javax.swing.text.Element)
	 */
	@Override
	public View create(Element elem) {
		return new SubstancePasswordView(this.passwordField, elem);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#installListeners()
	 */
	@Override
	protected void installListeners() {
		super.installListeners();

		this.substanceFadeStateListener = new FadeStateListener(
				this.passwordField, null, null);
		this.substanceFadeStateListener.registerListeners(false);

		this.substancePropertyChangeListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if ("font".equals(evt.getPropertyName())) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							passwordField.updateUI();
						}
					});
				}
			}
		};
		this.passwordField
				.addPropertyChangeListener(this.substancePropertyChangeListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#uninstallListeners()
	 */
	@Override
	protected void uninstallListeners() {
		this.substanceFadeStateListener.unregisterListeners();
		this.substanceFadeStateListener = null;

		this.passwordField
				.removePropertyChangeListener(this.substancePropertyChangeListener);
		this.substancePropertyChangeListener = null;

		super.uninstallListeners();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#installDefaults()
	 */
	@Override
	protected void installDefaults() {
		super.installDefaults();
		Border b = this.passwordField.getBorder();
		if (b == null || b instanceof UIResource) {
			Border newB = new BorderUIResource.CompoundBorderUIResource(
					new SubstanceBorder(SubstanceSizeUtils
							.getTextBorderInsets(SubstanceSizeUtils
									.getComponentFontSize(this.passwordField))),
					new BasicBorders.MarginBorder());
			this.passwordField.setBorder(newB);
		}
	}

	// /*
	// * (non-Javadoc)
	// *
	// * @see
	// javax.swing.plaf.basic.BasicTextUI#paintBackground(java.awt.Graphics)
	// */
	// @Override
	// protected void paintBackground(Graphics g) {
	// SubstanceCoreUtilities.paintCompBackground(g, this.passwordField);
	// }

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#createHighlighter()
	 */
	@Override
	protected Highlighter createHighlighter() {
		return new SubstanceHighlighter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#paintSafely(java.awt.Graphics)
	 */
	@Override
	protected void paintSafely(Graphics _g) {
		// // Have to call the super implementation since it sets a
		// // private "painted" flag which affects many other places.
		// // Without this there will be many visual artifacts with
		// // painting the caret and highlights
		// super.paintSafely(_g);
		//
		// SubstanceTextPainter textPainter = SubstanceLookAndFeel
		// .getCurrentTextPainter();
		//
		// SubstanceTextPainter.BackgroundPaintingCallback callback = new
		// SubstanceTextPainter.BackgroundPaintingCallback() {
		// public void paintBackground(Graphics g) {
		// Highlighter highlighter = getComponent().getHighlighter();
		//
		// // paint the background
		// if (getComponent().isOpaque()) {
		// SubstancePasswordFieldUI.this.paintBackground(g);
		// }
		//
		// // paint the highlights
		// if (highlighter != null) {
		// highlighter.paint(g);
		// }
		// }
		// };
		// textPainter.init(passwordField, null, true);
		// if (textPainter.needsBackgroundImage()) {
		// textPainter.setBackgroundFill(passwordField, passwordField
		// .getBackground(), false, 0, 0);
		// textPainter.attachCallback(callback);
		// } else {
		// callback.paintBackground(_g);
		// }
		//
		// // paint the view hierarchy
		// Rectangle alloc = getVisibleEditorRect();
		// if (alloc != null) {
		// getRootView(passwordField).paint(_g, alloc);
		// }
		// textPainter.renderSurface(_g);
		//
		// // paint the caret
		// Caret caret = getComponent().getCaret();
		// if (caret != null) {
		// caret.paint(_g);
		// }
		// Have to call the super implementation since it sets a
		// private "painted" flag which affects many other places.
		// Without this there will be many visual artifacts with
		// painting the caret and highlights
		Graphics2D dummy = (Graphics2D) _g.create(0, 0, 0, 0);
		super.paintSafely(dummy);
		dummy.dispose();

		SubstanceCoreUtilities.paintTextComponent(_g, this.passwordField, this
				.getRootView(this.passwordField), this.getVisibleEditorRect());
	}
}