/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		OSThread.cpp

	Contains:	Thread abstraction implementation

	$Log: OSThread.cpp,v $
	Revision 1.2  1999/02/19 23:06:15  ds
	Created
		
	
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef __MW_
#if __PTHREADS__
#include <pthread.h>
#else
#include <mach/mach.h>
#include <mach/cthreads.h>
#endif
#endif

#include "OSThread.h"
#include "Exception.h"
#include "MyAssert.h"

//
// OSThread.cp
//

OSThread::OSThread(char *threadName)
{
    int strlt = strlen(threadName);
	strlt = (strlt > 31) ? 31 : strlt;
	strncpy(fName, threadName, strlt);
	fName[strlt] = 0;	
   
	fStopRequested = false;
	fStarted = false;
	fRunning = false;
	fCancelThrown = false;
	fDetached = false;
	fJoined =false;
	fBlockStopRequest = false;
	fDoDetach = false;
	
#if DEBUG
	fLocksHeld = 0;
	memset(fLockTagArray, 0, sizeof(fLockTagArray));
#endif
}

OSThread::~OSThread()
{
	fDoDetach = false;//we must JOIN with the thread!
	//send the thread a stop exception
	this->SendStopRequest();
		
	fDoDetach = false;
	if (!fJoined && !fDetached)
		Join();
}

void OSThread::Start()
{
#if __PTHREADS__
	int err = pthread_create((pthread_t*)&fThreadID, NULL, _Entry, (void*)this);
	Assert(err == 0);
#else
	fThreadID = (UInt32)cthread_fork((cthread_fn_t)_Entry, (any_t)this);
#endif
}

void OSThread::Join()
{
	// What we're trying to do is allow the thread we want to delete to complete
	// running. So we wait for it to stop.
	Assert(!fJoined && !fDetached);
	fJoined = true;
#if __PTHREADS__
	void *retVal;
	pthread_join(pthread_self(), &retVal);
#else
	cthread_join((cthread_t)fThreadID);
#endif
}

void OSThread::Entry()
{
	Assert(false);
}


void OSThread::Detach()
{
	Assert(!fDetached && !fJoined);
	fDetached = true;
}

void OSThread::SendStopRequest()
{
	fStopRequested = true;
	if (fDoDetach)
		Detach();
}

void OSThread::CheckForStopRequest()
{
	if (!fBlockStopRequest && fStopRequested && !fCancelThrown)
		ThrowStopRequest();
}


bool OSThread::WillThrowStopRequest()
{
	return !fBlockStopRequest && fStopRequested && !fCancelThrown;
}


void OSThread::ThrowStopRequest()
{
	if (fStopRequested && !fCancelThrown) {
		fCancelThrown = true;
		Throw_(Cancel_E);
	}
}

void OSThread::CallEntry(OSThread* thread) 	// static method
{
	thread->fStarted = true;
	thread->fRunning = true;

	try
	{
		thread->Entry();
	} 
	catch(...)
	{
		Assert(thread->IsStopRequested());
	}

	thread->fRunning = false;
	
	if (thread->fDetached) {
		Assert(!thread->fJoined);
#if __PTHREADS__
		pthread_detach((pthread_t)thread->fThreadID);
#else
		cthread_detach((cthread_t)thread->fThreadID);
#endif
		delete thread;
	}
}


void* OSThread::_Entry(void *inThread)  //static
{
	OSThread* theThread = (OSThread*)inThread;
#if __PTHREADS__
	theThread->fThreadID = (UInt32)pthread_self();
#else
	theThread->fThreadID = (UInt32)cthread_self();
	cthread_set_data(cthread_self(), (any_t)theThread);
	cthread_set_name((cthread_t)theThread->fThreadID, theThread->GetThreadName());
#endif
	OSThread::CallEntry(theThread);
	return NULL;
}

#if DEBUG
void OSThread::IncrementLocksHeld(UInt32 tag)
{
	if ((fLocksHeld * 5) < (kTagArraySize - 5))
		//put this tag into the tag array
		::memcpy(&fLockTagArray[fLocksHeld * 5], &tag, sizeof(tag));
	fLocksHeld++;
}


void OSThread::DecrementLocksHeld(UInt32 tag)
{
	fLocksHeld--;
	//find a matching tag to remove out of the array
	for (UInt32 lockIndex = 0; lockIndex < kTagArraySize; lockIndex+=5)
	{
		if (::memcmp(&fLockTagArray[lockIndex], &tag, sizeof(tag)) == 0)
		{
			::memmove(&fLockTagArray[lockIndex], &fLockTagArray[lockIndex + 5], kTagArraySize - (lockIndex + 5));
			lockIndex = kTagArraySize; 
		}
	}
}

void OSThread::PrintLockTagArray()
{
	for (UInt32 lockIndex = 0; ((lockIndex < kTagArraySize) && (lockIndex < fLocksHeld)); lockIndex++)
		printf("Lock %ld: %s\n", lockIndex, &fLockTagArray[lockIndex * 5]);
}
#endif
