/*
 *   dspalloc.c -- Allocate module step of dspLoadModule 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspbios.h"
#include "dspstruc.h"
#include "dspmem.h"
#include "dsppcmem.h"
#include "dspfixup.h"
#include "dspispos.h"                 /* CH01 Mwave/OS prototypes */
#include "dspalloc.h"
#include "dspgpc.h"
#include "dspipc.h"
#include "dspfxutl.h"
#include "dspparse.h"
#include <math.h>
#include "dspfree.h"
#include "dspstcmp.h"                  /* Function Prototypes               */

#define  hiword(x)     (USHORT)(x>>16)     /* CH01 move upper word of ULONG */
                                           /* to lower word & convert to US */
#define  loword(x)     (USHORT)(x & 0x0000FFFFL) /* CH05 lower word of ULONG */

#define  StrideMax     2047            /* Maximum Stride value              */
#define  HoldMax       31              /* Maximum Hold value                */
#define  HoldShift     11              /* Shift value for table entry       */
#define  StrideHoldNum 32              /* Stride Hold table size            */
#define  DMA_FrameRate 1379            /* DMA windows/sec = ceil(44100/32)  */
#define  PktDefault    40              /*                                   */

ULONG      ulcintcnt = 0;              /*                                   */
ULONG      ulSDMA_CYC = 0;             /*                                   */
LONG       lPktMax = 0;                /*                                   */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateModule                                          */
/*                                                                          */
/* FUNCTION: This procedure allocates resources for tasks in module         */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRMOD          - pointer to module for resource allocation            */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occured                              */
/*    DSP_NO_INSTNS        - No code segments given, can't load             */
/*    DSP_NO_DATA          - No data segments given, can't load             */
/*    DSP_INVALID_OS_LOAD  - Loading an OS without using DSPINIT            */
/*       TDB               - errors                                         */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*    fDSPINIT_FLAG  - set and reset by DspInit                             */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*                                                                          */
/*    OS_Load                                                               */
/*    AllocateTask                                                          */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating resources */
/*    the entire allocation processed will be aborted, and the error code   */
/*    returned.  Some of the tasks may not have been processed upon         */
/*    termination in case of a error.                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateModule(PRMOD prmod)

{
   PRTSK      prtsk;
   RC         ulRC;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateModule entry prmod %x\n",(int)prmod);

   if ((prtsk = prmod->MOD_prtskTail) == NULL)
      return  DSP_NOERROR;

/* TRAVERSE THE MODULE - IE FOR ALL TASKS                                   */

   do {
      prtsk = prtsk->TSK_prtskNext;    /* 1st time, move to head of list    */

   /* CHECK FOR AN OS "TASK"                                                */

      if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_OS)

      /* WE HAVE AN OS, CAN WE PROCESS IT?                                  */

         if (fDSPINIT_FLAG)            /* global variable set and reset     */
                                       /* by DSP INIT                       */
            {

         /* DO THE OS LOAD, CHECK FOR ERRORS                                */

            if ((ulRC = OS_Load(prtsk)) != DSP_NOERROR)
               return  ulRC;
         }

      /* ELSE WE HAVE AN ERROR - TRYING TO LOAD OS WITHOUT USING DSPINIT    */

         else
            return  DSP_INV_OS_LOAD;

   /* WE HAVE A NORMAL TASK, PROCESS IT                                     */

      else

      /* DO THE NORMAL TASK, CHECK FOR ERRORS                               */

         if ((ulRC = AllocateTask(prtsk)) != DSP_NOERROR)
            return  ulRC;

   }  while (prtsk != prmod->MOD_prtskTail);/* end traverse the module      */
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::AllocateModule exit ulRC=0\n");
   return  DSP_NOERROR;
}                                      /* END AllocateModule                */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: OS_Load                                                 */
/*                                                                          */
/* FUNCTION: This procedure loads the OS into the DSP and starts it running.*/
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK          - pointer to task struct holding OS information        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*    DSP_NO_INSTNS  - No code segments given, can't load                   */
/*    DSP_NO_DATA    - No data segments given, can't load                   */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*                                                                          */
/*    AllocSegRes                                                           */
/*    FixupTask                                                             */
/*    DspBIOS_Reset - in DSPBIOS.H                                          */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of loading the os,      */
/*    the entire load process will be aborted, and the error code returned. */
/*    Some of the load may not have been complete upon termination in       */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC OS_Load(PRTSK prtsk)

{
   PRSEG      prseg; 
   PRGPC      prgpc;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   ULONG      ulAddr;
   USHORT     usBuf[1];                /* temp buffer used for BIOS read and*/
                                       
   
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::OS_Load entry prtsk %x\n",(int)prtsk);

   /* EXAMINE ALL THE INSTRUCTION SEGMENTS                                  */

   if ((prseg = prtsk->TSK_prsegITail) == NULL)
      return  DSP_NO_INST;

   do {
      prseg = prseg->SEG_prsegNext;    /* 1st time, move to head of list    */

      /* ALLOCATE RESOURCES FOR THE SEGMENT, CHECK FOR ERRORS               */

      if ((ulRC = AllocSegRes(prseg)) != DSP_NOERROR)
         return  ulRC;

   }  while (prseg != prtsk->TSK_prsegITail);

   /* EXAMINE ALL THE DATA SEGMENTS                                         */

   if ((prseg = prtsk->TSK_prsegDTail) == NULL)
      return  DSP_NO_DATA;

   do {
      prseg = prseg->SEG_prsegNext;    /* 1st time, move to head of list    */

      /* ALLOCATE RESOURCES FOR THE SEGMENT, CHECK FOR ERRORS               */

      if ((ulRC = AllocSegRes(prseg)) != DSP_NOERROR)
         return  ulRC;

   }  while (prseg != prtsk->TSK_prsegDTail);

   /*************************************************************************/
   /* Add an IPC for the Task.                                              */
   /*************************************************************************/

   if (prtsk->TSK_prseg_IPC != NULL) {
      if (ulRC == DSP_NOERROR)
         ulRC = AddIPC(prtsk);
   }

   /* DO FIXUP ON THE OS SEGMENTS, CHECK FOR ERRORS                         */

   if ((ulRC = FixupTask(prtsk)) != DSP_NOERROR)
      return  ulRC;

   /*************************************************************************/
   /* Create the GPC buffer For Each Owner GPC hanging off the GPC list.    */
   /* The GPC control blocks should only be hanging off a Task.             */
   /* The GPC buffer will be allocate from DSP memory.                      */
   /*************************************************************************/

   if ((prgpc = prtsk->TSK_prgpcTail) != NULL) {/* Check for one block      */
      do {
         prgpc = prgpc->GPC_prgpcNext; /* Get next GPC block                */
         if (((prgpc->GPC_prseg->SEG_usflg&SEG_TYPEMASK) == SEG_DATA) &&
            (prgpc->GPC_usflgKind == GPC_OWNER)) {

            if ((ulRC = AllocateGPCBuf(prgpc)) != DSP_NOERROR)
               return  ulRC;
         }

      }  while (prgpc != prtsk->TSK_prgpcTail);
   } /* endif                                                               */

   /* START THE OS RUNNING                                                  */

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspalloc::OS_Load");
         PRINT_DEBUG(EMBEDFILEPTR, "RUN25:  prdsp (0x%X)", prtsk->TSK_prdsp);
         PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", prtsk);
         PRINT_DEBUG_FCLOSE(EMBEDFILEPTR);

         if ((ulRC = DspBIOS_Run(prtsk->TSK_prdsp)) != DSP_NOERROR)
            return  ulRC;

         if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "ACTIVATE",
                                         &ulAddr )) != DSP_NOERROR)
         {
            MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::OS_Load, error resolving ACTIVATE label ulRC %lx\n",ulRC);
            return (ulRC);
         }

         if ((ulRC = DspBIOS_D_Read(prtsk->TSK_prdsp, ulAddr, (ULONG)1, &usBuf
            )) != DSP_NOERROR)
            return (ulRC);

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspalloc::OS_Load");
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
             "  READ26:  contents of ACTIVATE address", EOF, (ULONG)1,
             prtsk->TSK_prdsp, ulAddr, (ULONG)1, &usBuf);

#ifndef  NODSP
         if (usBuf[0] != 0)
            return (DSP_MWAVEOS_ERROR);
#endif

   /*************************************************************************/
   /* Allocate MIPS for ISPOS.                                              */
   /*************************************************************************/

   prtsk->TSK_ulCPS = prtsk->TSK_ulCPF;
   prtsk->TSK_prdsp->DSP_ulCPSFree -= prtsk->TSK_ulCPS;

   /* SET STATE FLAG TO ACTIVE                                              */

   prtsk->TSK_usflgState = TSK_ACTIVE;

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::OS_Load exit ulRC=0\n");
   return  DSP_NOERROR;

}                                      /* END OS_Load                       */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateTask                                            */
/*                                                                          */
/* FUNCTION: This procedure allocates resources for a task in the module    */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK          - pointer to task struct holding task information      */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*    DSP_NO_INSTNS  - No code segments given, can't load                   */
/*    DSP_NO_DATA    - No default data segment given, can't load            */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*                                                                          */
/*    FindFrameMgr                                                          */
/*    AllocateCPS                                                           */
/*    AddIPC                                                                */
/*    AllocateDMA                                                           */
/*    AllocateStrideHold                                                    */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating the       */
/*    resources, the entire allocation process will be aborted, and the     */
/*    error code returned.                                                  */
/*    Some of the load may not have been complete upon termination in       */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateTask(PRTSK prtsk)

{
   PRSEG      prseg;
   PRGPC      prgpc;
   PRHW       prhw;
   PRHW       prtaskhw;
   PRLDMA     pldmaNode;
   RC         ulRC = DSP_NOERROR;      /* Initialization required           */
   ULONG      ulDummy;
   LONG       lDummy;
   BOOL       bfound;
   BOOL       bStrideHold;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateTask entry prtsk %x\n",(int)prtsk);
   /*************************************************************************/
   /* If the Task is a BIOS task, allocate                                  */
   /* HW resources for the task.                                            */
   /*************************************************************************/

   if ((prtsk->TSK_usflgType&TSK_BIOSMASK) == TSK_BIOS) {
      if (prtsk->TSK_prhwTail != NULL) {

         /*******************************************************************/
         /* Traverse the HW list this task claims                           */
         /*******************************************************************/

         prtaskhw = prtsk->TSK_prhwTail;
         do {

            /****************************************************************/
            /* Traverse the HW Resource List on DSP Assign the HW token to  */
            /* the Task when found.  If a match is found but the Token is   */
            /* already assigned (HW_prtsk!=NULL) Return Insufficient        */
            /* Resources.  If the entire DSP HW list is traversed, and no   */
            /* Match was found, Return Insufficient Resources.              */
            /****************************************************************/

            bfound = FALSE;
            prhw = prtsk->TSK_prdsp->DSP_prhwTail;

            do {
               if (prtaskhw->HW_ulhwType == prhw->HW_ulhwType) {
                  if (prhw->HW_prtsk == NULL) {
                     prhw->HW_prtsk = prtaskhw->HW_prtsk;
                     bfound = TRUE;
                  }
                  else
                     ulRC = DSP_HWDEVICE_NOT_AVAILABLE;
               }                       /* endif                             */
               prhw = prhw->HW_prhwNext;
            }  while ((prhw != prtsk->TSK_prdsp->DSP_prhwTail) && (ulRC ==
               DSP_NOERROR) && (bfound == FALSE));

            /****************************************************************/
            /* If the entire DSP HW list is traversed, and no Match was     */
            /* found, Return Insufficient Resources.                        */
            /****************************************************************/

            if (bfound == FALSE)
               ulRC = DSP_HWDEVICE_NOT_AVAILABLE;
            else
               prtaskhw = prtaskhw->HW_prhwNext;

         }  while ((prtaskhw != prtsk->TSK_prhwTail)&&(ulRC == DSP_NOERROR));                          /* enddo                             */
      } /* endif                                                            */
   } /* endif                                                               */
   if (ulRC == DSP_NOERROR) {

      /**********************************************************************/
      /* EXAMINE ALL THE INSTRUCTION SEGMENTS                               */
      /**********************************************************************/

      if ((prseg = prtsk->TSK_prsegITail) == NULL)
         return  DSP_NO_INST;

      do {
         prseg = prseg->SEG_prsegNext; /* 1st time, move to head of list    */

         /* ALLOCATE RESOURCES FOR THE SEGMENT, CHECK FOR ERRORS            */

         if ((ulRC = AllocSegRes(prseg)) != DSP_NOERROR)
            return  ulRC;

      }  while (prseg != prtsk->TSK_prsegITail);

      /**********************************************************************/
      /* EXAMINE ALL THE DATA SEGMENTS                                      */
      /**********************************************************************/

      if ((prseg = prtsk->TSK_prsegDTail) == NULL)
         return  DSP_NO_DATA;

      /* PLACE TCB ON DDS, the DDS is at the head of the DSeg list          */

      prseg->SEG_prsegNext->SEG_prsegtmp->SEG_ulPrefix = sizeof(RTCB)/2;

      do {
         prseg = prseg->SEG_prsegNext; /* Advance to next in head of list   */

         /* ALLOCATE RESOURCES FOR THE SEGMENT, CHECK FOR ERRORS            */

         if ((ulRC = AllocSegRes(prseg)) != DSP_NOERROR)
            return  ulRC;

      }  while (prseg != prtsk->TSK_prsegDTail);

      /**********************************************************************/
      /* FIND OR CREATE THE FRAME MANAGER FOR THIS TASK                     */
      /**********************************************************************/

      if ((ulRC = FindFrameMgr(prtsk)) != DSP_NOERROR)
         return  ulRC;

      /******************************************************************** */
      /* Create the GPC buffer For Each Owner GPC hanging off the GPC list. */
      /* The GPC control blocks should only be hanging off a Task.          */
      /* The GPC buffer will be allocate from DSP memory.                   */
      /**********************************************************************/

      if ((prgpc = prtsk->TSK_prgpcTail) != NULL) {  /* Check for one block */
         do {
            prgpc = prgpc->GPC_prgpcNext;            /* Get next GPC block  */
            if (((prgpc->GPC_prseg->SEG_usflg&SEG_TYPEMASK) == SEG_DATA) &&
               (prgpc->GPC_usflgKind == GPC_OWNER)) {

               if ((ulRC = AllocateGPCBuf(prgpc)) != DSP_NOERROR)
                  return  ulRC;
            }

         }  while (prgpc != prtsk->TSK_prgpcTail);
      } /* endif                                                            */

      /**********************************************************************/
      /* Allocate CPS for the Task if its a RealTime Task                   */
      /**********************************************************************/

      if (((prtsk->TSK_usflgType&TSK_TYPEMASK) != TSK_NRT1) &&
         ((prtsk->TSK_usflgType&TSK_TYPEMASK) != TSK_NRT2))
         if ((ulRC = AllocCPS(prtsk)) != DSP_NOERROR)
            return (ulRC);

      /**********************************************************************/
      /* Add an IPC for the Task.                                           */
      /**********************************************************************/

      if (prtsk->TSK_prseg_IPC != NULL) {
         if (ulRC == DSP_NOERROR)
            ulRC = AddIPC(prtsk);
      }

      /**********************************************************************/
      /* Check to assure DMA packet usage not exceeded                      */
      /**********************************************************************/

      if (prtsk->TSK_prldmaTail != NULL)/* any dma requests for this task   */
         {
         if (ulRC == DSP_NOERROR)
            ulRC = AllocateDMA(&lDummy, &ulDummy);
         if (ulRC != DSP_NOERROR)
            return (ulRC);

         /*******************************************************************/
         /* See if any stride/hold usage requested                          */
         /*******************************************************************/

         bStrideHold = FALSE;
         pldmaNode = prtsk->TSK_prldmaTail;
         do {
            if (pldmaNode->LDMA_ulStride != 0)/* find a stride entry?       */
               {
               bStrideHold = TRUE;
               break;
            }
         }  while (pldmaNode != prtsk->TSK_prldmaTail);/* still not end     */
         if (bStrideHold == TRUE)      /* any stride/hold requested?        */
            ulRC = AllocateStrideHold(prtsk);
      }                                /* endif                             */
   }                                   /* endif Check for INSUFF_RESOURCES  */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateTask exit ulRC %lx\n",ulRC);
   return (ulRC);

}                                      /* END AllocateTask                  */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateSegment                                         */
/*                                                                          */
/* FUNCTION: This procedure allocates an instruction or data segment in DSP */
/*           memory and zeros the buffer.                                   */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK  prtsk       - pointer to task struct holding task information  */
/*    PSZ    pszVirtName - Name of NEW segment.                             */
/*    ULONG  ulSize      - Size of New segment.                             */
/*    ULONG  ulAlign     - Alignment of new segment.                        */
/*    USHORT usFlags     - Segment type.                                    */
/*    HSEG   hSegAddr    - Segment to recieve address of NEW segment.       */
/*    ULONG  ulOffsetAddr- Offset into the segment that gets addr.          */
/*    PHSEG  phSeg       - Handle to new segment.                           */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*    phSeg          - Contains the new segment.                            */
/*                                                                          */
/* SIDE EFFECTS:     - The hSegAddr Segment will get the address of the     */
/*                     new segment.                                         */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*                                                                          */
/*    FindFrameMgr                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateSegment(PRTSK prtsk,PSZ pszVirtName,ULONG ulSize,ULONG ulAlign,
                    USHORT usFlags,ULONG ulDSPAddr,PRSEG *pprSeg)
{

   ULONG      ulGoodAlign[] =  { 1,2,4,8,16,32,64,128,256,512,1024,2048 };
   PRSEG      prNewSeg;
   PRSEG      prseg;
   RC         ulRC;
   BOOL       bCkit;
   BOOL       bFound;
   USHORT     usTemp;                  /* CH01 Temp variable for DS writes */
   int        i;

   MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspalloc::AllocateSegment entry prtsk %x pszVirtName %s ulDSPAddr %lx\n",
	   (int)prtsk,pszVirtName,ulDSPAddr);

   /*************************************************************************/
   /*  Verify that all the parameters are OK.                               */
   /*************************************************************************/

   if (prtsk == NULL)
      return (DSP_INV_HANDLE);

   if (prtsk->TSK_achValid != TSKVALIDATE)   /* Valid Task                  */
      return (DSP_INV_HANDLE);

   if ((ulRC=CheckDSPStatus(prtsk->TSK_prdsp)))
      return (ulRC);  /* CH02 Check for card unplugged */

   if (ulSize == 0)                          /* Valid Size                  */
      return (DSP_INV_PARAMETER);

   if (((usFlags&SEG_TYPEMASK) != SEG_INST) &&
      ((usFlags&SEG_TYPEMASK) != SEG_DATA ))   /* Valid Flags                 */
      return (DSP_INV_FLAG);

   /*** CH03 LBMIV 02/05/97 Begin ***/
   if (ulDSPAddr > prtsk->TSK_prdsp->DSP_ulDMemTotal) {
     MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspalloc::AllocateSegment error Segment %s Addr %lx\n",pszVirtName,ulDSPAddr);
     return(DSP_INV_ADDRESS);
   } /* endif */
   /*** CH03 LBMIV 02/05/97 End ***/

   /*************************************************************************/
   /* Verify that align values are valid                                    */
   /*************************************************************************/

   bCkit = FALSE;                      /* assume bad align value            */
   for (i = 0; i < (sizeof ulGoodAlign)/(sizeof(ULONG)); i++)
      if (ulAlign == ulGoodAlign[i]) {
         bCkit = TRUE;
         break;
      }
   if (!bCkit)
      return (DSP_INV_PARAMETER);

   /*************************************************************************/
   /*  If the Name is NON-NULL Verify that its Unique.                      */
   /*  Add a segment to the Tasks Code or Data Segment list .               */
   /*************************************************************************/

   if (usFlags == SEG_INST) {

      bFound = FALSE;
      if (pszVirtName != NULL) {

         prseg = prtsk->TSK_prsegITail;
         do {
            if (prseg->SEG_pszVirtName != NULL) {
               ulRC = mwstrcmp(prseg->SEG_pszVirtName, pszVirtName);
               if (ulRC == 0)
                  bFound = TRUE;
            }                          /* endif                             */
            prseg = prseg->SEG_prsegNext;
         }  while ((prseg != prtsk->TSK_prsegITail) && (bFound == FALSE));/**/
                                       /* enddo                             */
      }                                /* endif                             */
      if (bFound == FALSE)
         ulRC = AddRSegmentTail(&(prtsk->TSK_prsegITail), &prNewSeg);
      else {
         ulRC = DSP_DUPLICATE_NAME;
	 MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspalloc::AllocateSegment error %lx Name %s\n",ulRC,pszVirtName);
      }

   }                                   /* end if SEG_INST                   */
   else if (usFlags == SEG_DATA) {
      bFound = FALSE;
      if (pszVirtName != NULL) {

         prseg = prtsk->TSK_prsegDTail;
         do {
            if (prseg->SEG_pszVirtName != NULL) {
               ulRC = mwstrcmp(prseg->SEG_pszVirtName, pszVirtName);

               if (ulRC == 0)
                  bFound = TRUE;
            }                       /* endif                             */
            prseg = prseg->SEG_prsegNext;
         }  while ((prseg != prtsk->TSK_prsegDTail) && (bFound == FALSE));

      }                             /* endif                             */
      if (bFound == FALSE)
         ulRC = AddRSegmentTail(&(prtsk->TSK_prsegDTail), &prNewSeg);
      else {
         ulRC = DSP_DUPLICATE_NAME;
	 MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspalloc::AllocateSegment error ulRC %lx Name %s\n",ulRC,pszVirtName);
      }

   }                                /* endif SEG_DATA                    */
   else
      ulRC = DSP_INV_FLAG;

   /*************************************************************************/
   /*  Fill out any left over info.                                         */
   /*  Create a buffer for the Name.                                        */
   /*************************************************************************/

   if (ulRC == DSP_NOERROR) {

      if (pszVirtName != NULL)
         ulRC = GetStr(pszVirtName, &prNewSeg->SEG_pszVirtName);
      else
         prNewSeg->SEG_pszVirtName = pszVirtName;

      prNewSeg->SEG_usflg = usFlags;
      prNewSeg->SEG_prtsk = prtsk;
      prNewSeg->SEG_prsegtmp->SEG_ulMemSize = ulSize;
      prNewSeg->SEG_prsegtmp->SEG_ulAlign = ulAlign;

      /**********************************************************************/
      /*  Allocate Segment resources.                                       */
      /**********************************************************************/

      if ((ulRC = AllocSegRes(prNewSeg)) != DSP_NOERROR) {
         FreeSegment(prNewSeg);        /* Free segment if cannot            */
                                       /* allocate any resources            */
         return (ulRC);
      }

      /**********************************************************************/
      /*  Allocate an Image for the DSP memory allocated.                   */
      /**********************************************************************/
      /**********************************************************************/
      /*  Write the address of the new segment in the receiving             */
      /*  DSP Location If an address was specified.                         */
      /**********************************************************************/

      if (ulDSPAddr != 0) {

         /* CH01 : All DS writes of size 1 must use USHORT for AIX */
         usTemp = prNewSeg->SEG_prmem->MEM_ulSegAddr; /* CH01 */

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a",
              "dspalloc::AllocateSegment");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", prtsk);
         PRINT_DEBUG(EMBEDFILEPTR, "  pszVirtName \"%s\"", pszVirtName);
         PRINT_DEBUG(EMBEDFILEPTR, "  ulSize (0x%X)\n", ulSize);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE27: addr of new segment in receiving DSP loc if specced",
              EOF, 01L,
              prtsk->TSK_prdsp, ulDSPAddr, 01L, (PVOID)&usTemp);

         if ((ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, ulDSPAddr, 01L,
             (PVOID)&usTemp)) != DSP_NOERROR)         /* CH01 */
            return (ulRC);
      }

      /**********************************************************************/
      /*  Remove Temporary Section of Segment.                              */
      /**********************************************************************/

      *pprSeg = prNewSeg;

   }                                   /* endif                             */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateSegment exit ulRC %lx\n",ulRC);
   return (ulRC);

}                                      /* END AllocateSegment               */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FindFrameMgr                                            */
/*                                                                          */
/* FUNCTION: This procedure searches for an existing frame mgr if available */
/*    otherwise, it will create and install a frame mgr in the dsp          */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK          - pointer to task struct holding task information      */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating the       */
/*    resources, the entire allocation process will be aborted, and the     */
/*    error code returned.                                                  */
/*    Some of the load may not have been complete upon termination in       */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FindFrameMgr(PRTSK prtsk)

{
   PRDSP      prdsp;
   PRFM       prfm;
   PRTSKTMP   prtsktmp;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   ULONG      ulFM_CPS, ulOldFM_CPS, ulFrameMgrSize;
   BOOL       fFound;
   BOOL       fAperiodic;
   enum { CREATE_RFM,CALC_FM_CPS,ALLOC_FM_CPS,ALLOC_DSP_MEM,INSERT_FM } step;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::FindFrameMgr entry prtsk %x\n",(int)prtsk);

/****************************************************************************/
/* GET THE DSP THIS TASK IS RUNNING ON                                      */
/****************************************************************************/

   prdsp = prtsk->TSK_prdsp;
   prtsktmp = prtsk->TSK_prtsktmp;

/****************************************************************************/
/* IF TIMER, CALCULATE THE CLOSEST SAMPLERATE AND CONVERT TO USE IT         */
/****************************************************************************/
/* TO BE DONE LATER                                                         */
/****************************************************************************/
/* SEARCH THE FRAME MGR LIST FOR THIS TASKS FRAME MGR                       */
/****************************************************************************/

   fFound = FALSE;

/* APERIODIC TASKS NEED CHECK ONLY THE GROUP NAME                           */

   fAperiodic = (BOOL)((prtsk->TSK_usflgType&(USHORT)TSK_TYPEMASK) == (USHORT)
      TSK_APERIODIC);
   if (fAperiodic) {

   /* IF THIS IS APERIODIC WITH A GROUP NAME OF NULL, ALWAYS MAKE NEW FM    */

      if (prtsktmp->TSK_pszFMGroupName != NULL) {

      /* SEARCH FOR APERIODIC FM WITH SAME NAME                             */

         if ((prfm = prdsp->DSP_prfmTail) != NULL) {
            do {
               prfm = prfm->FM_prfmNext;/* 1st time, move to head of list   */

            /* COMPARE THIS FRAME MGR WITH THE ONE I NEED                   */

               if (((prfm->FM_usflgType&FM_TYPEMASK) == FM_APERIODIC) &&
		   (prtsktmp->TSK_pszFMGroupName == prfm->FM_pszName)) {

               /* AS ERROR CHECKING, MAKE SURE THEY ARE AT LEAST ON SAME HW */

                  if (prtsktmp->TSK_hwidHWIntSrc != prfm->FM_hwid)
                     return (DSP_INV_APERIODIC_FM);
                  else {

                 /* found the FM, set task's FrameMgr pointer               */

                     prtsk->TSK_prfm = prfm;
                     fFound = TRUE;
                  }
	       }

            }  while (prfm != prdsp->DSP_prfmTail && !fFound);
         }                             /* end if tail != null               */
      }                                /* end search of frame mgr list      */
   }                                   /* end aperiodic search              */

/* IF THIS IS A NON-REAL-TIME TASK, SET PRFM TO NULL                        */

   else if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_NRT1 ||
            (prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_NRT2) {
       prtsk->TSK_prfm = NULL;
       fFound = TRUE;
   }

/* IF THIS IS A NORMAL TASK, SEARCH THROUGH FM LIST                         */

   else if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_REALTIME) {
      if ((prtsk->TSK_usflgType&TSK_USEMASK) == TSK_USETIMER)
         return (DSP_NOT_SUPPORTED);
      if ((prfm = prdsp->DSP_prfmTail) != NULL) {
         do {
            prfm = prfm->FM_prfmNext;/* 1st time, move to head of list*/

   /* COMPARE THIS FRAME MGR WITH THE ONE I NEED                      */

            if ((prfm->FM_hwid == prtsktmp->TSK_hwidHWIntSrc) &&
               ((prfm->FM_usflgType&FM_TYPEMASK) ==
               (prtsk->TSK_usflgType&TSK_TYPEMASK)) &&
               (prfm->FM_ulSampleRate == prtsktmp->TSK_ulTimeBase) &&
               (prfm->FM_usSamplesPerFrame ==
               prtsktmp->TSK_usSamplesPerFrame)) {

      /* found the FM, set task's FrameMgr pointer, update FM uscount */

               prtsk->TSK_prfm = prfm;
               prfm->FM_usUseCount += 1;
               fFound = TRUE;
            }

         }  while (prfm != prdsp->DSP_prfmTail && !fFound);
      }                          /* end if tail != null               */
   }                             /* END NORMAL TASK SEARCH            */
   else {
      return (DSP_INV_TASK_TYPE);
   }

/* IF NOT FOUND, MUST CREATE ONE                                            */

   if (!fFound) {

  /**************************************************************************/
  /*Determine cycles allocated for existing frame managers                  */
  /**************************************************************************/

      if ((ulRC = CalcFM_CPS(prtsk, &ulOldFM_CPS)) != DSP_NOERROR)
         return  ulRC;

      for (step = CREATE_RFM; step <= INSERT_FM && ulRC == DSP_NOERROR; step++
         ) {
         switch (step) {
            case  CREATE_RFM :

      /* CREATE FM STRUCT AND ADD TO FM LINKED LIST                         */

               if ((ulRC = AddRFrameMgrTail(&(prdsp->DSP_prfmTail), &prfm)) ==
                  DSP_NOERROR) {       /* STORE PTR TO FM IN TASK           */
                  prtsk->TSK_prfm = prfm;

        /* FILL IN REST OF RFM INFORMATION                                  */

                  if (fAperiodic)
                     prfm->FM_pszName = prtsktmp->TSK_pszFMGroupName;

                  prfm->FM_hwid = prtsktmp->TSK_hwidHWIntSrc;
                  prfm->FM_pszHWintsource = NULL;
                  prfm->FM_ulSampleRate = prtsktmp->TSK_ulTimeBase;
                  prfm->FM_usSamplesPerFrame = prtsktmp->TSK_usSamplesPerFrame;
                  prfm->FM_usflgType |= prtsk->TSK_usflgType&TSK_TYPEMASK;

      /* make sure usecount is set to one by AddTailRFrameMgr & usflgType to 0*/

               }
               break;

            case  CALC_FM_CPS :

      /**********************************************************************/
      /*Calculate CPS for the new collection of frame managers              */
      /**********************************************************************/

               ulRC = CalcFM_CPS(prtsk, &ulFM_CPS);
               break;

            case  ALLOC_FM_CPS :

      /*Want to up total allocation by delta of new fm MIPS - old FM MIPS   */

               ulFM_CPS = ulFM_CPS-ulOldFM_CPS;/* Get MIPS delta            */

      /**********************************************************************/
      /* Allocate CPS for the added Frame Mgr if cycles are available       */
      /**********************************************************************/

               if (ulFM_CPS < prtsk->TSK_prdsp->DSP_ulCPSFree)
                  prtsk->TSK_prdsp->DSP_ulCPSFree -= ulFM_CPS;
               else {
                  ulRC = DSP_INSUFF_CPS;
		  MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspalloc::FindFrameMgr error %lx, Task %s CPSfree %ld Required %ld\n",ulRC,
                            prtsk->TSK_pszVirtName,
                            prtsk->TSK_prdsp->DSP_ulCPSFree, ulFM_CPS);
               }
               break;

            case  ALLOC_DSP_MEM :

    /************************************************************************/
    /* Now that we are assured MIPS are available for the new Frame Mgr, we */
    /* can go ahead and write the control block to the dsp and activate the */
    /* Frame Mgr.                                                           */
    /************************************************************************/

   /************************************************************************/
   /* CH01 Determine size of the FM control block required by this version */
   /* of MWAVEOS in words of data store.  Older versions use the standard  */
   /* structure defined in RFMCB.  Newer versions have been extended and   */
   /* are recognized by the presence of the FMCBSIZE label. LBM 01/10/96   */
   /************************************************************************/
   if (IsposLabelToAddress(prdsp, "FMCBSIZE", &ulFrameMgrSize)
                           != DSP_NOERROR)
      {
         ulFrameMgrSize = (ULONG)sizeof(RFMCB)/2;
      }
    /* ALLOCATE DSP MEM FOR FMCB                                            */
               ulRC = AllocateMem(prdsp, ulFrameMgrSize, MEM_DATA|
                  MEM_ALLOCATE, (ULONG)1, (ULONG)0, &(prfm->FM_prmemFMCB));
               break;

            case  INSERT_FM :

               ulRC = InsertFrameMgr(prtsk, prfm);

         }                             /* endswitch                         */
      }                                /* endfor                            */
      if (ulRC != DSP_NOERROR)
         {
         step--;                       /* back up one step                  */
         switch (step) {               /* on error, undo all that has been  */
                                       /* done                              */
            case  INSERT_FM :

               (VOID)FreeMem(prtsk->TSK_prdsp, prfm->FM_prmemFMCB, MEM_DATA);

            case  ALLOC_DSP_MEM :

               prtsk->TSK_prdsp->DSP_ulCPSFree += ulFM_CPS;

            case  ALLOC_FM_CPS :
            case  CALC_FM_CPS :

               (VOID)RemoveRFrameMgr(prfm, prtsk);

            case  CREATE_RFM :

               return (ulRC);

         }                             /* endswitch                         */
      }                                /* endif                             */
   }                                   /* end create new frame mgr          */
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::FindFrameMgr exit ulRC=0\n");
   return  DSP_NOERROR;
}                                      /* end FindFrameMgr                  */

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocSegRes                                             */
/*                                                                          */
/* FUNCTION: This procedure searches for resources to be allocated for a    */
/*    given segment and allocates all resources needed for the segment.     */
/*    First pass, this will only allocate dsp memory.                       */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRSEG          - pointer to segment struct to be searched             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating the       */
/*    resources, the entire allocation process will be aborted, and the     */
/*    error code returned.                                                  */
/*    Some of the allocation may not have been complete upon termination in */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocSegRes(PRSEG prseg)

{
   PRDSP      prdsp;
   RC         ulRC;
   USHORT     usFlag;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocSegRes entry prseg %x\n",(int)prseg);

   /*************************************************************************/
   /* ALLOCATE DSP MEMORY FOR THE SEGMENT                                   */
   /*************************************************************************/

   prdsp = prseg->SEG_prtsk->TSK_prdsp;

   /* SET FLAGS FOR AllocateMem                                             */

   if ((prseg->SEG_usflg&SEG_TYPEMASK) == SEG_DATA)
      usFlag = MEM_DATA|MEM_ALLOCATE;
   else
      usFlag = MEM_INST|MEM_ALLOCATE;

   /* ALLOCATE DSP MEM FOR THIS SEGMENT                                     */

   if ((ulRC = AllocateMem(prdsp, prseg->SEG_prsegtmp->SEG_ulMemSize, usFlag,
      prseg->SEG_prsegtmp->SEG_ulAlign, prseg->SEG_prsegtmp->SEG_ulPrefix,
      &(prseg->SEG_prmem))) != DSP_NOERROR)
      return  ulRC;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocSegRes exit ulRC %lx\n",ulRC);
   return  ulRC;

}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocCPS                                                */
/*                                                                          */
/* FUNCTION: This procedure will allocate MIPS for a Task from the DSP      */
/*    MIPs Pool.                                                            */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK          - pointer to task                                      */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR       -       No errors occured                           */
/*    DSP_INSUFF_CPS    -       Insufficient IPS available                  */
/*    DSP_NOT_SUPPORTED -       Timer or 0 Sample rate specified            */
/*                                                                          */
/* SIDE EFFECTS: DSP_ulCPSFREE will have the new MIPS number.               */
/*                                                                          */
/* NOTES:                                                                   */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocCPS(PRTSK prtsk)

{
   double     dSampRate;
   PRLDMA     prldmaNode;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   ULONG      ulMaxCPF;
   static ULONG ulint_overhead;
   static ULONG ultco_cpf;
   ULONG      ulCycleTemp;
   ULONG      ulIPS;
   ULONG      ulDMA_IPS;
   ULONG      ulFrameRate;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocCPS entry prtsk %x\n",(int)prtsk);

   /*************************************************************************/
   /* ALLOCATE MIPS                                                         */
   /* We only allocate MIPS for RealTime tasks.                             */
   /*************************************************************************/
   /*************************************************************************/
   /*                                                                       */
   /* Timer based tasks are not supported.  If timer is                     */
   /* indicated in TSK_usflgType, return DSP_NOT_SUPPORTED.                 */
   /*                                                                       */
   /* The type TSK_UNDEFINED is a vestige of the module compiler            */
   /* and should not occur in a legal DSP file.                             */
   /*************************************************************************/

   if (ulcintcnt == 0)                 /* if ISPOS values not yet retrieved */
      {

      /**********************************************************************/
      /* get 'cycle interrupt count overhead' value from ISPOS              */
      /**********************************************************************/

      if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "CINTCNT", &ulcintcnt)
         ) != DSP_NOERROR)
         return (ulRC);

      /**********************************************************************/
      /* get 'task call overhead' value from ISPOS                          */
      /**********************************************************************/

      if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "TCO_CPF", &ultco_cpf)
         ) != DSP_NOERROR)
         return (ulRC);

      /**********************************************************************/
      /* get 'safe dma cycle overhead' from Mwaveos                         */
      /* Use default of zero if label not found                             */
      /**********************************************************************/

      ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "SDMA_CYC", &ulSDMA_CYC);

      if ((ulRC != DSP_NOERROR) && (ulRC != DSP_LABEL_NOT_FOUND))
         return (ulRC);
      ulRC = DSP_NOERROR;
   }

   /*************************************************************************/
   /*                                                                       */
   /* for R2 manager, only sample rate supported (no timer support)         */
   /*************************************************************************/

if ((prtsk->TSK_usflgType&TSK_USEMASK&!TSK_BIOSMASK) != TSK_USESAMPLERATE)
      return (DSP_NOT_SUPPORTED);

   /*************************************************************************/
   /* Assure non-zero sample rate                                           */
   /*************************************************************************/

   if (prtsk->TSK_prtsktmp->TSK_usSamplesPerFrame < 1)
      return (DSP_NOT_SUPPORTED);

   /*************************************************************************/
   /* Establish overhead associated with >16 bit cycle count                */
   /*************************************************************************/

   ulCycleTemp = prtsk->TSK_ulCPF+2;   /* extra 2 might bump ctr            */
   ulint_overhead = ulcintcnt * hiword(ulCycleTemp);

   /*************************************************************************/
   /* Calculate cycles per frame associated with the task base, the         */
   /* task overhead, and the extra >16 bit cycle count overhead             */
   /*************************************************************************/

   ulMaxCPF = prtsk->TSK_ulCPF + ultco_cpf + ulint_overhead;

   /*************************************************************************/
   /* Calculate IPS associated with these cycles.  IPS = cycles per         */
   /* frame * frame rate.  Frame rate = Sample rate / Samples per frame     */
   /*************************************************************************/

   dSampRate = prtsk->TSK_prtsktmp->TSK_ulTimeBase;/* sample rate           */
   ulFrameRate = (ULONG)ceil(dSampRate/
      prtsk->TSK_prtsktmp->TSK_usSamplesPerFrame);/* round up               */
   ulIPS = ulMaxCPF *ulFrameRate;

   /*************************************************************************/
   /* Determine cycles associated with DMA cycle steal                      */
   /* CycleStealMIPS=(MaxPkSiz+5) * MaxPerDMA * (44.1K / 32)                */
   /* Also incorporate the delta associated with Mwaveos safe dma           */
   /*  support:                                                             */
   /*      SafeDMAMIPS=ulSDMA_CYC * MaxPerDMA * (44.1K / 32)                */
   /*************************************************************************/

   ulDMA_IPS = 0;

   if (prtsk->TSK_prldmaTail != NULL)  /* Any dma for this task             */
      {

      /*Thread through ldma structure associated with this task and         */
      /*total up dma cycle steal IPS +safe dma Mwaveos delta IPS            */

      prldmaNode = prtsk->TSK_prldmaTail;
      do                               /* until all dmas in this task       */
                                       /* traversed                         */
         {
         /* Add: ( MaxPkSize + 5 + safe dma delta ips per packet *          */
         /*        MaxPerDMA * (44.1K / 32) )                               */
         ulDMA_IPS = ulDMA_IPS+(prldmaNode->LDMA_ulMaxPkSize + 5 +
                     ulSDMA_CYC) *
                     prldmaNode->LDMA_lMaxPerDMA *
                     DMA_FrameRate;

         prldmaNode = prldmaNode->LDMA_prldmaNext;/* to next dma in tsk     */
      }  while (prldmaNode != prtsk->TSK_prldmaTail);/* not end of dma list */
   }

   /*************************************************************************/
   /* Now determine grand total                                             */
   /*************************************************************************/

   prtsk->TSK_ulCPS = ulIPS+ulDMA_IPS;

   /*************************************************************************/
   /* Allocate CPS for the Task if there is resource                        */
   /*************************************************************************/

   if (prtsk->TSK_ulCPS < prtsk->TSK_prdsp->DSP_ulCPSFree)
      prtsk->TSK_prdsp->DSP_ulCPSFree -= prtsk->TSK_ulCPS;
   else {
      ulRC = DSP_INSUFF_CPS;
      MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspalloc::AllocCPS error ulRC %lx Task=%s. CPSfree=%ld. Required=%ld",ulRC,
               prtsk->TSK_pszVirtName, prtsk->TSK_prdsp->DSP_ulCPSFree,
               prtsk->TSK_ulCPS);
      prtsk->TSK_ulCPS = 0L;
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocCPS exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: CalcFM_CPS                                              */
/*                                                                          */
/* FUNCTION: This procedure will calculate the ISPOS MIPS overhead          */
/*              associated with an added frame manager                      */
/*                                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK     prtsk    : ptr to task using the frame manager              */
/*    ULONG     pulFM_CPS: ptr to location that will contain the frame      */
/*                      Mgr MIPS                                            */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    ULONG     PULfm_CPS: ptr to location containing the calculated        */
/*                      frame mgr MIPS                                      */
/*                                                                          */
/*    DSP_NOERROR       - No errors occurred                                */
/*    DSP_INTERNAL_ERROR- Failing IsposLabelToAddress()                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* NOTES:                                                                   */
/*      Called by FindFrameMgr when adding a new frame mgr and by           */
/*              FreeTask when removing a frame mgr.                         */
/*      Evertime a frame mgr is added or removed, the ISPOS overhead        */
/*              associated with all the frame mgrs must be recalculated.    */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC CalcFM_CPS(PRTSK prtsk,ULONG *pulFM_CPS)

{
   double     dSampRate;
   PRFM       pfmNode,pfmBeginNode;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   static ULONG ulFMIPSA = 0;
   static ULONG ulFMIPSB,ulFMIPSC;
   ULONG      ulMipsCtr;
   ULONG      ulFrameRate;
   HWID       hwid;
   SHORT      sfmTotal,sfmSame;
   int        i;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::CalcFM_CPS entry prtsk %x\n",(int)prtsk);
   /*************************************************************************/
   /*If there are no frame managers, just indicate no fm mips and return    */
   /*************************************************************************/

   pfmNode = prtsk->TSK_prdsp->DSP_prfmTail;/* get ptr to fm list           */
   if (pfmNode == NULL)                /* if no frame mgrs, no MIPS         */
      {
      *pulFM_CPS = 0;                  /* Indicate no frame manger mips     */
      return (ulRC);                   /* Nothing more to do                */
   }

   /*************************************************************************/
   /* Establish IPS associated with all the Frame Managers                  */
   /*************************************************************************/
   /*Retrieve ISPOS defined FMIPS values, if this has not yet been done     */

   if (ulFMIPSA == 0) {
      if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "FMIPS_A", &ulFMIPSA))
         != DSP_NOERROR)
         return (DSP_INTERNAL_ERROR);  /* couldn't find FMIPS_A ISPOS label */
      if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "FMIPS_B", &ulFMIPSB))
         != DSP_NOERROR)
         return (DSP_INTERNAL_ERROR);  /* couldn't find FMIPS_B ISPOS label */
      if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "FMIPS_C", &ulFMIPSC))
         != DSP_NOERROR)
         return (DSP_INTERNAL_ERROR);  /* couldn't find FMIPS_C ISPOS label */
   }

   /*************************************************************************/
   /*Determine the total number of frame managers                           */
   /*************************************************************************/

   pfmBeginNode = pfmNode;
   sfmTotal = 0;                       /* will contain tot number of frame  */
                                       /* mgrs                              */
   do {
      sfmTotal++;                      /* counting up total frame mgrs      */
      pfmNode = pfmNode->FM_prfmNext;  /* bump to next frame mgr            */
   }  while (pfmNode != pfmBeginNode); /* not yet back to start             */

   /*************************************************************************/
   /*Now total the IPS contribution of each of the individual frame mgrs    */
   /*Contribution of each manager depends on the total number of mgrs       */
   /*and on the number of mangers on the same interrupt source.             */
   /*************************************************************************/

   ulMipsCtr = 0;
   for (i = 0; i < sfmTotal; i++)      /* Include every fm in calculation   */
      {
      pfmNode = pfmNode->FM_prfmNext;  /* Bump to next node in list         */
      pfmBeginNode = pfmNode;          /* Use to tell when we're back to    */
                                       /* start                             */
      hwid = pfmNode->FM_hwid;         /* pick up hwid of this frame mgr    */
      sfmSame = 0;                     /* Will contain count of mgrs w/ this*/
                                       /* id                                */
      do {
         if (hwid == pfmNode->FM_hwid) /* Same id? Will always be at        */
            sfmSame++;                 /* least 1                           */
         pfmNode = pfmNode->FM_prfmNext;/* bump to next mgr in list         */
      }  while (pfmNode != pfmBeginNode);/* back to start?                  */

      /**********************************************************************/
      /*Now calculate the IPS for the frame mgr pointed to by pfmNode.      */
      /*Mips=Frame_rate(FMIPSA + FMIPSB * sfmTotal + FMIPSC * sfmSame).     */
      /*Frame_rate=SampleRate/SamplesPerFrame (rounded up)                  */
      /**********************************************************************/

      dSampRate = pfmNode->FM_ulSampleRate;/* sample rate                   */
      if (pfmNode->FM_usSamplesPerFrame == 0) {
	MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspalloc::CalcFM_CPS, error ulRC %x Task=%s. Invalid samples/frame=0\n",
	       DSP_INV_FM,
	       prtsk->TSK_pszVirtName);
	return (DSP_INV_FM);          /* prevent divide by zero            */
      }
      ulFrameRate = (ULONG)ceil(dSampRate/pfmNode->FM_usSamplesPerFrame);
      ulMipsCtr = ulMipsCtr+ulFrameRate *(ulFMIPSA+ulFMIPSB *sfmTotal+ulFMIPSC
         *sfmSame);
   }                                   /* end for i                         */
   *pulFM_CPS = ulMipsCtr;             /* pass total fm IPS count back to   */
                                       /* caller                            */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::CalcFM_CPS exit ulRC %lx\n",ulRC);

   return (ulRC);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateDMA                                             */
/*                                                                          */
/* FUNCTION: This procedure allocates dma packet list entries for the dsp.  */
/*           Pkt counts for each element are contained in the LDMA_lPackets */
/*           field.  The total of LDMA_lPackets field in all of the linked  */
/*           LDMA structures must be less the Maximum packets available.    */
/*           If not, a INS_DMA_PKT_ENTRIES code is returned.  In addition,  */
/*           the procedure checks that DMA usage does not exceed the User   */
/*           Specified Bus Capacity.                                        */
/*                                                                          */
/* INPUT:                                                                   */
/*    none                                                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*    DSP_INS_PKT_ENTRIES  - Insufficient dma entries remaining             */
/*    lTotcount            - Total count of DMA packets                     */
/*    ulBusUse             - Total bus usage                                */
/* GLOBAL VARIABLES (referenced)                                            */
/*      pg                                                                  */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    none                                                                  */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating resources */
/*    the entire allocation processed will be aborted, and the error code   */
/*    returned.  Some of the tasks may not have been processed upon         */
/*    termination in case of a error.                                       */
/*                                                                          */
/*    THIS FUNCTION ASSUMES A SINGLE DSP.  IT DOES NOT SUPPORT A MULTIPLE   */
/*      DSP ENVIRONMENT.                                                    */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateDMA(LONG *lTotCount,ULONG *ulBusUse)
{
   PRTSK      pTskTail,prTskNod2;
   PRMOD      prMod,pModTail;
   PRLDMA     pLdmaTail,prldmaNode;
   PRDSP      prdspNode;
   LONG       lPacketCount;
   ULONG      ulBusUsage = 0;          /* Initialize to 0                   */


   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::AllocateDMA entry\n");

/****************************************************************************/
/* Assure lPktMax initialized properly.  If Mwave/OS label IDMAPKTS not     */
/* defined, lPktMax=PktDefault-2.                                           */
/* If IDMAPKTS defined, lPktMax=IDMAPKTS-2.                                 */
/* Note that this function supports only single dsp environment.            */
/****************************************************************************/

   if (lPktMax == 0)                   /* if lPktMax not yet initialized,   */
      {
      prdspNode = pg->prmgrTail->MGR_prdspTail;
      if (IsposLabelToAddress(prdspNode, "IDMAPKTS", (PULONG)&lPktMax) ==
                                                           DSP_NOERROR)
         lPktMax = lPktMax-2;
      else
         lPktMax = PktDefault-2;
   }

   lPacketCount = 0;                   /* set to begin counting             */

/****************************************************************************/
/*Thread through entire structure tree & add up packet allocations.         */
/*Code relies on assumption that all tasks associated with the module       */
/*tree are on the same dsp.                                                 */
/****************************************************************************/

   pModTail = prMod = pg->prmgrTail->MGR_prmodTail;/* set start/end         */
   do                                  /* until all modules in this dsp     */
                                       /* traversed                         */
      {
      pTskTail = prTskNod2 = prMod->MOD_prtskTail;/* ptrs to TSK structure  */
      if (pTskTail != NULL)            /* skip modules without tasks        */
         do                            /* until all tasks in this module    */
                                       /* traversed                         */
            {
            pLdmaTail = prldmaNode = prTskNod2->TSK_prldmaTail;/* ptrs to   */
                                       /* dma                               */
            if (prldmaNode != NULL)    /* are there dmas for this task?     */
               {                       /* add up packets for all dmas for   */
                                       /* this task                         */
               do                      /* until all dmas in this tsk in this*/
                                       /* mod traversed                     */
                  {
                  lPacketCount = lPacketCount+prldmaNode->LDMA_lPackets;
                                       /* bump cnt                          */
                  ulBusUsage += prldmaNode->LDMA_ulMaxPkSize *
                                prldmaNode->LDMA_lMaxPerDMA;
                  prldmaNode = prldmaNode->LDMA_prldmaNext;/* to nxt dma in */
                                                           /* task          */
               }  while (prldmaNode != pLdmaTail); /* still not end of this */
                                                   /* dma list              */
            }
            prTskNod2 = prTskNod2->TSK_prtskNext;/* to nxt tsk in this mod  */
         }  while (prTskNod2 != pTskTail);/* still not end of this task list*/
      prMod = prMod->MOD_prmodNext;    /* bump to nxt mod in this dsp       */
   }  while (prMod != pModTail);       /* still not end of this module list */
   *lTotCount = lPacketCount;          /* provide total packet count to     */
                                       /* caller                            */
   *ulBusUse = ulBusUsage * DMA_FrameRate; /* provide bus usage to caller CH01*/
   if (lPacketCount > lPktMax) {       /* if allocation exceeded            */
     MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspalloc::AllocateDMA error ulRC %x Available=%ld, Required=%ld\n",
	    DSP_INS_DMA_PKT_ENTRIES,
	    lPktMax, lPacketCount);
     return (DSP_INS_DMA_PKT_ENTRIES);
   }

/********Bus Capacity Checking***********************************************/

   if (*ulBusUse > pg->prmgrTail->MGR_ulBusCapacity) {
     MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspalloc::AllocateDMA error ulRC %x Available=%ld, Required=%ld\n",
	    DSP_INS_BUS_CAPACITY,
	    pg->prmgrTail->MGR_ulBusCapacity, *ulBusUse);
      return (DSP_INS_BUS_CAPACITY);
   }
   pg->prmgrTail->MGR_ulRemBusCapacity =
      pg->prmgrTail->MGR_ulBusCapacity - *ulBusUse;
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::AllocateDMA exit \n");

   return (DSP_NOERROR);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateStrideHold                                      */
/*                                                                          */
/* FUNCTION: This procedure allocates stride/hold     entries for the dsp.  */
/*           A  maximum of 31 entries may be allocated.  Information        */
/*           comes from the LDMA_ulStride and LDMA_ulHold fields of the     */
/*           LDMA structure.  All LDMA structure elements are pointed to    */
/*           by the TSK_prldmaTail pointer.  If no table entries are        */
/*           available for allocation, DSP_INS_STRIDE_HOLD_ENTRIES is       */
/*           returned.                                                      */
/*           This procedure is entered only if at least one stride/hold     */
/*           entry is being requested for the task                          */
/*                                                                          */
/*           This procedure assures that sufficient stride/hold table       */
/*           entries are available and updates the stride hold table in     */
/*           ISPOS data                                                     */
/*           storage as required.  The updating of the dsp data segment     */
/*           index into the stride/hold table is done later in FixupTask    */
/*           after the image is downloaded into the dsp.  This can't be done*/
/*           here because the downloading of the image would overlay the    */
/*           data segment update.                                           */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK prtsk          - Ptr to task for which the stride hold entries  */
/*                              are requested                               */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*    DSP_INS_STRIDE_HOLD_ENTRIES - Insufficient stride/hold entries avail  */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*      pg                                                                  */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    FindMatch                                                             */
/*    DspBIOS_D_Write                                                       */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating resources */
/*    the entire allocation processed will be aborted, and the error code   */
/*    returned.  Some of the tasks may not have been processed upon         */
/*    termination in case of a error.                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateStrideHold(PRTSK prtsk)
{
   PRTSK      pTskTail,prTskNod2;
   PRMOD      prMod,pModTail;
   PRLDMA     prldmaNod1,prldmaNod2,MatchNode;
   RC         ulRC;
   static ULONG ulStrideStart = 0;
   BOOL       bInUse[32];
   USHORT     usTableOffset,usStrideHoldEntry,usTemp;
   int        i;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateStrideHold entry prtsk %x\n",(int)prtsk);

/****************************************************************************/
/* First thread through all the dma structures for this task and see if any */
/* of them are sharable and if any existing sharable elements provide       */
/* identical stride/hold values.  If so, allocate the shared stride/hold    */
/* table entry                                                              */
/****************************************************************************/

   prldmaNod1 = prtsk->TSK_prldmaTail; /* set starting pointer              */
   do {
      if (prldmaNod1->LDMA_usflg == LDMA_SHARABLE)/* is this one sharable   */
         {
         if (FindMatch(prldmaNod1, &MatchNode) == TRUE)/* match found?      */
            {
            prldmaNod1->LDMA_usSlotNum = MatchNode->LDMA_usSlotNum;/* share */

       /***bypass debug******************************************************/
       /*if (cdbg) printf("Stride=%d, Hold=%d shared on Slot Num %d\n",               */
       /*       prldmaNod1->LDMA_ulStride,                                  */
       /*       prldmaNod1->LDMA_ulHold,                                    */
       /*       prldmaNod1->LDMA_usSlotNum);                                */
       /*****************end bypass******************************************/

         }
      }
      prldmaNod1 = prldmaNod1->LDMA_prldmaNext;/* to nxt dma                */
   }  while (prldmaNod1 != prtsk->TSK_prldmaTail);/* not back to start      */

/****************************************************************************/
/*All possible shared entries allocated.  Now allocate new ones             */
/****************************************************************************/

   if (ulStrideStart == 0)             /* get dsp stride table start if     */
                                       /* first time                        */
      {
      ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "STRIDE", &ulStrideStart);
      if (ulRC != DSP_NOERROR)
         return (ulRC);
   }

   for (i = 0; i < 32; i++)
      bInUse[i] = FALSE;               /* init table; FALSE means entry not */
                                       /* in use                            */

/****************************************************************************/
/*Prepare to find an available slot by first                                */
/*                   threading through entire ldma structure for            */
/*this dsp and marking all used slots in bInUse array.                      */
/****************************************************************************/

   pModTail = prMod = pg->prmgrTail->MGR_prmodTail;/* set start/end         */
   do                                  /* until all mods on this dsp        */
                                       /* traversed                         */
      {
      pTskTail = prTskNod2 = prMod->MOD_prtskTail;/* ptrs to TSK struct     */
      if (pTskTail != NULL)            /* skip modules without tasks        */
         do                            /* until all tasks in this mod       */
                                       /* traversed                         */
            {
            prldmaNod2 = prTskNod2->TSK_prldmaTail;
            if (prldmaNod2 != NULL)    /* Any dmas for this task            */
               {
               do {

               /* Mark slot used.  If usSlotNum=0, dummy entry 0 is marked. */

                  bInUse[prldmaNod2->LDMA_usSlotNum] = TRUE;
                  prldmaNod2 = prldmaNod2->LDMA_prldmaNext;/* to nxt in dma */
                                                           /* list          */
               }  while (prldmaNod2 != prTskNod2->TSK_prldmaTail);/* end of */
                                                               /* dma list? */
            }
            prTskNod2 = prTskNod2->TSK_prtskNext;/* Bump to next tsk in this*/
                                       /* mod                               */
         }  while (prTskNod2 != pTskTail);/* still not end of task list     */
      prMod = prMod->MOD_prmodNext;    /* bump to next mod in this dsp      */
   }  while (prMod != pModTail);       /* while still not end of module list*/

/****************************************************************************/
/*Now allocate available slots for each of the stride/hold                  */
/*requesters                                                                */
/****************************************************************************/

   prldmaNod1 = prtsk->TSK_prldmaTail; /* set start of list pointer         */
   do                                  /* until all dmas in the requesting  */
                                       /* task traversed                    */
      {
      if ((prldmaNod1->LDMA_ulStride != 0) && /* if stride requested        */
         (prldmaNod1->LDMA_usSlotNum == 0))/* and slot not yet allocated    */
         {
         for (i = 1; i < 32; i++) {
            if (bInUse[i] == FALSE)
               break;
         }
         if (i < 32)                   /* available slot was found?         */
            {
            bInUse[i] = TRUE;          /* mark slot as used                 */
            prldmaNod1->LDMA_usSlotNum = (USHORT)i;/* allocate slot to task */
            usTableOffset = 2*i;       /* Offset into stride table          */

         /*bypass debug******************************************************/
         /*  if (cdbg) printf("Stride=%d, Hold=%d allocated to Slot Num %d\n",        */
         /*     prldmaNod1->LDMA_ulStride,                                  */
         /*     prldmaNod1->LDMA_ulHold,                                    */
         /*     prldmaNod1->LDMA_usSlotNum);                                */
         /*************end bypass********************************************/
         /*******************************************************************/
         /*Build stride hold entry                                          */
         /*******************************************************************/

            usStrideHoldEntry = prldmaNod1->LDMA_ulStride;/* pick up Stride */
            usTemp = (prldmaNod1->LDMA_ulHold) << 11;/* shift Hold left     */
            usStrideHoldEntry = usStrideHoldEntry|usTemp;/* Or Hold in      */

         /*******************************************************************/
         /*Write new entry into proper slot in dsp stride/hold table        */
         /*******************************************************************/

            PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a",
              "dspalloc::AllocateStrideHold");
            PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", prtsk);
            PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                 "WRITE28: new entry in proper slot of dsp stride/hold table",
                 EOF, 1, prtsk->TSK_prdsp, ulStrideStart+usTableOffset, 1,
                 &usStrideHoldEntry);

            ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, /* dsp handle          */
               ulStrideStart+usTableOffset, /* dsp address                  */
               1,                      /* number of dsp words               */
               &usStrideHoldEntry);    /* pc address                        */
            if (ulRC != DSP_NOERROR)
               return (ulRC);

         /*******************************************************************/
         /*Stride/hold tbl  offset value in data segment is fixed up        */
         /*later after segment image is written into dsp memory.            */
         /*******************************************************************/

         }
         else
            return (DSP_INS_STRIDE_HOLD_ENTRIES);/* No more entries         */
      }
      prldmaNod1 = prldmaNod1->LDMA_prldmaNext;/* to next dma in task       */
   }  while (prldmaNod1 != prtsk->TSK_prldmaTail);/* still not end of this  */
                                       /* dma list                          */
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::AllocateStrideHold exit\n");

   return (DSP_NOERROR);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FindMatch                                               */
/*                                                                          */
/* FUNCTION: This procedure threads through the entire dma structure list   */
/*           looking for a sharable entry with matching stride hold values. */
/*           If a match is found, TRUE is returned and pMatchNode is set    */
/*           with the address of the matching node.  If no match is found,  */
/*           FALSE is returned                                              */
/*           This procedure is called only from AllocateStrideHold.         */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRLDMA pldmaNode     - Ptr to node defining sharable entry for which  */
/*                              a match is desired                          */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    TRUE                 - Match is found.  In this case, pMatchNode is   */
/*                              set with address of matching node.          */
/*    FALSE                - No match found                                 */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*      pg                                                                  */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    none                                                                  */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating resources */
/*    the entire allocation processed will be aborted, and the error code   */
/*    returned.  Some of the tasks may not have been processed upon         */
/*    termination in case of a error.                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

BOOL FindMatch(PRLDMA pldmaNode,PRLDMA *pMatchNode)

{
   PRTSK      pTskTail,prTskNod2;
   PRMOD      prMod,pModTail;
   PRLDMA     prldmaNod2;
   ULONG      ulStride,ulHold;


   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::FindMatch entry \n");

   ulStride = pldmaNode->LDMA_ulStride;/* pick up stride value we want to   */
                                       /* match                             */
   ulHold = pldmaNode->LDMA_ulHold;    /* pick up hold value we want to     */
                                       /* match                             */

/****************************************************************************/
/*Thread through entire dma structure looking for a sharable matching       */
/*entry.                                                                    */
/****************************************************************************/

   pModTail = prMod = pg->prmgrTail->MGR_prmodTail;/* set start/end         */
   do                                  /* until all mods on this dsp        */
                                       /* traversed                         */
      {
      pTskTail = prTskNod2 = prMod->MOD_prtskTail;/* ptrs to TSK struct     */
      if (pTskTail != NULL)            /* skip modules without tasks        */
         do                            /* until all tasks in this mod       */
                                       /* traversed                         */
            {
            prldmaNod2 = prTskNod2->TSK_prldmaTail;
            if (prldmaNod2 != NULL)    /* Any dmas for this task            */
               {
               do {
                  if (prldmaNod2 != pldmaNode)/* assure not looking at      */
                                       /* ourself                           */
                     if ((prldmaNod2->LDMA_usflg == LDMA_SHARABLE) &&
                        (prldmaNod2->LDMA_ulStride == ulStride) && /* same  */
                                       /* stride                            */
                        (prldmaNod2->LDMA_ulHold == ulHold) && /* same hold */
                        (prldmaNod2->LDMA_usSlotNum != 0))/* has a slot     */
                        {              /* found a match                     */
                        *pMatchNode = prldmaNod2;/* return matching node    */
                        return (TRUE); /* Indicate a match was found        */
                     }
                  prldmaNod2 = prldmaNod2->LDMA_prldmaNext;/* to nxt in dma */
                                       /* list                              */
               }  while (prldmaNod2 != prTskNod2->TSK_prldmaTail);/* end of */
                                       /* dma list?                         */
            }
            prTskNod2 = prTskNod2->TSK_prtskNext;/* Bump to next tsk in this*/
                                       /* mod                               */
         }  while (prTskNod2 != pTskTail);/* still not end of task list     */
      prMod = prMod->MOD_prmodNext;    /* bump to next mod in this dsp      */
   }  while (prMod != pModTail);       /* while still not end of module list*/
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::FindMatch exit\n");
   return (FALSE);                     /* no matching entry was found       */
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateVGPC_CPS                                        */
/*                                                                          */
/* FUNCTION: This procedure allocates cycles associated with VGPC           */
/*           connections.  It is entered only on the second or later        */
/*           connection.  The cycles for the first connection are included  */
/*           in the basic task cpf value.                                   */
/*           In addition to accounting for the vgpc mips, this procedure    */
/*           updates the dsp task cycle counter to account for the new cpf  */
/*           value.                                                         */
/*           If cycles are avaialble to make the connection, the cycles are */
/*           allcoated.  Otherwise DSP_INSUFF_CPS is returned               */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK  prtsk         - Ptr to the user task for this GPC              */
/*    PRGPC  prgpc         - Ptr to the GPC                                 */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*    DSP_INSUFF_CPS       - Insufficient cycles available                  */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*      none                                                                */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    TCB is updated with new CPF value                                     */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    none                                                                  */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateVGPC_CPS(PRTSK prtsk,PRGPC prgpc)
{
   RC         ulRC;
   ULONG      ulFrameRate,ulCycles;
   ULONG      ulAddrTCB_Cycles,ulTemp;
   ULONG      ulCPF_Plus2;
   USHORT     usHiword;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateVGPC_CPS entry prtsk %x\n",(int)prtsk);
/****************************************************************************/
/*Get frame rate information from the frame mgr controlling the task        */
/* using this VGPC                                                          */
/****************************************************************************/

   ulFrameRate =                       /* FrameRate = SampleRate /          */
                                       /* SamplesPer Frame                  */
      prtsk->TSK_prfm->FM_ulSampleRate/     /* sample rate                  */
      prtsk->TSK_prfm->FM_usSamplesPerFrame;/* samples per frame            */
   if (prtsk->TSK_prfm->FM_ulSampleRate%prtsk->TSK_prfm->FM_usSamplesPerFrame
      != 0)
      ulFrameRate++;                   /* round up FrameRate value          */
   ulCycles =                          /* Cycles = FrameRate * DeltaCPF     */
      ulFrameRate *                    /* FrameRate                         */
      prgpc->GPC_ulMaxCycles;          /* DeltaCPF                          */

/****************************************************************************/
/*Account for possibility that DeltaCPF caused rollover into hiword         */
/****************************************************************************/

   ulTemp = prtsk->TSK_ulCPF+2;
   usHiword = hiword(ulTemp);          /* pick up current hiword            */
   prtsk->TSK_ulCPF += prgpc->GPC_ulMaxCycles;/* bump cpf by DeltaCPF       */
   ulTemp = prtsk->TSK_ulCPF;
   if (usHiword != hiword(ulTemp))     /* Did bump cause hiword rollover    */
      ulCycles = ulCycles+ulcintcnt;   /* Account for additional cycles     */

/****************************************************************************/
/*Are there this many cycles left?                                          */
/****************************************************************************/

   if (ulCycles < prtsk->TSK_prdsp->DSP_ulCPSFree) {
      prtsk->TSK_prdsp->DSP_ulCPSFree -= ulCycles;/* Allocate cycles        */
      prtsk->TSK_ulCPS += ulCycles;    /* Update TSK struct w/ new value    */
   }
   else {
     MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspalloc::AllocateVGPC_CPS error ulRC %x VGPC Allocation. VGPC=%s. CPSfree=%ld. Required=%ld\n",
	    DSP_INSUFF_CPS,
	    prgpc->GPC_pszName, prtsk->TSK_prdsp->DSP_ulCPSFree, ulCycles);
     return (DSP_INSUFF_CPS);
   }

/****************************************************************************/
/*Fix up TCB of user task with new CPS value                                */
/****************************************************************************/

/*Get address of dsp location to fix up                                     */

   ulAddrTCB_Cycles =
      prtsk->TSK_prsegDTail->SEG_prsegNext->SEG_prmem->MEM_ulSegAddr-(ULONG)
         sizeof(RTCB)+FIELDOFFSET(RTCB, TCB_ulCycles);

/*Write new cpf value into TCB                                              */
   ulCPF_Plus2 = prtsk->TSK_ulCPF+2;                /* New value for TCB    */

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspalloc::AllocateVGPC_CPS");
   PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", prtsk);
   PRINT_DEBUG(EMBEDFILEPTR, "  prgpc (0x%X)\n",prgpc);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "WRITE29: new cpf value into TCB", EOF, (sizeof(ULONG)/2),
        prtsk->TSK_prdsp, ulAddrTCB_Cycles, (sizeof(ULONG)/2), &ulCPF_Plus2);

   ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, ulAddrTCB_Cycles,
             (sizeof(ULONG)/2 ), &ulCPF_Plus2);     /* Write new TCB value  */

   /*Assure task operating with new cycle count before continuing  **CH01** */
   if (DSP_NOERROR == ulRC)
      ulRC = Wait_DONERT(prtsk->TSK_prdsp);
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::AllocateVGPC_CPS exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeVGPC_CPS                                            */
/*                                                                          */
/* FUNCTION: This procedure frees     cycles associated with VGPC           */
/*           connections.  It is entered only when there are more than one  */
/*           outstanding VGPC connections.  (MIPS for the first VGPC        */
/*           connection is included in the basic task cpf value.)           */
/*           In addition to freeing        the vgpc mips, this procedure    */
/*           updates the dsp task cycle counter to account for the new cpf  */
/*           value.                                                         */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK  prtsk         - Ptr to the user task for this GPC              */
/*    PRGPC  prgpc         - Ptr to the GPC                                 */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*      none                                                                */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    TCB is updated with new CPF value                                     */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    FixupTCB(prtsk)                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeVGPC_CPS(PRTSK prtsk,PRGPC prgpc)
{
   RC         ulRC;
   ULONG      ulFrameRate,ulCycles;
   ULONG      ulAddrTCB_Cycles,ulTemp;
   ULONG      ulCPF_Plus2;
   USHORT     usHiword;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::FreeVGPC_CPS entry prtsk %x\n",(int)prtsk);

/****************************************************************************/
/*Get frame rate information from the frame mgr controlling the task        */
/* using this VGPC                                                          */
/****************************************************************************/

   ulFrameRate =                       /* FrameRate = SampleRate /          */
                                       /* SamplesPer Frame                  */
      prtsk->TSK_prfm->FM_ulSampleRate/     /* sample rate                  */
      prtsk->TSK_prfm->FM_usSamplesPerFrame;/* samples per frame            */
   if (prtsk->TSK_prfm->FM_ulSampleRate%prtsk->TSK_prfm->FM_usSamplesPerFrame
      != 0)
      ulFrameRate++;                   /* round up FrameRate value          */
   ulCycles =                          /* Cycles = FrameRate * DeltaCPF     */
      ulFrameRate *                    /* FrameRate                         */
      prgpc->GPC_ulMaxCycles;          /* DeltaCPF                          */

/****************************************************************************/
/*Account for possibility that DeltaCPF caused rollover into hiword         */
/****************************************************************************/

   ulTemp = prtsk->TSK_ulCPF+2;
   usHiword = hiword(ulTemp);          /* pick up current hiword            */
   prtsk->TSK_ulCPF -= prgpc->GPC_ulMaxCycles;/* reduce cpf by DeltaCPF     */
   ulTemp = prtsk->TSK_ulCPF;
   if (usHiword != hiword(ulTemp))     /* Did DeltaCPF cause rollover       */
      ulCycles = ulCycles+ulcintcnt;   /* Account for additional cycles     */
   prtsk->TSK_prdsp->DSP_ulCPSFree += ulCycles;/* Free up cycles            */
   prtsk->TSK_ulCPS -= ulCycles;       /* Update TSK structure w/ cycle     */
                                       /* count                             */

/****************************************************************************/
/*Fix up TCB with new CPS value                                             */
/****************************************************************************/

/*Get address of dsp location to fix up                                     */
   ulAddrTCB_Cycles =
      prtsk->TSK_prsegDTail->SEG_prsegNext->SEG_prmem->MEM_ulSegAddr -
      (ULONG)sizeof(RTCB)+FIELDOFFSET(RTCB, TCB_ulCycles);

/*Write new cpf value into TCB                                              */

/* Begin CH01 Swap words for non-Intel systems */
   ulCPF_Plus2 = prtsk->TSK_ulCPF+2;             /* New value for TCB       */
/* End CH01 Swap words for non-Intel systems */

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspalloc::FreeVGPC_CPS");
   PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", prtsk);
   PRINT_DEBUG(EMBEDFILEPTR, "  prgpc (0x%X)\n", prgpc);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "WRITE30: new cpf value into TCB", EOF, (sizeof(ULONG)/2),
        prtsk->TSK_prdsp, ulAddrTCB_Cycles, (sizeof(ULONG)/2), &ulCPF_Plus2);

   ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, ulAddrTCB_Cycles,
             (sizeof(ULONG)/2 ), &ulCPF_Plus2);  /* Write new TCB value     */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::FreeVGPC_CPS exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FixupStrideHold                                         */
/*                                                                          */
/* FUNCTION: This procedure fixes up the index into the stride hold table   */
/*           that exists in the data segment for every segment requesting   */
/*           a dma stride/hold entry.  The actual allocation of the entries */
/*           is done in AllocateStrideHold.  Fixing up of the index value   */
/*           is done here after the image has been loaded into dsp memory.  */
/*                                                                          */
/*           This procedure uses the slotnumber (stride/hold table entry    */
/*           number) and stride hold values contained in teh LDMA structure */
/*                                                                          */
/*           This procedure is called from FixupTask.                       */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK prtsk          - Ptr to task for which the stride hold entries  */
/*                              are requested                               */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*      none                                                                */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    DspBIOS_D_Write                                                       */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*    If an error is encountered during the process of allocating resources */
/*    the entire allocation processed will be aborted, and the error code   */
/*    returned.  Some of the tasks may not have been processed upon         */
/*    termination in case of a error.                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FixupStrideHold(PRTSK prtsk)
{
   PRLDMA     prldmaNod1;
   RC         ulRC;
   ULONG      ulDmaSpot;
   USHORT     usTableOffset;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::FixupStrindHold entry prtsk %x\n",(int)prtsk);

/****************************************************************************/
/*Now fix up the stride/hold index for each of the stride hold              */
/*requesters                                                                */
/****************************************************************************/

   prldmaNod1 = prtsk->TSK_prldmaTail; /* set start of list pointer         */
   do                                  /* until all dmas in the requesting  */
                                       /* task traversed                    */
      {
      if (prldmaNod1->LDMA_ulStride != 0)/* if stride requested             */
         {

         /*******************************************************************/
         /*Fix up stride/hold tbl  offset value in data segment.            */
         /*The offset value to be placed in the data segment is a word      */
         /*offset equal to the slot number.  The location in the data       */
         /*segment to be fixed up is determined by adding the offset        */
         /*(from the LDMA structure)                                        */
         /*to the address of the data segment start (from the memory        */
         /*memory structure .                                               */
         /*******************************************************************/

         usTableOffset = prldmaNod1->LDMA_usSlotNum;
         ulDmaSpot = prldmaNod1->LDMA_prseg->SEG_prmem->MEM_ulSegAddr+
            prldmaNod1->LDMA_ulOffset;

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a",
              "dspalloc::FixupStrideHold");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", prtsk);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE31:  Fix up stride/hold tbl offset value in data seg",
              EOF, 1,
              prtsk->TSK_prdsp, ulDmaSpot, 1, &usTableOffset);

         ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, /* dsp handle             */
            ulDmaSpot,                 /* dsp address                       */
            1,                         /* number of dsp words               */
            &usTableOffset);           /* pc address                        */
         if (ulRC != DSP_NOERROR)
            return (ulRC);
      }
      prldmaNod1 = prldmaNod1->LDMA_prldmaNext;/* to next dma in task       */
   }  while (prldmaNod1 != prtsk->TSK_prldmaTail);/* still not end of this  */
                                       /* dma list                          */
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspalloc::FixupStrindHold exit \n");
   return (DSP_NOERROR);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ChangeCPF                                               */
/*                                                                          */
/* FUNCTION: This procedure processes the dspChangeCPF API.  It allocates   */
/*           MIPS associated with the change and updates theCPF field of the*/
/*           dsp TCB to reflect the appropriate value.  If insufficient MISP*/
/*           are available to support an increased CPF, the API is          */
/*           ignored and DSP_INSUFF_CPS is returned.                        */
/*                                                                          */
/*           This procedure is called from DAEMON_DspChangeCPF              */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK prtsk          - Ptr to task for which the CPF change is        */
/*                              requested                                   */
/*    LONG lCPF            - Positive or negative CPF update delta value    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*    DSP_INSUFF_CPS       - Insufficient CPS available for requested delta */
/*    DSP_INV_PARAMETER    - Requested delta reduces CPF to < 1             */
/*    DSP_INV_HANDLE       - Invalid task handle.                           */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*                                                                          */
/*      none                                                                */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    DspBIOS_D_Write                                                       */
/*                                                                          */
/*                                                                          */
/* NOTES:                                                                   */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ChangeCPF(PRTSK prtsk,LONG lCPF)
{
   double     dSampRate;
   RC         ulRC;
   ULONG      ulFrameRate;
   LONG       lcpf;
   LONG       lNewCycles;
   ULONG      ulAddrTCB_Cycles;
   ULONG      ulCPF_Plus2;
   SHORT      sHiwdDelta;
   USHORT     usCPF_OldLowWord;     /* Ch 03 */
   USHORT     usTemp;               /* Ch 03 */

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::ChangeCPF entry prtsk %x\n",(int)prtsk);

/* Verify that prtsk is a valid task handle                                 */

   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);

   if ((ulRC=CheckDSPStatus(prtsk->TSK_prdsp)))
      return (ulRC);  /* CH02 Check for card unplugged */

   lcpf = prtsk->TSK_ulCPF+lCPF;       /* Add delta cpf to current value    */

   if (lcpf < 1)
      return (DSP_INV_PARAMETER);      /* CPF less than 1 is invalid        */

/****************************************************************************/
/*Calculate Frame Rate using  information in associated frame manager.      */
/*Frame Rate = SamplePerSecond  / SamplesPerFrame                           */
/****************************************************************************/

   dSampRate = prtsk->TSK_prfm->FM_ulSampleRate;

   /* Frame Rate = rounded up sample rate divided by samples per frame      */
   ulFrameRate = (ULONG)ceil(dSampRate/prtsk->TSK_prfm->FM_usSamplesPerFrame);

   usCPF_OldLowWord = loword((prtsk->TSK_ulCPF+2));  /* Save old CPF Low Word */

/*check for hiword rollowver                                                */

   sHiwdDelta = hiword(lcpf)-hiword(prtsk->TSK_ulCPF);
   lCPF = lCPF+ulcintcnt *sHiwdDelta;  /* add any cpf for hiword rollov     */
   lNewCycles = lCPF *ulFrameRate;     /* lNewCycles is delta change        */
   if (lCPF > 0)                       /* If increase, see if cycles        */
                                       /* available                         */
      {
      if (lNewCycles > prtsk->TSK_prdsp->DSP_ulCPSFree) {   /* Enough left? */
	MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspalloc::ChangeCPF error ulRC %x CCPF. Task=%s. CPSfree=%ld. Required=%ld\n",
	       DSP_INSUFF_CPS,
	       prtsk->TSK_pszVirtName,
	       prtsk->TSK_prdsp->DSP_ulCPSFree, lNewCycles);
	return (DSP_INSUFF_CPS);      /* Not enough cycles left            */
      }
   }

/*Fix up all the data structures                                            */

   prtsk->TSK_ulCPF = lcpf;            /* set new CPF in TSK structure      */
   prtsk->TSK_ulCPS += lNewCycles;     /* apply delta CPF to TSK structure  */
   prtsk->TSK_prdsp->DSP_ulCPSFree -= lNewCycles;/* apply delta CPS to DSP  */
                                       /* struct                            */

/****************************************************************************/
/*Fix up TCB with new CPS value                                             */
/****************************************************************************/

/*Get address of dsp location to fix up                                     */
   ulAddrTCB_Cycles =
      prtsk->TSK_prsegDTail->SEG_prsegNext->SEG_prmem->MEM_ulSegAddr -
      (ULONG)sizeof(RTCB)+FIELDOFFSET(RTCB, TCB_ulCycles);

/*Write new cpf value into TCB                                              */

/* Begin CH01 Swap words for non-Intel systems */
   ulCPF_Plus2 = prtsk->TSK_ulCPF+2;             /* New value for TCB       */
/* End CH01 Swap words for non-Intel systems */

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspalloc::ChangeCPF");
   PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", prtsk);
   PRINT_DEBUG(EMBEDFILEPTR, "  lCPF (0x%X)\n",lCPF);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "WRITE32:  new cpf value into TCB", EOF, (sizeof(ULONG)/2),
        prtsk->TSK_prdsp, ulAddrTCB_Cycles, (sizeof(ULONG)/2), &ulCPF_Plus2);

   /* *** Begin CH03 ***************************************************** */

   if ( loword(ulCPF_Plus2) < usCPF_OldLowWord )
      {
      usTemp = hiword(ulCPF_Plus2);
      ulRC = DspBIOS_D_Write( prtsk->TSK_prdsp,
                              ulAddrTCB_Cycles+2,
                              (sizeof(USHORT)/2),
                              &usTemp);       /* Write new hiword TCB value */

      usTemp = loword(ulCPF_Plus2);
      ulRC = DspBIOS_D_Write( prtsk->TSK_prdsp,
                              ulAddrTCB_Cycles,
                              (sizeof(USHORT)/2),
                              &usTemp);       /* Write new loword TCB value */
      }
   else
      ulRC = DspBIOS_D_Write( prtsk->TSK_prdsp,
                              ulAddrTCB_Cycles,
                              (sizeof(ULONG)/2),
                              &ulCPF_Plus2);  /* Write new TCB value */

   /* *** End CH03 ******************************************************* */

   if (DSP_NOERROR == ulRC)
      ulRC = Wait_DONERT(prtsk->TSK_prdsp);     /* Sync operation to MWAVEOS */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::ChangeCPF exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/* $PAGE                                                                    */
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ChangeDMA                                               */
/*                                                                          */
/* FUNCTION: This procedure processes the dspChangeDMA API.  It allocates   */
/*           MIPS and DMA packet entries associated with the change.  If    */
/*           insufficient MIPS or packet entries are available for an       */
/*           increment request, an error return code is generated and the   */
/*           API is ignored.                                                */
/*           If stride/hold change is requested, that request is validated  */
/*           and processed or  diagnosed and handles with a return code as  */
/*           appropriate                                                    */
/*                                                                          */
/*           This procedure is called from DAEMON_DspChangeDMA              */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK prtsk          - Ptr to task for which the DMA change is        */
/*                              requested                                   */
/*    PSZ  pszDMAName      - ASCII name of DMA channel                      */
/*    LONG lPacket         - Positive or negative packet number delta       */
/*    LONG lPacketSize     - Positive or negative max packet number delta   */
/*    LONG lMaxPacketPF    - Positive or negative max packets per frame delta*/
/*    USHORT usStride      - New stride value                               */
/*    USHORT usHold        - New Hold value                                 */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR          - No errors occurred                             */
/*    DSP_INSUFF_CPS       - Insufficient CPS available for requested delta */
/*    DSP_INV_PARAMETER    - Requested delta reduces lPacket, lPacketSize,  */
/*                              or lMaxPacketPF to < 1.                     */
/*                                                                          */
/* GLOBAL VARIABLES (referenced)                                            */
/*      none                                                                */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*    none                                                                  */
/*                                                                          */
/* REFERENCED FUNCTIONS:                                                    */
/*    AllocateDMA                                                           */
/*    DspBIOS_D_Write                                                       */
/*                                                                          */
/* NOTES:                                                                   */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */

/************************** END OF SPECIFICATIONS ***************************/

RC ChangeDMA(PRTSK prtsk,PSZ pszDMAName,LONG lPackets,LONG lMaxPacketSize,LONG
              lMaxPacketPF,USHORT usStride,USHORT usHold)

{
   PRLDMA     prldmaNod1;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   LONG       lNewPackets,lNewMaxSize,lNewMaxPF;
   static ULONG ulStrideStart = 0;    /* Must initialize to 0               */
   ULONG      ulNewMips,ulOldMips;
   LONG       lDeltaMips = 0;          /* Assume lDeltaMips = 0             */
   LONG       lDeltaBusUse = 0;  /* Delta Bus usage resulting from this API */
   LONG       lCount;
   ULONG      ulBusUse;
   USHORT     usStrideHoldEntry,usTemp,usTableOffset;
   BOOL       bFound;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::ChangeDMA entry prtsk %x\n",(int)prtsk);
/*assure prtsk points to a task handle                                      */

   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);

   if ((ulRC=CheckDSPStatus(prtsk->TSK_prdsp)))
      return (ulRC);  /* CH03 Check for card unplugged */

/****************************************************************************/
/*Thread through all the dma structures for this task looking for the       */
/*one with a matching name                                                  */
/****************************************************************************/

   prldmaNod1 = prtsk->TSK_prldmaTail; /* set starting pointer              */
   if (prldmaNod1 == NULL) {           /* error if no dma structures        */
     MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspalloc::ChangeDMA error ulRC %x ChangeDMA: Task %s has no DMA entries.\n",
	    DSP_NAME_NOT_FOUND,
	    prtsk->TSK_pszVirtName);
     return (DSP_NAME_NOT_FOUND);
   }
   bFound = FALSE;
   do {
      if (prldmaNod1->LDMA_pszDMAName != NULL)/* protect against NULL       */
         {
         if (mwstrcmp(prldmaNod1->LDMA_pszDMAName, pszDMAName) == 0) {
            bFound = TRUE;
            break;
         }
      }
      prldmaNod1 = prldmaNod1->LDMA_prldmaNext;
   }  while (prldmaNod1 != prtsk->TSK_prldmaTail);/* not back to start      */
   if (!bFound) {
     MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspalloc::ChangeDMA error ulRC %x DMA name=%s\n", DSP_NAME_NOT_FOUND,pszDMAName);
      return (DSP_NAME_NOT_FOUND);
   }

/****************************************************************************/
/* We've gotten the information about the requested DMA                     */
/* Now add the requested deltas                                             */
/****************************************************************************/

   lNewPackets = prldmaNod1->LDMA_lPackets+lPackets;
   if (lNewPackets < 0)
      return (DSP_INV_PARAMETER);
   lNewMaxSize = prldmaNod1->LDMA_ulMaxPkSize+lMaxPacketSize;
   if ((lNewMaxSize < 1) || (lNewMaxSize > 256))
      return (DSP_INV_PARAMETER);
   lNewMaxPF = prldmaNod1->LDMA_lMaxPerDMA+lMaxPacketPF;
   if (lNewMaxPF < 0)
      return (DSP_INV_PARAMETER);
   if ((usStride > StrideMax) || (usHold > HoldMax))
      return (DSP_INV_PARAMETER);

   if (lPackets > 0)                   /* if number of packets is being     */
                                       /* increased                         */

   /*Assure sufficient entries are available                                */

      {

   /*************************************************************************/
   /* Determine existing DMA packet entry count and bus usage               */
   /* These are the values for the entire system prior to the effect of     */
   /* the dspChangeDMA.  (ulBusUse is extraneous here.  Packet entries is   */
   /* what we really want).                                                 */
   /*************************************************************************/

      ulRC = AllocateDMA(&lCount, &ulBusUse);/* Determine existing dma      */
                                       /* values                            */
      if (ulRC != DSP_NOERROR)
         return (ulRC);
      if ((lCount+lPackets) > lPktMax) /* Room for new ones?                */
         return (DSP_INS_DMA_PKT_ENTRIES);
   }

/****************************************************************************/
/*If any change to MaxPacketSize or MaxPacketPF, establish effect on        */
/*MIPS and bus capacity                                                     */
/****************************************************************************/

   if ((lMaxPacketSize != 0) || (lMaxPacketPF != 0))/* any change           */
      {

   /*************************************************************************/
   /*Here we calculate lDeltaBusUse.  This  is the bus usage delta          */
   /*associated with this dspChangeDMA.  We then assure that this delta     */
   /*does not exceed the remaining bus capacity.                            */
   /*************************************************************************/

   /* begin CH02 DEC rearranged comments and code to make more readable     */
      /* (new values from dspChangeDMA - old dma values for this task) *    */
      /* DMA_FrameRate = delta bus use in xfrs/sec                          */

      lDeltaBusUse = (lNewMaxSize * lNewMaxPF -
         prldmaNod1->LDMA_ulMaxPkSize * prldmaNod1->LDMA_lMaxPerDMA) *
         DMA_FrameRate;

      /* if delta is positive and  delta > remaining, return error code     */
      if ((lDeltaBusUse > 0) &&
         (lDeltaBusUse > pg->prmgrTail->MGR_ulRemBusCapacity)) {
	MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspalloc::ChangeDMA error ulRC %x Remaining=%ld, Required=%ld\n",
	       DSP_INS_BUS_CAPACITY,
	       pg->prmgrTail->MGR_ulRemBusCapacity, lDeltaBusUse);
	return (DSP_INS_BUS_CAPACITY);
      }
   /* end CH02 DEC                                                          */

   /*ulRemBusCapacity is updated later after all other validation done      */
   /*Allocate DMA MIPS                                                      */

   /* Begin CH01 MTS #1234 and #2532: Include ulSDMA_CYC (safe DMA Cycles)  */
   /* in calculation  of delta CPS (MIPS)                                   */

      ulOldMips = DMA_FrameRate *             /* determine current DMA mips */
         (prldmaNod1->LDMA_ulMaxPkSize + 5 + ulSDMA_CYC) *         /* CH01  */
         prldmaNod1->LDMA_lMaxPerDMA;
      ulNewMips = DMA_FrameRate *                 /* determine new DMA mips */
         (lNewMaxSize + 5 + ulSDMA_CYC) * lNewMaxPF;               /* CH01  */
      lDeltaMips = ulNewMips-ulOldMips;         /* determine delta DMA mips */
      if (lDeltaMips > (LONG)prtsk->TSK_prdsp->DSP_ulCPSFree) {
	MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspalloc::ChangeDMA error ulRC %x CDMA. Task=%s. CPSfree=%ld. Required=%ld\n",
	       DSP_INSUFF_CPS,
	       prtsk->TSK_pszVirtName,
	       prtsk->TSK_prdsp->DSP_ulCPSFree, lDeltaMips);
	return (DSP_INSUFF_CPS);                    /* not enough CPS left */
      }
   /* End   CH01 MTS #2544 and #2532                                        */

   /*************************************************************************/
   /*CPS values will be updated after assuring nothing wrong with           */
   /*stride/hold.                                                           */
   /*************************************************************************/

   }

   if ((usStride != 0) || (usHold != 0))/* Any stride/hold modification?    */
      {
      if (prldmaNod1->LDMA_usflg == LDMA_SHARABLE)
         return (DSP_INV_PARAMETER);   /* Can't modify sharable entry       */
      if (prldmaNod1->LDMA_usSlotNum == 0)
         return (DSP_INV_PARAMETER);   /* Can't modify non-existent entry   */

   /*************************************************************************/
   /*Build stride hold entry                                                */
   /*************************************************************************/

      if (usStride == 0)               /* if stride not being changed       */
         usStride = prldmaNod1->LDMA_ulStride;/* use existing stride        */
      if (usHold == 0)                 /* if hold not being changed         */
         usHold = prldmaNod1->LDMA_ulHold;/* use existing hold              */
      if (usStride < usHold)           /* stride must be >=hold             */
         return (DSP_INV_PARAMETER);
      usStrideHoldEntry = usStride;    /* pick up Stride                    */
      usTemp = usHold << HoldShift;    /* shift Hold left                   */
      usStrideHoldEntry = usStrideHoldEntry|usTemp;/* Or Hold in            */
      if (ulStrideStart == 0)          /* get dsp stride table start if     */
                                       /* first time                        */
         {
         ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, "STRIDE", &ulStrideStart
            );
         if (ulRC != DSP_NOERROR)
            return (ulRC);
      }

   /*************************************************************************/
   /*Write new entry into proper slot in dsp stride/hold table              */
   /*************************************************************************/

      usTableOffset = 2*prldmaNod1->LDMA_usSlotNum;/* offset into s/h table */

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspalloc::ChangeDMA");
   PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", prtsk);
   PRINT_DEBUG(EMBEDFILEPTR, "  pszDMAName \"%s\"", pszDMAName);
   PRINT_DEBUG(EMBEDFILEPTR, "  lPackets (0x%X)\n", lPackets);
   PRINT_DEBUG(EMBEDFILEPTR, "  lMaxPacketSize (0x%X)", lMaxPacketSize);
   PRINT_DEBUG(EMBEDFILEPTR, "  lMaxPacketPF (0x%X)\n", lMaxPacketPF);
   PRINT_DEBUG(EMBEDFILEPTR, "  usStride (0x%X)", usStride);
   PRINT_DEBUG(EMBEDFILEPTR, "  usHold (0x%X)\n", usHold);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "WRITE33:  new entry into proper slot in dsp stride/hold table",
        EOF, 1,
        prtsk->TSK_prdsp, ulStrideStart+usTableOffset, 1, &usStrideHoldEntry);

      ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, /* dsp handle                */
         ulStrideStart+usTableOffset,  /* dsp address                       */
         1,                            /* number of dsp words               */
         &usStrideHoldEntry);          /* pc address                        */
      if (ulRC != DSP_NOERROR)
         return (ulRC);
      prldmaNod1->LDMA_ulStride = usStride;/* fix up stride in dma structure*/
      prldmaNod1->LDMA_ulHold = usHold;/* fix up hold in dma structure      */
   }

/****************************************************************************/
/*All parameters have been validated.  Now the manager structure            */
/*values can be updated.                                                    */
/*First fix up CPS values                                                   */
/****************************************************************************/

   prtsk->TSK_ulCPS += lDeltaMips;     /* update cps value for this task    */
   prtsk->TSK_prdsp->DSP_ulCPSFree -= lDeltaMips;/* update Free Mips        */

/****************************************************************************/
/*Now Update DMA structure packet information                               */
/****************************************************************************/

   prldmaNod1->LDMA_lPackets = lNewPackets;
   prldmaNod1->LDMA_ulMaxPkSize = lNewMaxSize;
   prldmaNod1->LDMA_lMaxPerDMA = lNewMaxPF;

/****************************************************************************/
/*Update Bus Capacity                                                       */
/****************************************************************************/

   pg->prmgrTail->MGR_ulRemBusCapacity = /* remaining capacity =            */
      pg->prmgrTail->MGR_ulRemBusCapacity-/* previous value -               */
      lDeltaBusUse;                    /* delta for this dspChangeDMA       */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspalloc::ChangeDMA exit ulRC %lx\n",ulRC);
   return (ulRC);
}

