/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.util;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.SAMTextHeaderCodec;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.FormatUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalCoordinateComparator;
import htsjdk.samtools.util.ListMap;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.OverlapDetector;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringLineReader;
import htsjdk.samtools.util.StringUtil;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

public class IntervalList
implements Iterable<Interval> {
    private final SAMFileHeader header;
    private final List<Interval> intervals = new ArrayList<Interval>();
    private static final Log log = Log.getInstance(IntervalList.class);

    public IntervalList(SAMFileHeader header) {
        if (header == null) {
            throw new IllegalArgumentException("SAMFileHeader must be supplied.");
        }
        this.header = header;
    }

    public SAMFileHeader getHeader() {
        return this.header;
    }

    @Override
    public Iterator<Interval> iterator() {
        return this.intervals.iterator();
    }

    public void add(Interval interval) {
        this.intervals.add(interval);
    }

    public void addall(Collection<Interval> intervals) {
        this.intervals.addAll(intervals);
    }

    @Deprecated
    public void sort() {
        Collections.sort(this.intervals, new IntervalCoordinateComparator(this.header));
        this.header.setSortOrder(SAMFileHeader.SortOrder.coordinate);
    }

    public IntervalList sorted() {
        IntervalList sorted = IntervalList.copyOf(this);
        Collections.sort(sorted.intervals, new IntervalCoordinateComparator(sorted.header));
        sorted.header.setSortOrder(SAMFileHeader.SortOrder.coordinate);
        return sorted;
    }

    public IntervalList uniqued() {
        return this.uniqued(true);
    }

    public IntervalList uniqued(boolean concatenateNames) {
        List<Interval> tmp = IntervalList.getUniqueIntervals(this.sorted(), concatenateNames);
        IntervalList value = new IntervalList(this.header.clone());
        value.intervals.addAll(tmp);
        return value;
    }

    @Deprecated
    public void unique() {
        this.unique(true);
    }

    @Deprecated
    public void unique(boolean concatenateNames) {
        this.sort();
        List<Interval> tmp = this.getUniqueIntervals(concatenateNames);
        this.intervals.clear();
        this.intervals.addAll(tmp);
    }

    public List<Interval> getIntervals() {
        return Collections.unmodifiableList(this.intervals);
    }

    @Deprecated
    public List<Interval> getUniqueIntervals() {
        return this.getUniqueIntervals(true);
    }

    public static List<Interval> getUniqueIntervals(IntervalList list, boolean concatenateNames) {
        return IntervalList.getUniqueIntervals(list, concatenateNames, false);
    }

    public static List<Interval> getUniqueIntervals(IntervalList list, boolean concatenateNames, boolean enforceSameStrands) {
        List<Interval> intervals = list.getHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate ? list.sorted().intervals : list.intervals;
        ArrayList<Interval> unique = new ArrayList<Interval>();
        TreeSet<Interval> toBeMerged = new TreeSet<Interval>();
        Interval current = null;
        for (Interval next : intervals) {
            if (current == null) {
                toBeMerged.add(next);
                current = next;
                continue;
            }
            if (current.intersects(next) || current.abuts(next)) {
                if (enforceSameStrands && current.isNegativeStrand() != next.isNegativeStrand()) {
                    throw new SAMException("Strands were not equal for: " + current.toString() + " and " + next.toString());
                }
                toBeMerged.add(next);
                current = new Interval(current.getSequence(), current.getStart(), Math.max(current.getEnd(), next.getEnd()), current.isNegativeStrand(), null);
                continue;
            }
            unique.add(IntervalList.merge(toBeMerged, concatenateNames));
            toBeMerged.clear();
            current = next;
            toBeMerged.add(current);
        }
        if (toBeMerged.size() > 0) {
            unique.add(IntervalList.merge(toBeMerged, concatenateNames));
        }
        return unique;
    }

    @Deprecated
    public List<Interval> getUniqueIntervals(boolean concatenateNames) {
        if (this.getHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            this.sort();
        }
        return IntervalList.getUniqueIntervals(this, concatenateNames);
    }

    static Interval merge(SortedSet<Interval> intervals, boolean concatenateNames) {
        String chrom = intervals.first().getSequence();
        int start = intervals.first().getStart();
        int end = intervals.last().getEnd();
        boolean neg = intervals.first().isNegativeStrand();
        LinkedHashSet<String> names = new LinkedHashSet<String>();
        for (Interval i : intervals) {
            if (i.getName() != null) {
                names.add(i.getName());
            }
            start = Math.min(start, i.getStart());
            end = Math.max(end, i.getEnd());
        }
        String name = concatenateNames ? (names.isEmpty() ? null : StringUtil.join("|", names)) : (String)names.iterator().next();
        return new Interval(chrom, start, end, neg, name);
    }

    public long getBaseCount() {
        return Interval.countBases(this.intervals);
    }

    public long getUniqueBaseCount() {
        return this.uniqued().getBaseCount();
    }

    public int size() {
        return this.intervals.size();
    }

    public static IntervalList copyOf(IntervalList list) {
        IntervalList clone = new IntervalList(list.header.clone());
        clone.intervals.addAll(list.intervals);
        return clone;
    }

    public static IntervalList fromFile(File file) {
        BufferedReader reader = IOUtil.openFileForBufferedReading(file);
        IntervalList list = IntervalList.fromReader(reader);
        try {
            reader.close();
        }
        catch (IOException e) {
            throw new SAMException(String.format("Failed to close file %s after reading", file));
        }
        return list;
    }

    public static IntervalList fromFiles(Collection<File> intervalListFiles) {
        ArrayList<IntervalList> intervalLists = new ArrayList<IntervalList>();
        for (File file : intervalListFiles) {
            intervalLists.add(IntervalList.fromFile(file));
        }
        return IntervalList.union(intervalLists);
    }

    public static IntervalList fromReader(BufferedReader in) {
        IntervalList intervalList;
        try {
            StringBuilder builder = new StringBuilder(4096);
            String line = null;
            while ((line = in.readLine()) != null && line.startsWith("@")) {
                builder.append(line).append('\n');
            }
            if (builder.length() == 0) {
                throw new IllegalStateException("Interval list file must contain header. ");
            }
            StringLineReader headerReader = new StringLineReader(builder.toString());
            SAMTextHeaderCodec codec = new SAMTextHeaderCodec();
            IntervalList list = new IntervalList(codec.decode(headerReader, "BufferedReader"));
            SAMSequenceDictionary dict = list.getHeader().getSequenceDictionary();
            if (line == null) {
                IntervalList intervalList2 = list;
                return intervalList2;
            }
            FormatUtil format = new FormatUtil();
            do {
                boolean negative;
                if (line.trim().length() == 0) continue;
                String[] fields = line.split("\t");
                if (fields.length != 5) {
                    throw new SAMException("Invalid interval record contains " + fields.length + " fields: " + line);
                }
                String seq = fields[0];
                int start = format.parseInt(fields[1]);
                int end = format.parseInt(fields[2]);
                if (fields[3].equals("-")) {
                    negative = true;
                } else if (fields[3].equals("+")) {
                    negative = false;
                } else {
                    throw new IllegalArgumentException("Invalid strand field: " + fields[3]);
                }
                String name = fields[4];
                Interval interval = new Interval(seq, start, end, negative, name);
                if (dict.getSequence(seq) == null) {
                    log.warn("Ignoring interval for unknown reference: " + interval);
                    continue;
                }
                list.intervals.add(interval);
            } while ((line = in.readLine()) != null);
            intervalList = list;
        }
        catch (IOException ioe) {
            throw new SAMException("Error parsing interval list.", ioe);
        }
        finally {
            try {
                in.close();
            }
            catch (Exception e) {}
        }
        return intervalList;
    }

    public void write(File file) {
        try {
            BufferedWriter out = IOUtil.openFileForBufferedWriting(file);
            FormatUtil format = new FormatUtil();
            if (this.header != null) {
                SAMTextHeaderCodec codec = new SAMTextHeaderCodec();
                codec.encode(out, this.header);
            }
            for (Interval interval : this) {
                out.write(interval.getSequence());
                out.write(9);
                out.write(format.format(interval.getStart()));
                out.write(9);
                out.write(format.format(interval.getEnd()));
                out.write(9);
                out.write(interval.isPositiveStrand() ? 43 : 45);
                out.write(9);
                if (interval.getName() != null) {
                    out.write(interval.getName());
                } else {
                    out.write(".");
                }
                out.newLine();
            }
            out.flush();
            out.close();
        }
        catch (IOException ioe) {
            throw new SAMException("Error writing out interval list to file: " + file.getAbsolutePath(), ioe);
        }
    }

    public static IntervalList intersection(IntervalList list1, IntervalList list2) {
        SequenceUtil.assertSequenceDictionariesEqual(list1.getHeader().getSequenceDictionary(), list2.getHeader().getSequenceDictionary());
        IntervalList result = new IntervalList(list1.getHeader().clone());
        OverlapDetector<Interval> detector = new OverlapDetector<Interval>(0, 0);
        detector.addAll(list1.getIntervals(), list1.getIntervals());
        for (Interval i : list2.getIntervals()) {
            Collection as = detector.getOverlaps(i);
            for (Interval j : as) {
                Interval tmp = i.intersect(j);
                result.add(tmp);
            }
        }
        return result.uniqued();
    }

    public static IntervalList intersection(Collection<IntervalList> lists) {
        IntervalList intersection = null;
        for (IntervalList list : lists) {
            if (intersection == null) {
                intersection = list;
                continue;
            }
            intersection = IntervalList.intersection(intersection, list);
        }
        return intersection;
    }

    public static IntervalList concatenate(Collection<IntervalList> lists) {
        if (lists.isEmpty()) {
            throw new SAMException("Cannot concatenate an empty list of IntervalLists.");
        }
        SAMFileHeader header = lists.iterator().next().getHeader().clone();
        header.setSortOrder(SAMFileHeader.SortOrder.unsorted);
        IntervalList merged = new IntervalList(header);
        for (IntervalList in : lists) {
            SequenceUtil.assertSequenceDictionariesEqual(merged.getHeader().getSequenceDictionary(), in.getHeader().getSequenceDictionary());
            merged.addall(in.intervals);
        }
        return merged;
    }

    public static IntervalList union(Collection<IntervalList> lists) {
        IntervalList merged = IntervalList.concatenate(lists);
        return merged.uniqued();
    }

    public static IntervalList union(IntervalList list1, IntervalList list2) {
        List<IntervalList> duo = CollectionUtil.makeList(list1, list2);
        return IntervalList.union(duo);
    }

    public static IntervalList invert(IntervalList list) {
        IntervalList inverse = new IntervalList(list.header.clone());
        ListMap<Integer, Interval> map = new ListMap<Integer, Interval>();
        for (Interval i : list.uniqued().getIntervals()) {
            map.add(list.getHeader().getSequenceIndex(i.getSequence()), i);
        }
        int intervals = 0;
        for (SAMSequenceRecord samSequenceRecord : list.getHeader().getSequenceDictionary().getSequences()) {
            Integer sequenceIndex = samSequenceRecord.getSequenceIndex();
            String sequenceName = samSequenceRecord.getSequenceName();
            int sequenceLength = samSequenceRecord.getSequenceLength();
            Integer lastCoveredPosition = 0;
            if (map.containsKey(sequenceIndex)) {
                for (Interval i : (List)map.get(sequenceIndex)) {
                    if (i.getStart() > lastCoveredPosition + 1) {
                        inverse.add(new Interval(sequenceName, lastCoveredPosition + 1, i.getStart() - 1, false, "interval-" + ++intervals));
                    }
                    lastCoveredPosition = i.getEnd();
                }
            }
            if (sequenceLength <= lastCoveredPosition) continue;
            inverse.add(new Interval(sequenceName, lastCoveredPosition + 1, sequenceLength, false, "interval-" + ++intervals));
        }
        return inverse;
    }

    public static IntervalList subtract(Collection<IntervalList> listsToSubtractFrom, Collection<IntervalList> listsToSubtract) {
        return IntervalList.intersection(IntervalList.union(listsToSubtractFrom), IntervalList.invert(IntervalList.union(listsToSubtract)));
    }

    public static IntervalList difference(Collection<IntervalList> lists1, Collection<IntervalList> lists2) {
        return IntervalList.union(IntervalList.subtract(lists1, lists2), IntervalList.subtract(lists2, lists1));
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        IntervalList intervals1 = (IntervalList)o;
        return this.header.equals(intervals1.header) && ((Object)this.intervals).equals(intervals1.intervals);
    }

    public int hashCode() {
        int result = this.header.hashCode();
        result = 31 * result + ((Object)this.intervals).hashCode();
        return result;
    }
}

