//==============================================================================================
//  font3d.cc                                                                      Font3D
//----------------------------------------------------------------------------------------------
//
//  Copyright (c) 1994-1996 by Todd A. Prater                                   Version 1.60
//  All rights reserved.
//
//----------------------------------------------------------------------------------------------
//
//  Permission to copy and distribute Font3D in its entirety, for noncommercial purposes,
//  is hereby granted without fee, provided that this license information and copyright 
//  notice appear in all copies. 
//
//  If you redistribute Font3D, the entire contents of this distribution must be distributed,
//  including the readme.txt, and register.txt files, and the complete set of documentation,
//  both ASCII text, and PostScript files. 
//
//  The software may be modified for your own purposes, but modified versions may not be
//  distributed without prior consent of the author.
//
//  This software is provided 'asis', without any express or implied warranty.  In no event
//  will the author be held liable for any damages arising from the use of this software.  
//
//  If you would like to do something with Font3D that this copyright prohibits (such as 
//  distributing it with a commercial product, using portions of the source in some other
//  program, distributing registered copies, etc.), please contact the author (preferably
//  via email).  Arrangements can probably be worked out.
//
//==============================================================================================

#include <math.h>
#include <stdlib.h>
#include <iostream.h>
#include <iomanip.h>
#include <stdio.h>
#include <string.h>

#include "truetype.h"
#include "build.h"
#include "font3d.h"


//==============================================================================================
//  PrintGreeting()
//----------------------------------------------------------------------------------------------
//  This function prints out the greeting that is seen every time the program is run.
//==============================================================================================

   void PrintGreeting(void)
   {
      cout<<"-------------------------------------------------------------------"<<endl;
      cout<<"                       Font3D Version 1.60                         "<<endl;
      cout<<"                 3-D Text Object Creation Utility                  "<<endl;
      cout<<endl;
      cout<<"   Copyright 1994-1996 by Todd A. Prater.  All rights reserved.    "<<endl;
      cout<<"-------------------------------------------------------------------"<<endl;
   }


//==============================================================================================
//  PrintPixarCopyright()
//----------------------------------------------------------------------------------------------
//  This function prints out the Pixar copyright notice that is written out whenever a RIB
//  type output file is written.
//==============================================================================================

   void PrintPixarCopyright(void)
   {
      cout<<"     The RenderMan Interface Procedures and RIB Protocol are:      "<<endl;
      cout<<"                   Copyright 1988, 1989, Pixar.                    "<<endl;
      cout<<"                       All rights reserved.                        "<<endl;
      cout<<"           RenderMan is a registered trademark of Pixar.           "<<endl;
      cout<<"-------------------------------------------------------------------"<<endl;
   }


//==============================================================================================
//  OutputHeader()
//==============================================================================================
//  This function writes the informational header for an output file.  This header contains
//  a listing of all program options and their current values.
//==============================================================================================

   void OutputHeader(ostream& outputFile, const char* commentBegin, Font3DOptions& options,
                     const char* fontName, int triangleCount)
   {
      outputFile<<commentBegin
                <<"====================================================================="<<endl;
      outputFile<<commentBegin
                <<"                     This file was generated by                      "<<endl;
      outputFile<<commentBegin
                <<"                        Font3D Version 1.60                          "<<endl;
      outputFile<<commentBegin
                <<"                 A 3-D Text Object Creation Utility                  "<<endl;
      outputFile<<commentBegin
                <<"====================================================================="<<endl;
      outputFile<<commentBegin<<endl;

      outputFile<<commentBegin<<"  String:               ";
      for (int i=0;i<options.stringLength;i++) outputFile<<(char)options.string[i];
      outputFile<<endl;
      outputFile<<commentBegin<<"  Font:                 "<<options.fontFileName
                              <<" ("<<fontName<<")"<<endl;
      outputFile<<commentBegin<<"  Encoding:             ";
      switch (options.mapType)
      {
         case MACINTOSH:  outputFile<<"MACINTOSH"<<endl; break;
         case MICROSOFT:  outputFile<<"MICROSOFT"<<endl; break;
         default:   outputFile<<"Unknown"<<endl;
      }
      outputFile<<commentBegin<<"  Number of Triangles:  "<<triangleCount<<endl;
      outputFile<<commentBegin<<endl;
      outputFile<<commentBegin<<"  Configuration Information:"<<endl;
      outputFile<<commentBegin<<endl;

      outputFile<<commentBegin<<"                      name="<<options.objectName<<endl;
      outputFile<<commentBegin<<"                    format=";
      switch (options.outputFormat)
      {
         case POV:   outputFile<<"POV"<<endl; break;
         case POV3:  outputFile<<"POV3"<<endl; break;
         case RAW:   outputFile<<"RAW"<<endl; break;
         case RIB:   outputFile<<"RIB"<<endl; break;
         case DXF:   outputFile<<"DXF"<<endl; break;
         case DXF2:  outputFile<<"DXF2"<<endl; break;
         case VIVID: outputFile<<"VIVID"<<endl; break;
         case RAD:   outputFile<<"RAD"<<endl; break;
         default:    outputFile<<"Unknown"<<endl;
      }
      outputFile<<commentBegin<<"             triangle-type=";
      switch (options.triangleType)
      {
         case SMOOTH:  outputFile<<"SMOOTH"<<endl; break;
         case FLAT:    outputFile<<"FLAT"<<endl; break;
         default:      outputFile<<"Unknown"<<endl;
      }
      outputFile<<commentBegin<<"                bevel-type=";
      switch (options.bevelType)
      {
         case ROUND:   outputFile<<"ROUND"<<endl; break;
         case FLAT:    outputFile<<"FLAT"<<endl; break;
         default:      outputFile<<"Unknown"<<endl;
      }

      outputFile<<commentBegin<<"                     depth="<<options.depth<<endl;
      outputFile<<commentBegin<<"                      xpos=";
      switch (options.xPosition)
      {
         case LEFT:    outputFile<<"LEFT"<<endl; break;
         case CENTER:  outputFile<<"CENTER"<<endl; break;
         case RIGHT:   outputFile<<"RIGHT"<<endl; break;
         default:      outputFile<<"Unknown"<<endl;
      }
      outputFile<<commentBegin<<"                      ypos=";
      switch (options.yPosition)
      {
         case BOTTOM:   outputFile<<"BOTTOM"<<endl; break;
         case BASELINE: outputFile<<"BASELINE"<<endl; break;
         case CENTER:   outputFile<<"CENTER"<<endl; break;
         case TOP:      outputFile<<"TOP"<<endl; break;
         default:       outputFile<<"Unknown"<<endl;
      }
      outputFile<<commentBegin<<"                      zpos=";
      switch (options.zPosition)
      {
         case BACK:     outputFile<<"BACK"<<endl; break;
         case CENTER:   outputFile<<"CENTER"<<endl; break;
         case FRONT:    outputFile<<"FRONT"<<endl; break;
         default:       outputFile<<"Unknown"<<endl;
      }
      outputFile<<commentBegin<<"                front-face=";
      if (options.frontFaceVisible) outputFile<<"ON"<<endl; else outputFile<<"OFF"<<endl;
      outputFile<<commentBegin<<"               front-bevel=";
      if (options.frontBevelVisible) outputFile<<"ON"<<endl; else outputFile<<"OFF"<<endl;
      outputFile<<commentBegin<<"                     sides=";
      if (options.sideVisible) outputFile<<"ON"<<endl; else outputFile<<"OFF"<<endl;
      outputFile<<commentBegin<<"                back-bevel=";
      if (options.backBevelVisible) outputFile<<"ON"<<endl; else outputFile<<"OFF"<<endl;
      outputFile<<commentBegin<<"                 back-face=";
      if (options.backFaceVisible) outputFile<<"ON"<<endl; else outputFile<<"OFF"<<endl;
      outputFile<<commentBegin<<"            front-face-cut="<<options.frontFaceCut<<endl;
      outputFile<<commentBegin<<"            front-side-cut="<<options.frontSideCut<<endl;
      outputFile<<commentBegin<<"             back-face-cut="<<options.backFaceCut<<endl;
      outputFile<<commentBegin<<"             back-side-cut="<<options.backSideCut<<endl;
      if (options.frontFaceTextureName)
      {
         outputFile<<commentBegin<<"        front-face-texture=";
         outputFile<<options.frontFaceTextureName<<endl;
      }
      if (options.frontBevelTextureName) 
      {
         outputFile<<commentBegin<<"       front-bevel-texture=";
         outputFile<<options.frontBevelTextureName<<endl;
      }
      if (options.sideTextureName)
      {
         outputFile<<commentBegin<<"              side-texture=";
         outputFile<<options.sideTextureName<<endl;
      }
      if (options.backBevelTextureName)
      {
         outputFile<<commentBegin<<"        back-bevel-texture=";
         outputFile<<options.backBevelTextureName<<endl;
      }
      if (options.backFaceTextureName)
      {
         outputFile<<commentBegin<<"         back-face-texture=";
         outputFile<<options.backFaceTextureName<<endl;
      }
      outputFile<<commentBegin<<"       smoothing-threshold="<<options.threshold<<endl;
      outputFile<<commentBegin<<"                resolution="<<options.resolution<<endl;
      outputFile<<commentBegin<<"                 precision="<<options.outputPrecision<<endl;

      outputFile<<commentBegin<<"         coordinate-system=";
      if (options.csystem==RIGHT_HANDED) outputFile<<"RIGHT"<<endl;
      else outputFile<<"LEFT"<<endl;

      outputFile<<commentBegin<<endl;
      outputFile<<commentBegin
                <<"---------------------------------------------------------------------"<<endl;
   }


//==============================================================================================
//  OutputTriangles()
//==============================================================================================
//
//  SYNTAX:       void OutputTriangles(ostream& outputFile, int outputFormat,
//                                     int triangleType, int spacesToIndent,
//                                     TRIANGLELIST& triangleList);
//
//  DESCRIPTION:  This function outputs a list of triangles to the stream 'outputFile'.
//                The 'outputFormat' parameter specifies the output format (ie. POV, RAW,
//                RIB, etc.);  the following symbolic constants are defined in config.h:
//
//                     Symbol           Description 
//                 ------------------------------------------------------------------------
//                     RAW..............Output raw vertex data (ie. nine space-delimited
//                                      values per line, corresponding to the x-y-z coor-
//                                      dinates of each vertex.)
//                     RIB..............Output RenderMan compatible triangles.
//                     POV..............Output Persistence of Vision V2.x compatible
//                                      triangles.
//                     RAD..............Output Radiance 2.x compatible triangles.
//
//                The 'triangleType' parameter specifies the type of triangle to output.
//                The following symbolic constants are defined in config.h:
//
//                     Symbol           Description
//                 ------------------------------------------------------------------------ 
//                     FLAT.............Output flat triangles (only vertex data).
//                     SMOOTH...........Output smooth triangles (vertices and normals).
//
//                The 'spacesToIndent' parameter specifies the number of spaces to indent
//                each line of output.
//
//  NOTES:        When adding a new output format, you must add a case to BOTH conditionals
//                SMOOTH and FLAT, even if they are both exactly the same.
//
//==============================================================================================

   void OutputTriangles(ostream&      outputFile,
                        int           outputFormat,
                        int           triangleType,
                        int           spacesToIndent,
                        TRIANGLELIST& triangleList,
			char*	      texture,
                        int           csystem)
   {

      int       i;
      char*     indentString;
      TRIANGLE* t;

      double    v1x,v1y,v1z,v2x,v2y,v2z,v3x,v3y,v3z;
      double    n1x,n1y,n1z,n2x,n2y,n2z,n3x,n3y,n3z;
      vector    side1, side2;
      vector    normal;

      indentString = new char[spacesToIndent+1];
      for (i=0;i<spacesToIndent;i++) indentString[i]=' ';
      indentString[i]=0x00;

      triangleList.gotoFirst();
      for (i=0;i<triangleList.Count();i++)
      {
         t = triangleList.Current();

         v1x=t->v1.x; v1y=t->v1.y; if (csystem==RIGHT_HANDED) v1z=t->v1.z; else v1z=-t->v1.z;
         v2x=t->v2.x; v2y=t->v2.y; if (csystem==RIGHT_HANDED) v2z=t->v2.z; else v2z=-t->v2.z;
         v3x=t->v3.x; v3y=t->v3.y; if (csystem==RIGHT_HANDED) v3z=t->v3.z; else v3z=-t->v3.z;
         n1x=t->n1.x; n1y=t->n1.y; if (csystem==RIGHT_HANDED) n1z=t->n1.z; else n1z=-t->n1.z;
         n2x=t->n2.x; n2y=t->n2.y; if (csystem==RIGHT_HANDED) n2z=t->n2.z; else n2z=-t->n2.z;
         n3x=t->n3.x; n3y=t->n3.y; if (csystem==RIGHT_HANDED) n3z=t->n3.z; else n3z=-t->n3.z;

         side1  = t->v1 - t->v2;
         side2  = t->v3 - t->v2;
         normal = side1 ^ side2;

         if (length(normal) < DEGENERATE_THRESHOLD)
	 {
            triangleList.gotoNext();
            continue;
	 }

         if (triangleType==SMOOTH)
         {
            if (outputFormat==POV)
            {
               outputFile<<indentString<<"smooth_triangle{";
               outputFile<<'<'<<v1x<<','<<v1y<<','<<v1z<<'>'<<',';
               outputFile<<'<'<<n1x<<','<<n1y<<','<<n1z<<'>'<<',';
               outputFile<<'<'<<v2x<<','<<v2y<<','<<v2z<<'>'<<',';
               outputFile<<'<'<<n2x<<','<<n2y<<','<<n2z<<'>'<<',';
               outputFile<<'<'<<v3x<<','<<v3y<<','<<v3z<<'>'<<',';
               outputFile<<'<'<<n3x<<','<<n3y<<','<<n3z<<'>'<<'}'<<endl;
            }
            else if (outputFormat==POV3)
            {
               outputFile<<indentString<<"smooth_triangle{";
               outputFile<<'<'<<v1x<<','<<v1y<<','<<v1z<<'>'<<',';
               outputFile<<'<'<<n1x<<','<<n1y<<','<<n1z<<'>'<<',';
               outputFile<<'<'<<v2x<<','<<v2y<<','<<v2z<<'>'<<',';
               outputFile<<'<'<<n2x<<','<<n2y<<','<<n2z<<'>'<<',';
               outputFile<<'<'<<v3x<<','<<v3y<<','<<v3z<<'>'<<',';
               outputFile<<'<'<<n3x<<','<<n3y<<','<<n3z<<'>';
               if (texture) outputFile<<"texture{"<<texture<<'}';
               outputFile<<'}'<<endl;
            }
            else if (outputFormat==RAW)
            {
               outputFile<<indentString;
               outputFile<<v1x<<' '<<v1y<<' '<<v1z<<' ';
               outputFile<<v2x<<' '<<v2y<<' '<<v2z<<' ';
               outputFile<<v3x<<' '<<v3y<<' '<<v3z<<endl;
            }
            else if (outputFormat==RIB)
            {
               outputFile<<indentString<<"Polygon \"P\" [";
               outputFile<<v1x<<' '<<v1y<<' '<<v1z<<' ';
               outputFile<<v2x<<' '<<v2y<<' '<<v2z<<' ';
               outputFile<<v3x<<' '<<v3y<<' '<<v3z<<" ] ";
               outputFile<<"\"N\" [";
               outputFile<<n1x<<' '<<n1y<<' '<<n1z<<' ';
               outputFile<<n2x<<' '<<n2y<<' '<<n2z<<' ';
               outputFile<<n3x<<' '<<n3y<<' '<<n3z<<" ]"<<endl;
            }
            else if (outputFormat==DXF)
	    {
               outputFile<<"  0"<<endl;
               outputFile<<"3DFACE"<<endl;
               outputFile<<"  8"<<endl;
               outputFile<<0<<endl;
               outputFile<<" 10"<<endl;
               outputFile<<v1x<<endl;
               outputFile<<" 20"<<endl;
               outputFile<<v1y<<endl;
               outputFile<<" 30"<<endl;
               outputFile<<v1z<<endl;
               outputFile<<" 11"<<endl;
               outputFile<<v2x<<endl;
               outputFile<<" 21"<<endl;
               outputFile<<v2y<<endl;
               outputFile<<" 31"<<endl;
               outputFile<<v2z<<endl;
               outputFile<<" 12"<<endl;
               outputFile<<v3x<<endl;
               outputFile<<" 22"<<endl;
               outputFile<<v3y<<endl;
               outputFile<<" 32"<<endl;
               outputFile<<v3z<<endl;
               outputFile<<" 13"<<endl;
               outputFile<<v3x<<endl;
               outputFile<<" 23"<<endl;
               outputFile<<v3y<<endl;
               outputFile<<" 33"<<endl;
               outputFile<<v3z<<endl;
	    }
            else if (outputFormat==VIVID)
	    {
               outputFile<<indentString;
               outputFile<<"patch { vertex "<<v1x<<" "<<v1y<<" "<<v1z<<" "
                                 <<"normal "<<n1x<<" "<<n1y<<" "<<n1z<<" "
                                 <<"vertex "<<v2x<<" "<<v2y<<" "<<v2z<<" "
                                 <<"normal "<<n2x<<" "<<n2y<<" "<<n2z<<" "
                                 <<"vertex "<<v3x<<" "<<v3y<<" "<<v3z<<" "
                                 <<"normal "<<n3x<<" "<<n3y<<" "<<n3z<<" }"<<endl;
	    }
            else if (outputFormat==RAD)
            {
               outputFile<<indentString<<(texture?texture:"text_mat")<<" polygon P"<<i<<endl;
	       outputFile<<"0"<<endl<<"0"<<endl<<"9"<<endl;
               outputFile<<v1x<<' '<<v1y<<' '<<v1z<<endl;
               outputFile<<v2x<<' '<<v2y<<' '<<v2z<<endl;
               outputFile<<v3x<<' '<<v3y<<' '<<v3z<<endl<<endl;
            }
         }
         else
         {
            if (outputFormat==POV)
            {
               outputFile<<indentString<<"triangle{";
               outputFile<<'<'<<v1x<<','<<v1y<<','<<v1z<<'>'<<',';
               outputFile<<'<'<<v2x<<','<<v2y<<','<<v2z<<'>'<<',';
               outputFile<<'<'<<v3x<<','<<v3y<<','<<v3z<<'>'<<'}'<<endl;
            }
            else if (outputFormat==POV3)
            {
               outputFile<<indentString<<"triangle{";
               outputFile<<'<'<<v1x<<','<<v1y<<','<<v1z<<'>'<<',';
               outputFile<<'<'<<v2x<<','<<v2y<<','<<v2z<<'>'<<',';
               outputFile<<'<'<<v3x<<','<<v3y<<','<<v3z<<'>';
               if (texture) outputFile<<"texture{"<<texture<<'}';
               outputFile<<'}'<<endl;
            }
            else if (outputFormat==RAW)
            {
               outputFile<<indentString;
               outputFile<<v1x<<' '<<v1y<<' '<<v1z<<' ';
               outputFile<<v2x<<' '<<v2y<<' '<<v2z<<' ';
               outputFile<<v3x<<' '<<v3y<<' '<<v3z<<endl;
            }
            else if (outputFormat==RIB)
            {
               outputFile<<indentString<<"Polygon \"P\" [";
               outputFile<<v1x<<' '<<v1y<<' '<<v1z<<' ';
               outputFile<<v2x<<' '<<v2y<<' '<<v2z<<' ';
               outputFile<<v3x<<' '<<v3y<<' '<<v3z<<" ]"<<endl;
            }
            else if (outputFormat==DXF)
	    {
               outputFile<<"  0"<<endl;
               outputFile<<"3DFACE"<<endl;
               outputFile<<"  8"<<endl;
               outputFile<<0<<endl;
               outputFile<<" 10"<<endl;
               outputFile<<v1x<<endl;
               outputFile<<" 20"<<endl;
               outputFile<<v1y<<endl;
               outputFile<<" 30"<<endl;
               outputFile<<v1z<<endl;
               outputFile<<" 11"<<endl;
               outputFile<<v2x<<endl;
               outputFile<<" 21"<<endl;
               outputFile<<v2y<<endl;
               outputFile<<" 31"<<endl;
               outputFile<<v2z<<endl;
               outputFile<<" 12"<<endl;
               outputFile<<v3x<<endl;
               outputFile<<" 22"<<endl;
               outputFile<<v3y<<endl;
               outputFile<<" 32"<<endl;
               outputFile<<v3z<<endl;
               outputFile<<" 13"<<endl;
               outputFile<<v3x<<endl;
               outputFile<<" 23"<<endl;
               outputFile<<v3y<<endl;
               outputFile<<" 33"<<endl;
               outputFile<<v3z<<endl;
	    }
            else if (outputFormat==VIVID)
	    {
               outputFile<<indentString;
               outputFile<<"patch { vertex "<<v1x<<" "<<v1y<<" "<<v1z<<" "
                                 <<"normal "<<n1x<<" "<<n1y<<" "<<n1z<<" "
                                 <<"vertex "<<v2x<<" "<<v2y<<" "<<v2z<<" "
                                 <<"normal "<<n2x<<" "<<n2y<<" "<<n2z<<" "
                                 <<"vertex "<<v3x<<" "<<v3y<<" "<<v3z<<" "
                                 <<"normal "<<n3x<<" "<<n3y<<" "<<n3z<<" }"<<endl;
	    }
            else if (outputFormat==RAD)
            {
               outputFile<<indentString<<(texture?texture:"text_mat")<<" polygon P"<<i<<endl;
	       outputFile<<"0"<<endl<<"0"<<endl<<"9"<<endl;
               outputFile<<v1x<<' '<<v1y<<' '<<v1z<<endl;
               outputFile<<v2x<<' '<<v2y<<' '<<v2z<<endl;
               outputFile<<v3x<<' '<<v3y<<' '<<v3z<<endl<<endl;
            }
	 }         

         triangleList.gotoNext();
      }

      delete indentString;

   }


//==============================================================================================
//  OutputObjects()
//==============================================================================================
//
//  SYNTAX:       USHORT OutputObjects(ostream& outputFile, TTFont& font,
//                                     Font3DOptions& options);
//
//  DESCRIPTION:  This function outputs the triangle information that makes up a group of
//                glyphs to a specified stream.  The string to generate is found in the
//                Font3DOptions structure 'options', the font to use is given by 'font',
//                and the output stream is 'outputFile'.
//
//  ERRORS:       Possible error conditions upon return from this function:
//
//                   ERR_NoError...............Successful completion.
//                   ERR_OutOfMemory...........An attempt to allocate memory failed.
//                   There are more...
//
//==============================================================================================

   USHORT OutputObjects(ostream&       outputFile,
                        TTFont&        font,
                        Font3DOptions& options)
   {

      int          i;
      int          triangleCount;
      TRIANGLELIST frontFaceTriangleList;
      TRIANGLELIST backFaceTriangleList;
      TRIANGLELIST frontBevelTriangleList;
      TRIANGLELIST backBevelTriangleList;
      TRIANGLELIST sideTriangleList;
      int          success;

      double       startX,endX;

      double xmin,xmax,ymin,ymax,zmin,zmax,upem,xdelta,ydelta;
      double thisymin,thisymax;
      vector offset(0.0,0.0,0.0);

      int          partsCount;

      char*        constComment = "";

      outputFile<<setprecision(options.outputPrecision);

      upem = (DOUBLE)font.UnitsPerEm();

         if (options.verbose) { cout<<"Generating triangles...["; cout.flush(); }

         startX = font.Glyph(options.string[0])->LeftSideBearing()/upem;
         endX=0;

         ymin = (DOUBLE)font.GlyphYMin(font.CharacterMap(options.string[0]))/(DOUBLE)upem;
         ymax = (DOUBLE)font.GlyphYMax(font.CharacterMap(options.string[0]))/(DOUBLE)upem;

         for (i=0;i<options.stringLength;i++)
         {
            endX+=( font.Glyph(options.string[i])->AdvanceWidth())/upem;
            if (i<options.stringLength) 
               endX += font.Kerning(font.CharacterMap(options.string[i]),
                                    font.CharacterMap(options.string[i+1]))/upem;

            thisymin=(DOUBLE)font.GlyphYMin(font.CharacterMap(options.string[i]))/(DOUBLE)upem;
            thisymax=(DOUBLE)font.GlyphYMax(font.CharacterMap(options.string[i]))/(DOUBLE)upem;

            if (ymin>thisymin) ymin=thisymin;
            if (ymax<thisymax) ymax=thisymax;

         }

         endX -= font.Glyph(options.string[i-1])->RightSideBearing()/upem;

         xmin = startX;
         xmax = endX;

         if (options.xPosition==LEFT) xdelta = -xmin;
         else if (options.xPosition==CENTER) xdelta = -(xmin+xmax)/2;
         else xdelta = -xmax;

         xmin+=xdelta;
         xmax+=xdelta;

         if (options.yPosition==BOTTOM) ydelta = -ymin;
         else if (options.yPosition==BASELINE) ydelta = 0.0;
         else if (options.yPosition==CENTER) ydelta = -(ymin+ymax)/2;
         else ydelta = -ymax;

         ymin+=ydelta;
         ymax+=ydelta;

         if (options.zPosition==FRONT) { zmin=0; zmax=options.depth; }
         else if (options.zPosition==CENTER) { zmin=-options.depth/2; zmax=options.depth/2; }
         else { zmin=-options.depth; zmax=0; }

         offset = vector(xdelta,ydelta,0.0);

         for (i=0;i<options.stringLength;i++)
         {

            success = CreateFaces(font, font.CharacterMap(options.string[i]),
                                  options, offset, frontFaceTriangleList,
                                  backFaceTriangleList);

            if (success!=F3D_ERR_NoError)
            {
               return success;
            }

            success = CreateSides(font, font.CharacterMap(options.string[i]),
                                  options, offset, frontBevelTriangleList,
                                  backBevelTriangleList, sideTriangleList);
            if (success!=F3D_ERR_NoError)
            {
               frontFaceTriangleList.Empty();
               backFaceTriangleList.Empty();
               return success;
            }

            offset.x += (  font.Glyph(options.string[i])->AdvanceWidth() )/upem; 
            if (i<options.stringLength-1) 
               offset.x += font.Kerning(font.CharacterMap(options.string[i]),
                                        font.CharacterMap(options.string[i+1]))/upem;

            if (options.verbose) { cout<<(char)options.string[i];cout.flush(); }

         }

         if (options.verbose) { cout<<"]...Done!"<<endl; }


      triangleCount = 0;

      if (options.frontFaceVisible)
         triangleCount+=frontFaceTriangleList.Count();  
      if (options.backFaceVisible)
         triangleCount+=backFaceTriangleList.Count();
      if (options.frontBevelVisible
          && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         triangleCount+=frontBevelTriangleList.Count();
      if (options.backBevelVisible
          && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         triangleCount+=backBevelTriangleList.Count();
      if (options.sideVisible)
         triangleCount+=sideTriangleList.Count();

      if (options.verbose) { cout<<"Writing output..."; cout.flush(); }

     //---------------------------------------------------------------------------
     //  Output RAW Triangles...
     //---------------------------------------------------------------------------

      if (options.outputFormat==RAW)
      {
         OutputTriangles(outputFile,RAW,FLAT,0,frontFaceTriangleList,NULL,options.csystem);
         OutputTriangles(outputFile,RAW,FLAT,0,backFaceTriangleList,NULL,options.csystem);
         OutputTriangles(outputFile,RAW,FLAT,0,frontBevelTriangleList,NULL,options.csystem);
         OutputTriangles(outputFile,RAW,FLAT,0,backBevelTriangleList,NULL,options.csystem);
         OutputTriangles(outputFile,RAW,FLAT,0,sideTriangleList,NULL,options.csystem);
      }

     //---------------------------------------------------------------------------
     //  Output RenderMan RIB File...
     //---------------------------------------------------------------------------

      else if (options.outputFormat==RIB)
      {
         OutputHeader(outputFile,"#",options,font.FullName(),triangleCount);
         outputFile<<endl;

         if (options.frontFaceVisible)
         {
            outputFile<<"# Front Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,frontFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<endl<<endl;
	 }
         if (options.backFaceVisible)
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,backFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<endl<<endl;
         }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,frontBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<endl<<endl;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,backBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<endl<<endl;
         }
         if (options.sideVisible)
         {
            outputFile<<"# Back Face Triangles:"<<endl<<endl;
            OutputTriangles(outputFile,RIB,options.triangleType,0,sideTriangleList,
                            NULL,options.csystem);
            outputFile<<endl<<endl;
         }
      }


     //---------------------------------------------------------------------------
     //  Output POV 2.x Triangles...
     //---------------------------------------------------------------------------

      else if (options.outputFormat==POV)
      {
         OutputHeader(outputFile,"//",options,font.FullName(),triangleCount);

         if (!options.constants) constComment="//";
         else constComment="";

         outputFile<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_XMin   = "<<xmin<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_YMin   = "<<ymin<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_ZMin   = "<<zmin<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_XMax   = "<<xmax<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_YMax   = "<<ymax<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_ZMax   = "<<zmax<<endl;
         outputFile<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_Width  = "<<(xmax-xmin)
                   <<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_Height = "<<(ymax-ymin)
                   <<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_Depth  = "<<(zmax-zmin)
                   <<endl;
         outputFile<<endl<<endl;

         partsCount = 0;

         if (options.frontFaceVisible)
         {
            outputFile<<"#declare FF_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,frontFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.backFaceVisible)
         {
            outputFile<<"#declare BF_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,backFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         {
            outputFile<<"#declare FB_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,frontBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"#declare BB_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,backBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }
         if (options.sideVisible)
         {
            outputFile<<"#declare S_"<<options.objectName<<" = "<<endl;
            outputFile<<"  union {"<<endl;
            OutputTriangles(outputFile,POV,options.triangleType,4,sideTriangleList,
                            NULL,options.csystem);
            outputFile<<"  }"<<endl<<endl;
            partsCount++;
         }

         if (partsCount < 2)
         {
            outputFile<<"#declare "<<options.objectName<<" = object { ";

            if (options.frontFaceVisible)
            {
               outputFile<<"FF_"<<options.objectName<<" ";
               if (options.frontFaceTextureName != NULL)
                  outputFile<<"texture { "<<options.frontFaceTextureName<<" }";
            }
            else if (options.backFaceVisible)
            {
               outputFile<<"BF_"<<options.objectName<<" ";
               if (options.backFaceTextureName != NULL)
	          outputFile<<"texture { "<<options.backFaceTextureName<<" }";
            }
            else if (options.frontBevelVisible
                     && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
            {
               outputFile<<"FB_"<<options.objectName<<" ";
               if (options.frontBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.frontBevelTextureName<<" }";
            }
            else if (options.backBevelVisible
                     && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
            {
               outputFile<<"BB_"<<options.objectName<<" ";
               if (options.backBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.backBevelTextureName<<" }";
            }
            else if (options.sideVisible)
            {
               outputFile<<"S_"<<options.objectName<<" ";
               if (options.sideTextureName != NULL)
                  outputFile<<"texture { "<<options.sideTextureName<<" }";
            }

            outputFile<<" }"<<endl;
         }
         else
         {
            outputFile<<"#declare "<<options.objectName<<" = ";
            outputFile<<"  union {"<<endl;
            if (options.frontFaceVisible)
            {
               outputFile<<"    object { FF_"<<options.objectName<<" ";
               if (options.frontFaceTextureName != NULL)
                  outputFile<<"texture { "<<options.frontFaceTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.backFaceVisible)
            {
               outputFile<<"    object { BF_"<<options.objectName<<" ";
               if (options.backFaceTextureName != NULL)
	          outputFile<<"texture { "<<options.backFaceTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.frontBevelVisible
                && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
            {
               outputFile<<"    object { FB_"<<options.objectName<<" ";
               if (options.frontBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.frontBevelTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.backBevelVisible
                && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
            {
               outputFile<<"    object { BB_"<<options.objectName<<" ";
               if (options.backBevelTextureName != NULL)
                  outputFile<<"texture { "<<options.backBevelTextureName<<" }";
               outputFile<<" }"<<endl;
            }
            if (options.sideVisible)
            {
               outputFile<<"    object { S_"<<options.objectName<<" ";
               if (options.sideTextureName != NULL)
                  outputFile<<"texture { "<<options.sideTextureName<<" }";
               outputFile<<" }"<<endl;
            }

            outputFile<<" }"<<endl;
         }

      }


     //---------------------------------------------------------------------------
     //  Output POV 3.0 Triangles...
     //---------------------------------------------------------------------------

      else if (options.outputFormat==POV3)
      {
         OutputHeader(outputFile,"//",options,font.FullName(),triangleCount);

         if (!options.constants) constComment="//";
         else constComment="";

         outputFile<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_XMin   = "<<xmin<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_YMin   = "<<ymin<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_ZMin   = "<<zmin<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_XMax   = "<<xmax<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_YMax   = "<<ymax<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_ZMax   = "<<zmax<<endl;
         outputFile<<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_Width  = "<<(xmax-xmin)
                   <<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_Height = "<<(ymax-ymin)
                   <<endl;
         outputFile<<constComment<<"#declare "<<options.objectName<<"_Depth  = "<<(zmax-zmin)
                   <<endl;
         outputFile<<endl<<endl;

         outputFile<<"#declare "<<options.objectName<<" = "<<endl;
         outputFile<<"  mesh {"<<endl;
         if (options.frontFaceVisible)
         {
            OutputTriangles(outputFile,POV3,options.triangleType,4,frontFaceTriangleList,
                            options.frontFaceTextureName,options.csystem);
            outputFile<<endl;
         }
         if (options.backFaceVisible)
         {
            OutputTriangles(outputFile,POV3,options.triangleType,4,backFaceTriangleList,
                            options.backFaceTextureName,options.csystem);
            outputFile<<endl;
         }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
         {
            OutputTriangles(outputFile,POV3,options.triangleType,4,frontBevelTriangleList,
                            options.frontBevelTextureName,options.csystem);
            outputFile<<endl;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            OutputTriangles(outputFile,POV,options.triangleType,4,backBevelTriangleList,
                            options.backBevelTextureName,options.csystem);
            outputFile<<endl;
         }
         if (options.sideVisible)
         {
            OutputTriangles(outputFile,POV,options.triangleType,4,sideTriangleList,
                            options.sideTextureName,options.csystem);
            outputFile<<endl;
         }
         outputFile<<"  }"<<endl;

      }


     //---------------------------------------------------------------------------
     //  Output AutoCad DXF Triangles...
     //---------------------------------------------------------------------------

      else if (options.outputFormat==DXF)
      {
         OutputHeader(outputFile,"999\n",options,font.FullName(),triangleCount);

         if (options.frontFaceVisible)
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,frontFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
	 }
         if (options.backFaceVisible)
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,backFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
	 }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,frontBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
	 }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,backBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
         }
         if (options.sideVisible)
	 {
            outputFile<<"  0"<<endl;
            outputFile<<"SECTION"<<endl;
            outputFile<<"  2"<<endl;
            outputFile<<"ENTITIES"<<endl;
            OutputTriangles(outputFile,DXF,options.triangleType,0,sideTriangleList,
                            NULL,options.csystem);
            outputFile<<"  0"<<endl;
            outputFile<<"ENDSEC"<<endl;
         }
         outputFile<<"  0"<<endl;
         outputFile<<"EOF"<<endl;
      }


     //---------------------------------------------------------------------------
     //  Output AutoCad DXF2 Triangles...WITHOUT SEPARATE ENTITIES
     //
     //  NOTE:  Since the actual triangles are written the same was as the
     //         original DXF files, there is not a separate DXF2 entry in the
     //         OutputTriangles() function.
     //---------------------------------------------------------------------------

      else if (options.outputFormat==DXF2)
      {
         OutputHeader(outputFile,"999\n",options,font.FullName(),triangleCount);
         outputFile<<"  0"<<endl;
         outputFile<<"SECTION"<<endl;
         outputFile<<"  2"<<endl;
         outputFile<<"ENTITIES"<<endl;
         OutputTriangles(outputFile,DXF,options.triangleType,0,frontFaceTriangleList,
                         NULL,options.csystem);
         OutputTriangles(outputFile,DXF,options.triangleType,0,backFaceTriangleList,
                         NULL,options.csystem);
         OutputTriangles(outputFile,DXF,options.triangleType,0,frontBevelTriangleList,
                         NULL,options.csystem);
         OutputTriangles(outputFile,DXF,options.triangleType,0,backBevelTriangleList,
                         NULL,options.csystem);
         OutputTriangles(outputFile,DXF,options.triangleType,0,sideTriangleList,
                         NULL,options.csystem);
         outputFile<<"  0"<<endl;
         outputFile<<"ENDSEC"<<endl;
         outputFile<<"  0"<<endl;
         outputFile<<"EOF"<<endl;
      }


     //---------------------------------------------------------------------------
     //  Output VIVID 2.0 Triangles...
     //---------------------------------------------------------------------------

      else if (options.outputFormat==VIVID)
      {
         OutputHeader(outputFile,"//",options,font.FullName(),triangleCount);

         if (options.frontFaceVisible)
	 {
            outputFile<<"// Front Face Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,frontFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<endl;
	 }
         if (options.backFaceVisible)
	 {
            outputFile<<"// Back Face Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,backFaceTriangleList,
                            NULL,options.csystem);
            outputFile<<endl;
	 }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
	 {
            outputFile<<"// Front Bevel Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,frontBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<endl;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"// Back Bevel Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,backBevelTriangleList,
                            NULL,options.csystem);
            outputFile<<endl;
	 }
         if (options.sideVisible)
	 {
            outputFile<<"// Side Triangles"<<endl<<endl;
            OutputTriangles(outputFile,VIVID,options.triangleType,0,sideTriangleList,
                            NULL,options.csystem);
            outputFile<<endl;
	 }
      }


     //---------------------------------------------------------------------------
     //  Output Radiance 2.x Triangles...
     //---------------------------------------------------------------------------

      else if (options.outputFormat==RAD)
      {
         OutputHeader(outputFile,"#",options,font.FullName(),triangleCount);

         if (options.frontFaceVisible)
	 {
            outputFile<<"# Front Face Triangles"<<endl<<endl;
            OutputTriangles(outputFile,RAD,options.triangleType,0,frontFaceTriangleList,
                            options.frontFaceTextureName,options.csystem);
            outputFile<<endl;
	 }
         if (options.backFaceVisible)
	 {
            outputFile<<"# Back Face Triangles"<<endl<<endl;
            OutputTriangles(outputFile,RAD,options.triangleType,0,backFaceTriangleList,
                            options.backFaceTextureName,options.csystem);
            outputFile<<endl;
	 }
         if (options.frontBevelVisible
             && (options.frontFaceCut!=0.0 || options.frontSideCut!=0.0))
	 {
            outputFile<<"# Front Bevel Triangles"<<endl<<endl;
            OutputTriangles(outputFile,RAD,options.triangleType,0,frontBevelTriangleList,
                            options.frontBevelTextureName,options.csystem);
            outputFile<<endl;
         }
         if (options.backBevelVisible
             && (options.backFaceCut!=0.0 || options.backSideCut!=0.0))
         {
            outputFile<<"# Back Bevel Triangles"<<endl<<endl;
            OutputTriangles(outputFile,RAD,options.triangleType,0,backBevelTriangleList,
                            options.backBevelTextureName,options.csystem);
            outputFile<<endl;
	 }
         if (options.sideVisible)
	 {
            outputFile<<"# Side Triangles"<<endl<<endl;
            OutputTriangles(outputFile,RAD,options.triangleType,0,sideTriangleList,
                            options.sideTextureName,options.csystem);
            outputFile<<endl;
	 }
      }

      if (options.verbose) { cout<<"Done!"<<endl; }

     //---------------------------------------------------------------------------
     //  Clean up...
     //---------------------------------------------------------------------------

      frontFaceTriangleList.Empty();
      backFaceTriangleList.Empty();
      frontBevelTriangleList.Empty();
      backBevelTriangleList.Empty();
      sideTriangleList.Empty();

      return F3D_ERR_NoError;

   }


//==============================================================================================
//  ReadOptions()                                                                               
//----------------------------------------------------------------------------------------------
//                                                                                              
//  SYNTAX:       int  ReadOptions(char* fileName, int numArgs, char & * argument[]);           
//                                                                                              
//  DESCRIPTION:  This function processes a text file that contains a list of program           
//                options.  The options should be separated by a whitespace character,
//                and will be stored in an array of strings 'argument' with their number        
//                in 'numArgs'.  Possible error conditions upon return from this function       
//                are:                                                                          
//                                                                                              
//                     Symbol                  Description                                      
//                 ------------------------------------------------------------------------     
//                     F3D_F3D_ERR_NoError.............The routine completed successfully.
//                     F3D_F3D_ERR_OutOfMemory.........An attempt to allocate memory failed.
//                     F3D_F3D_ERR_UnableToOpenFile....An attempt to open a file failed.
//
//  RETURNS:      One of the defined error conditions in font3d.h.
//                                                                                              
//  NOTE:         Remember to delete 'argument' and the strings contained in it when you're     
//                done with it!                                                                 
//                                                                                              
//==============================================================================================

   int ReadOptions(char* fileName, int& numArgs, char** &argument)
   {
      int  i;                                   // A counter
      char currentChar;                         // The current character being scanned
      char nextChar;                            // The next character to be scanned
      int  currentPosition;                     // A placekeeper for the 'currentString' array
      int  argumentPosition;                    // A placekeeper for the 'currentArgument' array
      char currentString[MAX_STRING_SIZE+1];    // Holds the current string being scanned
      char currentArgument[MAX_STRING_SIZE+1];  // Holds the current argument being scanned

      int  insideQuotes;                        // Flags whether we are inside a pair of quotes
      int  insideArgument;                      // Flags whether we are in the middle of an arg.
      int  quoteChar;                           // Which type of quote (single or double)
                                                //   opened this string

      ifstream inputFile(fileName);                           // Open the configuration file
      if (!inputFile)                                         //   and make sure all is OK
      {
         return F3D_ERR_UnableToOpenFile;
      }

      argument = new CHARPTR[MAX_OPTIONS];                    // Allocate the array of ptrs to
                                                              //   argument strings

      inputFile.width(MAX_LINE_LENGTH);                       // Make sure we don't overwrite
                                                              //   the static arrays

      inputFile.unsetf(ios::skipws);                          // Make sure we don't skip ws.

      numArgs=0;
      argumentPosition=0;
      insideArgument=FALSE;
      do                                                      // For each line of text in the
      {                                                       //   configuration file:
         currentPosition=0;
         inputFile>>currentChar;                              // (1) Read the current line of
         while ((currentChar!='\n') && (!inputFile.eof())     //   text into a buffer...
                && (currentPosition<MAX_LINE_LENGTH))      
         {
            currentString[currentPosition++]=currentChar;
            inputFile>>currentChar;
         }
         if (currentChar=='\n')                               // (2) Make sure the buffer is
         {                                                    //   null terminated
            currentString[currentPosition]=0;
         }

         insideQuotes=FALSE;

         if (insideArgument==TRUE)                            // (3) If we just finished an
         {                                                    //   argument on the previous  
            insideArgument=FALSE;                             //   line, allocate space and
            currentArgument[argumentPosition]=0;              //   copy it to the 'argument'
            argument[numArgs]=new char[argumentPosition+1];   //   array.
            if (argument[numArgs]==NULL)
               return F3D_ERR_OutOfMemory;
            strcpy(argument[numArgs],currentArgument);
            numArgs++;
            argumentPosition=0;
         }

         for (i=0;i<currentPosition;i++)                      // (4) For each character in the
         {                                                    //   buffer:
            currentChar = currentString[i];
            nextChar = currentString[i+1];

            if (currentChar==' ' && insideQuotes==FALSE)      // (4.1) If we find whitespace, and
            {                                                 //   we've just come out of an
               if (insideArgument==TRUE)                      //   argument, allocate space and
               {                                              //   copy it to the 'argument' 
                  insideArgument=FALSE;                       //   array.
                  currentArgument[argumentPosition]=0;
                  argument[numArgs]=new char[argumentPosition+1];
                  if (argument[numArgs]==NULL)
                     return F3D_ERR_OutOfMemory;
                  strcpy(argument[numArgs],currentArgument);
                  numArgs++;
                  argumentPosition=0;
               }
               continue;
            }

            else if (currentChar==COMMENT_CHAR 
                     && insideQuotes==FALSE)                  // (3.2) If we find a comment char
            {                                                 //   not in quotes, we're finished
               break;                                         //   with this line
            }

            else if (currentChar==SINGLE_QUOTE_CHAR           // (3.3) Look for quote characters.
                     || currentChar==DOUBLE_QUOTE_CHAR)       //   If we were not already inside
            {                                                 //   quotes, we are now.  If we
               if (insideQuotes==TRUE)                        //   are in quotes, and this is the
               {                                              //   same type of quote we opened
                  if (currentChar==quoteChar)                 //   we're out now.  In either case
                  {                                           //   go on to the next character.
                     insideQuotes=FALSE;
                     continue;
                  }
               }
               else
               {
                  insideQuotes=TRUE;
                  quoteChar=currentChar;
                  continue;
               }
            }

            if (insideArgument==FALSE)
               insideArgument=TRUE;

            currentArgument[argumentPosition++]=currentChar;  // (3.4) Put the current char into
                                                              //   the current argument string
         }

         if (insideQuotes==TRUE)                              // (3.5) If we are still inside
            return F3D_ERR_UnmatchedQuotes;                   //   quotes at the end of the line
      }                                                       //   it's an error.
      while ((!inputFile.eof()) && (numArgs<MAX_OPTIONS));


      if (insideArgument==TRUE)                               // If we were in the middle of
      {                                                       //   reading an argument before 
         insideArgument=FALSE;                                //   we hit the EOF, allocate
         currentArgument[argumentPosition]=0;                 //   space and copy it to the
         argument[numArgs]=new char[argumentPosition+1];      //   'argument' array.
         if (argument[numArgs]==NULL)
            return F3D_ERR_OutOfMemory;
         strcpy(argument[numArgs],currentArgument);
         numArgs++;
         argumentPosition=0;
      }

      return F3D_ERR_NoError;
   }



//==============================================================================================
//  ParseOptions()                                                                              
//----------------------------------------------------------------------------------------------
//                                                                                              
//  SYNTAX:       int ParseOptions(int numArgs, CHAR* argument[],                              
//                                 Font3DOptions& defaultOptions,                              
//                                 USHORT& errorCode, USHORT& errorPos);                       
//                                                                                              
//  DESCRIPTION:  This function parses a list of option strings, storing any configuration      
//                information in a 'Font3DOptions' structure.  'numArgs' contains the           
//                number of option strings, 'argument[]' is the array of strings, and           
//                'defaultOptions' is the structure used to hold the information recovered      
//                by parsing the strings.  Two USHORT values 'errorCode' and 'errorPos'         
//                hold information about any errors that may have occurred.  'errorPos'         
//                reports the index of the string that caused the error, while 'errorCode'      
//                reports the type of error.  The following symbolic constants are defined      
//                in font3d.h:
//                                                                                              
//                     Symbol           Description                                             
//                 ------------------------------------------------------------------------     
//                     F3D_F3D_ERR_NoError..........The routine completed successfully.
//                     F3D_F3D_ERR_OutOfMemory......An attempt to allocate memory failed.
//                     F3D_F3D_ERR_NoOptionsFound...No option strings were found.
//                     F3D_F3D_ERR_InvalidOption....An invalid option was encountered.
//                                                                                              
//  RETURNS:      TRUE is returned if there were any option strings to process, and if they     
//                were all processed successfully; FALSE otherwise.                             
//                                                                                              
//  NOTES:        All option strings are assumed to be of the following form:                   
//                                                                                              
//                                    optionname=optiondata                                
//                                                                                              
//==============================================================================================

   int ParseOptions(int numArgs, char* argument[], Font3DOptions& defaultOptions,
                    int& errorCode, int& errorPos)
   {
      char  tempChar;
      int   tempInt;
      char* tempString;
      int   i,j,c;
      char* option;                          // The entire option string
      char* optionData;                      // The data part of the option, after the '='
      int   optionDataLength;                // The length of the data part of the option

      int    success;
      char*  configFileName;
      int    numConfigArgs;
      char** configArgs;
      int    configErrorCode;
      int    configErrorPos;

      for (i=0;i<numArgs;i++)
      {
         option = new CHAR[strlen(argument[i])+1];             // Make a new character array to
         if (option==NULL)                                     //  hold the current option;
	 {                                                     //  for in case we shouldn't
            errorCode = F3D_ERR_OutOfMemory;                       //  modify the argument string
            errorPos = i;                                      //  array.
            return FALSE;
         }
         strcpy(option,argument[i]);
                                                               
         optionData = strchr(option,'=');                      // Find the '=' char. If there
         if (optionData==NULL)                                 //  isn't one, this is an inva-
         {                                                     //  lid option.
            errorCode = F3D_ERR_InvalidOption;
            errorPos  = i;
            return FALSE;
         }                                                     // Now, replace the '=' with a 
         *optionData = 0x00;                                   //  null char so we can use the  
         optionData++;                                         //  string lib routines on both
                                                               //  parts of the option string.

         optionDataLength = strlen(optionData);                // Get the length of the data
                                                               //  part of the option string 
         if (optionDataLength==0)                              //  (the part after the '=').
         {                                                     //  If there is nothing there,
            errorCode = F3D_ERR_InvalidOption;                     //  this is an invalid option.
            errorPos = i;
            return FALSE;
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   back-bevel                                                                   
     //-----------------------------------------------------------------------------------------

         if (strcasecmp(option,"back-bevel")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.backBevelVisible = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.backBevelVisible = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   back-bevel-texture                                                            
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"back-bevel-texture")==0)
         {
            if (defaultOptions.backBevelTextureName!=NULL)
               delete defaultOptions.backBevelTextureName;

            defaultOptions.backBevelTextureName = new char[optionDataLength+1];
            if (defaultOptions.backBevelTextureName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }
            strcpy(defaultOptions.backBevelTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   back-face                                                                     
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"back-face")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.backFaceVisible = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.backFaceVisible = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   back-face-cut                                                                 
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"back-face-cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.backFaceCut))
	         {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   back-face-texture                                                             
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"back-face-texture")==0)
         {
            if (defaultOptions.backFaceTextureName!=NULL)
               delete defaultOptions.backFaceTextureName;

            defaultOptions.backFaceTextureName = new char[optionDataLength+1];
            if (defaultOptions.backFaceTextureName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }
            strcpy(defaultOptions.backFaceTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   back-side-cut                                                                 
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"back-side-cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.backSideCut))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   bevel-texture                                                                 
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"bevel-texture")==0)
         {
            if (defaultOptions.frontBevelTextureName!=NULL)
               delete defaultOptions.frontBevelTextureName;
            if (defaultOptions.backBevelTextureName!=NULL)
               delete defaultOptions.backBevelTextureName;

            defaultOptions.frontBevelTextureName = new char[optionDataLength+1];
            defaultOptions.backBevelTextureName = new char[optionDataLength+1];

            if (   defaultOptions.frontBevelTextureName==NULL
                || defaultOptions.backBevelTextureName==NULL  )
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.frontBevelTextureName,optionData);
            strcpy(defaultOptions.backBevelTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   bevel-type                                                                    
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"bevel-type")==0)
         {
            if (strcasecmp(optionData,"ROUND")==0)
               defaultOptions.bevelType = ROUND;
            else if (strcasecmp(optionData,"FLAT")==0)
               defaultOptions.bevelType = FLAT;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   bevels                                                                        
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"bevels")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
	    {
               defaultOptions.frontBevelVisible = TRUE;
               defaultOptions.backBevelVisible  = TRUE;
            }
            else if (strcasecmp(optionData,"OFF")==0)
	    {
               defaultOptions.frontBevelVisible = FALSE;
               defaultOptions.backBevelVisible  = FALSE;
	    }
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   char                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"char")==0)
         {
            if (!sscanf(optionData,"%c",&tempChar))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
            else
            {
               defaultOptions.stringLength=1;
               if (defaultOptions.string!=NULL)
                  delete defaultOptions.string;
               defaultOptions.string=new int[1];
               defaultOptions.string[0]=tempChar;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   code                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"code")==0)
         {
            if (!sscanf(optionData,"%i",&tempInt))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
            else
            {
               defaultOptions.stringLength=1;
               if (defaultOptions.string!=NULL)
                  delete defaultOptions.string;
               defaultOptions.string=new int[1];
               defaultOptions.string[0]=tempInt;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   config                                                                        
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"config")==0)
	 {

            configFileName = new char[optionDataLength+1];

            if (configFileName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(configFileName,optionData);

            success = ReadOptions(configFileName, numConfigArgs, configArgs);

            if (success==F3D_ERR_OutOfMemory)
	    {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos = i;
               return FALSE;
	    }
            else if (success==F3D_ERR_UnableToOpenFile)
	    {
               cout<<endl;
               cout<<"   WARNING: Unable to open configuration file \""
                   <<configFileName<<"\"."<<endl;
               cout<<"            Ignoring..."<<endl<<endl;
	    }
            else if (success==F3D_ERR_UnmatchedQuotes)
            {
               cout<<endl
                   <<"   WARNING: Unmatched quotes in default configuration file \""
                   <<configFileName<<"\"."<<endl;
               cout<<"            Ignoring..."<<endl<<endl;
            }
            else
            {
               success = ParseOptions(numConfigArgs, configArgs, defaultOptions, 
                                      configErrorCode,configErrorPos);
               if (!success)
               {
                  cout<<endl;
                  cout<<"   ERROR: Invalid option \""<<configArgs[configErrorPos]
                      <<"\""<<endl;
                  cout<<"          in configuration file \""<<configFileName
                      <<"\""<<endl<<endl;
                  exit(1);
               }
               for (c=0;c<numConfigArgs;c++)
                  delete configArgs[c];
               delete configArgs;

               delete configFileName;
 	    }

	 }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   constants                                                                     
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"constants")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.constants = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.constants = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   coordinate-system                                                             
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"coordinate-system")==0)
         {
            if (strcasecmp(optionData,"RIGHT")==0)
               defaultOptions.csystem = RIGHT_HANDED;
            else if (strcasecmp(optionData,"LEFT")==0)
               defaultOptions.csystem = LEFT_HANDED;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   cut                                                                           
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.frontFaceCut))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
            defaultOptions.backFaceCut = defaultOptions.frontFaceCut;
            defaultOptions.frontSideCut = defaultOptions.frontFaceCut;
            defaultOptions.backSideCut = defaultOptions.frontFaceCut;
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   depth                                                                         
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"depth")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.depth))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   face-cut                                                                      
     //-----------------------------------------------------------------------------------------
         else if (strcasecmp(option,"face-cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.frontFaceCut))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
            defaultOptions.backFaceCut = defaultOptions.frontFaceCut;
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   face-texture                                                                  
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"face-texture")==0)
         {
            if (defaultOptions.frontFaceTextureName!=NULL)
               delete defaultOptions.frontFaceTextureName;
            if (defaultOptions.backFaceTextureName!=NULL)
               delete defaultOptions.backFaceTextureName;

            defaultOptions.frontFaceTextureName = new char[optionDataLength+1];
            defaultOptions.backFaceTextureName = new char[optionDataLength+1];

            if (   defaultOptions.frontFaceTextureName==NULL
                || defaultOptions.backFaceTextureName==NULL  )
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.frontFaceTextureName,optionData);
            strcpy(defaultOptions.backFaceTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   faces                                                                         
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"faces")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
	    {
               defaultOptions.frontFaceVisible = TRUE;
               defaultOptions.backFaceVisible  = TRUE;
            }
            else if (strcasecmp(optionData,"OFF")==0)
	    {
               defaultOptions.frontFaceVisible = FALSE;
               defaultOptions.backFaceVisible  = FALSE;
	    }
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   font                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"font")==0)
         {
            if (defaultOptions.fontFileName!=NULL)
            {
               delete defaultOptions.fontFileName;
            }

            defaultOptions.fontFileName = new char[optionDataLength+1];
            if (defaultOptions.fontFileName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.fontFileName,optionData);

         }

     //-----------------------------------------------------------------------------------------
     // SWITCH:   font-path                                                                     
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"font-path")==0)
         {
            if (defaultOptions.fontPathName!=NULL)
            {
               delete defaultOptions.fontPathName;
            }

            defaultOptions.fontPathName = new char[optionDataLength+1];
            if (defaultOptions.fontPathName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.fontPathName,optionData);

         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   format                                                                        
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"format")==0)
         {
            if (strcasecmp(optionData,"RAW")==0)
               defaultOptions.outputFormat = RAW;
            else if (strcasecmp(optionData,"POV")==0)
               defaultOptions.outputFormat = POV;
            else if (strcasecmp(optionData,"POV3")==0)
               defaultOptions.outputFormat = POV3;
            else if (strcasecmp(optionData,"RIB")==0)
               defaultOptions.outputFormat = RIB;
            else if (strcasecmp(optionData,"DXF")==0)
               defaultOptions.outputFormat = DXF;
            else if (strcasecmp(optionData,"DXF2")==0)
               defaultOptions.outputFormat = DXF2;
            else if (strcasecmp(optionData,"VIVID")==0)
               defaultOptions.outputFormat = VIVID;
            else if (strcasecmp(optionData,"RAD")==0)
               defaultOptions.outputFormat = RAD;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   front-bevel                                                                   
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"front-bevel")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.frontBevelVisible = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.frontBevelVisible = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   front-bevel-texture                                                           
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"front-bevel-texture")==0)
         {
            if (defaultOptions.frontBevelTextureName!=NULL)
               delete defaultOptions.frontBevelTextureName;

            defaultOptions.frontBevelTextureName = new char[optionDataLength+1];
            if (defaultOptions.frontBevelTextureName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }
            strcpy(defaultOptions.frontBevelTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   front-face                                                                    
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"front-face")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.frontFaceVisible = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.frontFaceVisible = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   front-face-cut                                                                
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"front-face-cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.frontFaceCut))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   front-face-texture                                                            
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"front-face-texture")==0)
         {
            if (defaultOptions.frontFaceTextureName!=NULL)
               delete defaultOptions.frontFaceTextureName;

            defaultOptions.frontFaceTextureName = new char[optionDataLength+1];
            if (defaultOptions.frontFaceTextureName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }
            strcpy(defaultOptions.frontFaceTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   front-side-cut                                                                
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"front-side-cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.frontSideCut))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   map                                                                           
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"map")==0)
         {
            if (strcasecmp(optionData,"MAC")==0)
               defaultOptions.mapType = MACINTOSH;
            else if (strcasecmp(optionData,"MS")==0)
               defaultOptions.mapType = MICROSOFT;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos  = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   name                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"name")==0)
         {
            if (defaultOptions.objectName!=NULL)
               delete defaultOptions.objectName;

            defaultOptions.objectName = new char[optionDataLength+1];
            if (defaultOptions.objectName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos = i;
               return FALSE;
            }

            strcpy(defaultOptions.objectName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   output                                                                        
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"output")==0)
         {
            if (defaultOptions.outputFileName!=NULL)
               delete defaultOptions.outputFileName;

            defaultOptions.outputFileName = new char[optionDataLength+1];
            if (defaultOptions.outputFileName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.outputFileName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   output-path                                                                   
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"output-path")==0)
         {
            if (defaultOptions.outputPathName!=NULL)
               delete defaultOptions.outputPathName;

            defaultOptions.outputPathName = new char[optionDataLength+1];
            if (defaultOptions.outputPathName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.outputPathName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   precision                                                                     
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"precision")==0)
         {
            if (!sscanf(optionData,"%i",&defaultOptions.outputPrecision))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   resolution                                                                    
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"resolution")==0)
         {
            if (!sscanf(optionData,"%i",&defaultOptions.resolution))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   side-cut                                                                      
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"side-cut")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.frontSideCut))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
            defaultOptions.backSideCut = defaultOptions.frontSideCut;
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   side-texture                                                                  
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"side-texture")==0)
         {
            if (defaultOptions.sideTextureName!=NULL)
               delete defaultOptions.sideTextureName;

            defaultOptions.sideTextureName = new char[optionDataLength+1];
            if (defaultOptions.sideTextureName==NULL)
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }
            strcpy(defaultOptions.sideTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   sides                                                                         
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"sides")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.sideVisible = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.sideVisible = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   smoothing-threshold                                                           
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"smoothing-threshold")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.threshold))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
            defaultOptions.threshold = defaultOptions.threshold*M_PI/180;
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   string                                                                        
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"string")==0)
         {
            if (defaultOptions.string!=NULL)
            {
               delete defaultOptions.string;
               defaultOptions.stringLength = 0;
            }

            defaultOptions.string = new int[optionDataLength+1];
            tempString = new char[optionDataLength+1];
            if ((defaultOptions.string==NULL) || (tempString==NULL))
	         {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }
            strcpy(tempString,optionData);

            for (j=0;j<(int)strlen(tempString);j++)
               defaultOptions.string[j]=tempString[j];

            defaultOptions.stringLength = strlen(tempString);
            delete tempString;
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   texture                                                                       
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"texture")==0)
         {
            if (defaultOptions.frontFaceTextureName!=NULL)
               delete defaultOptions.frontFaceTextureName;
            if (defaultOptions.backFaceTextureName!=NULL)
               delete defaultOptions.backFaceTextureName;
            if (defaultOptions.frontBevelTextureName!=NULL)
               delete defaultOptions.frontBevelTextureName;
            if (defaultOptions.backBevelTextureName!=NULL)
               delete defaultOptions.backBevelTextureName;
            if (defaultOptions.sideTextureName!=NULL)
               delete defaultOptions.sideTextureName;

            defaultOptions.frontFaceTextureName = new char[optionDataLength+1];
            defaultOptions.backFaceTextureName = new char[optionDataLength+1];
            defaultOptions.frontBevelTextureName = new char[optionDataLength+1];
            defaultOptions.backBevelTextureName = new char[optionDataLength+1];
            defaultOptions.sideTextureName = new char[optionDataLength+1];

            if (   defaultOptions.frontFaceTextureName==NULL
                || defaultOptions.backFaceTextureName==NULL
                || defaultOptions.frontBevelTextureName==NULL
                || defaultOptions.backBevelTextureName==NULL
                || defaultOptions.sideTextureName==NULL )
            {
               errorCode = F3D_ERR_OutOfMemory;
               errorPos  = i;
               return FALSE;
            }

            strcpy(defaultOptions.frontFaceTextureName,optionData);
            strcpy(defaultOptions.backFaceTextureName,optionData);
            strcpy(defaultOptions.frontBevelTextureName,optionData);
            strcpy(defaultOptions.backBevelTextureName,optionData);
            strcpy(defaultOptions.sideTextureName,optionData);
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   triangle-type                                                                 
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"triangle-type")==0)
         {
            if (strcasecmp(optionData,"SMOOTH")==0)
               defaultOptions.triangleType = SMOOTH;
            else if (strcasecmp(optionData,"FLAT")==0)
               defaultOptions.triangleType = FLAT;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos  = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   verbose                                                                       
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"verbose")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
               defaultOptions.verbose = TRUE;
            else if (strcasecmp(optionData,"OFF")==0)
               defaultOptions.verbose = FALSE;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos  = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   xpos                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"xpos")==0)
         {
            if (strcasecmp(optionData,"LEFT")==0)
               defaultOptions.xPosition = LEFT;
            else if (strcasecmp(optionData,"CENTER")==0)
               defaultOptions.xPosition = CENTER;
            else if (strcasecmp(optionData,"RIGHT")==0)
               defaultOptions.xPosition = RIGHT;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   ypos                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"ypos")==0)
         {
            if (strcasecmp(optionData,"BOTTOM")==0)
               defaultOptions.yPosition = BOTTOM;
            else if (strcasecmp(optionData,"BASELINE")==0)
               defaultOptions.yPosition = BASELINE;
            else if (strcasecmp(optionData,"CENTER")==0)
               defaultOptions.yPosition = CENTER;
            else if (strcasecmp(optionData,"TOP")==0)
               defaultOptions.yPosition = TOP;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   zpos                                                                          
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"zpos")==0)
         {
            if (strcasecmp(optionData,"BACK")==0)
               defaultOptions.zPosition = BACK;
            else if (strcasecmp(optionData,"CENTER")==0)
               defaultOptions.zPosition = CENTER;
            else if (strcasecmp(optionData,"FRONT")==0)
               defaultOptions.zPosition = FRONT;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }





     //-----------------------------------------------------------------------------------------
     //  These last few options are going to remain either unimplemented, or undocumented for   
     //  the time being...                                                                      
     //-----------------------------------------------------------------------------------------

     //-----------------------------------------------------------------------------------------
     // SWITCH:   codelist                                                                      
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"codelist")==0)
         {
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   reverse                                                                       
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"reverse")==0)
         {
            if (strcasecmp(optionData,"ON")==0)
	    {
               defaultOptions.reverse = TRUE;
            }
            else if (strcasecmp(optionData,"OFF")==0)
	    {
               defaultOptions.reverse = FALSE;
	    }
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   reverse-border-vertical                                                       
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"reverse-border-vertical")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.reverseBorderVertical))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   reverse-border-horizontal                                                     
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"reverse-border-horizontal")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.reverseBorderHorizontal))
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   spacing                                                                       
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"spacing")==0)
         {
            if (!sscanf(optionData,"%lf",&defaultOptions.spacing))
	    {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
            }
         }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   spaceto                                                                       
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"spaceto")==0)
	 {
            if (!sscanf(optionData,"%lf",&defaultOptions.spaceto))
	    {
               errorCode = F3D_ERR_InvalidOption;
               errorPos = i;
               return FALSE;
	    }
	 }


     //-----------------------------------------------------------------------------------------
     // SWITCH:   language                                                                      
     //-----------------------------------------------------------------------------------------

         else if (strcasecmp(option,"language")==0)
         {
            if (strcasecmp(optionData,"USENGLISH")==0)
               defaultOptions.language = USENGLISH;
            else if (strcasecmp(optionData,"UKENGLISH")==0)
               defaultOptions.language = UKENGLISH;
            else
            {
               errorCode = F3D_ERR_InvalidOption;
               errorPos  = i;
               return FALSE;
            }
         }





     //-----------------------------------------------------------------------------------------
     // SWITCH:  UNKNOWN!                                                                       
     //-----------------------------------------------------------------------------------------

         else
         {
            errorCode = F3D_ERR_InvalidOption;
            errorPos=i;
            return FALSE;
         }

         delete option;


      } // END Looping thru arguments 

      return TRUE;

   }





//==============================================================================================
//  main()
//==============================================================================================

int main(int argc, char* argv[])
{
   int           errorCode;
   int           errorPos;
   int           success;
   Font3DOptions options;
   USHORT        pid,sid,lid;

   int           numFileArgs;
   char**        fileArgs;

   char          fileName[MAX_FILENAME_SIZE+1];
   char*         defaultConfigFile=NULL;
   char*         charstring;
   int           i;

   defaultConfigFile=getenv("FONT3D_DEFAULT_CONFIG");

   if (defaultConfigFile==NULL)
   {
      defaultConfigFile = "font3d.def";
   }

   success = ReadOptions(defaultConfigFile, numFileArgs, fileArgs);

   if (success==F3D_ERR_OutOfMemory)
   {
      cout<<endl<<"ERROR: Out of memory."<<endl;
      exit(1);
   }
   else if (success==F3D_ERR_UnmatchedQuotes)
   {
      cout<<endl<<"ERROR: Unmatched quotes in default configuration file \""
          <<defaultConfigFile<<"\""<<endl;
   }
   else if (success!=F3D_ERR_UnableToOpenFile)
   {
      success = ParseOptions(numFileArgs,fileArgs,options,errorCode,errorPos);

      if (!success)
      {
         cout<<endl;
         cout<<"ERROR: Invalid option #"<<errorPos+1<<": \""<<fileArgs[errorPos]<<"\""<<endl;
         cout<<"       in default configuration file \""
             <<defaultConfigFile<<"\""<<endl<<endl;
         exit(1);
      }

      for (i=0;i<numFileArgs;i++)
         delete fileArgs[i];
      delete fileArgs;
   }

   if (argc>1)
   {
      success = ParseOptions(argc-1,&argv[1],options,errorCode,errorPos);

      if (!success)
      {
         cout<<endl;
         cout<<"ERROR: Invalid command line option \""
             <<argv[errorPos+1]<<"\""<<endl<<endl;;
         exit(1);
      }
   }

   if (options.fontFileName==NULL)
   {
       cout<<endl;
       cout<<"ERROR: No Font File Specified."<<endl<<endl;;
       exit (1);
   }

   if (options.mapType==MICROSOFT)
   {
      pid = PID_Microsoft;
      sid = SID_MS_UGL;
      lid = LID_MS_USEnglish;
   }
   else
   {
      pid = PID_Macintosh;
      sid = SID_MAC_Roman;
      lid = LID_MAC_English;
   }

   TTFont UserFont(options.fontFileName,options.fontPathName,pid,sid,lid);

   switch(UserFont.LastError())
   {
      case F3D_ERR_NoError:
         break;

      case F3D_ERR_OutOfMemory:
         cout<<endl; 
         cout<<"ERROR: Out of memory."<<endl<<endl;
         exit(1);
         break;

      case F3D_ERR_UnableToOpenFile:
         cout<<endl;
         cout<<"ERROR: Unable to open font file \""<<options.fontFileName
             <<"\"."<<endl<<endl;
         exit(1);
         break;

      case F3D_ERR_TableNotFound:
         cout<<endl;
         cout<<"ERROR: A required part of this TrueType font is missing."<<endl<<endl;;
         exit(1);
         break;

      case F3D_ERR_UnknownCmapFormat:
         cout<<endl;
         cout<<"ERROR: An invalid or unsupported character encoding method"<<endl;
         cout<<"       was specified."<<endl<<endl;
         exit(1);
         break;

      case F3D_ERR_UnknownKernFormat:
         break;

      default:
         cout<<endl;
         cout<<"ERROR: Unable to read font file."<<endl;
         exit(1);
   }


   if (options.outputPathName==NULL)
      strcpy(fileName, options.outputFileName);
   else
      sprintf(fileName,"%s/%s",options.outputPathName,options.outputFileName);
   ofstream outputfile(fileName);
   if (!outputfile)
   {
      cout<<endl;
      cout<<"ERROR: Unable to open \""<<options.outputFileName<<"\" for output."<<endl<<endl;
      exit(1);
   }

   if (options.verbose)
   {
      PrintGreeting();
      if (options.outputFormat==RIB)
         PrintPixarCopyright();
      cout<<endl;

      cout<<"Font:     "<<UserFont.FullName()<<endl;
      cout<<"Object:   ";

      charstring = new char[options.stringLength+1];
      for (i=0;i<options.stringLength;i++)
         charstring[i]=(char)options.string[i];
      charstring[i]=0;
      cout<<"\""<<charstring<<"\""<<endl;

      cout<<"Encoding: ";
      if (options.mapType==MICROSOFT)
         cout<<"Microsoft"<<endl;
      else
         cout<<"Macintosh"<<endl;

      cout<<endl;

      cout<<"Output Filename: "<<options.outputFileName<<endl;
      cout<<"Object Format:   ";
      switch (options.outputFormat)
      {
         case RAW: cout<<"RAW"<<endl; break;
         case POV: cout<<"POV-Ray 2.x"<<endl; break;
         case POV3: cout<<"POV-Ray 3.0"<<endl; break;
         case RIB: cout<<"RenderMan RIB"<<endl; break;
         case DXF: cout<<"AutoCad DXF"<<endl; break;
         case DXF2: cout<<"AutoCad DXF (single entity)"<<endl; break;
         case RAD: cout<<"Radiance 2.x"<<endl; break;
         case VIVID: cout<<"Vivid 2.0"<<endl; break;
      }
      cout<<"Object Name:     "<<options.objectName<<endl;
      cout<<endl;

   }

   success = OutputObjects(outputfile,UserFont,options);

   switch (success)
   {
      case F3D_ERR_NoError:
         if (options.verbose) cout<<endl<<"Output File Successfully Created!"<<endl<<endl;
         break;
      case F3D_ERR_OutOfMemory:
         cout<<"ERROR:  Out Of Memory."<<endl;
         cout<<"        Output file is probably incomplete."<<endl<<endl;
         break;
      case F3D_ERR_NoPolyFound:
         cout<<"ERROR:  Unable to Triangulate."<<endl;
         cout<<"        Output file is probably incomplete."<<endl<<endl;
         break;
      default:
         cout<<"ERROR:  Fatal Error."<<endl;
         cout<<"        Output file is probably incomplete."<<endl<<endl;
   }

   return 0;

}

