##########################################################################
#                                                                        #
#           copyright (c) 2003, 2005 ITB, Humboldt-University Berlin     #
#           written by: Raphael Ritz, r.ritz@biologie.hu-berlin.de       #
#                                                                        #
##########################################################################

"""Article reference main class"""

from AccessControl import ClassSecurityInfo
from Products.CMFCore.permissions import View

from Products.Archetypes.public import Schema
from Products.Archetypes.public import StringField, StringWidget
from Products.Archetypes.public import registerType

from Products.CMFBibliographyAT.marshall import BibtexMarshaller
from Products.CMFBibliographyAT.content.base import BaseEntry
from Products.CMFBibliographyAT.content.schemata \
    import HeaderSchema, AuthorSchema, CoreSchema, TrailingSchema
from Products.CMFBibliographyAT.content.fields \
    import journalField, volumeField, numberField, pagesField


SourceSchema = Schema((
    journalField,
    volumeField,
    numberField,
    pagesField,
    StringField('pmid',
        widget=StringWidget(label="PMID",
            label_msgid="label_source_pmid",
            description="The reference's number in the PubMed database.",
            description_msgid="help_source_pmid",
            i18n_domain="plone",),
            ),
    ))

ArticleSchema = HeaderSchema + AuthorSchema + CoreSchema +  \
                SourceSchema + TrailingSchema


class ArticleReference(BaseEntry):
    """ content type to make reference to a (scientific) article.
    """
    security = ClassSecurityInfo()
    archetype_name = "Article Reference"
    source_fields = ('journal', 'volume', 'number', 'pages', 'pmid')

    schema = ArticleSchema
    
    security.declareProtected(View, 'PMID')
    def PMID(self):
        """ returns the pmid if set
        """
        value = self.getPmid()
        if value:
            return value
        else:
            return None
        
    security.declareProtected(View, 'getPubMedLink')
    def getPubMedLink(self, defaultURL=None):
        """ a link to PubMed
            if pmid is set or the default otherwise
        """
        if self.getPmid():
            url = "http://www.ncbi.nlm.nih.gov/entrez/query.fcgi?cmd=Retrieve"
            return url + "&db=PubMed&list_uids=%s" % self.getPmid()
        else: return defaultURL

    security.declareProtected(View, 'Source')
    def Source(self):
        """ the default article source format
        """
        try:
            return self.ArticleSource()
        except AttributeError:
            journal = self.getJournal()
            volume  = self.getVolume()
            number  = self.getNumber()
            pages   = self.getPages()

            source = journal
            if volume: source += ' ' + volume
            if number: source += '(' + number + ')'
            if pages: source += ':' + pages
            return source + '.'


registerType(ArticleReference)
