#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../include/Y2/Y.h"
#include "../include/Y2/Ylib.h"

#include "../include/string.h"


static void print_help(void);


/*
 *	Setting parameter codes:
 */
#define PARM_NONE		0	/* Do nothing. */
#define PARM_CYCLE		1	/* Change cycle. */
#define PARM_AUDIO_PRESET	2	/* Change Audio mode preset. */
#define PARM_AUDIO_VALUES	3	/* Change Audio mode values. */
#define PARM_AUDIO_SHELLOUT	4	/* Shell out recorder. */


/*
 *      Prints help message.
 */
static void print_help(void)
{
	printf("\
Usage: yset <parameter [value...]> [options]\n\
\n\
    <parameter> can be any of the following:\n\
\n\
        audio <name>                 Change to a preset audio mode.\n\
\n\
        audiovalues <sample_rate> <channels> <sample_size> <fragment_size>\n\
                                     Change Audio mode values:\n\
                                      <sample_rate>   Sample rate in Hz.\n\
                                      <channels>      Number of channels.\n\
                                      <sample_size>   Sample size in bits.\n\
                                      <fragment_size> Fragment size in bytes.\n\
\n\
        cycle <microseconds>         Adjust the cycle.\n\
\n\
        shellout                     Shells out the recorder.\n\
\n\
    [options] can be any of the following:\n\
\n\
        --recorder <address:port>    Specify which Y server to connect to.\n\
\n\
    Return values:\n\
\n\
        0       Success.\n\
        1       General error.\n\
        2       Cannot connect to Y server error.\n\
        3       Systems error.\n\
\n"
	);

	return;
}


int main(int argc, char *argv[])
{
	int i, status;

	const char *con_arg = NULL;

	int parm = PARM_NONE;
	char val[1024];

	YConnection *con = NULL;

	int	sample_rate		= 11025,
		channels		= 1,
		sample_size		= 8,
		fragment_size_bytes	= 512;
	long	cycle_us		= 30000;


	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               !strcmp(argv[i], "?")
	    )
	    {
		print_help();
		return(0);
	    }
	    /* Cycle. */
	    else if(strcasepfx(argv[i], "--cycle") ||
                    strcasepfx(argv[i], "-cycle") ||
                    strcasepfx(argv[i], "cycle")
	    )
	    {
                i++;
                if(i < argc)
                {
                    parm = PARM_CYCLE;
		    strncpy(val, argv[i], 1024);
		    val[1024 - 1] = '\0';
                }   
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
            /* Audio mode values. */
            else if(strcasepfx(argv[i], "--audiovalue") ||
                    strcasepfx(argv[i], "-audiovalue") ||
                    strcasepfx(argv[i], "audiovalue")
            )
            {
                parm = PARM_AUDIO_VALUES;

                i++;
                if(i < argc)
                    sample_rate = atoi(argv[i]);
                else
                    fprintf(stderr,
                        "Insufficient arguments.\n"
                    );

		i++;
                if(i < argc)
                    channels = atoi(argv[i]);
                else
                    fprintf(stderr,
                        "Insufficient arguments.\n" 
                    );

                i++;
                if(i < argc)
                    sample_size = atoi(argv[i]);
                else
                    fprintf(stderr,
                        "Insufficient arguments.\n" 
                    );

                i++;
                if(i < argc)
                    fragment_size_bytes = atoi(argv[i]);
                else
                    fprintf(stderr,
                        "Insufficient arguments.\n" 
                    );
            }
            /* Audio mode preset. */
            else if(strcasepfx(argv[i], "--audio") ||
                    strcasepfx(argv[i], "-audio") ||
                    strcasepfx(argv[i], "audio")

            )
            {
                i++;
                if(i < argc)
                {
                    parm = PARM_AUDIO_PRESET;
                    strncpy(val, argv[i], 1024);
                    val[1024 - 1] = '\0';
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
            /* Shell out. */
            else if(strcasepfx(argv[i], "--shellout") ||
                    strcasepfx(argv[i], "-shellout") ||
                    strcasepfx(argv[i], "shellout")
            )
            {
		parm = PARM_AUDIO_SHELLOUT;
            }
	    /* Connect address. */
	    else if(strcasepfx(argv[i], "--rec") ||
                    strcasepfx(argv[i], "-rec")
	    )
	    {
		i++;
		if(i < argc)
		{
		    con_arg = argv[i];
		}
		else
		{
		    fprintf(stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	}

	/* If no operation specified, then just print help. */
	if(parm == PARM_NONE)
	{
            print_help();   
            return(1);
	}


        /* Connect to Y server. */
        con = YOpenConnection(
            NULL,               /* No start argument. */
            con_arg
        );
        if(con == NULL)
        {
            fprintf(stderr, "Cannot connect to Y server");
            if(con_arg == NULL)
                con_arg = getenv("RECORDER");
            if(con_arg == NULL)
                fprintf(stderr, ".\n");
            else
                fprintf(stderr, ": %s\n", con_arg);
            return(2);
        }

	/* Begin setting new value by the given parameter code. */
	status = -1;
	switch(parm)
	{
	  /* Set cycle. */
	  case PARM_CYCLE:
            status = YSetCycle(
		con,
		atol(val)
	    );
            if(status)
            {
                fprintf(stderr,
                    "Could not set cycle to value `%s'.\n",
                    val
                );
            }
            break;

	  /* Set audio mode preset. */
	  case PARM_AUDIO_PRESET:
            status = YChangeAudioModePreset(con, val);
            if(status)
            {
                fprintf(stderr,
                    "Could not set audio mode to value `%s'.\n",
                    val
                );
            }
	    break;

	  /* Set audio mode values. */
	  case PARM_AUDIO_VALUES:
	    /* Sanitize values. */
	    if((channels != 1) &&
               (channels != 2)
	    )
	    {
		fprintf(stderr,
                    "Invalid number of channels %i.\n",
                    channels
                );
		break;
	    }
            if((sample_size != 8) &&
               (sample_size != 16)
            )
            {
                fprintf(stderr,
                    "Invalid sample size %i bits.\n",
                    sample_size
                );
                break;
            }
            if((fragment_size_bytes != 256) &&
               (fragment_size_bytes != 512) &&
               (fragment_size_bytes != 1024) &&
               (fragment_size_bytes != 2048) &&
               (fragment_size_bytes != 4096) &&
               (fragment_size_bytes != 8192) &&
               (fragment_size_bytes != 16384)
            )
            {
                fprintf(stderr,
 "Invalid fragment size %i bytes (needs to be a 2^n value).\n",
                    fragment_size_bytes
                );
                break;
            }

	    /* Calculate cycle. */
	    cycle_us = YCalculateCycle(
                con,
                sample_rate, channels,
                sample_size, fragment_size_bytes
	    );
	    if(cycle_us < 100)
		cycle_us = 100;

	    status = YSetAudioModeValues(
                con,
                sample_size,
                channels,
                sample_rate,
                0,		/* Direction: 0 = play. */
                1,		/* Allow fragmenting. */
		2, 		/* Number of fragments. */
                fragment_size_bytes
	    );
            if(status)
            {
                fprintf(stderr,
                    "Could not set audio mode values.\n"
                );
            }
	    else
	    {
		YSetCycle(con, cycle_us);
	        YSyncAll(con, True);	/* Ensures full affect of cycle. */
	    }
	    break;

	  /* Shell out. */
	  case PARM_AUDIO_SHELLOUT:
            status = YSetAudioModeValues(
                con,  
                8,	/* Sample size. */
                1,	/* Channels. */
                0,	/* Sample rate (set to 0 to mean shell out). */
                0,      /* Direction: 0 = play. */
                1,      /* Allow fragmenting. */
                2,      /* Number of fragments. */
                512	/* Fragment size in bytes. */
	    );
	    if(status)
            {
                fprintf(stderr,
                    "Could not shell out recorder.\n"
                );
            }
	    else
	    {
                printf(
"Recorder has been shelled out.\n\
To reininitialize the recorder, use `yset' and specify any valid Audio\n\
mode or any valid Audio values.\n"
);
	    }
	    break;
	}

        /* Disconnect from Y server. */
        YCloseConnection(con, False);
        con = NULL;

	return(0);
}
