/* This is for Emacs: -*-Mode: C;-*- */
%{

#include <stdio.h>

#include <glib.h>
#include <gc/gc.h>

#include "tree.h"
#include "parse.h"
#include "util.h"
#include "process.h"

int yylex(void);
 
%}

%option noyywrap
%option yylineno

%%

"{" {
  return '{';
}

"}" {
  return '}';
}

":" {
  return ':';
}

"," {
  return ',';
}

";" {
  return ';';
}

"(" {
  return '(';
}

")" {
  return ')';
}

"<" {
  return '<';
}

">" {
  return '>';
}

"*" {
  return '*';
}

"&" {
  return '&';
}

"->" {
  return POINTSAT;
}

"=" {
  return '=';
}

"::" {
  return SCOPE;
}

"/*" { /* C comments */
  register int c;

  while (1)
  {
    while ((c = input()) != '*' && c != EOF)
      ;
    if (c == '*')
    {
      while ((c = input()) == '*')
        ;
      if (c == '/')
        break;
    }
    if (c == EOF)
      process_error("EOF in comment");
  }
}

"//"[^\n]* { /* C++ comments, ignore */
}

"%{" { /* C++ code block, read literally into buffer */
  register int c;
  char buffer[1024];
  int strsz = 0, bufsz = 64;
  char *text = GC_malloc(bufsz);
  char *cp;
  int had_percent = FALSE;
  int lineno = yylineno;
  const char *fname = process_get_infile_name();
  
  text[0] = '\0';
  cp = buffer;
  
  do {
    if ((c = input()) == EOF)
      break;
    
    if (had_percent)
    {
      if (c == '}')
        break;
      if (c != '%')
      {
        *cp++ = '%';
        had_percent = FALSE;
      }
      *cp++ = c;
    }
    else if (c == '%')
      had_percent = TRUE;
    else
      *cp++ = c;
    
    if (cp - buffer >= sizeof(buffer) - 1)
    {
      text = GC_str_append(text, buffer, cp - buffer, &strsz, &bufsz);
      cp = buffer;
    }
  } while (!had_percent || c != '}');
  text = GC_str_append(text, buffer, cp - buffer, &strsz, &bufsz);
  
  if (c == EOF)
    process_error("EOF in C code block");

  yylval.tree = tree_build_verbatim(text);
  TREE_LINENO(yylval.tree) = lineno;
  TREE_FILENAME(yylval.tree) = fname;
  
  return C_CODE;
}

namespace {
  return NAMESPACE;
}

value {
  return VALUE;
}

class {
  return CLASS;
}

using {
  return USING;
}

constructor {
  return CONSTRUCTOR;
}

method {
  return METHOD;
}

getter {
  return GETTER;
}

setter {
  return SETTER;
}

signal {
  return SIGNAL;
}

unsigned|signed|long|short|int|char|bool|float|double {
  yylval.tree = tree_build_typespec(yytext);
  return TYPESPEC;
}

const {
  return CONST;
}

wrap {
  return WRAP;
}

virtual {
  return VIRTUAL;
}

\"[^\"]*\" { /* string constant */
  /* FIXME: escaped " */
  yylval.tree = tree_build_string(yytext);
  return STRING_CONSTANT;
}

(0x)?[0-9]+ { /* integer constant */
  yylval.tree = tree_build_integer(yytext);
  return INTEGER_CONSTANT;
}

[a-zA-Z_]+[a-zA-Z_0-9]* { /* identifier */
  yylval.tree = tree_build_identifier(yytext);
  return IDENTIFIER;
}  

[ \t\n]+ { /* ignore */
}

#plugin {
  return SHARP_PLUGIN;
}

