/*
 * gdd_notes.c
 */
/*
 * yank  -  yet another NoteKeeper
 * Copyright (C) 1999, 2000, 2001 Michael Humann <m.hussmann@home.ins.de>
 *
 * This program is free software; you  can redistribute it and/or modify it
 * under the terms of  the GNU General  Public License as published  by the
 * Free Software Foundation;  either version 2 of  the License, or (at your
 * option) any later version.
 *
 * This program  is  distributed in the  hope  that it will  be useful, but
 * WITHOUT  ANY   WARRANTY;  without   even   the  implied    warranty   of
 * MERCHANTABILITY or    FITNESS FOR A PARTICULAR   PURPOSE.   See  the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write to the  Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gdd_notes.h"
#include "app.h"
#include "callbacks.h"

#include <dirent.h>
#include <stdlib.h>

static GList     *gdd_list = NULL;  /* list of gdd_note(s) */
static gdd_note  *last_changed = NULL;

static void      gdd_notes_search_icons(GList *list);
static gint      gdd_notes_read_dir(const gchar *path, GList **list);
static gdd_note* gdd_notes_find_by_menuname(const gchar *name, GList *list);
static void      gdd_notes_add_cb(GtkWidget *w, gpointer p);
static void      gdd_notes_note_changed_cb(GtkWidget *gdd, gpointer gdd_entry);

/* ------------------------------------------------------ */
/*
 * search for icons
 */

static void
gdd_notes_search_icons(GList *list)
{
    GdkImlibImage *iml;
    gdd_note      *gdd_entry;
    gchar         *fname;
    
    if (list == NULL)
    {
        return;
        /* notreached */
    }

    for (list = g_list_first(list);
         list != NULL;
         list = g_list_next(list))
    {
        gdd_entry = list->data;

        /* FIXME: support for .jpg ... */
        fname = g_strconcat(g_dirname(gdd_entry->gladepath), PATH_SEP_STR,
                            gdd_entry->menuname, ".xpm", NULL);
        if ((iml = gdk_imlib_load_image(fname)))
        {
            gdd_entry->iconpath = fname;
            gdd_entry->iconwidget = gnome_pixmap_new_from_imlib_at_size(iml, 15,
                                                                        15);
            /* FIXME: destroy iml? */
        }
        else
        {
            g_free(fname);
            gdd_entry->iconpath = NULL;
            gdd_entry->iconwidget = NULL;
        }
    }    
}

/* ------------------------------------------------------ */
/*
 * returns pointer to gdd_note with menuname from list or NULL
 */

static gdd_note*
gdd_notes_find_by_menuname(const gchar *name,
                           GList *list)
{
    gdd_note *gdd_entry;
    gdd_note *ret;

    if (list == NULL)
    {
        return (NULL);
        /* notreached */
    }

    if (name == NULL)
    {
        return (NULL);
        /* notreached */
    }
    
    ret = NULL;
    for (list = g_list_first(list);
         list != NULL && ret == NULL;
         list = g_list_next(list))
    {
        gdd_entry = list->data;
        if (strcmp(gdd_entry->menuname, name) == 0)
        {
            ret = gdd_entry;
        }
    }
    
    return (ret);
}

/* ------------------------------------------------------ */
/*
 * scan dir for glade-files, create a gdd wiget for each of them and
 * add it to the list
 * return: 0 = OK
 *         1 = error
 */

static gint
gdd_notes_read_dir(const gchar *path,
                   GList **list)
{
    DIR           *dir;
    struct dirent *entry;
    gdd_note      *gdd_entry;
    gchar         buf;
    gint          name_len;
    
    g_return_val_if_fail(path != NULL, 1);
    g_return_val_if_fail(list != NULL, 1);
    
    dir = opendir(path);
    if (dir == FALSE)
    {
        return (1);
        /* notreached */
    }

    while ((entry = readdir(dir)))
    {
        name_len = strlen(entry->d_name) - 6; /* length of name w/out ext. */
        
        if (strcmp(&(entry->d_name[name_len]), ".glade") != 0)
        {
            continue;
            /* notreached */
        }

        buf = entry->d_name[name_len];
        entry->d_name[name_len] = '\0';

        /* don't add notes with the same name twice */
        if (gdd_notes_find_by_menuname(entry->d_name, (*list)) != NULL)
        {
            entry->d_name[name_len] = buf;
            continue;
            /* notreached */            
        }
        
        gdd_entry = g_new(gdd_note, 1);
        if (gdd_entry == NULL)
        {
            g_warning(__FUNCTION__ ": g_new() failed!");
            closedir(dir);
            return (1);
            /* notreached */
        }

        gdd_entry->menuname = g_strdup(entry->d_name);
        entry->d_name[name_len] = buf;
        gdd_entry->gladepath = g_strconcat(path, PATH_SEP_STR ,entry->d_name,
                                           NULL);
        gdd_entry->gdd = gdd_new(gdd_entry->gladepath, GDD_NOTES_ROOT_WIDGET,
                                 NULL);
        gtk_signal_connect(GTK_OBJECT(gdd_entry->gdd), "changed",
                           GTK_SIGNAL_FUNC(gdd_notes_note_changed_cb),
                           gdd_entry);
      
        gdd_entry->widget = glade_xml_get_widget(GLADE_XML(gdd_entry->gdd),
                                                 GDD_NOTES_ROOT_WIDGET);
        gdd_entry->iconpath = NULL;
        gdd_entry->iconwidget = NULL;
        
        yank_register_note_widget(gdd_entry->widget);
        (*list) = g_list_append((*list), gdd_entry);
    }

    closedir(dir);
    
    return (0);
}

/* ------------------------------------------------------ */
/*
 * build list of gdd widgets with path and menuname
 * return: 0 = OK
 *         1 = error (empty list)
 */

gint
gdd_notes_build_list(void)
{
    gchar *dir;
    gchar *home;
    
    gdd_notes_read_dir("." PATH_SEP_STR "GddNotes", &gdd_list);

    home = g_get_home_dir();
    if (home)
    {
        dir = g_strdup_printf("%s" PATH_SEP_STR ".%s" PATH_SEP_STR "GddNotes"
                              PATH_SEP_STR "%s", home, PACKAGE, VERSION);
        gdd_notes_read_dir(dir, &gdd_list);
        g_free(dir);
        g_free(home);
    }
    
    dir = NULL;
#ifdef YANK_LIBDIR
    dir = g_strdup(YANK_LIBDIR PATH_SEP_STR PACKAGE PATH_SEP_STR "GddNotes"
                   PATH_SEP_STR VERSION PATH_SEP_STR);
#else  /* YANK_LIBDIR */
    dir = gnome_unconditional_libdir_file(PACKAGE PATH_SEP_STR "GddNotes"
                                          PATH_SEP_STR VERSION PATH_SEP_STR);
#endif /* YANK_LIBDIR */
    gdd_notes_read_dir(dir, &gdd_list);
    g_free(dir);
    
    gdd_notes_search_icons(gdd_list);
    
    return (!(g_list_length(gdd_list) > 0));
}

/* ------------------------------------------------------ */
/*
 * create menu from list
 * return: 0 = OK
 *         1 = error
 */

gint
gdd_notes_build_menu(const gchar *mpath)
{
    GnomeUIInfo *menu_entry;
    gdd_note    *gdd_entry;
    GList       *list;
    gint        lsize;
    gboolean    has_abs_path;
    gchar       *cwd;
    
    g_return_val_if_fail(mpath != NULL, 1);
    g_return_val_if_fail(gdd_list != NULL, 1);
    
    lsize = g_list_length(gdd_list);
    if (lsize < 1)
    {
        return (0);
        /* notreached */
    }

    cwd = NULL;
    
    for (list = g_list_first(gdd_list);
         list != NULL;
         list = g_list_next(list))
    {
        gdd_entry = list->data;

        menu_entry = g_new0(GnomeUIInfo, 2);
        if (menu_entry == NULL)
        {
            g_warning(_("gdd_notes_build_menu: out of memory"));
            return (1);
            /* notreached */
        }
        
        menu_entry[0].type = GNOME_APP_UI_ITEM;
        menu_entry[0].label = g_strdup(gdd_entry->menuname);
        menu_entry[0].hint = g_strdup(gdd_entry->gladepath); 
        menu_entry[0].moreinfo = gdd_notes_add_cb;
        menu_entry[0].user_data = gdd_entry;
        menu_entry[0].unused_data = NULL;
        if (gdd_entry->iconpath != NULL)
        {
            menu_entry[0].pixmap_type = GNOME_APP_PIXMAP_FILENAME;
            menu_entry[0].pixmap_info = gdd_entry->iconpath;
        }
        else
        {
            menu_entry[0].pixmap_type = GNOME_APP_PIXMAP_NONE;
            menu_entry[0].pixmap_info = 0;
        }
        menu_entry[0].accelerator_key = (gchar) NULL;
        menu_entry[0].ac_mods = GDK_CONTROL_MASK;
        
        menu_entry[1].type = GNOME_APP_UI_ENDOFINFO;

        /*
         * fool gnome_dirrelative_file(); to find icons for the menu
         * in ~/.yank/GddNotes etc.
         *
         * I do not like it so it's a FIXME
         */
        
        has_abs_path = FALSE;

        if (gdd_entry->iconpath != NULL)
        {
            if (gdd_entry->iconpath[0] == PATH_SEP)
            {
                cwd = g_new0(gchar, (PATH_MAX + 1));
                if (cwd == NULL)
                {
                    g_warning(_("gdd_notes_build_menu: can't alloc path"));
                }
                else
                {
                    has_abs_path = TRUE;
                    getcwd(cwd, PATH_MAX);
                    chdir(PATH_SEP_STR);
                }
            }
        }

        gnome_app_insert_menus(GNOME_APP(yank_main_app(NULL)), mpath,
                               menu_entry);

        if (has_abs_path == TRUE)
        {
            chdir(cwd);
            g_free(cwd);
        }
    }
    
    return (0);
}

/* ------------------------------------------------------ */
/*
 * called from the main menu
 */

static void
gdd_notes_add_cb(GtkWidget *w,
                 gpointer p)
{
    gdd_note *gdd_entry;
    Gdd      *gdd;
    
    g_return_if_fail(p != NULL);

    gdd_entry = p;
    if (!IS_GDD(gdd_entry->gdd))
    {
        g_warning(_("gdd_notes_add_cb: data failure"));
        return;
        /* notereached */
    }

    gdd = gdd_entry->gdd;
    gdd_cmonitor_on(gdd, FALSE);
    prepare_add(Testing, gdd_entry->widget);
}

/* ------------------------------------------------------ */
/*
 * the note has been edited
 */

static void
gdd_notes_note_changed_cb(GtkWidget *gdd,
                          gpointer gdd_entry)
{
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));
    g_return_if_fail(gdd_entry != NULL);

    last_changed = gdd_entry;
    note_changed();
}

/* ------------------------------------------------------ */
/*
 * read from gui
 */

void
gdd_notes_fill_note_data_from_page(note_data *note)
{
    xmlNodePtr list;
    xmlNodePtr gdd_val;

    gdd_val = gdd_get_values(last_changed->gdd, NULL, last_changed->menuname);
    
    (xmlNodePtr) note->text = gdd_val;
    (gdd_note *) note->user = last_changed;
    
    note->title = NULL;
    
    for (list = gdd_val->childs;
         list != NULL && note->title == NULL;
         list = list->next)
    {
        /* FIXME: choose val from 1.st child if title is not found */
        if (strcmp(list->name, "title") == 0)
        {
            note->title = xmlNodeListGetString(list->doc, list->childs, 1);
        }
    }
    note->id = NULL;
}

/* ------------------------------------------------------ */
/*
 * write to gui
 */

void 
gdd_notes_fill_page_from_note_data(note_data *note)
{
    gdd_set_values(GDD(((gdd_note*) note->user)->gdd), (xmlNodePtr) note->text);
}

/* ------------------------------------------------------ */
/*
 * designed to be changed in later versions
 */

GtkWidget*
gdd_notes_choose_icon(note_data *note)
{
    GtkWidget *ret;

    ret = ((gdd_note *) note->user)->iconwidget;
    
    return (ret);
}

/* ------------------------------------------------------ */
/*
 * clear gui values
 */

void
gdd_notes_clear_gui(GtkWidget *widget)
{
    gdd_note *gdd_entry;
    GList    *list;
    gint     found;
    
    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_WIDGET(widget));

    found = 0;
    for (list = g_list_first(gdd_list);
         list != NULL && found == 0;
         list = g_list_next(list))
    {
        gdd_entry = list->data;
        if (gdd_entry->widget == widget)
        {
            gdd_clear_values(gdd_entry->gdd);
            found = 1;
        }
    }
}

/* ------------------------------------------------------ */

void
gdd_notes_fill_widget(note_data *note)
{
    gdd_note *gdd_entry;

    g_return_if_fail(note->notetype == Testing);
    
    gdd_entry = gdd_notes_find_by_menuname(((xmlNodePtr)note->text)->name,
                                            gdd_list);
    if (gdd_entry == NULL)
    {
        g_warning(_("gdd_notes_fill_widget: can't find list entry"));
    }
    ((gdd_note *) note->user) = gdd_entry;
}

/* ------------------------------------------------------ */
