// skycal.cc -- Functions for sunrise, sunset, phase of moon.
// Last modified 1998-05-11

// This source file began its life as skycalendar.c and skycalc.c in
// John Thorstensen's skycal distribution (version 4.1, 1994-09) at
// ftp://iraf.noao.edu/contrib/skycal.tar.Z.  Those portions that are
// unchanged from the original sources are covered by the original
// license statement, included below.  The new portions and "value
// added" by David Flater are covered under the GNU General Public
// License:

/*
    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// The original skycal comments and license statement are as follows:

/*
   This is a self-contained c-language program to print a nighttime
   astronomical calendar for use in planning observations.
   It prints to standard output (usually the terminal); the
   operator should capture this output (e. g., using redirection
   in UNIX or the /out= switch in VMS) and then print it on an
   appropriate output device.  The table which is printed is some
   125 columns wide, so a wide device is required (either a line
   printer or a laserprinter in LANDSCAPE mode.)  It is assumed that
   the ASCII form-feed character will actually begin a new page.
   The original program was to run on VMS, but it should be very
   transportable.  Non-vms users will probably want to change
   'unixio.h' to 'stdio.h' in the first line.
   An explanatory text is printed at the beginning of the output, which
   includes the appropriate CAUTIONS regarding accuracy and applicability.

   A number of 'canned site' parameters have been included.  Be
   careful of time zones, DST etc. for foreign sites.
   To customize to your own site, install an option in the
   routine 'load_site'.  The code is very straightforward; just do
   it exactly the same as the others.  You might also want to erase
   some seldom-used choices.  One can also specify new site parameters
   at run time.

   This program may be used freely by anyone for scientific or educational
   purposes.  If you use it for profit, I want a cut, and claim
   a copyright herewith.  In any case please acknowledge the source:

			John Thorstensen
			Dept. of Physics and Astronomy
			Dartmouth College
			Hanover, NH 03755
			John.Thorstensen@dartmouth.edu

			May 26, 1993.
*/

#include "common.hh"

static double atan_circ(double x, double y)
{
  /* returns radian angle 0 to 2pi for coords x, y */
  double theta;
  if(x == 0) {
    if(y > 0.) theta = M_PI/2.;
    else if(y < 0.) theta = 3.*M_PI/2.;
    else theta = 0.;   /* x and y zero */
  }
  else theta = atan(y/x);
  if(x < 0.) theta = theta + M_PI;
  if(theta < 0.) theta = theta + 2.*M_PI;
  return(theta);
}

static double altit (double dec, double ha, double lat)
/* dec deg, dec hrs, dec deg */
{
  double x;
  dec = dec / DEG_IN_RADIAN;
  ha = ha / HRS_IN_RADIAN;
  lat = lat / DEG_IN_RADIAN;  /* thank heavens for pass-by-value */
  x = DEG_IN_RADIAN * asin(cos(dec)*cos(ha)*cos(lat) + sin(dec)*sin(lat));
  return(x);
}

static double lst(double jd, double longit)

{
	/* returns the local MEAN sidereal time (dec hrs) at julian date jd
	   at west longitude long (decimal hours).  Follows
           definitions in 1992 Astronomical Almanac, pp. B7 and L2.
           Expression for GMST at 0h ut referenced to Aoki et al, A&A 105,
	   p.359, 1982. */

	double t, ut, jdmid, jdint, jdfrac, sid_g;
	double jdnoon2000jan1 = 2451545.;
	long jdin, sid_int;

	jdin = (long)jd;         /* fossil code from earlier package which
			split jd into integer and fractional parts ... */
	jdint = jdin;
	jdfrac = jd - jdint;
	if(jdfrac < 0.5) {
		jdmid = jdint - 0.5;
		ut = jdfrac + 0.5;
	}
	else {
		jdmid = jdint + 0.5;
		ut = jdfrac - 0.5;
	}
	t = (jdmid - jdnoon2000jan1)/36525;
	sid_g = (24110.54841+8640184.812866*t+0.093104*t*t-6.2e-6*t*t*t)/86400.;
	sid_int = (long)sid_g;
	sid_g = sid_g - (double) sid_int;
	sid_g = sid_g + 1.0027379093 * ut - longit/24.;
	sid_int = (long)sid_g;
	sid_g = (sid_g - (double) sid_int) * 24.;
	if(sid_g < 0.) sid_g = sid_g + 24.;
	return(sid_g);
}

static void
lpsun(double jd, double *ra, double *dec)

/* Low precision formulae for the sun, from Almanac p. C24 (1990) */
/* ra and dec are returned as decimal hours and decimal degrees. */

{
	double n, L, g, lambda,epsilon,x,y,z;

	n = jd - 2451545.0;
	L = 280.460 + 0.9856474 * n;
	g = (357.528 + 0.9856003 * n)/DEG_IN_RADIAN;
	lambda = (L + 1.915 * sin(g) + 0.020 * sin(2. * g))/DEG_IN_RADIAN;
	epsilon = (23.439 - 0.0000004 * n)/DEG_IN_RADIAN;

	x = cos(lambda);
	y = cos(epsilon) * sin(lambda);
	z = sin(epsilon)*sin(lambda);

	*ra = (atan_circ(x,y))*HRS_IN_RADIAN;
	*dec = (asin(z))*DEG_IN_RADIAN;
}

// This function combines a few steps that are always used together to
// find the altitude of the sun.
static double
sun_altitude (double jd, double lat, double longit) {
  double ra, dec;
  lpsun(jd, &ra, &dec);
  return altit(dec, lst(jd,longit)-ra, lat);
}

// I messed with this a good bit.
//
//   *  It now converges to within 1 minute.
//   *  It converges better from bad initial guesses.  (Deriv is now
//      updated inside of the loop.)
//   *  It won't roam more than half a day in either direction.
//   *  Max iterations chosen conservatively.
//   *  It finishes with a check to determine what it found.

static double jd_sun_alt(double alt, double jdorig, double lat, double longit,
int &is_sunrise)
{
	/* returns jd at which sun is at a given
		altitude, given jdguess as a starting point. */

        double jdguess = jdorig;
	double jdout, adj = 1.0;
	double deriv, err, del = 0.002;
	double alt2,alt3;
	short i = 0;

        // Convergence when new diff is less than 1/4 minute.
        double convrg = 0.25 / (24.0 * 60.0);

	/* first guess */

	alt2 = sun_altitude (jdguess, lat, longit);
	jdguess = jdguess + del;
	alt3 = sun_altitude (jdguess, lat, longit);
	err = alt3 - alt;
	deriv = (alt3 - alt2) / del;
        if (deriv == 0.0) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
          cerr << "jd_sun_alt got nuked!" << endl;
#endif
          return (-1.0e10);
        }
        adj = -err/deriv;
	while(fabs(adj) >= convrg) {
	  if (i++ == 12) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
            cerr << "jd_sun_alt exceeded max iterations" << endl;
#endif
            return(-1.0e10); /* bad status flag */
  	  }
	  jdguess += adj;
          if (fabs (jdguess - jdorig) > 0.5) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
            cerr << "jd_sun_alt ran outside of its box" << endl;
#endif
            return (-1.0e10);
          }
          alt2 = alt3;
	  alt3 = sun_altitude (jdguess, lat, longit);
	  err = alt3 - alt;
  	  deriv = (alt3 - alt2) / adj;
          if (deriv == 0.0) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
            cerr << "jd_sun_alt got nuked!" << endl;
#endif
            return (-1.0e10);
          }
          adj = -err/deriv;
	}
	jdout = jdguess;

        // Figure out whether this is a sunrise or a sunset by shifting
        // by 1 second.
	{
          jdguess -= 1.0 / SEC_IN_DAY;
	  alt2 = sun_altitude (jdguess, lat, longit);
          is_sunrise = (alt2 < alt3);
	}

#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
        cerr << "jd_sun_alt converged in " << i << " iterations" << endl;
#endif
	return(jdout);
}

static void flmoon(int n, int nph, double *jdout)

/* Gives jd (+- 2 min) of phase nph on lunation n; replaces
less accurate Numerical Recipes routine.  This routine
implements formulae found in Jean Meeus' *Astronomical Formulae
for Calculators*, 2nd edition, Willman-Bell.  A very useful
book!! */

{
  double jd, cor;
  double M, Mpr, F;
  double T;
  double lun;

  lun = (double) n + (double) nph / 4.;
  T = lun / 1236.85;
  jd = 2415020.75933 + 29.53058868 * lun
	  + 0.0001178 * T * T
	  - 0.000000155 * T * T * T
	  + 0.00033 * sin((166.56 + 132.87 * T - 0.009173 * T * T)/DEG_IN_RADIAN);
  M = 359.2242 + 29.10535608 * lun - 0.0000333 * T * T - 0.00000347 * T * T * T;
  M = M / DEG_IN_RADIAN;
  Mpr = 306.0253 + 385.81691806 * lun + 0.0107306 * T * T + 0.00001236 * T * T * T;
  Mpr = Mpr / DEG_IN_RADIAN;
  F = 21.2964 + 390.67050646 * lun - 0.0016528 * T * T - 0.00000239 * T * T * T;
  F = F / DEG_IN_RADIAN;
  if((nph == 0) || (nph == 2)) {/* new or full */
	  cor =   (0.1734 - 0.000393*T) * sin(M)
		  + 0.0021 * sin(2*M)
		  - 0.4068 * sin(Mpr)
		  + 0.0161 * sin(2*Mpr)
		  - 0.0004 * sin(3*Mpr)
		  + 0.0104 * sin(2*F)
		  - 0.0051 * sin(M + Mpr)
		  - 0.0074 * sin(M - Mpr)
		  + 0.0004 * sin(2*F+M)
		  - 0.0004 * sin(2*F-M)
		  - 0.0006 * sin(2*F+Mpr)
		  + 0.0010 * sin(2*F-Mpr)
		  + 0.0005 * sin(M+2*Mpr);
	  jd = jd + cor;
  }
  else {
	  cor = (0.1721 - 0.0004*T) * sin(M)
		  + 0.0021 * sin(2 * M)
		  - 0.6280 * sin(Mpr)
		  + 0.0089 * sin(2 * Mpr)
		  - 0.0004 * sin(3 * Mpr)
		  + 0.0079 * sin(2*F)
		  - 0.0119 * sin(M + Mpr)
		  - 0.0047 * sin(M - Mpr)
		  + 0.0003 * sin(2 * F + M)
		  - 0.0004 * sin(2 * F - M)
		  - 0.0006 * sin(2 * F + Mpr)
		  + 0.0021 * sin(2 * F - Mpr)
		  + 0.0003 * sin(M + 2 * Mpr)
		  + 0.0004 * sin(M - 2 * Mpr)
		  - 0.0003 * sin(2*M + Mpr);
	  if(nph == 1) cor = cor + 0.0028 -
			  0.0004 * cos(M) + 0.0003 * cos(Mpr);
	  if(nph == 3) cor = cor - 0.0028 +
			  0.0004 * cos(M) - 0.0003 * cos(Mpr);
	  jd = jd + cor;

  }
  *jdout = jd;
}

// This began as print_phase in skycalc.c.
// Set direction = 1 to go forward, -1 to go backward.
static void
find_next_moon_event (double &jd, int &phase, int direction) {
  assert (direction == 1 || direction == -1);
  double newjd, lastnewjd, nextjd;
  short kount=0;

  // Originally, there was no problem with getting snagged, but since
  // I introduced the roundoff error going back and forth with Timestamp,
  // now it's a problem.
  // Move ahead by 1 second to avoid snagging.
  jd += (double)direction / SEC_IN_DAY;

  // Find current lunation.  I have doubled the safety margin since
  // it seemed biased for forwards search.
  int nlast = (int)((jd - 2415020.5) / 29.5307 - 2*direction);

  flmoon(nlast,0,&lastnewjd);
  nlast += direction;
  flmoon(nlast,0,&newjd);
  while ((direction == 1 && newjd <= jd) ||
         (direction == -1 && newjd >= jd)) {
    lastnewjd = newjd;
    nlast += direction;
    flmoon(nlast,0,&newjd);
    assert (kount++ < 5); // Original limit was 35 (!)
  }
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
  cerr << "Finding lunation took " << kount << " iterations." << endl;
#endif

  // We might save some work here by estimating, i.e.:
  //   x = jd - lastnewjd;
  //   noctiles = (int)(x / 3.69134);  /* 3.69134 = 1/8 month; truncate. */
  // However....

  if (direction == 1) {
    assert (lastnewjd <= jd && newjd > jd);
    phase = 1;
    nlast--; // Lunation is lastnewjd's lunation
    flmoon (nlast, phase, &nextjd);       // Phase = 1
    if (nextjd <= jd) {
      flmoon (nlast, ++phase, &nextjd);   // Phase = 2
      if (nextjd <= jd) {
        flmoon (nlast, ++phase, &nextjd); // Phase = 3
        if (nextjd <= jd) {
          phase = 0;
          nextjd = newjd;
        }
      }
    }
  } else {
    assert (lastnewjd >= jd && newjd < jd);
    phase = 3;
    // Lunation is newjd's lunation
    flmoon (nlast, phase, &nextjd);       // Phase = 3
    if (nextjd >= jd) {
      flmoon (nlast, --phase, &nextjd);   // Phase = 2
      if (nextjd >= jd) {
        flmoon (nlast, --phase, &nextjd); // Phase = 1
        if (nextjd >= jd) {
          phase = 0;
          nextjd = newjd;
        }
      }
    }
  }
  jd = nextjd;
}

// Front end with XTide data types
void
find_next_moon_event (Timestamp &t, Station::EventType &etype_out,
Station::Direction d) {
  int dir, phase;
  double jd = t.jd();
  if (d == Station::forward)
    dir = 1;
  else
    dir = -1;
  find_next_moon_event (jd, phase, dir);
  t = Timestamp (jd);
  switch (phase) {
  case 0:
    etype_out = Station::newmoon;
    break;
  case 1:
    etype_out = Station::firstquarter;
    break;
  case 2:
    etype_out = Station::fullmoon;
    break;
  case 3:
    etype_out = Station::lastquarter;
    break;
  default:
    assert (0);
  }
}

// Here's another opportunity for Jeff Dairiki to write a better root
// finder :-)
//
// jd_sun_alt needed good initial guesses to find sunrises and
// sunsets.  This was not a problem since good guesses were easy to
// come by.  The original skycalendar did this with estimates based on
// the local midnight:
//
//    jd = date_to_jd(date); /* local midnight */
//    jdmid = jd + zone(use_dst,stdz,jd,jdbdst,jdedst) / 24.;  /* corresponding ut */
//    stmid = lst(jdmid,longit);
//    lpsun(jdmid,&rasun,&decsun);
//    hasunset = ha_alt(decsun,lat,-(0.83+horiz));
//    jdsunset = jdmid + adj_time(rasun+hasunset-stmid)/24.; /* initial guess */
//    jdsunset = jd_sun_alt(-(0.83+horiz), jdsunset,lat,longit); /* refinement */
//    jdsunrise = jdmid + adj_time(rasun-hasunset-stmid)/24.;
//    jdsunrise = jd_sun_alt(-(0.83+horiz),jdsunrise,lat,longit);
//
// While efficient, this is an inconvenient way to go about it when
// I'm looking for the next event from time t, and don't even know
// when midnight is.  So I messed with jd_sun_alt to make it converge
// better from bad initial guesses, and substituted three bad guesses
// for one good one.  Normally, two would suffice, but I wanted to
// add a safety margin in case one of them happens to land at a point
// that nukes jd_sun_alt.

// Set direction = 1 to go forward, -1 to go backward.
static void
find_next_sun_event (double &jd, double lat, double longit, int &is_sunrise,
int direction) {
  assert (direction == 1 || direction == -1);
  int its = 0;

  // Move ahead by 1 minute to avoid snagging.
  jd += (double)direction / (24.0 * 60.0);

  double jdorig = jd;
  double inc = (double)direction / 6.0; // 4 hours
  int looking_for;

  // First we want to know what we are looking for.
  looking_for = (sun_altitude (jdorig, lat, longit) < sunrise_altitude);
  if (direction == -1)
    looking_for = !looking_for;

  // Now give it a decent try.  Because jd_sun_alt is so unpredictable,
  // we can even find things out of order (which is one reason we need
  // to know what we're looking for).
  double jdlooper = jdorig;
  do {
    its++;
    jd = jd_sun_alt (sunrise_altitude, jdlooper, lat, longit, is_sunrise);
    jdlooper += inc;
  // Loop either on error return (which is a negative number), or if we
  // found an event in the wrong direction, or the wrong kind of event.
  } while ((jd < 0.0) ||
           (direction == 1 && jd <= jdorig) ||
           (direction == -1 && jd >= jdorig) ||
           (is_sunrise != looking_for));

#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
  cerr << "find_next_sun_event converged in " << its << " iterations" << endl;
#endif
}

// Front end with XTide data types
void
find_next_sun_event (Timestamp &t, Coordinates c,
Station::EventType &etype_out, Station::Direction d) {
  assert (!(c.isNull()));
  int dir, issunrise;
  double jd = t.jd();
  if (d == Station::forward)
    dir = 1;
  else
    dir = -1;
  // skycal "longit" is measured in HOURS WEST, not degrees east.
  // (lat is unchanged)
  find_next_sun_event (jd, c.lat(), -(c.lng())/15.0, issunrise, dir);
  t = Timestamp (jd);
  if (issunrise)
    etype_out = Station::sunrise;
  else
    etype_out = Station::sunset;
}

// Simple question deserving a simple answer...
int sun_is_up (Timestamp &t, Coordinates c) {
  assert (!(c.isNull()));
  return (sun_altitude (t.jd(), c.lat(), -(c.lng())/15.0) >= sunrise_altitude);
}
