/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef XSID_WB_DEBUG
#include <iomanip>
#endif
#include <fstream>
#include <iostream>
using namespace std;

#include "IniFile.h"

IniFile::IniFile(const int newMaxLineLen) : TextFile(newMaxLineLen)
{
    groups.setAutoDelete(true);
    curGroup = 0;
    curEntry = 0;
}

IniFile::~IniFile()
{
}

void IniFile::setHeader(const char* comment)
{
    configHeaderComment = comment;
}

void IniFile::setGroup(const char* name)
{
    if ( curGroup!=0 && curGroup->name==name )
        return;
#ifdef XSID_WB_DEBUG
    cout << "::setGroup(" << name << ")" << endl;
#endif
    if ( !haveGroup(name) )
    {
        // Group not found. Create it.
        Group* ns = new Group(name);
        groups.append(ns);
        curGroup = ns;
#ifdef XSID_WB_DEBUG
        cout << "Not found. Created it." << endl;
#endif
    }
#ifdef XSID_WB_DEBUG
    cout << curGroup->name << endl;
#endif
}

// Overloaded.
void IniFile::setEntry(const char* key, const char* value)
{
    if ( haveEntry(key) )
    {
        // Key found. Assign/copy new value.
        curEntry->value = value;
    }
    else
    {
#ifdef XSID_WB_DEBUG
        cout << "Not found. Created it." << endl;
#endif
        // Key not found. Create key/value pair.
        curEntry = new Entry(key,value);
        curGroup->entries.append(curEntry);
    }
#ifdef XSID_WB_DEBUG
    cout << "key = " << curEntry->key << endl;
    cout << "value = " << curEntry->value << endl;
#endif
}

// Overloaded.
void IniFile::setEntry(const char* key, int value)
{
    QString tmp;
    tmp.setNum(value);
    setEntry(key,tmp);
}

// Overloaded.
void IniFile::setEntry(const char* key, long int value)
{
    QString tmp;
    tmp.setNum(value);
    setEntry(key,tmp);
}

// Overloaded.
void IniFile::setEntry(const char* key, float value)
{
    QString tmp;
    tmp.setNum(value);
    setEntry(key,tmp);
}

// Overloaded.
bool IniFile::getValue(const char* key, QString& value)
{
    if ( haveEntry(key) )
    {
        value = curEntry->value;
        return true;
    }
    else
    {
        return false;
    }
}

// Overloaded.
bool IniFile::getValue(const char* key, float& value)
{
    if ( haveEntry(key) )
    {
        value = curEntry->value.toFloat();
        return true;
    }
    else
    {
        return false;
    }
}

// Overloaded.
bool IniFile::getValue(const char* key, int& value)
{
    long int i;
    bool ret = getValue(key,i);  // get long int value
    if ( ret )
        value = i;               // truncate it
    return ret;
}

// Overloaded.
bool IniFile::getValue(const char* key, long int& value)
{
    if ( haveEntry(key) )
    {
        value = curEntry->value.toLong();
        return true;
    }
    else
    {
        return false;
    }
}

// Overloaded.
bool IniFile::getValue(const char* key, bool& value)
{
    long int i;
    bool ret = getValue(key,i);  // get long int value
    if ( ret )
        value = (i!=0);          // convert to int
    return ret;
}

const QString& IniFile::getString(const char* key)
{
    if ( haveEntry(key) )
        return curEntry->value;
    else
        return empty;
}

bool IniFile::getBool(const char* key)
{
    if ( haveEntry(key) )
        return (curEntry->value.toInt()!=0);
    else
        return false;
}

long int IniFile::getLongInt(const char* key)
{
    if ( haveEntry(key) )
        return curEntry->value.toLong();
    else
        return 0;
}

bool IniFile::haveGroup(const char* name)
{
    QListIterator<Group> it(groups);
    for ( it.toFirst(); it.current(); ++it ) 
    {
        if ( it.current()->name==name )
        {
            curGroup = it.current();  
            return true;  // group found
        }
    }
    return false;  // group not found
}

bool IniFile::haveEntry(const char* key)
{
    if ( curGroup==0 )
        return false;
    QListIterator<Entry> it(curGroup->entries);
    for ( it.toFirst(); it.current(); ++it ) 
    {
        if ( it.current()->key==key )
        {
            curEntry = it.current();
            return true;  // entry found
        }
    }
    return false;
}

void IniFile::removeEntry(const char* key)
{
    if ( haveEntry(key) )
    {
        curGroup->entries.remove(curEntry);
    }
    else
    {
#ifdef XSID_WB_DEBUG
        cout << "Not found. Nothing to remove." << endl;
#endif
    }
}

bool IniFile::load(const char* fileName)
{
    open(fileName);
    while (status && isGood && !isEOF())  // line-by-line loop
    {
        getLine();
        // Skip blank and comment lines.
        while (status && !isEOF() && isBlank() || isComment())
        {
            getLine();
        };
        if ( isEOF() )
            break;
        // Evaluate line.
#ifdef XSID_WB_DEBUG
        cout << "Line " << getLineNum() << ", " << getLineLen() << ": ";
        cout << getLineBuf() << endl;
        cout << "ParseBuf: " << getParseBuf() << endl;
#endif
        if ( lineBuf[0]=='[' )  // group start?
        {
            QString tmp = lineBuf+1;
            int closePos = tmp.find(']');
            if ( closePos>0 )
            {
                tmp.truncate(closePos);
                setGroup(tmp);
            }
        }
        else  // some other line content
        {
            QString tmp = lineBuf;
            int equalPos = tmp.find('=');
            if ( equalPos>0 )
            {
                setEntry(tmp.left(equalPos),tmp.mid(equalPos+1));
            }
        }
    }
    close();
    return isGood;
}

bool IniFile::save(const char* fileName)
{
    bool wasSuccess = false;

#ifdef XSID_HAVE_IOS_BIN
    ofstream toFile(fileName,ios::out|ios::bin|ios::trunc);
#else
    ofstream toFile(fileName,ios::out|ios::binary|ios::trunc);
#endif
    if ( !toFile.fail() )
    {
        if ( !configHeaderComment.isEmpty() )
            toFile << configHeaderComment << endl;

        QListIterator<Group> secit(groups);
        for ( secit.toFirst(); secit.current(); ++secit ) 
        {
            toFile << '[' << secit.current()->name << ']' << endl;

            QListIterator<Entry> setit(secit.current()->entries);
            for ( setit.toFirst(); setit.current(); ++setit ) 
            {
                toFile << setit.current()->key << '=' 
                    << setit.current()->value << endl;

            }
            toFile << endl;
        }
        toFile.close();
        wasSuccess = !toFile.fail();
    }
    return wasSuccess;
}
