/* $Header: /fridge/cvs/xscorch/sgtk/swidgets/sdialog.c,v 1.7 2001/07/08 07:22:11 justins Exp $ */
/*
   
   xscorch - sdialog.c        Copyright(c) 2001,2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   Scorched dialogues
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sdialog.h>
#include <slabel.h>
#include <gdk/gdkkeysyms.h>
#include <sgtk.h>
#include <sstr.h>
#include <sgtkfont.h>


static GtkWindowClass *parent_class;



enum _ScDialogSignals {
   APPLY_SIGNAL,
   LAST_SIGNAL
};
static gint _sc_dialog_signals[LAST_SIGNAL] = { 0 };



static void _sc_dialog_destroy(GtkObject *obj) {

   ScDialog *dlg = SC_DIALOG(obj);

   if(dlg->state != NULL) *dlg->state = SC_DIALOG_DESTROYED;

   if(GTK_OBJECT_CLASS(parent_class)->destroy != NULL) {
      GTK_OBJECT_CLASS(parent_class)->destroy(obj);
   }

}



static void _sc_dialog_ok_clicked(GtkWidget *button, ScDialog *dlg) {

   gtk_widget_ref(GTK_WIDGET(dlg));
   gtk_signal_emit(GTK_OBJECT(dlg), _sc_dialog_signals[APPLY_SIGNAL]);
   gtk_widget_unref(GTK_WIDGET(dlg));

   if(dlg->state != NULL) *dlg->state = SC_DIALOG_ACCEPTED;
   if(!(dlg->flags & SC_DIALOG_DELAY_DESTROY)) gtk_widget_destroy(GTK_WIDGET(dlg));

}



static void _sc_dialog_cancel_clicked(GtkWidget *button, ScDialog *dlg) {

   if(dlg->state != NULL) *dlg->state = SC_DIALOG_REJECTED;
   if(!(dlg->flags & SC_DIALOG_DELAY_DESTROY)) gtk_widget_destroy(GTK_WIDGET(dlg));

}



static void _sc_dialog_apply_clicked(GtkWidget *button, ScDialog *dlg) {

   gtk_widget_ref(GTK_WIDGET(dlg));
   gtk_signal_emit(GTK_OBJECT(dlg), _sc_dialog_signals[APPLY_SIGNAL]);
   gtk_widget_unref(GTK_WIDGET(dlg));

}



static void _sc_dialog_close_clicked(GtkWidget *button, ScDialog *dlg) {

   if(dlg->state != NULL) *dlg->state = SC_DIALOG_REJECTED;
   if(!(dlg->flags & SC_DIALOG_DELAY_DESTROY)) gtk_widget_destroy(GTK_WIDGET(dlg));

}



static gint _sc_dialog_key_press(GtkWidget *widget, GdkEventKey *key) {

   ScDialog *dlg = SC_DIALOG(widget);
   
   /* Try to run parent keyhandler first */
   if(GTK_WIDGET_CLASS(parent_class)->key_press_event != NULL) {
      if(GTK_WIDGET_CLASS(parent_class)->key_press_event(widget, key)) {
         return(TRUE);
      }
   }

   switch(key->keyval) {
      case GDK_Return:
      case GDK_KP_Enter:
         if(dlg->flags & SC_DIALOG_OK)          _sc_dialog_ok_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_YES)    _sc_dialog_ok_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_APPLY)  _sc_dialog_apply_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_CANCEL) _sc_dialog_cancel_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_NO)     _sc_dialog_cancel_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_CLOSE)  _sc_dialog_close_clicked(widget, dlg);
         else return(FALSE);
         return(TRUE);
      case GDK_Escape:
         if(dlg->flags & SC_DIALOG_CANCEL)      _sc_dialog_cancel_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_NO)     _sc_dialog_cancel_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_CLOSE)  _sc_dialog_close_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_OK)     _sc_dialog_ok_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_YES)    _sc_dialog_ok_clicked(widget, dlg);
         else if(dlg->flags & SC_DIALOG_APPLY)  _sc_dialog_apply_clicked(widget, dlg);
         else return(FALSE);
         return(TRUE);
   }

   return(FALSE);

}



static void _sc_dialog_class_init(ScDialogClass *klass) {

   GtkObjectClass *object_class = (GtkObjectClass *)klass;
   
   /* Get parent class */
   parent_class = gtk_type_class(gtk_window_get_type());
   
   _sc_dialog_signals[APPLY_SIGNAL] =
      gtk_signal_new("apply", 
                     GTK_RUN_LAST,
                     object_class->type,
                     GTK_SIGNAL_OFFSET(ScDialogClass, apply),
                     gtk_marshal_NONE__NONE,
                     GTK_TYPE_NONE, 0);

   gtk_object_class_add_signals(object_class, _sc_dialog_signals, LAST_SIGNAL);

   /* Setup new signals */
   klass->apply = NULL;
   
   /* Setup signals from parent */
   GTK_OBJECT_CLASS(klass)->destroy = _sc_dialog_destroy;
   GTK_WIDGET_CLASS(klass)->key_press_event = _sc_dialog_key_press;

}



static void _sc_dialog_init_obj(ScDialog *dlg) {
   
   dlg->state = NULL;

}



GtkType sc_dialog_get_type(void) {

   static GtkType sc_dialog_type = 0;
   
   if(sc_dialog_type == 0) {
      static const GtkTypeInfo sc_dialog_info = {
         (char *)"ScDialog",
         sizeof(ScDialog),
         sizeof(ScDialogClass),
         (GtkClassInitFunc)_sc_dialog_class_init,
         (GtkObjectInitFunc)_sc_dialog_init_obj,
         (GtkArgSetFunc)NULL,
         (GtkArgGetFunc)NULL
      };
      sc_dialog_type = gtk_type_unique(gtk_window_get_type(), &sc_dialog_info);
   }
    
   return(sc_dialog_type);

}  



GtkWidget *sc_dialog_new(const char *title, const char *msgtext, guint flags) {

   ScDialog *dialog;
   GtkWidget *vbox;
   GtkWidget *hbox;
   GtkWidget *msg;
   GtkWidget *btn;
   
   dialog = gtk_type_new(sc_dialog_get_type());
   g_return_val_if_fail(dialog != NULL, NULL);

   if(title != NULL) gtk_window_set_title(GTK_WINDOW(dialog), title);

   vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(dialog), 10);
   gtk_container_add(GTK_CONTAINER(dialog), vbox);

   if(msgtext != NULL) {
      msg = gtk_label_new(msgtext);
      gtk_widget_set_usize(msg, 350, 0);
      gtk_label_set_line_wrap(GTK_LABEL(msg), TRUE);
      gtk_box_pack_start(GTK_BOX(vbox), msg, TRUE, TRUE, 0);
   }
   if(flags & SC_DIALOG_NO_GRID) {
      dialog->grid = NULL;
   } else {
      dialog->grid = gtk_table_new(1, 1, FALSE);
      gtk_box_pack_start(GTK_BOX(vbox), dialog->grid, TRUE, TRUE, 0);
   }

   gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);
   hbox = gtk_hbox_new(FALSE, 5);
   /* This box houses the command buttons and should not rescale vertically */
   gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

   if(flags & SC_DIALOG_CLOSE) {
      btn = gtk_button_new_with_label(" Close ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(_sc_dialog_close_clicked), dialog);
      gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & SC_DIALOG_APPLY) {
      btn = gtk_button_new_with_label(" Apply ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(_sc_dialog_apply_clicked), dialog);
      gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & SC_DIALOG_NO) {
      btn = gtk_button_new_with_label(" No ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(_sc_dialog_cancel_clicked), dialog);
      gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & SC_DIALOG_CANCEL) {
      btn = gtk_button_new_with_label(" Cancel ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(_sc_dialog_cancel_clicked), dialog);
      gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & SC_DIALOG_YES) {
      btn = gtk_button_new_with_label(" Yes ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(_sc_dialog_ok_clicked), dialog);
      gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & SC_DIALOG_OK) {
      btn = gtk_button_new_with_label(" Ok ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(_sc_dialog_ok_clicked), dialog);
      gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }

   dialog->flags = flags;
   return(GTK_WIDGET(dialog));

}



gboolean sc_dialog_run(ScDialog *dialog) {

   ScDialogState state;

   dialog->flags |= SC_DIALOG_DELAY_DESTROY;
   state = SC_DIALOG_WAITING;
   dialog->state = &state;

   gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
   gtk_widget_show_all(GTK_WIDGET(dialog));
   if(!(dialog->flags & SC_DIALOG_NONMODAL)) gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

   while(state == SC_DIALOG_WAITING) {
      do {
         gtk_main_iteration();
      } while(gtk_events_pending());
   }

   dialog->state = NULL;
   if(state != SC_DIALOG_DESTROYED) gtk_widget_destroy(GTK_WIDGET(dialog));
   return(state == SC_DIALOG_ACCEPTED);

}



void sc_dialog_show(ScDialog *dialog) {

   dialog->flags &= ~SC_DIALOG_DELAY_DESTROY;
   dialog->state = NULL;

   gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
   gtk_widget_show_all(GTK_WIDGET(dialog));
   if(!(dialog->flags & SC_DIALOG_NONMODAL)) gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

}



void sc_dialog_grid_attach(ScDialog *dlg, GtkWidget *widget, int row, int col) {

   gtk_table_attach(GTK_TABLE(dlg->grid), widget, col, col + 1, row, row + 1, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 2, 2);
   if(GTK_IS_MISC(widget)) gtk_misc_set_alignment(GTK_MISC(widget), 0, 0.5);
   
}



void sc_dialog_grid_attach_label(ScDialog *dlg, const char *msg, int row, int col) {

   GtkWidget *label = sc_label_new(msg);
   gtk_table_attach(GTK_TABLE(dlg->grid), label, col, col + 1, row, row + 1, GTK_FILL, GTK_FILL, 2, 2);
   gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
   
}



/***  General-purpose dialogs    ***/



void sc_dialog_message(const char *title, const char *textmsg) {

   GtkWidget *dlg;
   
   dlg = sc_dialog_new(title, textmsg, SC_DIALOG_OK | SC_DIALOG_NO_GRID);
   sc_dialog_show(SC_DIALOG(dlg));

}


gboolean sc_dialog_query(const char *title, const char *textmsg) {

   gboolean result;
   GtkWidget *dlg;
   
   dlg = sc_dialog_new(title, textmsg, SC_DIALOG_YES | SC_DIALOG_NO | SC_DIALOG_NO_GRID);
   result = sc_dialog_run(SC_DIALOG(dlg));
   return(result);

}


void sc_dialog_error(const char *s) {

   char *err;
   char *buf;
   
   err = serror(errno);
   if(err == NULL) {
      sc_dialog_message("Error", s);
      return;
   }
   
   buf = (char *)malloc(strlen(s) + strlen(err) + 4);
   if(buf == NULL) {
      sc_dialog_message("Error", s);
      return;
   }
   
   snprintfn(buf, strlen(s) + strlen(err) + 4, "%s: %s", s, err);
   sc_dialog_message("Error", buf);
   free(buf);
   return;

}



void sc_dialog_text(const char *filename) {

   ScDialog *dlg;
   char buf[SC_DIALOG_STRING_BUFFER];
   char out[SC_DIALOG_STRING_BUFFER];
   char *pin;
   char *pout;
   int  width;
   int height;
   FILE *f;
   GdkFont *font;
   GdkFont *lfont;
   GdkFont *font_normal;
   GdkFont *font_italic;
   GdkFont *font_bold;
   GtkWidget *scroll;
   GtkWidget *message;
   
   dlg = SC_DIALOG(sc_dialog_new(filename, NULL, SC_DIALOG_CLOSE | SC_DIALOG_NONMODAL));

   if((f = fopen(filename, "r")) == NULL) {
      snprintfn(buf, sizeof(buf), "Cannot open file %s.\n", filename);
      sc_dialog_error(buf);
      return;
   }
   
   /* Load the needed fonts */
   font_normal = gdk_font_load(SC_GTK_FIXED_FONT);
   font_italic = gdk_font_load(SC_GTK_ITALIC_FIXED_FONT);
   font_bold   = gdk_font_load(SC_GTK_BOLD_FIXED_FONT);
   if(font_normal == NULL) {
      width = 8;
      height = 16;
   } else {
      width = gdk_char_width(font_normal, 'W');
      height = (font_normal->ascent + font_normal->descent);
   }
   width  *= 82;
   height *= 33;

   scroll = gtk_scrolled_window_new(NULL, NULL);
   sc_dialog_grid_attach(dlg, scroll, 0, 0);
   gtk_widget_set_usize(scroll, width, height);

   message = gtk_text_new(NULL, NULL);
   gtk_text_set_editable(GTK_TEXT(message), FALSE);
   gtk_container_add(GTK_CONTAINER(scroll), message);

   /* Load in the text data */
   gtk_text_freeze(GTK_TEXT(message));
   while(fgets(buf, SC_DIALOG_STRING_BUFFER, f)) {
      pin   = buf;
      pout  = out;
      font  = NULL;
      lfont = font_normal;
      while(*pin != '\0') {
         if(*(pin + 1) != 0x08) {
            font = font_normal;
         } else {
            font = (*pin == '_' ? font_italic : font_bold);
            pin += 2;
         }
         if(font != lfont) {
            if(pout - out > 0) {
               gtk_text_insert(GTK_TEXT(message), lfont, NULL, NULL, out, pout - out);
            }
            pout  = out;
            lfont = font;
         }
         *(pout++) = *pin;
         if(*pin == '\n') {
            *(pout++) = ' ';
            *(pout++) = ' ';
         }
         pin++;
      }
      if(pout - out > 0) {
         gtk_text_insert(GTK_TEXT(message), lfont, NULL, NULL, out, pout - out);
      }
   }
   if(font_normal != NULL) gdk_font_unref(font_normal);
   if(font_italic != NULL) gdk_font_unref(font_italic);
   if(font_bold   != NULL) gdk_font_unref(font_bold);
   gtk_text_thaw(GTK_TEXT(message));
   fclose(f);

   sc_dialog_show(dlg);
   return;

}



