/* $Header: /fridge/cvs/xscorch/sgame/sweapon.c,v 1.45 2001/08/15 05:34:35 jacob Exp $ */
/*

   xscorch - sweapon.c        Copyright(c) 2001,2000 Jacob Luna Lundberg
                              Copyright(c) 2001,2000 Justin David Smith
   jacob(at)chaos2.org        http://chaos2.org/~jacob
   justins(at)chaos2.org      http://chaos2.org/

   Scorched basic weapon system


   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <xscorch.h>
#include <sconfig.h>
#include <sweapon.h>
#include <splayer.h>
#include <saddconf.h>
#include <sexplosion.h>
#include <sinventory.h>
#include <sphysics.h>
#include <sphoenix.h>
#include <swindow.h>
#include <sshield.h>
#include <scolor.h>
#include <sland.h>
#include <math.h>
#include <sutil/sstr.h>
#include <sutil/shash.h>



/***  Weapon Information    ***/



/*
 * How much weapon selection information do ya wanna see?
 * (bit)  (entertainment)
 *   0  None but deadly errors.
 *   1  Contents of weapon_info structs.
 *   2  Entry and exit values from functions.
 *   4  Contents of weapon_info structs in _sc_weapon_viewable.
 */
#define  SC_WEAPON_INFO_DEBUG   0



inline void _sc_weapon_dump_info(const sc_weapon_info *info) {
/* _sc_weapon_dump_info
   Waddaya think it does, eh? */

   if(info == NULL)
      printf("NULL");
   else
      printf("ident=%i,armslevel=%i,price=%i,bundle=%i,state=0x%.6x,radius=%i,force=%i,liquid=%i,scatter=%i,children=%i,ph_ch=%i,ph_fl=0x%.4x,useless=%s,indirect=%s,name=%s\n",
             info->ident,info->armslevel,info->price,info->bundle,info->state,info->radius,info->force,info->liquid,info->scatter,info->children,info->ph_ch,info->ph_fl,info->useless?"true":"false",info->indirect?"true":"false",info->name);

   fflush(stdout);

}



inline bool _sc_weapon_viewable(const sc_weapon_config *wc, const sc_weapon_info *info, int flags) {
/* sc_weapon_viewable
   Is the weapon viewable under the current rules? */

   #if (SC_WEAPON_INFO_DEBUG & 2)
      printf("DEBUG -   entering _sc_weapon_viewable(wc(%p), info(%p), flags=%i)\n", wc, info, flags);
      fflush(stdout);
   #endif

   if(wc == NULL || info == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG -   exit early, returning false\n");
         fflush(stdout);
      #endif
      return(false);
   }

   #if (SC_WEAPON_INFO_DEBUG & 2)
      printf("DEBUG -     info->armslevel(%p)=%i\n", &info->armslevel, info->armslevel);
      printf("DEBUG -     info->useless(%p)=%i\n", &info->useless, info->useless);
      printf("DEBUG -     info->indirect(%p)=%i\n", &info->indirect, info->indirect);
      fflush(stdout);
   #endif

   #if (SC_WEAPON_INFO_DEBUG & 4)
      printf("DEBUG -     dumping info struct (%p) ...\n", info);
      _sc_weapon_dump_info(info);
   #endif

   if( (!(flags & SC_WEAPON_LIMIT_ARMS)     || info->armslevel <= wc->armslevel)   &&
       (!(flags & SC_WEAPON_LIMIT_USELESS)  || wc->uselessitems || !info->useless) &&
       (!(flags & SC_WEAPON_LIMIT_INDIRECT) || !info->indirect)                    ) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG -   exit, returning true\n");
         fflush(stdout);
      #endif
      return(true);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG -   exit, returning false\n");
         fflush(stdout);
      #endif
      return(false);
   }

}



int sc_weapon_count(const sc_weapon_config *wc, int flags) {
/* sc_weapon_count
   Counts the number of weapons in an sc_weapon_config struct's weaponlist.
   This is optimized but of course if you're calling it a lot you should cache the data.
   The only exception is that the count may change if a new weapon list file is appended. */

   sc_weapon_list_item *iter = wc->weaponlist->defs_head;
   int counter = 0;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_count(wc(%p), flags=%i) pass %i\n", wc, flags, ++track);
      fflush(stdout);
   #endif

   /* Count the weapons */
   while(iter != NULL) {
      if(_sc_weapon_viewable(wc, iter->item, flags))
         ++counter;
      iter = iter->next;
   }

   /* Return the count */
   #if (SC_WEAPON_INFO_DEBUG & 2)
      printf("DEBUG - exit, returning counter=%i\n", counter);
      fflush(stdout);
   #endif
   return(counter);

}



sc_weapon_info *sc_weapon_first(const sc_weapon_config *wc, int flags) {
/* sc_weapon_first
   Return the first viable weapon in the weapon config space. */

   sc_weapon_list_item *iter;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_first(wc(%p), flags=%i) pass %i\n", wc, flags, ++track);
      fflush(stdout);
   #endif

   if(wc == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }
   iter = wc->weaponlist->defs_head;

   while(iter != NULL && !_sc_weapon_viewable(wc, iter->item, flags))
      iter = iter->next;

   if(iter != NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_weapon_dump_info(iter->item);
      #endif
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_weapon_info *sc_weapon_last(const sc_weapon_config *wc, int flags) {
/* sc_weapon_last
   Return the last viable weapon in the weapon config space. */

   sc_weapon_list_item *iter;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_last(wc(%p), flags=%i) pass %i\n", wc, flags, ++track);
      fflush(stdout);
   #endif

   if(wc == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }
   iter = wc->weaponlist->defs_tail;

   while(iter != NULL && !_sc_weapon_viewable(wc, iter->item, flags))
      iter = iter->prev;

   if(iter != NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_weapon_dump_info(iter->item);
      #endif
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_weapon_info *sc_weapon_lookup(const sc_weapon_config *wc, int id, int flags) {
/* sc_weapon_lookup
   Returns the weapon information for index given.  If the item is
   not found or out-of-bounds, then NULL is returned. */

   sc_weapon_list_item *iter;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_lookup(wc(%p), id=%i, flags=%i) pass %i\n", wc, id, flags, ++track);
      fflush(stdout);
   #endif

   if(wc == NULL || id < 0 || id > SC_WEAPON_MAX_ID) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = wc->weaponlist->defs_hash[shash(SC_WEAPON_HASH_BITS, id)];
   while(iter != NULL && iter->item->ident != id)
      iter = iter->chain;

   if(iter != NULL && _sc_weapon_viewable(wc, iter->item, flags)) {
      #if (SC_WEAPON_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_weapon_dump_info(iter->item);
      #endif
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



inline bool _sc_weapon_compare_names(const char *A, const char *B) {
/* _sc_weapon_compare_names
   Basically, make sure the names are the same, allowing for case and punctuation. */

   if(A == NULL || B == NULL)
      return(false);

   while(*A != '\0' && B != '\0') {
      /* next valid A char */
      while(!( (*A == '\0') || (*A >= '0' && *A <= '9') ||
               (*A >= 'A' && *A <= 'Z') || (*A >= 'a' && *A <= 'z') ))
         ++A;
      /* next valid B char */
      while(!( (*B == '\0') || (*B >= '0' && *B <= '9') ||
               (*B >= 'A' && *B <= 'Z') || (*B >= 'a' && *B <= 'z') ))
         ++B;
      /* test for disparate strings at this char pair */
      if( ((*A >= 'a' && *A <= 'z') ? (*A - ('a' - 'A')) : *A) !=
          ((*B >= 'a' && *B <= 'z') ? (*B - ('a' - 'A')) : *B) )
         break;
      ++A;
      ++B;
   }

   /* If both made it to the end then they're ``equal'' enough for us. */
   if(*A == '\0' && *B == '\0')
      return(true);
   else
      return(false);

}



sc_weapon_info *sc_weapon_lookup_by_name(const sc_weapon_config *wc, char *name, int flags) {
/* sc_weapon_lookup
   Returns the weapon information for index given.  If the item is
   not found or out-of-bounds, then NULL is returned. */

   sc_weapon_list_item *iter;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_lookup_by_name(wc(%p), name(%p)=\"%s\", flags=%i) pass %i\n", wc, name, name, flags, ++track);
      fflush(stdout);
   #endif

   if(wc == NULL || name == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = wc->weaponlist->defs_head;
   while(iter != NULL && !_sc_weapon_compare_names(name, iter->item->name))
      iter = iter->next;

   if(iter != NULL && _sc_weapon_viewable(wc, iter->item, flags)) {
      #if (SC_WEAPON_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_weapon_dump_info(iter->item);
      #endif
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_weapon_info *sc_weapon_prev(const sc_weapon_config *wc, const sc_weapon_info *info, int flags) {
/* sc_weapon_prev
   Advance to the previous weapon in the list (with wrapping).  */

   sc_weapon_list_item *iter;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_prev(wc(%p), info(%p), flags=%i) pass %i\n", wc, info, flags, ++track);
      fflush(stdout);
   #endif

   if(wc == NULL || info == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = wc->weaponlist->defs_hash[shash(SC_WEAPON_HASH_BITS, info->ident)];
   while(iter != NULL && iter->item->ident != info->ident)
      iter = iter->chain;

   if(iter == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early (no valid weapons), returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   do {
      iter = iter->prev;
      if(iter == NULL) iter = wc->weaponlist->defs_tail;
   } while(iter->item->ident != info->ident && !_sc_weapon_viewable(wc, iter->item, flags));

   if(iter != NULL && iter->item->ident != info->ident) {
      #if (SC_WEAPON_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_weapon_dump_info(iter->item);
      #endif
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_weapon_info *sc_weapon_next(const sc_weapon_config *wc, const sc_weapon_info *info, int flags) {
/* sc_weapon_next
   Advance to the next weapon in the list (with wrapping).  */

   sc_weapon_list_item *iter;

   #if (SC_WEAPON_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_weapon_next(wc(%p), info(%p), flags=%i) pass %i\n", wc, info, flags, ++track);
      fflush(stdout);
   #endif

   if(wc == NULL || info == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = wc->weaponlist->defs_hash[shash(SC_WEAPON_HASH_BITS, info->ident)];
   while(iter != NULL && iter->item->ident != info->ident)
      iter = iter->chain;

   if(iter == NULL) {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit early (no valid weapons), returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   do {
      iter = iter->next;
      if(iter == NULL) iter = wc->weaponlist->defs_head;
   } while(iter->item->ident != info->ident && !_sc_weapon_viewable(wc, iter->item, flags));

   if(iter != NULL && iter->item->ident != info->ident) {
      #if (SC_WEAPON_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_weapon_dump_info(iter->item);
      #endif
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_WEAPON_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



/***  Weapon Statistics  ***/



double _sc_weapon_stat_yield(const sc_weapon_config *wc, const sc_weapon_info *info) {
/* _sc_weapon_stat_yield
   Find an upper bound yield for a weapon.  */

   double yield;                 /* Total weapon yield */
   const sc_weapon_info *child;  /* Info on child weapon(s) */
   int numchildren;              /* Number of children for phoenix weapon */
   int count;                    /* A counter for frog type phoenix weapons */

   /* Yield from self */
   yield = SQR(info->radius) * info->force;

   /* If liquid (napalm), must multiply by potential area of flooding 
      -- this should scale hot napalm to near the vicinity of a nuke
      (but less), which is reasonable, considering both can take down
      large tanks.  */
   if(SC_WEAPON_IS_LIQUID(info)) yield *= info->liquid;

   /* Phoenix weapons have special yield rules. */
   if(SC_WEAPON_IS_PHOENIX(info)) {
      if(!sc_phoenix_verify(wc, info)) {
         /* This means somebody snuck in a bad child pointer on a
            phoenix weapon.  It shouldn't happen, but if it does, we
            promise to return a negative yield.  AIs are advised to
            select another weapon as this one may cause the game to
            segfault or die and nobody wants that! */
         fprintf(stderr, "Bad phoenix weapon: %s.  Children loop or do not exist.\n", info->name);
         return(yield ? -abs(yield) : -1);
      }
      child = sc_weapon_lookup(wc, SC_PHOENIX_CHILD_TYPE(info), SC_WEAPON_LIMIT_NONE);
      numchildren = SC_PHOENIX_CHILD_COUNT(info);
      if(SC_PHOENIX_IS_CHAIN(info)) {
         /* Each child is made smaller by a constant factor, _including_ the first child */
         for(count = 1; count <= numchildren; ++count)
            yield += _sc_weapon_stat_yield(wc, child) * pow(SC_PHOENIX_INCREASE_FACTOR, count);
      } else if(SC_PHOENIX_IS_CONVERT(info)) {
         /* conversions probably aren't well represented here */
         if(child != info) yield = _sc_weapon_stat_yield(wc, child);
      } else if(SC_PHOENIX_IS_SCATTER(info)) {
         /* scattering doesn't detonate the original weapon */
         if(child != info) yield = numchildren * _sc_weapon_stat_yield(wc, child);
         else              yield = numchildren * yield;
         /* as a policy, scattering takes a slight bonus for "effectiveness" */
         yield *= (SC_PHOENIX_YIELD_SCATTER_SCALE + SC_PHOENIX_SCATTER(info)) / SC_PHOENIX_YIELD_SCATTER_SCALE;
      } else if(SC_PHOENIX_IS_SPIDER(info)) {
         /* spiders detonate the original weapon in the center, and on average 75% of their children */
         if(child != info) yield += 0.75 * numchildren * _sc_weapon_stat_yield(wc, child);
         else              yield += 0.75 * numchildren * yield;
      }
   }

   return(yield);

}



double _sc_weapon_stat_precision(const sc_weapon_config *wc, const sc_weapon_info *info, bool triple) {
/* _sc_weapon_stat_precision
   Find a rough estimator of a weapon's precision.
   Later on we might have a wind shear value, and/or a
   weapon eccentricity value, which would also contribute. */

   double precision;             /* Estimate of weapon precision */
   const sc_weapon_info *child;  /* Info on child weapon(s) */
   int numchildren;              /* Number of children for phoenix weapon */
   int count;                    /* A counter for frog type phoenix weapons */

   /* Horizonal (i.e. not area) space covered, usually 2 * radius.
      If liquid (napalm), the area of flooding is the horizonal space. */
   if(SC_WEAPON_IS_LIQUID(info))
      precision = info->liquid;
   else
      precision = 2 * info->radius;

   /* Triple Turret weapons get three times the kaboom, but only if we have such a turret. */
   if(!triple && SC_WEAPON_IS_TRIPLE(info))
      precision /= 3;

   /* Phoenix weapons have special precision rules. */
   if(SC_WEAPON_IS_PHOENIX(info)) {
      if(!sc_phoenix_verify(wc, info)) {
         /* This means somebody snuck in a bad child pointer on a
            phoenix weapon.  It shouldn't happen, but if it does, we
            promise to return a negative yield.  AIs are advised to
            select another weapon as this one may cause the game to
            segfault or die and nobody wants that! */
         fprintf(stderr, "Bad phoenix weapon: %s.  Children loop or do not exist.\n", info->name);
         return(precision ? -abs(precision) : -1);
      }
      child = sc_weapon_lookup(wc, SC_PHOENIX_CHILD_TYPE(info), SC_WEAPON_LIMIT_NONE);
      numchildren = SC_PHOENIX_CHILD_COUNT(info);
      if(SC_PHOENIX_IS_CHAIN(info)) {
         /* Allow the children to vote on their precisions. */
         for(count = 1; count <= numchildren; ++count)
            precision += pow(SC_PHOENIX_INCREASE_FACTOR, count) * 100 / _sc_weapon_stat_precision(wc, child, triple);
      } else if(SC_PHOENIX_IS_CONVERT(info)) {
         /* Conversions could be more precise, but for now AI's don't know about them. */
         precision *= 100;  /* TEMP - Arbitrary! - JL */
      } else if(SC_PHOENIX_IS_SCATTER(info)) {
         /* Scattering doesn't detonate the original weapon. */
         if(child != info) precision = numchildren * 100 / _sc_weapon_stat_precision(wc, child, triple);
         else              precision = numchildren * precision;
         /* As a rule, scattering is very imprecise, and we hit its precision value hard here. */
         precision *= SC_PHOENIX_SCATTER(info);
      } else if(SC_PHOENIX_IS_SPIDER(info)) {
         /* Spiders detonate the original weapon in the center, and on average 75% of their children */
         if(child != info) precision += 0.75 * numchildren * 100 / _sc_weapon_stat_precision(wc, child, triple);
         else              precision += 0.75 * numchildren * precision;
         /* Spiders are also docked for their imprecise nature. */
         precision *= 2 * SC_PHOENIX_SCATTER(info);
      }
   }

   /* Invert the precision so bigger is better, and normalize a bit.
      This will give a Missile a precision of 1.25, by the way.
      If the weapon is completely imprecise, it scores a zero.
      Likewise, if it doesn't really explode, it gets a zero. */
   precision = (precision && info->force) ? (100 / precision) : 0;

   return(precision);

}



double sc_weapon_statistic(const sc_weapon_config *wc, const sc_weapon_info *info, const sc_player *p, sc_weapon_stat statistic) {
/* sc_weapon_statistic
   Calculate various weapon statistics used in AI purchasing.
   The sc_player pointer is optional (can be NULL, in which case you get defaults). */

   bool triple;
   double result;

   /* Can't allow people to pass us bad info, now can we...? */
   if(wc == NULL || info == NULL)
      return(0);

   switch(statistic) {
      case SC_WEAPON_STAT_PRICE:
         if(info->bundle)
            result = info->price / info->bundle;
         else
            result = 0;
         break;
      case SC_WEAPON_STAT_YIELD:
         result = _sc_weapon_stat_yield(wc, info);
         break;
      case SC_WEAPON_STAT_PRECISION:
         triple = p && (p->ac_state & SC_ACCESSORY_STATE_TRIPLE);
         result = _sc_weapon_stat_precision(wc, info, triple);
         break;
      case SC_WEAPON_STAT_ECON_YIELD:
         if(info->price)
            result = _sc_weapon_stat_yield(wc, info) * info->bundle / info->price;
         else
            result = _sc_weapon_stat_yield(wc, info) * info->bundle / SC_INVENTORY_CHEAPO_FACTOR;
         break;
      case SC_WEAPON_STAT_ECON_PRECISION:
         triple = p && (p->ac_state & SC_ACCESSORY_STATE_TRIPLE);
         if(info->price)
            result = _sc_weapon_stat_precision(wc, info, triple) * info->bundle / info->price;
         else
            result = _sc_weapon_stat_precision(wc, info, triple) * info->bundle / SC_INVENTORY_CHEAPO_FACTOR;
         break;
      case SC_WEAPON_STAT_PRECISION_YIELD:
         triple = p && (p->ac_state & SC_ACCESSORY_STATE_TRIPLE);
         result = _sc_weapon_stat_precision(wc, info, triple) * _sc_weapon_stat_yield(wc, info);
         break;
      default:
         printf("Warning - someone asked for a weapon statistic we don't know how to calculate!\n");
         result = 0;
   }

   return(result);

}



void sc_weapon_info_line(const sc_weapon_config *wc, const sc_weapon_info *info, char *buffer, int buflen) {
/* sc_weapon_info_line
   Create a line of information about the weapon. */

   int moving;

   if(buffer == NULL || buflen <= 0)
      return;

   if(info == NULL) {
      buffer[0] = '\0';
      return;
   }

   /* Clear the buffer. */
   memsetn(buffer, 0, buflen * sizeof(char));
   /* Terminating NULL character. */
   --buflen;

   /* Add the name to the buffer. */
   snprintfn(buffer, buflen, "%s:", info->name);
   moving = strnlen(info->name, SC_INVENTORY_MAX_NAME_LEN - 1) + 1;
   buffer += moving;
   buflen -= moving;

   /* Add spaces out to the end of the name area. */
   while(++moving < 20 && --buflen > 0)
      *(buffer++) = ' ';

   /* Display the weapon's arms level. */
   snprintfn(buffer, buflen, "arms = %1i, ", info->armslevel);
   moving = 10;
   buffer += moving;
   buflen -= moving;

   /* Display the weapon's yield. */
   moving = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_YIELD) / 1000;
   snprintfn(buffer, buflen, "yield = %-8i", moving);
   moving = 8 + (moving ? (int)log10(moving) : 1);
   buffer += moving;
   buflen -= moving;

   /* Add the comma. */
   if(buflen-- > 0)
      *(buffer++) = ',';

   /* Add spaces out to the end of the yield area, plus two. */
   while(++moving < 18 && --buflen > 0)
      *(buffer++) = ' ';

   /* Write out some weapon info flags. */
   snprintfn(buffer, buflen, "%7s %8s %5s %6s %7s",
             SC_WEAPON_IS_PHOENIX(info) ? "complex" : "",
             SC_WEAPON_IS_INFINITE(info) ? "infinite" : "",
             SC_WEAPON_IS_SMOKING(info) ? "smoke" : "",
             SC_WEAPON_IS_TRIPLE(info) ? "triple" : "",
             info->useless ? "useless" : "");

}



void sc_weapon_print_yields(const sc_weapon_config *wc) {
/* sc_weapon_print_yields
   Print current weapon yields. */

   int index;
   double price, yield, precision, econ_yield, econ_prec, prec_yield;
   const int wpcount = sc_weapon_count(wc, SC_WEAPON_LIMIT_INDIRECT);
   const sc_weapon_info *info;

   printf("\n   %-16s  %12s  %12s  %12s  %12s  %12s  %12s\n",
          "Weapon Name", "Price", "Yield", "Precision", "Econ Yield", "Econ Prec", "Prec Yield");
   for(index = 0; index < wpcount; ++index) {
      info = sc_weapon_lookup(wc, index, SC_WEAPON_LIMIT_INDIRECT);
      if(info->bundle > 0) {
         price      = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_PRICE);
         yield      = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_YIELD);
         precision  = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_PRECISION);
         econ_yield = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_ECON_YIELD);
         econ_prec  = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_ECON_PRECISION);
         prec_yield = sc_weapon_statistic(wc, info, NULL, SC_WEAPON_STAT_PRECISION_YIELD);
         printf("   %-16s  %12.0f  %12.0f  %12.9f  %12.0f  %12.9f  %12.0f\n",
                info->name, price, yield, precision, econ_yield, econ_prec, prec_yield);
      }
   }

}



/***  Weapon Creation / Destruction  ***/



sc_weapon *sc_weapon_new(const sc_config *c, sc_weapon_info *i, double x, double y, double vx, double vy, bool ct, int playerid) {
/* sc_weapon_new
   Create a new weapon with the parametres described.  */

   sc_weapon *wp;    /* Weapon pointer */
   int flags;        /* Trajectory flags */

   /* Sanity check */
   if(i == NULL) return(NULL);

   /* Allocate the new weapon structure */   
   wp = (sc_weapon *)malloc(sizeof(sc_weapon));
   if(wp == NULL) return(NULL);

   /* Set all weapon creation defaults */
   wp->playerid = playerid;
   wp->state = i->state;
   wp->weaponinfo = i;
   wp->children = 0;
   wp->chain = NULL;
   wp->exp_res = 1;
   
   /* Are we tunnelling? This is important, 
      as we must set the trajectory flags now! */
   wp->triggered = ct;
   flags = SC_WEAPON_TRAJ_FLAGS(c, wp);

   /* Setup initial trajectory */
   wp->tr = sc_traj_new_velocities(c, c->players[playerid], flags, x, y, vx, vy);

   /* Return the newly created weapon. */
   return(wp);

}



sc_weapon *sc_weapon_fire_new(sc_config *c, sc_player *p, sc_explosion **e) {
/* sc_weapon_fire_new
   A special case, when the tank fires a new weapon.  */

   sc_weapon *wp;    /* A pointer to point to the weapon to return */
   double factorx;   /* Based on turret angle; contribution to X velocity */
   double factory;   /* Based on turret angle; contribution to Y velocity */
   sc_weapon_info *i, *ti = NULL;  /* Weapon info pointers */
   double x, y, vx, vy;            /* Temporary weapon info */
   bool ct;                        /* Temporary weapon info */

   /* Make sure not firing a NULL weapon */
   i = p->selweapon;
   if(SC_WEAPON_IS_NULL(i)) return(NULL);

   /* Calculate contribution to each direction, based on turret angle */
   factorx = cos(p->turret * M_PI / 180);
   factory = sin(p->turret * M_PI / 180);   

   /* Set initial weapon position (based on turret radius and angle) */
   /* Also set initial weapon velocity based on the tank's power. */
   x = sc_player_turret_x(p, p->turret);
   y = sc_player_turret_y(p, p->turret);
   vx = p->power * SC_PHYSICS_VELOCITY_SCL * factorx;
   vy = p->power * SC_PHYSICS_VELOCITY_SCL * factory;
   
   /* Do we need to apply a contact trigger? */
   ct = sc_player_use_contact_trigger(c, p);

   /* If it's a triple turret type weapon and we have a triple turret, then
      we must temporarily replace its weaponinfo with a new phoenix one.
      NOTE that this info will be free'd before this function exits... */
   if((p->ac_state & SC_ACCESSORY_STATE_TRIPLE) && (i->state & SC_WEAPON_STATE_TRIPLE)) {
      ti = (sc_weapon_info *)malloc(sizeof(sc_weapon_info));
      if(ti == NULL) {
         printf("Your weapon hasn't enough memory to fire, sir!  (fire_new() malloc failure)\n");
         return(NULL);
      }
      memcpyn((void *)ti, (void *)i, sizeof(sc_weapon_info));
      ti->ph_fl |= SC_PHOENIX_AT_TANK | SC_PHOENIX_TYPE_SCATTER;
      ti->ph_ch = i->ident;
      /* NOTE that we hard-code behavior of the triple turret here.
         It is a TRIPLE turret after all...  ;-)
         Still, this should probably change at some point. */
      ti->children = 3;
      ti->scatter = 3;
      /* Allocate the new weapon structure */
      wp = sc_weapon_new(c, ti, x, y, vx, vy, ct, p->index);
   } else {
      /* Allocate the new weapon structure */
      wp = sc_weapon_new(c, i, x, y, vx, vy, ct, p->index);
   }

   /* If it is the right phoenix type, the weapon should be split now. */
   if(SC_WEAPON_IS_PHOENIX(wp->weaponinfo) && SC_PHOENIX_IS_AT_TANK(wp->weaponinfo))
      switch(sc_phoenix(SC_PHOENIX_AT_TANK, c, &wp, e)) {
         /* This is the case we expect if it is a tank phoenix weapon */
         case SC_PHOENIX_SIZZLE:
            sc_weapon_landfall(c, wp);
            sc_weapon_free(&wp);
            break;

         /* Neither of these _should_ happen */
         case SC_PHOENIX_DETONATE:
         case SC_PHOENIX_FAILURE:
            sc_weapon_free_chain(&wp);
            break;

         /* This is the case we expect if it is not a tank phoenix weapon. */
         case SC_PHOENIX_NO_ACTION:
            wp->weaponinfo = i; /* safety */
         default:
      }

   /* Decrement player weapon inventory */
   if(!SC_WEAPON_IS_INFINITE(i) && (--(i->inventories[p->index]) <= 0)) {
      /* We ran out, so return to the default weapon. */
      i = SC_WEAPON_DEFAULT(c->weapons);
      p->selweapon = i;
   }

   /* This is quite important, despite being usually free(NULL). */
   free(ti);
   return(wp);

}



void sc_weapon_landfall(sc_config *c, const sc_weapon *wp) {
/* sc_weapon_landfall
   Perform any landfall operations pending that are associated with
   this particular weapon.  Note that landfall may also affect player
   tank positions. */

   if(c == NULL || wp == NULL || wp->tr == NULL) return;
   sc_traj_landfall(c, wp->tr);

}



void sc_weapon_free(sc_weapon **wp) {
/* sc_weapon_free
   Releases the weapon at the head of the list, and sets *wp to its old
   chain pointer (the next weapon in the list).  */

   sc_weapon *chain; /* On success, wp will be set to the chain weapon */
   
   /* Release the weapon */
   if(wp == NULL || *wp == NULL) return;
   chain = (*wp)->chain;
   sc_traj_free(&(*wp)->tr);
   free(*wp);
   (*wp) = chain;

}



void sc_weapon_free_chain(sc_weapon **wp) {
/* sc_weapon_free_chain
   Deletes an entire weapon chain.  *wp == NULL after this call.  */

   sc_weapon *chain; /* We must follow the chain and exterminate it */

   /* Release each weapon */
   if(wp == NULL) return;
   while(*wp != NULL) {
      chain = (*wp)->chain;
      free(*wp);
      (*wp) = chain;
   }

}



void sc_weapon_create_all(sc_config *c, sc_explosion **e) {
/* sc_weapon_create_all
   Create the weapons for each player, to be launched. */

   sc_player *p;
   int i;

   for(i = c->numplayers - 1; i >= 0; --i) {
      p = c->players[i];
      if(!p->dead && p->armed) {
         p->weapons = sc_weapon_fire_new(c, p, e);
         p->armed = false;
      }
   }

}



/***  Weapon Lists and Configuration Structs  ***/



void sc_weapon_info_free(sc_weapon_info **wi) {
/* sc_weapon_info_free
   Invalidate memory used by an sc_weapon_info. */

   /* Make sure there is an item to free */
   if(wi == NULL || *wi == NULL) return;
   /* Free the item's name if it has one */
   if((*wi)->name != NULL) free((*wi)->name);
   /* Free the item */
   free(*wi);
   *wi = NULL;

}



void sc_weapon_list_item_free(sc_weapon_list_item **wli) {
/* sc_weapon_list_item_free
   Invalidate a list item. */

   if(wli == NULL || *wli == NULL) return;
   sc_weapon_info_free(&(*wli)->item);
   free(*wli);
   *wli = NULL;

}



void sc_weapon_list_free(sc_weapon_list **wl) {
/* sc_weapon_list_free
   Invalidate a weapon list and subordinate chain. */

   sc_weapon_list_item *item = (*wl)->defs_head;

   if(wl == NULL || *wl == NULL) return;
   memsetn((*wl)->defs_hash, 0, sizeof(void *) * SC_WEAPON_HASH_SIZE);
   while(item != NULL) {
      (*wl)->defs_head = item->next;
      sc_weapon_list_item_free(&item);
      item = (*wl)->defs_head;
   }
   free(*wl);
   *wl = NULL;

}



void sc_weapon_config_free(sc_weapon_config **wc) {
/* sc_weapon_config_free
   Invalidate memory used by a weapon config struct. */

   if(wc == NULL || *wc == NULL) return;
   sc_weapon_list_free(&(*wc)->weaponlist);
   free(*wc);
   *wc = NULL;

}



void sc_weapon_inventory_clear(sc_weapon_config *wc) {
/* sc_weapon_inventory_clear
   Clear out the player weapon inventories. */

   int count = sc_weapon_count(wc, SC_WEAPON_LIMIT_NONE);
   sc_weapon_info *info = sc_weapon_first(wc, SC_WEAPON_LIMIT_NONE);

   for(; count > 0; --count) {
      if(!SC_WEAPON_IS_INFINITE(info))
         memsetn(info->inventories, 0, SC_MAX_PLAYERS * sizeof(int));
      info = sc_weapon_next(wc, info, SC_WEAPON_LIMIT_NONE);
   }

}



sc_weapon_list_item *sc_weapon_list_item_new() {
/* sc_weapon_list_item_new
   Allocate and set to NULL a new sc_weapon_list_item struct. */

   sc_weapon_list_item *wli;

   wli = (sc_weapon_list_item *)malloc(sizeof(sc_weapon_list_item));
   if(wli == NULL) return(NULL);

   wli->chain = NULL;
   wli->prev = NULL;
   wli->next = NULL;

   wli->item = (sc_weapon_info *)malloc(sizeof(sc_weapon_info));
   if(wli->item == NULL) {
      free(wli);
      return(NULL);
   }

   wli->item->ident = -1;
   wli->item->name = NULL;

   return(wli);

}



sc_weapon_list *sc_weapon_list_new(void) {
/* sc_weapon_list_new
   Allocate space and set defaults (NULL, heh) on a new sc_weapon_list struct. */

   sc_weapon_list *wl;

   wl = (sc_weapon_list *)malloc(sizeof(sc_weapon_list));
   if(wl == NULL) return(NULL);

   /* Not necessary on Linux but I don't trust some others (*cough* Solaris). */
   memsetn(wl->defs_hash, 0, sizeof(void *) * SC_WEAPON_HASH_SIZE);

   wl->defs_head = NULL;
   wl->defs_tail = NULL;

   return(wl);

}



sc_weapon_config *sc_weapon_config_new(void) {
/* sc_weapon_config_new
   Allocate space and set defaults on a new sc_weapon_config struct. */

   const char *filename;
   sc_weapon_config *wc;

   wc = (sc_weapon_config *)malloc(sizeof(sc_weapon_config));
   if(wc == NULL) return(NULL);

   wc->armslevel    = SC_ARMS_LEVEL_DEF;
   wc->bombiconsize = SC_WEAPON_BOMB_ICON_DEF;
   wc->tunneling    = false;
   wc->scaling      = 1.0;
   wc->tracepaths   = false;
   wc->uselessitems = true;
   wc->weaponlist   = NULL;

   wc->weaponlist = sc_weapon_list_new();
   if(wc->weaponlist == NULL) {
      free(wc);
      return(NULL);
   }

   /* Read in the weapon info list */
   filename = SC_GLOBAL_DIR "/" SC_WEAPON_FILE;
   if(!sc_weapon_append_file(wc, filename)) {
      /* This is the root weapon list...  Die! */
      sc_weapon_list_free(&wc->weaponlist);
      free(wc);
      return(NULL);
   }

   return(wc);

}



/***  Weapon Configuration File Support  ***/



/* Summary of the weapon state bits, for use in saddconf.c */
static const char *_sc_weapon_state_bit_names[] = {
   "digger",
   "roller",
   "triple",
   "riot",
   "beam",
   "dirt",
   "liquid",
   "plasma",
   "smoke",
   "defer",
   "sapper",
   NULL
};
static const int _sc_weapon_state_bit_items[] = {
   SC_WEAPON_STATE_DIGGER,
   SC_WEAPON_STATE_ROLLER,
   SC_WEAPON_STATE_TRIPLE,
   SC_WEAPON_STATE_RIOT,
   SC_WEAPON_STATE_BEAM,
   SC_WEAPON_STATE_DIRT,
   SC_WEAPON_STATE_LIQUID,
   SC_WEAPON_STATE_PLASMA,
   SC_WEAPON_STATE_SMOKE,
   SC_WEAPON_STATE_DEFER,
   SC_WEAPON_STATE_SAPPER,
   0
};



const char **sc_weapon_state_bit_names(void) {

   return(_sc_weapon_state_bit_names);

}



const int *sc_weapon_state_bit_items(void) {

   return(_sc_weapon_state_bit_items);

}
