/*

    xpuyopuyo - pnetconnect.c Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Network code to initiate and close connections
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <xpuyopuyo.h>
#if USE_NETWORK /* Allow network support? */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <pnetint.h>
#include <pconfig.h>
#include <pwindow.h>
#include <pconfigm.h>


static int pnet_set_nonblocking(int socket) {

   /* Setup the socket as nonblocking */
   return(fcntl(socket, F_SETFL, O_NONBLOCK));

}


void pnet_shutdown(int *socket) {

   /* Close the socket if it is a valid socket descriptor */
   if(socket != NULL && *socket >= 0) {
      shutdown(*socket, 2);   /* Perform a shutdown - disallow R/W */
      close(*socket);         /* Close the socket in the system */
      *socket = -1;           /* Invalidate the descriptor */
   } /* Only if the descriptor was valid ... */

}


void pnet_shutdown_all(psocket *s) {

   pnet_shutdown(&s->socket);
   pnet_shutdown(&s->remote);

}


void pnet_close(pconfig *c, psocket **s) {

   /* Only proceed if the pointers are valid */
   if(s == NULL || *s == NULL) return;

   /* Shutdown both associated sockets */
   pnet_shutdown_all(*s);
   
   if(!c->quiet) {
      printf("Network Statistics:\n");
      printf("   Bytes sent:  %d\n", (*s)->sent);
      printf("   Bytes recv:  %d\n\n", (*s)->recv);
   }
   
   /* Free and Nullify the pointer */
   free(*s);
   *s = NULL;
   
   /* Undo network game */
   p_config_undo_network(c);

}


psocket *pnet_server_new(pconfig *c, int port) {
   
   struct sockaddr_in socket_info;     /* Information about us */
   struct sockaddr remote_socket;      /* Information about client */
   int remote_socket_size;             /* Size of remote_socket */
   dword version;                      /* Protocol version */
   psocket *s;                         /* A new socket ... */
   int errors = 0;                     /* Nonzero on an error */

   /* Construct a new socket */
   s = (psocket *)malloc(sizeof(psocket));
   if(s == NULL) {
      pnet_set_error("server_new", "failed to allocate memory");
      return(NULL);
   }
   s->sent = 0;
   s->recv = 0;

   /* Setup the server socket */
   s->socket = socket(AF_INET, SOCK_STREAM, DEFAULT_PROTOCOL);
   if(s->socket < 0) {
      free(s);
      pnet_set_error("server_new", strerror(errno));
      return(NULL);
   }

   /* Setup the socket information about us; attempt to bind to a port */   
   bzero((char *)&socket_info, sizeof(socket_info));
   socket_info.sin_family = AF_INET;
   socket_info.sin_port = htons(port);
   if(bind(s->socket, (struct sockaddr *)&socket_info, sizeof(socket_info)) < 0) {
      free(s);
      pnet_set_error("server_new", strerror(errno));
      return(NULL);
   }

   /* Listen on that port for a client connection */   
   if(listen(s->socket, 0) < 0) {
      free(s);
      pnet_set_error("server_new", strerror(errno));
      return(NULL);
   }
   
   /* setup server socket not to block */
   pnet_set_nonblocking(P_COMM_SOCKET(s));
   
   /* Tell the user we are about to sit and wait ... */
   #ifdef P_NET_DEBUG_COMM
   printf("server:  Waiting for connection to port %d\n", port);
   #endif /* P_NET_DEBUG_COMM */
   
   p_window_set_status(c->window, "Waiting for connection to server ...");
   p_window_set_waiting(c->window, 1);
   p_window_idle(c->window);

   /* Loop until we get a valid connection */
   s->remote = accept(s->socket, &remote_socket, &remote_socket_size);
   while(s->remote < 0) {
      /* Idle for a minute */
      p_window_idle(c->window);

      /* Try again */
      s->remote = accept(s->socket, &remote_socket, &remote_socket_size);
   } /* While waiting for a valid connection ... */

   /* Close the server socket */
   pnet_shutdown(&s->socket);

   /* Now, we need to wait for the client to request a version */   
   p_window_set_status(c->window, "Checking protocol version number ...");
   p_window_idle(c->window);
   
   /* setup socket to client */
   #ifdef P_NET_NONBLOCKING_SOCKETS
      pnet_set_nonblocking(P_COMM_SOCKET(s));
   #endif /* Allow nonblocking client sockets? */
   
   
   /* Await version number request */
   if(pnet_recv_dword(c, s, &version) < 0) {
      errors = 1;
   } else { /* Client Version received */
      /* Confirm version code */
      if(version != P_NET_PROTO_VERSION) {
         pnet_set_error("server_new", "Protocol version mismatch");
         errors = 1;
      } else { /* Version is correct */
         /* Send version reply code */
         if(pnet_send_dword(c, s, P_NET_PROTO_VERSION) < 0) errors = 1;
      } /* Version agrees? */
   } /* Version received? */
      
   /* Clear messages; we are done */
   p_window_set_status(c->window, NULL);
   p_window_set_waiting(c->window, 0);
   
   /* Any errors occur through version negotiation? */
   if(errors != 0) {
      pnet_close(c, &s);
      return(NULL);
   }
   
   #ifdef P_NET_DEBUG_COMM
   printf("server:  Looks like we're communicating with protocol %d\n", version);
   #endif /* P_NET_DEBUG_COMM */
   
   /* Setup config */
   p_config_network(c);

   /* Return the new socket info */   
   return(s);
   
}


psocket *pnet_client_new(pconfig *c, const char *server, int port) {

   struct sockaddr_in socket_info;  /* Socket information to connect to remote */
   struct hostent *hostinfo;        /* Host information about remote server */
   dword version;                   /* Version code to send/received */
   psocket *s;                      /* New socket information */
   int errors = 0;                  /* Nonzero if an error occurred */

   /* Construct a new psocket */
   s = (psocket *)malloc(sizeof(psocket));
   if(s == NULL) {
      pnet_set_error("client_new", "failed to allocate psocket");
      return(NULL);
   }
   s->sent = 0;
   s->recv = 0;

   /* To identify us as a client */
   s->remote = -1;

   /* Get information about the remote host */
   hostinfo = gethostbyname(server);
   if(hostinfo == NULL) {
      free(s);
      pnet_set_error("client_new", "Cannot resolve hostname.");
      return(NULL);
   }

   /* Create a new socket to use */   
   s->socket = socket(AF_INET, SOCK_STREAM, DEFAULT_PROTOCOL);
   if(s->socket < 0) {
      free(s);
      pnet_set_error("client_new", strerror(errno));
      return(NULL);
   }

   /* Setup socket_info to contain info to connect to remote host */
   bzero((char *)&socket_info, sizeof(socket_info));
   bcopy(hostinfo->h_addr, (char *)&socket_info.sin_addr, hostinfo->h_length);
   socket_info.sin_family = hostinfo->h_addrtype;
   socket_info.sin_port = htons(port);

   /* Tell user we are about to connect to the remote host */
   #ifdef P_NET_DEBUG_COMM
   printf("client:  Attempting connection to server %s, port %d\n", server, port);
   #endif /* P_NET_DEBUG_COMM */
   
   p_window_set_status(c->window, "Attempting a connection to server ...");
   p_window_set_waiting(c->window, 1);
   p_window_idle(c->window);

   /* Attempt the connect */   
   if(connect(s->socket, (struct sockaddr *)&socket_info, sizeof(socket_info)) < 0) {
      errors = 1;
      pnet_set_error("client_new", strerror(errno));
   } else { /* Connect call succeeded */
      /* Tell user the connection is established */
      p_window_set_status(c->window, "Checking protocol version number ...");
      p_window_idle(c->window);
   
      /* setup socket */
      #ifdef P_NET_NONBLOCKING_SOCKETS
         pnet_set_nonblocking(P_COMM_SOCKET(s));
      #endif /* Allow nonblocking client sockets? */
   
   
      /* Send version number request */
      if(pnet_send_dword(c, s, P_NET_PROTO_VERSION) < 0) {
         errors = 1;
      } else { /* Sent our version request */
         /* Await version reply code */
         if(pnet_recv_dword(c, s, &version) < 0) {
            errors = 1;
         } else { /* Received version reply */
            /* Confirm version code */
            if(version != P_NET_PROTO_VERSION) {
               errors = 1;
               pnet_set_error("client_new", "Protocol version mismatch.");
            } /* Is the version acceptable? */
         } /* Did we receive a version reply? */
      } /* Successfully sent version? */
   } /* Connect succeed? */

   p_window_set_status(c->window, NULL);
   p_window_set_waiting(c->window, 0);
   
   /* Any errors through all that? */
   if(errors != 0) {
      pnet_close(c, &s);
      return(NULL);
   }

   /* We are done */
   #ifdef P_NET_DEBUG_COMM
   printf("client:  Looks like we're communicating with protocol %d\n", version);
   #endif /* P_NET_DEBUG_COMM */
   
   /* Setup config */
   p_config_network(c);
   
   /* Return socket information */   
   return(s);
   
}


#endif /* Allow network? */
