/*  This file is part of "xprintmon"
 *  Copyright (C) 2006 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/select.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <malloc.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <alloca.h>
#include <stdarg.h>

#include <X11/Intrinsic.h>

#include "global.h"

static struct child {
	struct child *next;
	XtInputId xid;
	pid_t pid;
	int fd;
	int status;
	char *buffer;
	size_t len;
	size_t size;
	collectorcallback *callback;
	void *privdata;
	bool deleted;
} *children;


static void terminateChild(struct child *child, const char *fmt, ...)
	__attribute__ ((format (printf, 2, 3)));
static void terminateChild(struct child *child, const char *fmt, ...) {

	if( fmt != NULL ) {
		va_list ap;
		char buffer[100]; char *p = buffer;
		int l;

		if( child->len > 0 && child->buffer[child->len-1] != '\n')
			*(p++) = '\n';
		strcpy(p, "XPRINTMON: ");
		p+=11;

		va_start(ap, fmt);
		l = vsnprintf(p, sizeof(buffer)-(p-buffer), fmt, ap);
		assert( l >= 0 );
		l += p-buffer;
		assert( (size_t)l < sizeof(buffer) );
		if( child->size < child->len+l+1 ) {
			char *n = realloc(child->buffer, child->len+l+1);
			if( n != NULL ) {
				child->size = child->len+l+1;
				child->buffer = n;
			}
		}
		if( child->size >= child->len+l+1 ) {
			memcpy(child->buffer+child->len, buffer, l+1);
			child->len += l;
		}
	}

	if( child->callback != NULL)
		child->callback(child->privdata, child->buffer, child->len, true);
}

static void processTerminatedChild(struct child *child) {
	assert(child->pid == -1);
	assert(child->fd == -1);

	if( WIFEXITED(child->status) ) {
		if( WEXITSTATUS(child->status) == 0 ) {
			terminateChild(child, "child returned successfully\n");
		} else {
			terminateChild(child, "child exited with errorcode %d\n",
					(int)(WEXITSTATUS(child->status)));
		}
	} else if( WIFSIGNALED(child->status) ) {
		if( WTERMSIG(child->status) == SIGUSR2 && child->buffer != NULL &&
			strncmp(child->buffer, "XPRINTMON:", 10) == 0) {
			terminateChild(child, NULL);
		} else {
			int signum = WTERMSIG(child->status);
			static const char * const signames[30] = {
				"???", "HUP", "INT", "QUIT", "ILL",
				"TRAP", "ABRT", "UNUSED", "FPE",
				"KILL", "USR1", "SEGV", "USR2",
				"PIPE", "ALRM", "TERM", "STKFLT",
				"CHLD", "CONT", "STOP", "TSTP",
				"TTIN", "TTOU", "IO", "XCPU",
				"XFSZ", "VTALRM", "PROF", "???",
				"WINCH"};

			// TODO: names but numbers...
			terminateChild(child, "child killed by signal SIG%s (%d)\n",
					signames[(signum>29||signum<0)?0:signum],
					signum);
		}
	} else {
		terminateChild(child, "unrecognized termination of child\n");
	}
}

static void processChild(XtPointer data, int *source, XtInputId *id) {
	struct child *child = data;
	ssize_t got;

	assert( child->fd == *source );
	assert( child->xid == *id );
	if( child->fd < 0 )
		return;
	if( child->size - child->len < 256 ) {
		char *n = realloc(child->buffer, child->size+2048);
		if( n == NULL )
			return;
		child->buffer = n;
		child->size+=2048;
	}

	got = read(child->fd, child->buffer+child->len,
			child->size-1-child->len);
	if( got == 0 ) {
		XtRemoveInput(child->xid);
		close(child->fd);
		child->fd = -1;

		if( child->pid < 0 )
			processTerminatedChild(child);
		return;
	} else if( got < 0 ) {
		int e = errno;
		if( e == EAGAIN || e == EINTR )
			return;
		XtRemoveInput(child->xid);
		close(child->fd);
		child->fd = -1;
		fprintf(stderr, "Error reading from child: %s",
				strerror(e));
		if( child->pid > 0 )
			kill(child->pid, SIGPIPE);
		return;
	}
	while( got > 0 ) {
		if( child->buffer[child->len] == '\0' )
			child->buffer[child->len] = '~';
		child->len++;
		got--;
	}
	child->buffer[child->len] = '\0';
	if( child->callback != NULL)
		child->callback(child->privdata, child->buffer, child->len, false);
}

int null_fd = -1;

void abortCollector(struct child *child) {
	assert( child != NULL);
	if( child->fd >= 0 ) {
		XtRemoveInput(child->xid);
		close(child->fd);
		child->fd = -1;
	}
	child->callback = NULL;
	child->privdata = NULL;
	child->deleted = true;
}
void killCollector(struct child *child) {
	assert( child != NULL);
	if( child->pid > 0 ) {
		kill(child->pid, SIGABRT);
	}
}

struct child *newCollector(XtAppContext app, collectorcallback callback, void *frame, const char *command, char **const args) {
	struct child *child;
	int filedes[2];
	int e;
	pid_t pid;

	child = malloc(sizeof(struct child));
	if( child == NULL )
		return NULL;
	child->next = children;
	child->buffer = NULL;
	child->len = 0;
	child->size = 0;
	child->status = 0;
	child->callback = callback;
	child->privdata = frame;
	child->deleted = false;

	assert(null_fd > 0);

	e = pipe(filedes);
	if( e != 0 ) {
		fprintf(stderr, "Error creating pipe: %s\n", strerror(errno));
		free(child);
		return NULL;
	}
	closeonexec(filedes[0]);
	closeonexec(filedes[1]);
	child->fd = filedes[0];

	pid = fork();
	if( pid < 0 ) { /* error */
		fprintf(stderr,"XPRINTMON: error forking: %s\n",
				strerror(errno));
		return NULL;
	} else if( pid == 0) { /* child */
		e = dup2(null_fd, 0);
		if( e != 0 ) {
			fprintf(stderr,"XPRINTMON: error calling dup2(%d,0): %s\n",
				null_fd, strerror(errno));
			raise(SIGUSR2);
			exit(EXIT_FAILURE);
		}
		e = dup2(filedes[1], 1);
		if( e != 1 ) {
			fprintf(stderr,"XPRINTMON: error calling dup2(%d,1): %s\n",
				filedes[1], strerror(errno));
			raise(SIGUSR2);
			exit(EXIT_FAILURE);
		}
		e = dup2(filedes[1], 2);
		if( e != 2 ) {
			fprintf(stderr,"XPRINTMON: error calling dup2(%d,2): %s\n",
				filedes[1], strerror(errno));
			raise(SIGUSR2);
			exit(EXIT_FAILURE);
		}
		close(filedes[0]);
		close(filedes[1]);
		e = execvp(command, args);
		fprintf(stderr,"XPRINTMON: error calling exec: %s\n",
				strerror(errno));
		raise(SIGUSR2);
		exit(EXIT_FAILURE);
	}
	close(filedes[1]);
	child->pid = pid;
	child->xid = XtAppAddInput(app, child->fd,
			(XtPointer)(XtInputReadMask|XtInputExceptMask),
			processChild, child);
	children = child;
	return child;
}

struct child *newCollectorVa(XtAppContext app, collectorcallback callback, void *frame, const char *command, ...) {
	va_list ap;
	size_t count = 0;
	const char **arguments, **p, *a;

	va_start(ap, command);
	while( va_arg(ap, char*) != NULL )
		count++;
	va_end(ap);
	arguments = alloca(sizeof(char*)*(count+2));
	p = arguments;
	*(p++) = command;
	va_start(ap, command);
	while( (a = va_arg(ap, char*)) != NULL )
		*(p++) = a;
	*p = NULL;
	assert( (size_t)(p-arguments) == count+1);
	return newCollector(app, callback, frame, command, (char**)arguments);
}

bool closeonexec(int fd) {
	int e = fcntl(fd, F_SETFD, FD_CLOEXEC);
	if( e < 0 ) {
		e = errno;
		fprintf(stderr, "Cannot mark fd %d as close-on-exec: %s\n",
				fd, strerror(e));
		return false;
	} else
		return true;
}
static bool unblock(int fd) {
	int e;

	e = fcntl(fd, F_GETFL);
	if( e < 0 ) {
		fprintf(stderr, "Error getting file status flags for fd %d: %s\n",
				fd, strerror(errno));
		return false;
	}
	e = fcntl(fd, F_SETFL, O_NONBLOCK|e);
	if( e < 0 ) {
		fprintf(stderr, "Error setting fd %d to non-blocking: %s\n",
				fd, strerror(errno));
		return false;
	}
	return true;
}

/* Ugly, but it works. when getting ECHILD write a byte to the pipe,
 * so the select call catches it. No races, no blocks */
int childpipe[2];

#define DEBUG 1

static void processChildSignal(XtPointer data UNUSED, int *source UNUSED, XtInputId *id UNUSED) {
	pid_t pid;
	struct child *child, **p_child;
	int status;
	char buffer[1];

	pid = 1;
	while( read(childpipe[0], buffer, 1) > 0 ) {
		pid = waitpid(-1, &status, WNOHANG);
		if( pid <= 0 )
			break;
		for( child = children ; child ; child = child->next ) {
			if( child->pid == pid ) {
				child->pid = -1;
				child->status = status;
				if( child->fd < 0 )
					processTerminatedChild(child);
				break;
			}
		}
	}
	if( pid < 0 && errno != ECHILD ) {
		fprintf(stderr, "Error locking for children (waitpid): %s\n",
				strerror(errno));
#ifdef DEBUG
	} else if( pid <= 0) {
		fprintf(stderr, "Missed Child?\n");
#endif
	}
	for( p_child = &children ; (child = *p_child) != NULL ;
			p_child = &child->next ) {
		if( child->deleted && child->pid < 0 && child->fd < 0 ) {
			free(child->buffer);
			*p_child = child->next;
			free(child);
			child = *p_child;
			if( child == NULL )
				break;
		}
	}
}


void initCollectors(void) {
	int e;
	do {
		null_fd = open("/dev/null", O_WRONLY|O_NOCTTY|O_APPEND);
	} while( null_fd >= 0 && null_fd <= 2 );

	if( null_fd < 0 ) {
		fprintf(stderr, "Error opening /dev/null: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}
	if( !closeonexec(null_fd) )
		exit(EXIT_FAILURE);
	e = pipe(childpipe);
	if( e != 0 ) {
		fprintf(stderr, "Error creating child-status pipe: %s\n",
				strerror(errno));
		exit(EXIT_FAILURE);
	}
	if( !closeonexec(childpipe[0]) || !unblock(childpipe[0]))
		exit(EXIT_FAILURE);
	if( !closeonexec(childpipe[1]) || !unblock(childpipe[1]))
		exit(EXIT_FAILURE);
}

static void catchECHLD(int dummy UNUSED) {
	char c = 'B';
	write(childpipe[1], &c, 1);
}

void registerCollectors(XtAppContext app) {
	struct sigaction action;
	action.sa_handler = catchECHLD;
	action.sa_flags = SA_NOCLDSTOP;
	sigemptyset(&action.sa_mask);
	sigaction(SIGCHLD, &action, NULL);

	XtAppAddInput(app, childpipe[0], (XtPointer)(XtInputReadMask|XtInputExceptMask),
			processChildSignal, NULL);
}
