/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_kiviat.c,v 1.2 1999/11/08 06:20:25 bbarrett Exp $
 * 
 *	Function:	- Kiviat subwindow management functions
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>

#define _NO_PROTO

#include <Xm/DialogS.h>
#include <Xm/DrawingA.h>

#include "xmpi.h"

/*
 * global functions
 */
void			xmpi_kv_popup();
void			xmpi_kv_popdown();
void			xmpi_kv_update();
void			xmpi_kv_busy();
void			xmpi_kv_unbusy();

/*
 * local functions
 */
static void		kv_create();
static void		popdown_cb();
static void		popup_cb();
static void		expose_resize_cb();
static void		resize();
static void		gcinit();
static void		plot();

/*
 * local variables
 */
static Widget		kiviat_w = 0;		/* kiviat window */
static Widget		draw_w;			/* kiviat drawing area */
static Pixmap		pixmap;			/* kiviat pixmap */
static Dimension	drawsize;		/* drawing area size */
static Dimension	kv_xheight;		/* extra height pixels */
static Dimension	kv_xwidth;		/* extra width pixels */
static GC		fgpen = 0;		/* foreground pen */
static GC		bgpen = 0;		/* background pen */
static GC		blkpen = 0;		/* blocked process pen */
static GC		syspen = 0;		/* system overhead pen */
static GC		runpen = 0;		/* running process pen */
static Display		*disp;			/* drawing area display */
static Drawable		root;			/* root window */
static unsigned int	depth;			/* screen depth */
static int		fl_position = 1;	/* kiviat position flag */
static int		fl_up = 0;		/* kiviat popped up flag */
static int		kv_nprocs = -1;		/* # processes */
static double		*kv_cosine = 0;		/* processes cos(angle) */
static double		*kv_sine = 0;		/* processes sin(angle) */
static double		*kv_block = 0;		/* % blocking times */
static double		*kv_system = 0;		/* % sys. overhead times */

/*
 *	xmpi_kv_popup
 *
 *	Function:	- popup the Kiviat diagram window
 *			- create it if necessary
 */
void
xmpi_kv_popup()

{
/*
 * If Kiviat already created pop it up else create it.
 */
	if (!kiviat_w) {
		kv_create();
	}

	XtPopup(kiviat_w, XtGrabNone);
}

/*
 *	xmpi_kv_popdown
 *
 *	Function:	- popdown the Kiviat diagram window
 */
void
xmpi_kv_popdown()

{
	if (kiviat_w) {
		XtPopdown(kiviat_w);
	}
}

/*
 *	kv_create
 *
 *	Function:	- create the Kiviat diagram window
 */
static void
kv_create()

{
	Dimension	height, width;		/* popup dimensions */
/*
 * Create the Kiviat window.
 */
	kiviat_w = XtVaCreatePopupShell("kiviat_pop",
			xmDialogShellWidgetClass, xmpi_shell,
			XmNtitle, "XMPI Kiviat",
			XmNallowShellResize, True,
			XmNdeleteResponse, XmUNMAP,
                	XmNmappedWhenManaged, False,
			NULL);
/*
 * Set Kiviat window callbacks.
 */
	XtAddCallback(kiviat_w, XmNpopupCallback, popup_cb, NULL);
	XtAddCallback(kiviat_w, XmNpopdownCallback, popdown_cb, NULL);
/*
 * Create the Kiviat drawing area.
 */
	drawsize = XMPI_KVDEFSIZE;

	draw_w = XtVaCreateManagedWidget("kiviat_draw",
			xmDrawingAreaWidgetClass, kiviat_w,
			XmNwidth, drawsize,
			XmNheight, drawsize,
			NULL);

	XtAddCallback(draw_w, XmNexposeCallback, expose_resize_cb, NULL);
	XtAddCallback(draw_w, XmNresizeCallback, expose_resize_cb, NULL);
/*
 * Initialize graphic contexts.
 */
	gcinit();
/*
 * Create the Kiviat pixmap.
 */
	pixmap = XCreatePixmap(disp, root, drawsize, drawsize, depth);
/*
 * Compute the extra pixels between drawing area and popup.
 * These are used when resizing.
 */
	XtVaGetValues(kiviat_w, XmNheight, &height, XmNwidth, &width, NULL);

	kv_xheight = height - drawsize;
	kv_xwidth = width - drawsize;
}

/*
 *	popdown_cb
 *
 *	Function:	- Kiviat popup/unmap callback
 */
static void
popdown_cb()

{
	fl_up = 0;
}

/*
 *	popup_cb
 *
 *	Function:	- Kiviat popup positioning callback
 *			- center window only after its creation
 *	Accepts:	- widget
 */
static void
popup_cb(w)

Widget			w;

{
	Position	x, y;			/* parent coordinates */
	Dimension	width, height;		/* parent dimensions */
	Dimension	ww, wh;			/* widget dimensions */

	fl_up = 1;
	xmpi_kv_update();

	XtManageChild(draw_w);

	if (!fl_position) return;

	XtVaGetValues(w, XmNwidth, &ww, XmNheight, &wh, NULL);

	XtVaGetValues(XtParent(w), XmNx, &x, XmNy, &y,
			XmNwidth, &width, XmNheight, &height, NULL);

	XtVaSetValues(w, XmNx, x + ((width - ww) / 2),
			XmNy, y + ((height - wh) / 2), NULL);

	fl_position = 0;
}

/*
 *	expose_resize_cb
 *
 *	Function:	- callback function for expose/resize events
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
expose_resize_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmDrawingAreaCallbackStruct
			*p_callback;

{
	XEvent		*event;

	event = p_callback->event;

	if (p_callback->reason == XmCR_EXPOSE) {
		XCopyArea(event->xany.display, pixmap, event->xany.window,
				fgpen, 0, 0, drawsize, drawsize, 0, 0);
	}
	else if (p_callback->reason == XmCR_RESIZE) {
		resize();
	}
}

/*
 *	resize
 *
 *	Function:	- resize the Kiviat window
 */
static void
resize()

{
	Dimension	height, width;		/* window sizes */
/*
 * Force it to be a square of the smallest dimension.
 */
	XtVaGetValues(draw_w, XmNheight, &height, XmNwidth, &width, NULL);
	drawsize = (width > height) ? height : width;
/*
 * Change the popup dimensions to match.
 */
	XtVaSetValues(kiviat_w, XmNheight, drawsize + kv_xheight,
				XmNwidth, drawsize + kv_xwidth, NULL);
/*
 * Create a new pixmap and re-plot.
 */
	XFreePixmap(disp, pixmap);
	pixmap = XCreatePixmap(disp, root, drawsize, drawsize, depth);

	plot();
}

/*
 *	gcinit
 *
 *	Function:	- allocate graphic contexts
 */
static void
gcinit()

{
	Pixel		fg;			/* foreground colour */
	XGCValues	gcval;			/* graphic context info */
	XColor		col_def;		/* colour definition */
	XColor		dev_null;		/* unused functionality */

	if (fgpen) return;

	disp = XtDisplay(draw_w);
	root = RootWindowOfScreen(XtScreen(draw_w));
	depth = DefaultDepthOfScreen(XtScreen(draw_w));
/*
 * Create foreground pen.
 */
	XtVaGetValues(draw_w, XmNforeground, &fg, NULL);

	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
				XMPI_LBLCOLOUR, &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	gcval.background = fg;
	gcval.font = app_res.ap_rankfont->fid;

	fgpen = XCreateGC(disp, root,
			(GCFont | GCForeground | GCBackground), &gcval);
/*
 * Create background pen.
 */
	gcval.foreground = fg;
	bgpen = XCreateGC(disp, root, GCForeground, &gcval);
/*
 * Create system overhead pen.
 */
	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
				XMPI_SYSCOLOUR, &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	syspen = XCreateGC(disp, root, GCForeground, &gcval);
/*
 * Create blocked process pen.
 */
	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
				XMPI_BLKCOLOUR, &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	blkpen = XCreateGC(disp, root, GCForeground, &gcval);
/*
 * Create running process pen.
 */
	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
				XMPI_RUNCOLOUR, &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	runpen = XCreateGC(disp, root, GCForeground, &gcval);
}

/*
 *	xmpi_kv_update
 *
 *	Function:	- get Kiviat info from dbase and plot it
 */
void
xmpi_kv_update()

{
	int		i;			/* favourite index */
	int		n;			/* # processes */
	double		angle;			/* text angle position */
	double		delta;			/* angle increment */

	if (!fl_up) return;
/*
 * If # processes changed, reallocate information storage space.
 */
	n = xmpi_db_getnprocs();

	if (n == 0) return;
	
	if (n != kv_nprocs) {

		kv_nprocs = n;

		if (kv_block) free((char *) kv_block);
		kv_block = (double *) malloc((unsigned)
					4 * kv_nprocs * sizeof(double));
		if (kv_block == 0) {
			kv_nprocs = 0;
			return;
		}

		kv_system = kv_block + kv_nprocs;
		kv_cosine = kv_system + kv_nprocs;
		kv_sine = kv_cosine + kv_nprocs;
/*
 * Compute sin/cos of angles (location is mid-slices).
 */
		delta = (2.0 * PI) / kv_nprocs;
		angle = delta / 2.0;

		for (i = 0; i < kv_nprocs; ++i, angle += delta) {
			kv_cosine[i] = cos(angle);
			kv_sine[i] = sin(angle);
		}
	}
/*
 * Get load information from database and plot Kiviat.
 */
	xmpi_db_getload(kv_nprocs, kv_block, kv_system);

	plot();
}

/*
 *	plot
 *
 *	Function:	- plot current Kiviat info
 */
static void
plot()

{
	int		center;			/* circle center */
	int		radius;			/* circle radius */
	int		spoke;			/* spoke length */
	int		x, y;			/* coordinates */
	int		i;			/* favourite index */
	int		dir;			/* string direction */
	int		ascent;			/* font ascent */
	int		descent;		/* font descent */
	int		fmtlen;			/* string length */
	char		fmtbuf[8];		/* formatting buffer */
	XCharStruct	fmtinfo;		/* string font info */

	if (!fl_up) return;
/*
 * Determine circle center and radius.
 * Clear the screen.
 */
	center = drawsize / 2;
	radius = center - XMPI_KVBORDER;

	XFillRectangle(disp, pixmap, bgpen, 0, 0, drawsize, drawsize);
/*
 * Plot the process ranks.
 */
	spoke = radius + XMPI_KVTEXT;

	for (i = 0; i < kv_nprocs; ++i) {

		x = center + (int) (spoke * kv_cosine[i]);
		y = center - (int) (spoke * kv_sine[i]);

		sprintf(fmtbuf, "%d", i);
		fmtlen = strlen(fmtbuf);

		XTextExtents(app_res.ap_rankfont, fmtbuf, fmtlen,
				&dir, &ascent, &descent, &fmtinfo);

		XDrawString(disp, pixmap, fgpen, x - (fmtinfo.width / 2),
				y + (fmtinfo.ascent / 2), fmtbuf, fmtlen);
	}
/*
 * Loop over processes plotting the pie slices.
 * Angles are in 1/64th of a degree (i.e. 23040 = 360 * 64).
 */
	x = 0;
	y = 23040 / kv_nprocs;

	for (i = 0; i < kv_nprocs; ++i, x += y) {

		if (kv_block[i] < 0) continue;
/*
 * Plot blocked slice.
 */
		XFillArc(disp, pixmap, blkpen,
				center - radius, center - radius,
				radius + radius, radius + radius, x, y);
/*
 * Plot system slice.
 */
		spoke = (int) ((double) radius * (1.0 - kv_block[i]));

		XFillArc(disp, pixmap, syspen,
				center - spoke, center - spoke,
				spoke + spoke, spoke + spoke, x, y);
/*
 * Plot user slice.
 */
		spoke = (int) ((double) radius
					* (1.0 - kv_block[i] - kv_system[i]));

		XFillArc(disp, pixmap, runpen,
				center - spoke, center - spoke,
				spoke + spoke, spoke + spoke, x, y);
	}
/*
 * Refresh the screen.
 */
	XCopyArea(disp, pixmap, XtWindow(draw_w), fgpen,
				0, 0, drawsize, drawsize, 0, 0);
}

/*
 *	xmpi_kv_busy
 *
 *	Function:	- set the busy cursor for the Kiviat window
 */
void
xmpi_kv_busy()

{
	if (kiviat_w) xmpi_busy_widget(kiviat_w);
}

/*
 *	xmpi_kv_unbusy
 *
 *	Function:	- set the normal cursor for the Kiviat window
 */
void
xmpi_kv_unbusy()

{
	if (kiviat_w) xmpi_unbusy_widget(kiviat_w);
}
