/*
 *	$COPYRIGHT$
 *
 *	$Id: xmpi_matrix.cc,v 1.4 2000/10/29 19:32:02 bbarrett Exp $
 *
 *	Function:	- message matrix subwindow management functions
 */

#include <stdlib.h>
#include <string.h>

#include <Xm/DialogS.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/ScrolledW.h>

#include "xmpi.h"
#include "xmpi_view.h"
#include "xmpi_misc.h"
#include "xmpi_matrix.h"

#include "Bitmaps/src1.xbm"
#include "Bitmaps/src2.xbm"
#include "Bitmaps/src3.xbm"
#include "Bitmaps/src4.xbm"
#include "Bitmaps/src5.xbm"
#include "Bitmaps/src6.xbm"
#include "Bitmaps/src7.xbm"
#include "Bitmaps/srcx.xbm"


/*
 * external variables
 */
extern struct _gps *xmpi_app_procs;    /* appl. GPS array */

/*
 * local functions
 */
static void destroy_cb();

static void expose_cb(Widget, XtPointer, 
		      XmDrawingAreaCallbackStruct*);

static void gcinit();

static void mat_clear();

static void mat_create();

static void mat_plot_grid();

static void popdown_cb(Widget);

static void popup_cb(Widget);

/*
 * local variables
 */
static Widget draw_w;		       /* matrix drawing area */

static Widget matrix_w = 0;	       /* matrix window */

static Widget mgr_w;		       /* matrix manager */

static Widget scroll_w;		       /* scrolled window */

static Pixmap pixmap;		       /* matrix pixmap */

static Pixmap src_pix[XMPI_MTCNTMAX + 1];

 /* msg source count pixmaps */
static Dimension bordersize;	       /* space around grid */

static Dimension drawsize;	       /* drawing area size */

static GC lblpen = 0;		       /* labelling pen */

static GC bgpen = 0;		       /* background pen */

static GC gridpen = 0;		       /* grid line pen */

static Display *disp;		       /* drawing area display */

static Drawable root;		       /* root window */

static unsigned int depth;	       /* screen depth */

static int fl_up = 0;		       /* matrix popped up flag */

static int mat_nprocs = 0;	       /* # processes */

static int *mat_src;		       /* message source count */

const char *src_colours[XMPI_MTCNTMAX + 1] = {
  XMPI_RUNCOLOUR,
  XMPI_SYSCOLOUR,
  XMPI_BLKCOLOUR,
  XMPI_RCOMCOLOUR,
  XMPI_LCOMCOLOUR,
  XMPI_SYSCOLOUR,
  XMPI_RCOMCOLOUR,
  XMPI_LBLCOLOUR
};

static unsigned char *src_bits[XMPI_MTCNTMAX + 1] = {
  src1_bits,
  src2_bits,
  src3_bits,
  src4_bits,
  src5_bits,
  src6_bits,
  src7_bits,
  srcx_bits
};

/*
 *	xmpi_mat_popup
 *
 *	Function:	- popup the message matrix window
 *			- creates it if necessary
 */
void
xmpi_mat_popup()
{
  struct xmproc *mat_procs;	       /* process status array */

/*
 * If matrix already created pop it up else create it.
 */
  if (!matrix_w) {
    mat_nprocs = xmpi_vw_getnprocs();
    mat_procs = xmpi_vw_getprocs();

    if (mat_nprocs <= 0)
      return;

    mat_src = (int *) malloc((unsigned) (mat_nprocs *
					 sizeof(int)));
    if (mat_src == 0)
      xmpi_fail((char*) "xmpi (malloc)");

    mat_create();
    fl_up = 1;
    xmpi_mat_update(mat_procs);
  }
  XtPopup(matrix_w, XtGrabNone);
}

/*
 *	mat_create
 *
 *	Function:	- creates the message matrix window
 */
static void
mat_create()
{
  Dimension minscrollsize;	       /* min scroller resize */

  Dimension scrollsize;		       /* initial scroller size */

  if (mat_nprocs < XMPI_MTMINPROC) {
    bordersize = (Dimension) (((XMPI_MTMINPROC - mat_nprocs) / 2.0) *
		  XMPI_MTCELLSIZE) + XMPI_MTCELLSIZE;
  } else {
    bordersize = 2 * XMPI_MTCELLSIZE;
  }

  drawsize = (mat_nprocs * XMPI_MTCELLSIZE) + (2 * bordersize);
  minscrollsize = (XMPI_MTMINPROC * XMPI_MTCELLSIZE) +
    (2 * XMPI_MTCELLSIZE) + XMPI_VWXWIDTH;
  scrollsize = (((XMPI_MTMAXPROC < mat_nprocs) ?
		 XMPI_MTMAXPROC : mat_nprocs) *
		XMPI_MTCELLSIZE) + (2 * bordersize) + XMPI_VWXWIDTH;
/*
 * Create the matrix window.
 */
  matrix_w = XtVaCreatePopupShell("matrix_pop",
				  xmDialogShellWidgetClass, xmpi_shell,
				  XmNtitle, "XMPI Message Matrix",
				  XmNmappedWhenManaged, False,
				  XmNmaxWidth, drawsize + XMPI_VWXWIDTH,
				  XmNmaxHeight, drawsize + XMPI_VWXHEIGHT,
				  XmNminWidth, minscrollsize,
				  XmNminHeight, minscrollsize,
				  NULL);
/*
 * Set matrix window callbacks.
 */
  XtAddCallback(matrix_w, XmNpopupCallback, 
		(XtCallbackProc) popup_cb, NULL);
  XtAddCallback(matrix_w, XmNpopdownCallback, 
		(XtCallbackProc) popdown_cb, NULL);
  XtAddCallback(matrix_w, XmNdestroyCallback, 
		(XtCallbackProc) destroy_cb, NULL);

  mgr_w = XtVaCreateWidget("matrix_mgr",
			   xmFormWidgetClass, matrix_w,
			   NULL);
/*
 * Build the scrolled window.
 */
  scroll_w = XtVaCreateManagedWidget("matrix_view",
				     xmScrolledWindowWidgetClass, mgr_w,
				     XmNscrollingPolicy, XmAUTOMATIC,
				     XmNheight, scrollsize,
				     XmNwidth, scrollsize,
				     XmNspacing, 0,
				     XmNrightAttachment, XmATTACH_FORM,
				     XmNleftAttachment, XmATTACH_FORM,
				     XmNtopAttachment, XmATTACH_FORM,
				     XmNbottomAttachment, XmATTACH_FORM,
				     NULL);
/*
 * Create the matrix drawing area.
 */
  draw_w = XtVaCreateManagedWidget("matrix_draw",
				   xmDrawingAreaWidgetClass, scroll_w,
				   XmNwidth, drawsize,
				   XmNheight, drawsize,
				   NULL);

  XtAddCallback(draw_w, XmNexposeCallback, 
		(XtCallbackProc) expose_cb, NULL);

  XtManageChild(mgr_w);
/*
 * Initialize graphic contexts.
 */
  gcinit();
/*
 * Create the matrix pixmap.
 */
  pixmap = XCreatePixmap(disp, root, drawsize, drawsize, depth);
}

/*
 *	xmpi_mat_destroy
 *
 *	Function:	- destroys matrix window
 */
void
xmpi_mat_destroy()
{
  Widget w;

  if (matrix_w) {
    XtUnmapWidget(matrix_w);
    w = matrix_w;
    mat_clear();
    XtDestroyWidget(w);
    fl_up = 0;
  }
}

/*
 *	destroy_cb
 *
 *	Function:	- matrix window destroy callback
 */
static void
destroy_cb()
{
  mat_clear();
}

/*
 *	mat_clear
 *
 *	Function:	- cleans up resources associated with matrix window
 */
static void
mat_clear()
{
  int i;

  if (!matrix_w)
    return;
/*
 * Flushing the dispay here eliminates a BadGC (invalid GC parameter)
 * error that occurs in some cases.  I am not quite sure what the exact
 * cause of the error is but I suspect it is some attempt to update the
 * matrix window after the GCs have been freed.
 */
  XmUpdateDisplay(matrix_w);
  matrix_w = 0;
/*
 * Free information storage space.
 */
  free(mat_src);
/*
 * Destroy pixmaps and pens.
 */
  for (i = 0; i <= XMPI_MTCNTMAX; ++i) {
    XFreePixmap(disp, src_pix[i]);
  }

  XFreePixmap(disp, pixmap);
  XFreeGC(disp, lblpen);
  XFreeGC(disp, bgpen);
  XFreeGC(disp, gridpen);
}

/*
 *	popup_cb
 *
 *	Function:	- matrix popup callback
 *	Accepts:	- widget
 */
static void
popup_cb(Widget)
{
  fl_up = 1;
  XtManageChild(mgr_w);
}

/*
 *	popdown_cb
 *
 *	Function:	- matrix popdown callback
 *	Accepts:	- widget
 */
static void
popdown_cb(Widget)
{
  fl_up = 0;
}

/*
 *	expose_cb
 *
 *	Function:	- callback function for expose events
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
expose_cb(Widget, XtPointer, 
	  XmDrawingAreaCallbackStruct *p_callback)
{
  XEvent *event;

  event = p_callback->event;

  XCopyArea(event->xany.display, pixmap, event->xany.window,
	    lblpen, 0, 0, drawsize, drawsize, 0, 0);
}

/*
 *	gcinit
 *
 *	Function:	- allocate graphic contexts
 */
static void
gcinit()
{
  Pixel fg, bg;			       /* fore/background colours */

  XGCValues gcval;		       /* graphic context info */

  XColor col_def;		       /* colour definition */

  XColor dev_null;		       /* unused functionality */

  Pixmap bm;			       /* source count bitmap */

  GC pen;			       /* src cnt bitmap GC */

  int i;

  disp = XtDisplay(draw_w);
  root = RootWindowOfScreen(XtScreen(draw_w));
  depth = DefaultDepthOfScreen(XtScreen(draw_w));

  XtVaGetValues(draw_w, XmNforeground, &fg, XmNbackground, &bg, NULL);
/*
 * Create the label, grid and background pens.
 */
  gcval.line_width = XMPI_MTLNWIDTH;
  gcval.foreground = bg;
  gcval.background = fg;

  gridpen = XCreateGC(disp, root,
		      GCLineWidth | GCForeground | GCBackground, &gcval);

  gcval.foreground = fg;
  bgpen = XCreateGC(disp, root, GCForeground, &gcval);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
		   XMPI_LBLCOLOUR, &col_def, &dev_null);

  gcval.foreground = col_def.pixel;
  gcval.background = fg;
  gcval.font = app_res.ap_rankfont->fid;
  lblpen = XCreateGC(disp, root,
		     GCFont | GCForeground | GCBackground, &gcval);
/*
 * Create the source message count pixmaps.
 */
  for (i = 0; i <= XMPI_MTCNTMAX; ++i) {
    bm = XCreateBitmapFromData(disp, root,
		    (const char *) src_bits[i], src1_width, src1_height);

    XAllocNamedColor(disp,
		     DefaultColormapOfScreen(XtScreen(draw_w)),
		     src_colours[i], &col_def, &dev_null);

    gcval.foreground = col_def.pixel;
    gcval.background = fg;
    gcval.clip_mask = bm;
    pen = XCreateGC(disp, root,
		    GCForeground | GCBackground | GCClipMask,
		    &gcval);

    src_pix[i] = XCreatePixmapFromBitmapData(disp, root,
		   (char *) src_bits[i], src1_width, src1_height, bg, fg,
					     depth);

    XCopyPlane(disp, bm, src_pix[i], pen, 0, 0,
	       src1_width, src1_height, 0, 0,
	       (unsigned long) 1);

    XFreePixmap(disp, bm);
    XFreeGC(disp, pen);
  }
}

/*
 *	xmpi_mat_update
 *
 *	Function:	- get matrix info from dbase and plot it
 *	Accepts:	- process status information
 */
void
xmpi_mat_update(struct xmproc *mat_procs)
{
  int i, j;

  Dimension x, y;		       /* coordinates */

  Pixmap pix;			       /* source count pixmap */

  struct xmmsg *p;		       /* message aggregate */

  if (!fl_up)
    return;
/*
 * Clear the screen.
 */
  XFillRectangle(disp, pixmap, bgpen, 0, 0, drawsize, drawsize);
  mat_plot_grid();

  y = bordersize;

  for (i = 0; i < mat_nprocs; ++i) {

    if (!mat_procs[i].xmp_msgs) {
      y += XMPI_MTCELLSIZE;
      continue;
    }
    for (j = 0; j < mat_nprocs; ++j) {
      mat_src[j] = 0;
    }

    p = (xmmsg*) al_top(mat_procs[i].xmp_msgs);

    while (p) {

      if (p->xmm_nmsg > 0) {
	mat_src[p->xmm_gsrc] += p->xmm_nmsg;
      }
      p = (xmmsg*) al_next(mat_procs[i].xmp_msgs, p);
    }

    x = bordersize;

    for (j = 0; j < mat_nprocs; ++j) {

      if (mat_src[j] == 0) {
	x += XMPI_MTCELLSIZE;
	continue;
      }
      if (mat_src[j] > XMPI_MTCNTMAX) {
	pix = src_pix[XMPI_MTCNTMAX];
      }
      pix = src_pix[mat_src[j] - 1];

      XCopyArea(disp, pix, pixmap, bgpen,
		0, 0, src1_width, src1_height,
		x + XMPI_MTCNTINSET,
		y + XMPI_MTCNTINSET);

      x += XMPI_MTCELLSIZE;
    }

    y += XMPI_MTCELLSIZE;
  }

  XCopyArea(disp, pixmap, XtWindow(draw_w), bgpen,
	    0, 0, drawsize, drawsize, 0, 0);
}

/*
 *	mat_plot_grid
 *
 *	Function:	- plots the matrix outline and labels
 */
static void
mat_plot_grid()
{
  int i;

  int x, y;			       /* coordinates */

  int w, h;			       /* width and height */

  int dir;			       /* string direction */

  int ascent;			       /* font ascent */

  int descent;			       /* font descent */

  int fmtlen;			       /* string length */

  char *destxt, *srctxt;	       /* labels */

  char fmtbuf[8];		       /* formatting buffer */

  XCharStruct fmtinfo;		       /* string font info */

  XFontStruct *finfo;		       /* font information */

/*
 * Clear the screen.
 */
  XFillRectangle(disp, pixmap, bgpen, 0, 0, drawsize, drawsize);
/*
 * Plot the horizontal grid lines.
 */
  x = bordersize;
  y = bordersize;

  for (i = 0; i <= mat_nprocs; ++i) {
    XDrawLine(disp, pixmap, gridpen, x, y,
	      x + (XMPI_MTCELLSIZE * mat_nprocs), y);
    y += XMPI_MTCELLSIZE;
  }
/*
 * Plot the vertical grid lines.
 */
  x = bordersize;
  y = bordersize;

  for (i = 0; i <= mat_nprocs; ++i) {
    XDrawLine(disp, pixmap, lblpen, x, y,
	      x, y + (XMPI_MTCELLSIZE * mat_nprocs));

    x += XMPI_MTCELLSIZE;
  }
/*
 * Plot the process rank labels.
 */
  for (i = 0; i < mat_nprocs; ++i) {
    sprintf(fmtbuf, "%d", i);
    fmtlen = strlen(fmtbuf);

    XTextExtents(app_res.ap_rankfont, fmtbuf, fmtlen,
		 &dir, &ascent, &descent, &fmtinfo);

    x = bordersize - ((XMPI_MTLNWIDTH / 2) + 1);
    y = bordersize + (XMPI_MTCELLSIZE * i) + (XMPI_MTCELLSIZE / 2);
    XDrawString(disp, pixmap, lblpen, x - fmtinfo.width,
		y + (fmtinfo.ascent / 2), fmtbuf, fmtlen);

    x = bordersize + (XMPI_MTCELLSIZE * i) + (XMPI_MTCELLSIZE / 2);
    y = bordersize - ((XMPI_MTLNWIDTH / 2) + 2);
    XDrawString(disp, pixmap, lblpen, x - (fmtinfo.width / 2),
		y, fmtbuf, fmtlen);
  }

  if (mat_nprocs > 1) {
    if (mat_nprocs > 3) {
      srctxt = (char*) "source";
      destxt = (char*) "destination";
    } else {
      srctxt = (char*) "src";
      destxt = (char*) "des";
    }
/*
 * Plot the source ranks label.
 */
    finfo = XQueryFont(disp, app_res.ap_rankfont->fid);

    XTextExtents(app_res.ap_rankfont, srctxt, strlen(srctxt),
		 &dir, &ascent, &descent, &fmtinfo);

    x = drawsize / 2 - (fmtinfo.width / 2);
    y = bordersize - ((XMPI_MTLNWIDTH / 2) + 4)
      - 2 * (finfo->max_bounds.descent + finfo->max_bounds.descent);

    XDrawString(disp, pixmap, lblpen, x, y, srctxt, strlen(srctxt));
/*
 * Plot the destination ranks label.
 */
    xmpi_vstr_extent(destxt, *app_res.ap_rankfont, &w, &h);

    x = bordersize - ((XMPI_MTLNWIDTH / 2) + 2)
      - finfo->max_bounds.width - w;

    y = (drawsize - h) / 2;

    xmpi_vstr_draw(disp, pixmap, lblpen,
		   x, y, destxt, *app_res.ap_rankfont);

    XFreeFontInfo((char **) 0, finfo, 1);
  }
/*
 * Refresh the screen.
 */
  XCopyArea(disp, pixmap, XtWindow(draw_w), gridpen,
	    0, 0, drawsize, drawsize, 0, 0);
}

/*
 *	xmpi_mat_busy
 *
 *	Function:	- set the busy cursor for the matrix window
 */
void
xmpi_mat_busy()
{
  if (matrix_w)
    xmpi_busy_widget(matrix_w);
}

/*
 *	xmpi_mat_unbusy
 *
 *	Function:	- set the normal cursor for the matrix window
 */
void
xmpi_mat_unbusy()
{
  if (matrix_w)
    xmpi_unbusy_widget(matrix_w);
}
