/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <pwd.h>
#include <errno.h>
#include <unistd.h>
#include <ghttp.h>
#include <locale.h>

#include "editor_query_http.h"

enum {
	LRS_SERVER = 1 << 0,
	LRS_PROXY  = 1 << 1,
	LRS_AGENT  = 1 << 2,
};

enum {
	REQ_UNDEF = 0,
	REQ_SEARCH,
	REQ_GET,
	REQ_ADD,
};

// * Read a single line from the buffer and move the pointer along *
/*
static gchar *LyrixReadLine(gchar **dataptr)
{
	gchar *http_data = *dataptr;
	gchar *pos;

	if ((http_data == NULL) || ((*http_data) == '\0') || ((*http_data) == '.')) {
		*dataptr = NULL;
		return NULL;
	}

	for (pos = http_data; *pos; pos++) {
		if (*pos=='\n') {
			*pos='\0';
#ifdef CODEDEBUG
			DEBUG(4, ("[%s]\n", http_data));
#endif
			*dataptr = pos + 1;

			return http_data;
		}
	}

#ifdef CODEDEBUG
	DEBUG(4, ("[%s]\n", http_data));
#endif
	*dataptr=NULL;

	return http_data;
}
*/

// * Input length has to be a minimum of 4 characters!!!
static void encode_char(gchar* c, gboolean param)
{
	static const gchar convert_chars[] = {
		'$', '&', '+', ',', '/', ':', ';', '=',
		'?', '@', '\"', '<', '>', '#', '%', '{',
		'}', '|', '\\', '^', '~', '[', ']', '`', 0
	};

	gint value = 0;

	if (c[0] == ' ') {
		// We can encode spaces in parameters with '+'
		// Much better to read than %20
		if (param) {
			c[0] = '+';
			value =  0;
		}
		else { value = 0x20; }
	}
	else if ((c[0] <= 0x1F) || (c[0] >= 0x7F)) {
		value = c[0];
	}
	else {
		while ((convert_chars[value] != 0) && (convert_chars[value] != c[0]))
			{ value++; }

		value = convert_chars[value];
	}

	if (value == 0) {
		c[1] = '\0';
		return;
	}

	c[0] = '%';
	c[1] = ((value / 16) >= 10) ? 'A' + ((value / 16) - 10) : '0' + (value / 16);
	c[2] = ((value % 16) >= 10) ? 'A' + ((value % 16) - 10) : '0' + (value % 16);
	c[3] = '\0';
}

static gchar* url_encode(gchar* str, gint length, gboolean param)
{
	GString* buffer;
	gchar append[4], *result;
	gint i;

	if (length < 0) {
		buffer = g_string_sized_new(64);
		for (i = 0; str[i] != '\0'; i++) {
			append[0] = str[i];
			encode_char(append, param);
			buffer = g_string_append(buffer, append);
		}
	}
	else {
		buffer = g_string_sized_new(length);
		for (i = 0; i < length; i++) {
			append[0] = str[i];
			encode_char(append, param);
			buffer = g_string_append(buffer, append);
		}
	}

	result = buffer->str;
	g_string_free(buffer, FALSE);

	return result;
}

LyrixRequest* lyrix_request_new(void)
{
	LyrixRequest* result = NULL;

	result = g_malloc(sizeof(LyrixRequest));

	if (result != NULL) {
		result->request_type = REQ_UNDEF;

		result->artist = NULL;
		result->album = NULL;
		result->title = NULL;

		result->u.add.lyric_stream = NULL;
		result->u.add.author = NULL;

		result->use_proxy = FALSE;
		result->redirect = NULL;
		result->http_data = NULL;
	}

	return result;
}

#define FREE(var) \
	if (var != NULL) { g_free(var); }

#define FREE_N(var) \
	if (var != NULL) { g_free(var); var = NULL; }

static void lyrix_request_clear(LyrixRequest* req)
{
	g_return_if_fail(req != NULL);

	switch (req->request_type) {
	case REQ_ADD:
		FREE(req->u.add.author);
		FREE(req->u.add.lyric_stream);
		break;
	default:
		break;
	}

	req->request_type = REQ_UNDEF;
}

void lyrix_request_free(LyrixRequest* req)
{
	g_return_if_fail(req != NULL);

	if (req->http_data != NULL)
		lyrix_request_unlink_http_data(req);

	lyrix_request_clear(req);

	FREE(req->artist);
	FREE(req->album);
	FREE(req->title);

	g_free(req);
}

LyrixRequestHTTP* lyrix_request_data_new (void)
{
	LyrixRequestHTTP* result = NULL;

	result = g_malloc(sizeof(LyrixRequestHTTP));

	if (result != NULL) {
		result->is_set = 0;
		result->ref = 0;

		result->server.name = NULL;
		result->server.search_uri = NULL;
		result->server.get_uri = NULL;
		result->server.add_uri = NULL;

		result->proxy.name = NULL;
		result->proxy.user = NULL;
		result->proxy.passwd = NULL;

		result->agent.program = NULL;
		result->agent.version = NULL;
	}

	return result;
}

void lyrix_request_data_free (LyrixRequestHTTP* http_data)
{
	g_return_if_fail(http_data != NULL);
	g_return_if_fail(http_data->ref == 0);

	FREE(http_data->server.name);
	FREE(http_data->server.search_uri);
	FREE(http_data->server.get_uri);
	FREE(http_data->server.add_uri);

	FREE(http_data->proxy.name);
	FREE(http_data->proxy.user);
	FREE(http_data->proxy.passwd);

	FREE(http_data->agent.program);
	FREE(http_data->agent.version);

	g_free(http_data);
}

void lyrix_request_data_set_server (LyrixRequestHTTP *http_data,
	const gchar *name, gint port, const gchar *search_uri,
	const gchar *get_uri, const gchar *add_uri)
{
	g_return_if_fail(http_data != NULL);

	FREE_N(http_data->server.name);
	FREE_N(http_data->server.search_uri);
	FREE_N(http_data->server.get_uri);
	FREE_N(http_data->server.add_uri);

	if ((name == NULL) || (search_uri == NULL) || (get_uri == NULL) ||
		(add_uri == NULL) || (port <= 0) || (port >= 65536))
	{
		http_data->is_set &= !LRS_SERVER;
	}
	else { http_data->is_set |= LRS_SERVER; }

	http_data->server.name = g_strdup(name);
	http_data->server.port = port;
	http_data->server.search_uri = g_strdup(search_uri);
	http_data->server.get_uri = g_strdup(get_uri);
	http_data->server.add_uri = g_strdup(add_uri);
}

void lyrix_request_data_set_proxy
	(LyrixRequestHTTP *http_data, const gchar *name, gint port, const gchar *user, const gchar *passwd)
{
	g_return_if_fail(http_data != NULL);

	FREE_N(http_data->proxy.name);
	FREE_N(http_data->proxy.user);
	FREE_N(http_data->proxy.passwd);

	if ((name == NULL) || (port <= 0) || (port >= 65536))
	{
		http_data->is_set &= !LRS_PROXY;
		return;
	}
	else { http_data->is_set |= LRS_PROXY; }

	http_data->proxy.name = g_strdup(name);
	http_data->proxy.user = (user != NULL) ? g_strdup(user) : NULL;
	http_data->proxy.passwd = (passwd != NULL) ? g_strdup(passwd) : NULL;
	http_data->proxy.port = port;
}

void lyrix_request_data_set_agent
	(LyrixRequestHTTP *http_data, const gchar *program, const gchar *version)
{
	g_return_if_fail(http_data != NULL);

	FREE_N(http_data->agent.program);
	FREE_N(http_data->agent.version);

	if ((program == NULL) || (version == NULL)) {
		http_data->is_set &= !LRS_AGENT;
		return;
	}
	else { http_data->is_set |= LRS_AGENT; }

	http_data->agent.program = g_strdup(program);
	http_data->agent.version = g_strdup(version);
}

void lyrix_request_set_redirect(LyrixRequest *req, const gchar *redirect)
{
	g_return_if_fail(req != NULL);

	if (req->redirect != NULL) {
		g_warning("Last redirect (%s) not followed.\n", req->redirect);
		g_free(req->redirect);
	}

	req->redirect = g_strdup(redirect);
}

void lyrix_request_link_http_data(LyrixRequest *req, LyrixRequestHTTP *data)
{
	g_return_if_fail(req != NULL);
	g_return_if_fail(data != NULL);

	lyrix_request_unlink_http_data(req);

	req->http_data = data;
	req->http_data->ref++;
}

void lyrix_request_unlink_http_data(LyrixRequest *req)
{
	g_return_if_fail(req != NULL);

	if (req->http_data != NULL) {
		req->http_data->ref--;
		req->http_data = NULL;
	}
}

void lyrix_request_use_proxy(LyrixRequest *req, gboolean use)
{
	g_return_if_fail(req != NULL);

	req->use_proxy = use;
}

void lyrix_request_set_lyrics_info(LyrixRequest *req,
	const gchar *artist, const gchar *album, const gchar* title)
{
	g_return_if_fail(req != NULL);

	FREE(req->artist);
	FREE(req->album);
	FREE(req->title);

	req->artist = g_strdup(artist);
	req->album = g_strdup(album);
	req->title = g_strdup(title);
}

void lyrix_request_set_search (LyrixRequest *req,
	SearchMatch artist, SearchMatch title)
{
	g_return_if_fail(req != NULL);
	g_return_if_fail((req->artist != NULL) || (req->album != NULL) || (req->title != NULL));
	g_return_if_fail((artist > 0) && (artist < MATCH_LAST));
	g_return_if_fail((title > 0) && (title < MATCH_LAST));

	lyrix_request_clear(req);

	req->request_type = REQ_SEARCH;

	req->u.search.artist_match = artist;
	req->u.search.title_match = title;
	req->u.search.page = 0;
}

void lyrix_request_set_get
	(LyrixRequest *req, gint db_id)
{
	g_return_if_fail(req != NULL);
	g_return_if_fail(db_id >= 0);
	g_return_if_fail((req->artist != NULL) || (req->album != NULL) || (req->title != NULL));

	req->request_type = REQ_GET;

	req->u.get.db_id = db_id;
}

void lyrix_request_set_add
	(LyrixRequest *req, const gchar *author, const gchar *stream)
{
	g_return_if_fail(req != NULL);
	g_return_if_fail(author != NULL);
	g_return_if_fail(stream != NULL);
	g_return_if_fail((req->artist != NULL) && (req->title != NULL));

	lyrix_request_clear(req);

	req->request_type = REQ_ADD;

	req->u.add.author = g_strdup(author);
	req->u.add.lyric_stream = g_strdup(stream);
}

void lyrix_request_set_add_from_search
	(LyrixRequest *req, const gchar *artist, const gchar *album,
	const gchar* title, const gchar *author, const gchar *stream)
{
#define __set_var(var) \
	if (var != NULL) { \
		FREE(req->var); \
		req->var = g_strdup(var); \
	}

	g_return_if_fail(req != NULL);
	g_return_if_fail(req->request_type == REQ_SEARCH);
	g_return_if_fail(author != NULL);
	g_return_if_fail(stream != NULL);
	g_return_if_fail((artist != NULL) || (req->artist != NULL));
	g_return_if_fail((album != NULL) || (req->album != NULL));
	g_return_if_fail((title != NULL) || (req->title != NULL));

	lyrix_request_clear(req);

	__set_var(artist);
	__set_var(album);
	__set_var(title);

	req->request_type = REQ_ADD;

	req->u.add.author = g_strdup(author);
	req->u.add.lyric_stream = g_strdup(stream);
#undef __set_var
}

static GString *lyrix_request_compose_uri(LyrixRequest *req)
{
	GString *uri;
	gchar *tmp[4];

	uri = g_string_new(NULL);

	tmp[0] = url_encode(req->artist, -1, TRUE);
	tmp[1] = url_encode(req->album, -1, TRUE);
	tmp[2] = url_encode(req->title, -1, TRUE);

	switch (req->request_type) {
	case REQ_SEARCH:
		if (req->u.search.page > 0) {
			tmp[3] = g_strdup_printf
				("?page=%i\n", req->u.search.page);
		}
		else { tmp[3] = g_strdup(""); }
		g_string_sprintf(uri,
			"http://%s:%i/%s?artist=%s&album=%s&title=%s&sart=%i&stit=%i%s",
			req->http_data->server.name,
			req->http_data->server.port,
			req->http_data->server.search_uri,
			tmp[0], tmp[1], tmp[2],
			req->u.search.artist_match,
			req->u.search.title_match,
			tmp[3]);
		g_free(tmp[3]);
		break;
	case REQ_GET:
		g_string_sprintf(uri,
			"http://%s:%i/%s?id=%i&artist=%s&album=%s&title=%s",
			req->http_data->server.name,
			req->http_data->server.port,
			req->http_data->server.get_uri,
			req->u.get.db_id,
			tmp[0], tmp[1], tmp[2]);
		break;
	case REQ_ADD:
		g_string_sprintf(uri,
			"http://%s:%i/%s?artist=%s&album=%s&title=%s",
			req->http_data->server.name, req->http_data->server.port,
			req->http_data->server.add_uri,
			tmp[0], tmp[1], tmp[2]);
		break;
	default:
		g_free(tmp[2]);
		g_free(tmp[1]);
		g_free(tmp[0]);
		g_string_free(uri, TRUE);
		return NULL;
	}

	if (ghttp_uri_validate(uri->str) == -1) {
		g_string_free(uri, TRUE);
		uri = NULL;
	}

	return uri;
}

static gchar* lyrix_request_exec
	(LyrixRequest *req, ghttp_request *request, HTTPProxy *default_proxy)
{
	HTTPProxy *proxy;
	GString *uri;
	GString *proxy_uri;
	char user_agent[256];

	proxy = (default_proxy != NULL) ? default_proxy :
		((req->use_proxy == TRUE) ? &(req->http_data->proxy) : NULL);
	if (proxy != NULL) {
		proxy_uri = g_string_new(NULL);
		g_string_sprintf(proxy_uri, "http_data://%s:%d",
			proxy->name, proxy->port);

		ghttp_set_proxy(request, proxy_uri->str);

		g_string_free(proxy_uri, TRUE);

		if (proxy->user) {
			ghttp_set_proxy_authinfo
				(request, proxy->user, proxy->passwd);
		}
	}

	if (req->redirect == NULL) {
		uri = lyrix_request_compose_uri(req);
		if (uri == NULL)
			{ return NULL; }
	}
	else {
		uri = g_string_new(req->redirect);

		// We just follow a redirect once!
		FREE_N(req->redirect);

		if ((req->request_type == REQ_SEARCH) &&
			(req->u.search.page > 0))
		{
			g_string_sprintfa(uri, "&page=%i",
				req->u.search.page);
		}
	}

#ifdef CODE_DEBUG
	DEBUG(4, ("URI is %s\n", uri->str));
#endif
	ghttp_set_uri(request, uri->str);

	g_snprintf(user_agent, 256, "%s %s",
		req->http_data->agent.program, req->http_data->agent.version);
	ghttp_set_header(request, "User-Agent", user_agent);

	ghttp_set_header(request, http_hdr_Connection, "close");
	ghttp_prepare(request);
	ghttp_process(request);

	g_string_free(uri, TRUE);

	return ghttp_get_body(request);
}

gboolean lyrix_request_data_is_valid(LyrixRequestHTTP *http_data)
{
	g_return_val_if_fail(http_data != NULL, FALSE);
	g_return_val_if_fail(http_data->is_set & LRS_SERVER, FALSE);
	g_return_val_if_fail(http_data->is_set & LRS_AGENT, FALSE);

	return TRUE;
}

gboolean lyrix_request_is_valid(LyrixRequest *req)
{
	g_return_val_if_fail(req != NULL, FALSE);

	g_return_val_if_fail(req->request_type != REQ_UNDEF, FALSE);

	// Is the request http_data set ?
	g_return_val_if_fail
		(lyrix_request_data_is_valid(req->http_data) == TRUE, FALSE);

	// Are we using a proxy but don't have the proxy data
	g_return_val_if_fail((req->use_proxy == FALSE) ||
		(req->http_data->is_set & LRS_PROXY), FALSE);

	return TRUE;
}

gboolean lyrix_request_do_query
	(LyrixRequest *req, LyrixQueryResult **res, HTTPProxy *default_proxy)
{
	ghttp_request    *http_request = NULL;
	gboolean          result = FALSE;
	gchar            *old_locale;
	gchar            *http_body;
	ParseType         parse_type;

	g_return_val_if_fail(req != NULL, FALSE);
	g_return_val_if_fail(res != NULL, FALSE);
	g_return_val_if_fail(req->http_data != NULL, FALSE);

	if (lyrix_request_is_valid(req) == FALSE)
		return FALSE;

	http_request = ghttp_request_new();
	if (http_request == NULL)
		return FALSE;

	if (((*res) != NULL) && (req->request_type == REQ_SEARCH))
		req->u.search.page = (*res)->list.pages;

	// * work around a bug in libghttpd *
	old_locale = strdup(setlocale(LC_NUMERIC,"C"));

	http_body = lyrix_request_exec
		(req, http_request, default_proxy);

	setlocale(LC_NUMERIC, old_locale);
	free(old_locale);

	if ((ghttp_get_body_len(http_request) > 0) &&
		(ghttp_status_code(http_request) == 200))
	{
		if ((*res) == NULL)
			(*res) = lyrix_query_result_new();

		switch (req->request_type) {
		case REQ_SEARCH: parse_type = PT_QUE; break;
		case REQ_GET: parse_type = PT_LYR; break;
		default: parse_type = PT_UDF; break;
		}

		if (parse_type != PT_UDF) {
			result = lyrix_query_result_parse
				(*res, http_body,
				ghttp_get_body_len(http_request),
				PT_BUF | PT_TXT | parse_type);
		}
	}

	ghttp_request_destroy(http_request);

	return result;
}

/*
gboolean LyrixDoQuery(LyrixRequest *req)
{
	inbuffer = LyrixReadLine(&result);

#ifdef CODE_DEBUG
	DEBUG(4, ("Reply is [%s]\n", inbuffer));
#endif

	switch (strtol(strtok(inbuffer," "),NULL,10)) {
	// * 200 - exact match *
	case 200:
		query->query_match = MATCH_EXACT;
		query->query_matches = 1;

		LyrixParseTitle(g_strstrip(strtok(NULL,"")),
			query->result_list[0].title,
			query->result_list[0].artist, "/");
		break;
	// * 210 - multiple exact matches *
	case 210:
		query->query_match = MATCH_EXACT;
		query->query_matches = 0;

		while((inbuffer=LyrixReadLine(&result))) {
			LyrixParseTitle(g_strstrip(strtok(NULL,"")),
				query->result_list[query->query_matches].title,
				query->result_list[query->query_matches].artist,
				"/");

			query->query_matches++;
		}
		break;
	// * 211 - inexact match *
	case 211:
		query->query_match=MATCH_INEXACT;
		query->query_matches=0;

		while ((inbuffer=LyrixReadLine(&result))) {
			LyrixParseTitle(g_strstrip(strtok(NULL,"")),
				query->result_list[query->query_matches].title,
				query->result_list[query->query_matches].artist,
				"/");

			query->query_matches++;
		}
		break;
	// * No match *
	default:
		query->query_match = MATCH_NOMATCH;

		ghttp_request_destroy(request);

		return FALSE;
	}

	ghttp_request_destroy(request);

	return TRUE;
}
*/

// * Process a line of input http_data *
/*
static void LyrixProcessLine
	(gchar *inbuffer, LyrixData *http_data, gint numtracks)
{
	hint track;
	gint len=0;
	gchar *st;

	if(!strncasecmp(inbuffer,"# Revision: ",12)) {
		http_data->revision=atoi(inbuffer+12);
	}
	else if(!strncasecmp(inbuffer,"DTITLE",6)) {
		len=strlen(http_data->data_title);

		g_snprintf(http_data->data_title+len,256-len,"%s",g_strstrip(inbuffer+7));
	}
	else if(!strncasecmp(inbuffer,"DYEAR",5)) {
		strtok(inbuffer,"=");

		st = strtok(NULL, "");
		if(st == NULL)
				return;

		http_data->data_year=atoi(g_strstrip(st));
	}
	else if(!strncasecmp(inbuffer,"DGENRE",6)) {
		strtok(inbuffer,"=");

		st = strtok(NULL, "");
		if(st == NULL)
				return;

		st=g_strstrip(st);

		if(*st) {
			http_data->data_genre=LyrixGenreValue(g_strstrip(st));
			http_data->data_id3genre=ID3GenreValue(g_strstrip(st));
		}
	}
	else if(!strncasecmp(inbuffer,"DID3",4)) {
		strtok(inbuffer,"=");

		st = strtok(NULL, "");
		if(st == NULL)
				return;

		http_data->data_id3genre=atoi(g_strstrip(st));
	}
	else if(!strncasecmp(inbuffer,"TTITLE",6)) {
		track=atoi(strtok(inbuffer+6,"="));

		if(track<numtracks)
			len=strlen(http_data->data_track[track].track_name);

		g_snprintf(http_data->data_track[track].track_name+len,256-len,"%s",
			g_strstrip(strtok(NULL,"")));
	}
	else if(!strncasecmp(inbuffer,"TARTIST",7)) {
		http_data->data_multi_artist=TRUE;

		track=atoi(strtok(inbuffer+7,"="));

		if(track<numtracks)
			len=strlen(http_data->data_track[track].track_artist);

		st = strtok(NULL, "");
		if(st == NULL)
				return;

		g_snprintf(http_data->data_track[track].track_artist+len,256-len,"%s",
			g_strstrip(st));
	}
	else if(!strncasecmp(inbuffer,"EXTD",4)) {
		len=strlen(http_data->data_extended);

		g_snprintf(http_data->data_extended+len,4096-len,"%s",g_strstrip(inbuffer+5));
	}
	else if(!strncasecmp(inbuffer,"EXTT",4)) {
		track=atoi(strtok(inbuffer+4,"="));

		if(track<numtracks)
			len=strlen(http_data->data_track[track].track_extended);

		st = strtok(NULL, "");
		if(st == NULL)
				return;

		g_snprintf(http_data->data_track[track].track_extended+len,4096-len,"%s",
			g_strstrip(st));
	}
	else if(!strncasecmp(inbuffer,"PLAYORDER",5)) {
		len=strlen(http_data->data_playlist);

		g_snprintf(http_data->data_playlist+len,256-len,"%s",g_strstrip(inbuffer+10));
	}
}

// * Read the actual Lyrix entry *
gboolean LyrixRead(SingitSong *song, LyrixServer *server,
	LyrixHello *hello, ResultListEntry *entry, LyrixData *http_data)
{

	int index;
	ghttp_request *request=NULL;
	GString *cmd;
	char *result,*inbuffer;

	request=ghttp_request_new();
	if(!request) return FALSE;

	if(!disc->have_info) CDStat(disc,TRUE);

	http_data->data_genre=entry->entry_genre;
	http_data->data_id=LyrixLyrixid(disc);
	*(http_data->data_extended)='\0';
	*(http_data->data_title)='\0';
	*(http_data->data_artist)='\0';
	*(http_data->data_playlist)='\0';
	http_data->data_multi_artist=FALSE;
	http_data->data_year=0;
	http_data->data_id3genre=-1;
	http_data->revision=-1;

	for(index=0;index<MAX_TRACKS;index++) {
		*(http_data->data_track[index].track_name)='\0';
		*(http_data->data_track[index].track_artist)='\0';
		*(http_data->data_track[index].track_extended)='\0';
	}

	cmd=g_string_new(NULL);

	g_string_sprintf(cmd,"cddb+read+%s+%08x",LyrixGenre(entry->entry_genre),
			 entry->entry_id);

	result=LyrixMakeRequest(server,hello,cmd->str,request);

	g_string_free(cmd,TRUE);

	if(!result) {
		ghttp_request_destroy(request);

		return FALSE;
	}

	inbuffer=LyrixReadLine(&result);

	while((inbuffer=LyrixReadLine(&result)))
		LyrixProcessLine(inbuffer,http_data,disc->num_tracks);

	// * Both disc title and artist have been stuffed in the title field, so the *
	// * need to be separated *

	LyrixParseTitle(http_data->data_title,http_data->data_title,http_data->data_artist,"/");

	ghttp_request_destroy(request);

	// * Don't allow the genre to be overwritten *
	http_data->data_genre=entry->entry_genre;

	return TRUE;
}

// * See if a disc is in the local database *

gboolean LyrixStatLyrixData(SingitSong *song)
{
	int index,id;
	struct stat st;
	char root_dir[256],file[256];

	if(!disc->have_info) CDStat(disc,TRUE);

	id = LyrixLyrixid(disc);

	g_snprintf(root_dir,256,"%s/.cddb",getenv("HOME"));

	if(stat(root_dir, &st) < 0)
		return FALSE;
	else {
		if(!S_ISDIR(st.st_mode))
			return FALSE;
	}

	g_snprintf(file,256,"%s/%08x",root_dir,id);
	if(stat(file,&st)==0) return TRUE;

	for(index=0;index<12;index++) {
		g_snprintf(file,256,"%s/%s/%08x",root_dir,LyrixGenre(index),id);

		if(stat(file,&st) == 0)
			return TRUE;
	}

	return FALSE;
}

// * Read from the local database *
int LyrixReadLyrixData(SingitSong *song, LyrixData *ddata)
{
	FILE *discdb_data=NULL;
	int index,genre;
	char root_dir[256],file[256],inbuf[512];
	struct stat st;

	g_snprintf(root_dir,256,"%s/.cddb",getenv("HOME"));

	if(stat(root_dir, &st) < 0) {
		return -1;
	} else {
		if(!S_ISDIR(st.st_mode)) {
			errno = ENOTDIR;
			return -1;
		}
	}

	if(!disc->have_info) CDStat(disc,TRUE);

	ddata->data_id=LyrixLyrixid(disc);
	*(ddata->data_extended)='\0';
	*(ddata->data_title)='\0';
	*(ddata->data_artist)='\0';
	*(ddata->data_playlist)='\0';
	ddata->data_multi_artist=FALSE;
	ddata->data_year=0;
	ddata->data_genre=7;
	ddata->data_id3genre=-1;
	ddata->revision=-1;

	for(index=0;index<MAX_TRACKS;index++) {
		*(ddata->data_track[index].track_name)='\0';
		*(ddata->data_track[index].track_artist)='\0';
		*(ddata->data_track[index].track_extended)='\0';
	}

	g_snprintf(file,256,"%s/%08x",root_dir,ddata->data_id);
	if(stat(file,&st)==0) {
		discdb_data=fopen(file, "r");
	}
	else {
		for(genre=0;genre<12;genre++) {
			g_snprintf(file,256,"%s/%s/%08x",root_dir,LyrixGenre(genre),
				 ddata->data_id);

			if(stat(file,&st)==0) {
	discdb_data=fopen(file, "r");

	ddata->data_genre=genre;
	break;
			}
		}

		if(genre==12) return -1;
	}

	while(fgets(inbuf,512,discdb_data))
		LyrixProcessLine(inbuf,ddata,disc->num_tracks);

	// * Both disc title and artist have been stuffed in the title field, so the *
	// * need to be separated *

	LyrixParseTitle(ddata->data_title,ddata->data_title,ddata->data_artist,"/");

	fclose(discdb_data);

	return 0;
}

static void LyrixWriteLine(char *header,int num,char *http_data,FILE *outfile)
{
	int len;
	char *offset;

	len=strlen(http_data);
	offset=http_data;

	for(;;) {
		if(len>70) {
			if(num==-1)
	fprintf(outfile,"%s=%.70s\n",header,offset);
			else fprintf(outfile,"%s%d=%.70s\n",header,num,offset);

			offset+=70;
			len-=70;
		}
		else {
			if(num==-1) fprintf(outfile,"%s=%s\n",header,offset);
			else fprintf(outfile,"%s%d=%s\n",header,num,offset);
			break;
		}
	}
}

// * Write to the local cache *
int LyrixWriteLyrixData(SingitSong *song, LyrixData *ddata,
	FILE *outfile, gboolean gripext,gboolean freedbext)
{
	FILE *discdb_data;
	int track;
	char root_dir[256],file[256],tmp[512];
	struct stat st;

	if(!disc->have_info) CDStat(disc,TRUE);

	if(!outfile) {
		g_snprintf(root_dir,256,"%s/.cddb",getenv("HOME"));
		g_snprintf(file,256,"%s/%08x",root_dir,ddata->data_id);

		if(stat(root_dir,&st)<0) {
			if(errno != ENOENT) {
	Debug(_("Stat error %d on %s\n"),errno,root_dir);
	return -1;
			}
			else {
	Debug(_("Creating directory %s\n"),root_dir);
	mkdir(root_dir,0777);
			}
		} else {
			if(!S_ISDIR(st.st_mode)) {
	Debug(_("Error: %s exists, but is a file\n"),root_dir);
	errno=ENOTDIR;
	return -1;
			}
		}

		if((discdb_data=fopen(file,"w"))==NULL) {
			Debug(_("Error: Unable to open %s for writing\n"),file);
			return -1;
		}
	}
	else discdb_data=outfile;

#ifndef GRIPCD
	fprintf(discdb_data,"# xmcd CD database file generated by Grip %s\n",
		VERSION);
#else
	fprintf(discdb_data,"# xmcd CD database file generated by GCD %s\n",
		VERSION);
#endif
	fputs("# \n",discdb_data);
	fputs("# Track frame offsets:\n",discdb_data);

	for(track=0;track<disc->num_tracks;track++)
		fprintf(discdb_data, "#			 %d\n",disc->track[track].start_frame);

	fputs("# \n",discdb_data);
	fprintf(discdb_data,"# Lyrix length: %d seconds\n",disc->length.mins *
		60 + disc->length.secs);
	fputs("# \n",discdb_data);

	if(gripext) fprintf(discdb_data,"# Revision: %d\n",ddata->revision);
	else fprintf(discdb_data,"# Revision: %d\n",ddata->revision+1);

	fprintf(discdb_data,"# Submitted via: Grip %s\n",VERSION);
	fputs("# \n",discdb_data);
	fprintf(discdb_data,"DISCID=%08x\n",ddata->data_id);

	g_snprintf(tmp,512,"%s / %s",ddata->data_artist,ddata->data_title);
	LyrixWriteLine("DTITLE",-1,tmp,discdb_data);

	if(gripext||freedbext) {
		if(ddata->data_year)
			fprintf(discdb_data,"DYEAR=%d\n",ddata->data_year);
		else fprintf(discdb_data,"DYEAR=\n");
	}

	if(gripext) {
		fprintf(discdb_data,"DGENRE=%s\n",LyrixGenre(ddata->data_genre));
		fprintf(discdb_data,"DID3=%d\n",ddata->data_id3genre);
	}
	else if(freedbext) {
		fprintf(discdb_data,"DGENRE=%s\n",ID3GenreString(ddata->data_id3genre));
	}

	for(track=0;track<disc->num_tracks;track++) {
		if(gripext||!*(ddata->data_track[track].track_artist)) {
			LyrixWriteLine("TTITLE",track,ddata->data_track[track].track_name,
					discdb_data);
		}
		else {
			g_snprintf(tmp,512,"%s / %s",ddata->data_track[track].track_artist,
		 ddata->data_track[track].track_name);
			LyrixWriteLine("TTITLE",track,tmp,discdb_data);
		}

		if(gripext&&*(ddata->data_track[track].track_artist))
			LyrixWriteLine("TARTIST",track,ddata->data_track[track].track_artist,
					discdb_data);
	}

	LyrixWriteLine("EXTD",-1,ddata->data_extended,discdb_data);

	for(track=0;track<disc->num_tracks;track++)
		LyrixWriteLine("EXTT",track,
			ddata->data_track[track].track_extended,discdb_data);

	if(outfile)
		fprintf(discdb_data,"PLAYORDER=\n");
	else {
		fprintf(discdb_data,"PLAYORDER=%s\n",ddata->data_playlist);
		fclose(discdb_data);
	}

	return 0;
}
*/
