
#include "PluginGlue.h"

#include "EgOSUtils.h"
#include "RectUtils.h"
#include "XFloatList.h"
#include <math.h>
#include "ConsolePixPort.h"
#include "WindowDevice.h"
#include "CEgIOFile.h"

#if ITUNES
#include "iTunesVisualAPI.h"
#endif

#if SOUNDJAM || AUDION
#include "VisFramework.h"
#endif

#if EG_MAC
#define __defaultFont		"Palatino"
#include <Windows.h>
#include <Palettes.h>
#endif

#if UNSANITY_ECHO
#include "Echo_Utilities.h"
#endif


#if EG_WIN
#define __defaultFont 		"Times"
#include "RectUtils.h"
#endif


#if EG_X
#define __defaultFont 		""
#include "RectUtils.h"
#endif

// #include <stdio.h>

PluginGlue::PluginGlue( long inHostVers, void* inRefCon,  const PluginInfo info):
	mConsoleLines( ASA_OrderImportant),
	mLineExpireTimes( cOrderImportant )
{

	// Init runtime vars...
	mHostVersion			= inHostVers;
	mRefCon				= inRefCon;
	mDoingSetPortWin		= false;
	mAtFullScreen			= false;
	mMouseWillAwaken		= false;
	mTrackTextIsReset		= true;
	mShowConsole			= true;
	mWinRectHolder.top		= -1;

	mVideoOutput = 0;
	mParentWind = 0;

	mT_MS		= 0;
	mT_MS_Base	= 0;
	mT		= 0;

	mNeedsInit = true;

	// Init screensaver/keypressed record...
	for ( int i = 0; i < 4; i++ )
		mCurKeys[ i ] = 0;

	__invalidate( mConsoleDirtyRgn )

	// Init some vars...
	CEgIOFile::sCreatorType = info.mAuthor;

	// Link some important vars into the pref compiler dict
	mDict.AddVar( "T", &mT );
	mDict.AddVar( "LAST_SONG_START", &mLastSongStart );
	mDict.AddVar( "X", &mFFT_X );
	mDict.AddVar( "W", &mFFT_W );
	mDict.AddVar( "I", &mTransition_I );

	// Set the name of our prefs file
	mPrefs.SetFilename( info.mPrefFile, true );

	// Establish the root dir
	mMainFolder.SetRootDir( info.mConfigsFolder );

	mPluginInfo = info;
}

PluginGlue::~PluginGlue() {

}

void PluginGlue::PrepareToExit() {

	Rect r;

#if !ITUNES
	SetFullscreen( false );
#endif

	SavePrefs( mPrefs );

#if EG_WIN
	// Very annoying:  in Win32, the call to GetWindowRect() is returning garbage in ~WhiteCap(), so just
	// forget about saving the window position for now in windows :_(

	//GetWinRect( r );
	r = mWinRectHolder;  // Just use a recent rect for now
#else
	GetWinRect( r );
#endif
	mPrefs.SetPref( MCC4_TO_INT( "wTop" ), r.top );
	mPrefs.SetPref( MCC4_TO_INT( "wLft" ), r.left );
	mPrefs.SetPref( MCC4_TO_INT( "wBot" ), r.bottom );
	mPrefs.SetPref( MCC4_TO_INT( "wRgt" ), r.right );

	mPrefs.Store();

	EgOSUtils::ResetCursor();
}

void PluginGlue::SetPrefsFactory(FactoryPrefPresets presets) {

	// Init Prefs...
	mConsoleLineDur		= 14;
	mFFT_Bin_Start		= presets.mFFT_Bin_Start;
	mFFT_StepsPerBin	= presets.mFFT_StepsPerBin;
	mFFT_Smooth		= presets.mFFT_Smooth;
	mFFT_NumBins		= presets.mFFT_NumBins;
	mFFT_FadeAway		= presets.mFFT_FadeAway;
	mCacheDepth		= 1;
	mScrnSaverDelay		= -1 * 60;			// Factory: screen saver mode disabled
	mBorderlessWind		= 0;
	mAlwaysOnTop		= 0;
	mFullscreenDepth	= presets.mFullscreenDepth;
	mFullscreenMode		= 0;
	mMaxDispSize		= presets.mMaxDispSize;
	mSample_Scale		= 1;
	mSample_Smooth		= presets.mSample_Smooth;
	mSample_NumBins		= presets.mSample_NumBins;
	mTextSize		= 18;
	mTextFont		.Assign( __defaultFont );
	mTrackTextStartStr	.Assign( "sqwv( ( t - LAST_SONG_START ) % 600 )" );
	mKeyMap			.Assign( "" );
	mFFT_TransformStr	.Assign( presets.mFFT_TransformStr );
	mStartupCmdList		.Assign( "" );

	mForeFPS		= presets.mForeFPS;
	mBackFPS		= presets.mBackFPS;

	mPrefs.SetPref( MCC4_TO_INT( "Vers" ), mPluginInfo.mPrefCompatVersion );
}


void PluginGlue::LoadPrefs( Prefs& inPrefs ) {

	// Load Prefs...
	mConsoleLineDur		= inPrefs.GetPref( MCC4_TO_INT( "CLin" ) );
	mFFT_Bin_Start		= inPrefs.GetPref( MCC4_TO_INT( "FSrt" ) );
	mFFT_StepsPerBin	= inPrefs.GetPref( MCC4_TO_INT( "FRge" ) );
	mFFT_Smooth		= inPrefs.GetPref( MCC4_TO_INT( "FSmo" ) ) / 10000.0;
	mFFT_NumBins		= inPrefs.GetPref( MCC4_TO_INT( "FNum" ) );
	mFFT_FadeAway		= inPrefs.GetPref( MCC4_TO_INT( "FFde" ) ) / 1000.0;
	mCacheDepth		= inPrefs.GetPref( MCC4_TO_INT( "Cach" ) );
	mScrnSaverDelay		= inPrefs.GetPref( MCC4_TO_INT( "SSvr" ) ) * 60.0;
	mBorderlessWind		= inPrefs.GetPref( MCC4_TO_INT( "NoBo" ) );
	mAlwaysOnTop		= inPrefs.GetPref( MCC4_TO_INT( "FWin" ) );
	mFullscreenDepth	= inPrefs.GetPref( MCC4_TO_INT( "FS_D" ) );
	mFullscreenMode		= inPrefs.GetPref( MCC4_TO_INT( "FS_M" ) );
	mMaxDispSize.h		= inPrefs.GetPref( MCC4_TO_INT( "MaxX" ) );
	mMaxDispSize.v		= inPrefs.GetPref( MCC4_TO_INT( "MaxY" ) );
	mSample_Scale		= inPrefs.GetPref( MCC4_TO_INT( "SScl" ) ) / 1000.0;
	mSample_Smooth		= inPrefs.GetPref( MCC4_TO_INT( "SSmo" ) ) / 10000.0;
	mSample_NumBins		= inPrefs.GetPref( MCC4_TO_INT( "SNum" ) );
	mTextSize		= inPrefs.GetPref( MCC4_TO_INT( "TSze" ) );
	mForeFPS		= inPrefs.GetPref( MCC3_TO_INT( "FPS" ) );
	mBackFPS		= inPrefs.GetPref( MCC4_TO_INT( "FPS*" ) );

	inPrefs.GetPref( MCC3_TO_INT( "TFn" ), mTextFont );
	inPrefs.GetPref( MCC2_TO_INT( "T?" ),   mTrackTextStartStr );
	inPrefs.GetPref( MCC4_TO_INT( "KMap" ), mKeyMap );
	inPrefs.GetPref( MCC4_TO_INT( "FFTT" ), mFFT_TransformStr );
	inPrefs.GetPref( MCC4_TO_INT( "SCmd" ), mStartupCmdList );
}

void PluginGlue::SavePrefs( Prefs& inPrefs ) {

	// Save Prefs...
	inPrefs.SetPref( MCC4_TO_INT( "CLin" ), mConsoleLineDur );
	inPrefs.SetPref( MCC4_TO_INT( "FSrt" ), mFFT_Bin_Start );
	inPrefs.SetPref( MCC4_TO_INT( "FRge" ), mFFT_StepsPerBin );
	inPrefs.SetPref( MCC4_TO_INT( "FSmo" ), mFFT_Smooth * 10000.0 );
	inPrefs.SetPref( MCC4_TO_INT( "FNum" ), mFFT_NumBins );
	inPrefs.SetPref( MCC4_TO_INT( "FFde" ), mFFT_FadeAway * 1000.0 );
	inPrefs.SetPref( MCC4_TO_INT( "Cach" ), mCacheDepth );
	inPrefs.SetPref( MCC4_TO_INT( "SSvr" ), mScrnSaverDelay / 60.0 );
	inPrefs.SetPref( MCC4_TO_INT( "NoBo" ), mBorderlessWind );
	inPrefs.SetPref( MCC4_TO_INT( "FWin" ), mAlwaysOnTop );
	inPrefs.SetPref( MCC4_TO_INT( "FS_D" ), mFullscreenDepth );
	inPrefs.SetPref( MCC4_TO_INT( "FS_M" ), mFullscreenMode );
	inPrefs.SetPref( MCC4_TO_INT( "MaxY" ), mMaxDispSize.v );
	inPrefs.SetPref( MCC4_TO_INT( "MaxX" ), mMaxDispSize.h );
	inPrefs.SetPref( MCC4_TO_INT( "SScl" ), mSample_Scale * 1000.0 );
	inPrefs.SetPref( MCC4_TO_INT( "SSmo" ), mSample_Smooth * 10000.0 );
	inPrefs.SetPref( MCC4_TO_INT( "SNum" ), mSample_NumBins );
	inPrefs.SetPref( MCC4_TO_INT( "TSze" ), mTextSize );
	inPrefs.SetPref( MCC3_TO_INT( "FPS" ),  mForeFPS );
	inPrefs.SetPref( MCC4_TO_INT( "FPS*" ), mBackFPS );
	inPrefs.SetPref( MCC4_TO_INT( "TFnt" ), mTextFont );
	inPrefs.SetPref( MCC2_TO_INT( "T?" ),   mTrackTextStartStr );
	inPrefs.SetPref( MCC4_TO_INT( "KMap" ), mKeyMap );
	inPrefs.SetPref( MCC4_TO_INT( "FFTT" ), mFFT_TransformStr );
	inPrefs.SetPref( MCC4_TO_INT( "SCmd" ), mStartupCmdList );
}

void PluginGlue::Init() {

	// Read the prefs from disk
 	mPrefs.Load();
	if ( mPrefs.GetPref( MCC4_TO_INT( "Vers" ) ) != mPluginInfo.mPrefCompatVersion ) {
		mPrefs.ErasePrefData();		// Chuck any partially loaded pref data
		SetPrefsFactory();
		SavePrefs( mPrefs );
	}
	LoadPrefs( mPrefs );

	// Keep the host (of any) what fullscreen settings we prefer
	AssertFullscreenSettings();

	// Track Text stuff
	mTrackTextStartFcn.Compile( mTrackTextStartStr, mDict );

	// FFT processing init
	mFFT_TransformFcn.Compile( mFFT_TransformStr, mDict );

	// Alloc/setup the sound mem our virtual machines will access...
	SetFFT_NumBins( mFFT_NumBins );
	SetSample_NumBins( mSample_NumBins );

#if STANDALONE
	mSoundPort.SetSampleSkip( 0 );
	OSErr err = mSoundPort.StartRecording();
#endif

	// We have to init the time index to something in case something in the init process uses it
	ResetTimeIndex( EgOSUtils::CurTimeMS() );
}

void PluginGlue::Initialize() {

	// We have to init the time index to something in case something in the init process uses it
	ResetTimeIndex( EgOSUtils::CurTimeMS() );

	// Make a list of all the items in the "Scripts" folder
	CacheFolder( "Scripts", mScripts, nil, mCacheDepth );

	// Store the scripts folder ID
	mMainFolder.GetItemID( FILE_SYS_ROOT_ID, "Scripts", mScriptFolderID );

	// Show the welcome msg...
	Println( "" );
	Println( mPluginInfo.mLongVersionStr );
	Print( "   v" );
	Println( mPluginInfo.mVersionStr );
	Println( "   www.55ware.com" );

#ifndef CANT_ACCEPT_KEYSTROKES
	Print( "   Press " );
	Print( mKeyMap.getChar( cShowHelp ) );
	Print( " for help, SHIFT+" );
	Print( mKeyMap.getChar( cShowHelp ) );
	Println( " for docs." );
#endif

	Println( "" );

	// Queue the pref's startup command line
	QueueCmdList( mStartupCmdList );
}

void PluginGlue::ResetTimeIndex( long inNewTimeIndex_MS ) {

	long oldTimeBase = mT_MS_Base;

	mT_MS_Base		= inNewTimeIndex_MS;
	mT_MS			= 0;
	mT			= 0;
	mLastCursorUpdate	=
	mLastKeyPollTime	= mT - 100;
	mLastActiveTime		= mT;
	mLastSongStart		= mT - 10000;

	// Modify the console expiration times...
	long numLines = mConsoleLines.Count();
	for ( int i = 0; i < numLines; i++ ) {
		mLineExpireTimes[ i ] = mLineExpireTimes[ i ] + oldTimeBase - mT_MS_Base;
	}

	KillRunningScripts();

	TimeIndexHasChanged( oldTimeBase );
}

void PluginGlue::Draw( SPluginData* inData, long inTime_MS ) {

	// We need access to inData in RecordSample()
	mClientData = inData;

	// Are we using the system clock as the time index?
	if ( inTime_MS < 0 )
		inTime_MS = EgOSUtils::CurTimeMS();

	// If we run for dozens of hours, mT loses percision because the number of elapsed secs
	//   since startup is large while the frame speed is on the order of a hundredth of a sec.
	//   (ex, 550000.01 gets truncated to 550000.0 because the mantissa can only be so precise.
	//   The end result is a very ugly and bothersome jitter of time-based objects.
	// Float percision gives us a mantissa of about 6 and a half base 10 digits.  20-50 FPS is on the order of
	//   hundredth of a second accuracy, leaving us roughly 5 and a half digits--that's 50000 secs (about 14 hours).
	if ( mT_MS > 50000000 )
		ResetTimeIndex( inTime_MS );

	/* Index the time from the start of the plugin (ie mT_MS_Base).  This is done because we use the original time indexes, we have
	 very large numbers.  With 32bit ints this is fine, but when they're put in 32 bit floats, we lose critical
	 percision (because each sample's index only varies by 10 or 20 milliseconds.  */
	mT_MS = inTime_MS - mT_MS_Base;
	mT = 0.001 * mT_MS;

	// BeginFrame and EndFrame must encapsulate all blocks containing possible drawing
	mVideoOutput -> BeginFrame();
	{
		RemoveExpiredConsoleLines();

		// Check running time-based scripts to see if we need to do something
		ManageTimeScripts();

		// Do we start new text.  New text is started only when startText goes from false to true (not
		//   when startText is just true, or else we'd cause "new" text each frame)--we use
		//   mTrackTextIsReset to keep track of this.  An exception is when NewSong() is called--we
		//   reset it then so the user will see the new song info asap (assuming the 'T?' parameter
		//   depends on LAST_SONG_START)
		bool startText = mTrackTextStartFcn.Evaluate() > 0;
		if ( mTrackTextIsReset && startText ) {
			StartTrackText();
			mTrackTextIsReset = false; }
		else if ( ! startText && ! mTrackTextIsReset )
			mTrackTextIsReset = true;

		// Don't bother drawing anything if we're too small
		if ( mDimX > 3 && mDimY > 3 )
			DoFrame();
	}
	mVideoOutput -> EndFrame();

	// Manage screen saver bahavior and system activity
	IdleMonitor();

	if ( ! IsFullscreen() )
		EgOSUtils::ResetCursor();
}

void PluginGlue::RecordSample( bool inDoFFT, bool inDoSamples ) {

	bool doRecord = true;

	if ( doRecord ) {
		if ( inDoSamples )
			RecordSample();

		if ( inDoFFT )
			RecordFFT();
	}
}

/*
	// Calculate bass1
	// Assign mBassLvl here -- choose the larger bins
	char* chosen = sTemp.Dim( BINS_TO_COMPARE );
	for ( i = 0; i < BINS_TO_COMPARE; i++ )
		chosen[ i ] = 0;

	// sum the 4 largest bins...
	sum = 0;
	for ( j = 0; j < 4; j++ ) {

		// Default to the bin not taken yet
		for ( idx = 0; chosen[ idx ]; idx++ ) {
		}
		bin = inSample[ idx ];

		// Find next largest bin, then 'choose' it
		for ( i = idx + 1; i < BINS_TO_COMPARE; i++ ) {
			if ( samp[ i ] > bin && chosen[ i ] == 0 ) {
				bin = samp[ i ];
				idx = i;
			}
		}

		sum += bin;
		chosen[ idx ] = 1;
	}
	mBass[ 0 ] = sum;


	// Calculate bass2 -- sum some low freq bins
	sum = 0;
	for ( i = 0; i < inNumBins >> 4; i++ )
		sum += samp[ i ];
	mBass[ 1 ] = sum;

	*/

/*
SJ:
RenderVisualData* data = messageInfo->u.renderMessage.renderData;
RecordFFT( data -> spectrumData[0], data -> spectrumData[1], kVisualNumSpectrumEntries );
*/

void PluginGlue::RecordFFT() {

	long i, numBins = mFFT_NumBins;

	// If no input is available, zero it out
	if ( ! mClientData ) {
		mFFT_Bass = 0;
		for ( i = 0; i < mFFT_NumBins; i++ )
			mFFT_Fcn -> mFcn[ i ] = 0;
		return;
	}

#if STANDALONE
	float newX;

	// Scale the integer term that controls to bin freq range into a usable value
	float binRange = .080 * mFFT_StepsPerBin;

	mSoundPort.GetSpectrum( numBins, binRange, .0174 * mFFT_Bin_Start, mFFT_Fcn -> mFcn );

	// Transform the FFT spectrum
	for ( i = 0; i < numBins; i++ ) {
		mFFT_X = mFFT_Fcn -> mFcn[ i ];
		mFFT_W = ( (float) i ) / ( (float) numBins );
		newX = mFFT_FadeAway * mFFT_Copy[ i ] + mSample_Scale * mFFT_TransformFcn.Evaluate();
		mFFT_Copy[ i ] = mFFT_Fcn -> mFcn[ i ] = newX;
	}
#else
	long fftNum	= mClientData -> mFFT_NumBins - mFFT_Bin_Start;
	if ( fftNum < 0 )
		fftNum = 0;

	// Limit number of bins if there's not enough mp3 player bins to go around
	if ( mFFT_NumBins * mFFT_StepsPerBin > fftNum )
		numBins = fftNum / mFFT_StepsPerBin;
	if ( numBins > mFFT_NumBins )
		numBins = mFFT_NumBins;

	// Calc, transform, and copy FFT spectrum
	long sum;
	long idx = mFFT_Bin_Start;

	for ( i = 0; i < numBins; i++ ) {
		sum = 0;

#if SOUNDJAM || WINAMP || SONIQUE || ULTRAPLAYER || AUDION || ITUNES || XMMS
		for ( long j = 0; j < mFFT_StepsPerBin; j++, idx++ ) {
			sum += (unsigned long) mClientData -> mSpectrum_L[idx];
			sum += (unsigned long) mClientData -> mSpectrum_R[idx];
		}
#endif

#if ENTHEOGEN || UNSANITY_ECHO
		for ( long j = 0; j < mFFT_StepsPerBin; j++, idx++ )
			sum += mClientData -> mSpectrum[idx];
#endif

#if XMMS
		// At the beginning XMMS delivers 0 or even negative
		// Catch these failures
		if (sum <= 0)
			{ sum = 1; }
#endif
		mFFT_X = ( (float) sum ) / mFFT_StepsPerBin;
		mFFT_W = ( (float) i ) / ( (float) numBins );
		mFFT_Fcn -> mFcn[ i ] =
			mFFT_FadeAway * mFFT_Fcn -> mFcn[ i ] +
			mSample_Scale * mFFT_TransformFcn.Evaluate();

//		if (i % 4 == 0) { printf("%7li", sum); }
//		if (i % 4 == 0) { printf("%7.4f ", mFFT_Fcn -> mFcn[ i ]); }
//		if (i % 4 == 0) { printf("%4i", lround(mFFT_Fcn -> mFcn[ i ])); }
	}
//	if (numBins > 0) { printf("\n"); }
#endif // else STANDALONE

	// If we didn't have enough bins to fill the whole sample, zero out what remains
	for ( i = numBins; i < mFFT_NumBins; i++ )
		mFFT_Fcn -> mFcn[ i ] = 0;

	// Flatten the ends of the sample a bit...
	if ( mFFT_NumBins >= 2 ) {
		mFFT_Fcn -> mFcn[ 0 ] *= .65;
		mFFT_Fcn -> mFcn[ 0 ] *= .85;

		mFFT_Fcn -> mFcn[ mFFT_NumBins - 2 ] *= .85;
		mFFT_Fcn -> mFcn[ mFFT_NumBins - 1 ] *= .65;
	}

	// Smooth the sample
	if ( mFFT_Smooth > 0 )
		XFloatList::GaussSmooth( mFFT_Smooth * (float) mFFT_NumBins,
			numBins, true, mFFT_Fcn -> mFcn );

	// Normalize the sample

	// Set the bass level variable
	/*
	float tb, b =	mFFT_Fcn -> mFcn[ (long) (.3 * mFFT_NumBins) ];
	tb = mFFT_Fcn -> mFcn[ (long) (.4 * mFFT_NumBins) ];
	if ( tb > b )
		b = tb;
	tb = mFFT_Fcn -> mFcn[ (long) (.5 * mFFT_NumBins) ];
	if ( tb > b )
		b = tb;
	mFFT_Bass = b;*/

	mFFT_Bass = mFFT_Fcn -> mFcn[ (long) (.2 * mFFT_NumBins) ] +
		mFFT_Fcn -> mFcn[ (long) (.4 * mFFT_NumBins) ] + 0;

	mFFT_Bass = 2.1 * pow( mFFT_Bass,  (float) 1.2 );
}

void PluginGlue::RecordSample() {

	float scale = SAMPLE_SCALE * mSample_Scale;
	long i, n, numSamples;

	// If no input is available, zero it out
	if ( ! mClientData ) {
		for ( i = 0; i < mSample_NumBins; i++ )
			mSample_Fcn -> mFcn[ i ] = 0;
		return;
	}

#if STANDALONE
	numSamples = mSample_NumBins;
	mSoundPort.GetSamples( mSample_Fcn -> mFcn, numSamples, scale );
#else
	numSamples = mClientData -> mSample_NumBins;
	if ( numSamples > mSample_NumBins )
		numSamples = mSample_NumBins;
	//long unused = mClientData -> mSample_NumBins - numSamples;
#endif

#if SOUNDJAM || AUDION || ITUNES
	long sample;

	for ( i = 0; i < numSamples; i++ ) {
		sample = ( mClientData -> mWaveform )[ 2*i ];
		mSample_Fcn -> mFcn[ i ] = scale * ( sample - 128 );
	}
#endif

#if WINAMP || SONIQUE  || UNSANITY_ECHO
	for ( i = 0; i < numSamples; i++ )
		mSample_Fcn -> mFcn[ i ] = scale * mClientData -> mWaveform[ 2*i ];
#endif

#if ENTHEOGEN
	for ( i = 0; i < numSamples; i++ )
		mSample_Fcn -> mFcn[ i ] = scale * mClientData -> mWaveform[ i ];
#endif

#if XMMS
	for ( i = 0; i < numSamples; i++ ) {
		mSample_Fcn -> mFcn[ i ] = scale *
			(mClientData -> mWaveform_L[ i ] + mClientData -> mWaveform_R[ i ]);
	}
#endif

	// Flatten the ends of the sample...
	n = numSamples / 25 + 1;
	if ( n <= numSamples ) {
		for ( i = 0; i < n; i++ ) {
			scale = sin( 1.5 * ( ( i / n ) * .7 + .3 ) );
			mSample_Fcn -> mFcn[ i ] *= scale;
			mSample_Fcn -> mFcn[ numSamples - i - 1 ] *= scale;
		}
	}

	// Smooth out the waveform if the prefs says so
	n = mSample_Fcn -> mNumFcnBins;
	if ( mSample_Smooth > 0 )
		XFloatList::GaussSmooth( mSample_Smooth * ( (float) n ) , n, true, mSample_Fcn -> mFcn );
}

void PluginGlue::SetSample_NumBins( long inNum ) {

	if ( inNum > 0 && inNum < 10000 ) {
		mSample_Fcn = (ExprUserFcn*) mSamplesBuf.Dim( sizeof( float ) * inNum + sizeof( ExprUserFcn ) + 32 );
		mSample_NumBins = inNum;
		mSample_Fcn -> mNumFcnBins = mSample_NumBins;

		for ( int i = 0; i < inNum; i++ )
			mSample_Fcn -> mFcn[ i ] = 0;
	}
}

void PluginGlue::SetFFT_NumBins( long inNum ) {

	if ( inNum > 0 && inNum < 1000 ) {
		mFFT_Fcn = (ExprUserFcn*) mFFTBuf.Dim( sizeof( float ) * inNum + sizeof( ExprUserFcn ) + 64 );
		mFFT_NumBins = inNum;
		mFFT_Fcn -> mNumFcnBins = mFFT_NumBins;

		for ( int i = 0; i < inNum; i++ )
			mFFT_Fcn -> mFcn[ i ] = 0;


#if STANDALONE
		mFFT_Copy = (float*) mFFT_CopyBuf.Dim( sizeof( float ) * inNum + 64 );
		for ( int i = 0; i < inNum; i++ )
			mFFT_Copy[ i ] = 0;
#endif
	}
}

void PluginGlue::SetWindowOutput( PP_WindowPtr inWind, const Rect* inRect, bool inFullscreen ) {

	// mDoingSetPortWin == true is a signal that another thread is in SetWindowOutput()
	if ( mDoingSetPortWin )
		return;

	Rect r;
	if ( inRect )
		r = *inRect;

	// If an invalid win rect, fix it, you monkey!
	if ( r.right - r.left < 20  || r.bottom - r.top < 20 || ! inRect ) {

		r.top		= mPrefs.GetPref( MCC4_TO_INT( "wTop" ) );
		r.left		= mPrefs.GetPref( MCC4_TO_INT( "wLft" ) );
		r.right		= mPrefs.GetPref( MCC4_TO_INT( "wRgt" ) );
		r.bottom	= mPrefs.GetPref( MCC4_TO_INT( "wBot" ) );

#if EG_MAC
		// Invalidate the window if it's offscreen...
		RgnHandle deskRgn = ::GetGrayRgn();
#if TARGET_API_MAC_CARBON
		Rect rgnBounds;
		GetRegionBounds(deskRgn, &rgnBounds);
		::SectRect( &rgnBounds, &r, &r );
#else
		::SectRect( &(*deskRgn) -> rgnBBox, &r, &r );
#endif
#endif // EG_MAC
	}

	// If no prefs avail (or an older version, use factory rect) or a bad win, hard code a size
	if ( mPrefs.GetPref( MCC4_TO_INT( "Vers" ) ) != mPluginInfo.mPrefCompatVersion ||
		r.right - r.left < 20 ||
		r.bottom - r.top < 20 ||
		r.left < -100 ||
		r.top < -100 ||
		r.left > 1700 ||
		r.top > 1500 )
	{
		r.top = 52;
		r.left = 16;
		r.right = 540;
		r.bottom = 400;
	}

	mVideoOutput = &mWindowOutput;

	mWindowOutput.SetOSWindow( inWind, mParentWind );

	mWindowOutput.Resize( r );

	SetOccupiedRect( inFullscreen );

	// Signal that this thread is done with SetWindowOutput()
	mDoingSetPortWin = false;
}

void PluginGlue::SetVideoOutput( VideoDevice* inDevice, bool inFullscreen ) {

	mVideoOutput	= inDevice;

	SetOccupiedRect( inFullscreen );
}

void PluginGlue::SetOccupiedRect( bool inFullScreen ) {

	Rect r;

	if ( ! mVideoOutput )
		return;

	mAtFullScreen = inFullScreen;

	mVideoOutput -> GetOutputRect( r );

	long x = r.right - r.left;
	long y = r.bottom - r.top;

	// The pane rect is the rect within inPort th plugin frame occupies
	mPaneRect = r;

	// mDispRect is the rect within inPort G-Force is drawing in (ex, the letterbox)
	// Change the disp rect if the desired size exceeds the pixel ceiling
	mDispRect = r;
	if ( y > mMaxDispSize.v ) {
		InsetRect( &mDispRect, 0, ( y - mMaxDispSize.v ) / 2 );
		y = mDispRect.bottom - mDispRect.top;
	}
	if ( x > mMaxDispSize.h ) {
		InsetRect( &mDispRect, ( x - mMaxDispSize.h ) / 2, 0 );
		x = mDispRect.right - mDispRect.left;
	}

	// Init our output port
	mVideoOutput -> SetOrigin( mDispRect.left - mPaneRect.left, mDispRect.top - mPaneRect.top );
	mVideoOutput -> SetDesiredFPS( mForeFPS );

	// Changing the port (and the resolution) may change the mouse cords
	EgOSUtils::GetMouse( mLastMousePt );

	// Remember what the plugin is dimensioned to
	mDimX = x;
	mDimY = y;

	// BeginFrame and EndFrame must encapsulate all blocks containing possible drawing
	mVideoOutput -> BeginFrame();
	{
		// Resize callback
		PortResized( x, y );

		// Refresh the window...
		Refresh( nil );

		// If we're starting the plugin...
		if ( mNeedsInit ) {
			Initialize();
			mNeedsInit = false;
		}
	}
	mVideoOutput -> EndFrame();

	// Init the cursor
	if ( mAtFullScreen )
		EgOSUtils::HideCursor();
	else
		EgOSUtils::ResetCursor();
}

void PluginGlue::GetWinRect( Rect& outRect ) {

	mWindowOutput.GetWindowRect( outRect );
}

void PluginGlue::Refresh( Rect* inRect ) {

	Rect r = mPaneRect;

	if ( mVideoOutput ) {

		if ( inRect )
			r = *inRect;
		else
			mVideoOutput -> Erase( false );

		if ( PositiveArea( r ) ) {
			OffsetRect( &r, - mPaneRect.left, - mPaneRect.top );
			RefreshRect( &r );
		}
	}
}

Point PluginGlue::GetFullscreenSize() {

	if ( mFullscreenMode < 0 || mFullscreenMode > 10 )
		mFullscreenMode = 0;

	return PluginPrefs::GetFullscreenSize( mFullscreenMode );
}

void PluginGlue::AssertFullscreenSettings() {

	// Don't change FS settings if we're already in fullscreen.
	if ( mAtFullScreen )
		return;

#if ITUNES
	VisualPluginData* handlerData = (VisualPluginData*) mRefCon;
	if ( handlerData ) {
		Point size = GetFullscreenSize();
		PlayerSetFullScreenOptions( handlerData->appCookie,
			handlerData ->appProc,
			8, 32, GetFullscreenDepth(),
			size.h, size.v );
	}
#endif

#if SONIQUE || ENTHEOGEN || ULTRAPLAYER || UNSANITY_ECHO
	return;
#endif

#if UNSANITY_ECHO
/*
	SInt16 depth	= mFullscreenDepth;
	SInt16 width	= mFullscreenSize.h;
	SInt16 height	= mFullscreenSize.v;
	FullScreenState st = Echo_BeginFullScreen( PLUGIN_ID, &depth, &width, &height );
*/
#endif

#if SOUNDJAM || AUDION
	VisHandlerData*	handlerData = (VisHandlerData*) mRefCon;

	if ( handlerData ) {
		Point size = GetFullscreenSize();
		PlayerSetFullScreenOptions
			( handlerData->appCookie, handlerData->playerProc,
			8, 32, GetFullscreenDepth(), size.h, size.v );
		}
#endif
}

void PluginGlue::SetFullscreen( bool inFullScreen ) {

	UtilStr str;

	if ( inFullScreen == mAtFullScreen )
		return;

	AssertFullscreenSettings();

#if ITUNES
	VisualPluginData* handlerData = (VisualPluginData*) mRefCon;
	if ( handlerData ) {
		if ( PlayerSetFullScreen( handlerData -> appCookie, handlerData -> appProc , inFullScreen ) == noErr )
			mAtFullScreen = inFullScreen;
	}
#endif

#if SONIQUE || ENTHEOGEN || ULTRAPLAYER || UNSANITY_ECHO
	return;
#endif

#if UNSANITY_ECHO
	/*
	SInt16 depth	= mFullscreenDepth;
	SInt16 width	= mFullscreenSize.h;
	SInt16 height	= mFullscreenSize.v;
	FullScreenState st = Echo_BeginFullScreen( PLUGIN_ID, &depth, &width, &height ); */
#endif

#if SOUNDJAM || AUDION
	VisHandlerData*	handlerData = (VisHandlerData*) mRefCon;

	if ( handlerData ) {
		long result = PlayerSetFullScreen(handlerData->appCookie, handlerData->playerProc, inFullScreen );
		if ( result == noErr )
			mAtFullScreen = inFullScreen;
	}
#endif

#if STANDALONE || WINAMP
	Point 		size;
	bool		ok;
	int		dispID;

	if ( inFullScreen && ! mAtFullScreen ) {

		// Make the device currently hosting the window the fullscreen device
		dispID = mWindowOutput.GetParentDisplayID();
		size = GetFullscreenSize();
		ok = mWindowOutput.EnterFullscreen( dispID, size, mFullscreenDepth );

		// The port changed weather we failed or not
		SetOccupiedRect( ok );

		// Default: mouse movement will not exit fullscreen mode
		mMouseWillAwaken = false;

		if ( ! ok ) {
			size = GetFullscreenSize();
			str.Assign( "Swtich to " );
			str.Append( (long) size.h );
			str.Append( "x" );
			str.Append( (long) size.v );
			str.Append( " failed.  Try a different fullscreen mode." );
			Println( &str );
		}
	}

	// If exiting from fullscreen
	else if ( ! inFullScreen && mAtFullScreen ) {

		// Restore the window
		mWindowOutput.ExitFullscreen();

		SetOccupiedRect( false );

		// Use this oppurtunity to store the cur window position
		StoreWinRect();
	}
#endif

	// Prevent sleep
	if ( ! mAtFullScreen )
		mLastActiveTime = mT;

	// Changing the port (and the resolution) may change the mouse cords
	EgOSUtils::GetMouse( mLastMousePt );

	// Redraw the entire window when you get a chance
	mVideoOutput -> Erase( false );

	// Init the cursor
	if ( mAtFullScreen )
		EgOSUtils::HideCursor();
	else
		EgOSUtils::ResetCursor();
}

/*
void PluginGlue::TakeScreenshot() {

	CEgFileSpec fileSpec;
}
*/

void PluginGlue::NewTrack( char* inTitle, char* inArtist, char* inAlbum  ) {

	long pos;

	if ( inTitle ) {

		// mTrackTextStartFcn can access this, so that's what will trigger track text
		mLastSongStart = mT;

		// Small hack: supress track text if the plugin just started (causes too much clutter on screen)
		// It's a hack in that we don't expect the text draw trigger fcn to want to draw at 30 secs into a track
		if ( mT < .5 )
			mLastSongStart -= 30;

		// See the track text portion of Draw()
		mTrackTextIsReset = true;

		mSongTitle.Assign( inTitle );
		mArtist.Assign( inArtist );
		mAlbum.Assign( inAlbum );

		// Chop off extension
		pos = mSongTitle.FindLastInstanceOf( '.' );
		if ( pos > 0 && mSongTitle.length() - pos <= 3 )
			mSongTitle.Keep( pos - 1 );

		// Look for matching script name.  Forcing ExecuteCmd() to PluginGlue ensures a script of the matching
		// name is run, not a matching config.
		PluginGlue::ExecuteCmd( mSongTitle );

		Refresh( nil );  }

	else {
		mArtist		.Wipe();
		mAlbum		.Wipe();
		mSongTitle	.Wipe();
	}
}

void PluginGlue::ReplaceTrackInfo( UtilStr& ioText ) {

	/* This is how a particle "accesses" the track info.  I'd like to have the text continuously
	updated, but the current way things are set up, it'd be hackish  */
	ioText.Replace( "#ARTIST#", mArtist.getCStr(), false );
	ioText.Replace( "#ALBUM#",  mAlbum.getCStr(), false );
	ioText.Replace( "#TITLE#",  mSongTitle.getCStr(), false );
}

void PluginGlue::IdleMonitor() {

	bool kybdPress = false;
	float pollDelay;
	float secsUntilSleep = mScrnSaverDelay - ( mT - mLastActiveTime );
	Point pt;
	bool screenSaverMode = mScrnSaverDelay > 0;

	// Calc time till next kybd poll (Don't waste time checking the kybd unless we've been idle a while)
	// Also don't bother rapildly checking the kybd until we're really close to going into screen saver mode
	if ( IsFullscreen() )
		pollDelay = .7;
	else if ( secsUntilSleep < 90 )
		pollDelay = secsUntilSleep / 120.0;
	else
		pollDelay = 10;

	// If it's time to poll for activity...
	if ( mT > mLastKeyPollTime + pollDelay && screenSaverMode ) {

		mLastKeyPollTime = mT;

#if EG_MAC

#if TARGET_API_MAC_CARBON
		::GetKeys( (long*) mCurKeys );
#else
		::GetKeys( mCurKeys );
#endif

		// If the keys are pressed then that counts as activity
		if ( ( mCurKeys[0] != mPastKeys[0] ) || ( mCurKeys[1] != mPastKeys[1] ) || ( mCurKeys[2] != mPastKeys[2] ) || ( mCurKeys[3] != mPastKeys[3] ) ) {
			kybdPress = true;
			mPastKeys[0] = mCurKeys[0];
			mPastKeys[1] = mCurKeys[1];
			mPastKeys[2] = mCurKeys[2];
			mPastKeys[3] = mCurKeys[3];
		}
#endif // EG_MAC
	}

	// Check for user activity
	EgOSUtils::GetMouse( pt );
	if ( pt.h != mLastMousePt.h || pt.v != mLastMousePt.v || kybdPress ) {
		mLastMousePt		= pt;
		mLastActiveTime		= mT;

		if ( mAtFullScreen && mMouseWillAwaken )
			SetFullscreen( false );
	}

	// If we're elligible to enter fullscreen then do it
	if ( screenSaverMode ) {
		if ( ! mAtFullScreen && ( mT - mLastActiveTime > mScrnSaverDelay ) ) {
			SetFullscreen( true );
			mMouseWillAwaken = true;
		}
	}

	// Don't let the computer sleep on us, dammit
	EgOSUtils::PreventSleep();
}

void PluginGlue::CacheFolder( const char* inFolder, FileSysIDList& outIDList, XLongList* outPlayList, long inNumLevels ) {

	FileObj		folderID;
	XLongList	tempList;

	if ( mMainFolder.GetItemID( FILE_SYS_ROOT_ID, inFolder, folderID ) == FILE_SYS_SUCCESS ) {

		// Clue the file sys in that we're gonna access this folder a lot
		mMainFolder.Cache( folderID, inNumLevels );

		// Get a listing of all the items in this folder and sort them
		mMainFolder.Catalog( folderID, tempList );

		outIDList.Assign( &mMainFolder, tempList, FILE_SYS_SORT_CI | FILE_SYS_STRIP_EXTNS );

		if ( outPlayList ) {
			outPlayList -> RemoveAll();
			for ( int i = outIDList.Count(); i > 0; i-- ) {

				outPlayList -> Add( i );
			}

			// Mix up the playlist, baby!
			outPlayList -> Randomize();
		}
	}
}

void PluginGlue::TakeScreenshot() {

	CEgFileSpec imageSpec;
	PixPort* screenCapture;

	if ( GetScreenshotSpec( imageSpec ) ) {

		Print( "Writing " );
		Print( imageSpec.GetFileName( true ) );
		Println( "..." );

		screenCapture = Screenshot_PixPort();
		if ( screenCapture )
			screenCapture -> WriteImage( imageSpec );
	}
}

bool PluginGlue::GetScreenshotSpec( CEgFileSpec& outSpec ) {

	CEgFileSpec rootSpec;
	UtilStr fileName;
	long i;

	if ( mMainFolder.FileObjToSpec( FILE_SYS_ROOT_ID, rootSpec ) ) {

		for ( i = 1; i < 100; i++ ) {
			fileName.Assign( "Snapshot " );
			fileName.Append( i );
			fileName.Append( ".bmp" );
			outSpec.AssignPathName( fileName.getCStr(), &rootSpec );

			// Once we find a unique file name, use that as the image file
			if ( ! outSpec.Exists() )
				return true;
		}
	}

	return false;
}

/*
#if STANDALONE

void PluginGlue::VideoExport_Start( PixPort& inSample ) {
	bool ok = false;

	// Stop capturing if capturing sound...
	mPlaybackFile.close();

	if ( ! IsFullscreen() ) {

		// Ask user for capture file to export
		if ( EgOSUtils::AskOpen( nil, mCaptureSpec, c55_CaptureType ) ) {
			mPlaybackFile.open( &mCaptureSpec );
			if ( mPlaybackFile.noErr() ) {

				if ( mPlaybackFile.GetLong() == PREFS_COMPAT_VERSION ) {

					// Read the time index...
					ResetTimeIndex( mPlaybackFile.GetLong() );

					// Skip to the next sample to sync our times right
					mNextT_MS = mT_MS;
					#pragma warning( "Fix this" )
					//ManageTimeIndex();
					ok = true;

					mShowConsole = false;

					mVideoExporter.StartExport( inSample );
				}
			}

			if ( ! ok )
				VideoExport_End();
		}
	}
}



void PluginGlue::VideoExport_End() {

	mPlaybackFile.close();
	mVideoExporter.EndExport();

	mShowConsole = true;

	ResetTimeIndex( EgOSUtils::CurTimeMS() );
}

#endif
*/

bool PluginGlue::ExecuteCmdCode( long inCode, bool inShiftKey ) {

	Point pt;
	bool handled = true;
	long n;

	switch ( inCode ) {

		case cNextFullscreenMode:
			mFullscreenMode++;
			pt = GetFullscreenSize();
			mTemp.Assign( "New mode: " );
			mTemp.Append( (long) pt.h );
			mTemp.Append( "x" );
			mTemp.Append( (long) pt.v );
			if ( IsFullscreen() )
				mTemp.Append( "   (Reenter fullscreen mode for changes)" );
			Println( &mTemp );

			// Inform the environment our new preferred screen settings
			AssertFullscreenSettings();
			break;

		case cShowHelp:
			if ( ! inShiftKey ) {

				// Force text to become visible even if it's hidden
				mShowConsole = true;

				ShowHelp();

				// Make the help text last longer than usual
				long num = mConsoleLines.Count();
				mLineExpireTimes[ num - 1 ] += 10000;  }
			else
				Show_HTML_Help();
			break;

		case cTakeSnapshot:
			TakeScreenshot();
			break;

		case cDispTrackTitle:
			StartTrackText();
			break;

		case cToggleConsole:
			if ( mShowConsole ) {
				Refresh( &mConsoleDirtyRgn );
				Println( "Console OFF" );
			}
			mShowConsole = ! mShowConsole;
			if ( mShowConsole )
				Println( "Console ON" );
			break;

		case cToggleFullsceen:
			SetFullscreen( ! IsFullscreen() );
			break;

		case cIncFFT_Smooth:
		case cDecFFT_Smooth:
			if ( inCode == cIncFFT_Smooth )
				mFFT_Smooth += .001;
			else if ( mFFT_StepsPerBin > 0 )
				mFFT_Smooth -= .001;
			if ( mFFT_Smooth < 0 )
				mFFT_Smooth = 0;
			Print( "FFT smoothing: " );
			Println( mFFT_Smooth );
			break;

		case cIncFFT_BinRange:
		case cDecFFT_BinRange:
			if ( inCode == cIncFFT_BinRange )
				mFFT_StepsPerBin++;
			else if ( mFFT_StepsPerBin > 1 )
				mFFT_StepsPerBin--;
			Print( "FFT range: " );
			Println( mFFT_StepsPerBin );
			break;

		case cIncFFT_Bin_Start:
		case cDecFFT_Bin_Start:
			n = ( inCode == cDecFFT_Bin_Start ) ? -1 : 1;
			if ( mFFT_Bin_Start > 10 )
				n *= 2;
			mFFT_Bin_Start += n;
			if ( mFFT_Bin_Start < 0 )
				mFFT_Bin_Start = 0;
			Print( "FFT start: " );
			Println( mFFT_Bin_Start );
			break;

		case cDecNumFFTBins:
		case cIncNumFFTBins:
			if ( inCode == cDecNumFFTBins )
				n = - 2;
			else
				n = + 2;
			SetFFT_NumBins( mFFT_NumBins + n );
			Print( "Num FFT bins: " );
			Println( mFFT_NumBins );
			break;

		case cDecNumSampBins:
		case cIncNumSampBins:
			if ( inCode == cDecNumSampBins )
				n = - 4;
			else
				n = + 4;
			SetSample_NumBins( mSample_NumBins + n );
			Print( "Num sample bins: " );
			Println( mSample_NumBins );
			break;

		case cDecSampScale:
		case cIncSampScale:
			if ( inCode == cIncSampScale )
				mSample_Scale *= 1.2;
			else if ( mSample_Scale > .0001 )
				mSample_Scale /= 1.2;
			else
				mSample_Scale = .0001;
			Print( "Sample scale: " );
			Println( mSample_Scale );
			break;

		case cFrameRate:
			mTemp.SetValue( mVideoOutput -> FPS() );
			mTemp.Append( " frames/sec " );
			Println( &mTemp );
			break;
	/*
		case cToggleCapture:
			if ( mCaptureFile.is_open() )
				mCaptureFile.close();
			else {
				if ( EgOSUtils::AskSaveAs( "Save data capture:", "Sound Capture", mCaptureSpec, c55_CaptureType ) ) {

					mCaptureFile.open( &mCaptureSpec );
					if ( mCaptureFile.noErr() ) {
						mCaptureFile.PutLong( PREFS_COMPAT_VERSION );
						mPrevCaptureT = mT; }
					else
						EgOSUtils::ShowFileErr( mCaptureSpec, mCaptureFile, false );
				}
			}
			break; */

#if STANDALONE
		/*
		case cResetTimeIndex:
			ResetTimeIndex( mT_MS );
			break; */
			/*
		case cToggleVideoExport:

			// Turn the exporter on/off
			if ( mVideoExporter.Exporting() )
				VideoExport_End();
			else if ( ! IsFullscreen() )
				mStartExport = true;
			else
				Println( "You can't export video while in fullscreen mode" );
			break;*/
#endif


		default:
			handled = false;
	}

	return handled;
}

bool PluginGlue::ExecuteCmd( UtilStr& inCmd ) {

	bool ok = false;
	FileObj itemID;

	// Look for a matching script file...
	itemID = mScripts.FindID( &inCmd );
	if ( itemID ) {
		CEgFileSpec spec;

		if ( mMainFolder.FileObjToSpec( itemID, spec ) )
			ok = ExecuteScript( spec );
	}

	return ok;
}

bool PluginGlue::ExecuteScript( CEgFileSpec& inSpec ) {

	bool ok;
	TimeBasedScript* newScript;

	newScript = new TimeBasedScript;
	ok = newScript -> Assign( &inSpec );
	if ( ok ) {
		newScript -> Start( mT );
		mRunningScripts.addToHead( newScript );
	}

	return ok;
}

void PluginGlue::QueueCmdList( const UtilStr& inCmdList ) {

	if ( inCmdList.length() ) {
		TimeBasedScript* script = new TimeBasedScript;

		script -> Assign( inCmdList );
		script -> Start( mT );
		mRunningScripts.addToHead( script );
	}
}

bool PluginGlue::ExecuteKey( long inChar, bool inShiftKey, bool inCtrlKey ) {

	bool handled = true;
	long command;

	// Make the char uppercase
	if ( inChar >= 'a' && inChar <= 'z' )
		inChar = 'A' + ( inChar - 'a' );

	// ESC key
	if ( inChar == 27 )
		SetFullscreen( false );

	// RETURN or ENTER key
	else if ( inChar == 13 || inChar == 3 ) {
		SetFullscreen( ! IsFullscreen() ); }

	else if ( inChar >= ' ' && inChar < 256 ) {

		// CONTROL and SHIFT mean "store state as key"
		if ( inCtrlKey && inShiftKey )
			handled = StoreStateAsScript( inChar );

		// Just inCtrlKey means play key script
		else if ( inCtrlKey )
			handled = ExecuteScriptFromKey( inChar, false );

		else {
			command = mKeyMap.FindNextInstanceOf( 0, inChar );
			handled = ExecuteCmdCode( command, inShiftKey );
		}
	}

	else
		handled = false;

	return handled;
}

void PluginGlue::ExecuteCmdList( UtilStr& cmdList ) {

	UtilStr cmd;
	long pos;

	cmdList.Capitalize();

	// Let's light this candle!!
	goto doneCmd;

	while ( cmdList.length() ) {

		// Move the cmd str from the pref str to a temp str
		cmd.Assign( cmdList, pos - 1 );
		cmdList.Trunc( pos, false );

		// Pass the cmd string off to be executed
		if ( cmd.length() == 1 )
			ExecuteKey( cmd.getChar( 1 ), false, false );
		else
			ExecuteCmd( cmd );

doneCmd:
		// Find the end of the next command line
		pos = cmdList.FindNextInstanceOf( 0, ';' );
		if ( pos == 0 )
			pos = cmdList.length() + 1;
	}
}

bool PluginGlue::ExecuteScriptFromKey( long inKey, bool inSuppressMsgs ) {

	bool handled;

	// The script we'll run is "'X' Key"
	UtilStr scriptTitle;
	scriptTitle.Append( (char) inKey );
	scriptTitle.Append( " Key" );

	// Look for (and run) a config or script named "Keyboard X"
	handled = ExecuteCmd( scriptTitle );

	if ( handled ) {
		Print( "Started script: " );
		Println( &scriptTitle ); }
	else if ( ! inSuppressMsgs ) {
		Print( "Error loading script: " );
		Println( &scriptTitle );
	}

	return handled;
}

bool PluginGlue::StoreStateAsScript( long inKey ) {

	FileObj scriptObj;
	bool result;

	// Generate a list of commands that will reproduce the current state
	UtilStr cmdLine;
	MakeStateCmdLine( cmdLine );
	cmdLine.Prepend( "0:00   " );

	// We'll name the script "'X' Key"
	UtilStr scriptTitle;
	scriptTitle.Append( (char) inKey );
#if EG_WIN
	scriptTitle.Append( " Key.txt" );
#else
	scriptTitle.Append( " Key" );
#endif

	// Actually write the scipt file
	long temp = CEgIOFile::sCreatorType;
	CEgIOFile::sCreatorType = MCC4_TO_INT( "ttxt" );
	result = mMainFolder.Create( mScriptFolderID, scriptTitle, scriptObj, FILE_SYS_DATA );
	result = mMainFolder.WriteData( scriptObj, cmdLine.getCStr(), cmdLine.length() );
	CEgIOFile::sCreatorType = temp;

	if ( result ) {

		Print( "Wrote script: " );
		Println( &scriptTitle );

#if EG_WIN
		scriptTitle.Trunc( 4 );
#endif

		// Put the script in out script list (so we can find it later)
		mScripts.RemoveItem( &scriptTitle, false );
		mScripts.Add( &scriptTitle, scriptObj, FILE_SYS_STRIP_EXTNS ); }
	else {
		Print( "Error writing script: " );
		Println( &scriptTitle );
	}

	return result;
}

void PluginGlue::ManageTimeScripts() {

	// Keep fetching command lines until we're up to date...
	TimeBasedScript *next, *script = (TimeBasedScript*) mRunningScripts.GetHead();
	while ( script ) {
		next = (TimeBasedScript*) script -> GetNext();

		// Exhaust each scripts of commands due
		while ( script -> GetNextCmd( mT, mTemp ) ) {
			ExecuteCmdList( mTemp );
		}

		// Remove/delete the script if we've finished it
		if ( script -> Done() )
			delete script;

		script = next;
	}
}

void PluginGlue::KillRunningScripts() {

	mRunningScripts.deleteContents();
}

void PluginGlue::Show_HTML_Help() {

	bool ok = false;
	FileObj itemID;
	CEgFileSpec spec;

	SetFullscreen( false );

	if ( mMainFolder.GetItemID( FILE_SYS_ROOT_ID, "Documentation.html", itemID ) ) {
		if ( mMainFolder.FileObjToSpec( itemID, spec ) ) {
			ok = true;
			Println( "Launching documentation, please wait..." );
			if ( ! EgOSUtils::LaunchHTMLFile( spec, mPluginInfo.mID ) )
				Println( "Error launching Documentation.html" );
		}
	}
	if ( ! ok ) {
		UtilStr msg;
		mMainFolder.GetLastError( msg );
		Println( &msg );
	}
}

#define PIX_PER_LINE 11
void PluginGlue::DrawConsole( ConsolePixPort *inPort ) {

	long i, start, numLines = mConsoleLines.Count(), width, height;
	long top = 3;
	UtilStr* theLine;

	// Init the dirty rect
	::SetRect( &mConsoleDirtyRgn, 0, 0, -1, top + 2 );

	// Don't draw console if text is toggled off
	if ( mShowConsole && numLines > 0 ) {

		// Clear the clip rect
		inPort -> SetClipRect();

		inPort -> SetConsoleFont();

		// Check if console runs off the display rect...
		if ( numLines * PIX_PER_LINE > mPaneRect.bottom - mPaneRect.top )
			start = 1 + numLines - ( mPaneRect.bottom - mPaneRect.top ) / PIX_PER_LINE;
		else
			start = 1;

		// Draw each line of the console...
		for ( i = start; i <= numLines; i++ ) {
			theLine = mConsoleLines.Edit( i );
			inPort -> DrawText( 4, top + (i-start) * PIX_PER_LINE, *theLine );

			// Keep track of what region we're drawing over
			mConsoleDirtyRgn.bottom += PIX_PER_LINE;
			inPort -> TextRect( theLine -> getCStr(), width, height );
			width += 4 + 2;
			if ( width > mConsoleDirtyRgn.right )
				mConsoleDirtyRgn.right = width;
		}

		inPort -> SetDefaultFont();
	}
}

void PluginGlue::RemoveExpiredConsoleLines() {

	long numLines = mConsoleLines.Count();
	bool refreshOrdered = false;

	// Delete console lines that are too old...
	while ( mLineExpireTimes.Fetch( 1 ) < mT_MS && numLines > 0 ) {
		mConsoleLines.Remove( 1 );
		mLineExpireTimes.RemoveElement( 1 );
		numLines--;

		// Force the region containing text to be redrawn
		if ( ! refreshOrdered ) {

			// Force the region containing text to be redrawn (Note: Refresh is in port, not frame, cords)
			OffsetRect( &mConsoleDirtyRgn, mPaneRect.left, mPaneRect.top );
			Refresh( &mConsoleDirtyRgn );
			__invalidate( mConsoleDirtyRgn )

			refreshOrdered = true;
		}
	}
}

void PluginGlue::Print( const char* inStr ) {

	long num = mConsoleLines.Count();
	UtilStr* lastLine = mConsoleLines.Edit( num );

	// Append the text to the console text..
	if ( lastLine )
		lastLine -> Append( inStr );
	else {
		mConsoleLines.Add( inStr );
		num = 1;
	}

	// Setup when this line will be deleted
	mLineExpireTimes[ num - 1 ] = mT_MS + mConsoleLineDur * 1000;

	// Force the region containing text to be redrawn  (Note: Refresh is in port, not frame, cords)
	OffsetRect( &mConsoleDirtyRgn, mPaneRect.left, mPaneRect.top );
	Refresh( &mConsoleDirtyRgn );
	__invalidate( mConsoleDirtyRgn )
}

void PluginGlue::Print( char inChar ) {

	char str[ 2 ];

	str[ 0 ] = inChar;
	str[ 1 ] = 0;
	Print( str );
}

void PluginGlue::Println( const char* inStr ) {

	Print( inStr );
	mConsoleLines.Add( "" );
}

void PluginGlue::Println( float inNum ) {

	mTemp.SetFloatValue( inNum );

	Println( &mTemp );
}

void PluginGlue::CommandClick()  {

}

#if EG_MAC
void PluginGlue::HandleEvt( WindowPtr inWin, EventRecord& evt ) {

	static long sLastWhen = 0;
	long 	curTime = ::TickCount();
	Point	pt;
	GrafPtr origPort;

	switch ( evt.what ) {

		case updateEvt:
			if ( inWin == (WindowPtr) evt.message ) {
				::GetPort( &origPort );
#if TARGET_API_MAC_CARBON
				::SetPortWindowPort( inWin );
#else
				::SetPort( inWin );
#endif
				::BeginUpdate( inWin );
				Refresh( nil );
				::EndUpdate( inWin );
				::SetPort( origPort );
			}
			break;

		case mouseDown:
			pt = evt.where;
			if ( evt.modifiers & cmdKey ) {

				// If we're in full screen mode, show the mouse ptr
				if ( mAtFullScreen )
					EgOSUtils::ShowCursor();

				::GetPort( &origPort );
#if TARGET_API_MAC_CARBON
				::SetPortWindowPort( inWin );
#else
				::SetPort( inWin );
#endif
				CommandClick();
				::SetPort( origPort );  }
			if ( IsFullscreen() )
				SetFullscreen( false );
			else if ( curTime - sLastWhen < ::GetDblTime() )
				SetFullscreen( true );
			else if ( ! IsFullscreen() ) {
#if STANDALONE
				Rect r, growBox, lim;
				GetWinRect( r );
				growBox = r;
				growBox.left = r.right - 35;
				growBox.top = r.bottom - 35;
				if ( ::PtInRect( pt, &growBox ) ) {
					lim.top = lim.left = 30;
					lim.bottom = lim.right = 9000;
					unsigned long newSize = ::GrowWindow( inWin, pt, &lim );
					::SizeWindow( inWin, newSize & 0xFFFF, newSize >> 16, false );
					GetWinRect( r );
					SetWindowOutput( inWin, &r, false );	}	// SizeWindow changes the size of the win, but gWC doesn't know that
				else if ( BorderlessWindow() && ::Button() ) {
					lim.top = lim.left = 0;
					lim.bottom = lim.right = 9000;
					::DragWindow( inWin, pt, &lim );
				}
#endif
				sLastWhen = curTime;
			}
			break;

	}
}
#endif // EG_MAC
