/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifndef __SINGIT_SONG_H__
#define __SINGIT_SONG_H__

#include <glib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_ID3
#include <id3.h>
#endif

#include <semaphore.h>

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

typedef enum _LyricType {

	lt_none = 0,
	lt_text,
	lt_id3v2xx,
	lt_midi
}
LyricType;

typedef struct {

	guint line;	// Lyric line
	guint time;	// Play time in milliseconds
	guint pos;	// Used by synchronized lyrics to get the position in the line

}
LToken;

typedef struct {

	GList  *first_token;
	GList  *last_token;
	GList  *active_token;

	gchar **lyrics;         // Lyrics text
	guint   lyric_lines;    // Number of lyric lines

	gchar  *song_filename; 	// Filename of the song
	gchar  *lyric_filename; // Filename of the lyrics text
	guint   song_length;    // Length in ms
	gulong  file_size;
	guchar  sha_digest[20]; // Hash of the textfile

	gchar *delimiter;       // Delimiter type of text
	LyricType lyric_type;

#ifdef HAVE_ID3
	ID3Tag *id3tag;
#endif

	guint attachments;	// usage control via attach
	gint song_sema_id;	// semaphore controlled attachments
}
LSong;

#define tTime(a) (((LToken*) (a->data))->time)
#define tLine(a) (((LToken*) (a->data))->line)
#define tPos(a) (((LToken*) (a->data))->pos)
#define tText(a, b) (((LSong *) a)->lyrics[((LToken*) (b->data))->line])

#define l_song_is_last_token(a) (((LSong *) a)->active_token) == last_token)
#define l_song_is_first_token(a) (((LSong *) a)->active_token) == first_token)
#define l_song_set_length(a, b) (((LSong *) a)->song_length) = b)

#define l_song_text_found(a) (((LSong *) a)->lyrics != NULL)
#define l_song_lyrics_found(a) (((LSong *) a)->first_token != NULL)

LSong* l_song_new(gchar* song_filename);
void l_song_clear(LSong* song);
void l_song_free(LSong* song);

inline LSong* l_song_attach(LSong *song);
inline LSong* l_song_detach(LSong *song, gboolean free);

/*
  Returns the position of the time in the token list compared to the current token
  Results:	-1 = time is before the current
	0 = time is between the current and the next token
	1 = time is after the next token
*/
gint l_song_is_time_ok(LSong *song, gint time);
inline gint inl_l_song_is_time_ok(LSong *song, gint time);
inline gboolean l_song_is_empty_item(LSong *song, GList *item);

/* Returns the first time with the line */
gint l_song_find_line(LSong* song, gint line);
/* Returns the first line with the time */
gint l_song_find_time(LSong* song, gint time);

/*
  Returns the token directly before the time (current token)
  State is most time the result of "l_song_is_time_ok"
*/
GList* l_song_find_current_token(LSong *song, gint time, gint state);
inline GList *inl_l_song_get_next_token(LSong *song);
inline GList *inl_l_song_get_prev_token(LSong *song);

/*
  Returns the token with the next lyric line.
  Set Empty if you accept an empty line
  Set hops if you want to get the item distance else use NULL
*/
GList* l_song_find_next_lyric_line(LSong *song, GList *item, gboolean empty, guint *hops);
GList* l_song_find_prev_lyric_line(LSong *song, GList *item, gboolean empty, guint *hops);

/* Tells you if the lyrics file has changed size or hash */
gboolean l_song_lyrics_changed(LSong *song);

/*
  Get the length of the of the text to the next string
  Result:	-2 = Error in the lyrics file should not happen in "good" lyrics
  	-1 = To the end of line
	> 0 = Length of substring
*/
gint l_song_get_text_length(LSong *song);
inline gint inl_l_song_get_text_length(LSong *song);

#ifdef HAVE_ID3
/* Refreshs the id3_tag */
void l_song_get_id3_tag(LSong *song, gchar *filename);
#endif

/*
  Checks if every later timetag in the song list has a greater line number or text offset
*/
gint l_song_check_sync_lyric_consistency(LSong *song);

/*
  Adds an overall time to a song (adding < 0 = sub)
*/
void l_song_modify_overall_time(LSong *song, gint time);

/*
  Guess if the lyrics are syncronized by checking every tPos > 0
*/
gboolean l_song_guess_sync_lyrics(LSong *song);

#ifdef __cplusplus
}
#endif // __cplusplus

#endif // __SINGIT_SONG_H__
