/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>

#include "utilities.h"
#include "cpicture.h"

gint32 ToL (gchar *puffer) { return (puffer[0] | puffer[1]<<8 | puffer[2]<<16 | puffer[3]<<24); }
gint16 ToS (gchar *puffer) { return (puffer[0] | puffer[1]<<8); }

inline unsigned short get_big_u_short(void *p) {
    unsigned short *s_p = (unsigned short*)p;
    return (*s_p >> 8) | (*s_p << 8);
}

typedef struct
{
	guint8 idLength;
	guint8 colorMapType;
	guint8 imageType;
	guint8 colorMapIndexLo, colorMapIndexHi,
		colorMapLengthLo, colorMapLengthHi,
		colorMapSize;
	guint8 origX[2]; // lo, hi
	guint8 origY[2]; // lo, hi
	guint8 width[2];
	guint8 height[2];
	guint8 bpp;
	guint8 imageDes; // don't use, space eater
}
tgaHeader_t;
/*
ID Length - Field 1 (1 byte): Byte-length of Field 6, the Image ID Field.

Color Map Type - Field 2 (1 byte):
  0 -	indicates that no color-map data is included with this image.
  1 -	indicates that a color-map is included with this image.

Image Type - Field 3 (1 byte):
  0 - No image data includet
  1 - Uncompressed color-map (CM) image
  2 - Uncompressed true color (TC) image
  3 - Uncomp black + white
  9 - Run length encoded (RLE) CM img
 10 - RLE TC image
 11 - RLE BW image

Color Map Specification - Field 4 (5 bytes):
Field 4.1 (2 bytes) -	First Entry Index:
Field 4.2 (2 bytes) -	Color map Length:
Field 4.3 (1 byte) -	Color map Entry Size: 15, 16, 24 or 32-bit values are used.

Image Specification - Field 5 (10 bytes):
Field 5.1 (2 bytes) -	X-origin of Image:
Field 5.2 (2 bytes) -	Y-origin of Image:
Field 5.3 (2 bytes) -	Image Width:
Field 5.4 (2 bytes) -	Image Height:
Field 5.5 (1 byte) -	Pixel Depth:
Field 5.6 (1 byte) -	Image Descriptor:
  bpp 3-0: Alpha Channel bpp
  bpp 5-4: Picture Transfer
    bot - left: 0 / 0
    bot - right : 0 / 1
    top - left: 1 / 0
    top - right : 1 / 1
  bpp 7-6: Must be 0

Image ID - Field 6 (variable):
Color Map Data - Field 7 (variable):
Image Data - Field 8 (variable):
Developer Data - Field 9 (variable):
*/
typedef struct
{
	guint32 extensionAreaOffset;
	guint32 developerDirectoryOffset;
	#define TGA_SIGNATURE "TRUEVISION-XFILE"
	gchar signature[16];
	gchar dot;
	gchar null;
}
tgaFooter_t;

enum tga_types
{
	unknown = 0,
	mapped = 1,
	color = 2,
	gray = 3,
	mapped_rle = 9,
	color_rle = 10,
	gray_rle = 11
};

enum tga_descriptor
{
	abpp = 0x0f,
	horizontal = 0x10,
	vertical = 0x20
};

typedef struct
{
	gchar    zzMagic[2];	/* 00 "BM" */
	guint32  bfSize;      /* 02 */
	guint16  zzHotX;	/* 06 */
	guint16  zzHotY;	/* 08 */
	guint32  bfOffs;      /* 0A */
	guint32  biSize;      /* 0E */
	guint32  biWidth;     /* 12 */
	guint32  biHeight;    /* 16 */
	guint16  biPlanes;    /* 1A */
	guint16  biBitCnt;    /* 1C */
	guint32  biCompr;     /* 1E */
	guint32  biSizeIm;    /* 22 */
	guint32  biXPels;     /* 26 */
	guint32  biYPels;     /* 2A */
	guint32  biClrUsed;	/* 2E */
	guint32  biClrImp;	/* 32 */
				/* 36 */
}
bmpHeader_t;

typedef struct {

	unsigned short magic;		// magic?

	char compress;
	char bpp;

	unsigned short dimension;		// 3?

	unsigned short width;
	unsigned short height;
	unsigned short format;		// depth(GRAY,GRAY Alpha,RGB,RGB Alpha)

	char unknown[512 - 12];
}
sgiHeader_t;

enum {
	SGI_MAGIC = 0x01da,

	SGI_COMPRESS_NONE = 0x00,
	SGI_COMPRESS_RLE = 0x01,
	SGI_COMPRESS_ARLE = 0x02,

	SGI_FORMAT_GRAY = 0x01,
	SGI_FORMAT_GRAYA = 0x02,
	SGI_FORMAT_RGB = 0x03,
	SGI_FORMAT_RGBA = 0x04,
};

CPicture::CPicture()
{
	type = PT_NON;
	bpp = width = height = size = lastError = 0;
	data = 0;
}

CPicture::CPicture(gchar i_bpp, guint32 i_width, guint32 i_height, const gchar *i_data)
{
//	printf("CPicture\n");
	type = PT_RAW;
	bpp = i_bpp;
	width = i_width;
	height = i_height;
	size = i_width * i_height * i_bpp / 8;
	data = new gchar[size];
	memcpy(data, i_data, size);
}

CPicture::~CPicture()
{
//	printf("~CPicture\n");
	this->free();
}

void CPicture::free()
{
//	printf("Test\n");
	if (data != 0) {
		delete [] data;
		data = 0;
	}
	bpp = width = height = size = 0;
}

#define RLE_PACKETSIZE 0x80

/* Decode a RLE encoded TGA. */
gint32 CPicture::decodeRLE (const gchar *buffer, guint32 buf_size)
{
	gchar *buffer_offset = (gchar*) buffer;
	guint32 bytesPerPixel = (bpp / 8);
	guint32 data_size = size * bytesPerPixel;
	data = new gchar[data_size];
	gchar *data_offset = data;
	guint32 i = 0, count;

	while ((guint32)(buffer_offset - buffer) < buf_size) {
		count = buffer_offset[0];
		if ((guint32)(data_offset - data) < data_size) { return -1; }

		if (count > ((data - data_offset + data_size) / bytesPerPixel)  ) { return -1; }
		else {
			for (i = 0; i < count; i++) {
				memcpy(data_offset, buffer_offset+1, bytesPerPixel);
				data_offset += bytesPerPixel;
			}
			buffer_offset += (bytesPerPixel + 1);
		}
	}
	return 0;
}

gint32 CPicture::TGA_ParseBuffer(const gchar *buffer, guint32 buf_size)
{
	gchar *where = (gchar*) buffer;
	tgaHeader_t *header = (tgaHeader_t *) where;

	where += (sizeof(tgaHeader_t) + header->idLength);

	if (header->colorMapType != 0) { return -1; }
	bpp = header->bpp;
	if (bpp != 32 && bpp != 24 && bpp != 8) { return -2; }
	width = header->width[0] + header->width[1] * 256;
	height = header->height[0] + header->height[1] * 256;
	size = width * height;

	switch (header->imageType) {
	case 2:
	case 3:
		data = (gchar*) grabData(where, size);
		break;
	case 10:
	case 11:
		decodeRLE(buffer, buf_size);
		break;
	default:
		return -1;
	}

	// no image data
	if (data == 0) { return -3; }

	return 0;
}

gint32 CPicture::BMP_ParseBuffer(const gchar *buffer, guint32 buf_size)
{
	unsigned long i;       // standard counter.
	char temp;             // temporary color storage for bgr-rgb conversion.

	gchar *where = (gchar*) buffer;
	bmpHeader_t header;

	if (buf_size < 0x36) { return -1; }
	if (strncmp((const char*) where,"BM",2)) { return -1; }

	header.bfSize = ToL (&where[0x02]);
	header.zzHotX = ToS (&where[0x06]);
	header.zzHotY = ToS (&where[0x08]);
	header.bfOffs = ToL (&where[0x0a]);
	header.biSize = ToL (&where[0x0e]);
	where += 18;

	switch (header.biSize) {
	case 12:  // OS/2 1.x
		header.biWidth   = ToS (&where[0x00]);       /* 12 */
		header.biHeight  = ToS (&where[0x02]);       /* 14 */
		header.biPlanes  = ToS (&where[0x04]);       /* 16 */
		header.biBitCnt  = ToS (&where[0x06]);       /* 18 */
		where += 8;
		break;
	default:
		if (header.biSize <= 64) {
			header.biWidth   =ToL (&where[0x00]);	/* 12 */
			header.biHeight  =ToL (&where[0x04]);	/* 16 */
			header.biPlanes  =ToS (&where[0x08]);	/* 1A */
			header.biBitCnt  =ToS (&where[0x0A]);	/* 1C */
			header.biCompr   =ToL (&where[0x0C]);	/* 1E */
			header.biSizeIm  =ToL (&where[0x10]);	/* 22 */
			header.biXPels   =ToL (&where[0x14]);	/* 26 */
			header.biYPels   =ToL (&where[0x18]);	/* 2A */
			header.biClrUsed =ToL (&where[0x1C]);	/* 2E */
			header.biClrImp  =ToL (&where[0x20]);	/* 32 */
			where += header.biSize - 4;
		}
		else { return -1; }
	}
	bpp = header.biBitCnt;
	if (header.biPlanes != 1) { return -1; }
	if ((bpp != 8) && (bpp != 24) && (bpp != 32)) { return -1; }

	width = header.biWidth;
	height = header.biHeight;
	size = width * height;

	if (buf_size < 0x36 + size * bpp / 8) { return -1; }

	data = (gchar*) grabData(where, size);

	if ((bpp != 24) || (bpp != 32))
		for (i = 0; i < size; i+=(bpp / 8)) {	// bgr -> rgb
			temp = data[i];
			data[i] = data[i+2];
			data[i+2] = temp;
		}

	return 0;
}

gint32 CPicture::SGI_ParseBuffer(const gchar *buffer, guint32 buf_size)
{
	sgiHeader_t *header_p = (sgiHeader_t*) buffer;
	gchar *data_p = (gchar*) buffer;

	data_p += sizeof(sgiHeader_t);

	if ((get_big_u_short(&header_p->magic) == SGI_MAGIC) &&
		(header_p->compress == SGI_COMPRESS_NONE) &&
		(get_big_u_short(&header_p->format) == SGI_FORMAT_RGBA))
	{
		bpp = 32;
		width = get_big_u_short(&header_p->width);
		height = get_big_u_short(&header_p->height);
		size = width * height * bpp / 8;

//		g_print("p: %i / %i\n", width, height);

		data = new gchar[size];

		for (guint t = 0;t != height; t++) {
			for (guint s = 0;s != width; s++) {
				*(data + 0 + s*4 + t*4*width) = *(data_p + s + t*width + 0*width*height);
				*(data + 1 + s*4 + t*4*width) = *(data_p + s + t*width + 1*width*height);
				*(data + 2 + s*4 + t*4*width) = *(data_p + s + t*width + 2*width*height);
				*(data + 3 + s*4 + t*4*width) = *(data_p + s + t*width + 3*width*height);
			}
		}
	}
	else {
		return -1;
	}

	return 0;
}

gint32 CPicture::RAW_ParseBuffer(const gchar *buffer, guint32 buf_size)
{
	size = buf_size;
	data = new gchar[size];
	memcpy(data, buffer, size);
	return 0;
}

bool CPicture::loadData(const gchar *data_p, const guint size, PicType i_type)
{
	gint32 ret = 1;

	type = i_type;

	switch (type) {
	case PT_TGA: ret = TGA_ParseBuffer(data_p, size); break;
	case PT_BMP: ret = BMP_ParseBuffer(data_p, size); break;
	case PT_SGI: ret = SGI_ParseBuffer(data_p, size); break;
	case PT_RAW: ret = RAW_ParseBuffer(data_p, size); break;
	case PT_NON:
	default:
		ret = -3;
	}

	if (ret) { lastError = ret; }
	return (ret == 0);
}

bool CPicture::loadFile(const gchar *name, PicType i_type)
{
	FILE *iFile = 0;
	gchar *buffer = 0;
	guint32 fSize;
	gint32 ret = 0;

	if (!(iFile = fopen((const char*) name, "rb"))) { return false; }
	fSize = FileGetSize (iFile);
	if (!(buffer = new gchar[fSize + 1])) {
		fclose(iFile);
		return false;
	}
	fread(buffer, 1, fSize, iFile);
	fclose(iFile);

	type = i_type;

	switch (type) {
	case PT_TGA: ret = TGA_ParseBuffer(buffer, fSize); break;
	case PT_BMP: ret = BMP_ParseBuffer(buffer, fSize); break;
	case PT_SGI: ret = SGI_ParseBuffer(buffer, fSize); break;
	case PT_RAW: ret = RAW_ParseBuffer(buffer, fSize); break;
	case PT_NON:
	default:
		ret = -3;
	}

	delete [] buffer;
	if (ret) { lastError = ret; }
	return (ret == 0);
}

gchar* CPicture::grabData(const gchar *buff, guint32 size, guint32 fakesize = 0)
{
	guint32 bytesPerPixel = bpp / 8;
	gchar *result = new gchar[size * bytesPerPixel];
	if (fakesize >= 8) { bytesPerPixel = fakesize / 8; }

	if (result == 0) { return 0; }
	if (!memcpy(result, buff, size * bytesPerPixel)) {
		delete [] result;
		return 0;
	}

	if (bytesPerPixel > 1) {
		gchar temp;
		for (guint32 i = 0; i < size * bytesPerPixel; i += bytesPerPixel) {
			temp = result[i];
			result[i] = result[i + 2];
			result[i + 2] = temp;
		}
	}

	return result;
}

guint32 CPicture::operator[] (guint32 ndx)
{
	guint32 result = 0;

	if (ndx > size) { return 0; }

	switch (bpp) {
	case 32:
	case 24:
		memcpy(&result, &data[ndx], bpp);
		break;
   	case 8:
		result = data[ndx];
		break;
	}
	return result;
}

bool CPicture::setRAW(gchar i_bpp, guint32 i_width, guint32 i_height)
{
	if (type != PT_RAW) { return false; }

	bpp = i_bpp;
	width = i_width;
	height = i_height;

	return true;
}