/* gui.c
 * Copyright (C) 2006 Gauvain Pocentek <gauvainpocentek@gmail.com>
 * 
 * A part of this file comes from the gnome keyboard capplet (control-center):
 * Copyright (C) 2003 Sergey V. Oudaltsov <svu@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include <libxfcegui4/libxfcegui4.h>

#include "xfkc.h"
#include "gui.h"

// from the gnome control center keyboard applet
char *
xci_desc_to_utf8 (XklConfigItem * ci)
{
  char *sd = g_strstrip (ci->description);
  return sd[0] == 0 ? g_strdup (ci->name) :
    g_locale_to_utf8 (sd, -1, NULL, NULL, NULL);
}
//

void
save_settings ()
{
	gchar *layouts, *variants, *kbdmodel;
	model = GTK_TREE_MODEL (xfkc_gui->layout_store);
	
	gtk_tree_model_get_iter_first (model, &iter);
	gtk_tree_model_get (model, &iter, LAYOUTS, &layouts, VARIANTS, &variants, -1);
	xfkc_settings->layouts = layouts;
	if (variants != NULL)
		xfkc_settings->variants = variants;
	else
		xfkc_settings->variants = "";
	
	while (gtk_tree_model_iter_next (model, &iter))
	{
		gtk_tree_model_get (model, &iter, LAYOUTS, &layouts, VARIANTS, &variants, -1);
		xfkc_settings->layouts = g_strdup(g_strconcat(xfkc_settings->layouts, ",", layouts, NULL));
		if (variants != NULL)
			xfkc_settings->variants = g_strdup(g_strconcat(xfkc_settings->variants, ",", variants, NULL));
		else
			xfkc_settings->variants = g_strdup(g_strconcat(xfkc_settings->variants, ",", NULL));
	}

	//
	model = GTK_TREE_MODEL (xfkc_gui->combo_store);
	gtk_combo_box_get_active_iter (GTK_COMBO_BOX (xfkc_gui->kbd_model_combo), &iter);
	gtk_tree_model_get (model, &iter, NOM, &kbdmodel, -1);
	xfkc_settings->model = kbdmodel;

	xfkc_settings->use_X_config = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (xfkc_gui->use_X_check_btn));
	
	write_settings_in_rc_file ();

	g_free(layouts);
	g_free(variants);
	g_free(kbdmodel);
}

void
fill_tree_model_with_config ()
{
	while (user_rec_config->layouts[xfkc_gui->layouts_nb] != NULL)
	{
		gtk_list_store_append (xfkc_gui->layout_store, &iter);
		gtk_list_store_set ( xfkc_gui->layout_store, &iter,
							LAYOUTS, user_rec_config->layouts[xfkc_gui->layouts_nb ],
							VARIANTS, user_rec_config->variants[xfkc_gui->layouts_nb ],
							-1);
		xfkc_gui->layouts_nb += 1;
	}
}

static void
add_model_to_list (XklConfigRegistry * config_registry,
                   XklConfigItem * config_item,
                   xfkcDialog * dialog)
{
	char *utf_model_name = xci_desc_to_utf8 (config_item);
	gtk_list_store_append( dialog->combo_store, &iter );
	gtk_list_store_set( dialog->combo_store, &iter, 
                      DESC, utf_model_name,
                      NOM, config_item->name, -1 );
	g_free (utf_model_name);
}

void
set_combo_default_value ()
{
	gchar *id;

	model = GTK_TREE_MODEL (xfkc_gui->combo_store);
	gtk_tree_model_get_iter_first (model, &iter);
	gtk_tree_model_get (model, &iter, NOM, &id, -1);
	if (strcmp (id, xfkc_settings->model) == 0 )
		gtk_combo_box_set_active_iter (GTK_COMBO_BOX (xfkc_gui->kbd_model_combo), &iter);
	else
	{
		while (gtk_tree_model_iter_next(model, &iter))
		{
			gtk_tree_model_get (model, &iter, NOM, &id, -1);

			if (strcmp (id, xfkc_settings->model) == 0)
			{
				gtk_combo_box_set_active_iter (GTK_COMBO_BOX (xfkc_gui->kbd_model_combo), &iter);
				break;
			}
		}
	}
	
	g_free (id);
}
                                                        
void
define_dialog ()
{
	GtkCellRenderer * renderer;
	GtkWidget *vbox1, *vbox2, *hbox, *label, *tree, *frame;
	int response;
	
	xfkc_gui = g_new0 (xfkcDialog, 1);

	xfkc_gui->max_layouts_nb = xkl_engine_get_max_num_groups (engine);
	xfkc_gui->layouts_nb = 0;
	
	renderer = gtk_cell_renderer_text_new ();

	xfkc_gui->dialog = xfce_titled_dialog_new_with_buttons (_("Keyboard settings"),
											NULL,
											GTK_DIALOG_MODAL,
											GTK_STOCK_CANCEL,
											GTK_RESPONSE_CANCEL,
											GTK_STOCK_OK,
											GTK_RESPONSE_OK,
											NULL);
	gtk_window_set_icon_name (GTK_WINDOW (xfkc_gui->dialog), "xfce4-keyboard");

	xfkc_gui->combo_store = gtk_list_store_new (COMBO_NUM, G_TYPE_STRING, G_TYPE_STRING);
	xfkc_gui->layout_store = gtk_list_store_new (TREE_NUM, G_TYPE_STRING, G_TYPE_STRING);
	
	vbox1 = gtk_vbox_new (FALSE, 5);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (xfkc_gui->dialog)->vbox), vbox1);
	gtk_widget_show (vbox1);

	xfkc_gui->use_X_check_btn = gtk_check_button_new_with_label (_("Use X configuration"));
	gtk_box_pack_start (GTK_BOX (vbox1), xfkc_gui->use_X_check_btn, FALSE, FALSE, 0);
	gtk_widget_show (xfkc_gui->use_X_check_btn);
	
	xfkc_gui->global_vbox = gtk_vbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (vbox1), xfkc_gui->global_vbox, FALSE, FALSE, 5);
	gtk_widget_show (xfkc_gui->global_vbox);

	frame = xfce_framebox_new (_("Keyboard model:"), TRUE);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (xfkc_gui->global_vbox), frame, TRUE, TRUE, 0);

	xfkc_gui->kbd_model_combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL (xfkc_gui->combo_store));
	gtk_widget_set_size_request (xfkc_gui->kbd_model_combo, 230, -1);
	xfce_framebox_add (XFCE_FRAMEBOX (frame), xfkc_gui->kbd_model_combo);
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (xfkc_gui->kbd_model_combo), renderer, TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (xfkc_gui->kbd_model_combo), renderer, "text", 0);
	xkl_config_registry_foreach_model (registry,
                                       (ConfigItemProcessFunc) add_model_to_list,
                                       xfkc_gui);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (xfkc_gui->combo_store),
                                        0, GTK_SORT_ASCENDING);

	set_combo_default_value ();
	gtk_widget_show (xfkc_gui->kbd_model_combo);
	
	frame = xfce_framebox_new (_("Keyboard layouts:"), TRUE);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (xfkc_gui->global_vbox), frame, TRUE, TRUE, 0);
	
	hbox = gtk_hbox_new (FALSE, 5);
	xfce_framebox_add (XFCE_FRAMEBOX (frame), hbox);
	gtk_widget_show (hbox);
	
	// TreeView
	xfkc_gui->layout_tree_view = gtk_tree_view_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (xfkc_gui->layout_tree_view),
                                               -1,      
                                               "Layout",  
                                               renderer,
                                               "text", LAYOUTS,
                                               NULL);
                                               
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (xfkc_gui->layout_tree_view),
                                               -1,      
                                               "Variant",  
                                               renderer,
                                               "text", VARIANTS,
                                               NULL);
	
	xfkc_gui->layout_store = gtk_list_store_new (TREE_NUM, G_TYPE_STRING, G_TYPE_STRING);
	fill_tree_model_with_config ();
	
	gtk_tree_view_set_model (GTK_TREE_VIEW (xfkc_gui->layout_tree_view),
							GTK_TREE_MODEL (xfkc_gui->layout_store));
	
	gtk_box_pack_start (GTK_BOX (hbox), xfkc_gui->layout_tree_view, TRUE, TRUE, 0);
	gtk_widget_set_size_request (xfkc_gui->layout_tree_view, -1, 112);
	gtk_widget_show (xfkc_gui->layout_tree_view);


	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 5);
	gtk_widget_show (vbox2);

	xfkc_gui->add_btn = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_box_pack_start (GTK_BOX (vbox2), xfkc_gui->add_btn, FALSE, FALSE, 0);
	gtk_widget_show (xfkc_gui->add_btn);

	xfkc_gui->edit_btn = gtk_button_new_from_stock (GTK_STOCK_EDIT);
	gtk_box_pack_start (GTK_BOX (vbox2), xfkc_gui->edit_btn, FALSE, FALSE, 0);
	gtk_widget_show (xfkc_gui->edit_btn);

	xfkc_gui->rm_btn = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_box_pack_start (GTK_BOX (vbox2), xfkc_gui->rm_btn, FALSE, FALSE, 0);
	gtk_widget_show (xfkc_gui->rm_btn);

	show_hide_buttons ();
	edit_btn_show (GTK_TREE_VIEW (xfkc_gui->layout_tree_view), NULL);

	// update the GUI with the settings
	if (xfkc_settings->use_X_config == 1)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (xfkc_gui->use_X_check_btn),
								TRUE);
	else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (xfkc_gui->use_X_check_btn),
								FALSE);
	check_button_toggled (GTK_TOGGLE_BUTTON (xfkc_gui->use_X_check_btn), NULL);
	
	// Signals
	g_signal_connect (G_OBJECT (xfkc_gui->use_X_check_btn), "toggled",
		G_CALLBACK (check_button_toggled), NULL);
	g_signal_connect (G_OBJECT (xfkc_gui->add_btn), "clicked",
		G_CALLBACK (add_layout), NULL);
	g_signal_connect (G_OBJECT (xfkc_gui->rm_btn), "clicked",
		G_CALLBACK (rm_layout), NULL);
	g_signal_connect (G_OBJECT (xfkc_gui->edit_btn), "clicked",
		G_CALLBACK (edit_layout), NULL);
	g_signal_connect (G_OBJECT (xfkc_gui->layout_tree_view), "cursor-changed",
		G_CALLBACK (edit_btn_show), NULL);
	g_signal_connect (G_OBJECT (xfkc_gui->dialog), "response",
		G_CALLBACK (dialog_response), NULL);

	gtk_dialog_run (GTK_DIALOG (xfkc_gui->dialog));
}

static void
show_hide_buttons ()
{
	int nb = (int) xfkc_gui->layouts_nb;
	int max_nb = (int) xfkc_gui->max_layouts_nb;
	if (nb < max_nb)
		gtk_widget_set_sensitive (xfkc_gui->add_btn, TRUE);
	else
		gtk_widget_set_sensitive (xfkc_gui->add_btn, FALSE);
	
	if (nb > 1)
		gtk_widget_set_sensitive (xfkc_gui->rm_btn, TRUE);
	else
		gtk_widget_set_sensitive (xfkc_gui->rm_btn, FALSE);
}

//           //
// Callbacks //
//           //

static void
edit_btn_show (GtkTreeView *tree_view,
               gpointer     user_data)
{
	GtkTreePath *p;
	GtkTreeViewColumn *c;
	gtk_tree_view_get_cursor (GTK_TREE_VIEW (tree_view), &p, &c);
	if (p != NULL)
		gtk_widget_set_sensitive (xfkc_gui->edit_btn, TRUE);
	else
		gtk_widget_set_sensitive (xfkc_gui->edit_btn, FALSE);
}

// dialog response //
static void
dialog_response (GtkWidget *widget,
		 gint       response,
		 gpointer data)
{
	if (response == GTK_RESPONSE_OK)
		save_settings ();

	gtk_widget_destroy (xfkc_gui->dialog);

	gtk_tree_path_free (path);

	g_free (xfkc_gui);
	g_free (xfkc_settings);
	g_free (config_file);

    g_object_unref(G_OBJECT(registry));
    g_object_unref(G_OBJECT(user_rec_config));

	exit (EXIT_SUCCESS);
}

// remove layout //
static void
rm_layout (GtkWidget *widget, gpointer data)
{
	selection = gtk_tree_view_get_selection (
									GTK_TREE_VIEW (xfkc_gui->layout_tree_view));
	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		gtk_list_store_remove(xfkc_gui->layout_store, &iter);
		gtk_widget_set_sensitive (xfkc_gui->add_btn, TRUE);
		xfkc_gui->layouts_nb -= 1;
		show_hide_buttons ();
	}
	edit_btn_show (GTK_TREE_VIEW (xfkc_gui->layout_tree_view), NULL);
}

// check button toggled //
static void
check_button_toggled (GtkToggleButton *btn, gpointer data)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (xfkc_gui->use_X_check_btn)))
		gtk_widget_set_sensitive (GTK_WIDGET (xfkc_gui->global_vbox), FALSE);
	else
		gtk_widget_set_sensitive (GTK_WIDGET (xfkc_gui->global_vbox), TRUE);
}

// Add Layout //

// from the gnome control center keyboard applet
static void
add_variant_to_available_layouts_tree (XklConfigRegistry * config_registry,
                                       XklConfigItem * config_item, 
                                       GtkTreeStore *treestore)
{
  char *utf_variant_name = xci_desc_to_utf8 (config_item);

  gtk_tree_store_append (treestore, &child, &iter);
  gtk_tree_store_set (treestore, &child, 
		      AVAIL_LAYOUT_TREE_COL_DESCRIPTION, utf_variant_name, 
		      AVAIL_LAYOUT_TREE_COL_ID, config_item->name, -1);
  g_free (utf_variant_name);
}

static void
add_layout_to_available_layouts_tree (XklConfigRegistry * config_registry,
                                      XklConfigItem * config_item, 
                                      GtkTreeStore *treestore)
{
  char *utf_layout_name = xci_desc_to_utf8 (config_item);

  gtk_tree_store_append (treestore, &iter, NULL);
  gtk_tree_store_set (treestore, &iter, 
		      AVAIL_LAYOUT_TREE_COL_DESCRIPTION, utf_layout_name, 
		      AVAIL_LAYOUT_TREE_COL_ID, config_item->name, -1);
  g_free (utf_layout_name);

  xkl_config_registry_foreach_layout_variant (config_registry, config_item->name,
			       (ConfigItemProcessFunc)add_variant_to_available_layouts_tree, 
                               treestore);
}
//
static void
edit_layout (GtkWidget *widget, gpointer data)
{
	gchar *c;
	c = layout_dialog_run();
	if (c != NULL)
	{
		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (xfkc_gui->layout_tree_view));
		gchar **strings;
		strings = g_strsplit_set(c, ",", 0);

		gtk_tree_selection_get_selected (selection, &model, &iter);
		gtk_list_store_set (xfkc_gui->layout_store, &iter,
							LAYOUTS, strings[0],
							VARIANTS, strings[1],
							-1);
		g_free(strings);
	}
	g_free(c);
	edit_btn_show (GTK_TREE_VIEW (xfkc_gui->layout_tree_view), NULL);
}

static void
add_layout (GtkWidget *widget, gpointer data)
{
	gchar *c;
	c = layout_dialog_run();
	if (c != NULL)
	{
		gchar **strings;
		strings = g_strsplit_set(c, ",", 0);
		gtk_list_store_append (xfkc_gui->layout_store, &iter);
		gtk_list_store_set (xfkc_gui->layout_store, &iter,
						LAYOUTS, strings[0],
						VARIANTS, strings[1],
						-1);
		xfkc_gui->layouts_nb += 1;
		show_hide_buttons ();
		g_free (strings);
	}
	g_free (c);
	edit_btn_show (GTK_TREE_VIEW (xfkc_gui->layout_tree_view), NULL);
}
	
gchar *
layout_dialog_run ()
{
	GtkWidget *dialog;
	GtkTreeStore *treestore;
	GtkWidget *tree_view = gtk_tree_view_new ();
	GtkCellRenderer *renderer;
	GtkWidget *scrolledw;

	dialog = xfce_titled_dialog_new_with_buttons(_("Add layout"),
										GTK_WINDOW (xfkc_gui->dialog),
										GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
										GTK_STOCK_CANCEL,
										GTK_RESPONSE_CANCEL,
										GTK_STOCK_OK,
										GTK_RESPONSE_OK,
										NULL);
	gtk_window_set_icon_name (GTK_WINDOW (dialog), "xfce4-keyboard");

	treestore = gtk_tree_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	
	xkl_config_registry_foreach_layout (registry, (ConfigItemProcessFunc)
			add_layout_to_available_layouts_tree, treestore);

	renderer = gtk_cell_renderer_text_new ();
	
	GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes (NULL,
									renderer,
									"text",
									AVAIL_LAYOUT_TREE_COL_DESCRIPTION,
									NULL);
	gtk_tree_view_set_model (GTK_TREE_VIEW (tree_view),
			   GTK_TREE_MODEL (treestore));
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (treestore),
                                        0, GTK_SORT_ASCENDING);

	scrolledw = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), scrolledw);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledw),
									GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC);
	gtk_widget_show (scrolledw);
	
	gtk_container_add (GTK_CONTAINER (scrolledw), tree_view);
	gtk_widget_show (GTK_WIDGET (tree_view));
	
	gtk_window_set_default_size(GTK_WINDOW (dialog), 400, 400);
	gtk_widget_show (dialog);

	int response = gtk_dialog_run (GTK_DIALOG (dialog));
	
	if (response == GTK_RESPONSE_OK)
	{
		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
		GtkTreeIter selected_iter;
		gchar *id;
		gchar *strings[2];
				
		gtk_tree_selection_get_selected (selection, &model, &iter);
		gtk_tree_model_get (model, &iter, AVAIL_LAYOUT_TREE_COL_ID, &id, -1);
		
		path = gtk_tree_model_get_path (model, &iter);
		if (gtk_tree_path_get_depth (path) == 1)
		{
			strings[0] = id;
			strings[1] = "";
		}
		else
		{
			strings[1] = id;
			gtk_tree_path_up(path);
			gtk_tree_model_get_iter(model, &iter, path);
			gtk_tree_model_get (model, &iter, AVAIL_LAYOUT_TREE_COL_ID, &id, -1);
			strings[0] = id;
		}

		gtk_widget_destroy (dialog);
		return g_strconcat(strings[0], ",", strings[1], NULL);
		
	}
	gtk_widget_destroy (dialog);
	return NULL;
}
