/*
 *	Xenophilia GTK+ Theme Engine
 *	
 *	xeno_theme_style.c:
 *		XenoStyle / XenoStyleData
 *
 *	Copyright  1999-2001 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_theme_rc.h"
#include "xeno_theme_style.h"
#include "xeno_theme_draw.h"
#include "xeno_theme_color.h"
#include "xeno_theme_images.h"

/*
 *	Prototypes
 */

#if XENO_GTK2
static void 		xeno_style_class_init	(XenoStyleClass *klass);
static void			xeno_style_init			(XenoStyle		*style);
static void			xeno_style_finalize		(XenoStyle		*style);

static void			xeno_style_copy			(GtkStyle		*dest,
											 GtkStyle		*src);
#endif


/*
 *	Globals
 */

#if XENO_GTK2
static GtkStyleClass *	xeno_style_parent_class = NULL;
#else
GMemChunk *		xeno_style_data_chunk = NULL;
#endif


/*
 *	XenoStyle / XenoStyleData
 */

#if XENO_GTK2
GType
xeno_style_get_type ()
{
	static GType style_type = 0;
	
	if (!style_type) {
		static const GTypeInfo style_info = {
			sizeof(XenoStyleClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) xeno_style_class_init,
			NULL,
			NULL,
			sizeof(XenoStyle),
			8,
			(GInstanceInitFunc) xeno_style_init,
		};
		style_type = g_type_register_static (GTK_TYPE_STYLE, "XenoStyle", &style_info, 0);
	}
	return style_type;
}

static void
xeno_style_class_init (XenoStyleClass *klass)
{
	GtkStyleClass	*style_class;
	GObjectClass	*object_class;

	xeno_style_parent_class = g_type_class_peek_parent (klass);
	
	object_class = G_OBJECT_CLASS(klass);
	object_class->finalize	= (GObjectFinalizeFunc) xeno_style_finalize;
	
	style_class = GTK_STYLE_CLASS(klass);
	style_class->copy				= xeno_style_copy;
	style_class->realize			= xeno_style_realize;
	style_class->unrealize			= xeno_style_unrealize;
	
	style_class->draw_hline			= xeno_draw_hline;
	style_class->draw_vline			= xeno_draw_vline;
	style_class->draw_shadow		= xeno_draw_shadow;
	style_class->draw_polygon		= xeno_draw_polygon;
	style_class->draw_arrow			= xeno_draw_arrow;
	style_class->draw_diamond		= xeno_draw_diamond;
	style_class->draw_string		= xeno_draw_string;
	style_class->draw_box			= xeno_draw_box;
	style_class->draw_flat_box		= xeno_draw_flat_box;
	style_class->draw_check			= xeno_draw_check;
	style_class->draw_option		= xeno_draw_option;
	style_class->draw_tab			= xeno_draw_tab;
	style_class->draw_shadow_gap	= xeno_draw_shadow_gap;
	style_class->draw_box_gap		= xeno_draw_box_gap;
	style_class->draw_extension		= xeno_draw_extension;
	style_class->draw_focus			= xeno_draw_focus;
	style_class->draw_slider		= xeno_draw_slider;
	style_class->draw_handle		= xeno_draw_handle;
}
#endif

#if XENO_GTK2
static void
xeno_style_init (XenoStyle *data)
{
	gint i;
	
	{
#else
XenoStyleData *
xeno_style_data_new ()
{
	XenoStyleData *data;
	gint i;

	if (xeno_style_data_chunk == NULL) {
		if ((xeno_style_data_chunk = g_mem_chunk_create(XenoStyleData, 256, G_ALLOC_AND_FREE)) == NULL)
			return NULL;
	}
	
	data = g_mem_chunk_alloc (xeno_style_data_chunk);
	if (data) {
		data->ref_count = 1;
#endif
		for (i=0; i<XENO_IMAGE_LAST; ++i)
			data->pixmaps[i] = NULL;
		for (i=0; i<5; ++i) {
			data->white_gc[i] = NULL;
			data->black_gc[i] = NULL;
			data->fgmid_gc[i] = NULL;
		}
	}
	xeno_gradient_set_init (&data->gradient_set);
  #if !XENO_GTK2
	return data;
  #endif
}

#if XENO_GTK2
static void
xeno_style_finalize (XenoStyle *data)
#else
void
xeno_style_data_destroy (XenoStyleData *data)
#endif
{
	gint i;
	
	g_return_if_fail (data != NULL);
  #if !XENO_GTK2
	g_return_if_fail (data->ref_count == 0);
  #endif
	
	for (i=0; i<XENO_IMAGE_LAST; i++) {
		if (data->pixmaps[i]) {
			gdk_pixmap_unref(data->pixmaps[i]);
			xeno_image_unref(i);
			data->pixmaps[i] = NULL;
		}
	}
	for (i=0; i<5; ++i) {
		if (data->white_gc[i]) gtk_gc_release (data->white_gc[i]);
		if (data->black_gc[i]) gtk_gc_release (data->black_gc[i]);
	}
  #if !XENO_GTK2	
	g_mem_chunk_free (xeno_style_data_chunk, data);
  #endif
}

#if XENO_GTK2
static void
xeno_style_copy (GtkStyle *dest, GtkStyle *src)
{
	g_return_if_fail (XENO_IS_STYLE(src));
	g_return_if_fail (XENO_IS_STYLE(dest));
	
	xeno_style_parent_class->copy(dest, src);
	/* the rest will be recalculated when realize()'d */
}
#endif


/* Realization */

static void
xeno_realize_gc	(GdkGCValues	*gc_values,
				 GdkColormap	*colormap,
				 XenoColor		*color,
				 GdkColor		*color_p,
				 GdkGC			**gc_p)
{
	GdkColor *ptr;
	
	g_return_if_fail (gc_values != NULL);
	g_return_if_fail (colormap != NULL);
	g_return_if_fail (color != NULL);
	g_return_if_fail (gc_p != NULL);
	
	ptr = &gc_values->foreground;
	xeno_color_to_gdk (color, ptr);
	if (color_p)
		*color_p = *ptr;
	
	if (!gdk_colormap_alloc_color(colormap, (ptr), FALSE, TRUE)) {
		g_warning ("unable to allocate color #%02x%02x%02x\n", ptr->red>>8, ptr->green>>8, ptr->blue>>8);
	} else if (gc_p) {
		if (*gc_p)
			gtk_gc_release (*gc_p);
		
		*gc_p = gtk_gc_get (xeno_visual->depth, xeno_colormap, gc_values, GDK_GC_FOREGROUND | GDK_GC_FONT );
	}
}

void
xeno_style_realize (GtkStyle *style)
{
	GdkGCValues		gc_values;
	XenoColor		bg, light, dark, *white_p, *black_p;
	
	XenoRcData		*rc_data;
	XenoStyleData	*style_data;
	
	extern GdkFont	*default_font;	/* exported from gtk/gtkstyle.c */
	gint			i;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	
	rc_data	   = XENO_STYLE_RC_DATA(style);
	style_data = XENO_STYLE_DATA(style);
	
  #if XENO_GTK2
	xeno_style_parent_class->unrealize(style);
  #else
	g_return_if_fail (rc_data != NULL);
	g_return_if_fail (style_data != NULL);
  #endif
	
	if (!default_font)
		default_font = gdk_font_load ("-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*");
	
	if (style->font->type == GDK_FONT_FONT) {
		gc_values.font = style->font;
	} else if (style->font->type == GDK_FONT_FONTSET) {
		gc_values.font = default_font;
	}
	
	for (i = 0; i < 5; ++i) {
		xeno_color_from_gdk (&bg, &style->bg[i]);
		if (style->bg_pixmap[i] && gdk_color_equal (&style->bg[i], &style->rc_style->bg[i])) {
			if (xeno_pixmap_get_average_color (style->bg_pixmap[i], xeno_visual, &bg))
				xeno_color_to_gdk (&bg, &style->bg[i]);
		}
		
		white_p = &style_data->white[i];
		black_p = &style_data->black[i];
		
		xeno_color_shade (&bg, rc_data->white[i], white_p);
		xeno_color_shade (&bg, rc_data->black[i], black_p);
		
		xeno_color_blend (&bg, white_p, rc_data->shine[i] - 1.0, &light);
		xeno_color_blend (&bg, black_p, 1.0 - rc_data->shade[i], &dark);
		
		xeno_realize_gc (&gc_values, style->colormap, white_p, NULL, &style_data->white_gc[i]);
		xeno_realize_gc (&gc_values, style->colormap, black_p, NULL, &style_data->black_gc[i]);
		
		if (!xeno_pseudocolor)
			xeno_color_blend (&light, &dark, 0.5, &bg);
		
		/* Allocate new gc's */
		xeno_realize_gc (&gc_values, style->colormap, &light, &style->light[i], &style->light_gc[i]);
		xeno_realize_gc (&gc_values, style->colormap, &dark,  &style->dark[i],  &style->dark_gc[i]);
		xeno_realize_gc (&gc_values, style->colormap, &bg,	&style->mid[i],	  &style->mid_gc[i]);
	}
	
	xeno_gradient_set_realize (&style_data->gradient_set, style);
}

void
xeno_style_unrealize (GtkStyle *style)
{
	XenoStyleData	*style_data;
	gint i;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	
	style_data = XENO_STYLE_DATA(style);
	
#if XENO_GTK2
	xeno_style_parent_class->unrealize(style);
#else
	if (style_data->ref_count == 1)
#endif
	{
		for (i=0; i<5; i++) {
			if (style_data->white_gc[i]) {
				gtk_gc_release (style_data->white_gc[i]);
				style_data->white_gc[i] = NULL;
			}
			if (style_data->black_gc[i]) {
				gtk_gc_release (style_data->black_gc[i]);
				style_data->black_gc[i] = NULL;
			}
			if (style_data->fgmid_gc[i]) {
				gtk_gc_release (style_data->fgmid_gc[i]);
				style_data->fgmid_gc[i] = NULL;
			}
		}
		xeno_gradient_set_unrealize (&style_data->gradient_set);
	}
}


#if !XENO_GTK2
/* The style classes */
XenoStyleClass xeno_style_classes[4] = {
	{
		{
			0,
			0,
			xeno_draw_hline,
			xeno_draw_vline,
			xeno_draw_shadow,
			xeno_draw_polygon,
			xeno_draw_arrow,
			xeno_draw_diamond,
			xeno_draw_oval,
			xeno_draw_string,
			xeno_draw_box,
			xeno_draw_flat_box,
			xeno_draw_check,
			xeno_draw_option,
			xeno_draw_cross,
			xeno_draw_ramp,
			xeno_draw_tab,
			xeno_draw_shadow_gap,
			xeno_draw_box_gap,
			xeno_draw_extension,
			xeno_draw_focus,
			xeno_draw_slider,
			xeno_draw_handle
		},
		XENO_STYLE_MAGIC,
	},
	{
		{
			1,
			1,
			xeno_draw_hline,
			xeno_draw_vline,
			xeno_draw_shadow,
			xeno_draw_polygon,
			xeno_draw_arrow,
			xeno_draw_diamond,
			xeno_draw_oval,
			xeno_draw_string,
			xeno_draw_box,
			xeno_draw_flat_box,
			xeno_draw_check,
			xeno_draw_option,
			xeno_draw_cross,
			xeno_draw_ramp,
			xeno_draw_tab,
			xeno_draw_shadow_gap,
			xeno_draw_box_gap,
			xeno_draw_extension,
			xeno_draw_focus,
			xeno_draw_slider,
			xeno_draw_handle
		},
		XENO_STYLE_MAGIC,
	},
	{
		{
			2,
			2,
			xeno_draw_hline,
			xeno_draw_vline,
			xeno_draw_shadow,
			xeno_draw_polygon,
			xeno_draw_arrow,
			xeno_draw_diamond,
			xeno_draw_oval,
			xeno_draw_string,
			xeno_draw_box,
			xeno_draw_flat_box,
			xeno_draw_check,
			xeno_draw_option,
			xeno_draw_cross,
			xeno_draw_ramp,
			xeno_draw_tab,
			xeno_draw_shadow_gap,
			xeno_draw_box_gap,
			xeno_draw_extension,
			xeno_draw_focus,
			xeno_draw_slider,
			xeno_draw_handle
		},
		XENO_STYLE_MAGIC,
	},
	{
		{
			3,
			3,
			xeno_draw_hline,
			xeno_draw_vline,
			xeno_draw_shadow,
			xeno_draw_polygon,
			xeno_draw_arrow,
			xeno_draw_diamond,
			xeno_draw_oval,
			xeno_draw_string,
			xeno_draw_box,
			xeno_draw_flat_box,
			xeno_draw_check,
			xeno_draw_option,
			xeno_draw_cross,
			xeno_draw_ramp,
			xeno_draw_tab,
			xeno_draw_shadow_gap,
			xeno_draw_box_gap,
			xeno_draw_extension,
			xeno_draw_focus,
			xeno_draw_slider,
			xeno_draw_handle
		},
		XENO_STYLE_MAGIC,
	}
};
#endif


