#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2006 Jd & Hap Hazard
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#

import ConfigParser, subprocess, platform
import sys, os, os.path, socket
import shutil, urllib,urlparse
import constants
from NodeProxy import Node


class XMConfig(ConfigParser.SafeConfigParser):
    """ XenMan's configuration management class. """

    # the default list of sections in the config
    DEFAULT = 'DEFAULT'
    ENV = 'ENVIRONMENT'
    PATHS = 'PATHS'
    APP_DATA = 'APPLICATION DATA'
    CLIENT_CONFIG = 'CLIENT CONFIGURATION'
    IMAGE_STORE = 'IMAGE STORE'
    
    DEFAULT_REMOTE_FILE = '/etc/xenman.conf'
    
    def __init__(self, node, searchfiles = None, create_file = None):
        """Looks for xenman.conf in current, user home and /etc directories. If
        it is not found, seeds one in the local directory."""

        ConfigParser.SafeConfigParser.__init__(self)
        self.node = node
        self.std_sections = [self.ENV,self.PATHS,self.APP_DATA,
                             self.CLIENT_CONFIG, self.IMAGE_STORE]
        
        if searchfiles is None:
            # no search path give. apply heuristics
            if not self.node.isRemote:
                # localhost
                filelist = [x for x in [os.path.join(os.getcwd(),'xenman.conf'),
                                        os.path.expanduser('~/.xenman/xenman.conf'),
                                        '/etc/xenman.conf'] if self.node.file_exists(x)]
            else:
                # remote host
                if self.node.file_exists(self.DEFAULT_REMOTE_FILE):
                    filelist =  [self.DEFAULT_REMOTE_FILE]
                else:
                    filelist = []
        else:
            # search path specified
            filelist = [x for x in searchfiles if self.node.file_exists(x)]

        if len(filelist) < 1:
            print 'No Configuration File Found'
            if create_file is None:
                # no default creation file is specified. use heuristics
                if not self.node.isRemote:
                    # localhost. create in cwd
                    print 'Creating default xenman.conf in current directory'            
                    self.configFile = os.path.join(os.getcwd(), 'xenman.conf')
                else:
                    # remote host. create in default remote location
                    print 'Creating default xenman.conf at %s:%s' \
                          % (self.node.hostname,self.DEFAULT_REMOTE_FILE)
                    self.configFile = self.DEFAULT_REMOTE_FILE                
            else:
                # default creation location is specified
                print 'Creating default xenman.conf at %s:%s' \
                      % (self.node.hostname,create_file)                
                self.configFile = create_file            

            # new file created, populate w/ default entries
            self.__createDefaultEntries()
            self.__commit()
            
        else:            
            # config file(s) found. choose a writable file,
            # otherwise create a new default file in the user's
            # home directory (only for localhost)
            self.configFile = None
            for f in filelist:
                try:
                    if self.node.file_is_writable(f):
                        # file is writable
                        if self.__validateFile(f):
                            # file is valid
                            self.configFile = f
                            print 'Valid, writable configuration found, using %s' % f
                        else:
                            # file is writable but not valid
                            # back it up (a new one will get created)                            
                            self.node.rename(f,f+'.bak')
                            print 'Old configuration found. Creating backup: %s.bak' % f                            

                        break
                    else:
                        print 'Confguration File not writable, skipping: %s' % f
                except IOError:
                    print 'Confguration File accessable, skipping: %s' % f
                    continue
                    
            if self.configFile is None:
                # no writable config file found
                print "No writable configuration found ... "
                if not self.node.isRemote:
                    # localhost
                    if not os.path.exists(os.path.expanduser('~/.xenman/')):
                        os.mkdir(os.path.expanduser('~/.xenman/'))
                    self.configFile = os.path.expanduser('~/.xenman/xenman.conf')
                    print "\t Creating %s" % self.configFile                    
                    self.__createDefaultEntries()
                    self.__commit()
                else:
                    # TBD: what to do in the remote case
                    raise Exception('No writable configuration found on remote host: %s' % self.node.hostname)
                
            #self.configFile = filelist[0]
            fp = self.node.open(self.configFile)
            self.readfp(fp)
            fp.close()

        
        self.__commit()


    def __createDefaultEntries(self):

        # cleanup first
        for s in self.sections():
            self.remove_section(s)
            
        # add the standard sections
        for s in self.std_sections:
            self.add_section(s)                


        # seed the defaults
        self.set(self.PATHS, constants.prop_disks_dir, '')
        self.set(self.PATHS, constants.prop_snapshots_dir, '')
        self.set(self.PATHS, constants.prop_snapshot_file_ext, '.snapshot.xm')
        self.set(self.PATHS, constants.prop_xenconf_dir,'/etc/xen')
        self.set(self.PATHS, constants.prop_cache_dir, '/var/cache/xenman')        
        self.set(self.CLIENT_CONFIG, constants.prop_browser, '/usr/bin/yelp')
        #self.set(self.PATHS, constants.prop_staging_location, '')
        #self.set(self.PATHS, constants.prop_kernel, '')
        #self.set(self.PATHS, constants.prop_ramdisk, '')
        #self.set(self.ENV, constants.prop_lvm, 'True')
        #self.__commit()

        # set localhost specific properties
        if not self.node.isRemote:
            #self.add_section(constants.LOCALHOST)
            self.set(self.ENV,constants.prop_dom0_kernel,platform.release())
        
        
    def __commit(self):
        outfile = self.node.open(self.configFile,'w')
        self.write(outfile)
        outfile.close()

    def __validateFile(self, filename):
        temp = ConfigParser.ConfigParser()
        fp = self.node.open(filename)
        temp.readfp(fp)
        fp.close()
        for s in self.std_sections:
            if not temp.has_section(s):
                return False
        return True
    

    def getDefault(self, option):
        """ retrieve a default option/key value """
        return self.get(self.DEFAULT, option)


    def get(self, section, option):

        # does the option exist? return None if not
        if option is None: return None
        if not self.has_option(section, option): return None

        # option is available in the config. get it.
        retVal = ConfigParser.SafeConfigParser.get(self, section, option)
        
        # check if the value is blank. if so, return None
        # otherwise, return the value.
        if not retVal.strip():
            return None
        else:
            return retVal
        

    def setDefault(self, option, value):
        """set the default for option to value.
        POSTCONDITION: option, value pair has been set in the default
        configuration, and committed to disk"""
        if option is not None:
            self.set(self.DEFAULT, option, value)


    def set(self, section, option, value):
        ConfigParser.SafeConfigParser.set(self, section, option, value)
        self.__commit()

        
    def getHostProperty(self, option, hostname=constants.LOCALHOST):
        """ retrieve the value for 'option' for 'hostname',
        'None', if the option is not set"""

        # does the option exist? return None if not
        if not self.has_option(hostname, option): return None

        # option is available in the config. get it.
        retVal = self.get(hostname, option)
        
        # check if the value is blank. if so, return None
        # otherwise, return the value.
        if not retVal.strip():
            return None
        else:
            return retVal        

    def setHostProperty(self, option, value, hostname=constants.LOCALHOST):
        """ set config 'option' to 'value' for 'hostname'.
        If the a config section for 'hostname' doesn't exit,
        one is created."""
        
        if not self.has_section(hostname): self.add_section(hostname)
        self.set(hostname, option, value)
        self.__commit()

    def removeHost(self, hostname):
        """ remove 'hostname' from the list of configured hosts.
        The configuration is deleted from both memory and filesystem"""

        if self.has_section(hostname):
            self.remove_section(hostname)
            self.__commit()

    def getHosts(self):
        """ return the list configured hosts"""
        return set(self.sections())-set(self.std_sections)


    
class LVMProxy:
    """A thin, (os-dependent) wrapper around the shell's LVM
    (Logical Volume Management) verbs"""
    # TODO: move this class to an OSD module
    
    @classmethod
    def isLVMEnabled(cls, node_proxy):
        retVal = True
        if node_proxy.exec_cmd('vgs 2> /dev/null')[1]:
            if node_proxy.exec_cmd('/usr/sbin/vgs 2> /dev/null')[1]:
                retVal = False
        return retVal
    
        
    def __init__(self, node_proxy):
        """ The constructor simply checks if LVM services are available
        for use via the shell at the specified 'node'.
        RAISES: OSError"""
        self.node = node_proxy
        self.call_prefix = ''

        if node_proxy.exec_cmd('vgs 2> /dev/null')[1]:
            self.call_prefix = '/usr/sbin/'
            if node_proxy.exec_cmd('/usr/sbin/vgs 2> /dev/null')[1]:
                raise OSError("LVM facilities not found")


    def listVolumeGroups(self):
        """ Returns the list of existing Volume Groups"""
        try:
            vglist = self.node.exec_cmd(self.call_prefix + 'vgs -o vg_name --noheadings')[0]
            return [s.strip() for s in vglist.splitlines()]
        except OSError, err:
            print err
            return None

    def listLogicalVolumes(self, vgname):
        """ Returns the list of Logical Volumes in a Volume Group"""
        try:            
            lvlist = self.node.exec_cmd(self.call_prefix + 'lvs -o lv_name --noheadings '+ vgname)[0]
            return [s.strip() for s in lvlist.splitlines()]
        except OSError, err:
            print err
            return None

    def createLogicalVolume(self, lvname, lvsize, vgname):
        """ Create a new LV with in the specified Volume Group.
        'lvsize' denotes size in number of megabytes.
        RETURNS: True on sucees
        RAISES: OSError"""
        error,retcode = self.node.exec_cmd(self.call_prefix + 'lvcreate %s -L %sM -n %s'%(vgname,lvsize,lvname))
        if retcode:
            raise OSError(error.strip('\n'))
        else:
            return True
        
                
    def removeLogicalVolume(self, lvname, vgname=None):
        """ Remove the logical volume 'lvname' from the
        Volume Group 'vgname'. If the latter is not specified,
        'lvname' is treated as a fully specified path
        RETURNS: True on success
        RAISES: OSError"""
        if (vgname):
            lvpath = vgname + '/' + lvname
        else:
            lvpath = lvname
            
        error,retcode = self.node.exec_cmd(self.call_prefix + 'lvremove -f %s'% lvpath)
        if retcode:
            raise OSError(error.strip('\n'))
        else:
            return True



class ImageStore:
    """A class that manages the list of known kernel/ramdisk pairs (images) in
    a given XMConfig object. Images specified as url's are downloaded to a
    local cache automatically. NOTE: This class is intended for use by a client
    application managing multiple DomU images.
    """
    KERNEL = 'kernel'
    RAMDISK = 'ramdisk'
    BOOTLOADER = 'bootloader'
    
    def __init__(self, config):
        self.config = config
        self.list = []
        self.default = None
        
        # create an image store section if required
        if not config.has_section(XMConfig.IMAGE_STORE):
            config.add_section(XMConfig.IMAGE_STORE)

        # read in the image list and default image
        self.__refresh()
        
        # set the default image to none if required
        if self.default == None:
            self.config.set(XMConfig.CLIENT_CONFIG, constants.prop_imagestore_default, 'None')
            
        
    def addImage(self, name,
                 kernel, ramdisk,
                 bootloader = '',
                 other = []):
        entry = [kernel,ramdisk,bootloader,other]
        self.config.set(XMConfig.IMAGE_STORE,name,str(entry))
        self.__refresh()


    def getImage(self, name=None):
        if name is None:
            name = self.default
        entry = self.config.get(XMConfig.IMAGE_STORE, name)
        if entry:
            return eval(entry)
        else:
            return None

    def getFilenames(self, name=None):
        if name is None:
            name = self.default
        return self.__validateImage(name)

    def setDefault(self, name):
        self.config.set(XMConfig.CLIENT_CONFIG, constants.prop_imagestore_default, name)
        self.default = name
        

    def __refresh(self):
        # read in the list of known images
        self.list = []
        for prop in self.config.options(XMConfig.IMAGE_STORE):
            self.list.append(prop)
        self.default = self.config.get(XMConfig.CLIENT_CONFIG,
                                       constants.prop_imagestore_default) 

                

    def __validateImage(self,name):
        entry = self.getImage(name)
        if entry is None:
            raise Exception("Invalid Image: %s"%name)
        
        cachedir = self.config.get(XMConfig.PATHS,constants.prop_cache_dir)
        kernel, ramdisk = None, None
            
        # check kernel entry
        if entry[0].startswith("http://") or entry[0].startswith("ftp://"):
            # kernel is to be network fetched. check if already in cache
            cache_path = os.path.join(cachedir,os.path.join(name,'vmlinuz.default'))
            if not os.path.exists(cache_path):                                           
                # kernel is not in cache, download it
                print '%s: downloading kernel.' % name
                try:
                    if not os.path.exists(os.path.join(cachedir,name)):
                        # create a directory for the image in the cache
                        os.mkdir(os.path.join(cachedir,name))
                    fetchImage(entry[0], cache_path)
                except (Exception, StandardError), e: 
                    raise Exception('Invalid Config: kernel download failed. '+str(e))
            # kernel is in the cache dir
            kernel = cache_path
        else:
            # kernel is a local file
            kernel = entry[0]

        
        # check ramdisk entry
        if entry[1].startswith("http://") or entry[1].startswith("ftp://"):
            # ramdisk is to be network fetched. check if already in cache
            cache_path = os.path.join(cachedir,os.path.join(name,'initrd.default'))
            if not os.path.exists(cache_path):                                           
                # ramdisk is not in cache, download it
                print '%s: downloading kernel.'% name
                try:
                    if not os.path.exists(os.path.join(cachedir,name)):
                        # create a directory for the image in the cache
                        os.mkdir(os.path.join(cachedir,name))
                    fetchImage(entry[1], cache_path)
                except (Exception, StandardError), e: 
                    raise Exception('Invalid Config: ramdisk download failed. '+str(e))
            # kernel is in the cache dir
            ramdisk = cache_path
        else:
            # ramdisk is a local file
            ramdisk = entry[1]


        # return the kernel and ramdisk file locations in cache
        return (kernel, ramdisk)


        

   
def fetchImage(src, dest):
    """ Copies 'src' to 'dest'. 'src' can be an http or ftp URL
    or a filesystem location. dest must be a fully qualified
    filename."""
    
    print "Fetching: "+src
    if src.startswith("http://") or src.startswith("ftp://"):
        # newtwork fetch
        urllib.urlretrieve(src,dest)
    else:
        # filesystem fetch
        shutil.copyfile(src, dest)





def search_tree(tree, key):
    """Retrieve a value from a tree"""

    if tree == None or key == None:
        return None
    
    for elem in tree:
        if elem[0] == key: 
            return elem[1]
        elif type(elem[1]) is list:
            value = search_tree(elem[1], key)
            if value:
                return value
    return None


def is_host_remote(host):
    host_names = []
    try:
        (host_name, host_aliases,host_addrs) = socket.gethostbyaddr(host)
        host_names.append(host_name)
        host_names = host_aliases + host_addrs
    except:
        host_names.append(host)

    return len(set(l_names).intersection(set(host_names))) == 0



#
# Module initialization
#

l_names = []
try:
    (local_name, local_aliases,local_addrs) = \
                 socket.gethostbyaddr(constants.LOCALHOST)

    l_names.append(local_name)
    l_names = local_aliases + local_addrs
except socket.herror:
    print "ERROR : can not resolve localhost"
    pass
    



#########################
# SELF TEST
#########################

if __name__ == '__main__':

    REMOTE_HOST = '192.168.123.155'
    REMOTE_USER = 'root'
    REMOTE_PASSWD = ''

    REMOTE = False    
    
    local_node = Node(hostname=constants.LOCALHOST)
    if not REMOTE:
        remote_node = local_node  # for local-only testing
    else:        
        remote_node = Node(hostname=REMOTE_HOST,
                           username=REMOTE_USER,
                           password = REMOTE_PASSWD,
                           isRemote = True)    


    #
    # LVMProxy tests
    #
    lvm_local = LVMProxy(local_node)
    lvm_remote = LVMProxy(remote_node)

    print '\nLVMProxy interface test STARTING'
    for lvm in (lvm_local, lvm_remote):
        vgs =  lvm.listVolumeGroups()
        for g in vgs:
            print g
            print lvm.listLogicalVolumes(g)
            print '\t Creating test LV'
            lvm.createLogicalVolume('selfTest',0.1,g)
            print '\t Deleting test LV'
            lvm.removeLogicalVolume('selfTest',g)
    print 'LVMPRoxy interface test COMPLETED\n'


    #
    # XMConfig tests
    #    

    TEST_CONFIGFILE = '/tmp/xenman.conf'
    
    print "\nXMConfig interface test STARTING\n"
    
    print 'LOCALHOST ...'
    config_local = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE],
                            create_file=TEST_CONFIGFILE)    
    config_local.set(XMConfig.DEFAULT,'TEST_PROP','TEST_VAL')
    print "Default Property TEST_PROP:",config_local.getDefault('TEST_PROP')
    print "Default Sections:",config_local.sections()
    print "Known Hosts", config_local.getHosts()
    config_local2 = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE])
    print "Default Property TEST_PROP:",config_local2.getDefault('test_prop')    
    local_node.remove(TEST_CONFIGFILE)

    print '\nREMOTE HOST ...'
    config_remote = XMConfig(remote_node, searchfiles = [TEST_CONFIGFILE],
                            create_file=TEST_CONFIGFILE)
    config_remote.setDefault('TEST_PROP','TEST_VAL')
    print "Default Property TEST_PROP:",config_remote.get(XMConfig.DEFAULT,'TEST_PROP')
    print "Default Sections:",config_remote.sections()
    print "Known Hosts", config_remote.getHosts()
    config_remote2 = XMConfig(remote_node, searchfiles = [TEST_CONFIGFILE])
    print "Default Property TEST_PROP:",config_remote2.getDefault('test_prop')
    remote_node.remove(TEST_CONFIGFILE)

    print "\nXMConfig interface test COMPLETED"


    #
    # ImageStore tests
    #

    print "\nImageStore interface test STARTING\n"
    config_local = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE],
                            create_file=TEST_CONFIGFILE)
    store = ImageStore(config_local)
    print store.list
    
    store.addImage('test_image','/var/cache/xenman/vmlinuz.default','/var/cache/xenman/initrd.img.default')
    print store.list
    print store.getImage('test_image')
    print store.getFilenames('test_image')

    #store.addImage('test_image2',
    #               'http://linux.nssl.noaa.gov/fedora/core/5/i386/os/images/xen/vmlinuz',
    #               'http://linux.nssl.noaa.gov/fedora/core/5/i386/os/images/xen/initrd.img',
    #               )
    #print store.list
    #print store.getImage('test_image2')
    #print store.getFilenames('test_image2')

    store.addImage('test_image2',
                   'http://localhost/fedora/images/xen/vmlinuz',
                   'http://localhost/fedora/images/xen/initrd.img',
                   )
    print store.list
    print store.getImage('test_image2')
    print "First access, should fetch ...\n",store.getFilenames('test_image2')
    print "Second access, should get from cache ... "
    kernel, ramdisk = store.getFilenames('test_image2')
    print (kernel, ramdisk)
    local_node.remove(kernel)
    local_node.remove(ramdisk)
    local_node.rmdir('/var/cache/xenman/test_image2')
    
    local_node.remove(TEST_CONFIGFILE)
    
    print "\nImageStore interface test COMPLETED"
    sys.exit(0)
    
    
    
        
