/* -*-Mode: C++;-*-
 * $Id: nodec.cc 1.18 Sat, 28 Apr 2001 15:13:12 -0700 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

int
MAJORC::node_exists (const MKEY   &mkey,
		     NODEC        &node) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.get_minor (*this, mkey, node))) {
	PROP_ERROR (ret) (this) ("get_minor");
	return ret;
    }

    if (node.type () == FT_NotPresent) {
	return DBFS_NOTFOUND;
    }

    return 0;
}

int
MAJORC::node_mkey (const MKEY   &mkey,
		   NODEC        &node) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.get_minor (*this, mkey, node))) {
	PROP_ERROR (ret) (this) ("get_minor");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			   UPDATE OPERATIONS
//////////////////////////////////////////////////////////////////////

int
NODEC::mk_directory (XTYPE         type,
		     int           flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if (! (type.key () & FT_Container)) {
	ret = DBFS_INVAL;
	DBFS_ERROR (ret) (this) ("mk_directory");
	return ret;
    }

    if ((ret = dxn.create_index (*this, type, flags))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) (this) ("create_index");
	}
	return ret;
    }

    return 0;
}

int
NODEC::mk_view (XSIZ         length,
		VIEWDEF     &vdef,
		int          flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.create_view (*this, length, vdef, flags))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) (this) ("create_view");
	}
	return ret;
    }

    return 0;
}

int
NODEC::mk_reflink (const MAJORC &linkto,
		   int           flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.create_reflink (*this, linkto, flags))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) (this) ("create_reflink");
	}
	return ret;
    }

    linkto.incr_refcount (1);

    return 0;
}

int
NODEC::mk_arealink (const SAREA  &area,
		    int           flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.create_arealink (*this, area, flags))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) (this) ("create_arealink");
	}
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			   FILE OPERATIONS
//////////////////////////////////////////////////////////////////////

int
NODEC::put_segment (const guint8 *buf,
		    int           size,
		    int           flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.create_short (*this, buf, XSIZ (size), flags))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) (this) ("create_short");
	}
	return ret;
    }

    return 0;
}

int
NODEC::get_segment (guint8 *buf,
		    int     flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if (xtype () != XTYPE_SHORTSEG) {
	ret = DBFS_NOT_READABLE;
	DBFS_ERROR (ret) (this) ("get_segment");
	return ret;
    }

    if ((ret = dxn.get_short (*this, buf))) {
	PROP_ERROR (ret) (this) ("get_short");
	return ret;
    }

    return 0;
}

int
NODEC::repl_segment (FileHandle **fhp,
		     int          flags) const
{
    int ret;

    if ((ret = pre_overwrite_later (flags))) {
	return ret;
    }

    (*fhp) = dbfs_seg_replace (*this, flags);

    return 0;
}

int
NODEC::read_segment (FileHandle **fhp,
		     int          flags) const
{
    int ret;

    if (! (type () & FT_Readable)) {
	ret = DBFS_NOT_READABLE;
	DBFS_ERROR (ret) (this) ("read_segment");
	return ret;
    }

    if ((ret = dbfs_seg_read (*this, fhp, flags))) {
	PROP_ERROR (ret) (this) ("dbfs_seg_read");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			   READ OPERATIONS
//////////////////////////////////////////////////////////////////////

int
NODEC::read_reflink (MAJORC &node,
		     int     flags) const
{
    int  ret;
    DXN  dxn (txn (), ret);

    if (type () != FT_Reflink) {
	ret = DBFS_INVAL;
	DBFS_ERROR (ret) (this) ("read_reflink");
	return ret;
    }

    if ((ret = dxn.get_major (cont_id (), node))) {
	PROP_ERROR (ret) (this) ("get_major");
	return ret;
    }

    return 0;
}

int
NODEC::read_arealink (SAREA        &area,
		      int           flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if (type () != FT_Arealink) {
	ret = DBFS_INVAL;
	DBFS_ERROR (ret) (this) ("read_arealink");
	return ret;
    }

    if ((ret = dxn.get_area (cont_id ().area_id (), area))) {
	PROP_ERROR (ret) ("get_area");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		      MAJORC->NODEC redirection
//////////////////////////////////////////////////////////////////////

int
MAJORC::del_minor () const
{
    return def ().del_minor ();
}

int
MAJORC::dir_inverse_node (const MAJORC &node,
			  NODEC        &attr) const
{
    return def ().dir_inverse_node (node, attr);
}

int
MAJORC::dir_open (DIRC &curs) const
{
    return def ().dir_open (curs);
}

int
MAJORC::dir_link_invert (const MAJORC &major,
			 DKEY         &dkey) const
{
    return def ().dir_link_invert (major, dkey);
}

int
MAJORC::dir_link_invert_seqno (const MAJORC &major,
			       XSEQNO       &seqno) const
{
    return def ().dir_link_invert_seqno (major, seqno);
}

int
MAJORC::dir_link_insert (const DKEY   &key,
			 const MAJORC &node,
			 int           flags) const
{
    return def ().dir_link_insert (key, node, flags);
}

int
MAJORC::dir_link_insert_seqno (const MAJORC &node,
			       XSEQNO       &seqno,
			       int           flags) const
{
    return def ().dir_link_insert_seqno (node, seqno, flags);
}

int
MAJORC::dir_link_lookup (const DKEY   &key,
			 MAJORC       &node,
			 int         flags)  const
{
    return def ().dir_link_lookup (key, node, flags);
}

int
MAJORC::dir_link_lookup_seqno  (const XSEQNO &key,
				MAJORC       &node,
				int         flags)  const
{
    return def ().dir_link_lookup_seqno (key, node, flags);
}

int
MAJORC::dir_link_remove (const DKEY &key) const
{
    return def ().dir_link_remove (key);
}

int
MAJORC::dir_link_remove_seqno (const XSEQNO &key) const
{
    return def ().dir_link_remove_seqno (key);
}

int
MAJORC::put_segment (const guint8 *buf,
		     int           size,
		     int           flags) const
{
    return def ().put_segment (buf, size, flags);
}

int
MAJORC::get_segment (guint8       *buf,
		     int           flags) const
{
    return def ().get_segment (buf, flags);
}

int
MAJORC::repl_segment (FileHandle **fhp,
		      int          flags) const
{
    return def ().repl_segment (fhp, flags);
}

int
MAJORC::unsafe_create (int flags) const
{
    return def ().unsafe_create (flags);
}

int
MAJORC::unsafe_truncate (const XSIZ &size) const
{
    return def ().unsafe_truncate (size);
}

int
MAJORC::unsafe_write (const guint8 *buf,
		      const XSIZ   &nbytes,
		      const XSIZ   &offset) const
{
    return def ().unsafe_write (buf, offset, nbytes);
}

int
MAJORC::unsafe_read (guint8       *buf,
		     const XSIZ   &nbytes,
		     const XSIZ   &offset,
		     XSIZ         &nbytes_out) const
{
    return def ().unsafe_read (buf, offset, nbytes, nbytes_out);
}

int
MAJORC::unsafe_sync (void) const
{
    return def ().unsafe_sync ();
}

int
MAJORC::read_segment (FileHandle **fhp,
		      int          flags) const
{
    return def ().read_segment (fhp, flags);
}

int
MAJORC::read_reflink (MAJORC &node, int flags) const
{
    return def ().read_reflink (node, flags);
}

int
MAJORC::mk_view (XSIZ         length,
		 VIEWDEF     &vdef,
		 int          flags) const
{
    return def ().mk_view (length, vdef, flags);
}

int
MAJORC::mk_reflink (const MAJORC &linkto,
		    int           flags) const
{
    return def ().mk_reflink (linkto, flags);
}

int
MAJORC::mk_directory (XTYPE         type,
		      int          flags) const
{
    return def ().mk_directory (type, flags);
}


int
MAJORC::mk_arealink (const SAREA  &area,
		     int           flags) const
{
    return def ().mk_arealink (area, flags);
}

int
MAJORC::read_arealink (SAREA        &area,
		       int           flags) const
{
    return def ().read_arealink (area, flags);
}
