/*
    XCruise - A directory browser
    Copyright (C) 1999  Yusuke Shinyama <euske@cl.cs.titech.ac.jp>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 *  task.c
 *	Event loop
 */

#include "xcruise.h"

static int exitflag;
static int freezeflag;


/*  Initialize the universe and the ship
 */
void iniTask(void)
{
    int i;
    for(i = 0; i < GalaxyPoly; i++) {
	cirx[i] = cos(i * (M_PI*2.0/(double)GalaxyPoly));
	ciry[i] = sin(i * (M_PI*2.0/(double)GalaxyPoly));
    }
    
    universe.i.type = T_Galaxy;
    universe.i.stat = 0;
    universe.i.name[0] = '\0';
    universe.i.pos = zerop;
    universe.i.r = CurrentSize;
    universe.i.col = RootColor;
    universe.i.px.x = 1;      universe.i.px.y = 0;      universe.i.px.z = 0;
    universe.i.py.x = 0;      universe.i.py.y = 0;      universe.i.py.z = 1;
    
    expandGalaxy(&universe.g);
    curzone = &universe;
    
    viewp.x       = 0.0;  viewp.y       = 0.0;  viewp.z    = -200.0;
    forwardv.x    = 0.0;  forwardv.y    = 0.0;  forwardv.z = 1.0;
    upperv.x      = 0.0;  upperv.y      = 1.0;  upperv.z   = 0.0;
    rightv.x      = 1.0;  rightv.y      = 0.0;  rightv.z   = 0.0;
    vspeed = 0.0;
}

/*  Calculate the speed of the ship
 */
static void calcSpeed(unsigned int btn)
{
    double accela = accel1;
/* Change for Debian    if (btn & Button3Mask) */
    if (btn & Button2Mask)
	accela = accel2;
    if (btn & Button1Mask)
	vspeed += accela;
/* Change for Debian   else if (btn & Button2Mask) */
    else if (btn & Button3Mask)
	vspeed -= accela;
    else
	vspeed = 0.0;
}

/*  Calculate the direction of the ship
 */
static void calcDir(int mx, int my)
{
    double  dx, dy;
    double  dl;
    Point3D v0;
    
    dx = mx*diraccel / (screenWidth/2);
    dy = my*diraccel / (screenHeight/2);
    dl = sqrt(dx*dx + dy*dy);
    dx = dx * dl * dl;
    dy = dy * dl * dl;
    
    v0.x = dx * rightv.x + forwardv.x;
    v0.y = dx * rightv.y + forwardv.y;
    v0.z = dx * rightv.z + forwardv.z;
    formalize(v0);
    
    forwardv.x = dy * upperv.x + v0.x;
    forwardv.y = dy * upperv.y + v0.y;
    forwardv.z = dy * upperv.z + v0.z;
    formalize(forwardv);
    
    rightv.x = -dx * v0.x + rightv.x;
    rightv.y = -dx * v0.y + rightv.y;
    rightv.z = -dx * v0.z + rightv.z;
    formalize(rightv);
    
    upperv.x = -dy * v0.x + upperv.x;
    upperv.y = -dy * v0.y + upperv.y;
    upperv.z = -dy * v0.z + upperv.z;
    formalize(upperv);
    
    viewp.x += forwardv.x * vspeed;
    viewp.y += forwardv.y * vspeed;
    viewp.z += forwardv.z * vspeed; 
}

/*  Main task
 */
static void appTask(int titleflag, unsigned int btn, int ch, int cv)
{
    int rot1, rot2;

    if (freezeflag)
	return;

    /* move the ship */
    if (ch < 0) ch = 0;
    else if (screenWidth <= ch) ch = screenWidth-1;
    if (cv < 0) cv = 0;
    else if (screenHeight <= cv) cv = screenHeight-1;
    ch = ch - screenWidth/2;
    cv = screenHeight/2 - cv;
    
    calcSpeed(btn);
    calcDir(ch, cv);
    
    /* draw into the image buffer */
    XFillRectangle(disp, myscreen, bggc, 0, 0, screenWidth, screenHeight);
    lnupdate = false;
    drawTheUniverse();
    drawCursor(ch, cv);
    rot1 = asin(forwardv.x) * (180.0 / M_PI);
    if (forwardv.z < 0) rot1 = 180 - rot1;
    rot2 = asin(forwardv.y) * (180.0 / M_PI);
    if (forwardv.z < 0) rot2 = 180 - rot2;
    drawMeasure(rot1, rot2);
    drawInfo(titleflag);
    checkSwitch();
    cleanupGalaxy(&universe.g);
    if (lnupdate) {
	updateLink();
    }
    redrawScreen();
    
    XFlush(disp);
}

/*  Event handlers
 */

/* handle key-in */
static void doKey(char c)
{
    switch(c) {
    case 'q':
    case 'Q':
	exitflag = true;
	break;
    case 'f':
    case 'F':
	freezeflag = ! freezeflag;
	if (freezeflag) {
	    alert("Freeze", 0);
	}
	break;
    }
}

/* handle update events */
static void doUpdate(Region crg)
{
    XRectangle rect;
    rect.x = rect.y = 0;
    rect.width = screenWidth;
    rect.height = screenHeight;
    XSetRegion(disp, defaultgc, crg);
    redrawScreen();
    XSetClipRectangles(disp, defaultgc, 0, 0, &rect, 1, YXBanded);
}

/*  Event loop 
 */
void eventLoop(int titleflag)
{
    Region crg = NULL;
    XEvent ev;
    XRectangle r;

    char cbuff;
    KeySym ksym;
    Window w1, w2;
    unsigned int btn;
    int x0, y0, x, y;
    
    int nc;
    exitflag = false;
    freezeflag = false;
    
    while(!exitflag) {
	usleep(1000 * delayticks);
	while(XPending(disp)) {
	    XNextEvent(disp, &ev);
	    switch(ev.type) {
	    case KeyPress:
		nc = XLookupString((XKeyEvent*)&ev, &cbuff, 1, &ksym, NULL);
		doKey(cbuff);
		break;

	    case Expose:
		if (crg == NULL)
		    crg = XCreateRegion();
		r.x = ev.xexpose.x; r.y = ev.xexpose.y;
		r.width = ev.xexpose.width; r.height = ev.xexpose.height;
		XUnionRectWithRegion(&r, crg, crg);
		if (ev.xexpose.count == 0) {
		    doUpdate(crg);
		    XDestroyRegion(crg);
		    crg = NULL;
		}
		break;

	    case MappingNotify:
		XRefreshKeyboardMapping(&(ev.xmapping));
		break;
	    }
	}
	XQueryPointer(disp, mywin, &w1, &w2, &x0, &y0, &x, &y, &btn);
	appTask(titleflag, btn, x, y);
    }
}
