// $Id: TerrainView.cpp,v 1.26 2003/02/27 23:43:41 zongo Exp $
#include <math.h>
#include <iostream>

#include "TerrainView.h"
#include "Application.h"
#include "Command.h"
#include "Tool.h"

TerrainView::TerrainView() :
  m_ViewPos(),
  m_World(0),
  m_View(0),
  m_Tool(0)
{
  // Set default value of Fov, Distance, Angle & Pitch
  Center();
}


TerrainView::~TerrainView()
{
  if (m_Tool) delete m_Tool;
}


void 
TerrainView::UpdateTool()
{
    // Create undo command if possible
    if (m_Tool) {
	Command* command = m_Tool->CreateCommand();

	if (command)
	{
	    g_Application->PushCommand( command );
	    m_View->Redraw();
	}
	
	m_Tool->Update();
    }
}


void 
TerrainView::SetTool(Tool* tool)
{
    // Create undo command if possible
    if (m_Tool) {
	Command* command = m_Tool->CreateCommand();

	if (command)
	{
	    g_Application->PushCommand (command);
	    m_View->Redraw();
	}
	
	delete m_Tool;
    }

    m_Tool = tool;
}


void 
TerrainView::Update() 
{ 
   if (m_World)
   {
      if (m_View)
      {
	 // Will call this->Render() during openGL calls
	 m_View->Redraw();
      }
      
      // Update Tools to current world (currently, reset it)
      if (m_Tool) delete m_Tool;
      m_Tool = 0;
   }
}


void
TerrainView::ComputeView()
{
  Ark::Matrix44 m;

  m_ViewPos.Y = (m_World) ? m_World->GetHeight(m_ViewPos.X, m_ViewPos.Z) : 0.f;

  Ark::Vector3 Look(m_Distance, 0.0, 0.0);
  m.MakeRotationZ(-m_Pitch);
  Look = m.Transform(Look);
  m.MakeRotationY(m_AngleY);
  Look = m.Transform(Look);

  Ark::Vector3 Obs(m_ViewPos - Look);

  m_Camera.m_PointOfView = Obs;
  m_Camera.m_LookAt = m_ViewPos;
  m_Camera.m_FOV = m_FOV;

  g_Application->GetUI()->UpdateBrowseInfo (m_Camera.m_PointOfView,
					    m_Camera.m_LookAt);
}


bool
TerrainView::Render()
{
  m_World->Render(*m_View->GetRenderer(), m_Camera);

  return true;
}


// FIXME ClampWhatever should be made template or something
scalar
ClampScalar(scalar x, scalar max)
{
  if (x > max)
    return max;
  
  if (x < 0.0)
    return 0.0;
  
  return x;
}


void
TerrainView::Change(Ark::Vector3 newpos, const Ark::Collision &col)
{
  if (m_Tool)
  {
    //std::cerr << "Change: " << newpos << std::endl;
    scalar scale = m_World->m_Scale;

    const int x = static_cast<int>( newpos.X / scale );
    const int z = static_cast<int>( newpos.Z / scale );

    // Call generic interface
    m_Tool->Apply(x,z);
    m_Tool->Apply3D (newpos, col);
    m_World->Changed();
  }
}


void
TerrainView::Drag(scalar dx, scalar dy)
{
  if (!m_World) 
  {
    assert(0 && "TerrainView::Drag called with null m_World");
    return;
  }

  Ark::Matrix44 m;
  m.MakeRotationY(m_AngleY);

  Ark::Vector3 diff(-dy, 0, dx);
  diff.Scale(m_Distance);
  diff = m.Transform(diff);
  m_ViewPos -= diff;

  m_ViewPos.X = ClampScalar( m_ViewPos.X, (scalar) m_World->m_SizeX);
  m_ViewPos.Z = ClampScalar( m_ViewPos.Z, (scalar) m_World->m_SizeZ);
  
  ComputeView();
}


// Zoom (distance += zoom)
void 
TerrainView::Zoom(scalar zoom)
{
  // Change distance
  m_Distance += zoom;

  // Check for correct distance
  if (m_Distance <= 5.0)
  {
    m_Distance = 5.0;
  }
  // Check for maximum distance
  else if (60.0 <= m_Distance)
  {
    m_Distance = 60.0;
  }
  
  ComputeView();
}


// Rotate around Y axis
void 
TerrainView::Rotate(scalar angle)
{
  m_AngleY += angle;
  ComputeView();
}


// Change pitch (slope)
void 
TerrainView::Pitch(scalar angle)
{
  m_Pitch += angle;
  
  // Check for correct angle
  if (m_Pitch <= 15.0)
  {
    m_Pitch = 15.0;
  }
  // Check for maximum distance
  else if (89.0 <= m_Pitch)
  {
    m_Pitch = 89.0;
  }
  
  ComputeView();
}


void
TerrainView::Center()
{
  Ark::HeightField* hf = m_World;
  
  if (hf)
  {
    m_ViewPos = Ark::Vector3( hf->m_Scale * hf->m_SizeX/2.f, 
        0.f, hf->m_Scale * hf->m_SizeZ/2.f);

    //std::cout << "Pos: " << m_ViewPos.X << ", " << m_ViewPos.Z << std::endl;
    //std::cout << "Size: " << hf->m_SizeX << ", "<< hf->m_SizeZ << std::endl;
  }

  m_FOV = 60.0;
  m_AngleY = 45.0;
  m_Pitch = 60.0;
  m_Distance = 24.0;
  ComputeView();
}

