/* stringgadget.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: stringgadget.cc,v 1.21 2002/01/30 02:18:47 ralf Exp $ */

#include "stringgadget.h"
#include "awindow.h"
#include "guielement.h"

const char *StringGadget::type="StringGadget";

StringGadget::~StringGadget()
{
  aguix->cancelCutPaste(this);
  _freesafe(text);
  if(parent!=NULL) {
    parent->removeSubWin(twin);
    parent->removeSubWin(win);
  }
}

StringGadget::StringGadget(AGUIX *taguix):GUIElement(taguix)
{
  data=0;
  w=100;
  h=100;
  text=(char*)_allocsafe(1);
  text[0]='\0';
  cursorpos=0;
  xoffset=0;
  active=false;
  tbg=-1;
  bgset=false;
  selstart=selend=cursorpos;
  font=NULL;
  forbidPosChange=false;
  strongkeycapture=true;
  wantpaste=false;
  pasterequest=0;
}

StringGadget::StringGadget(AGUIX *taguix,int tx,int ty,int width,int height,const char *ttext,int tdata):GUIElement(taguix)
{
  this->data=tdata;
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=height;
  xoffset=0;
  cursorpos=0;
  active=false;
  this->text=dupstring(ttext);
  tbg=-1;
  bgset=false;
  selstart=selend=cursorpos;
  font=NULL;
  forbidPosChange=false;
  strongkeycapture=true;
  wantpaste=false;
  pasterequest=0;
}

StringGadget::StringGadget(AGUIX *taguix,int tx,int ty,int width,const char *ttext,int tdata):GUIElement(taguix)
{
  this->data=tdata;
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=taguix->getCharHeight()+2+6;
  xoffset=0;
  cursorpos=0;
  xoffset=0;
  active=false;
  this->text=dupstring(ttext);
  tbg=-1;
  bgset=false;
  selstart=selend=cursorpos;
  font=NULL;
  forbidPosChange=false;
  strongkeycapture=true;
  wantpaste=false;
  pasterequest=0;
}

void StringGadget::resize(int tw,int th)
{
  if ( tw < 8 ) tw = 8;
  if ( th < 8 ) th = 8;
  this->w=tw;
  this->h=th;
  if(parent!=NULL) {
    parent->resizeSubWin(win,tw,th);
  }
  updateWin();
  redraw();
}

int StringGadget::getData()
{
  return data;
}

void StringGadget::setData(int tdata)
{
  this->data=tdata;
}

void StringGadget::redraw()
{
  bredraw();
  tredraw();
}

void StringGadget::bredraw()
{
  if(parent!=NULL) {
    aguix->SetWindowBG(win,parent->getBG());
    aguix->ClearWin(win);
    aguix->setFG(2);
    aguix->DrawLine(win,0,h-1,0,0);
    aguix->DrawLine(win,0,0,w-1,0);
    aguix->DrawLine(win,2,h-1-2,w-1-2,h-1-2);
    aguix->DrawLine(win,w-1-2,h-1-2,w-1-2,2);
    aguix->setFG(1);
    aguix->DrawLine(win,0,h-1,w-1,h-1);
    aguix->DrawLine(win,w-1,h-1,w-1,0);
    aguix->DrawLine(win,2,h-1-2,2,2);
    aguix->DrawLine(win,2,2,w-1-2,2);
  }
}

void StringGadget::tredraw()
{
  int maxcol=aguix->getMaxCols();
  int newbg;
  GC usegc;
  if(parent==NULL) return;
  if(active==true) {
    if(maxcol>7) {
      newbg=7;
    } else {
      newbg=0;
    }
  } else {
    newbg=0;
  }
  aguix->SetWindowBG(twin,newbg);
  tbg=newbg;
  aguix->ClearWin(twin);
  if(font==NULL) usegc=0; else usegc=font->getGC();
  int cw,ch;
  if(font==NULL) {
    ch=aguix->getCharHeight();
    cw=aguix->getCharWidth();
  } else {
    ch=font->getCharHeight();
    cw=font->getCharWidth();
  }
  char *tstr=(char*)_allocsafe(maxdisplay+1);
  int dx;
  dx=1;
  double f1=((h-6-2)/2)-(ch/2);
  int dy=(int)f1;
  int x1,x2;
  x1=(selstart<=selend?selstart:selend);
  x2=(selstart>selend?selstart:selend);
  if((x2==(int)strlen(text))&&(selstart!=selend)) x2--;

  //TODO: warum zur Hoelle habe ich den zweiten Test hier eingebaut?????
  //  if((xoffset<x1)&&((x1-xoffset)<=maxdisplay)) {
  // so muesste es doch auch gehen
  if( xoffset < x1 ) {
    // links vom x1 ist noch Text
    aguix->setFG(usegc,1);
    strncpy(tstr,text+xoffset, a_min( x1-xoffset, maxdisplay ) );
    tstr[ a_min( x1-xoffset, maxdisplay) ]=0;
    if(font==NULL) aguix->DrawText(twin,tstr,dx,dy);
    else aguix->DrawText(twin,font,tstr,dx,dy);
    dx+=strlen(tstr)*cw;
  }
  //TODO: wiederum: wozu diese Berechnung, x1 ist doch < x2 (siehe oben)
  int maxx=(x1<x2)?x2:x1;
  int minx=(x1<x2)?x1:x2;
  if(!((maxx<xoffset)||((minx-xoffset)>/*=*/maxdisplay))) {
    // die Selektion ist sichtbar
    minx=(x1<xoffset)?xoffset:x1;
    maxx=(x2<(xoffset+maxdisplay))?x2:(xoffset+maxdisplay-1);
    int tl=maxx-minx+1;
    strncpy(tstr,text+minx,tl);
    tstr[tl]=0;
    if(active==true) {
      aguix->setFG(usegc,1);
    } else {
      aguix->setFG(usegc,0);
    }
    aguix->FillRectangle(twin,usegc,dx,dy,tl*cw,ch);
    if(active==true) {
      if(maxcol>7) {
	aguix->setFG(usegc,7);
      } else {
	aguix->setFG(usegc,0);
      }
    } else {
      aguix->setFG(usegc,1);
    }
    if(font==NULL) aguix->DrawText(twin,tstr,dx,dy);
    else aguix->DrawText(twin,font,tstr,dx,dy);
    dx+=tl*cw;
  }
  if((x2-xoffset)<(maxdisplay-1)) {
    int t1;
    t1=x2+1;
    if(t1<x2) t1=x2;
    if((int)strlen(text)>=t1) {
      strncpy(tstr,text+t1,maxdisplay-(x2-xoffset)-1);
      tstr[maxdisplay-(x2-xoffset)-1]=0;
      aguix->setFG(usegc,1);
      if(font==NULL) aguix->DrawText(twin,tstr,dx,dy);
      else aguix->DrawText(twin,font,tstr,dx,dy);
    }
  }
  _freesafe(tstr);
  aguix->Flush();
}

void StringGadget::flush()
{
}

bool StringGadget::isInside(int px,int py)
{
/*  if((px>x)&&(px<=(x+width))) {
    if((py>y)&&(py<=(y+height))) return true;
  }*/
  return false;
}

bool StringGadget::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue;
  returnvalue=false;
  int cw;
  int tx,ty;
  AGMessage *agmsg;
  
  if(font==NULL) {
    cw=aguix->getCharWidth();
  } else {
    cw=font->getCharWidth();
  }
  if(active==true) {
    if(msg->type==ButtonPress) {
      if(msg->window!=twin) {
        active=false;
        tredraw();
//        returnvalue=true;
        agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
        agmsg->type=AG_STRINGGADGET_DEACTIVATE;
        agmsg->stringgadget.sg=this;
	agmsg->stringgadget.ok = false;
        aguix->putAGMsg(agmsg);
      } else {
        if(forbidPosChange==false) {
	  aguix->queryPointer(msg->window,&tx,&ty);
          int nc=tx/cw;
          cursorpos=nc+xoffset;
          if(cursorpos>=(int)strlen(text)) cursorpos=(int)strlen(text);
          if(cursorpos<0) cursorpos=0;
          selstart=selend=cursorpos;
          if(msg->button==Button2) insertSelection();
          redraw();
	}
      }
    } else {
      if(msg->type==KeyPress) {
        ignoreRelease=false;
        int keystate=KEYSTATEMASK(msg->keystate);
        if((msg->key==XK_Right)||
	   (msg->key==XK_End)||
	   ((msg->key==XK_e)&&(keystate==ControlMask))||
	   ((msg->key==XK_f)&&(keystate==ControlMask))) {
	  if(strongkeycapture==true) returnvalue=true;
	  if(forbidPosChange==false) {
            if((((keystate==0)||(keystate==ShiftMask))&&
	       (msg->key==XK_Right))||
	       ((msg->key==XK_f)&&(keystate==ControlMask))) {
              if(cursorpos<(int)strlen(text)) {
                cursorpos++;
                if((cursorpos-xoffset)>=maxdisplay) xoffset++;
	        if(keystate!=ShiftMask) selstart=selend=cursorpos;
  	        else {
 	          selend=cursorpos;
  	          applySelection();
                }
                tredraw();
                returnvalue=true;
              }
            } else if(((keystate&Mod1Mask)==Mod1Mask)||
	              (msg->key==XK_End)||
		      ((msg->key==XK_e)&&(keystate==ControlMask))) {
              cursorpos=(int)strlen(text);
              while((cursorpos-xoffset)>=maxdisplay) xoffset++;
  	      if((keystate&ShiftMask)==0) selstart=selend=cursorpos;
  	      else selend=cursorpos;
              tredraw();
              returnvalue=true;
            }
            agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
            agmsg->type=AG_STRINGGADGET_CURSORCHANGE;
            agmsg->stringgadget.sg=this;
            aguix->putAGMsg(agmsg);
	  }
        } else if((msg->key==XK_Left)||
	          (msg->key==XK_Home)||
		  ((msg->key==XK_a)&&(keystate==ControlMask))||
		  ((msg->key==XK_b)&&(keystate==ControlMask))) {
	  if(strongkeycapture==true) returnvalue=true;
          if(forbidPosChange==false) {
            if((((keystate==0)||(keystate==ShiftMask))&&
	       (msg->key==XK_Left))||
	       ((msg->key==XK_b)&&(keystate==ControlMask))) {
              if(cursorpos>0) {
                cursorpos--;
                if((cursorpos-xoffset)<0) xoffset--;
	        if(keystate!=ShiftMask) selstart=selend=cursorpos;
	        else {
	          selend=cursorpos;
	  	  applySelection();
  	        }
                tredraw();
                returnvalue=true;
              }
            } else if(((keystate&Mod1Mask)==Mod1Mask)||
	              (msg->key==XK_Home)||
		      ((msg->key==XK_a)&&(keystate==ControlMask))) {
              xoffset=0;
              cursorpos=0;
	      if((keystate&ShiftMask)==0) selstart=selend=cursorpos;
  	      else selend=cursorpos;
              tredraw();
              returnvalue=true;
            }
            agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
            agmsg->type=AG_STRINGGADGET_CURSORCHANGE;
            agmsg->stringgadget.sg=this;
            aguix->putAGMsg(agmsg);
	  }
        } else if((msg->key==XK_Delete)||((msg->key==XK_d)&&(keystate==ControlMask))) {
          int len=strlen(text);
	  if(strongkeycapture==true) returnvalue=true;
	  if(selstart!=selend) {
	    removeSelection();
	    tredraw();
	  } else if((cursorpos<len)&&(len>0)) {
            char *tstr=(char*)_allocsafe(len);
            strncpy(tstr,text,cursorpos);
            tstr[cursorpos]=0;
            if(cursorpos<len) strcat(tstr,text+cursorpos+1);
            _freesafe(text);
            text=tstr;
            tredraw();
            returnvalue=true;
            agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
            agmsg->type=AG_STRINGGADGET_CONTENTCHANGE;
            agmsg->stringgadget.sg=this;
            aguix->putAGMsg(agmsg);
          }
        } else if((msg->key==XK_BackSpace)||((msg->key==XK_h)&&(keystate==ControlMask))) {
          int len=strlen(text);
	  if(strongkeycapture==true) returnvalue=true;
	  if(selend!=selstart) {
	    removeSelection();
	    tredraw();
	  } else if(cursorpos>0) {
            char *tstr=(char*)_allocsafe(len);
            tstr[0]=0;
            if(cursorpos>1) {
              strncpy(tstr,text,cursorpos-1);
              tstr[cursorpos-1]=0;
            }
            strcat(tstr,text+cursorpos);
            _freesafe(text);
            text=tstr;
            cursorpos--;
            if((cursorpos-xoffset)<0) xoffset--;
	    selstart=selend=cursorpos;
            tredraw();
            returnvalue=true;
            agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
            agmsg->type=AG_STRINGGADGET_CONTENTCHANGE;
            agmsg->stringgadget.sg=this;
            aguix->putAGMsg(agmsg);
          }
        } else if((((msg->key>=XK_space)&&(msg->key<=0xff))||
	           (msg->key==XK_KP_0)||
	           (msg->key==XK_KP_1)||
	           (msg->key==XK_KP_2)||
	           (msg->key==XK_KP_3)||
	           (msg->key==XK_KP_4)||
	           (msg->key==XK_KP_5)||
	           (msg->key==XK_KP_6)||
	           (msg->key==XK_KP_7)||
	           (msg->key==XK_KP_8)||
	           (msg->key==XK_KP_9))&&(keystate!=ControlMask)) {
          int len=strlen(text);
	  if(selstart!=selend) {
	    removeSelection();
	    tredraw();
	  }
          char *tstr=(char*)_allocsafe(len+2);
          tstr[0]=0;
          strncpy(tstr,text,cursorpos);
          tstr[cursorpos]=0;
          len=strlen(tstr);
	  switch(msg->key) {
	    case XK_KP_0:
	      tstr[len]='0';
	      break;
	    case XK_KP_1:
	      tstr[len]='1';
	      break;
	    case XK_KP_2:
	      tstr[len]='2';
	      break;
	    case XK_KP_3:
	      tstr[len]='3';
	      break;
	    case XK_KP_4:
	      tstr[len]='4';
	      break;
	    case XK_KP_5:
	      tstr[len]='5';
	      break;
	    case XK_KP_6:
	      tstr[len]='6';
	      break;
	    case XK_KP_7:
	      tstr[len]='7';
	      break;
	    case XK_KP_8:
	      tstr[len]='8';
	      break;
	    case XK_KP_9:
	      tstr[len]='9';
	      break;
	    default:
	      tstr[len]=(char)msg->key;
	      break;
	  }
          tstr[len+1]=0;
          strcat(tstr,text+cursorpos);
          _freesafe(text);
          text=tstr;
          cursorpos++;
          if((cursorpos-xoffset)>=maxdisplay) xoffset++;
	  selstart=selend=cursorpos;
          tredraw();
          returnvalue=true;
          agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_STRINGGADGET_CONTENTCHANGE;
          agmsg->stringgadget.sg=this;
          aguix->putAGMsg(agmsg);
        } else if(msg->key==XK_Return) {
	  if(strongkeycapture==true) returnvalue=true;
	}
      } else if((msg->type==KeyRelease)&&(ignoreRelease==false)) {
        int keystate=KEYSTATEMASK(msg->keystate);
        if((msg->key==XK_Return)||(msg->key==XK_Escape)) {
          active=false;
	  selstart=selend=cursorpos;
          tredraw();
          returnvalue=true;
          agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_STRINGGADGET_DEACTIVATE;
          agmsg->stringgadget.sg=this;
	  agmsg->stringgadget.ok = ( msg->key == XK_Return ) ? true : false;
          aguix->putAGMsg(agmsg);
          if(msg->key==XK_Return) {
            agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
            agmsg->type=AG_STRINGGADGET_OK;
            agmsg->stringgadget.sg=this;
            aguix->putAGMsg(agmsg);
          } else {
            agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
            agmsg->type=AG_STRINGGADGET_CANCEL;
            agmsg->stringgadget.sg=this;
            aguix->putAGMsg(agmsg);
          }
        } else if((msg->key==XK_Right)||
		  (msg->key==XK_End)||
		  ((msg->key==XK_e)&&(keystate==ControlMask))||
		  ((msg->key==XK_f)&&(keystate==ControlMask))) {
	  if(strongkeycapture==true) returnvalue=true;
        } else if((msg->key==XK_Left)||
	          (msg->key==XK_Home)||
		  ((msg->key==XK_a)&&(keystate==ControlMask))||
		  ((msg->key==XK_b)&&(keystate==ControlMask))) {
	  if(strongkeycapture==true) returnvalue=true;
        } else if((msg->key==XK_Delete)||((msg->key==XK_d)&&(keystate==ControlMask))) {
	  if(strongkeycapture==true) returnvalue=true;
        } else if((msg->key==XK_BackSpace)||((msg->key==XK_h)&&(keystate==ControlMask))) {
	  if(strongkeycapture==true) returnvalue=true;
        } else if((((msg->key>=XK_space)&&(msg->key<=0xff))||
	           (msg->key==XK_KP_0)||
	           (msg->key==XK_KP_1)||
	           (msg->key==XK_KP_2)||
	           (msg->key==XK_KP_3)||
	           (msg->key==XK_KP_4)||
	           (msg->key==XK_KP_5)||
	           (msg->key==XK_KP_6)||
	           (msg->key==XK_KP_7)||
	           (msg->key==XK_KP_8)||
	           (msg->key==XK_KP_9))&&(keystate!=ControlMask)) {
	  if(strongkeycapture==true) returnvalue=true;
        }
      } else if(msg->type==ButtonRelease) {
        applySelection();
      } else if(E->type==MotionNotify) {
        aguix->queryPointer(msg->window,&tx,&ty);
        int nc=tx/cw;
        int ncp=nc+xoffset;
        if(ncp>=(int)strlen(text)) ncp=strlen(text);
        if(ncp<0) ncp=0;
	cursorpos=ncp;
	if((nc<0)&&(xoffset>0)) {
	  xoffset--;
	  cursorpos=xoffset;
	} else if((nc>=maxdisplay)&&(((int)strlen(text)-xoffset)>=maxdisplay)) {
	  xoffset++;
	  cursorpos=xoffset+maxdisplay-1;
	}
	selend=cursorpos;
        redraw();
      }
    }
  } else if(msg->type==ButtonPress) {
    if(msg->window==twin) {
      active=true;
//      returnvalue=true;
      agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
      agmsg->type=AG_STRINGGADGET_ACTIVATE;
      agmsg->stringgadget.sg=this;
      aguix->putAGMsg(agmsg);
      if(forbidPosChange==false) {
        aguix->queryPointer(msg->window,&tx,&ty);
        int nc=tx/cw;
        cursorpos=nc+xoffset;
        if(cursorpos>=(int)strlen(text)) cursorpos=(int)strlen(text);
        if(cursorpos<0) cursorpos=0;
        selstart=selend=cursorpos;
        if(msg->button==Button2) insertSelection();
      }
      redraw();
    }
  }
  if(msg->type==Expose) {
    if(msg->window==win) bredraw();
    else if(msg->window==twin) tredraw();
  }
  return returnvalue;
}

void StringGadget::setText(const char *new_text)
{
  _freesafe(text);
  text=dupstring(new_text);
  if((xoffset+maxdisplay)>=(int)strlen(new_text)) {
    xoffset=(int)strlen(new_text)-maxdisplay;
    if(xoffset<0) xoffset=0;
  }
  if(cursorpos>=(int)strlen(new_text)) {
    cursorpos=(int)strlen(new_text);
  }
  selstart=selend=cursorpos;
  tredraw();
}

const char *StringGadget::getText()
{
  return text;
}

bool StringGadget::isActive()
{
  return active;
}

void StringGadget::updateWin()
{
  int i1;

  if(parent!=NULL) {
    parent->resizeSubWin(twin,w-6,h-6);
    if(font==NULL) maxdisplay=(w-6-2)/aguix->getCharWidth();
    else maxdisplay=(w-6-2)/font->getCharWidth();
    
    // make the cursor visible again
    if ( maxdisplay > 0 ) {
      // makes no sense if there is no char visible
      i1 = a_min( selstart, selend );
      if ( ( i1 - xoffset + 1) > maxdisplay )
	xoffset = i1 - maxdisplay + 1;
    }
    tredraw();
  }
}

void StringGadget::setParent(AWindow *tparent)
{
  this->parent=tparent;
  win=tparent->getSubWindow(0,x,y,w,h);
  twin=tparent->getSubWindow(win,3,3,w-6,h-6);
  if(font==NULL) maxdisplay=(w-6-2)/aguix->getCharWidth();
  else maxdisplay=(w-6-2)/font->getCharWidth();
}

void StringGadget::activate()
{
  active=true;
  ignoreRelease=true;
  tredraw();
}

void StringGadget::deactivate()
{
  active=false;
  tredraw();
}

void StringGadget::applySelection()
{
  int x1,x2;
  char *buffer;
  x1=(selstart<=selend?selstart:selend);
  x2=(selstart>selend?selstart:selend);
  if(x2==(int)strlen(text)) x2--;
  if(x1<x2) {
    //    XStoreBytes(aguix->getDisplay(),text+x1,x2-x1+1);
    buffer=dupstring(text+x1);
    buffer[x2-x1+1]=0;
    aguix->startCut(this,buffer);
    _freesafe(buffer);
    //    aguix->Flush();
  }
}

void StringGadget::removeSelection()
{
  int x1,x2;
  x1=(selstart<=selend?selstart:selend);
  x2=(selstart>selend?selstart:selend);
  if(x2==(int)strlen(text)) x2--;
  if(x1<x2) {
    char *tstr;
    tstr=(char*)_allocsafe(strlen(text)-x2+x1+1);
    strncpy(tstr,text,x1);
    tstr[x1]=0;
    strcat(tstr,text+x2+1);
    _freesafe(text);
    text=tstr;
    selstart=selend=cursorpos=x1;
    if(cursorpos<xoffset) xoffset=cursorpos;
  }
}

void StringGadget::insertSelection()
{
#if 0
  Display *display=aguix->getDisplay();
#endif
  int nbytes_return;
  char *bytes_return,*tstr;
#if 0
  bytes_return=XFetchBytes(display,&nbytes_return);
  if((nbytes_return>0)&&(bytes_return)) {
    // an Cursorpos einfgen
    int tl=strlen(text);
    tstr=(char*)_allocsafe(tl+nbytes_return+2);
    strncpy(tstr,text,cursorpos);
    tstr[cursorpos]=0;
    strncat(tstr,bytes_return,nbytes_return);
    tstr[cursorpos+nbytes_return]=0;
    strcat(tstr,text+cursorpos);
    _freesafe(text);
    text=tstr;
  }
  if(bytes_return) XFree(bytes_return);
#else
  if(aguix->amiOwner()==true) {
    bytes_return=dupstring(aguix->getCutBuffer());
    nbytes_return=strlen(bytes_return);
    // an Cursorpos einfgen
    int tl=strlen(text);
    tstr=(char*)_allocsafe(tl+nbytes_return+2);
    strncpy(tstr,text,cursorpos);
    tstr[cursorpos]=0;
    strncat(tstr,bytes_return,nbytes_return);
    tstr[cursorpos+nbytes_return]=0;
    strcat(tstr,text+cursorpos);
    _freesafe(text);
    text=tstr;
    _freesafe(bytes_return);
  } else {
    aguix->requestCut(getWindow());
    aguix->startPaste(this);
    wantpaste=true;
    pasterequest=time(NULL);
  }
#endif
}

int StringGadget::setFont(char *fontname)
{
  font=aguix->getFont(fontname);
  updateWin();
  if(font==NULL) return -1;
  return 0;
}

const char *StringGadget::getType()
{
  return type;
}

bool StringGadget::isType(const char *qtype)
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

bool StringGadget::isParent(Window child)
{
  if(child==win) return true;
  if(child==twin) return true;
  return false;
}

int StringGadget::getXOffset()
{
  return xoffset;
}

int StringGadget::getCursor()
{
  return cursorpos;
}

bool StringGadget::isPosChangeForbidden()
{
  return forbidPosChange;
}

void StringGadget::setForbidPosChange(bool nv)
{
  forbidPosChange=nv;
}

void StringGadget::setStrongKeyCapture(bool nv)
{
  strongkeycapture=nv;
}

void StringGadget::paste(unsigned char*buf)
{
  int nbytes_return;
  char *tstr;
  if((wantpaste==true)&&(difftime(time(NULL),pasterequest)<10.0)) {
    if(forbidPosChange==false) {
      selstart=selend=cursorpos;
      nbytes_return=strlen((char*)buf);
      // an Cursorpos einfgen
      int tl=strlen(text);
      tstr=(char*)_allocsafe(tl+nbytes_return+2);
      strncpy(tstr,text,cursorpos);
      tstr[cursorpos]=0;
      strncat(tstr,(char*)buf,nbytes_return);
      tstr[cursorpos+nbytes_return]=0;
      strcat(tstr,text+cursorpos);
      _freesafe(text);
      text=tstr;
      redraw();
    }
  }
}

void StringGadget::cancelpaste()
{
  wantpaste=false;
  pasterequest=0;
}

void StringGadget::cancelcut()
{
  selstart=selend=cursorpos;
  redraw();
}
