/* choosebutton.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: choosebutton.cc,v 1.7 2002/01/30 02:17:44 ralf Exp $ */

#include "choosebutton.h"
#include "awindow.h"
#include "guielement.h"

const char *ChooseButton::type="ChooseButton";

ChooseButton::~ChooseButton()
{
  if(label!=NULL) _freesafe(label);
  if(parent!=NULL) {
    parent->removeSubWin(win);
    parent->removeSubWin(lwin);
  }
}

ChooseButton::ChooseButton(AGUIX *taguix):GUIElement(taguix)
{
  this->data=0;
  this->state=false;
  this->instate=false;
  this->label=(char*)_allocsafe(1);
  this->label[0]=0;
  this->labelcolor=0;
  this->labelpos=LABEL_RIGHT;
  bgset=false;
  cross=false;
  laststate=false;
  font=NULL;
}

ChooseButton::ChooseButton(AGUIX *taguix,int tx,int ty,int width,int height,bool tstate,
                           const char *tlabel,int tlabelpos,int tlabelcolor,int tdata):GUIElement(taguix)
{
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=height;
  this->data=tdata;
  this->state=tstate;
  this->instate=tstate;
  this->label=dupstring(tlabel);
  this->labelpos=tlabelpos;
  this->labelcolor=tlabelcolor;
  bgset=false;
  cross=false;
  font=NULL;
  calcLabelGeometry();
  calcRealPos(tx,ty);
  laststate=tstate;
}

ChooseButton::ChooseButton(AGUIX *taguix,int tx,int ty,bool tstate,
                           const char *tlabel,int tlabelpos,int tlabelcolor,int tdata):GUIElement(taguix)
{
  int th;

  th = taguix->getCharHeight() + 6;

  this->x=tx;
  this->y=ty;
  this->w = th;
  this->h = th;
  this->data=tdata;
  this->state=tstate;
  this->instate=tstate;
  this->label=dupstring(tlabel);
  this->labelpos=tlabelpos;
  this->labelcolor=tlabelcolor;
  bgset=false;
  cross=false;
  font=NULL;
  calcLabelGeometry();
  calcRealPos(tx,ty);
  laststate=tstate;
}

const char *ChooseButton::getLabel()
{
  return label;
}

void ChooseButton::setLabel(const char *new_text)
{
  if(label!=NULL) _freesafe(label);
  label=dupstring(new_text);
  updateWin();
  lredraw();
}

void ChooseButton::setLabelColor(int color)
{
  if((color<(aguix->getMaxCols()))&&(color>=0)) {
    this->labelcolor=color;
    lredraw();
  }
}

int ChooseButton::getLabelColor()
{
  return labelcolor;
}

void ChooseButton::setLabelPos(int pos)
{
  this->labelpos=pos;
  updateWin();
  lredraw();
}

int ChooseButton::getLabelPos()
{
  return labelpos;
}

int ChooseButton::getData()
{
  return data;
}

void ChooseButton::setData(int tdata)
{
  this->data=tdata;
}

void ChooseButton::redraw()
{
  cbredraw();
  lredraw();
}

void ChooseButton::cbredraw()
{
  if(parent==NULL) return;
  if(win==0) return;
  if(bgset==false) {
    aguix->SetWindowBG(win,parent->getBG());
    aguix->SetWindowBG(lwin,parent->getBG());
    bgset=true;
  }
  aguix->ClearWin(win);
  aguix->setFG(2);
  aguix->DrawLine(win,0,h-1,0,0);
  aguix->DrawLine(win,0,0,w-1,0);
  aguix->setFG(1);
  aguix->DrawLine(win,0,h-1,w-1,h-1);
  aguix->DrawLine(win,w-1,h-1,w-1,1);
  int dx1,dy1;
  if(state==true) {
    dy1=h/2;
    dx1=w/4;
    int dy2=h*3/4;
    int dx2=w/2;
    aguix->DrawLine(win,dx1,dy1,dx2,dy2);
    dy1=h/4;
    dx1=w*3/4;
    aguix->DrawLine(win,dx2,dy2,dx1,dy1);
  }
  aguix->Flush();
}

void ChooseButton::lredraw()
{
  GC usegc;
  if(parent==NULL) return;
  if(lwin==0) return;
  if(font==NULL) usegc=0; else usegc=font->getGC();
  if(bgset==false) {
    aguix->SetWindowBG(lwin,parent->getBG());
    aguix->SetWindowBG(win,parent->getBG());
    bgset=true;
  }
  aguix->ClearWin(lwin);
  aguix->setFG(usegc,labelcolor);
  if(font==NULL) aguix->DrawText(lwin,label,0,0);
  else aguix->DrawText(lwin,font,label,0,0);
  aguix->Flush();
}

void ChooseButton::flush()
{
}

void ChooseButton::setState(bool tstate)
{
  this->state=tstate;
  cbredraw();
}

int ChooseButton::getState()
{
  return state;
}

bool ChooseButton::isInside(int px,int py)
{
  if((px>0)&&(px<=w)) {
    if((py>0)&&(py<=h)) return true;
  }
  return false;
}

bool ChooseButton::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue;
  returnvalue=false;
  if(msg->window==win) {
    if((msg->type==ButtonPress)||(msg->type==ButtonRelease)) {
      if(msg->type==ButtonPress) {
        if(msg->window==win) {
          AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_CHOOSEPRESSED;
          agmsg->choose.button=this;
          laststate=state;
          if(state==false) {
            setState(true);
            instate=true;
          } else {
            setState(false);
            instate=false;
          }
          cross=true;
          agmsg->choose.state=state;
          aguix->putAGMsg(agmsg);
          returnvalue=true;
        }
      } else {
        if(state==instate) {
          AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_CHOOSERELEASED;
          agmsg->choose.button=this;
          agmsg->choose.state=state;
          aguix->putAGMsg(agmsg);
          agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_CHOOSECLICKED;
          agmsg->choose.button=this;
          agmsg->choose.state=state;
          aguix->putAGMsg(agmsg);
          laststate=state;
        }
        cross=false;
      }
    } else if(msg->type==EnterNotify) {
      if(cross==true) {
        if(instate!=state) {
          setState(instate);
        }
      }
    } else if(msg->type==LeaveNotify) {
      if(cross==true) {
        setState(laststate);
      }
    }
  }
  if(msg->type==Expose) {
    if(msg->window==win) cbredraw();
    else if(msg->window==lwin) lredraw();
  }
//  return returnvalue;
  return false; // see cyclebutton.cc
}

void ChooseButton::resize(int tw,int th)
{
  this->w=tw;
  this->h=th;
  updateWin();
  if(parent!=NULL) parent->resizeSubWin(win,tw,th);
}

void ChooseButton::move(int nx,int ny)
{
  x=nx;
  y=ny;
  calcRealPos(x,y);
  updateWin();
  if(parent!=NULL) parent->moveSubWin(win,x,y);
}

int ChooseButton::getLabelWidth()
{
  if(label!=NULL) {
    int l;
    if(font==NULL) l=strlen(label)*aguix->getCharWidth();
    else l=strlen(label)*font->getCharWidth();
    return l;
  } else return 0;
}

int ChooseButton::getLabelHeight()
{
  if(font==NULL) return aguix->getCharHeight();
  else return font->getCharHeight();
}

void ChooseButton::calcLabelGeometry()
{
  double f1;
  lw=getLabelWidth();
  lh=getLabelHeight();
  switch(labelpos) {
    case LABEL_LEFT:
      lx=x-5-lw;
      f1=y+h/2-lh/2;
      ly=(int)f1;
      break;
    case LABEL_RIGHT:
      lx=x+w+5;
      f1=y+h/2-lh/2;
      ly=(int)f1;
      break;
    case LABEL_TOP:
      ly=y-5-lh;
      f1=x+w/2-lw/2;
      lx=(int)f1;
      break;
    case LABEL_BOTTOM:
      ly=y+h+5;
      f1=x+w/2-lw/2;
      lx=(int)f1;
      break;
  }
  if(lw<1) lw=1;
}

void ChooseButton::updateWin()
{
  calcLabelGeometry();
  if(parent!=NULL) {
    parent->moveSubWin(lwin,lx,ly);
    parent->resizeSubWin(lwin,lw,lh);
  }
}

void ChooseButton::setParent(AWindow *tparent)
{
  this->parent=tparent;
  win=parent->getSubWindow(0,x,y,w,h);
  calcLabelGeometry();
  lwin=parent->getSubWindow(0,lx,ly,lw,lh);
}

int ChooseButton::setFont(char *fontname)
{
  font=aguix->getFont(fontname);
  if(font==NULL) return -1;
  updateWin();
  return 0;
}

const char *ChooseButton::getType()
{
  return type;
}

bool ChooseButton::isType(const char *qtype)
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

bool ChooseButton::isParent(Window child)
{
  if(child==win) return true;
  if(child==lwin) return true;
  return false;
}

int ChooseButton::getX()
{
  return ((x<lx)?x:lx);
}

int ChooseButton::getY()
{
  return ((y<ly)?y:ly);
}

int ChooseButton::getWidth()
{
  if((labelpos==LABEL_TOP)||(labelpos==LABEL_BOTTOM)) {
    return ((w>lw)?w:lw);
  }
  return lw+w+5;
}

int ChooseButton::getHeight()
{
  if((labelpos==LABEL_LEFT)||(labelpos==LABEL_RIGHT)) {
    return ((h>lh)?h:lh);
  }
  return lh+h+5;
}

void ChooseButton::calcRealPos(int tx,int ty)
{
  switch(labelpos) {
    case LABEL_TOP:
    case LABEL_BOTTOM:
      if(lw<w) {
        x=tx;
      } else {
        double b=tx+lw/2-w/2;
        x=(int)b;
      }
      break;
    case LABEL_RIGHT:
      x=tx;
      break;
    default: // LABEL_LEFT
      x=tx+5+lw;
  }
  switch(labelpos) {
    case LABEL_TOP:
      y=ty+5+lh;
      break;
    case LABEL_BOTTOM:
      y=ty;
      break;
    default: // LABEL_LEFT & LABEL_RIGHT
      if(lh<h) {
        y=ty;
      } else {
        double b=ty+lh/2-h/2;
        y=(int)b;
      }
  }
}
