/* wpucontext.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2002-2005,2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "wpucontext.h"
#include "scriptop.h"
#include "normalmode.h"
#include "execlass.h"
#include "nmextlist.hh"

WPUStack::WPUStack()
{
  s = 10;
  elems = 0;
  stack = new std::string[s];
}

WPUStack::~WPUStack()
{
  delete [] stack;
}

void WPUStack::push( std::string elem )
{
  if ( elems >= s ) raise();
  stack[elems++] = elem;
}

std::string WPUStack::pop()
{
  if ( elems < 1 ) return "";
  return stack[--elems];
}

std::string WPUStack::top()
{
  if ( elems < 1 ) return "";
  return stack[elems - 1];
}

bool WPUStack::isEmpty()
{
  if ( elems < 1 ) return true;
  return false;
}

int WPUStack::size()
{
  return elems;
}

void WPUStack::raise()
{
  std::string *newstack;
  int i, old_s;
  
  old_s = s;
  s *= 2;
  newstack = new std::string[s];
  for ( i = 0; i < old_s; i++ )
    newstack[i] = stack[i];
  delete [] stack;
  stack = newstack;
}

/**************
 * WPUContext *
 **************/

struct wpucontext_deftokens {
  const char *str;
  enum WPUContext::ifp_t token;
} keywords[] = {
  { "isEmpty", WPUContext::IFP_ISEMPTY },
  { "size", WPUContext::IFP_SIZE },
  { "lasterror", WPUContext::IFP_LASTERROR },
  { "true", WPUContext::IFP_TRUE },
  { "false", WPUContext::IFP_FALSE },
  { "filelistEmpty", WPUContext::IFP_FILELISTEMPTY },
  { "toNum", WPUContext::IFP_TONUM },
  { "toStr", WPUContext::IFP_TOSTR },
  { "isLocal", WPUContext::IFP_ISLOCAL },
  { NULL, WPUContext::IFP_NONE }
};

//TODO vielleicht auch noch ONLYACTIVE untersttzen
WPUContext::WPUContext( Worker *w, List *coms, ActionMessage *msg )
{
  int id;
  FunctionProto *fp;
  struct wpucontext_deftokens *p;

  nr_of_stacks = 0;
  stacks = NULL;
  pc = 0;
  acoms = new ArrayList();
  
  id = coms->initEnum();
  fp = (FunctionProto*)coms->getFirstElement( id );
  while ( fp != NULL ) {
    acoms->addElement( fp );
    fp = (FunctionProto*)coms->getNextElement( id );
  }
  coms->closeEnum( id );

  extlist[0] = NULL;
  extlist[1] = NULL;
  worker = w;

  curpos = 0;
  tokenval = -1;
  lastentry = 0;
  lastchar = -1;
  ignore = 0;
  scanbuf = NULL;
  scanbuflen = -1;
  for ( p = keywords; p->token != IFP_NONE; p++ ) {
    ifp_insert( p->str, p->token );
  }
  extorder.wpu = this;
  lasterror = 0;
  
  userwin = NULL;
  cancelb = NULL;
  bb1 = NULL;
  bb2 = NULL;
  prosb = NULL;
  texts = NULL;
  nr_of_text = 0;
  progress = 100;

  activemode = worker->getActiveLister()->getActiveMode();
  nonactivemode = worker->getOtherLister( worker->getActiveLister() )->getActiveMode();

  if ( msg != NULL ) {
    if ( msg->mode != ActionMessage::AM_MODE_NORMAL ) {
      if ( msg->mode == ActionMessage::AM_MODE_DNDACTION ) {
        const DNDMsg *dndmsg = msg->dndmsg;
        ListerMode *lm1;
        NormalMode *nm1;

        lm1 = dndmsg->getDestMode();
        if ( lm1 != NULL ) {
          if ( lm1->isType( "NormalMode" ) == true ) {
            nm1 = (NormalMode*)lm1;
            extorder.destmode = nm1;
          }
        }
        lm1 = dndmsg->getSourceMode();
        if ( lm1 != NULL ) {
          if ( lm1->isType( "NormalMode" ) == true ) {
            nm1 = (NormalMode*)lm1;
            extlist[0] = new NMExtList();
            extlist[0]->createExtList( nm1, msg, false );
            extlist[0]->setTakeDirs( extorder.take_dirs );
          }
        }
        activemode = lm1;
        nonactivemode = worker->getOtherLister( dndmsg->getSourceLister() )->getActiveMode();
      } else if ( msg->mode == ActionMessage::AM_MODE_SPECIAL ) {
        NormalMode *nm1;

        if ( activemode != NULL ) {
          if ( activemode->isType( "NormalMode" ) == true ) {
            nm1 = (NormalMode*)activemode;
            extlist[0] = new NMExtList();
            extlist[0]->createExtList( nm1, msg, false );
            extlist[0]->setTakeDirs( extorder.take_dirs );
          }
        }
      }
    }
  }
}

WPUContext::~WPUContext()
{
  closeWin();
  delete acoms;
  for ( int i = 0; i < nr_of_stacks; i++ ) {
    delete stacks[i];
  }
  delete [] stacks;
  if ( extlist[0] != NULL ) {
    delete extlist[0];
  }
  if ( extlist[1] != NULL ) {
    delete extlist[1];
  }
  freeTempNameList();
}

int WPUContext::continueAtLabel( const char *label )
{
  /* find label and set pc to position */
  int pos, id;
  FunctionProto *fp;
  ScriptOp *sop;
  
  pos = 0;
  // whithout useful label jump to the end to avoid problems
  if ( label != NULL ) {
    if ( strlen( label ) > 0 ) {
      id = acoms->initEnum();
      while ( pos < acoms->size() ) {
        fp = (FunctionProto*)acoms->getElementAt( id, pos );
        if ( strcmp( fp->getName(), "ScriptOp" ) == 0 ) {
          sop = (ScriptOp*)fp;
          if ( sop->isLabel( label ) == true ) {
             break;
          }
        }
        pos++;
      }
      acoms->closeEnum( id );
    } else pos = acoms->size();
  } else pos = acoms->size();
  pc = pos;  /* when label was not found, finish whole execution
                because it's a error in list and I don't want to
                do something wrong */
  if ( pc >= acoms->size() ) return 1;
  return 0;
}

void WPUContext::push( int nr, std::string elem )
{
  if ( nr >= nr_of_stacks ) buildStack( nr );
  stacks[ nr ]->push( elem );
}

std::string WPUContext::pop( int nr )
{
  if ( nr >= nr_of_stacks ) buildStack( nr );
  if ( stacks[ nr ]->size() < 1 ) {
    lasterror = WPU_LASTERROR_STACKEMPTY;
    // we can still call top because it will return ""
  }
  return stacks[ nr ]->pop();
}

std::string WPUContext::top( int nr )
{
  if ( nr >= nr_of_stacks ) buildStack( nr );
  if ( stacks[ nr ]->size() < 1 ) {
    lasterror = WPU_LASTERROR_STACKEMPTY;
    // we can still call top because it will return ""
  }
  return stacks[ nr ]->top();
}

bool WPUContext::isEmpty( int nr )
{
  if ( nr >= nr_of_stacks ) buildStack( nr );
  return stacks[ nr ]->isEmpty();
}

int WPUContext::size( int nr )
{
  if ( nr >= nr_of_stacks ) buildStack( nr );
  return stacks[ nr ]->size();
}

int WPUContext::next( ActionMessage *msg )
{
  FunctionProto *fp;
  AGUIX *aguix = worker->getAGUIX();
  AGMessage *agmsg;
  bool cancel = false;

  if ( pc >= acoms->size() ) return 1;
  fp = (FunctionProto*)acoms->getElementAt( pc );
  if ( fp != NULL ) {
    fp->run( this, msg );
    pc++;
    
    if ( userwin != NULL ) {
      while ( ( agmsg = aguix->GetMessage( userwin ) ) != NULL ) {
        if ( agmsg->type == AG_BUTTONCLICKED ) {
          if ( agmsg->button.button == cancelb )
            cancel = true;
        } else if ( agmsg->type == AG_CLOSEWINDOW ) {
          if ( agmsg->closewindow.window == userwin->getWindow() )
            cancel = true;
        }
        aguix->ReplyMessage( agmsg );
      }
    }
  } else {
    return 1;
  }
  return ( cancel == true ) ? 1 : 0;
}

/*
 * buildStack
 * builds stack up to nr !
 */
void WPUContext::buildStack( int nr )
{
  WPUStack **newstacks;
  int i;
  
  if ( nr >= nr_of_stacks ) {
    newstacks = new WPUStack*[ nr + 1 ];
    for ( i = 0; i < nr_of_stacks; i++ )
      newstacks[ i ] = stacks[ i ];
    for ( i = nr_of_stacks; i < ( nr + 1); i++ )
      newstacks[ i ] = new WPUStack();
    delete [] stacks;
    stacks = newstacks;
    nr_of_stacks = nr + 1;
  }
}

int WPUContext::openWin()
{
  AGUIX *aguix;
  int tx, ty, tw;
  
  if ( userwin == NULL ) {
    aguix = worker->getAGUIX();
    userwin = new AWindow( aguix, 10, 10, 100, 50, 0, catalog.getLocale( 499 ) );
    userwin->create();
    tx = ty = 5;
    nr_of_text = 1;
    texts = new Text*[nr_of_text];
    texts[0] = (Text*)userwin->add( new Text( aguix, tx, ty, "", 1 ) );
    ty += texts[0]->getHeight() + 5;
    bb1 = (BevelBox*)userwin->add( new BevelBox( aguix, tx, ty, 10, 10, 1 ) );
    prosb = (SolidButton*)userwin->add( new SolidButton( aguix, tx + 1, ty + 1, 10, "", 2, 3, 0 ) );
    bb1->resize( bb1->getWidth(), prosb->getHeight() + 2 );
    ty += bb1->getHeight() + 5;
    bb2 = (BevelBox*)userwin->add( new BevelBox( aguix, tx, ty, 10, 2, 0 ) );
    ty += bb2->getHeight() + 5;
    cancelb = (Button*)userwin->add( new Button( aguix, tx, ty, catalog.getLocale( 8 ), 1, 0, 0 ) );
    ty += cancelb->getHeight() + 5;
    userwin->maximizeX();
    userwin->maximizeY();

    GUIElement *tbutton[1];
    tbutton[0] = cancelb;
    tw = AGUIX::scaleElementsW( userwin->getWidth(), 5, 0, -1, false, false, tbutton, NULL, 1 );
    
    if ( tw > userwin->getWidth() ) {
      userwin->resize( tw, userwin->getHeight() );
    }
    
    tw = userwin->getWidth();
    bb1->resize( tw - 10, bb1->getHeight() );
    setWinProgress( progress );
    bb2->resize( tw - 10, bb2->getHeight() );
    
    userwin->setMinSize( userwin->getWidth(), userwin->getHeight() );
    userwin->setMaxSize( userwin->getWidth(), userwin->getHeight() );
    userwin->show();
  }
  return 0;
}

int WPUContext::closeWin()
{
  if ( userwin != NULL ) {
    delete userwin;
    delete [] texts;
    userwin = NULL;
  }
  return 0;
}

char *WPUContext::parse( const char *str1, int maxlen, bool quote )
{
  char *tstr = NULL;
  NormalMode *nm;
  
  if ( worker != NULL ) {
    buildExtList();
    if ( activemode != NULL ) {
      if ( activemode->isType( "NormalMode" ) == true ) {
        nm = (NormalMode*)activemode;
        tstr = nm->parseComStrExt( str1, &extorder, maxlen, extlist, quote );
      }
    }
  }
  return tstr;
}

int WPUContext::ifp_lookup( const char *s )
{
  int p;

  for ( p = lastentry; p > 0; p = p - 1 )
    if ( strcasecmp( ifp_symtable[p].str, s ) == 0 )
      return p;
  return 0;
}

int WPUContext::ifp_insert( const char *s, enum ifp_t tok )
{
  int len;

  len = strlen( s );
  if ( lastentry + 1 >= IFP_SYMTABLE_MAX ) {
    // Symboltabelle voll
    return -1;
  }
  if ( lastchar + len + 1 >= STRMAX ) {
    // Lexemfeld voll
    return -1;
  }
  lastentry = lastentry + 1;
  ifp_symtable[lastentry].token = tok;
  ifp_symtable[lastentry].str = &lexemes[lastchar + 1];
  lastchar = lastchar + len + 1;
  strcpy( ifp_symtable[lastentry].str, s );
  return lastentry;
}

int WPUContext::ifp_lex()
{
  int t;

  while ( 1 ) {
    if ( curpos >= scanbuflen ) break;
    t = scanbuf[curpos++];
    if ( isspace( t ) ) {
    } else if ( isdigit( t ) ) {
      tokenval = t - '0';
      while ( isdigit( t = scanbuf[curpos++] ) ) {
	tokenval = 10 * tokenval + ( t - '0' );
      }
      curpos--;
      return IFP_NUM;
    } else if ( isalpha( t ) || ( t == '_' ) ) {
      int p, b = 0;

      while ( isalnum( t ) || ( t == '_' ) ) {
	ifp_lexbuf[b] = t;
	t = scanbuf[curpos++];
	b = b + 1;
	if ( ( b + 1 ) >= IFP_BUFSIZE ) {
	  // Pufferueberlauf
          break;
        }
      }
      ifp_lexbuf[b] = '\0';
      if ( t != '\0' ) {
	curpos--;
      }
      p = ifp_lookup( ifp_lexbuf );
      if ( p == 0 ) {
	return IFP_NONE;
      }
      tokenval = p;
      return ifp_symtable[p].token;
    } else if ( t == '"' ) {
      int b = 0;
      int bs = 0;

      for ( ;; ) {
	t = scanbuf[curpos++];
	if ( t == EOF )
	  break;
	if ( ( t == '\\' ) && ( bs == 0 ) ) {
	  bs = 1;
	  continue;
	}
	if ( ( t == '"' ) && ( bs == 0 ) )
	  break;
	bs = 0;
	ifp_lexbuf[b] = t;
	b = b + 1;
	if ( ( b + 1 ) >= IFP_BUFSIZE ) {
	  // Pufferueberlauf
          break;
        }
      }
      ifp_lexbuf[b] = '\0';
      return IFP_STRING;
    } else if ( t == '&' ) {
      t = scanbuf[curpos++];
      if ( t == '&' )
	return IFP_P_AND;
      return IFP_NONE;
    } else if ( t == '|' ) {
      t = scanbuf[curpos++];
      if ( t == '|' )
	return IFP_P_OR;
      return IFP_NONE;
    } else if ( t == '=' ) {
      t = scanbuf[curpos++];
      if ( t == '=' )
        return IFP_E;
      return IFP_NONE;
    } else if ( t == '!' ) {
      t = scanbuf[curpos++];
      if ( t == '=' )
        return IFP_NE;
      return IFP_NONE;
    } else if ( t == '<' ) {
      t = scanbuf[curpos++];
      if ( t == '=' )
        return IFP_LE;
      curpos--;
      return IFP_LT;
    } else if ( t == '>' ) {
      t = scanbuf[curpos++];
      if ( t == '=' )
        return IFP_GE;
      curpos--;
      return IFP_GT;
    } else if ( t == '\0' ) {
      return IFP_DONE;
    } else {
      tokenval = IFP_NONE;
      return t;
    }
  }
  return IFP_DONE;
}

int WPUContext::ifp_match( int t )
{
  if ( lookahead == t ) {
    lookahead = ifp_lex();
  } else {
    return 1;
  }
  return 0;
}

char *WPUContext::readBrace()
{
  int startpos, endpos;
  int bracecount = 0;
  int t, ig = 0, len, readpos;
  char *tstr;
  
  startpos = curpos;
  if ( scanbuf[curpos] == '{' ) {
    do {
      t = scanbuf[curpos++];
      if ( t == '\0' ) break;
      else if ( ( t == '\\' ) && ( ig == 0 ) ) {
        ig = 1;
        continue;
      } else if ( ( t == '{' ) && ( ig == 0 ) ) {
        bracecount++;
      } else if ( ( t == '}' ) && ( ig == 0 ) ) {
        bracecount--;
      }
      ig = 0;
    } while ( bracecount > 0 );
  }
  endpos = curpos;

  len = endpos - startpos;
  tstr = (char*)_allocsafe( len + 1 );
  strncpy( tstr, scanbuf + startpos, len );
  /* now turn backslashed {/} to normal {/} and remove first and last brace */
  t = 0; readpos = 1;
  for( readpos = 1; readpos < ( len - 1); readpos++ ) {
    if ( ( tstr[readpos] == '\\' ) &&
         ( ( tstr[readpos + 1 ] == '{' ) || ( tstr[readpos + 1] == '}' ) ) ) {
      tstr[t] = tstr[readpos + 1];
      readpos++;
    } else {
      tstr[t] = tstr[readpos];
    }
    t++;
  }
  tstr[t] = '\0';
  return tstr;
}

int WPUContext::ifp_parse()
{
  struct ifp_erg *e = NULL;
  int erg;
  
  lookahead = ifp_lex();
  if ( lookahead != IFP_DONE ) {
    e = ifp_expr();
  }
  if ( e == NULL ) return -1; /* error */
  erg = e->numval;
  _freesafe( e->strval );
  _freesafe( e );
  if ( lookahead != IFP_DONE ) return -1; /* Not at the end => error */
  return ( erg == 0 ) ? 0 : 1;
}

struct WPUContext::ifp_erg * WPUContext::ifp_expr()
{
  struct ifp_erg *e = NULL, *e2;
  int oldignore;
  int l;
  
  if ( ( e = ifp_factor() ) == NULL ) return NULL;
  while ( 1 ) {
    switch ( lookahead ) {
      case IFP_P_AND:
	if ( ifp_match( IFP_P_AND ) ) {
          _freesafe( e->strval );
          _freesafe( e );
          return NULL;
        }

        oldignore = ignore;
        if ( e->numval == 0 ) ignore = 1;

	if ( ( e2 = ifp_factor() ) == NULL ) {
          _freesafe( e->strval );
          _freesafe( e );
          return NULL;
        }

        ignore = oldignore;

        if ( e->numval != 0 ) {
          e->numval = e2->numval;
        }
        _freesafe( e->strval );
        e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
        e->wasstring = 0;
        sprintf( e->strval, "%d", e->numval );
        _freesafe( e2->strval );
        _freesafe( e2 );
        break;
      case IFP_P_OR:
	if ( ifp_match( IFP_P_OR ) ) {
          _freesafe( e->strval );
          _freesafe( e );
          return NULL;
        }

        oldignore = ignore;
        if ( e->numval != 0 ) ignore = 1;

	if ( ( e2 = ifp_factor() ) == NULL ) {
          _freesafe( e->strval );
          _freesafe( e );
          return NULL;
        }

        ignore = oldignore;

        if ( e->numval == 0 ) {
          e->numval = e2->numval;
        }
        _freesafe( e->strval );
        e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
        e->wasstring = 0;
        sprintf( e->strval, "%d", e->numval );
        _freesafe( e2->strval );
        _freesafe( e2 );
        break;
      case IFP_E:
      case IFP_NE:
      case IFP_LT:
      case IFP_LE:
      case IFP_GT:
      case IFP_GE:
        l = lookahead;
	if ( ifp_match( lookahead ) ) {
          _freesafe( e->strval );
          _freesafe( e );
          return NULL;
        }

	if ( ( e2 = ifp_factor() ) == NULL ) {
          _freesafe( e->strval );
          _freesafe( e );
          return NULL;
        }

        if ( ( e->wasstring == 1 ) && ( e2->wasstring == 1 ) ) {
          /* only do strcmp when both are strings */
          switch ( l ) {
            case IFP_NE:
              e->numval = ( strcmp( e->strval, e2->strval ) != 0 );
              break;
            case IFP_LT:
              e->numval = ( strcmp( e->strval, e2->strval ) < 0 );
              break;
            case IFP_LE:
              e->numval = ( strcmp( e->strval, e2->strval ) <= 0 );
              break;
            case IFP_GT:
              e->numval = ( strcmp( e->strval, e2->strval ) > 0 );
              break;
            case IFP_GE:
              e->numval = ( strcmp( e->strval, e2->strval ) >= 0 );
              break;
            default:
              e->numval = ( strcmp( e->strval, e2->strval ) == 0 );
              break;
          }
        } else {
          switch ( l ) {
            case IFP_NE:
              e->numval = ( e->numval != e2->numval );
              break;
            case IFP_LT:
              e->numval = ( e->numval < e2->numval );
              break;
            case IFP_LE:
              e->numval = ( e->numval <= e2->numval );
              break;
            case IFP_GT:
              e->numval = ( e->numval > e2->numval );
              break;
            case IFP_GE:
              e->numval = ( e->numval >= e2->numval );
              break;
            default:
              e->numval = ( e->numval == e2->numval );
              break;
          }
        }
        _freesafe( e->strval );
        e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
        e->wasstring = 0;
        sprintf( e->strval, "%d", e->numval );
        _freesafe( e2->strval );
        _freesafe( e2 );
        break;
      default:
	return e;
    }
  }
  return e;
}

struct WPUContext::ifp_erg *WPUContext::ifp_factor()
{
  struct ifp_erg *e = NULL;
  int tnum;
  char *tstr, *tstr2;
  int exeerror;

  switch ( lookahead ) {
    case '(':
      if ( ifp_match( '(' ) ) return NULL;
      if ( ( e = ifp_expr() ) == NULL ) return NULL;
      if ( ifp_match( ')' ) ) {
        _freesafe( e->strval );
        _freesafe( e );
        return NULL;
      }
      break;
    case '-':
      if ( ifp_match( '-' ) ) return NULL;
      tnum = tokenval;
      if ( ifp_match( IFP_NUM ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      e->numval = tnum * -1;
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_NUM:
      tnum = tokenval;
      if ( ifp_match( IFP_NUM ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      e->numval = tnum;
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_ISEMPTY:
      if ( ifp_match( IFP_ISEMPTY ) ) return NULL;
      if ( ifp_match( '(' ) ) return NULL;
      tnum = tokenval;
      if ( ifp_match( IFP_NUM ) ) return NULL;
      if ( ifp_match( ')' ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      if ( ( tnum < 0 ) || ( ignore == 1 ) ) {
        e->numval = 0;
      } else {
        e->numval = isEmpty( tnum );
      }
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_FILELISTEMPTY:
      if ( ifp_match( IFP_FILELISTEMPTY ) ) return NULL;
      if ( ifp_match( '(' ) ) return NULL;
      tnum = tokenval;
      if ( ifp_match( IFP_NUM ) ) return NULL;
      if ( ifp_match( ')' ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      if ( ( tnum < 0 ) || ( tnum > 1 ) || ( ignore == 1 ) ) {
        e->numval = 0;
      } else {
        e->numval = filelistEmpty( ( tnum == 0 ) ? false : true );
      }
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_TRUE:
      if ( ifp_match( IFP_TRUE ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      e->numval = 1;
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_FALSE:
      if ( ifp_match( IFP_FALSE ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      e->numval = 0;
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_SIZE:
      if ( ifp_match( IFP_SIZE ) ) return NULL;
      if ( ifp_match( '(' ) ) return NULL;
      tnum = tokenval;
      if ( ifp_match( IFP_NUM ) ) return NULL;
      if ( ifp_match( ')' ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      if ( ( tnum < 0 ) || ( ignore == 1 ) ) {
        e->numval = 0;
      } else {
        e->numval = size( tnum );
      }
      e->wasstring = 0;
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_LASTERROR:
      if ( ifp_match( IFP_LASTERROR ) ) return NULL;
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      e->numval = getLastError();
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      resetLastError();
      break;
    case '?':
      tstr = readBrace();
      if ( tstr != NULL ) {
        if ( ignore == 0 ) {
          /*execute tstr */
          tstr2 = parse( tstr, a_max( EXE_STRING_LEN - 1024, 256 ) );
          _freesafe( tstr );
          tstr = tstr2;
  
          if ( tstr != NULL ) {
            e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
            std::auto_ptr<ExeClass> ec( new ExeClass( ExeClass::EXECUTE_NORMAL, worker ) );
            ec->cdDir( getBaseDir() );
            ec->addCommand( "%s", tstr );
            e->numval = ec->getReturnCode( &exeerror );
            e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
            e->wasstring = 0;
            sprintf( e->strval, "%d", e->numval );
            
            if ( exeerror != 0 ) {
              if ( worker->getRequester() != NULL ) {
                char *reqstr;
                std::string error, str1;
                
                reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 749 ) ) + strlen( tstr ) + 1 );
                sprintf( reqstr, catalog.getLocale( 749 ), tstr );
                
                ec->readErrorOutput( error, 1024 );
                
                str1 = reqstr;
                str1 += error;
                _freesafe( reqstr );

                RefCount<AFontWidth> lencalc( new AFontWidth( worker->getAGUIX(), NULL ) );
                TextStorageString ts( str1, lencalc );

                worker->getRequester()->request( catalog.getLocale( 347 ), ts, catalog.getLocale( 11 ) );
              }
              _freesafe( e->strval );
              _freesafe( e );
              _freesafe( tstr );
              return NULL;
            }
            _freesafe( tstr );
          }
        } else {
          e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
          e->strval = dupstring( "0" );
          e->wasstring = 1;
          e->numval = atoi( e->strval );
          _freesafe( tstr );
        }
      }
      if ( ifp_match( '?' ) ) {
        if ( e != NULL ) {
          _freesafe( e->strval );
          _freesafe( e );
        }
        return NULL;
      }
      break;
    case '$':
      tstr = readBrace();
      if ( tstr != NULL ) {
        if ( ignore == 0 ) {
          tstr2 = parse( tstr, a_max( EXE_STRING_LEN - 1024, 256 ) );
          _freesafe( tstr );
          tstr = tstr2;
  
          if ( tstr != NULL ) {
            e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
            std::auto_ptr<ExeClass> ec( new ExeClass( ExeClass::EXECUTE_NORMAL, worker ) );
            ec->cdDir( getBaseDir() );
            ec->addCommand( "%s", tstr );
            e->strval = ec->getOutput( &exeerror );
            e->wasstring = 1;
            if ( e->strval == NULL ) {
              e->strval = dupstring( "" );
            }
            e->numval = atoi( e->strval );

            if ( exeerror != 0 ) {
              if ( worker->getRequester() != NULL ) {
                char *reqstr;
                std::string error, str1;
                
                reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 749 ) ) + strlen( tstr ) + 1 );
                sprintf( reqstr, catalog.getLocale( 749 ), tstr );
                
                ec->readErrorOutput( error, 1024 );
                
                str1 = reqstr;
                str1 += error;
                _freesafe( reqstr );

                RefCount<AFontWidth> lencalc( new AFontWidth( worker->getAGUIX(), NULL ) );
                TextStorageString ts( str1, lencalc );

                worker->getRequester()->request( catalog.getLocale( 347 ), ts, catalog.getLocale( 11 ) );
              }
              _freesafe( e->strval );
              _freesafe( e );
              _freesafe( tstr );
              return NULL;
            }
            _freesafe( tstr );
          }
        } else {
          e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
          e->strval = dupstring( "0" );
          e->wasstring = 1;
          e->numval = atoi( e->strval );
          _freesafe( tstr );
        }
      }
      if ( ifp_match( '$' ) ) {
        if ( e != NULL ) {
          _freesafe( e->strval );
          _freesafe( e );
        }
        return NULL;
      }
      break;
    case IFP_STRING:
      e = (struct ifp_erg*)_allocsafe( sizeof( struct ifp_erg ) );
      e->wasstring = 1;
      if ( ignore == 1 ) {
        e->strval = dupstring( "0" );
      } else {
        e->strval = parse( ifp_lexbuf, a_max( EXE_STRING_LEN - 1024, 256 ), false );
        if ( e->strval == NULL ) e->strval = dupstring( "0" );
      }
      e->numval = atoi( e->strval );
      if ( ifp_match( IFP_STRING ) ) {
        _freesafe( e->strval );
        _freesafe( e );
        return NULL;
      }
      break;
    case IFP_TONUM:
      if ( ifp_match( IFP_TONUM ) ) return NULL;
      if ( ifp_match( '(' ) ) return NULL;
      if ( ( e = ifp_expr() ) == NULL ) return NULL;
      if ( ifp_match( ')' ) ) {
        _freesafe( e->strval );
        _freesafe( e );
        return NULL;
      }
      _freesafe( e->strval );
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_TOSTR:
      if ( ifp_match( IFP_TOSTR ) ) return NULL;
      if ( ifp_match( '(' ) ) return NULL;
      if ( ( e = ifp_expr() ) == NULL ) return NULL;
      if ( ifp_match( ')' ) ) {
        _freesafe( e->strval );
        _freesafe( e );
        return NULL;
      }
      _freesafe( e->strval );
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 1;
      sprintf( e->strval, "%d", e->numval );
      break;
    case IFP_ISLOCAL:
      if ( ifp_match( IFP_ISLOCAL ) ) return NULL;
      if ( ifp_match( '(' ) ) return NULL;
      if ( ( e = ifp_expr() ) == NULL ) return NULL;
      if ( ifp_match( ')' ) ) {
        _freesafe( e->strval );
        _freesafe( e );
        return NULL;
      }
      e->numval = worker_islocal( e->strval );
      _freesafe( e->strval );
      e->strval = (char*)_allocsafe( A_BYTESFORNUMBER( e->numval ) );
      e->wasstring = 0;
      sprintf( e->strval, "%d", e->numval );
      break;
    default:
      return NULL;
  }
  return e;
}

int WPUContext::parse_if( const char *str)
{
  int erg;
  
  if ( str == NULL ) return -1;
  scanbuf = str;
  scanbuflen = strlen( str );
  curpos = 0;
  erg = ifp_parse();
  scanbuf = NULL;
  scanbuflen = -1;
  return erg;
}

bool WPUContext::filelistEmpty( bool other )
{
  buildExtList();
  if ( extlist[ ( other == true ) ? 1 : 0 ]->getRealSize() < 1 ) return true;
  return false;
}

void WPUContext::buildExtList()
{
  NormalMode *nm;
  
  if ( worker != NULL ) {
    if ( extlist[0] == NULL ) {
      extlist[0] = new NMExtList();
      if ( activemode != NULL ) {
        if ( activemode->isType( "NormalMode" ) == true ) {
          nm = (NormalMode*)activemode;
          extlist[0]->createExtList( nm, extorder.recursive );
        }
      }
      extlist[0]->setTakeDirs( extorder.take_dirs );
    }
    if ( extlist[1] == NULL ) {
      extlist[1] = new NMExtList();
      if ( nonactivemode != NULL ) {
        if ( nonactivemode->isType( "NormalMode" ) == true ) {
          nm = (NormalMode*)nonactivemode;
          extlist[1]->createExtList( nm, extorder.recursive );
        }
      }
      extlist[1]->setTakeDirs( extorder.take_dirs );
    }
  }
}

int WPUContext::getLastError()
{
  return lasterror;
}

void WPUContext::resetLastError()
{
  lasterror = 0;
}

int WPUContext::getCurrentLine()
{
  return pc;
}

void WPUContext::setRecursive( bool nv )
{
  extorder.recursive = nv;
}

bool WPUContext::getRecursive()
{
  return extorder.recursive;
}

void WPUContext::setWinProgress( int nv )
{
  progress = nv;
  char buf[ A_BYTESFORNUMBER( progress ) + 2 + 1 ];
  if ( ( progress < 0 ) || ( progress > 100 ) ) progress = 0;
  if ( userwin != NULL ) {
    prosb->resize( ( bb1->getWidth() - 2 ) * progress / 100, prosb->getHeight() );
    sprintf( buf, "%d %%", progress );
    prosb->setText( buf );
  }
}

void WPUContext::setWinText( const char *str )
{
  AGUIX *aguix = worker->getAGUIX();
  Text **ntexts;
  int i, lines;
  char **liness;
  int ty, tw;
  
  if ( userwin == NULL ) return;

  if ( str == NULL ) {
    ntexts = new Text*[1];
    ntexts[0] = texts[0];
    for ( i = 1; i < nr_of_text; i++ ) {
      userwin->remove( texts[i] );
      delete texts[i];
    }
    delete [] texts;
    texts = ntexts;
    nr_of_text = 1;
    texts[0]->setText( "" );
    ty = texts[0]->getY() + texts[0]->getHeight() + 5;
  } else {
    lines = createLines( str, &liness );
    ntexts = new Text*[lines];
    if ( lines >= nr_of_text ) {
      ty = 5;
      for ( i = 0; i < nr_of_text; i++ ) {
        ntexts[i] = texts[i];
        ntexts[i]->setText( liness[i] );
        ty += ntexts[i]->getHeight() + 5;
      }
      for ( i = nr_of_text; i < lines; i++ ) {
        ntexts[i] = (Text*)userwin->add( new Text( aguix, 5, ty, liness[i], 1 ) );
        ty += ntexts[i]->getHeight() + 5;
      }
    } else {
      ty = 5;
      for ( i = 0; i < lines; i++ ) {
        ntexts[i] = texts[i];
        ntexts[i]->setText( liness[i] );
        ty += ntexts[i]->getHeight() + 5;
      }
      for ( i = lines; i < nr_of_text; i++ ) {
        userwin->remove( texts[i] );
        delete texts[i];
      }
    }
    delete [] texts;
    texts = ntexts;
    nr_of_text = lines;
    for ( i = 0; i < lines; i++ )
      _freesafe( liness[i] );
    _freesafe( liness );
  }
  bb1->move( bb1->getX(), ty );
  prosb->move( prosb->getX(), bb1->getY() + 1 );
  ty += bb1->getHeight() + 5;
  bb2->move( bb2->getX(), ty );
  ty += bb2->getHeight() + 5;
  cancelb->move( cancelb->getX(), ty );
  ty += cancelb->getHeight() + 5;
  userwin->maximizeX();
  userwin->resize( userwin->getWidth(), ty );
  
  tw = userwin->getWidth();
  bb1->resize( tw - 10, bb1->getHeight() );
  i = ( tw - 10 - 2 ) * progress / 100;
  if ( i < 2 ) i = 2;
  prosb->resize( i, prosb->getHeight() );
  bb2->resize( tw - 10, bb2->getHeight() );

  GUIElement *tbutton[1];
  tbutton[0] = cancelb;
  AGUIX::scaleElementsW( userwin->getWidth(), 5, 0, -1, false, false, tbutton, NULL, 1 );
  userwin->setMinSize( userwin->getWidth(), userwin->getHeight() );
  userwin->setMaxSize( userwin->getWidth(), userwin->getHeight() );
}

const char *WPUContext::getBaseDir()
{
  const char *tstr = NULL;
  NormalMode *nm;
  
  if ( worker != NULL ) {
    if ( activemode != NULL ) {
      if ( activemode->isType( "NormalMode" ) == true ) {
        nm = (NormalMode*)activemode;
        tstr = nm->getCurrentDir();
      }
    }
  }
  return tstr;
}

int WPUContext::filelistSize( bool other )
{
  buildExtList();
  return extlist[ ( other == true ) ? 1 : 0 ]->getSize();
}

int WPUContext::filelistRealSize( bool other )
{
  buildExtList();
  return extlist[ ( other == true ) ? 1 : 0 ]->getRealSize();
}

void WPUContext::setTakeDirs( bool nv )
{
  extorder.take_dirs = nv;
  for ( int i = 0; i < 2; i++ ) {
    if ( extlist[i] != NULL ) {
      extlist[i]->setTakeDirs( nv );
    }
  }
}

bool WPUContext::getTakeDirs()
{
  return extorder.take_dirs;
}

const char *WPUContext::getTempName4File( const char *filename )
{
  std::list<temp4filename_t*>::iterator it1;
  worker_struct_stat stbuf;
  temp4filename_t *el1;
  char *newtempname;

  if ( filename == NULL ) return NULL;

  for ( it1 = tempnamelist.begin(); it1 != tempnamelist.end(); it1++ ) {
    if ( *it1 != NULL ) {
      if ( (*it1)->filename != NULL ) {
	if ( strcmp( (*it1)->filename, filename ) == 0 ) {
	  break;
	}
      }
    }
  }

  if ( it1 != tempnamelist.end() ) {
    return (*it1)->tempname;
  }

  if ( worker_lstat( filename, &stbuf ) != 0 ) return NULL;

  if ( ! S_ISREG( stbuf.st_mode ) ) return NULL;

  newtempname = Datei::createTMPName();

  if ( newtempname == NULL ) return NULL;
  
  if ( Datei::copyfile( newtempname, filename ) != 0 ) {
    if ( worker->getRequester() != NULL ) {
      char *reqstr;
      reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 618 ) ) + strlen( filename ) + 1 );
      sprintf( reqstr, catalog.getLocale( 618 ), filename );
      worker->getRequester()->request( catalog.getLocale( 347 ), reqstr, catalog.getLocale( 11 ) );
      _freesafe( reqstr );
    }
    
    worker_unlink( newtempname );
    _freesafe( newtempname );
    return NULL;
  }
  
  el1 = new temp4filename_t();
  el1->tempname = newtempname;
  el1->filename = dupstring( filename );
  if ( worker_lstat( newtempname, &stbuf ) != 0 ) el1->modtime = 0;
  else el1->modtime = stbuf.st_mtime;
  
  tempnamelist.push_back( el1 );
  return newtempname;
}

void WPUContext::freeTempNameList()
{
  std::list<temp4filename_t*>::iterator it1;
  worker_struct_stat stbuf;
  bool changed;

  for ( it1 = tempnamelist.begin(); it1 != tempnamelist.end(); it1++ ) {
    if ( *it1 != NULL ) {
      changed = false;

      if ( (*it1)->tempname != NULL ) {
	if ( worker_lstat( (*it1)->tempname, &stbuf ) != 0 ) changed = true;
	else if ( (*it1)->modtime != stbuf.st_mtime ) changed = true;
	
	if ( ( changed == true ) &&
	     ( (*it1)->filename != NULL ) &&
	     ( worker->getRequester() != NULL ) ) {
	  char *reqstr;
	  reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 628 ) ) + strlen( (*it1)->filename ) + 1 );
	  sprintf( reqstr, catalog.getLocale( 628 ), (*it1)->filename );
	  worker->getRequester()->request( catalog.getLocale( 125 ), reqstr, catalog.getLocale( 11 ) );
	  _freesafe( reqstr );
	}
	
        worker_unlink( (*it1)->tempname );
      }
      delete *it1;
    }
  }
}
