/*  Wordtrans, front-end for several dictionaries, for the Qt toolkit.
    Copyright (C) 2000-2004 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "QBabyDiccionario.h"
#include "intern.h"
#include <QConfig.h>
#include <qglobal.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextcodec.h>
#include "salidas.h"

#define USAR_TABLA

class DictStyle 
{
public:
	enum Formato { plain, normal, table };
	enum Alineacion { left, right };

	DictStyle( Formato f, Alineacion a );
	~DictStyle() { } ;

	QString open_def();
	QString append_word(const QString& word);
	QString open_par();
	QString append_line(const QString& line);
	QString close_par();
	QString close_def();

private:
	Alineacion align;
	Formato format;
	QString alineacion;
	QString font_family;
};

DictStyle::DictStyle( Formato f, Alineacion a ) {
	align = a;
	format = f;
	if (align == right) 
		alineacion="right";
	else
		alineacion="left";
}

QString DictStyle::open_def() {
	switch (format) {
		case table: 
			return "<table width=\"100%\" border=0>\n";
		case normal:
#if QT_VERSION < 300
			return QString("<p align=%1>\n").arg(alineacion);
#else
			return "<p>\n";
#endif
		default: 
			return "";
	}
}

QString DictStyle::append_word(const QString& word) {
	switch (format) {
		case table:
			//return "<tr><td bgcolor=\"#FFAA7F\">" + word + "</tr></td>\n";
#if QT_VERSION < 300
			return QString("<tr><td align=%1 bgcolor=\"%2\">%3</td></tr>\n")
        	       .arg(alineacion).arg(DiccionarioBase::color_header).arg(word);
#else
			return QString("<tr><td align=%1 bgcolor=\"%2\"><font face=%3>%4</font></td></tr>\n")
        	       .arg(alineacion).arg(DiccionarioBase::color_header).arg(DiccionarioBase::font_string).arg(word);
#endif
		case normal:
			return word + "<br>\n";
		default:
			return word + "\n";
	}
}

QString DictStyle::open_par() {
	switch (format) {
		case table:
			//return "<tr><td bgcolor=\"#00AAFF\">\n";
#if QT_VERSION < 300
			return QString("<tr><td align=%1>\n").arg(alineacion);
#else
			return QString("<tr><td align=%1><font face=%2>\n")
                          .arg(alineacion)
                          .arg(DiccionarioBase::font_string);
#endif
		case normal:
			return "";
		default:
			return "";
	}
}

QString DictStyle::append_line(const QString& line) {
	switch (format) {
		case table:
			return line;
		case normal:
			return line + "<br>\n";
		default:
			return line + "\n";
	}
}

QString DictStyle::close_par() {
	switch (format) {
		case table:
#if QT_VERSION < 300
			return "</td></tr>\n";
#else
			return "</font></td></tr>\n";
#endif
		case normal:
			return "";
		default:
			return " \n";
	}
}

QString DictStyle::close_def() {
	switch (format) {
		case table:
			return "</table>\n";
		case normal:
#if QT_VERSION < 300
			return "</p>\n";
#else
			return "</p><br>\n";
#endif
		default:
			return "";
	}
}



QBabyDiccionario::QBabyDiccionario() : DiccionarioBase() {
	setTipo( DiccionarioBase::Babylon_dict );  
	diccionario="/usr/share/babytrans/Engtospa.dic";
#ifndef NO_LIBBAB
	app_translator = NULL;
	//usar_tabla = true;
#endif
}

QBabyDiccionario::~QBabyDiccionario() {
}


bool QBabyDiccionario::update() {
#ifndef NO_LIBBAB
	QFileInfo fi( diccionario );
	if (!fi.exists()) {
		setError( _("File %1 doesn't exist").arg(diccionario) );
		sprintln( error );
		if (app_translator != NULL) delete app_translator;
		app_translator = NULL;
		return false;
	}

	QString nombre( fi.fileName() );
	QString path( fi.dirPath(true) );

	qDebug("Path:%s", path.latin1() );
	qDebug("Nombre:%s", nombre.latin1() );

	if (app_translator != NULL) delete app_translator;

	app_translator = babylon::create_babylon( nombre.latin1() );

	dict_charset = app_translator->get_charset().c_str();
#if QT_VERSION >= 300
	if (dict_charset.contains("8859-8")) dict_charset="ISO 8859-8-I";
#endif
	qDebug("Charset del nuevo diccionario: %s", dict_charset.data() );

	app_translator->set_path( path.latin1() ); //"/home/ricardo/.babytrans/" );

	if (! app_translator->open( nombre.latin1() ) ) { //"Engtospa.dic") ) {
		setError( _("Error from libbab: %1").arg( app_translator->get_error().c_str() ) );
		sprintln( error );
		return false;
	}
#endif

	return true;
}

QString QBabyDiccionario::getFullName() {
	return DiccionarioBase::getFullName();
	/*
    QString texto = getIdioma1();
    texto += " -> ";
    texto += getIdioma2();
	texto += " (babylon)";
    return texto;
	*/
}


bool QBabyDiccionario::busca(const QString& palabra, QString & resultado, 
                             bool con_formato) {

#define APPEND_LINE( texto ) \
	resultado += texto; \
	resultado += "\n";

	resultado="";
	n_matches=0;
	bestMatch="";

#ifndef NO_LIBBAB

	if (!app_translator) {
		resultado = _("Error: app_translator not initialized");
		return true;
	}

	QTextCodec *codec=NULL;
	codec= QTextCodec::codecForName( dict_charset.data() );
	if (codec==NULL) qWarning("codec == NULL");

	DictStyle::Alineacion align;
	if (app_translator->is_right_to_left() ) 
		align = DictStyle::right;
	else
		align = DictStyle::left;

	DictStyle::Formato formato;
	if (!con_formato) {
		formato = DictStyle::plain;
	} else {
		if (DiccionarioBase::usar_tabla)
			formato = DictStyle::table;
		else
			formato = DictStyle::normal;
	}

	DictStyle style(formato, align);

	babylon::container_type translations;

	bool ok = app_translator->translate( palabra.latin1(), translations );

	if (!ok) {
		//setError( QString( _("Error from libbab: %1") ).arg( app_translator.get_error().c_str() ) );
		resultado= _("Error from libbab: %1\n").arg( app_translator->get_error().c_str() );
		//sprintln( error );
		//return false;
		return true;
	}


	if ( !translations.empty() ) {
		babylon::container_type::iterator it = translations.begin();

		for ( ; ; ) {
			resultado += style.open_def();

			QString human_attrib ;
			if ( it->attrib.size() >= 8 ) {
		        if ( it->attrib[ 2 ] > '1' || it->attrib[ 3 ] != '0' ) {
		            /* noun */
        		    human_attrib = _( "n." ) ;
		        } else if ( it->attrib[ 6 ] == '1' ) {
        		    /* adverb */
		            human_attrib = _( "a." ) ;
		        } else if ( it->attrib[ 6 ] != '0' || it->attrib[ 9 ] != '0' ) {
		            /* verb */
		            human_attrib = _( "v." ) ;
		        } else if ( it->attrib[ 7 ] != '0' ) {
		            /* adverb */
		            human_attrib = _( "a." ) ;
		        } else if ( it->attrib[ 8 ] == '1' ) {
		            /* adjective */
		            human_attrib = _( "adv." ) ;
		        } else
		            human_attrib = ""; //it->attrib ;
			}

			QString texto;
			if (con_formato) texto += "<b>";
			texto += it->word.c_str();
			if (con_formato) texto += "</b>";
			if (!human_attrib.isEmpty()) {
				texto += " (";
				if (con_formato) texto += "<i><font color=blue>";
				texto += human_attrib;
				if (con_formato) texto += "</font></i>";
				texto += ")";
			}

			// This word is always in english, so I think this is not needed:
			//if (codec!=NULL)
			//	texto= codec->toUnicode( texto.latin1() );
			
			resultado += style.append_word(texto);
			//APPEND_LINE( texto );

			n_matches++;

			QString def_text = it->definition.c_str();
			//qDebug("Resultado: \"%s\"", def_text.latin1() );

			if (codec!=NULL) {
				//if (app_translator->is_right_to_left()) 
				//	def_text = 0xff + def_text + 0xfe;

				qDebug("Orig: '%s'", def_text.latin1() );
				def_text = codec->toUnicode( def_text.latin1() );
				//qDebug("Conv: '%s'", def_text.utf8().data() );
			}

			//Sustituye los "," por ", "
			for (unsigned int n=0; n < def_text.length(); n++) {
				if ( def_text[n] == ',' ) {
					if ( (n!=def_text.length()) && (def_text[n+1]!=' ') )
						def_text.insert( n+1, ' ' );
				}
			}

			if (! app_translator->is_right_to_left() ) {
				//Corta las definiciones en lneas
				QStringList lista= QStringList::split( QChar(';'), def_text, TRUE );
				for ( unsigned int n=0; n < lista.count(); n++ ) {
					QString linea =  lista[n];
					if ((n==0) && (bestMatch.isEmpty())) bestMatch= linea;
					if (n==0) resultado += style.open_par();
					if ( (n+1) < lista.count()) linea += "; ";
					resultado += style.append_line(linea);
					//APPEND_LINE( linea );
				}
				//Una lnea en blanco al final
				resultado += style.close_par();
			} else {
				//HEBREO
				//En hebreo no cortamos las definiciones
#ifdef USE_FRIBIDI
#if QT_VERSION < 300
				qDebug("Right-to-left language + Qt 2.x + Fribidi, so reversing parenthesis");
				//Invertimos los parntesis
				for (unsigned int n=0; n < def_text.length(); n++) {
					if ( def_text[n] == QChar('(') ) def_text[n]= QChar(')');
					else
					if ( def_text[n] == QChar(')') ) def_text[n]= QChar('(');
				}
#endif //QT_VERSION
#endif //USE_FRIBIDI
				//Sustituye los ";" por "; "
				for (unsigned int n=0; n < def_text.length(); n++) {
					if ( def_text[n] == ';' ) {
						if ( (n!=def_text.length()) && (def_text[n+1]!=' ') )
							def_text.insert( n+1, ' ' );
					}
				}
				if (bestMatch.isEmpty()) bestMatch= def_text;
				resultado += style.open_par();
				resultado += style.append_line(def_text);
				//APPEND_LINE( def_text );
				resultado += style.close_par();
			}

			resultado += style.close_def();

			++it;
			if ( it == translations.end() ) break;
		}
	}

	//if (resultado.count() >= 1)
	//	bestMatch=resultado[1];

	return true;
#else

	resultado= _("Wordtrans compiled without libbab support.");
	return true;

#endif
}

bool QBabyDiccionario::aprende(const QString& texto_idioma1, const QString& texto_idioma2) {
	setError( _("Cannot add new translations to Babylon Translator's dictionaries") );
	sprintln( error );
	return false;
}

void QBabyDiccionario::save() {
	conf->setGroup("General");

	conf->writeEntry("alias", alias.latin1() );
	conf->writeEntry("idioma1", idioma1.latin1() );
	conf->writeEntry("idioma2", idioma2.latin1() );
	conf->writeBoolEntry("deleted",getDeleted() );

	conf->setGroup("Diccionarios");
	conf->writeEntry("general", diccionario.latin1() );
	//conf->writeEntry("personal", personal.data() );

	//conf->setGroup("Opciones");
	//conf->writeBoolEntry("busqueda_inversa", busqueda_inversa );
	//conf->writeBoolEntry("case_sensitive", case_sensitive );
	//conf->writeBoolEntry("palabras_completas", palabras_completas );

	conf->setGroup("Iconos");
	conf->writeEntry("idioma1", getIconoIdioma1().latin1() );
	conf->writeEntry("idioma2", getIconoIdioma2().latin1() );

	//conf->setGroup("Style");
	//conf->writeBoolEntry("table", usar_tabla );

	save_desc();
	save_font();

	conf->sync();
}

void QBabyDiccionario::load(const QString& fichero) {
	//debug("QBabyDiccionario::load: leyendo %s", fichero.data() );

	conf->setFileName(fichero.latin1());

	conf->setGroup("General");

	alias = conf->readEntry("alias", "sin_alias");
	idioma1 = conf->readEntry("idioma1", "language1");
	idioma2 = conf->readEntry("idioma2", "language2");
	setDeleted( conf->readBoolEntry("deleted",false) );

	conf->setGroup("Diccionarios");
	diccionario = conf->readEntry("general", "/usr/share/babytrans/");
	//setPersonal( conf->readEntry("personal", ".i2e.dict") );

	//conf->setGroup("Opciones");
	//busqueda_inversa= conf->readBoolEntry("busqueda_inversa", false );
	//case_sensitive = conf->readBoolEntry("case_sensitive", false );
	//palabras_completas= conf->readBoolEntry("palabras_completas", true );

	conf->setGroup("Iconos");
	setIconoIdioma1( conf->readEntry("idioma1","/usr/share/wordtrans/i2e.xpm") );
	setIconoIdioma2( conf->readEntry("idioma2","/usr/share/wordtrans/e2i.xpm") );

	//conf->setGroup("Style");
	//usar_tabla = conf->readBoolEntry("table", true );

	load_desc();
	load_font();

	update();
}

