# virtual-server-lib.pl
# Common functions for Virtualmin
# XXX home-less operation
#	XXX when is uid chosen? It should be by setup_unix
# XXX links from edit_domain.cgi to Apache/BIND/etc modules
# XXX handle setup/delete/modify failure in save_domain.cgi

do '../web-lib.pl';
&init_config();
require '../ui-lib.pl';
%access = &get_module_acl();
if (!defined($access{'feature_unix'})) {
	$access{'feature_unix'} = 1;
	}
if (!defined($access{'feature_dir'})) {
	$access{'feature_dir'} = 1;
	}
$single_domain_mode = $access{'domains'} =~ /^\d+$/ &&
		      !$access{'edit'} && !$access{'create'} &&
		      !$access{'stop'} && !$access{'local'};
if (!&master_admin()) {
	# Allowed alias types are set by module config
	%can_alias_types = map { $_, 1 } split(/,/, $config{'alias_types'});
	}
else {
	# All types are allowed
	%can_alias_types = map { $_, 1 } (0 .. 10);
	}

$first_print = \&first_html_print;
$second_print = \&second_html_print;
$indent_print = \&indent_html_print;
$outdent_print = \&outdent_html_print;

@plugins = split(/\s+/, $config{'plugins'});
@opt_features = ( 'dir', 'unix', 'dns', 'mail', 'web', 'webalizer', 'ssl', 'logrotate', 'mysql', 'postgres', 'ftp', 'webmin' );
@vital_features = ( 'dir', 'unix' );
@features = ( @opt_features );
@backup_features = ( 'virtualmin', @features );
@opt_alias_features = ( 'dir', 'dns', 'mail', 'web' );
@alias_features = ( @opt_alias_features );
foreach $fname (@features) {
	do "$module_root_directory/feature-$fname.pl";
	}
foreach $pname (@plugins) {
	&foreign_require($pname, "virtual_feature.pl");
	}
@feature_plugins = grep { &plugin_defined($_, "feature_setup") } @plugins;
@mail_plugins = grep { &plugin_defined($_, "mailbox_inputs") } @plugins;

$backup_cron_cmd = "$module_config_directory/backup.pl";
$bw_cron_cmd = "$module_config_directory/bw.pl";

$custom_fields_file = "$module_config_directory/custom-fields";

sub require_useradmin
{
return if ($require_useradmin++);
&foreign_require("useradmin", "user-lib.pl");
%uconfig = &foreign_config("useradmin");
&foreign_require("quota", "quota-lib.pl");
$home_base = $config{'home_base'} || $uconfig{'home_base'};
if ($config{'ldap'}) {
	&foreign_require("ldap-useradmin", "ldap-useradmin-lib.pl");
	$usermodule = "ldap-useradmin";
	}
else {
	$usermodule = "useradmin";
	}
}

$domains_dir = "$module_config_directory/domains";

# list_domains()
# Returns a list of structures containing information about hosted domains
sub list_domains
{
local (@rv, $d);
opendir(DIR, $domains_dir);
foreach $d (readdir(DIR)) {
	if ($d !~ /^\./ && $d !~ /\.(lock|bak|rpmsave)$/i) {
		push(@rv, &get_domain($d));
		}
	}
closedir(DIR);
return @rv;
}

# get_domain(id, [file])
# Looks up a domain object by ID
sub get_domain
{
if ($_[0] && defined($main::get_domain_cache{$_[0]})) {
	return $main::get_domain_cache{$_[0]};
	}
local %dom;
local $file = $_[1] || "$domains_dir/$_[0]";
&read_file($file, \%dom) || return undef;
$dom{'file'} = "$domains_dir/$_[0]";
$dom{'id'} ||= $_[0];
$dom{'mail'} = 1 if (!defined($dom{'mail'}));	# compat - assume mail is on
$dom{'ugid'} = $dom{'gid'} if (!defined($dom{'ugid'}));	# compat - assume same
if ($dom{'disabled'} eq '1') {
	# compat - assume everything was disabled
	$dom{'disabled'} = "unix,web,dns,mail,mysql,postgres";
	}
elsif ($dom{'disabled'}) {
	# compat - user disabled has changed to unix
	$dom{'disabled'} =~ s/user/unix/g;
	}
if ($dom{'disabled'}) {
	# Manually disabled
	$dom{'disabled_reason'} ||= 'manual';
	}
if (!defined($dom{'created'})) {
	# compat - creation date can be inferred from ID
        $dom{'id'} =~ /^(\d{10})/;
        $dom{'created'} = $1;
        }
if (!defined($dom{'gid'})) {
	# compat - get GID from group name
	$dom{'gid'} = getgrnam($dom{'group'});
	}
if (!defined($dom{'unix'})) {
	# compat - unix is always on
	$dom{'unix'} = 1;
	}
if (!defined($dom{'dir'})) {
	# if unix is on, so is home
	$dom{'dir'} = $dom{'unix'};
	if ($dom{'parent'}) {
		# if server has a parent, it never has a Unix user
		$dom{'unix'} = 0;
		}
	}
if (!defined($dom{'limit_unix'})) {
	# compat - unix is always available for subdomains
	$dom{'limit_unix'} = 1;
	}
if (!defined($dom{'limit_dir'})) {
	# compat - home is always available for subdomains
	$dom{'limit_dir'} = 1;
	}
if (!defined($dom{'virt'})) {
	# compat - assume virtual IP if interface assigned
	$dom{'virt'} = $dom{'iface'} ? 1 : 0;
	}
if (!defined($dom{'web_port'}) && $dom{'web'}) {
	# compat - assume web port is current setting
	$dom{'web_port'} = $default_web_port;
	}
if (!defined($dom{'web_sslport'}) && $dom{'ssl'}) {
	# compat - assume SSL port is current setting
	$dom{'web_sslport'} = $web_sslport;
	}
if (!defined($dom{'prefix'})) {
	# compat - assume that prefix is same as group
	$dom{'prefix'} = $dom{'group'};
	}
if (!defined($dom{'home'})) {
	local @u = getpwnam($dom{'user'});
	$dom{'home'} = $u[7];
	}
if (!defined($dom{'proxy_pass_mode'}) && $dom{'proxy_pass'}) {
	# assume that proxy pass mode is proxy-based if not set
	$dom{'proxy_pass_mode'} = 1;
	}
if (!defined($dom{'template'})) {
	# assume default parent or sub-server template
	$dom{'template'} = $dom{'parent'} ? 1 : 0;
	}
if (!defined($dom{'db_mysql'}) && $dom{'mysql'}) {
	# Assume just one MySQL DB
	$dom{'db_mysql'} = $dom{'db'};
	}
if (!defined($dom{'db_postgres'}) && $dom{'postgres'}) {
	# Assume just one PostgreSQL DB
	$dom{'db_postgres'} = $dom{'db'};
	}
delete($dom{'missing'});	# never set in a saved domain
if ($_[0]) {
	$main::get_domain_cache{$_[0]} = \%dom;
	}
return \%dom;
}

# get_domain_by(field, value, [field, value, ...])
# Looks up a domain by some field(s)
sub get_domain_by
{
local $d;
local @rv;
foreach $d (&list_domains()) {
	local $i;
	local $allok = 1;
	for($i=0; $i<@_; $i+=2) {
		$allok = 0 if ($d->{$_[$i]} ne $_[$i+1]);
		}
	push(@rv, $d) if ($allok);
	}
return wantarray ? @rv : $rv[0];
}

# domain_id()
# Returns a new unique domain ID
sub domain_id
{
return time().$$;
}

# save_domain(&domain)
sub save_domain
{
mkdir($domains_dir, 0700);
&lock_file("$domains_dir/$_[0]->{'id'}");
$_[0]->{'id'} = &domain_id() if (!$_[0]->{'id'});
$_[0]->{'created'} = time() if (!$_[0]->{'created'});
&write_file("$domains_dir/$_[0]->{'id'}", $_[0]);
&unlock_file("$domains_dir/$_[0]->{'id'}");
$main::get_domain_cache{$_[0]->{'id'}} = $_[0];
return 1;
}

# delete_domain(&domain)
sub delete_domain
{
&lock_file("$domains_dir/$_[0]->{'id'}");
unlink("$domains_dir/$_[0]->{'id'}");
&unlock_file("$domains_dir/$_[0]->{'id'}");

# And the bandwidth file
unlink("$bandwidth_dir/$_[0]->{'id'}");
}

# list_domain_users([&domain], [skipunix], [no-virts], [no-quotas])
# List all Unix users who are in the domain's primary group.
# If domain is omitted, returns local users.
sub list_domain_users
{
# Get all aliases (and maybe generics) to look for those that match users
&require_mail();
local (%aliases, %generics);
if ($config{'mail'} && !$_[2]) {
	&require_mail();
	if ($config{'mail_system'} == 1) {
		# Find Postfix aliases for users
		%aliases = map { $_->{'name'}, $_ } grep { $_->{'enabled'} }
			       &sendmail::list_aliases($sendmail_afiles);
		if ($config{'generics'}) {
			%generics = map { $_->{'from'}, $_ }
				      &sendmail::list_generics($sendmail_gfile);
			}
		}
	elsif ($config{'mail_system'} == 0) {
		# Find Sendmail aliases for users
		%aliases = map { $_->{'name'}, $_ }
			       &postfix::list_aliases($postfix_afiles);
		if ($config{'generics'}) {
			local $cans = &postfix::get_maps($canonical_type);
			%generics = map { $_->{'name'}, $_ } @$cans;
			}
		}
	elsif ($config{'mail_system'} == 5) {
		# Find VPOPMail aliases to match with users
		%valiases = map { $_->{'from'}, $_ } &list_virtusers();
		}
	}

# Get all virtusers to look for those for users
local @virts;
if (!$_[2]) {
	@virts = &list_virtusers();
	}

local @users = &list_all_users_quotas();
if ($_[0]) {
	# Limit to domain users.
	@users = grep { $_->{'gid'} == $_[0]->{'gid'} ||
			$_->{'user'} eq $_[0]->{'user'} } @users;
	if ($_[0]->{'parent'}) {
		# This is a subdomain - exclude parent domain users
		@users = grep { $_->{'home'} =~ /^$_[0]->{'home'}\// } @users;
		}
	elsif (&get_domain_by("parent", $_[0]->{'id'})) {
		# This domain has subdomains - exclude their users
		@users = grep { $_->{'home'} !~ /^$_[0]->{'home'}\/domains\// } @users;
		}
	@users = grep { $_->{'user'} ne $_[0]->{'user'} } @users
		if ($_[1] || $_[0]->{'parent'});

	if ($config{'mail_system'} == 4) {
		# Add Qmail LDAP users (who have same GID?)
		local $ldap = &connect_qmail_ldap();
		local $rv = $ldap->search(base => $config{'ldap_base'},
				  filter => "(&(objectClass=qmailUser)(|(qmailGID=$_[0]->{'gid'})(gidNumber=$_[0]->{'gid'})))");
		&error($rv->error) if ($rv->code);
		local $u;
		foreach $u ($rv->all_entries) {
			local %uinfo = &qmail_dn_to_hash($u);
			next if (!$uinfo{'mailstore'});	# alias only
			$uinfo{'ldap'} = $u;
			@users = grep { $_->{'user'} ne $uinfo{'user'} } @users;
			push(@users, \%uinfo);
			}
		$ldap->unbind();
		}
	elsif ($config{'mail_system'} == 5) {
		# Add VPOPMail users for this domain
		local %attr_map = ( 'name' => 'user',
				    'passwd' => 'pass',
				    'clear passwd' => 'plainpass',
				    'comment/gecos' => 'real',
				    'dir' => 'home',
				    'quota' => 'qquota',
				   );
		local $user;
		open(UINFO, "$vpopbin/vuserinfo -D $_[0]->{'dom'} |");
		while(<UINFO>) {
			s/\r|\n//g;
			if (/^([^:]+):\s+(.*)$/) {
				local ($attr, $value) = ($1, $2);
				if ($attr eq "name") {
					# Start of a new user
					$user = { 'vpopmail' => 1,
						  'mailquota' => 1,
						  'person' => 1,
						  'fixedhome' => 1,
						  'noappend' => 1,
						  'noprimary' => 1 };
					push(@users, $user);
					}
				local $amapped = $attr_map{$attr};
				$user->{$amapped} = $value if ($amapped);
				if ($amapped eq "qquota") {
					if ($value eq "NOQUOTA") {
						$user->{$amapped} = 0;
						}
					else {
						$user->{$amapped} = int($value);
						}
					}
				}
			}
		close(UINFO);
		}
	}
else {
	# Limit to local users
	local @lg = getgrnam($config{'localgroup'});
	@users = grep { $_->{'gid'} == $lg[2] } @users;
	}
if (!$_[2]) {
	# Add email addresses and forwarding addresses to user structures
	local $u;
	foreach $u (@users) {
		next if ($u->{'qmail'});	# got from LDAP already
		$u->{'email'} = $u->{'virt'} = undef;
		$u->{'alias'} = $u->{'to'} = $u->{'generic'} = undef;
		$u->{'extraemail'} = $u->{'extravirt'} = undef;
		local ($al, $va);
		if ($al = $aliases{&escape_alias($u->{'user'})}) {
			$u->{'alias'} = $al;
			$u->{'to'} = $al->{'values'};
			}
		elsif ($va = $valiases{"$u->{'user'}\@$d->{'dom'}"}) {
			$u->{'valias'} = $va;
			$u->{'to'} = $va->{'to'};
			}
		elsif ($config{'mail_system'} == 2 ||
		       $config{'mail_system'} == 5) {
			# Find .qmail file
			local $alias = &get_dotqmail(&dotqmail_file($u));
			if ($alias) {
				$u->{'alias'} = $alias;
				$u->{'to'} = $u->{'alias'}->{'values'};
				}
			}
		$u->{'generic'} = $generics{$u->{'user'}};
		local $pop3 = $_[0] ? &remove_userdom($u->{'user'}, $_[0])
				    : $u->{'user'};
		local $email = $_[0] ? "$pop3\@$_[0]->{'dom'}" : undef;
		local $escuser = &escape_user($u->{'user'});
		local $escalias = &escape_alias($u->{'user'});
		local $v;
		foreach $v (@virts) {
			if (@{$v->{'to'}} == 1 &&
			    ($v->{'to'}->[0] eq $escuser ||
			     $v->{'to'}->[0] eq $escalias ||
			     $v->{'from'} eq $email &&
			      $v->{'to'}->[0] eq "BOUNCE") &&
			    (!$_[0] || $v->{'from'} ne $_[0]->{'dom'})) {
				if ($v->{'from'} eq $email) {
					if ($v->{'to'}->[0] ne "BOUNCE") {
						$u->{'email'} = $email;
						}
					$u->{'virt'} = $v;
					}
				else {
					push(@{$u->{'extraemail'}},
					     $v->{'from'});
					push(@{$u->{'extravirt'}}, $v);
					}
				}
			}
		}
	}
return @users;
}

# list_all_users_quotas()
# Returns a list of all Unix users, with quota info
sub list_all_users_quotas
{
# Get quotas for all users
&require_useradmin();
local $qv = $config{'hard_quotas'} ? 'hblocks' : 'sblocks';
if (!defined(%home_quotas) && $config{'home_quotas'}) {
	local $n = &quota::filesystem_users($config{'home_quotas'});
	local $i;
	for($i=0; $i<$n; $i++) {
		$home_quota{$quota::user{$i,'user'}} = $quota::user{$i,$qv};
		$used_home_quota{$quota::user{$i,'user'}} =
			$quota::user{$i,'ublocks'};
		}
	}
if (!defined(%mail_quotas) && $config{'mail_quotas'} &&
    $config{'mail_quotas'} ne $config{'home_quotas'}) {
	local $n = &quota::filesystem_users($config{'mail_quotas'});
	local $i;
	for($i=0; $i<$n; $i++) {
		$mail_quota{$quota::user{$i,'user'}} = $quota::user{$i,$qv};
		$used_mail_quota{$quota::user{$i,'user'}} =
			$quota::user{$i,'ublocks'};
		}
	}

# Get user list and add in quota info
local @users = &foreign_call($usermodule, "list_users");
local $u;
foreach $u (@users) {
	$u->{'quota'} = $home_quota{$u->{'user'}};
	$u->{'uquota'} = $used_home_quota{$u->{'user'}};
	$u->{'mquota'} = $mail_quota{$u->{'user'}};
	$u->{'umquota'} = $used_mail_quota{$u->{'user'}};
	$u->{'unix'} = 1;
	$u->{'person'} = 1;
	}
return @users;
}

# create_user(&user, [&domain])
# Create a mailbox or local user, his virtuser and possibly his alias
sub create_user
{
local $pop3 = &remove_userdom($_[0]->{'user'}, $_[1]);
&require_useradmin();
&require_mail();

if ($_[0]->{'qmail'}) {
	# Create user in Qmail LDAP
	local $ldap = &connect_qmail_ldap();
	local $_[0]->{'dn'} = "uid=$_[0]->{'user'},$config{'ldap_base'}";
	local @oc = ( "qmailUser" );
	push(@oc, "posixAccount") if ($_[0]->{'unix'});
	push(@oc, split(/\s+/, $config{'ldap_classes'}));
	local $attrs = &qmail_user_to_dn($_[0], \@oc, $_[1]);
	push(@$attrs, "objectClass" => \@oc);
	local $rv = $ldap->add($_[0]->{'dn'}, attr => $attrs);
	&error($rv->error) if ($rv->code);
	$ldap->unbind();
	}
elsif ($_[0]->{'vpopmail'}) {
	# Create user in VPOPMail
	local $quser = quotemeta($_[0]->{'user'});
	local $qdom = quotemeta($_[1]->{'dom'});
	local $qreal = quotemeta($_[0]->{'real'}) || '""';
	local $quota = $_[0]->{'qquota'} ? "-q $_[0]->{'qquota'}" : "-q NOQUOTA";
	local $qpass = quotemeta($_[0]->{'plainpass'});
	local $cmd = "$vpopbin/vadduser $quota -c $qreal $quser\@$qdom $qpass";
	local $out = &backquote_logged("$cmd 2>&1");
	&error("<tt>$cmd</tt> failed: <pre>$out</pre>") if ($?);
	$_[0]->{'home'} = "$config{'vpopmail_dir'}/domains/$_[1]->{'dom'}/$_[0]->{'user'}";
	}
else {
	# Add the Unix user
	&foreign_call($usermodule, "set_user_envs", $_[0], 'CREATE_USER', $_[0]->{'plainpass'}, [ ]);
	&foreign_call($usermodule, "making_changes");
	&foreign_call($usermodule, "lock_user_files");
	&userdom_substitutions($_[0], $_[1]);
	&foreign_call($usermodule, "create_user", $_[0]);
	&foreign_call($usermodule, "unlock_user_files");
	&foreign_call($usermodule, "made_changes");
	}

if (!$_[0]->{'qmail'}) {
	# Add his virtusers for non Qmail+LDAP users
	local $firstemail;
	local $vto = $_[0]->{'to'} ? &escape_alias($_[0]->{'user'})
				   : &escape_user($_[0]->{'user'});
	if ($_[0]->{'email'}) {
		local $virt = { 'from' => $_[0]->{'email'},
				'to' => [ $vto ] };
		&create_virtuser($virt);
		$_[0]->{'virt'} = $virt;
		$firstemail ||= $_[0]->{'email'};
		}
	elsif ($can_alias_types{9} && $_[1] && !$_[0]->{'noprimary'}) {
		# Add bouncer if email disabled
		local $virt = { 'from' => "$pop3\@$_[1]->{'dom'}",
				'to' => [ "BOUNCE" ] };
		&create_virtuser($virt);
		$_[0]->{'virt'} = $virt;
		}
	local @extravirt;
	local $e;
	foreach $e (@{$_[0]->{'extraemail'}}) {
		local $virt = { 'from' => $e,
				'to' => [ $vto ] };
		&create_virtuser($virt);
		push(@extravirt, $virt);
		$firstemail ||= $e;
		}
	$_[0]->{'extravirt'} = \@extravirt;
	}

if (!$_[0]->{'qmail'}) {
	# Add his alias, if any, for non Qmail+LDAP users
	local @to = @{$_[0]->{'to'}};
	if (@to) {
		local $alias = { 'name' => &escape_alias($_[0]->{'user'}),
				 'enabled' => 1,
				 'values' => $_[0]->{'to'} };
		&check_alias_clash($_[0]->{'user'}) &&
			&error(&text('alias_eclash2', $_[0]->{'user'}));
		if ($config{'mail_system'} == 1) {
			&sendmail::lock_alias_files($sendmail_afiles);
			&sendmail::create_alias($alias, $sendmail_afiles);
			&sendmail::unlock_alias_files($sendmail_afiles);
			}
		elsif ($config{'mail_system'} == 0) {
			&postfix::lock_alias_files($postfix_afiles);
			&postfix::create_alias($alias, $postfix_afiles);
			&postfix::unlock_alias_files($postfix_afiles);
			&postfix::regenerate_aliases();
			}
		elsif ($config{'mail_system'} == 2 ||
		       $config{'mail_system'} == 5) {
			# Set up user's .qmail file
			local $dqm = &dotqmail_file($_[0]);
			&lock_file($dqm);
			&save_dotqmail($alias, $dqm, $pop3);
			&unlock_file($dqm);
			}
		$_[0]->{'alias'} = $alias;
		}

	if ($config{'generics'} && $firstemail) {
		# Add genericstable entry too
		if ($config{'mail_system'} == 1) {
			local $gen = { 'from' => $_[0]->{'user'}, 'to' => $firstemail };
			&lock_file($sendmail_gfile);
			&sendmail::create_generic($gen, $sendmail_gfile,
						  $sendmail_gdbm, $sendmail_gdbmtype);
			&unlock_file($sendmail_gfile);
			}
		elsif ($config{'mail_system'} == 0) {
			local $gen = { 'name' => $_[0]->{'user'},
				       'value' => $firstemail };
			&lock_file($canonical_map_files[0]);
			&create_replace_mapping($canonical_type, $gen);
			&unlock_file($canonical_map_files[0]);
			&postfix::regenerate_canonical_table();
			}
		}
	}
}

# modify_user(&user, &old, &domain, [noaliases])
sub modify_user
{
local $pop3 = &remove_userdom($_[0]->{'user'}, $_[2]);
if ($_[1]->{'qmail'}) {
	# Update user in Qmail LDAP
	local $ldap = &connect_qmail_ldap();
	local $_[0]->{'dn'} = "uid=$_[0]->{'user'},$config{'ldap_base'}";
	local ($attrs, $delattrs) = &qmail_user_to_dn($_[0],
		[ $_[1]->{'ldap'}->get_value("objectClass") ], $_[2]);
	@$delattrs = grep { defined($_[1]->{'ldap'}->get_value($_))} @$delattrs;
	local (%attrs, $i);
	for($i=0; $i<@$attrs; $i+=2) {
		$attrs{$attrs->[$i]} = $attrs->[$i+1];
		}
	local $rv = $ldap->modify($_[1]->{'dn'},
				  replace => \%attrs,
				  delete => $delattrs);
	&error($rv->error) if ($rv->code);
	if ($_[0]->{'dn'} ne $_[1]->{'dn'}) {
		# Re-named too!
		$rv = $ldap->moddn($_[1]->{'dn'},
				   newrdn => "uid=$_[0]->{'user'}");
		&error($rv->error) if ($rv->code);
		}
	$ldap->unbind();
	}
elsif ($_[1]->{'vpopmail'}) {
	# Update VPOPMail user
	local $quser = quotemeta($_[1]->{'user'});
	local $qdom = quotemeta($_[2]->{'dom'});
	local $qreal = quotemeta($_[0]->{'real'}) || '""';
	local $qpass = quotemeta($_[0]->{'plainpass'});
	local $qquota = $_[0]->{'qquota'} ? $_[0]->{'qquota'} : "NOQUOTA";
	local $cmd = "$vpopbin/vmoduser -c $qreal ".
		     ($_[0]->{'passmode'} == 3 ? " -C $qpass" : "").
		     " -q $qquota $quser\@$qdom";
	local $out = &backquote_logged("$cmd 2>&1");
	if ($?) {
		&error("<tt>$cmd</tt> failed: <pre>$out</pre>");
		}
	if ($_[0]->{'user'} ne $_[1]->{'user'}) {
		# Need to rename manually
		local $vdomdir = "$config{'vpopmail_dir'}/domains/$_[2]->{'dom'}";
		&rename_logged("$vdomdir/$_[1]->{'user'}", "$vdomdir/$_[0]->{'user'}");
		&lock_file("$vdomdir/vpasswd");
		local $lref = &read_file_lines("$vdomdir/vpasswd");
		local $l;
		foreach $l (@$lref) {
			local @u = split(/:/, $l);
			if ($u[0] eq $_[1]->{'user'}) {
				$u[0] = $_[0]->{'user'};
				$u[5] =~ s/$_[1]->{'user'}$/$_[0]->{'user'}/;
				$l = join(":", @u);
				}
			}
		&flush_file_lines();
		&unlock_file("$vdomdir/vpasswd");
		&system_logged("$vpopbin/vmkpasswd $qdom");
		}
	}
else {
	# Modifying Unix user
	&require_useradmin();
	&require_mail();

	# Update the unix user
	&foreign_call($usermodule, "set_user_envs", $_[0], 'MODIFY_USER', $_[0]->{'plainpass'});
	&foreign_call($usermodule, "making_changes");
	&foreign_call($usermodule, "lock_user_files");
	&userdom_substitutions($_[0], $_[2]);
	&foreign_call($usermodule, "modify_user", $_[1], $_[0]);
	&foreign_call($usermodule, "unlock_user_files");
	&foreign_call($usermodule, "made_changes");
	return if ($_[3]);		# no need to touch aliases and virtusers
	}

if (!$_[0]->{'qmail'}) {
	# Take away all virtusers and add new ones, for non Qmail+LDAP users
	&delete_virtuser($_[1]->{'virt'}) if ($_[1]->{'virt'});
	local $e;
	foreach $e (@{$_[1]->{'extravirt'}}) {
		&delete_virtuser($e);
		}
	local $firstemail;
	local $vto = $_[0]->{'to'} ? &escape_alias($_[0]->{'user'})
				   : &escape_user($_[0]->{'user'});
	if ($_[0]->{'email'}) {
		local $virt = { 'from' => $_[0]->{'email'},
				'to' => [ $vto ] };
		&create_virtuser($virt);
		$_[0]->{'virt'} = $virt;
		$firstemail ||= $_[0]->{'email'};
		}
	elsif ($can_alias_types{9} && $_[2] && !$_[0]->{'noprimary'}) {
		# Add bouncer if email disabled
		local $virt = { 'from' => "$pop3\@$_[2]->{'dom'}",
				'to' => [ "BOUNCE" ] };
		&create_virtuser($virt);
		$_[0]->{'virt'} = $virt;
		}
	local @extravirt;
	foreach $e (@{$_[0]->{'extraemail'}}) {
		local $virt = { 'from' => $e,
				'to' => [ $vto ] };
		&create_virtuser($virt);
		push(@extravirt, $virt);
		$firstemail ||= $e;
		}
	$_[0]->{'extravirt'} = \@extravirt;
	}

local @to = @{$_[0]->{'to'}};
local @oldto = @{$_[1]->{'to'}};
if (!$_[0]->{'qmail'}) {
	# Update, create or delete alias, for non Qmail+LDAP users
	if (@to && !@oldto) {
		# Need to add alias
		local $alias = { 'name' => &escape_alias($_[0]->{'user'}),
				 'enabled' => 1,
				 'values' => $_[0]->{'to'} };
		&check_alias_clash($_[0]->{'user'}) &&
			&error(&text('alias_eclash2', $_[0]->{'user'}));
		if ($config{'mail_system'} == 1) {
			# Create Sendmail alias with same name as user
			&sendmail::lock_alias_files($sendmail_afiles);
			&sendmail::create_alias($alias, $sendmail_afiles);
			&sendmail::unlock_alias_files($sendmail_afiles);
			}
		elsif ($config{'mail_system'} == 0) {
			# Create Postfix alias with same name as user
			&postfix::lock_alias_files($postfix_afiles);
			&postfix::create_alias($alias, $postfix_afiles);
			&postfix::unlock_alias_files($postfix_afiles);
			&postfix::regenerate_aliases();
			}
		elsif ($config{'mail_system'} == 2 ||
		       $config{'mail_system'} == 5) {
			# Set up user's .qmail file
			local $dqm = &dotqmail_file($_[0]);
			&lock_file($dqm);
			&save_dotqmail($alias, $dqm, $pop3);
			&unlock_file($dqm);
			}
		$_[0]->{'alias'} = $alias;
		}
	elsif (!@to && @oldto) {
		# Need to delete alias
		if ($config{'mail_system'} == 1) {
			# Delete Sendmail alias
			&lock_file($_[0]->{'alias'}->{'file'});
			&sendmail::delete_alias($_[0]->{'alias'});
			&unlock_file($_[0]->{'alias'}->{'file'});
			}
		elsif ($config{'mail_system'} == 0) {
			# Delete Postfix alias
			&lock_file($_[0]->{'alias'}->{'file'});
			&postfix::delete_alias($_[0]->{'alias'});
			&unlock_file($_[0]->{'alias'}->{'file'});
			&postfix::regenerate_aliases();
			}
		elsif ($config{'mail_system'} == 2 ||
		       $config{'mail_system'} == 5) {
			# Remove user's .qmail file
			local $dqm = &dotqmail_file($_[0]);
			&lock_file($dqm);
			unlink($dqm);
			&unlock_file($dqm);
			}
		}
	elsif (@to && @oldto) {
		# Need to update the alias
		local $alias = { 'name' => &escape_alias($_[0]->{'user'}),
				 'enabled' => 1,
				 'values' => $_[0]->{'to'} };
		if ($config{'mail_system'} == 1) {
			# Update Sendmail alias
			&lock_file($_[1]->{'alias'}->{'file'});
			&sendmail::modify_alias($_[1]->{'alias'}, $alias);
			&unlock_file($_[1]->{'alias'}->{'file'});
			}
		elsif ($config{'mail_system'} == 0) {
			# Update Postfix alias
			&lock_file($_[1]->{'alias'}->{'file'});
			&postfix::modify_alias($_[1]->{'alias'}, $alias);
			&unlock_file($_[1]->{'alias'}->{'file'});
			&postfix::regenerate_aliases();
			}
		elsif ($config{'mail_system'} == 2 ||
		       $config{'mail_system'} == 5) {
			# Set up user's .qmail file
			local $dqm = &dotqmail_file($_[0]);
			&lock_file($dqm);
			&save_dotqmail($alias, $dqm, $pop3);
			&unlock_file($dqm);
			}
		$_[0]->{'alias'} = $alias;
		}

	if ($config{'generics'} && $_[0]->{'generic'}) {
		# Update genericstable entry too
		if ($config{'mail_system'} == 1) {
			&lock_file($sendmail_gfile);
			&sendmail::delete_generic($_[0]->{'generic'}, $sendmail_gfile,
					$sendmail_gdbm, $sendmail_gdbmtype);
			if ($firstemail) {
				local $gen = { 'from' => $_[0]->{'user'},
					       'to' => $firstemail };
				&sendmail::create_generic($gen, $sendmail_gfile,
						  $sendmail_gdbm, $sendmail_gdbmtype)
				}
			&unlock_file($sendmail_gfile);
			}
		elsif ($config{'mail_system'} == 0) {
			&lock_file($canonical_map_files[0]);
			&postfix::delete_mapping($canonical_type, $_[0]->{'generic'});
			if ($firstemail) {
				local $gen = { 'name' => $_[0]->{'user'},
					       'value' => $firstemail };
				&create_replace_mapping($canonical_type, $gen);
				}
			&unlock_file($canonical_map_files[0]);
			&postfix::regenerate_canonical_table();
			}
		}
	}
}

# delete_user(&user, domain)
# Delete a mailbox user and all associated virtusers and aliases
sub delete_user
{
if ($_[0]->{'qmail'}) {
	# Delete user in Qmail LDAP
	local $ldap = &connect_qmail_ldap();
	local $rv = $ldap->delete($_[0]->{'dn'});
	&error($rv->error) if ($rv->code);
	$ldap->unbind();
	}
elsif ($_[0]->{'vpopmail'}) {
	# Call VPOPMail delete user program
	local $quser = quotemeta($_[0]->{'user'});
	local $qdom = quotemeta($_[1]->{'dom'});
	local $cmd = "$vpopbin/vdeluser $quser\@$qdom";
	local $out = &backquote_logged("$cmd 2>&1");
	if ($?) {
		&error("<tt>$cmd</tt> failed: <pre>$out</pre>");
		}
	}
else {
	# Delete Unix user
	$_[0]->{'user'} eq 'root' && &error("Cannot delete root user!");
	$_[0]->{'uid'} == 0 && &error("Cannot delete UID 0 user!");
	&require_useradmin();
	&require_mail();

	# Delete the user
	&foreign_call($usermodule, "set_user_envs", $_[0], 'DELETE_USER')
	&foreign_call($usermodule, "making_changes");
	&foreign_call($usermodule, "lock_user_files");
	&foreign_call($usermodule, "delete_user",$_[0]);
	&foreign_call($usermodule, "unlock_user_files");
	&foreign_call($usermodule, "made_changes");
	}

if (!$_[0]->{'qmail'}) {
	# Delete any virtusers
	&delete_virtuser($_[0]->{'virt'}) if ($_[0]->{'virt'});
	local $e;
	foreach $e (@{$_[0]->{'extravirt'}}) {
		&delete_virtuser($e);
		}
	}

if (!$_[0]->{'qmail'}) {
	# Delete his alias, if any
	if ($_[0]->{'alias'}) {
		if ($config{'mail_system'} == 1) {
			# Delete Sendmail alias with same name as user
			&lock_file($_[0]->{'alias'}->{'file'});
			&sendmail::delete_alias($_[0]->{'alias'});
			&unlock_file($_[0]->{'alias'}->{'file'});
			}
		elsif ($config{'mail_system'} == 0) {
			# Delete Postfix alias with same name as user
			&lock_file($_[0]->{'alias'}->{'file'});
			&postfix::delete_alias($_[0]->{'alias'});
			&unlock_file($_[0]->{'alias'}->{'file'});
			&postfix::regenerate_aliases();
			}
		elsif ($config{'mail_system'} == 2 ||
		       $config{'mail_system'} == 5) {
			# .qmail will be deleted when user is
			}
		}

	if ($config{'generics'} && $_[0]->{'generic'}) {
		# Delete genericstable entry too
		if ($config{'mail_system'} == 1) {
			&lock_file($sendmail_gfile);
			&sendmail::delete_generic($_[0]->{'generic'}, $sendmail_gfile,
					$sendmail_gdbm, $sendmail_gdbmtype);
			&unlock_file($sendmail_gfile);
			}
		elsif ($config{'mail_system'} == 0) {
			&lock_file($_[0]->{'generic'}->{'file'});
			&postfix::delete_mapping($canonical_type, $_[0]->{'generic'});
			&unlock_file($_[0]->{'generic'}->{'file'});
			&postfix::regenerate_canonical_table();
			}
		}
	}
}

# domain_title(&domain)
sub domain_title
{
print "<center><font size=+1>",&domain_in($_[0]),"</font></center>\n";
}

# domain_in(&domain)
sub domain_in
{
return &text('indom', "<tt>$_[0]->{'dom'}</tt>");
}

# copy_skel_files(basedir, &user, home, [group])
# Copy files to the home directory of some new user
sub copy_skel_files
{
local $uf = $_[0];
return if (!$uf);
&require_useradmin();
if ($_[1]) {
	local $shell = $_[1]->{'shell'};
	$shell =~ s/^(.*)\///g;
	local $group = $_[3];
	$group = getgrgid($_[1]->{'gid'}) if (!$group);
	$uf =~ s/\$group/$group/g;
	$uf =~ s/\$gid/$_[1]->{'gid'}/g;
	$uf =~ s/\$shell/$shell/g;
	&useradmin::copy_skel_files($uf, $_[2],
				    $_[1]->{'uid'}, $_[1]->{'gid'});
	}
else {
	$uf =~ s/\$group/nogroup/g;
	$uf =~ s/\$gid/100/g;
	$uf =~ s/\$shell/\/bin\/false/g;
	&useradmin::copy_skel_files($uf, $_[2], 0, 0);
	}
}

# can_edit_domain(&domain)
# Returns 1 if the current user can edit some domain
sub can_edit_domain
{
return 1 if ($access{'domains'} eq "*");
local $d;
foreach $d (split(/\s+/, $access{'domains'})) {
	return 1 if ($d eq $_[0]->{'id'});
	}
return 0;
}

# master_admin()
# Returns 1 if the current user is the master Virtualmin admin, and can edit
# global settings
sub master_admin
{
return !$access{'noconfig'};
}

# domains_table(&domains)
# Display a list of domains in a table, with links for editing
sub domains_table
{
local @table_features = $config{'show_features'} ?
    (grep { $_ ne 'webmin' && $_ ne 'mail' && $_ ne 'unix' } @features) : ( );
print "<table border width=100%>\n";
print "<tr $tb> <td><b>$text{'index_domain'}</b></td> ",
      "<td><b>$text{'index_user'}</b></td> ",
      "<td><b>$text{'index_owner'}</b></td> ";
local $f;
foreach $f (@table_features) {
	print "<td><b>",$text{'index_'.$f},"</b></td> " if ($config{$f});
	}
if ($config{'mail'}) {
	print "<td><b>$text{'index_mail'}</b></td> ";
	print "<td><b>$text{'index_alias'}</b></td> ";
	}
if ($config{'home_quotas'}) {
	print "<td><b>$text{'index_quota'}</b></td> ".
	      "<td><b>$text{'index_uquota'}</b></td> ";
	}
print "</tr>\n";
local $d;
local %done;
foreach $d (sort { $a->{'user'} cmp $b->{'user'} ||
		   $a->{'created'} <=> $b->{'created'} } @{$_[0]}) {
	$done{$d->{'id'}}++;
	print "<tr $cb>\n";
	local $dn = $d->{'disabled'} ? "<i>$d->{'dom'}</i>" : $d->{'dom'};
	local $pfx;
	$pfx .= "&nbsp;&nbsp;" if ($d->{'parent'} && $done{$d->{'parent'}});
	$pfx .= "&nbsp;&nbsp;" if ($d->{'alias'} && $done{$d->{'alias'}});
	if ($access{'edit'}) {
		print "<td>$pfx<a href='edit_domain.cgi?dom=$d->{'id'}'>",
		      "$dn</a></td>\n";
		}
	else {
		print "<td>$pfx$dn</td>\n";
		}
	print "<td>$d->{'user'}</td>\n";
	print "<td>$d->{'owner'}\n";
	if ($d->{'alias'}) {
		local $aliasdom = &get_domain($d->{'alias'});
		local $of = &text('index_aliasof', $aliasdom->{'dom'});
		print $d->{'owner'} ? "($of)" : $of;
		}
	print "</td>\n";
	foreach $f (@table_features) {
		print "<td>",$d->{$f} ? $text{'yes'} : $text{'no'},"</td>\n"
			if ($config{$f});
		}
	local @users = &list_domain_users($d, 0, 1, 0);
	local ($duser) = grep { $_->{'user'} eq $d->{'user'} } @users;
	if ($config{'mail'}) {
		if ($d->{'mail'}) {
			local @aliases = &list_domain_aliases($d);
			if ($d->{'alias'}) {
				print "<td>0</td>\n";
				}
			else {
				printf "<td>%d&nbsp;(<a href='list_users.cgi?dom=$d->{'id'}'>$text{'index_list'}</a>)</td>\n", scalar(@users);
				}
			printf "<td>%d&nbsp;(<a href='list_aliases.cgi?dom=$d->{'id'}'>$text{'index_list'}</a>)</td>\n", scalar(@aliases);
			}
		else {
			print "<td colspan=2>$text{'index_nomail'}</td>\n";
			}
		}
	if ($config{'home_quotas'}) {
		if ($d->{'parent'}) {
			# Domains with parent have no quota
			if ($done{$d->{'parent'}}) {
				print "<td>&nbsp;&nbsp;\"</td>\n";
				}
			else {
				print "<td>$text{'index_samequ'}</td>\n";
				}
			}
		else {
			# Show quota for domain
			print "<td>",$d->{'quota'} ?
			  &quota_show($d->{'quota'}, $config{'home_quotas'}) :
			  $text{'form_unlimit'},"</td>\n";
			}
		if ($d->{'alias'}) {
			# Alias domains have no usage
			print "<td><br></td>\n";
			}
		else {
			# Show total usage for domain
			local $ut = $duser->{'uquota'} + $duser->{'umquota'};
			foreach $u (@users) {
				$ut += $u->{'uquota'} + $u->{'umquota'}
					if ($u->{'user'} ne $d->{'user'});
				}
			print "<td>",&quota_show($ut, $config{'home_quotas'}),"</td>\n";
			}
		}
	print "</tr>\n";
	}
print "</table>\n";
}

# userdom_name(name, &domain)
# Returns a username with the domain prefix (usually group) appended somehow
sub userdom_name
{
if ($config{'append_style'} == 0) {
	return $_[0].".".$_[1]->{'prefix'};
	}
elsif ($config{'append_style'} == 1) {
	return $_[0]."-".$_[1]->{'prefix'};
	}
elsif ($config{'append_style'} == 2) {
	return $_[1]->{'prefix'}.".".$_[0];
	}
elsif ($config{'append_style'} == 3) {
	return $_[1]->{'prefix'}."-".$_[0];
	}
elsif ($config{'append_style'} == 4) {
	return $_[0]."_".$_[1]->{'prefix'};
	}
elsif ($config{'append_style'} == 5) {
	return $_[1]->{'prefix'}."_".$_[0];
	}
elsif ($config{'append_style'} == 6) {
	return $_[0]."\@".$_[1]->{'dom'};
	}
else {
	&error("Unknown append_style $config{'append_style'}!");
	}
}

# remove_userdom(name, &domain)
# Returns a username with the domain prefix (group) stripped off
sub remove_userdom
{
return $_[0] if (!$_[1]);
local $g = $_[1]->{'prefix'};
local $d = $_[1]->{'dom'};
local $rv = $_[0];
($rv =~ s/\@(\Q$d\E)$//) || ($rv =~ s/(\.|\-|_)\Q$g\E$//) || ($rv =~ s/^\Q$g\E(\.|\-|_)//);
return $rv;
}

# too_long(name)
# Returns an error message if a username is too long for this Unix variant
sub too_long
{
local $max = &max_username_length();
if ($max && length($_[0]) > $max) {
	return &text('user_elong', "<tt>$_[0]</tt>", $max);
	}
else {
	return undef;
	}
}

sub max_username_length
{
&require_useradmin();
return $uconfig{'max_length'};
}

# get_default_ip()
# Returns this system's primary IP address
sub get_default_ip
{
if ($config{'defip'}) {
	return $config{'defip'};
	}
else {
	&foreign_require("net", "net-lib.pl");
	local ($iface) = grep { $_->{'fullname'} eq $config{'iface'} }
			      &net::active_interfaces();
	return $iface->{'address'};
	}
}

# check_apache_directives([directives])
# Returns an error string if the default Apache directives don't look valid
sub check_apache_directives
{
local ($d, $gotname, $gotdom, $gotdoc, $gotproxy);
local @dirs = split(/\t+/, defined($_[0]) ? $_[0] : $config{'apache_config'});
foreach $d (@dirs) {
	$d =~ s/#.*$//;
	if ($d =~ /^\s*ServerName\s+(\S+)$/i) {
		$gotname++;
		$gotdom++ if ($1 eq '$DOM' || $1 eq '${DOM}');
		}
	if ($d =~ /^\s*ServerAlias\s+(.*)$/) {
		$gotdom++ if (&indexof('$DOM', split(/\s+/, $1)) >= 0 ||
			      &indexof('${DOM}', split(/\s+/, $1)) >= 0);
		}
	$gotdoc++ if ($d =~ /^\s*DocumentRoot\s+(.*)$/);
	$gotproxy++ if ($d =~ /^\s*ProxyPass\s+(.*)$/);
	}
$gotname || return $text{'acheck_ename'};
$gotdom || return $text{'acheck_edom'};
$gotdoc || $gotproxy || return $text{'acheck_edoc'};
return undef;
}

# Print functions for HTML output
sub first_html_print { print @_,"<br>\n"; }
sub second_html_print { print @_,"<p>\n"; }
sub indent_html_print { print "<ul>\n"; }
sub outdent_html_print { print "</ul>\n"; }

# Print functions for text output
sub first_text_print
{
print $indent_text,(map { &entities_to_ascii($_) } @_),"\n";
}
sub second_text_print
{
print $indent_text,(map { &entities_to_ascii($_) } @_),"\n\n";
}
sub indent_text_print { $indent_text .= "    "; }
sub outdent_text_print { $indent_text = substr($indent_text, 4); }

sub null_print { }

sub set_all_null_print
{
$first_print = $second_print = $indent_print = $outdent_print = \&null_print;
}

# send_domain_email(&domain)
# Sends email to a new domain owner. Returns a pair containing a number
# (0=failed, 1=success) and an optional message. Also outputs status messages.
sub send_domain_email
{
local $tmpl = &get_template($_[0]->{'template'});
local $mail = $tmpl->{'mail'};
local $subject = $tmpl->{'mail_subject'};
local $cc = $tmpl->{'mail_cc'};
return (1, undef) if ($tmpl->{'mail_on'} eq 'none');
&$first_print($text{'setup_email'});
local @erv = &send_template_email($mail,
				  $_[0]->{'email'} ||
				    $_[0]->{'user'}.'@'.&get_system_hostname(),
			    	  $_[0], $subject,
				  $cc);
if ($erv[0]) {
	&$second_print(&text('setup_emailok', $erv[1]));
	}
else {
	&$second_print(&text('setup_emailfailed', $erv[1]));
	}
}

# will_send_user_email([&domain])
# Returns 1 if a new mailbox email would be sent to a user in this domain
sub will_send_user_email
{
local $tmode = $_[0] ? "user" : "local";
if ($config{$tmode.'_template'} eq 'none') {
        return 0;
        }
else {
        return 1;
        }
}

# send_user_email([&domain], &user, [to])
# Sends email to a new mailbox user. Returns a pair containing a number
# (0=failed, 1=success) and an optional message
sub send_user_email
{
local $tmode = $_[0] ? "user" : "local";
local $subject = $_[0] ? $config{'newuser_subject'}
		       : $config{'newlocal_subject'};
local $cc = $_[0] ? $config{'newuser_cc'}
		  : $config{'newlocal_cc'};
&ensure_template($tmode."-template");
return (1, undef) if ($config{$tmode.'_template'} eq 'none');
local $tmpl = $config{$tmode.'_template'} eq 'default' ?
	"$module_config_directory/$tmode-template" :
	$config{$tmode.'_template'};
local %hash;
local $email;
if ($_[0]) {
	%hash = ( %{$_[0]}, %{$_[1]} );
	$hash{'mailbox'} = &remove_userdom($_[1]->{'user'}, $_[0]);
	$email = $hash{'mailbox'}.'@'.$hash{'dom'};
	}
else {
	%hash = ( %{$_[1]} );
	$hash{'mailbox'} = $hash{'user'};
	$email = $hash{'user'}.'@'.&get_system_hostname();
	}
if ($_[2]) {
	$email = $_[2];
	}
$hash{'ftp'} = $_[1]->{'shell'} eq $config{'ftp_shell'} ||
	       $config{'jail_shell'} &&
		$_[1]->{'shell'} eq $config{'jail_shell'} ? 1 : 0;
return &send_template_email(&cat_file($tmpl), $email, \%hash,
			    $subject ||
			&entities_to_ascii($text{'mail_usubject'}), $cc);
}

# ensure_template(file)
sub ensure_template
{
&system_logged("cp $module_root_directory/$_[0] $module_config_directory/$_[0]")
	if (!-r "$module_config_directory/$_[0]");
}

# send_template_email(data, address, &substitions, subject, cc)
# Sends the given file to the specified address, with the substitions from
# a hash reference. The actual subs in the file must be like $XXX for entries
# in the hash like xxx - ie. $DOM is replaced by the domain name, and $HOME
# by the home directory
sub send_template_email
{
local $tmpl = $_[0];
$tmpl = &substitute_template($tmpl, $_[2]);

# Actually send using the mailboxes module
return (0, $text{'mail_system'})
	if (!$config{'mail'} || $config{'mail_system'} == 3);
local $subject = &substitute_template($_[3], $_[2]);
local $cc = &substitute_template($_[4], $_[2]);
&require_mail();
&foreign_require("mailboxes", "mailboxes-lib.pl");
local $mail = { 'headers' => [ [ 'From', $config{'from_addr'} ||
					 &mailboxes::get_from_address() ],
			       [ 'To', $_[1] ],
			       $cc ? ( [ 'Cc', $cc ] ) : ( ),
			       [ 'Subject', $subject ],
			       [ 'Content-type', 'text/plain' ] ],
		'body' => $tmpl };
&mailboxes::send_mail($mail);
return (1, &text('mail_ok', $_[1]));
}

# substitute_template(text, &hash)
# Given some text and a hash reference, for each ocurrance of $FOO or ${FOO} in
# the text replaces it with the value of the hash key foo
sub substitute_template
{
# Add some extra fixed parameters to the hash
local %hash = %{$_[1]};
$hash{'hostname'} = &get_system_hostname();

# Actually do the substition
local $rv = $_[0];
local $s;
foreach $s (keys %hash) {
	local $us = uc($s);
	local $sv = $hash{$s};
	$rv =~ s/\$\{\Q$us\E\}/$sv/g;
	$rv =~ s/\$\Q$us\E/$sv/g;
	if ($sv) {
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ELSE-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)/\2/g;
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)/\2/g;

		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ELSE-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)/\2/g;
		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)/\2/g;
		}
	else {
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ELSE-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)/\4/g;
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)//g;

		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ELSE-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)/\4/g;
		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)//g;
		}
	}
return $rv;
}

# userdom_substitutions(&user, &dom)
# Returns a hash reference of substitutions for a user in a domain
sub userdom_substitutions
{
if ($_[1]) {
	$_[0]->{'mailbox'} = &remove_userdom($_[0]->{'user'}, $_[1]);
	$_[0]->{'dom'} = $_[1]->{'dom'};
	$_[0]->{'dom_prefix'} = substr($_[1]->{'dom'}, 0, 1);
	}
return $_[0];
}

# alias_type(string, [alias-name])
# Return the type and destination of some alias string
sub alias_type
{
local @rv;
if ($_[0] =~ /^\|$module_config_directory\/autoreply.pl\s+(\S+)/) {
        @rv = (5, $1);
        }
elsif ($_[0] =~ /^\|$module_config_directory\/filter.pl\s+(\S+)/) {
        @rv = (6, $1);
        }
elsif ($_[0] =~ /^\|(.*)$/) {
        @rv = (4, $1);
        }
elsif ($_[0] eq "./Maildir/") {
	return (10);
	}
elsif ($_[0] =~ /^(\/.*)$/ || $_[0] =~ /^\.\//) {
        @rv = (3, $_[0]);
        }
elsif ($_[0] =~ /^:include:(.*)$/) {
        @rv = (2, $1);
        }
elsif ($_[0] =~ /^\\(\S+)$/) {
	if ($1 eq $_[1] || $1 eq "NEWUSER") {
		return (10);
		}
	else {
		@rv = (7, $1);
		}
        }
elsif ($_[0] =~ /^\%1\@(\S+)$/) {
        @rv = (8, $1);
        }
elsif ($_[0] eq "BOUNCE") {
        @rv = (9, undef);
        }
else {
        @rv = (1, $_[0]);
        }
return wantarray ? @rv : $rv[0];
}

# set_domain_envs(&domain, action)
# Sets up VIRTUALSERVER_ environment variables for a domain update or some kind,
# prior to calling making_changes or made_changes. action must be one of
# CREATE_DOMAIN, MODIFY_DOMAIN or DELETE_DOMAIN
sub set_domain_envs
{
local $e;
foreach $e (keys %ENVS) {
	delete($ENV{$e}) if ($e =~ /^VIRTUALSERVER_/);
	}
$ENV{'VIRTUALSERVER_ACTION'} = $_[1];
foreach $e (keys %{$_[0]}) {
	$ENV{'VIRTUALSERVER_'.uc($e)} = $_[0]->{$e};
	}
}

# making_changes()
# Called before a domain is created, modified or deleted to run the
# pre-change command
sub making_changes
{
if ($config{'pre_command'} =~ /\S/) {
	&clean_changes_environment();
	local $out = &backquote_logged("($config{'pre_command'}) 2>&1 </dev/null");
	&reset_changes_environment();
	return $? ? $out : undef;
	}
return undef;
}

# made_changes()
# Called after a domain has been created, modified or deleted to run the
# post-change command
sub made_changes
{
if ($config{'post_command'} =~ /\S/) {
	&clean_changes_environment();
	local $out = &backquote_logged("($config{'post_command'}) 2>&1 </dev/null");
	&reset_changes_environment();
	return $? ? $out : undef;
	}
return undef;
}

sub reset_changes_environment
{
%ENV = %UNCLEAN_ENV;
}

sub clean_changes_environment
{
local $e;
%UNCLEAN_ENV = %ENV;
foreach $e ('SERVER_ROOT', 'SCRIPT_NAME',
	    'FOREIGN_MODULE_NAME', 'FOREIGN_ROOT_DIRECTORY',
	    'SCRIPT_FILENAME') {
	delete($ENV{$e});
	}
}

# switch_to_domain_user(&domain)
# Changes the current UID and GID to that of the domain's unix user
sub switch_to_domain_user
{
($(, $)) = ( $_[0]->{'ugid'},
	     "$_[0]->{'ugid'} ".join(" ", $_[0]->{'ugid'},
					 &other_groups($_[0]->{'user'})) );
($<, $>) = ( $_[0]->{'uid'}, $_[0]->{'uid'} );
$ENV{'USER'} = $ENV{'LOGNAME'} = $_[0]->{'user'};
$ENV{'HOME'} = $_[0]->{'home'};
}

# print_subs_table(sub, ..)
sub print_subs_table
{
print "<table>\n";
foreach $k (@_) {
	print "<tr> <td><tt><b>\${$k}</b></td>\n";
	print "<td>",$text{"sub_".$k},"</td> </tr>\n";
	}
print "</table>\n";
print "$text{'sub_if'}<p>\n";
}

# alias_form(&to, left, &domain, "user"|"alias", user|alias)
sub alias_form
{
&require_mail();
local @typenames = map { $text{"alias_type$_"} } (0 .. 10);
$typenames[0] = "&lt;$typenames[0]&gt;";

local $left = $_[1];
local @values = @{$_[0]};
local $i;
for($i=0; $i<=@values+2; $i++) {
	print "<tr> <td>$left</td> <td>\n";
	$left = "";
	local ($type, $val) = $values[$i] ? &alias_type($values[$i], $_[4]) : (0, "");
	print "<select name=type_$i>\n";
	local $j;
	for($j=0; $j<@typenames; $j++) {
		next if ($j == 8 && $_[3] eq "user");	# to domain not valid
							# for users
		next if ($j == 10 && $_[3] ne "user");	# user's mailbox not
							# valid for aliases
		if ($j == 0 || $can_alias_types{$j} ||
		    $type == $j) {
			printf "<option value=$j %s>$typenames[$j]\n",
				$type == $j ? "selected" : "";
			}
		}
	print "</select>\n";
	if ($type == 7) {
		$val = &unescape_user($val);
		}
	print "<input name=val_$i size=30 value=\"$val\">\n";
	if ($config{'edit_afiles'} || &master_admin()) {
		local $prog = $type == 2 ? "edit_afile.cgi" :
			      $type == 5 ? "edit_rfile.cgi" :
			      $type == 6 ? "edit_ffile.cgi" : undef;
		if ($prog && $_[2]) {
			local $di = $_[2] ? $_[2]->{'id'} : undef;
			print "<a href='$prog?dom=$di&file=$val&$_[3]=$_[4]'>$text{'alias_afile'}</a>\n";
			}
		}
	print "</td> </tr>\n";
	}
}

# parse_alias(catchall, name)
# Returns a list of values for an alias, taken from the form generated by
# &alias_form
sub parse_alias
{
local (@values, $i, $t, $anysame);
for($i=0; defined($t = $in{"type_$i"}); $i++) {
	!$t || $can_alias_types{$t} ||
		&error($text{'alias_etype'}." : ".$text{'alias_type'.$t});
	local $v = $in{"val_$i"};
	$v =~ s/^\s+//;
	$v =~ s/\s+$//;
	if ($t == 1 && $v !~ /^(\S+)$/) {
		&error(&text('alias_etype1', $v));
		}
	elsif ($t == 3 && $v !~ /^\/(\S+)$/ && $v !~ /^\.\//) {
		&error(&text('alias_etype3', $v));
		}
	elsif ($t == 4) {
		$v =~ /^(\S+)/ || &error($text{'alias_etype4none'});
		(-x $1) && &check_aliasfile($1, 0) ||
			&error(&text('alias_etype4', $1));
		}
	elsif ($t == 7 && !defined(getpwnam($v)) &&
	       $config{'mail_system'} != 4 && $config{'mail_system'} != 5) {
		&error(&text('alias_etype7', $v));
		}
	elsif ($t == 8 && $v !~ /^[a-z0-9\.\-\_]+$/) {
		&error(&text('alias_etype8', $v));
		}
	elsif ($t == 8 && !$_[0]) {
		&error(&text('alias_ecatchall', $v));
		}
	if ($t == 1 || $t == 3) { push(@values, $v); }
	elsif ($t == 2) {
		$v = "$d->{'home'}/$v" if ($v !~ /^\//);
		push(@values, ":include:$v");
		}
	elsif ($t == 4) {
		$v = "$d->{'home'}/$v" if ($v !~ /^\//);
		push(@values, "|$v");
		}
	elsif ($t == 5) {
		# Setup autoreply script
		$v = "$d->{'home'}/$v" if ($v !~ /^\//);
		push(@values, "|$module_config_directory/autoreply.pl ".
			      "$v $name");
		&system_logged("cp autoreply.pl $module_config_directory");
		&system_logged("chmod 755 $module_config_directory/config");
		if (-d $sendmail::config{'smrsh_dir'}) {
			&system_logged("ln -s $module_config_directory/autoreply.pl $sendmail::config{'smrsh_dir'}/autoreply.pl");
			}
		}
	elsif ($t == 6) {
		# Setup filter script
		$v = "$d->{'home'}/$v" if ($v !~ /^\//);
		push(@values, "|$module_config_directory/filter.pl ".
			      "$v $name");
		&system_logged("cp filter.pl $module_config_directory");
		&system_logged("chmod 755 $module_config_directory/config");
		if (-d $sendmail::config{'smrsh_dir'}) {
			&system_logged("ln -s $module_config_directory/filter.pl $sendmail::config{'smrsh_dir'}/filter.pl");
			}
		}
	elsif ($t == 7) {
		push(@values, "\\".&escape_user($v));
		}
	elsif ($t == 8) {
		push(@values, "\%1\@$v");
		$anysame++;
		}
	elsif ($t == 9) {
		push(@values, "BOUNCE");
		}
	elsif ($t == 10) {
		push(@values, "\\$_[1]");
		}
	}
if (@values > 1 && $anysame) {
	&error(&text('alias_ecatchall2', $v));
	}
return @values;
}

# set_pass_change(&user)
sub set_pass_change
{
&require_useradmin();
local $pft = &useradmin::passfiles_type();
if ($pft == 2 || $pft == 5 || $config{'ldap'}) {
	$_[0]->{'change'} = int(time() / (60*60*24));
	}
elsif ($pft == 4) {
	$_[0]->{'change'} = time();
	}
}

sub check_aliasfile
{
return 0 if (!-r $_[0] && !$_[1]);
return 1;
}

# list_all_users()
# Returns all local and LDAP users, including those from Qmail
sub list_all_users
{
local @rv = &useradmin::list_users();
if ($config{'ldap'}) {
	push(@rv, &ldap_useradmin::list_users());
	}
if ($config{'mail_system'} == 4) {
	local $ldap = &connect_qmail_ldap();
	local $rv = $ldap->search(base => $config{'ldap_base'},
				  filter => "(objectClass=qmailUser)");
	local $u;
	foreach $u ($rv->all_entries) {
		local %uinfo = &qmail_dn_to_hash($u);
		push(@rv, \%uinfo);
		}
	$ldap->unbind();
	}
return @rv;
}

# list_all_groups()
# Returns all local and LDAP groups
sub list_all_groups
{
local @rv = &useradmin::list_groups();
if ($config{'ldap'}) {
	push(@rv, &ldap_useradmin::list_groups());
	}
return @rv;
}

# build_taken(&uid-taken, &username-taken, [&users])
# Fills in the the given hashes with used usernames and UIDs
sub build_taken
{
&require_useradmin();

# Add Unix users
local @users = $_[2] ? @{$_[2]} : &list_all_users();
local $u;
foreach $u (@users) {
	$_[0]->{$u->{'uid'}} = 1;
	$_[1]->{$u->{'user'}} = 1;
	}

# Add domain users
local $d;
foreach $d (&list_domains()) {
	$_[0]->{$d->{'uid'}} = 1;
	$_[1]->{$d->{'user'}} = 1;
	}
}

# build_group_taken(&gid-taken, &groupname-taken, [&groups])
# Fills in the the given hashes with used group names and GIDs
sub build_group_taken
{
&require_useradmin();
local @groups = $_[2] ? @{$_[2]} : &list_all_groups();
local $g;
foreach $g (@groups) {
	$_[0]->{$g->{'gid'}} = 1;
	$_[1]->{$g->{'group'}} = 1;
	}
local $d;
foreach $d (&list_domains()) {
	$_[0]->{$d->{'gid'}} = 1;
	$_[1]->{$d->{'group'}} = 1;
	}
}

# allocate_uid(&uid-taken)
sub allocate_uid
{
local $uid = $uconfig{'base_uid'};
while($_[0]->{$uid}) {
	$uid++;
	}
return $uid;
}

# allocate_gid(&gid-taken)
sub allocate_gid
{
local $gid = $uconfig{'base_gid'};
while($_[0]->{$gid}) {
	$gid++;
	}
return $gid;
}

# server_home_directory(&domain, [&parentdomain])
# Returns the home directory for a new virtual server user
sub server_home_directory
{
&require_useradmin();
if ($_[0]->{'parent'}) {
	# Owned by some existing user, so under his home
	return "$_[1]->{'home'}/domains/$_[0]->{'dom'}";
	}
elsif ($config{'home_format'}) {
	# Use the template from the module config
	local $home = "$home_base/$config{'home_format'}";
	return &substitute_template($home, $_[0]);
	}
else {
	# Just use the Users and Groups module settings
	return &useradmin::auto_home_dir($home_base, $_[0]->{'user'},
						     $_[0]->{'ugroup'});
	}
}

# set_quota(user, filesystem, quota)
sub set_quota
{
&require_useradmin();
if ($config{'hard_quotas'}) {
	&quota::edit_user_quota($_[0], $_[1],
				int($_[2]), int($_[2]), 0, 0);
	}
else {
	&quota::edit_user_quota($_[0], $_[1],
				int($_[2]), 0, 0, 0);
	}
}

# set_server_quotas(&domain)
# Set the user and possibly group quotas for a domain
sub set_server_quotas
{
if ($config{'home_quotas'}) {
	&set_quota($_[0]->{'user'}, $config{'home_quotas'}, $_[0]->{'uquota'});
	}
if ($config{'mail_quotas'} &&
    $config{'mail_quotas'} ne $config{'home_quotas'}) {
	&set_quota($_[0]->{'user'}, $config{'mail_quotas'}, $_[0]->{'uquota'});
	}
if ($config{'group_quotas'}) {
	&require_useradmin();
	if ($config{'hard_quotas'}) {
		&quota::edit_group_quota(
			$_[0]->{'group'}, $config{'home_quotas'},
			int($_[0]->{'quota'}), int($_[0]->{'quota'}), 0, 0);
		}
	else {
		&quota::edit_group_quota(
			$_[0]->{'group'}, $config{'home_quotas'},
			int($_[0]->{'quota'}), 0, 0, 0);
		}
	}
}

# nice_size(bytes)
sub nice_size
{
return $_[0] > 10*1024*1024 ? int($_[0]/1024/1024)." MB" :
       $_[0] > 10*1024 ? int($_[0]/1024)." kB" : $_[0]." b";
}

# users_table(&users, &dom)
# Output a table of mailbox users
sub users_table
{
local $can_quotas = $config{'home_quotas'} || $config{'mail_quotas'};
local $can_qquotas = $config{'mail_system'} == 4 || $config{'mail_system'} == 5;
print "<table border width=100%>\n";
print "<tr $tb> <td><b>$text{'users_name'}</b></td> ",
      "<td><b>$text{'users_pop3'}</b></td> ",
      "<td><b>$text{'users_real'}</b></td> ",
      $can_quotas ?
	  "<td><b>$text{'users_quota'}</b></td> " : "",
      $can_qquotas ?
	  "<td><b>$text{'users_qquota'}</b></td> " : "",
      "<td><b>$text{'users_size'}</b></td> ",
      "<td><b>$text{'users_ftp'}</b></td> ";
local ($f, %plugcol);
foreach $f (@mail_plugins) {
	local $col = &plugin_call($f, "mailbox_header", $_[1]);
	if ($col) {
		$plugcol{$f} = $col;
		print "<td><b>$col</b></td> ";
		}
	}
print "</tr>\n";
local $u;
local $did = $_[1] ? $_[1]->{'id'} : 0;
foreach $u (@{$_[0]}) {
	local $pop3 = $_[1] ? &remove_userdom($u->{'user'}, $_[1])
			    : $u->{'user'};
	local $domuser = $_[1] && $u->{'user'} eq $_[1]->{'user'} &&
			 $u->{'unix'};
	print "<tr $cb>\n";
	print "<td><a href='edit_user.cgi?dom=$did&",
	      "user=$u->{'user'}&unix=$u->{'unix'}'>",
	      ($domuser ? "<b>$pop3</b>" : $pop3),"</a></td>\n";
	print "<td>$u->{'user'}</td>\n";
	print "<td>",($u->{'real'} || "<br>"),"</td>\n";
	local $quota;
	$quota += $u->{'quota'} if ($config{'home_quotas'});
	$quota += $u->{'mquota'} if ($config{'mail_quotas'} &&
		      $config{'home_quotas'} ne $config{'mail_quotas'});
	if (defined($quota)) {
		print "<td>",$quota ? &quota_show($quota,$config{'home_quotas'})
				    : $text{'form_unlimit'},"</td>\n";
		}
	if ($u->{'mailquota'}) {
		print "<td>",$u->{'qquota'} ? &nice_size($u->{'qquota'}) :
			     $text{'form_unlimit'},"<br>","</td>\n";
		}
	elsif ($can_qquotas) {
		print "<td><br></td>\n";
		}
	local ($sz) = &mail_file_size($u);
	$sz = $sz ? &nice_size($sz) : $text{'users_empty'};
	local $lnk = &read_mail_link($u, $_[1]);
	if ($lnk) {
		print "<td><a href='$lnk'>$sz</a></td>\n";
		}
	else {
		print "<td>$sz</td>\n";
		}
	printf "<td>%s</td>\n",
		$domuser ? $text{'users_main'} :
		!$u->{'unix'} && !$u->{'shell'} ? $text{'users_qmail'} :
		$u->{'shell'} eq $config{'ftp_shell'} ? $text{'yes'} :
		$config{'jail_shell'} &&
		 $u->{'shell'} eq $config{'jail_shell'} ? $text{'users_jail'} :
		$u->{'shell'} eq $config{'shell'} ? $text{'no'} :
			&text('users_shell', "<tt>$u->{'shell'}</tt>");
	foreach $f (grep { $plugcol{$_} } @mail_plugins) {
		print "<td>",&plugin_call($f, "mailbox_column", $u, $_[1]),"</td>\n";
		}
	print "</tr>\n";
	}
print "</table>\n";
}

# quota_bsize(filesystem)
sub quota_bsize
{
&require_useradmin();
if (defined(&quota::block_size)) {
	local $bsize;
	if (!exists($bsize_cache{$_[0]})) {
		$bsize_cache{$_[0]} = &quota::block_size($_[0]);
		}
	return $bsize_cache{$_[0]};
	}
return undef;
}

# quota_show(number, filesystem)
# Returns text for the quota on some filesystem, in a human-readable format
sub quota_show
{
local $bsize = &quota_bsize($_[1]);
if ($bsize) {
	return int($_[0]*$bsize/1024)." ".$text{'form_k'};
	}
return $_[0]." ".$text{'form_b'};
}

# quota_input(name, number, filesystem)
# Returns HTML for an input for entering a quota, doing block->kb conversion
sub quota_input
{
local $bsize = &quota_bsize($_[2]);
return sprintf "<input name=%s size=10 value='%s'> %s",
	$_[0], $_[1] eq '' ? '' : $bsize ? int($_[1]*$bsize/1024) : $_[1],
	$bsize ? $text{'form_k'} : $text{'form_b'};
}

# quota_parse(name, filesystem)
# Converts an entered quota into blocks
sub quota_parse
{
local $bsize = &quota_bsize($_[1]);
return $bsize ? int($in{$_[0]}*1024/$bsize) : $in{$_[0]};
}

# setup_virt(&domain)
# Bring up an interface for a domain, if the IP isn't already enabled
sub setup_virt
{
&foreign_require("net", "net-lib.pl");
local @boot = &net::active_interfaces();
&$first_print(&text('setup_virt', $_[0]->{'ip'}));
local ($iface) = grep { $_->{'fullname'} eq $config{'iface'} } @boot;
local $b;
local $vmax = $config{'iface_base'} || int($net::min_virtual_number);
foreach $b (@boot) {
	$vmax = $b->{'virtual'}
		if ($b->{'name'} eq $iface->{'name'} &&
		    $b->{'virtual'} > $vmax);
	}
local $virt = { 'address' => $_[0]->{'ip'},
		'netmask' => $net::virtual_netmask || $iface->{'netmask'},
		'broadcast' =>
			$net::virtual_netmask eq "255.255.255.255" ?
				$_[0]->{'ip'} : $iface->{'broadcast'},
		'name' => $iface->{'name'},
		'virtual' => $vmax+1,
		'up' => 1 };
$virt->{'fullname'} = $virt->{'name'}.":".$virt->{'virtual'};
&net::save_interface($virt);
&net::activate_interface($virt);
$_[0]->{'iface'} = $virt->{'fullname'};
&$second_print(&text('setup_virtdone', $_[0]->{'iface'}));
}

# delete_virt(&domain)
# Take down the network interface for a domain
sub delete_virt
{
&$first_print($text{'delete_virt'});
&foreign_require("net", "net-lib.pl");
local ($biface) = grep { $_->{'fullname'} eq $_[0]->{'iface'} }
	 	       &net::boot_interfaces();
local ($aiface) = grep { $_->{'fullname'} eq $_[0]->{'iface'} }
	 	       &net::active_interfaces();
if ($biface->{'virtual'} ne '') {
	&net::delete_interface($biface);
	&net::deactivate_interface($aiface) if($aiface);
	&$second_print($text{'setup_done'});
	}
else {
	&$second_print(&text('delete_novirt', $d->{'iface'}));
	}
}

# check_virt_clash(ip)
# Returns the interface if some IP is already in use
sub check_virt_clash
{
&foreign_require("net", "net-lib.pl");
local @boot = &net::boot_interfaces();
local ($boot) = grep { $_->{'address'} eq $_[0] } @boot;
local @active = &net::active_interfaces();
local ($active) = grep { $_->{'address'} eq $_[0] } @active;
return $active || $boot;
}

# backup_domains(file, &domains, &features, dir-format, skip-errors, &options)
# Perform a backup of one or more domains into a single tar.gz file. Returns
# undef on success, or an error message.
sub backup_domains
{
# Create a temp dir for the backup, to be tarred up later
local $backupdir = &tempname();
mkdir($backupdir, 0700);

# Go through all the domains, and for each feature call the backup function
# to add it to the backup directory
local $d;
local $ok = 1;
local @donedoms;
local ($okcount, $errcount) = (0, 0);
DOMAIN: foreach $d (@{$_[1]}) {
	&$first_print(&text('backup_fordomain', $d->{'dom'}));
	&$second_print();
	&$indent_print();
	local $f;
	local $dok = 1;
	foreach $f (@{$_[2]}) {
		local $bfunc = "backup_$f";
		if (defined(&$bfunc) &&
		    ($d->{$f} || $f eq "virtualmin")) {
			local $ffile = "$backupdir/$d->{'dom'}_$f";
			local $fok = &$bfunc($d, $ffile, $_[5]->{$f});
			$dok = 0 if (!$fok);
			if (!$fok && !$_[4]) {
				$ok = 0;
				$errcount++;
				last DOMAIN;
				}
			push(@donedoms, $d);
			}
		}
	&$outdent_print();
	if ($dok) { $okcount++; }
	else { $errcount++; }
	}

# Work out where to write the final tar files to
local ($mode, $user, $pass, $server, $path) = &parse_backup_url($_[0]);
local ($dest, @destfiles);
if ($mode >= 1) {
	# Write archive to temporary file/dir first, for later upload
	$dest = &tempname();
	}
else {
	$dest = $path;
	}

if ($ok) {
	local $out;
	if ($_[3]) {
		# Create one tar file in the destination for each domain
		&$first_print($text{'backup_final2'});
		mkdir($dest, 0755);
		foreach $d (&unique(@donedoms)) {
			if (&has_command("gzip")) {
				$out = `cd $backupdir ; (tar cf - $d->{'dom'}_* | gzip -c) 2>&1 >$dest/$d->{'dom'}.tar.gz`;
				push(@destfiles, "$d->{'dom'}.tar.gz");
				}
			else {
				$out = `cd $backupdir ; tar cf $dest/$d->{'dom'}.tar $d->{'dom'}_* 2>&1`;
				push(@destfiles, "$d->{'dom'}.tar");
				}
			if ($?) {
				&$second_print(&text('backup_finalfailed', "<pre>$out</pre>"));
				$ok = 0;
				last;
				}
			}
		&$second_print($text{'setup_done'}) if ($ok);
		}
	else {
		# Tar up the directory into the final file
		&$first_print($text{'backup_final'});
		if (&has_command("gzip")) {
			$out = `cd $backupdir ; (tar cf - . | gzip -c) 2>&1 >$dest`;
			}
		else {
			$out = `cd $backupdir ; tar cf $dest . 2>&1`;
			}
		if ($?) {
			&$second_print(&text('backup_finalfailed', "<pre>$out</pre>"));
			$ok = 0;
			}
		else {
			&$second_print($text{'setup_done'});
			}
		}
	}

system("rm -rf ".quotemeta($backupdir));
local $sz = $_[3] ? &disk_usage_kb($dest)*1024
		  : (@st=stat($dest))[7];

if ($ok && $mode == 1) {
	# Upload file(s) to FTP server
	&$first_print($text{'backup_upload'});
	local $err;
	if ($_[3]) {
		# Need to upload entire directory .. which has to be created
		local $mkdirerr;
		&ftp_onecommand($server, "MKD $path", \$mkdirerr, $user, $pass);
		foreach $df (@destfiles) {
			&ftp_upload($server, "$path/$df", "$dest/$df", \$err,
				    undef, $user, $pass);
			if ($err) {
				&$second_print(
					&text('backup_uploadfailed', $err));
				$ok = 0;
				last;
				}
			}
		}
	else {
		# Just a single file
		&ftp_upload($server, $path, $dest, \$err, undef, $user, $pass);
		if ($err) {
			&$second_print(&text('backup_uploadfailed', $err));
			$ok = 0;
			}
		}
	&$second_print($text{'setup_done'}) if ($ok);
	}
elsif ($ok && $mode == 2) {
	# Upload to SSH server with scp
	&$first_print($text{'backup_upload2'});
	local $err;
	local $r = ($user ? "$user\@" : "")."$server:$path";
	if ($_[3]) {
		# Need to upload entire directory
		&scp_copy("$dest/*", $r, $pass, \$err);
		if ($err) {
			$err = undef;
			&scp_copy($dest, $r, $pass, \$err);
			}
		}
	else {
		# Just a single file
		&scp_copy($dest, $r, $pass, \$err);
		}
	if ($err) {
		&$second_print(&text('backup_uploadfailed', $err));
		$ok = 0;
		}
	&$second_print($text{'setup_done'}) if ($ok);
	}

if ($mode >= 1) {
	# Always delete the temporary destination
	system("rm -rf ".quotemeta($dest));
	}

# Show some status
if ($ok) {
	&$first_print(&text('backup_finalstatus', $okcount, $errcount));
	}

return ($ok, $sz);
}

# backup_virtualmin(&domain, file)
# Adds a domain's configuration file to the backup
sub backup_virtualmin
{
&$first_print($text{'backup_virtualmincp'});
system("cp ".quotemeta($_[0]->{'file'})." ".$_[1]);
&$second_print($text{'setup_done'});
return 1;
}

# restore_domains(file, &domains, &features, &options)
# Restore multiple domains from the given file
sub restore_domains
{
# Work out where the backup is located
local $ok = 1;
local $backup;
local ($mode, $user, $pass, $server, $path) = &parse_backup_url($_[0]);
if ($mode > 0) {
	# Need to download to temp file/directory first
	&$first_print($text{'restore_download'});
	$backup = &tempname();
	local $derr = &download_backup($_[0], $backup);
	if ($derr) {
		&$second_print(&text('restore_downloadfailed', $derr));
		$ok = 0;
		}
	else {
		&$second_print($text{'setup_done'});
		}
	}
else {
	$backup = $_[0];
	}

local $restoredir;
if ($ok) {
	# Create a temp dir for the backup archive contents
	$restoredir = &tempname();
	mkdir($restoredir, 0700);

	local @files;
	if (-d $backup) {
		# Extracting a directory of backup files
		&$first_print($text{'restore_first2'});
		opendir(DIR, $backup);
		@files = map { "$backup/$_" }
			     grep { $_ ne "." && $_ ne ".." } readdir(DIR);
		closedir(DIR);
		}
	else {
		# Extracting one backup file
		&$first_print($text{'restore_first'});
		@files = ( $backup );
		}

	# Extract each of the files
	local $f;
	foreach $f (@files) {
		open(BACKUP, $f);
		local $two;
		read(BACKUP, $two, 2);
		close(BACKUP);
		local $out;
		local $q = quotemeta($f);
		if ($two eq "\037\213") {
			# Assume gzipped tar
			$out = `cd '$restoredir' ; (gunzip -c $q | tar xf -) 2>&1`;
			}
		else {
			# Assume normal tar
			$out = `cd '$restoredir' ; tar xf $q 2>&1`;
			}
		if ($?) {
			&$second_print(&text('restore_firstfailed', "<tt>$f</tt>", "<pre>$out</pre>"));
			$ok = 0;
			last;
			}
		}
	&$second_print($text{'setup_done'}) if ($ok);
	}

if ($ok) {
	# Fill in missing domain details
	foreach $d (@{$_[1]}) {
		if ($d->{'missing'}) {
			$d = &get_domain(undef,
				"$restoredir/$d->{'dom'}_virtualmin");
			$d->{'missing'} = 1;
			}
		}

	# Now restore each of the domain/feature files
	local $d;
	DOMAIN: foreach $d (sort { $a->{'parent'} <=> $b->{'parent'} } @{$_[1]}) {
		if ($d->{'missing'}) {
			# This domain doesn't exist yet - need to re-create it
			&$first_print(&text('restore_createdomain',
				      $d->{'dom'}));
			local $cerr = &virtual_server_clashes($d);
			if ($cerr) {
				&$second_print(&text('restore_eclash', $cerr));
				$ok = 0;
				last DOMAIN;
				}
			local ($parentdom, $parentuser);
			if ($d->{'parent'}) {
				# Does the parent exist?
				$parentdom = &get_domain($d->{'parent'});
				if (!$parentdom) {
					&$second_print($text{'restore_epar'});
					$ok = 0;
					last DOMAIN;
					}
				$parentuser = $parentdom->{'user'};
				}
			# Create the domain, fixing the IP if needed
			&$indent_print();
			delete($d->{'missing'});
			if ($d->{'alias'}) {
				local $alias = &get_domain($d->{'alias'});
				$d->{'ip'} = $alias->{'ip'};
				}
			elsif (!$d->{'virt'} && !$config{'all_namevirtual'}) {
				$d->{'ip'} = &get_default_ip();
				}
			&create_virtual_server($d);
			&$outdent_print();
			}

		&$first_print(&text('restore_fordomain', $d->{'dom'}));
		&$indent_print();
		local $f;
		foreach $f (@{$_[2]}) {
			# Restore features
			local $rfunc = "restore_$f";
			if (defined(&$rfunc) &&
			    ($d->{$f} || $f eq "virtualmin")) {
				local $ffile = "$restoredir/$d->{'dom'}_$f";
				if (-r $ffile) {
					local $fok = &$rfunc($d, $ffile,
							     $_[3]->{$f});
					if (!$fok) {
						$ok = 0;
						&$outdent_print();
						last DOMAIN;
						}
					}
				}
			}
		&save_domain($d);

		# Re-setup Webmin user
		&modify_webmin($d, $d);
		&$outdent_print();
		}
	}

system("rm -rf ".quotemeta($restoredir));
if ($mode > 0) {
	# Clean up downloaded file
	system("rm -rf ".quotemeta($backup));
	}
return $ok;
}

# backup_contents(file)
# Returns a hash ref of domains and features in a backup file, or an error
# string if it is invalid
sub backup_contents
{
local $backup;
local ($mode, $user, $pass, $server, $path) = &parse_backup_url($_[0]);
if ($mode > 0) {
	# Need to download to temp file first
	$backup = &tempname();
	local $derr = &download_backup($_[0], $backup);
	return $derr if ($derr);
	}
else {
	$backup = $_[0];
	}

if (-d $backup) {
	# A directory of backup files, one per domain
	opendir(DIR, $backup);
	local $f;
	local %rv;
	foreach $f (readdir(DIR)) {
		next if ($f eq "." || $f eq "..");
		local $cont = &backup_contents("$backup/$f");
		if (ref($cont)) {
			local $d;
			foreach $d (keys %$cont) {
				if ($rv{$d}) {
					&clean_contents_temp();
					return &text('restore_edup', $d);
					}
				else {
					$rv{$d} = $cont->{$d};
					}
				}
			}
		else {
			&clean_contents_temp();
			return $backup."/".$f." : ".$cont;
			}
		}
	closedir(DIR);
	&clean_contents_temp();
	return \%rv;
	}
else {
	# A single file
	local $err;
	open(BACKUP, $backup);
	local $two;
	read(BACKUP, $two, 2);
	close(BACKUP);
	local $out;
	local $q = quotemeta($backup);
	if ($two eq "\037\213") {
		# Assume gzipped tar
		$out = `(gunzip -c $q | tar tf -) 2>&1`;
		}
	else {
		# Assume normal tar
		$out = `tar tf $q 2>&1`;
		}
	if ($?) {
		&clean_contents_temp();
		return $text{'restore_etar'};
		}
	local ($l, %rv, %done);
	foreach $l (split(/\n/, $out)) {
		if ($l =~ /^(.\/)?([^_]+)_([a-z0-9]+)$/) {
			push(@{$rv{$2}}, $3) if (!$done{$2,$3}++);
			}
		}
	&clean_contents_temp();
	return \%rv;
	}

	sub clean_contents_temp
	{
	system("rm -rf ".quotemeta($backup)) if ($mode > 0);
	}
}

# download_backup(url, tempfile)
# Downloads a backup file or directory to a local temp file or directory.
# Returns undef on success, or an error message.
sub download_backup
{
local ($mode, $user, $pass, $server, $path) = &parse_backup_url($_[0]);
if ($mode == 1) {
	# Download from FTP server
	local $cwderr;
	local $isdir = &ftp_onecommand($server, "CWD $path", \$cwderr,
				       $user, $pass);
	local $err;
	if ($isdir) {
		# Need to download entire directory
		mkdir($_[1], 0700);
		local $list = &ftp_listdir($server, $path, \$err, $user, $pass);
		return $err if (!$list);
		foreach $f (@$list) {
			$f =~ s/^$path[\\\/]//;
			&ftp_download($server, "$path/$f", "$_[1]/$f", \$err,
				      undef, $user, $pass);
			return $err if ($err);
			}
		return undef;
		}
	else {
		# Can just download a single file
		&ftp_download($server, $path, $_[1], \$err,
			      undef, $user, $pass);
		return $err;
		}
	}
elsif ($mode == 2) {
	# Download from SSH server
	&scp_copy(($user ? "$user\@" : "")."$server:$path",
		  $_[1], $pass, \$err);
	return $err;
	}
}

# restore_virtualmin(&domain, file)
# Restore the settings for a domain, such as quota, password and so on
sub restore_virtualmin
{
&$first_print($text{'restore_virtualmincp'});
local %oldd;
&read_file($_[1], \%oldd);
$_[0]->{'quota'} = $oldd{'quota'};
$_[0]->{'uquota'} = $oldd{'uquota'};
$_[0]->{'pass'} = $oldd{'pass'};
$_[0]->{'email'} = $oldd{'email'};
$_[0]->{'mailboxlimit'} = $oldd{'mailboxlimit'};
$_[0]->{'owner'} = $oldd{'owner'};
&save_domain($_[0]);

&$second_print($text{'setup_done'});
return 1;
}

# backup_strftime(path)
# Replaces stftime-style % codes in a path with the current time
sub backup_strftime
{
eval "use POSIX";
eval "use posix" if ($@);
local @tm = localtime(time());
return strftime($_[0], @tm);
}

# parse_backup_url(string)
# Converts a URL like ftp:// or a filename into its components
sub parse_backup_url
{
if ($_[0] =~ /^ftp:\/\/([^:]*):([^\@]*)\@([^\/]+)(\/.*)$/) {
	return (1, $1, $2, $3, $4);
	}
elsif ($_[0] =~ /^ssh:\/\/([^:]*):([^\@]*)\@([^\/]+)(\/.*)$/) {
	return (2, $1, $2, $3, $4);
	}
else {
	return (0, undef, undef, undef, $_[0]);
	}
}

# show_backup_destination(name, value)
# Returns HTML for a field for selecting a local or FTP file
sub show_backup_destination
{
local ($mode, $user, $pass, $server, $path) = &parse_backup_url($_[1]);
local $rv;

# Local file field
$rv .= "<table cellpadding=1 cellspacing=0>";
$rv .= sprintf "<tr> <td><input type=radio name=$_[0]_mode value=0 %s></td>\n",
	$mode == 0 ? "checked" : "";
$rv .= sprintf "<td colspan=2>%s <input name=$_[0]_file size=40 value='%s'> %s</td> </tr>\n",
	$text{'backup_mode0'}, $mode == 0 ? $path : "",
	&file_chooser_button("$_[0]_file");

# FTP file fields
$rv .= sprintf "<tr> <td><input type=radio name=$_[0]_mode value=1 %s></td>\n",
	$mode == 1 ? "checked" : "";
$rv .= sprintf "<td>%s <input name=$_[0]_server size=20 value='%s'></td>\n",
	$text{'backup_mode1'}, $mode == 1 ? $server : undef;
$rv .= sprintf "<td>%s <input name=$_[0]_path size=20 value='%s'></td> </tr>\n",
	$text{'backup_path'}, $mode == 1 ? $path : undef;
$rv .= "<tr> <td></td>\n";
$rv .= sprintf "<td>%s <input name=$_[0]_user size=15 value='%s'></td>\n",
	$text{'backup_login'}, $mode == 1 ? $user : undef;
$rv .= sprintf "<td>%s <input name=$_[0]_pass size=15 value='%s'></td> </tr>\n",
	$text{'backup_pass'}, $mode == 1 ? $pass : undef;

# SCP file fields
$rv .= sprintf "<tr> <td><input type=radio name=$_[0]_mode value=2 %s></td>\n",
	$mode == 2 ? "checked" : "";
$rv .= sprintf "<td>%s <input name=$_[0]_sserver size=20 value='%s'></td>\n",
	$text{'backup_mode2'}, $mode == 2 ? $server : undef;
$rv .= sprintf "<td>%s <input name=$_[0]_spath size=20 value='%s'></td> </tr>\n",
	$text{'backup_path'}, $mode == 2 ? $path : undef;
$rv .= "<tr> <td></td>\n";
$rv .= sprintf "<td>%s <input name=$_[0]_suser size=15 value='%s'></td>\n",
	$text{'backup_login'}, $mode == 2 ? $user : undef;
$rv .= sprintf "<td>%s <input name=$_[0]_spass size=15 value='%s'></td> </tr>\n",
	$text{'backup_pass'}, $mode == 2 ? $pass : undef;

$rv .= "</table>\n";
return $rv;
}

# parse_backup_destination(name, &in)
# Returns a backup destination string, or calls error
sub parse_backup_destination
{
local %in = %{$_[1]};
local $mode = $in{"$_[0]_mode"};
if ($mode == 0) {
	$in{"$_[0]_file"} =~ /^\/\S/ || &error($text{'backup_edest'});
	return $in{"$_[0]_file"};
	}
elsif ($mode == 1) {
	gethostbyname($in{"$_[0]_server"}) || &error($text{'backup_eserver1'});
	$in{"$_[0]_path"} =~ /^\/\S/ || &error($text{'backup_epath'});
	$in{"$_[0]_user"} =~ /^[^:\@\/]*$/ || &error($text{'backup_euser'});
	$in{"$_[0]_pass"} =~ /^[^:\@\/]*$/ || &error($text{'backup_epass'});
	return "ftp://".$in{"$_[0]_user"}.":".$in{"$_[0]_pass"}."\@".
	       $in{"$_[0]_server"}.$in{"$_[0]_path"};
	}
elsif ($mode == 2) {
	gethostbyname($in{"$_[0]_sserver"}) || &error($text{'backup_eserver2'});
	$in{"$_[0]_spath"} =~ /^\/\S/ || &error($text{'backup_epath'});
	$in{"$_[0]_suser"} =~ /^[^:\@\/]*$/ || &error($text{'backup_euser'});
	return "ssh://".$in{"$_[0]_suser"}.":".$in{"$_[0]_spass"}."\@".
	       $in{"$_[0]_sserver"}.$in{"$_[0]_spath"};
	}
}

# ftp_upload(host, file, srcfile, [&error], [&callback], [user, pass])
# Download data from a local file to an FTP site
sub ftp_upload
{
local($buf, @n);
local $cbfunc = $_[4];

$download_timed_out = undef;
local $SIG{ALRM} = "download_timeout";
alarm(60);

# connect to host and login
&open_socket($_[0], 21, "SOCK", $_[3]) || return 0;
alarm(0);
if ($download_timed_out) {
	if ($_[3]) { ${$_[3]} = $download_timed_out; return 0; }
	else { &error($download_timed_out); }
	}
&ftp_command("", 2, $_[3]) || return 0;
if ($_[5]) {
	# Login as supplied user
	local @urv = &ftp_command("USER $_[5]", [ 2, 3 ], $_[3]);
	@urv || return 0;
	if (int($urv[1]/100) == 3) {
		&ftp_command("PASS $_[6]", 2, $_[3]) || return 0;
		}
	}
else {
	# Login as anonymous
	local @urv = &ftp_command("USER anonymous", [ 2, 3 ], $_[3]);
	@urv || return 0;
	if (int($urv[1]/100) == 3) {
		&ftp_command("PASS root\@".&get_system_hostname(), 2,
			     $_[3]) || return 0;
		}
	}
&$cbfunc(1, 0) if ($cbfunc);

&ftp_command("TYPE I", 2, $_[3]) || return 0;

# get the file size and tell the callback
local @st = stat($_[2]);
if ($cbfunc) {
	&$cbfunc(2, $st[7]);
	}

# send the file
local $pasv = &ftp_command("PASV", 2, $_[3]);
defined($pasv) || return 0;
$pasv =~ /\(([0-9,]+)\)/;
@n = split(/,/ , $1);
&open_socket("$n[0].$n[1].$n[2].$n[3]", $n[4]*256 + $n[5], "CON", $_[3]) || return 0;
&ftp_command("STOR $_[1]", 1, $_[3]) || return 0;

# transfer data
local $got;
open(PFILE, $_[2]);
while(read(PFILE, $buf, 1024) > 0) {
	print CON $buf;
	$got += length($buf);
	&$cbfunc(3, $got) if ($cbfunc);
	}
close(PFILE);
close(CON);
if ($got != $st[7]) {
	if ($_[3]) { ${$_[3]} = "Upload incomplete"; return 0; }
	else { &error("Upload incomplete"); }
	}
&$cbfunc(4) if ($cbfunc);

# finish off..
&ftp_command("", 2, $_[3]) || return 0;
&ftp_command("QUIT", 2, $_[3]) || return 0;
close(SOCK);

return 1;
}

# ftp_onecommand(host, command, [&error], [user, pass])
# Executes one command on an FTP server, after logging in, and returns its
# exit status.
sub ftp_onecommand
{
local($buf, @n);

$download_timed_out = undef;
local $SIG{ALRM} = "download_timeout";
alarm(60);

# connect to host and login
&open_socket($_[0], 21, "SOCK", $_[2]) || return 0;
alarm(0);
if ($download_timed_out) {
	if ($_[2]) { ${$_[2]} = $download_timed_out; return 0; }
	else { &error($download_timed_out); }
	}
&ftp_command("", 2, $_[2]) || return 0;
if ($_[3]) {
	# Login as supplied user
	local @urv = &ftp_command("USER $_[3]", [ 2, 3 ], $_[2]);
	@urv || return 0;
	if (int($urv[1]/100) == 3) {
		&ftp_command("PASS $_[4]", 2, $_[2]) || return 0;
		}
	}
else {
	# Login as anonymous
	local @urv = &ftp_command("USER anonymous", [ 2, 3 ], $_[2]);
	@urv || return 0;
	if (int($urv[1]/100) == 3) {
		&ftp_command("PASS root\@".&get_system_hostname(), 2,
			     $_[2]) || return 0;
		}
	}

# make the directory
local @rv = &ftp_command($_[1], 2, $_[2]);
@rv || return 0;

# finish off..
&ftp_command("QUIT", 2, $_[3]) || return 0;
close(SOCK);

return $rv[1];
}

# ftp_listdir(host, dir, [&error], [user, pass])
# Returns a reference to a list of filenames in a directory
sub ftp_listdir
{
local($buf, @n);

$download_timed_out = undef;
local $SIG{ALRM} = "download_timeout";
alarm(60);

# connect to host and login
&open_socket($_[0], 21, "SOCK", $_[2]) || return 0;
alarm(0);
if ($download_timed_out) {
	if ($_[2]) { ${$_[2]} = $download_timed_out; return 0; }
	else { &error($download_timed_out); }
	}
&ftp_command("", 2, $_[2]) || return 0;
if ($_[3]) {
	# Login as supplied user
	local @urv = &ftp_command("USER $_[3]", [ 2, 3 ], $_[2]);
	@urv || return 0;
	if (int($urv[1]/100) == 3) {
		&ftp_command("PASS $_[4]", 2, $_[2]) || return 0;
		}
	}
else {
	# Login as anonymous
	local @urv = &ftp_command("USER anonymous", [ 2, 3 ], $_[2]);
	@urv || return 0;
	if (int($urv[1]/100) == 3) {
		&ftp_command("PASS root\@".&get_system_hostname(), 2,
			     $_[2]) || return 0;
		}
	}

# request the listing
local $pasv = &ftp_command("PASV", 2, $_[2]);
defined($pasv) || return 0;
$pasv =~ /\(([0-9,]+)\)/;
@n = split(/,/ , $1);
&open_socket("$n[0].$n[1].$n[2].$n[3]", $n[4]*256 + $n[5], "CON", $_[2]) || return 0;
&ftp_command("NLST $_[1]", 1, $_[2]) || return 0;

# transfer listing
local @list;
while(<CON>) {
	s/\r|\n//g;
	push(@list, $_);
	}
close(CON);

# finish off..
&ftp_command("", 2, $_[3]) || return 0;
&ftp_command("QUIT", 2, $_[3]) || return 0;
close(SOCK);

return \@list;
}

# scp_copy(source, dest, password, &error)
# Copies a file from some source to a destination. One or the other can be
# a server, like user@foo:/path/to/bar/
sub scp_copy
{
&foreign_require("proc", "proc-lib.pl");
local $cmd = "scp -r $_[0] $_[1]";
local ($fh, $fpid) = &proc::pty_process_exec($cmd);
local $out;
while(1) {
	local $rv = &wait_for($fh, "password:", "yes\\/no", ".*\n");
	$out .= $wait_for_input;
	if ($rv == 0) {
		syswrite($fh, "$_[2]\n");
		}
	elsif ($rv == 1) {
		syswrite($fh, "yes\n");
		}
	elsif ($rv < 0) {
		last;
		}
	}
close($fh);
local $got = waitpid($fpid, 0);
if ($? || $out =~ /permission\s+denied/i) {
	${$_[3]} = "scp failed : <pre>$out</pre>";
	}
}

# free_ip_address()
# Returns an IP address within the allocation range which is not currently used
sub free_ip_address
{
&foreign_require("net", "net-lib.pl");
local %taken = map { $_->{'address'}, $_ } (&net::boot_interfaces(),
					    &net::active_interfaces());
local @ranges = split(/\s+/, $config{'ip_ranges'});
local $r;
foreach $r (@ranges) {
	$r =~ /^(\d+\.\d+\.\d+)\.(\d+)\-(\d+)$/ || next;
	local ($base, $s, $e) = ($1, $2, $3);
	local $j;
	for($j=$s; $j<=$e; $j++) {
		local $try = "$base.$j";
		return $try if (!$taken{$try});
		}
	}
return undef;
}

# setup_for_subdomain(&parent-domain, subdomain-user, &sub-domain)
# Ensures that this virtual server can host sub-domains
sub setup_for_subdomain
{
&system_logged("mkdir '$_[0]->{'home'}/domains' 2>/dev/null");
&system_logged("chmod 755 '$_[0]->{'home'}/domains'");
&system_logged("chown $_[0]->{'uid'}:$_[0]->{'gid'} '$_[0]->{'home'}/domains'");
}

# count_domains(&parent)
# Returns the number of domains owned by this domain, and the max allowed
# for the current user. May exclude alias domains if they do not count towards
# the max.
sub count_domains
{
local $count = 1;
local @subs = &get_domain_by("parent", $_[0]->{'id'});
if ($config{'limitnoalias'}) {
	@subs = grep { !$_->{'alias'} } @subs;
	}
$count += @subs;
local $max;
if (&master_admin() || $_[0]->{'domslimit'} eq "*") {
	$max = 99999999;	# a huge number
	}
else {
	$max = $_[0]->{'domslimit'};
	}
return ($count, $max);
}

# count_mailboxes(&parent)
# Returns the number of mailboxes in this domain and all subdomains, and the
# max allowed for the current user
sub count_mailboxes
{
local $count = 0;
local $doms = 0;
local $parent = $_[0]->{'parent'} ? &get_domain($_[0]->{'parent'}) : $_[0];
local $d;
foreach $d ($parent, &get_domain_by("parent", $parent->{'id'})) {
	local @users = &list_domain_users($d, 0, 1, 1);
	$count += @users;
	$doms++;
	}
return ( $count, $parent->{'mailboxlimit'} ? $parent->{'mailboxlimit'} : 0,
	 $doms );
}

# database_name(&domain)
# Returns a suitable database name for a domain
sub database_name
{
local $tmpl = &get_template($_[0]->{'template'});
local $db = &substitute_template($tmpl->{'mysql'}, $_[0]);
$db ||= $_[0]->{'prefix'};
$db =~ s/[\.\-]/_/g;
return $db;
}

# unixuser_name(domainname)
# Returns a Unix username for soem domain, or undef if none can be found
sub unixuser_name
{
$_[0] =~ /^([^\.]+)/;
local ($try1, $user) = ($1, $1);
if (defined(getpwnam($try1)) || $config{'longname'}) {
	$user = $_[0];
	$try2 = $user;
	if (defined(getpwnam($try))) {
		return (undef, $try1, $try2);
		&error(&text('setup_eauto', $try1, $try2));
		}
	}
return ($user);
}

# virtual_server_clashes(&dom, [&features-to-check])
# Returns a clash error message if any were found for some new domain
sub virtual_server_clashes
{
local ($dom, $check) = @_;
my $f;
foreach $f (@features) {
	if ($dom->{$f} && (!$check || $check->{$f})) {
		local $cfunc = "check_${f}_clash";
		if (&$cfunc($dom)) {
			return &text('setup_e'.$f, $dom->{'dom'}, $dom->{'db'},
				     $dom->{'user'}, $dom->{'group'});
			}
		}
	}
foreach $f (@feature_plugins) {
	if ($dom->{$f} && (!$check || $check->{$f})) {
		local $cerr = &plugin_call($f, "feature_clash", $dom);
		return $cerr if ($cerr);
		}
	}
return undef;
}

# virtual_server_depends(&dom)
# Returns an error message if any of the features in the domain depend on
# missing features
sub virtual_server_depends
{
local $f;
foreach $f (grep { $_[0]->{$_} } @features) {
	local $dfunc = "check_depends_$f";
	if (defined(&$dfunc)) {
		# Call dependecy function
		local $derr = &$dfunc($_[0]);
		return $derr if ($derr);
		}
	else {
		# Check fixed dependency list
		local $fd;
		foreach $fd (@{$feature_depends{$f}}) {
			return &text('setup_edep'.$f) if (!$_[0]->{$fd});
			}
		}
	}
foreach $f (grep { $_[0]->{$_} } @feature_plugins) {
	local $derr = &plugin_call($f, "feature_depends", $_[0]);
	return $derr if ($derr);
	}
return undef;
}

# create_virtual_server(&domain, [&parent-domain], [parent-user])
# Given a complete domain object, setup all it's features
sub create_virtual_server
{
local ($dom, $parentdom, $parentuser) = @_;

# Run the before command
&set_domain_envs($_[0], "CREATE_DOMAIN");
local $merr = &making_changes();
return &text('setup_emaking', "<tt>$merr</tt>") if (defined($merr));

# Get ready for hosting a subdomain
if ($dom->{'parent'}) {
	&setup_for_subdomain($parentdom, $parentuser, $dom);
	}

# Set up all the selected features
my $f;
local %vital = map { $_, 1 } @vital_features;
foreach $f (@features) {
	if ($dom->{$f}) {
		local $sfunc = "setup_$f";
		if ($vital{$f}) {
			# Failure of this feature should halt the entire setup
			&$sfunc($dom);
			}
		else {
			# Failure can be ignored
			local $main::error_must_die = 1;
			eval { &$sfunc($dom) };
			if ($@) {
				&$second_print(&text('setup_failure',
					$text{'feature_'.$f}, $@));
				$dom->{$f} = 0;
				}
			}
		}
	}

# Set up all the selected plugins
foreach $f (@feature_plugins) {
	if ($dom->{$f}) {
		# Failure can be ignored
		local $main::error_must_die = 1;
		eval { &plugin_call($f, "feature_setup", $dom) };
		if ($@) {
			&$second_print(&text('setup_failure',
				&plugin_call($f, "feature_name"), $@));
			$dom->{$f} = 0;
			}
		}
	}

&run_post_actions();

# Add virtual IP address, if needed
if ($dom->{'virt'}) {
	&setup_virt($dom);
	}

# Add a virtuser for the unix user, if requested
if ($in{'mailbox'}) {
	&$first_print($text{'setup_mailbox'});
	local $virt = { 'from' => $user."\@".$dom->{'dom'},
			'to' => [ $user ] };
	&create_virtuser($virt);
	&$second_print($text{'setup_done'});
	}

# Save domain details
&$first_print($text{'setup_save'});
&save_domain($dom);
&$second_print($text{'setup_done'});

if (!$dom->{'alias'}) {
	# Notify the owner via email
	&send_domain_email($dom);
	}

# Update the parent domain Webmin user
if ($parentdom) {
	&modify_webmin($parentdom, $parentdom);
	}

if ($remote_user) {
	# Add to this user's list of domains if needed
	local %access = &get_module_acl();
	if (!&can_edit_domain($dom)) {
		$access{'domains'} = join(" ", split(/\s+/, $access{'domains'}),
					       $dom->{'id'});
		&save_module_acl(\%access);
		}
	}

# Run the after creation command
&made_changes();

return undef;
}

# register_post_action(&function, args)
sub register_post_action
{
push(@main::post_actions, [ @_ ]);
}

# run_post_actions()
# Run all registered post-modification actions
sub run_post_actions
{
local $a;
local %done;
foreach $a (@main::post_actions) {
	next if ($done{join(",", @$a)}++);
	local ($afunc, @aargs) = @$a;
	local $main::error_must_die = 1;
	eval { &$afunc(@aargs) };
	if ($@) {
		&$second_print(&text('setup_postfailure', $@));
		}
	}
@post_actions = ( );
}

# find_bandwidth_job()
# Returns the cron job used for bandwidth monitoring
sub find_bandwidth_job
{
&foreign_require("cron", "cron-lib.pl");
local @jobs = &cron::list_cron_jobs();
local ($job) = grep { $_->{'user'} eq 'root' &&
		      $_->{'command'} eq $bw_cron_cmd } @jobs;
return $job;
}

$bandwidth_dir = "$module_config_directory/bandwidth";

# get_bandwidth(&domain)
# Returns the bandwidth usage object for some domain
sub get_bandwidth
{
if (!defined($get_bandwidth_cache{$_[0]->{'id'}})) {
	local %bwinfo;
	&read_file("$bandwidth_dir/$_[0]->{'id'}", \%bwinfo);
	$get_bandwidth_cache{$_[0]->{'id'}} = \%bwinfo;
	}
return $get_bandwidth_cache{$_[0]->{'id'}};
}

# save_bandwidth(&domain, &info)
sub save_bandwidth
{
mkdir($bandwidth_dir, 0700);
&write_file("$bandwidth_dir/$_[0]->{'id'}", $_[1]);
$get_bandwidth_cache{$_[0]->{'id'}} ||= $_[1];
}

# bandwidth_input(name, value)
# Returns HTML for a bandwidth input field, with an 'unlimited' option
sub bandwidth_input
{
local $rv;
$rv .= sprintf "<input type=radio name=$_[0]_def value=1 %s> %s\n",
	$_[1] ? "" : "checked", $text{'edit_bwnone'};
$rv .= sprintf "<input type=radio name=$_[0]_def value=0 %s>\n",
	$_[1] ? "checked" : "";
local ($val, $u);
if ($_[1] && $_[1]%(1024*1024*1024) == 0) {
	$val = $_[1]/(1024*1024*1024);
	$u = "GB";
	}
elsif ($_[1] && $_[1]%(1024*1024) == 0) {
	$val = $_[1]/(1024*1024);
	$u = "MB";
	}
elsif ($_[1] && $_[1]%(1024) == 0) {
	$val = $_[1]/(1024);
	$u = "kB";
	}
else {
	$val = $_[1];
	$u = "bytes";
	}
local $sel = "<select name=$_[0]_units>";
local $t;
foreach $t ("GB", "MB", "kB", "bytes") {
	$sel .= sprintf "<option %s>%s\n",
		$t eq $u ? "selected" : "", $t;
	}
$sel .= "</select>\n";
$rv .= &text('edit_bwpast_'.$config{'bw_past'},
	     "<input name=$_[0] size=10 value='$val'> $sel",
	     $config{'bw_period'});
return $rv;
}

# parse_bandwidth(name, error)
sub parse_bandwidth
{
if ($in{"$_[0]_def"}) {
	return undef;
	}
else {
	$in{$_[0]} =~ /^\d+$/ && $in{$_[0]} > 0 || &error($_[1]);
	local $m = $in{"$_[0]_units"} eq "GB" ? 1024*1024*1024 :
		   $in{"$_[0]_units"} eq "MB" ? 1024*1024 :
		   $in{"$_[0]_units"} eq "kB" ? 1024 : 1;
	return $in{$_[0]} * $m;
	}
}

# email_template_input(template-file, subject, other-cc)
# Returns HTML for fields for editing an email template
sub email_template_input
{
local $rv;
$rv .= "<table>\n";
$rv .= "<tr> <td><b>$text{'newdom_subject'}</b></td>\n";
$rv .= sprintf "<td><input name=subject value='%s' size=60></td> </tr>\n",
	$_[1];
$rv .= "<tr> <td><b>$text{'newdom_cc'}</b></td>\n";
$rv .= sprintf "<td><input name=cc value='%s' size=60></td> </tr>\n",
	$_[2];
$rv .= "</table>\n";
if ($_[0]) {
	$rv .= "<textarea name=template rows=20 cols=70>";
	open(FILE, $_[0]);
	while(<FILE>) {
		$rv .= &html_escape($_);
		}
	close(FILE);
	$rv .= "</textarea>\n";
	}
return $rv;
}

# parse_email_template(file, subject-config, cc-config)
sub parse_email_template
{
$in{'template'} =~ s/\r//g;
&lock_file($_[0]);
open(FILE, ">$_[0]") || &error(&text('efilewrite', $file, $!));
print FILE $in{'template'};
close(FILE);
&unlock_file($_[0]);

&lock_file($module_config_file);
$config{$_[1]} = $in{'subject'};
$config{$_[2]} = $in{'cc'};
$config{'last_check'} = time()+1;	# no need for check.cgi to be run
&save_module_config();
&unlock_file($module_config_file);
}

# escape_user(username)
# Returns a Unix username with characters unsuitable for use in a mail
# destination (like @) escaped
sub escape_user
{
local $escuser = $_[0];
$escuser =~ s/\@/\\\@/g;
return $escuser;
}

# unescape_user(username)
# The reverse of escape_user
sub unescape_user
{
local $escuser = $_[0];
$escuser =~ s/\\\@/\@/g;
return $escuser;
}

# escape_alias(username)
# Converts a username into a suitable alias name
sub escape_alias
{
local $escuser = $_[0];
$escuser =~ s/\@/-/g;
return $escuser;
}

# dotqmail_file(&user)
sub dotqmail_file
{
return "$_[0]->{'home'}/.qmail";
}

# get_dotqmail(file)
sub get_dotqmail
{
$_[0] =~ /\.qmail(-(\S+))?$/;
local $alias = { 'file' => $_[0],
		 'name' => $2 };
open(AFILE, $_[0]) || return undef;
while(<AFILE>) {
	s/\r|\n//g;
	s/#.*$//g;
	if (/\S/) {
		push(@{$alias->{'values'}}, $_);
		}
	}
close(AFILE);
return $alias;
}

# save_dotqmail(&alias, file, username|aliasname)
sub save_dotqmail
{
if (@{$_[0]->{'values'}}) {
	open(AFILE, ">$_[1]");
	local $v;
	foreach $v (@{$_[0]->{'values'}}) {
		if ($v eq "\\$_[2]" || $v eq "\\NEWUSER") {
			# Delivery to this user means to his maildir
			print AFILE "./Maildir/\n";
			}
		else {
			print AFILE $v,"\n";
			}
		}
	close(AFILE);
	}
else {
	unlink($_[1]);
	}
}

$templates_dir = "$module_config_directory/templates";

# list_templates()
# Returns a list of all virtual server templates, including two defaults for
# top-level and sub-servers
sub list_templates
{
local @rv;
push(@rv, { 'id' => 0,
	    'name' => 'Default Settings',
	    'standard' => 1,
	    'default' => 1,
	    'web' => $config{'apache_config'},
	    'web_suexec' => $config{'suexec'},
	    'web_user' => $config{'web_user'},
	    'web_html_dir' => $config{'html_dir'},
	    'web_stats_dir' => $config{'stats_dir'},
	    'web_stats_hdir' => $config{'stats_hdir'},
	    'web_port' => $default_web_port,
	    'web_sslport' => $default_web_sslport,
	    'web_alias' => $config{'alias_mode'},
	    'dns' => $config{'bind_config'},
	    'dns_replace' => $config{'bind_replace'},
	    'dns_view' => $config{'dns_view'},
	    'ftp' => $config{'proftpd_config'},
	    'ftp_dir' => $config{'ftp_dir'},
	    'mail_on' => $config{'domain_template'},
	    'mail' => $config{'domain_template'} eq "none" ||
		      $config{'domain_template'} eq "default" ?
				&cat_file("domain-template") :
				&cat_file($config{'domain_template'}),
	    'mail_subject' => $config{'newdom_subject'} ||
			      &entities_to_ascii($text{'mail_dsubject'}),
	    'mail_cc' => $config{'newdom_cc'},
	    'user_aliases' => $config{'newuser_aliases'} || "none",
	    'mysql' => $config{'mysql_db'} || '${PREFIX}',
	    'mysql_wild' => $config{'mysql_wild'},
	    'mysql_hosts' => $config{'mysql_hosts'} || "none",
	    'skel' => $config{'virtual_skel'} || "none",
	    'frame' => &cat_file("framefwd-template"),
	    'gacl' => 1,
	    'gacl_umode' => $config{'gacl_umode'},
	    'gacl_uusers' => $config{'gacl_uusers'},
	    'gacl_ugroups' => $config{'gacl_ugroups'},
	    'gacl_groups' => $config{'gacl_groups'},
	    'gacl_root' => $config{'gacl_root'},
	    'for_parent' => 1,
	    'for_sub' => 0,
	    'for_alias' => 1,
	    'for_users' => 1,
	  } );
push(@rv, { 'id' => 1,
	    'name' => 'Defaults Settings For Sub-Servers',
	    'standard' => 1,
	    'mail_on' => $config{'subdomain_template'} eq "none" ? 0 : 1,
	    'mail' => $config{'subdomain_template'} eq "none" ||
		      $config{'subdomain_template'} eq "default" ?
				&cat_file("subdomain-template") :
				&cat_file($config{'subdomain_template'}),
	    'mail_subject' => $config{'newsubdom_subject'} ||
			      &entities_to_ascii($text{'mail_dsubject'}),
	    'mail_cc' => $config{'newsubdom_cc'},
	    'skel' => $config{'sub_skel'} || "none",
	    'for_parent' => 0,
	    'for_sub' => 1,
	    'for_alias' => 0,
	    'for_users' => 1
	  } );
local $f;
opendir(DIR, $templates_dir);
while($f = readdir(DIR)) {
	if ($f ne "." && $f ne "..") {
		local %tmpl;
		&read_file("$templates_dir/$f", \%tmpl);
		$tmpl{'mail'} =~ s/\t/\n/g;
		if ($tmpl{'id'} == 1) {
			foreach $k (keys %tmpl) {
				$rv[1]->{$k} = $tmpl{$k}
					if (!defined($rv[1]->{$k}));
				}
			}
		else {
			push(@rv, \%tmpl);
			}
		}
	}
closedir(DIR);
return @rv;
}

# save_template(&template)
# Create or update a template. If saving the standard template, updates the
# appropriate config options instead of the template file.
sub save_template
{
local $save_config = 0;
if (!defined($_[0]->{'id'})) {
	$_[0]->{'id'} = &domain_id();
	}
if ($_[0]->{'id'} == 0) {
	# Update appropriate config entries
	$config{'apache_config'} = $_[0]->{'web'};
	$config{'suexec'} = $_[0]->{'web_suexec'};
	$config{'web_user'} = $_[0]->{'web_user'};
	$config{'html_dir'} = $_[0]->{'web_html_dir'};
	$config{'stats_dir'} = $_[0]->{'web_stats_dir'};
	$config{'stats_hdir'} = $_[0]->{'web_stats_hdir'};
	$config{'web_port'} = $_[0]->{'web_port'};
	$config{'web_sslport'} = $_[0]->{'web_sslport'};
	$config{'alias_mode'} = $_[0]->{'web_alias'};
	$config{'bind_config'} = $_[0]->{'dns'};
	$config{'bind_replace'} = $_[0]->{'dns_replace'};
	$config{'dns_view'} = $_[0]->{'dns_view'};
	delete($config{'mx_server'});
	$config{'proftpd_config'} = $_[0]->{'ftp'};
	$config{'ftp_dir'} = $_[0]->{'ftp_dir'};
	if ($_[0]->{'mail_on'} ne "none") {
		if ($config{'domain_template'} eq "none") {
			$config{'domain_template'} = "default";
			}
		}
	else {
		$config{'domain_template'} = "none";
		}
	&uncat_file($config{'domain_template'} eq "none" ||
		    $config{'domain_template'} eq "default" ?
			"domain-template" :
			$config{'domain_template'}, $_[0]->{'mail'});
	$config{'newdom_subject'} = $_[0]->{'mail_subject'};
	$config{'newdom_cc'} = $_[0]->{'mail_cc'};
	$config{'newuser_aliases'} = $_[0]->{'user_aliases'} eq "none" ?
					"" : $_[0]->{'user_aliases'};
	$config{'mysql_db'} = $_[0]->{'mysql'};
	$config{'mysql_wild'} = $_[0]->{'mysql_wild'};
	$config{'mysql_hosts'} = $_[0]->{'mysql_hosts'} eq "none" ?
					"" : $_[0]->{'mysql_hosts'};
	$config{'virtual_skel'} = $_[0]->{'skel'} eq "none" ? "" :
				  $_[0]->{'skel'};
	$config{'gacl_umode'} = $_[0]->{'gacl_umode'};
	$config{'gacl_ugroups'} = $_[0]->{'gacl_ugroups'};
	$config{'gacl_users'} = $_[0]->{'gacl_users'};
	$config{'gacl_groups'} = $_[0]->{'gacl_groups'};
	$config{'gacl_root'} = $_[0]->{'gacl_root'};
	&uncat_file("framefwd-template", $_[0]->{'frame'});
	$save_config = 1;
	}
elsif ($_[0]->{'id'} == 1) {
	# For the default for sub-servers, update mail and skel in config only
	if ($_[0]->{'mail_on'} ne "none") {
		if ($config{'subdomain_template'} eq "none") {
			$config{'subdomain_template'} = "default";
			}
		}
	else {
		$config{'subdomain_template'} = "none";
		}
	&uncat_file($config{'subdomain_template'} eq "none" ||
		    $config{'subdomain_template'} eq "default" ?
			"subdomain-template" :
			$config{'subdomain_template'}, $_[0]->{'mail'});
	$config{'newsubdom_subject'} = $_[0]->{'mail_subject'};
	$config{'newsubdom_cc'} = $_[0]->{'mail_cc'};
	$config{'sub_skel'} = $_[0]->{'skel'} eq "none" ? "" :
			      $_[0]->{'skel'};
	$save_config = 1;
	}
if ($_[0]->{'id'} != 0) {
	# Just save the file
	mkdir($templates_dir, 0700);
	$_[0]->{'created'} ||= time();
	$_[0]->{'mail'} =~ s/\n/\t/g;
	&lock_file("$templates_dir/$_[0]->{'id'}");
	&write_file("$templates_dir/$_[0]->{'id'}", $_[0]);
	&unlock_file("$templates_dir/$_[0]->{'id'}");
	}
if ($save_config) {
	&lock_file($module_config_file);
	$config{'last_check'} = time()+1;
	&write_file($module_config_file, \%config);
	&unlock_file($module_config_file);
	}
}

# get_template(id)
# Returns a template, with any default settings filled in from real default
sub get_template
{
local @tmpls = &list_templates();
local ($tmpl) = grep { $_->{'id'} == $_[0] } @tmpls;
if (!$tmpl->{'default'}) {
	local $def = $tmpls[0];
	$tmpl->{'skel'} = $def->{'skel'} if (!$tmpl->{'skel'});
	local $p;
	foreach $p ("web", "dns", "ftp", "mail", "frame", "user_aliases") {
		if (!$tmpl->{$p}) {
			local $k;
			foreach $k (keys %$def) {
				$tmpl->{$k} = $def->{$k} if ($k =~ /^$p/);
				}
			}
		}
	}
return $tmpl;
}

# delete_template(&template)
# Just mark this template as deleted, as it may be needed later
sub delete_template
{
local %tmpl;
&lock_file("$templates_dir/$_[0]->{'id'}");
&read_file("$templates_dir/$_[0]->{'id'}", \%tmpl);
$tmpl{'deleted'} = 1;
&write_file("$templates_dir/$_[0]->{'id'}", \%tmpl);
&unlock_file("$templates_dir/$_[0]->{'id'}");
}

# cat_file(file)
# Returns the contents of some file
sub cat_file
{
local $path = $_[0] =~ /^\// ? $_[0] : "$module_config_directory/$_[0]";
local $rv;
open(FILE, $path);
while(<FILE>) {
	$rv .= $_;
	}
close(FILE);
return $rv;
}

# uncat_file(file, data)
# Writes to some file
sub uncat_file
{
local $path = $_[0] =~ /^\// ? $_[0] : "$module_config_directory/$_[0]";
&lock_file($path);
open(FILE, ">$path");
print FILE $_[1];
close(FILE);
&unlock_file($path);
}

# plugin_call(module, function, [arg, ...])
# If some plugin function is defined, call it and return the result,
# otherwise return undef
sub plugin_call
{
local ($mod, $func, @args) = @_;
if (&plugin_defined($mod, $func)) {
	if ($main::module_name ne "virtual_server") {
		# Set up virtual_server package
		&foreign_require("virtual-server", "virtual-server-lib.pl");
		$virtual_server::first_print = $first_print;
		$virtual_server::second_print = $second_print;
		$virtual_server::indent_print = $indent_print;
		$virtual_server::outdent_print = $outdent_print;
		}
	return &foreign_call($mod, $func, @args);
	}
else {
	return wantarray ? ( ) : undef;
	}
}

# plugin_defined(module, function)
# Returns 1 if some function is defined in a plugin
sub plugin_defined
{
local $pkg = $_[0];
$pkg =~ s/[^A-Za-z0-9]/_/g;
local $func = "${pkg}::$_[1]";
return defined(&$func);
}

# database_feature()
# Returns 1 if any feature that uses a database is enabled
sub database_feature
{
return $config{'mysql'} || $config{'postgres'};
}

# list_custom_fields()
# Returns a list of structures containing custom field details
sub list_custom_fields
{
local @rv;
open(FIELDS, $custom_fields_file);
while(<FIELDS>) {
	s/\r|\n//g;
	local @a = split(/:/, $_, 4);
	push(@rv, { 'name' => $a[0],
		    'type' => $a[1],
		    'opts' => $a[2],
		    'desc' => $a[3] });

	}
close(FIELDS);
return @rv;
}

# save_custom_fields(&fields)
sub save_custom_fields
{
open(FIELDS, ">$custom_fields_file");
foreach $a (@{$_[0]}) {
	print FIELDS $a->{'name'},":",$a->{'type'},":",
		     $a->{'opts'},":",$a->{'desc'},"\n";
	}
close(FIELDS);
}

# show_custom_fields([&domain])
# Returns HTML for custom field inputs, for inclusion in a table
sub show_custom_fields
{
local $rv;
local $f;
local $col = 0;
foreach $f (&list_custom_fields()) {
	$rv .= "<tr>\n" if ($col%2 == 0);
	$rv .= "<td valign=top><b>$f->{'desc'}</b></td>\n";
	$rv .= "<td>";
	local $n = "field_".$f->{'name'};
	local $v = $_[0] ? $_[0]->{"field_".$f->{'name'}} : undef;
	if ($f->{'type'} == 0) {
		local $sz = $f->{'opts'} || 30;
		$rv .= &ui_textbox($n, $v, $sz);
		}
	elsif ($f->{'type'} == 1 || $f->{'type'} == 2) {
		$rv .= &ui_user_textbox($n, $v);
		}
	elsif ($f->{'type'} == 3 || $f->{'type'} == 4) {
		$rv .= &ui_group_textbox($n, $v);
		}
	elsif ($f->{'type'} == 5 || $f->{'type'} == 6) {
		$rv .= &ui_textbox($n, $v, 30)." ".
			&file_chooser_button($n, $f->{'type'}-5);
		}
	elsif ($f->{'type'} == 7) {
		$rv .= &ui_radio($n, $v ? 1 : 0, [ [ 1, $text{'yes'} ],
						   [ 0, $text{'no'} ] ]);
		}
	elsif ($f->{'type'} == 8) {
		local $sz = $f->{'opts'} || 30;
		$rv .= &ui_password($n, $v, $sz);
		}
	elsif ($f->{'type'} == 9) {
		local @opts = &read_opts_file($f->{'opts'});
		local ($found) = grep { $_->[0] eq $v } @opts;
		push(@opts, [ $v, $v ]) if (!$found);
		$rv .= &ui_select($n, $v, \@opts);
		}
	elsif ($f->{'type'} == 10) {
		local ($w, $h) = split(/\s+/, $f->{'opts'});
		$h ||= 4;
		$w ||= 30;
		$v =~ s/\t/\n/g;
		$rv .= &ui_textarea($n, $v, $h, $w);
		}
	$rv .= "</td>\n";
	$rv .= "</tr>\n" if ($col++%2 == 1);
	}
$rv .= "</tr>\n" if ($col++%2 == 1);
return $rv;
}

# parse_custom_fields(&domain, &in)
# Updates a domain with custom fields
sub parse_custom_fields
{
local $f;
local %in = %{$_[1]};
foreach $f (&list_custom_fields()) {
	local $n = "field_".$f->{'name'};
	local $rv;
	if ($f->{'type'} == 0 || $f->{'type'} == 5 ||
	    $f->{'type'} == 6 || $f->{'type'} == 8) {
		$rv = $in{$n};
		}
	elsif ($f->{'type'} == 10) {
		$rv = $in{$n};
		$rv =~ s/\r//g;
		$rv =~ s/\n/\t/g;
		}
	elsif ($f->{'type'} == 1 || $f->{'type'} == 2) {
		local @u = getpwnam($in{$n});
		$rv = $f->{'type'} == 1 ? $in{$n} : $u[2];
		}
	elsif ($f->{'type'} == 3 || $f->{'type'} == 4) {
		local @g = getgrnam($in{$n});
		$rv = $f->{'type'} == 3 ? $in{$n} : $g[2];
		}
	elsif ($f->{'type'} == 7) {
		$rv = $in{$n} ? $f->{'opts'} : "";
		}
	elsif ($f->{'type'} == 9) {
		$rv = $in{$n};
		}
	$_[0]->{"field_".$f->{'name'}} = $rv;
	}
}

# read_opts_file(file)
sub read_opts_file
{
local @rv;
local $file = $_[0];
if ($file !~ /^\//) {
	local @uinfo = getpwnam($remote_user);
	if (@uinfo) {
		$file = "$uinfo[7]/$file";
		}
	}
open(FILE, $file);
while(<FILE>) {
	s/\r|\n//g;
	if (/^"([^"]*)"\s+"([^"]*)"$/) {
		push(@rv, [ $1, $2 ]);
		}
	elsif (/^"([^"]*)"$/) {
		push(@rv, [ $1, $1 ]);
		}
	elsif (/^(\S+)\s+(\S.*)/) {
		push(@rv, [ $1, $2 ]);
		}
	else {
		push(@rv, [ $_, $_ ]);
		}
	}
close(FILE);
return @rv;
}

# connect_qmail_ldap([return-error])
# Connect to the LDAP server used for Qmail. Returns an LDAP handle on success,
# or an error message on failure.
sub connect_qmail_ldap
{
eval "use Net::LDAP";
if ($@) {
	local $err = &text('ldap_emod', "<tt>Net::LDAP</tt>");
	if ($_[0]) { return $err; }
	else { &error($err); }
	}

# Connect to server
local $port = $config{'ldap_port'} || 389;
local $ldap = Net::LDAP->new($config{'ldap_host'}, port => $port);
if (!$ldap) {
	local $err = &text('ldap_econn',
			   "<tt>$config{'ldap_host'}</tt>","<tt>$port</tt>");
	if ($_[0]) { return $err; }
	else { &error($err); }
	}

# Start TLS if configured
if ($config{'ldap_tls'}) {
	$ldap->start_tls();
	}

# Login
local $mesg;
if ($config{'ldap_login'}) {
	$mesg = $ldap->bind(dn => $config{'ldap_login'},
			    password => $config{'ldap_pass'});
	}
else {
	$mesg = $ldap->bind(anonymous => 1);
	}
if (!$mesg || $mesg->code) {
	local $err = &text('ldap_elogin', "<tt>$config{'ldap_host'}</tt>",
		     $dn, $mesg ? $mesg->error : "Unknown error");
	if ($_[0]) { return $err; }
	else { &error($err); }
	}
return $ldap;
}

# qmail_dn_to_hash(&ldap-object)
# Given a LDAP object containing user details, convert it to a hash
sub qmail_dn_to_hash
{
local $x;
local %oc = map { $_, 1 } $_[0]->get_value("objectClass");
local %user = ( 'dn' => $_[0]->dn(),
		'qmail' => 1,
		'user' => scalar($_[0]->get_value("uid")),
		'plainpass' => scalar($_[0]->get_value("cuserPassword")),
		'uid' => $oc{'posixAccount'} ?
			scalar($_[0]->get_value("uidNumber")) :
			scalar($_[0]->get_value("qmailUID")),
		'gid' => $oc{'posixAccount'} ?
			scalar($_[0]->get_value("gidNumber")) :
			scalar($_[0]->get_value("qmailGID")),
		'real' => scalar($_[0]->get_value("cn")),
		'shell' => scalar($_[0]->get_value("loginShell")),
		'home' => scalar($_[0]->get_value("homeDirectory")),
		'pass' => scalar($_[0]->get_value("userPassword")),
		'mailstore' => scalar($_[0]->get_value("mailMessageStore")),
		'qquota' => scalar($_[0]->get_value("mailQuotaSize")),
		'email' => scalar($_[0]->get_value("mail")),
		'extraemail' => [ $_[0]->get_value("mailAlternateAddress") ],
	      );
local @fwd = $_[0]->get_value("mailForwardingAddress");
if (@fwd) {
	$user{'to'} = \@fwd;
	}
$user{'pass'} =~ s/^{[a-z0-9]+}//i;
$user{'qmail'} = 1;
$user{'unix'} = 1 if ($oc{'posixAccount'});
$user{'person'} = 1 if ($oc{'person'} || $oc{'inetOrgPerson'} ||
			$oc{'posixAccount'});
return %user;
}

# qmail_user_to_dn(&user, &classes, &domain)
# Given a useradmin-style user hash, returns a list of properties to 
# add/update and to delete
sub qmail_user_to_dn
{
local $pfx = $_[0]->{'pass'} =~ /^\{[a-z0-9]+\}/i ? undef : "{crypt}";
local @ee = @{$_[0]->{'extraemail'}};
local @to = @{$_[0]->{'to'}};
local @delrv;
local @rv = (
	 "uid" => $_[0]->{'user'},
	 "qmailUID" => $_[0]->{'uid'},
	 "qmailGID" => $_[0]->{'gid'},
	 "homeDirectory" => $_[0]->{'home'},
	 "userPassword" => $pfx.$_[0]->{'pass'},
	 "mailMessageStore" => $_[0]->{'mailstore'},
	 "mailQuotaSize" => $_[0]->{'qquota'},
	 "mail" => $_[0]->{'email'},
	 "mailHost" => &get_system_hostname(),
	 "accountStatus" => "active",
	);
if (@ee) {
	push(@rv, "mailAlternateAddress" => \@ee );
	}
else {	
	push(@delrv, "mailAlternateAddress");
	}
if (@to) {
	push(@rv, "mailForwardingAddress" => \@to );
	push(@rv, "deliveryMode", "nolocal");
	}
else {	
	push(@delrv, "mailForwardingAddress");
	push(@rv, "deliveryMode", "noforward");
	}
if ($_[0]->{'unix'}) {
	push(@rv, "uidNumber" => $_[0]->{'uid'},
		  "gidNumber" => $_[0]->{'gid'},
		  "loginShell" => $_[0]->{'shell'});
	}
if ($_[0]->{'person'}) {
	push(@rv, "cn" => $_[0]->{'real'});
	}
if (&indexof("person", @{$_[1]}) >= 0 ||
    &indexof("inetOrgPerson", @{$_[1]}) >= 0) {
	# Have to set sn
	push(@rv, "sn" => $_[0]->{'user'});
	}
# Add extra attribs, which can override those set above
local %subs = %{$_[0]};
&userdom_substitutions(\%subs, $_[2]);
local @props = &split_props($config{'ldap_props'}, \%subs);
local @addprops;
local $i;
local %over;
for($i=0; $i<@props; $i+=2) {
	if ($props[$i+1] ne "") {
		push(@addprops, $props[$i], $props[$i+1]);
		}
	else {
		push(@delrv, $props[$i]);
		}
	$over{$props[$i]} = $props[$i+1];
	}
for($i=0; $i<@rv; $i+=2) {
	if (exists($over{$rv[$i]})) {
		splice(@rv, $i, 2);
		$i -= 2;
		}
	}
push(@rv, @addprops);
return wantarray ? ( \@rv, \@delrv ) : \@rv;
}

# split_props(text, &user)
# Splits up LDAP properties
sub split_props
{
local %pmap;
foreach $p (split(/\t+/, &substitute_template($_[0], $_[1]))) {
	if ($p =~ /^(\S+):\s*(.*)/) {
		push(@{$pmap{$1}}, $2);
		}
	}
local @rv;
local $k;
foreach $k (keys %pmap) {
	local $v = $pmap{$k};
	if (@$v == 1) {
		push(@rv, $k, $v->[0]);
		}
	else {
		push(@rv, $k, $v);
		}
	}
return @rv;
}

# ui_subheading(text, ...)
# Returns HTML for a section heading
sub ui_subheading
{
return &theme_ui_subheading(@_) if (defined(&theme_ui_subheading));
return "<h3>",join("", @_),"</h3>\n";
}

# ui_buttons_hr()
sub ui_buttons_hr
{
return &theme_ui_buttons_hr(@_) if (defined(&theme_ui_buttons_hr));
return "<tr> <td colspan=2><hr></td> </tr>\n";
}

# create_initial_user(&dom)
# Returns a structure for a new mailbox user
sub create_initial_user
{
local $user;
if ($config{'mail_system'} == 4) {
	# User is for Qmail+LDAP
	$user = { 'qmail' => 1,
		  'mailquota' => 1,
		  'person' => $config{'ldap_classes'} =~ /person|inetOrgPerson/ || $config{'ldap_unix'} ? 1 : 0,
		  'unix' => $config{'ldap_unix'} };
	}
elsif ($config{'mail_system'} == 5) {
	# VPOPMail user
	$user = { 'vpopmail' => 1,
		  'mailquota' => 1,
		  'person' => 1,
		  'fixedhome' => 1,
		  'noappend' => 1,
		  'noprimary' => 1 };
	}
else {
	# Normal unix user
	$user = { 'unix' => 1,
		  'person' => 1 };
	}
if ($_[0]) {
	local $tmpl = &get_template($_[0]->{'template'});
	if ($tmpl->{'user_aliases'} ne 'none') {
		$user->{'to'} = [ split(/\t+/, $tmpl->{'user_aliases'}) ];
		}
	}
return $user;
}

# valid_domain_name(&parent, newdomain)
# Returns an error message if some domain name is invalid, or undef if OK
sub valid_domain_name
{
if ($access{'forceunder'}) {
	local $pd = $_[0]->{'dom'};
	if ($_[1] !~ /\.\Q$pd\E$/i) {
		return &text('setup_eforceunder', $parentdom->{'dom'});
		}
	}
return undef;
}

# domain_databases(&domain)
# Returns a list of structures for databases in a domain
sub domain_databases
{
local @dbs;
foreach $db (split(/\s+/, $_[0]->{'db_mysql'})) {
	push(@dbs, { 'name' => $db,
		     'type' => 'mysql' });
	}
foreach $db (split(/\s+/, $_[0]->{'db_postgres'})) {
	push(@dbs, { 'name' => $db,
		     'type' => 'postgres' });
	}
return @dbs;
}

# all_databases()
# Returns a list of all known databases on the system
sub all_databases
{
local @rv;
if ($config{'mysql'}) {
	&require_mysql();
	push(@rv, map { { 'name' => $_,
			  'type' => 'mysql',
			  'special' => $_ eq "mysql" } }
		      &mysql::list_databases());
	}
if ($config{'postgres'}) {
	&require_postgres();
	push(@rv, map { { 'name' => $_,
			  'type' => 'postgres',
			  'special' => ($_ =~ /^template/i) } }
		      &postgresql::list_databases());
	}
return @rv;
}

1;

