/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "definitionbuilder.h"

#include <QStringList>

using namespace WebIssues;

DefinitionBuilder::DefinitionBuilder()
{
    m_types.insert( DefinitionInfo::Text, "TEXT" );
    m_types.insert( DefinitionInfo::Enum, "ENUM" );
    m_types.insert( DefinitionInfo::Numeric, "NUMERIC" );
    m_types.insert( DefinitionInfo::DateTime, "DATETIME" );
    m_types.insert( DefinitionInfo::User, "USER" );
}

DefinitionBuilder::~DefinitionBuilder()
{
}

QString DefinitionBuilder::build( const DefinitionInfo& info )
{
    QString definition = buildType( info.type() );

    if ( definition.isEmpty() )
        return QString();

    if ( info.required() )
        definition += " required=1";

    QString defaultValue = info.defaultValue();
    if ( !defaultValue.isEmpty() )
        definition += QString( " default=%1" ).arg( buildString( defaultValue ) );

    QStringList keys = info.metadataKeys();
    for ( QStringList::iterator it = keys.begin(); it != keys.end(); ++it ) {
        QVariant meta = info.metadata( *it );
        QString value;
        if ( meta.type() == QVariant::StringList )
            value = buildStringArray( meta.toStringList() );
        else if ( meta.type() == QVariant::String )
            value = buildString( meta.toString() );
        else
            value = meta.toString();
        definition += QString( " %1=%2" ).arg( *it, value );
    }

    return definition;
}

QString DefinitionBuilder::buildType( DefinitionInfo::Type type )
{
    QMap<DefinitionInfo::Type, QString>::iterator it = m_types.find( type );
    if ( it == m_types.end() )
        return QString();
    return *it;
}

QString DefinitionBuilder::buildStringArray( const QStringList& array )
{
    QString result = "{";
    for ( QStringList::const_iterator it = array.begin(); it != array.end(); ++it ) {
        if ( it != array.begin() )
            result += ",";
        result += buildString( *it );
    }
    result += "}";
    return result;
}

QString DefinitionBuilder::buildString( const QString& string )
{
    QString result = string;
    result.replace( '\\', "\\\\" );
    result.replace( '\"', "\\\"" );
    return QString( "\"%1\"" ).arg( result );
}
