
/* 
 * w3m menu.c
 */
#include <stdio.h>

#include "fm.h"
#include "menu.h"
#include "func.h"
#include "myctype.h"
#include "regex.h"

#ifdef USE_MENU

static int call_menu_func(int key, int c);

#ifdef KANJI_SYMBOLS

#ifdef MANY_CHARSET
extern char **FRAME;
#else
static char *FRAME[] =
{
#ifdef MENU_THIN_FRAME
    "", "", "",
    "", "  ", "",
    "", "", "",
#else				/* not MENU_THIN_FRAME */
    "", "", "",
    "", "  ", "",
    "", "", "",
#endif				/* not MENU_THIN_FRAME */
    "", ""};
#endif

#define G_start
/**/
#define G_end    /**/

#else				/* not KANJI_SYMBOLS */
static char *N_FRAME[] =
{
    "+", "-", "+",
    "|", " ", "|",
    "+", "-", "+",
    ":", ":"};

static char *G_FRAME[] =
{
    "l", "q", "k",
    "x", " ", "x",
    "m", "q", "j",
    ":", ":"};

static char **FRAME = NULL;
static int graph_mode = FALSE;

#define G_start  {if (graph_mode) graphstart();}
#define G_end    {if (graph_mode) graphend();}
#endif				/* not KANJI_SYMBOLS */

static KeyTabItem w3mDefaultMenuKeyTab[] = {
#ifdef __EMX__
  K_SET_FUNC(K_GEN(0, 0x00), FUNCNAME_mPc),
#endif
  K_SET_FUNC(K_GEN(0, 0x01), FUNCNAME_mTop),
  K_SET_FUNC(K_GEN(0, 0x02), FUNCNAME_mPrev),
  K_SET_FUNC(K_GEN(0, 0x03), FUNCNAME_mClose),
  K_SET_FUNC(K_GEN(0, 0x05), FUNCNAME_mLast),
  K_SET_FUNC(K_GEN(0, 0x06), FUNCNAME_mNext),
  K_SET_FUNC(K_GEN(0, 0x08), FUNCNAME_mCancel),
  K_SET_FUNC(K_GEN(0, 0x0A), FUNCNAME_mOk),
  K_SET_FUNC(K_GEN(0, 0x0D), FUNCNAME_mOk),
  K_SET_FUNC(K_GEN(0, 0x0E), FUNCNAME_mDown),
  K_SET_FUNC(K_GEN(0, 0x10), FUNCNAME_mUp),
  K_SET_FUNC(K_GEN(0, 0x16), FUNCNAME_mNext),
  K_SET_FUNC(K_GEN(0, 0x1A), FUNCNAME_mSusp),
  K_SET_FUNC(K_GEN(0, 0x1B), FUNCNAME_mEsc),
  K_SET_FUNC(K_GEN(0, 0x20), FUNCNAME_mOk),
  K_SET_FUNC(K_GEN(0, 0x2F), FUNCNAME_mSrchF),
  K_SET_FUNC(K_GEN(0, 0x3F), FUNCNAME_mSrchB),
  K_SET_FUNC(K_GEN(0, 0x4E), FUNCNAME_mSrchP),
  K_SET_FUNC(K_GEN(0, 0x68), FUNCNAME_mSrchP),
  K_SET_FUNC(K_GEN(0, 0x6A), FUNCNAME_mDown),
  K_SET_FUNC(K_GEN(0, 0x6B), FUNCNAME_mUp),
  K_SET_FUNC(K_GEN(0, 0x6C), FUNCNAME_mOk),
  K_SET_FUNC(K_GEN(0, 0x6E), FUNCNAME_mSrchN),
  K_SET_FUNC(K_GEN(0, 0x7F), FUNCNAME_mCancel),
  K_SET_FUNC(K_GEN(0, K_ESC | 0x4F), FUNCNAME_mEscB),
  K_SET_FUNC(K_GEN(0, K_ESC | 0x5B), FUNCNAME_mEscB),
  K_SET_FUNC(K_GEN(0, K_ESC | 0x76), FUNCNAME_mPrev),
  K_SET_FUNC(K_GEN(0, K_UP), FUNCNAME_mUp),
  K_SET_FUNC(K_GEN(0, K_DOWN), FUNCNAME_mDown),
  K_SET_FUNC(K_GEN(0, K_RIGHT), FUNCNAME_mOk),
  K_SET_FUNC(K_GEN(0, K_LEFT), FUNCNAME_mCancel),
  K_SET_FUNC(K_GEN(0, K_PocketBSD_INS), FUNCNAME_mClose),
  K_SET_FUNC(K_GEN(0, K_FreeBSD_CONSOLE_INS), FUNCNAME_mClose),
  K_SET_FUNC(K_GEN(0, K_XTERM_MOUSE), FUNCNAME_mMouse),
  K_SET_FUNC(K_GEN(0, K_PAD_INS), FUNCNAME_mClose),
  K_SET_FUNC(K_GEN(0, K_PAD_PGUP), FUNCNAME_mBack),
  K_SET_FUNC(K_GEN(0, K_PAD_PGDN), FUNCNAME_mFore),
  K_SET_FUNC(K_GEN(0, K_PAD_HELP), FUNCNAME_mClose),
#ifdef USE_MOUSE
  K_SET_FUNC(K_GEN(0, K_MOUSE(CLICK) + MOUSE_BTN1_DOWN), FUNCNAME_mPositional),
  K_SET_FUNC(K_GEN(0, K_MOUSE(CLICK) + MOUSE_BTN3_DOWN), FUNCNAME_mPositional),
  K_SET_FUNC(K_GEN(0, K_MOUSE(DCLICK) + MOUSE_BTN1_DOWN), FUNCNAME_mPositional),
  K_SET_FUNC(K_GEN(0, K_MOUSE(DCLICK) + MOUSE_BTN3_DOWN), FUNCNAME_mPositional),
#endif
};

KeyTabList w3mDefaultMenuKeyTabList = {
  NULL,
  w3mDefaultMenuKeyTab,
  sizeof(w3mDefaultMenuKeyTab) / sizeof(w3mDefaultMenuKeyTab[0]),
  sizeof(w3mDefaultMenuKeyTab) / sizeof(w3mDefaultMenuKeyTab[0]),
};

#ifdef __EMX__
static int (*MenuPcKeymap[256])(char c)={
//			  Null
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
//							  S-Tab
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-q	  A-w	  A-E	  A-r	  A-t	  A-y	  A-u	  A-i
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-o	  A-p	  A-[	  A-]			  A-a	  A-s
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-d	  A-f	  A-g	  A-h	  A-j	  A-k	  A-l	  A-;
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-'    A-'		  A-\		  A-x	  A-c	  A-v
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mPrev,
// A-b	  A-n	  A-m	  A-,	  A-.	  A-/		  A-+
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
//			  F1	  F2	  F3	  F4	  F5
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// F6	  F7	  F8	  F9	  F10			  Home
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mTop,
// Up	  PgUp	  A-/	  Left	  5	  Right	  C-*	  End
  mUp,	  mUp,	  mNull,  mCancel,mNull,  mOk,	  mNull,  mLast,
// Down	  PgDn	  Ins	  Del	  S-F1	  S-F2	  S-F3	  S-F4
  mDown,  mDown,  mClose, mCancel,mNull,  mNull,  mNull,  mNull,
// S-F5	  S-F6	  S-F7	  S-F8	  S-F9	  S-F10	  C-F1	  C-F2
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// C-F3	  C-F4	  C-F5	  C-F6	  C-F7	  C-F8	  C-F9	  C-F10
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-F1	  A-F2	  A-F3	  A-F4	  A-F5	  A-F6	  A-F7	  A-F8
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-F9	  A-F10	  PrtSc	  C-Left  C-Right C-End	  C-PgDn  C-Home
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-1	  A-2	  A-3	  A-4	  A-5	  A-6	  A-7/8	  A-9
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// A-0	  A -	  A-=		  C-PgUp  F11	  F12	  S-F11
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// S-F12  C-F11	  C-F12	  A-F11	  A-F12	  C-Up	  C-/	  C-5
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
// S-*	  C-Down  C-Ins	  C-Del	  C-Tab	  C -	  C-+
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,
//				  A -	  A-Tab	  A-Enter
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 160
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 168
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 176
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 184
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 192
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 200
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 208
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 216
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 224
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 232
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,   // 240
  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull,  mNull	   // 248
};
#endif

/* --- SelectMenu --- */

static Menu SelectMenu;

static MenuItem SelectMenuDefaultItem[] = {
/* type        label           variable value func     popup keys data  */
    {MENU_NOP, "-- ", NULL, 0, nulcmd, NULL, "", NULL},
#if LANG == JA
    {MENU_MFUNC, "򤵤줿Хåե (D)", NULL, FUNCNAME_smDelBuf, NULL, NULL, "D", NULL},
#else				/* LANG != JA */
    {MENU_MFUNC, N_("Delete selected buffer (D)"), NULL, FUNCNAME_smDelBuf, NULL, NULL, "D", NULL},
#endif				/* LANG != JA */
    {MENU_END, "", NULL, 0, nulcmd, NULL, "", NULL},
};

static MenuItem *SelectMenuItem = SelectMenuDefaultItem;
static int SelectV = 0;
static void initSelectMenu(void);
static void smChBuf(void);

/* --- SelectMenu (END) --- */

/* --- MainMenu --- */

static Menu MainMenu;

static MenuItem MainMenuItem[] = {
/* type        label           variable value func     popup keys data  */
#if LANG == JA
    {MENU_FUNC, "         (b)", NULL, 0, backBf, NULL, "b", NULL},
    {MENU_POPUP, "Хåե (s)", NULL, 0, NULL, &SelectMenu, "s", NULL},
    {MENU_FUNC, "ɽ (v)", NULL, 0, vwSrc, NULL, "v V", NULL},
    {MENU_FUNC, "Խ (e)", NULL, 0, editBf, NULL, "e E", NULL},
    {MENU_FUNC, "¸ (S)", NULL, 0, svSrc, NULL, "S", NULL},
    {MENU_FUNC, "ɤ߹   (r)", NULL, 0, reload, NULL, "r R", NULL},
    {MENU_NOP,  "", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, "󥯤ɽ (a)", NULL, 0, followA, NULL, "a", NULL},
    {MENU_FUNC, "󥯤¸ (A)", NULL, 0, svA, NULL, "A", NULL},
    {MENU_FUNC, "ɽ   (i)", NULL, 0, followI, NULL, "i", NULL},
    {MENU_FUNC, "¸   (I)", NULL, 0, svI, NULL, "I", NULL},
    {MENU_FUNC, "ե졼ɽ (f)", NULL, 0, rFrame, NULL, "f F", NULL},
    {MENU_NOP,  "", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, "֥åޡ (B)", NULL, 0, ldBmark, NULL, "B", NULL},
    {MENU_FUNC, "إ       (h)", NULL, 0, ldhelp, NULL, "h H", NULL},
    {MENU_FUNC, "ץ   (o)", NULL, 0, ldOpt, NULL, "o O", NULL},
    {MENU_NOP,  "", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, "顼   (l)", NULL, 0, vwErrLog, NULL, "l L", NULL},
    {MENU_FUNC, "ץ (p)", NULL, 0, procList, NULL, "p P", NULL},
    {MENU_NOP,  "", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, "λ         (q)", NULL, 0, qquitfm, NULL, "q Q", NULL},
#else				/* LANG != JA */
    {MENU_FUNC, N_(" Back         (b) "), NULL, 0, backBf, NULL, "b", NULL},
    {MENU_POPUP, N_(" Select Buffer(s) "), NULL, 0, NULL, &SelectMenu, "s", NULL},
    {MENU_FUNC, N_(" View Source  (v) "), NULL, 0, vwSrc, NULL, "v V", NULL},
    {MENU_FUNC, N_(" Edit Source  (e) "), NULL, 0, editBf, NULL, "e E", NULL},
    {MENU_FUNC, N_(" Save Source  (S) "), NULL, 0, svSrc, NULL, "S", NULL},
    {MENU_FUNC, N_(" Reload       (r) "), NULL, 0, reload, NULL, "r R", NULL},
    {MENU_NOP, " ---------------- ", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, N_(" Go Link      (a) "), NULL, 0, followA, NULL, "a", NULL},
    {MENU_FUNC, N_(" Save Link    (A) "), NULL, 0, svA, NULL, "A", NULL},
    {MENU_FUNC, N_(" View Image   (i) "), NULL, 0, followI, NULL, "i", NULL},
    {MENU_FUNC, N_(" Save Image   (I) "), NULL, 0, svI, NULL, "I", NULL},
    {MENU_FUNC, N_(" View Frame   (f) "), NULL, 0, rFrame, NULL, "f F", NULL},
    {MENU_NOP, " ---------------- ", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, N_(" Bookmark     (B) "), NULL, 0, ldBmark, NULL, "B", NULL},
    {MENU_FUNC, N_(" Help         (h) "), NULL, 0, ldhelp, NULL, "h H", NULL},
    {MENU_FUNC, N_(" Option       (o) "), NULL, 0, ldOpt, NULL, "o O", NULL},
    {MENU_NOP, " ---------------- ", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, N_(" Error Log    (l) "), NULL, 0, vwErrLog, NULL, "l L", NULL},
    {MENU_FUNC, N_(" Process List (p) "), NULL, 0, procList, NULL, "p P", NULL},
    {MENU_NOP, " ---------------- ", NULL, 0, nulcmd, NULL, "", NULL},
    {MENU_FUNC, N_(" Quit         (q) "), NULL, 0, qquitfm, NULL, "q Q", NULL},
#endif				/* LANG != JA */
    {MENU_END, "", NULL, 0, nulcmd, NULL, "", NULL},
};

/* --- MainMenu (END) --- */

extern btri_string_tab_t w3mFuncTab[];
extern FuncList w3mFuncList[];
static MenuList *w3mMenuList;

#define mvaddch(y, x, c)        (move(y, x), addch(c))
#define mvaddstr(y, x, str)     (move(y, x), addstr(str))
#define mvaddnstr(y, x, str, n) (move(y, x), addnstr_sup(str, n))

void
new_menu(Menu *menu, MenuItem *item)
{
    int i, l, key;
    char *p, *q;
    KeyTabList menu_tab = {NULL, NULL, 0, 0}, sel_tab = {NULL, NULL, 0, 0};

    menu->cursorX = 0;
    menu->cursorY = 0;
    menu->x = 0;
    menu->y = 0;
    menu->nitem = 0;
    menu->item = item;
    menu->initial = 0;
    menu->select = 0;
    menu->offset = 0;
    menu->active = 0;

    if (item == NULL)
	return;

    for (i = 0; i <= K_FUNC_MAX && item[i].type != MENU_END; i++)
	;
    menu->nitem = i;
    menu->height = menu->nitem;
    menu_tab.next = w3mMenuKeyTabList;
    menu->width = 0;
    for (i = 0; i < menu->nitem; i++) {
	if ((p = item[i].keys) != NULL) {
	  if (item[i].type & MENU_MFUNC)
	    for (p = allocStr(p, -1) ; *p ;) {
	      q = getQWord(&p);
	      if ((key = getKey(q)) >= 0)
		addToKeyTab(&menu_tab, key, item[i].value);
	    }
	  else
	    for (p = allocStr(p, -1) ; *p ;) {
	      q = getQWord(&p);
	      if ((key = getKey(q)) >= 0) {
		addToKeyTab(&menu_tab, key, FUNCNAME_mSelect);
		addToKeyTab(&sel_tab, key, i);
	      }
	    }
	}
#ifdef MANY_CHARSET
	l = ttyfix_width(item[i].label);
#else
	l = strlen(item[i].label);
#endif
	if (l > menu->width)
	    menu->width = l;
    }
    menu->keymap = copyKeyTabList(&menu_tab);
    menu->keyselect = copyKeyTabList(&sel_tab);
}

void
geom_menu(Menu *menu, int x, int y, int mselect)
{
    int win_x, win_y, win_w, win_h;

    menu->select = mselect;

    if (menu->width % FRAME_WIDTH)
	menu->width = (menu->width / FRAME_WIDTH + 1) * FRAME_WIDTH;
    win_x = menu->x - FRAME_WIDTH;
    win_w = menu->width + 2 * FRAME_WIDTH;
    if (win_x + win_w > COLS)
	win_x = COLS - win_w;
    if (win_x < 0) {
	win_x = 0;
	if (win_w > COLS) {
	    menu->width = COLS - 2 * FRAME_WIDTH;
	    menu->width -= menu->width % FRAME_WIDTH;
	    win_w = menu->width + 2 * FRAME_WIDTH;
	}
    }
    menu->x = win_x + FRAME_WIDTH;

    win_y = menu->y - mselect - 1;
    win_h = menu->height + 2;
    if (win_y + win_h > LASTLINE)
	win_y = LASTLINE - win_h;
    if (win_y < 0) {
	win_y = 0;
	if (win_y + win_h > LASTLINE) {
	    win_h = LASTLINE - win_y;
	    menu->height = win_h - 2;
	    if (menu->height <= mselect)
		menu->offset = mselect - menu->height + 1;
	}
    }
    menu->y = win_y + 1;
}

void
draw_all_menu(Menu *menu)
{
    if (menu->parent != NULL)
	draw_all_menu(menu->parent);
    draw_menu(menu);
}

void
draw_menu(Menu *menu)
{
    int x, oy, y, w;
    int i, j;

    x = menu->x - FRAME_WIDTH;
    w = menu->width + 2 * FRAME_WIDTH;
    oy = y = menu->y - 1;

#ifndef KANJI_SYMBOLS
    if (FRAME == NULL) {
	if (graph_ok()) {
	    graph_mode = TRUE;
	    FRAME = G_FRAME;
	}
	else {
	    FRAME = N_FRAME;
	}
    }
#endif				/* not KANJI_SYMBOLS */

    if (menu->offset == 0) {
	G_start;
	mvaddstr(y, x, FRAME[0]);
	for (i = FRAME_WIDTH; i < w - FRAME_WIDTH; i += FRAME_WIDTH)
	    mvaddstr(y, x + i, FRAME[1]);
	mvaddstr(y, x + i, FRAME[2]);
	G_end;
    }
    else {
	G_start;
	mvaddstr(y, x, FRAME[3]);
	G_end;
	for (i = FRAME_WIDTH; i < w - FRAME_WIDTH; i += FRAME_WIDTH)
	    mvaddstr(y, x + i, FRAME[4]);
	G_start;
	mvaddstr(y, x + i, FRAME[5]);
	G_end;
	i = (w / 2 - 1) / FRAME_WIDTH * FRAME_WIDTH;
	mvaddstr(y, x + i, FRAME[9]);
    }

    for (j = 0; j < menu->height; j++) {
	y++;
	G_start;
	mvaddstr(y, x, FRAME[3]);
	G_end;
	draw_menu_item(menu, menu->offset + j);
	G_start;
	mvaddstr(y, x + w - FRAME_WIDTH, FRAME[5]);
	G_end;
    }
    y++;
    if (menu->offset + menu->height == menu->nitem) {
	G_start;
	mvaddstr(y, x, FRAME[6]);
	for (i = FRAME_WIDTH; i < w - FRAME_WIDTH; i += FRAME_WIDTH)
	    mvaddstr(y, x + i, FRAME[7]);
	mvaddstr(y, x + i, FRAME[8]);
	G_end;
    }
    else {
	G_start;
	mvaddstr(y, x, FRAME[3]);
	G_end;
	for (i = FRAME_WIDTH; i < w - FRAME_WIDTH; i += FRAME_WIDTH)
	    mvaddstr(y, x + i, FRAME[4]);
	G_start;
	mvaddstr(y, x + i, FRAME[5]);
	G_end;
	i = (w / 2 - 1) / FRAME_WIDTH * FRAME_WIDTH;
	mvaddstr(y, x + i, FRAME[10]);
    }
}

void
draw_menu_item(Menu *menu, int mselect)
{
    mvaddnstr(menu->y + mselect - menu->offset, menu->x,
	      menu->item[mselect].label, menu->width);
}

int
select_menu(Menu *menu, int mselect)
{
    if (mselect < 0 || mselect >= menu->nitem)
	return (MENU_NOTHING);
#ifdef USE_IMAGE
    stopImage();
#endif
    if (mselect < menu->offset)
	up_menu(menu, menu->offset - mselect);
    else if (mselect >= menu->offset + menu->height)
	down_menu(menu, mselect - menu->offset - menu->height + 1);

    if (menu->select >= menu->offset &&
	menu->select < menu->offset + menu->height)
	draw_menu_item(menu, menu->select);
    menu->select = mselect;
    standout();
    draw_menu_item(menu, menu->select);
    standend();
    move(menu->y + mselect - menu->offset, menu->x);
    toggle_stand();
#ifdef USE_IMAGE
    if (activeImage)
	untouchImageRegion();
#endif
    refresh();
#ifdef USE_IMAGE
    if (activeImage)
	drawImage();
#endif

    return (menu->select);
}

void
goto_menu(Menu *menu, int mselect, int down)
{
    int select_in;
    if (mselect >= menu->nitem)
	mselect = menu->nitem - 1;
    else if (mselect < 0)
	mselect = 0;
    select_in = mselect;
    while (menu->item[mselect].type == MENU_NOP || menu->item[mselect].type == MENU_MFUNC) {
	if (down > 0) {
	    if (++mselect >= menu->nitem)
	    {
		down_menu(menu, select_in - menu->select);
		mselect = menu->select;
		break;
	    }
	}
	else if (down < 0) {
	    if (--mselect < 0)
	    {
		up_menu(menu, menu->select - select_in);
		mselect = menu->select;
		break;
	    }
	}
	else {
	    return;
	}
    }
    select_menu(menu, mselect);
}

void
up_menu(Menu *menu, int n)
{
    if (n < 0 || menu->offset == 0)
	return;
    menu->offset -= n;
    if (menu->offset < 0)
	menu->offset = 0;

    draw_menu(menu);
}

void
down_menu(Menu *menu, int n)
{
    if (n < 0 || menu->offset + menu->height == menu->nitem)
	return;
    menu->offset += n;
    if (menu->offset + menu->height > menu->nitem)
	menu->offset = menu->nitem - menu->height;

    draw_menu(menu);
}

int
action_menu(Menu *menu)
{
    char c;
    int mselect;
    MenuItem item;
    CookedEvent cev;

    if (menu->active == 0) {
	if (menu->parent != NULL)
	    menu->parent->active = 0;
	return (0);
    }
    draw_all_menu(menu);
    select_menu(menu, menu->select);
    while (1) {
#ifdef USE_MOUSE
	if (use_mouse)
	    mouse_active();
#endif				/* USE_MOUSE */
	if (!getevent(&cev))
	  continue;
#ifdef USE_MOUSE
	if (use_mouse) {
	    mouse_inactive();
#if defined(USE_GPM) || defined(USE_SYSMOUSE)
	    if (cev.type == cooked_event_mouse) {
	      mselect = process_mMouse(cev.what, cev.x, cev.y);
	      if (mselect != MENU_NOTHING)
		break;
	      continue;
	    }
#endif				/* defined(USE_GPM) || defined(USE_SYSMOUSE) */
	}
#endif				/* USE_MOUSE */
	c = cev.what;
	if (IS_ASCII(c)) {	/* Ascii */
	    mselect = call_menu_func(0, c);
	    if (mselect != MENU_NOTHING)
		break;
	}
    }
    if (mselect >= 0 && mselect < menu->nitem) {
	item = menu->item[mselect];
	if (item.type & MENU_POPUP) {
	    popup_menu(menu, item.popup);
	    return (1);
	}
	if (menu->parent != NULL)
	    menu->parent->active = 0;
	if (item.type & MENU_VALUE)
	    *item.variable = item.value;
	if (item.type & MENU_FUNC) {
	    CurrentKey = 0;
	    ForcedKeyData = CurrentKeyData = NULL;
	    TargetX = TargetY = -1;
	    CurrentCmdData = item.data;
	    (*item.func) ();
	    CurrentCmdData = NULL;
	}
    }
    else if (mselect == MENU_CLOSE) {
	if (menu->parent != NULL)
	    menu->parent->active = 0;
    }
    return (0);
}

void
popup_menu(Menu *parent, Menu *menu)
{
    int active = 1, save_popup;

    if (menu->item == NULL || menu->nitem == 0)
	return;
    if (menu->active)
	return;

    menu->parent = parent;
    menu->select = menu->initial;
    menu->offset = 0;
    menu->active = 1;
    if (parent != NULL) {
	menu->cursorX = parent->cursorX;
	menu->cursorY = parent->cursorY;
	guess_menu_xy(parent, menu->width, &menu->x, &menu->y);
    }
    geom_menu(menu, menu->x, menu->y, menu->select);

    CurrentMenu = menu;
    while (active) {
	save_popup = CurrentMenuPopup;
	CurrentMenuPopup = TRUE;
	active = action_menu(CurrentMenu);
	CurrentMenuPopup = save_popup;
	displayBuffer(Currentbuf,
#ifdef USE_IMAGE
		      B_REDRAW_IMAGE
#else
		      B_FORCE_REDRAW
#endif
		      );
    }
    menu->active = 0;
    CurrentMenu = parent;
}

void
guess_menu_xy(Menu *parent, short width, short *x, short *y)
{
    *x = parent->x + parent->width + FRAME_WIDTH - 1;
    if (*x + width + FRAME_WIDTH > COLS) {
	*x = COLS - width - FRAME_WIDTH;
	if ((parent->x + parent->width / 2 > *x) &&
	    (parent->x + parent->width / 2 > COLS / 2))
	    *x = parent->x - width - FRAME_WIDTH + 1;
    }
    *y = parent->y + parent->select - parent->offset;
}

void
new_option_menu(Menu *menu, char **label, int *variable, void (*func)(void), MenuItem *more)
{
    int i, nitem, m;
    char **p;
    MenuItem *item;

    if (label == NULL || *label == NULL)
	return;

    for (i = 0, p = label; *p != NULL; i++, p++)
	;
    m = nitem = i;

    if (more)
      for (item = more ; item->type != MENU_END ; ++m, ++item)
	;

    item = New_N(MenuItem, m + 1);

    for (i = 0, p = label; i < nitem; i++, p++) {
	if (func != NULL)
	    item[i].type = MENU_VALUE | MENU_FUNC;
	else
	    item[i].type = MENU_VALUE;
	item[i].label = *p;
	item[i].variable = variable;
	item[i].value = i;
	item[i].func = func;
	item[i].popup = NULL;
	item[i].keys = "";
    }
    for (; i < m ; ++i)
      item[i] = more[i - nitem];
    item[i].type = MENU_END;

    new_menu(menu, item);
}

/* --- MenuFunctions --- */

static int
call_menu_func(int key, int c)
{
  if (K_LEN(key) < K_LEN_MAX) {
    FuncList *fl;

    key = K_GEN(key, c);

    if ((fl = lookupKey(key, CurrentMenu->keymap)))
      return fl->func.menu_func(key);
  }

  return MENU_NOTHING;
}

static int
call_menu_func_nostroke(int key, int c)
{
  FuncList *fl;

  key = K_OVER(key, c);

  if ((fl = lookupKey(key, CurrentMenu->keymap)))
    return fl->func.menu_func(key);

  return MENU_NOTHING;
}

int
mNextchar(int key)
{
  return call_menu_func(key, (unsigned char)getch());
}


#ifdef __EMX__
int
mPc(int c)
{
  c = (unsigned char)getch();
  return(MenuPcKeymap[(int)c](c));
}
#endif

int
mEsc(int key)
{
  int c;

  c = (unsigned int)getch();

  if (IS_ASCII(c))
    return call_menu_func_nostroke(key, K_ESC | c);

  return MENU_NOTHING;
}

static int
mEscD(int key, char c)
{
    int d;

    d = (int) c - (int) '0';
    c = getch();
    if (IS_DIGIT(c)) {
	d = d * 10 + (int) c - (int) '0';
	c = getch();
    }
    if (c == '~')
	return call_menu_func_nostroke(key, K_ESCD | d);
    else
	return (MENU_NOTHING);
}

int
mEscB(int key)
{
  int c;

  c = (unsigned char)getch();

  if (IS_DIGIT(c))
    return mEscD(key, c);
  else
    return call_menu_func_nostroke(key, K_ESCB | c);
}

int
mNull(int key)
{
    return (MENU_NOTHING);
}

int
mSelect(int key)
{
  if (CurrentMenu->keyselect) {
    int i;

    if ((i = lookupKeyIndex(key, CurrentMenu->keyselect)) >= 0)
      return select_menu(CurrentMenu, K_GET_FUNC(CurrentMenu->keyselect->key_table[i]));
  }

  return (MENU_NOTHING);
}

int
mDown(int c)
{
    if (CurrentMenu->select >= CurrentMenu->nitem - 1)
	return (MENU_NOTHING);
    goto_menu(CurrentMenu, CurrentMenu->select + 1, 1);
    return (MENU_NOTHING);
}

int
mUp(int c)
{
    if (CurrentMenu->select <= 0)
	return (MENU_NOTHING);
    goto_menu(CurrentMenu, CurrentMenu->select - 1, -1);
    return (MENU_NOTHING);
}

int
mLast(int c)
{
    goto_menu(CurrentMenu, CurrentMenu->nitem - 1, -1);
    return (MENU_NOTHING);
}

int
mTop(int c)
{
    goto_menu(CurrentMenu, 0, 1);
    return (MENU_NOTHING);
}

int
mNext(int c)
{
    int mselect = CurrentMenu->select + CurrentMenu->height;

    if (mselect >= CurrentMenu->nitem)
	return mLast(c);
    down_menu(CurrentMenu, CurrentMenu->height);
    goto_menu(CurrentMenu, mselect, -1);
    return (MENU_NOTHING);
}

int
mPrev(int c)
{
    int mselect = CurrentMenu->select - CurrentMenu->height;

    if (mselect < 0)
	return mTop(c);
    up_menu(CurrentMenu, CurrentMenu->height);
    goto_menu(CurrentMenu, mselect, 1);
    return (MENU_NOTHING);
}

int
mFore(int c)
{
    if (CurrentMenu->select >= CurrentMenu->nitem - 1)
	return MENU_NOTHING;
    goto_menu(CurrentMenu, CurrentMenu->select + CurrentMenu->height - 1,
	      CurrentMenu->height + 1);
    return (MENU_NOTHING);
}

int
mBack(int c)
{
    if (CurrentMenu->select <= 0)
	return (MENU_NOTHING);
    goto_menu(CurrentMenu, CurrentMenu->select - CurrentMenu->height + 1,
	      -1 - CurrentMenu->height);
    return (MENU_NOTHING);
}

int
mOk(int c)
{
    int mselect = CurrentMenu->select;

    if (CurrentMenu->item[mselect].type == MENU_NOP ||
	CurrentMenu->item[mselect].type == MENU_MFUNC)
	return (MENU_NOTHING);
    return (mselect);
}

int
mCancel(int c)
{
    return (MENU_CANCEL);
}

int
mClose(int c)
{
    return (MENU_CLOSE);
}

int
mSusp(int c)
{
    susp();
    draw_all_menu(CurrentMenu);
    select_menu(CurrentMenu, CurrentMenu->select);
    return (MENU_NOTHING);
}

static char *SearchString = NULL;

int (*menuSearchRoutine)(Menu *, char *, int);

static int
menuForwardSearch(Menu *menu, char *str, int from)
{
    int i;
    char* p;
    if ((p = regexCompile(str, IgnoreCase)) != NULL) {
	message(p);
	return -1;
    }
    if (from < 0)
	from = 0;
    for (i = from; i < menu->nitem; i++)
	if (menu->item[i].type != MENU_NOP &&
	    menu->item[i].type != MENU_MFUNC &&
	    regexMatch(menu->item[i].label, -1))
	    return i;
    return -1;
}

static int
menu_search_forward(Menu *menu, int from)
{
    char *str;
    int found;
    str = inputStrHist("Forward: ", NULL, TextHist);
    if (str != NULL && *str == '\0')
	str = SearchString;
    if (str == NULL || *str == '\0')
	return -1;
    SearchString = str;
    menuSearchRoutine = menuForwardSearch;
    found = menuForwardSearch(menu, SearchString, from + 1);
    if (WrapSearch && found == -1)
        found = menuForwardSearch(menu, SearchString, 0);
    if (found >= 0)
        return found;
    disp_message("Not found", TRUE);
    return -1;
}

int
mSrchF(int c)
{
    int mselect;
    mselect = menu_search_forward(CurrentMenu, CurrentMenu->select);
    if (mselect >= 0)
	goto_menu(CurrentMenu, mselect, 1);
    return (MENU_NOTHING);
}

static int
menuBackwardSearch(Menu *menu, char *str, int from)
{
    int i;
    char* p;
    if ((p = regexCompile(str, IgnoreCase)) != NULL) {
	message(p);
	return -1;
    }
    if (from >= menu->nitem)
	from = menu->nitem - 1;
    for (i = from; i >= 0; i--)
	if (menu->item[i].type != MENU_NOP &&
	    menu->item[i].type != MENU_MFUNC &&
	    regexMatch(menu->item[i].label, -1))
            return i;
    return -1;
}

static int
menu_search_backward(Menu *menu, int from)
{
    char *str;
    int found;
    str = inputStrHist("Backward: ", NULL, TextHist);
    if (str != NULL && *str == '\0')
	str = SearchString;
    if (str == NULL || *str == '\0')
	return (MENU_NOTHING);
    SearchString = str;
    menuSearchRoutine = menuBackwardSearch;
    found = menuBackwardSearch(menu, SearchString, from - 1);
    if (WrapSearch && found == -1)
        found = menuBackwardSearch(menu, SearchString, menu->nitem);
    if (found >= 0)
        return found;
    disp_message("Not found", TRUE);
    return -1;
}

int
mSrchB(int c)
{
    int mselect;
    mselect = menu_search_backward(CurrentMenu, CurrentMenu->select);
    if (mselect >= 0)
	goto_menu(CurrentMenu, mselect, -1);
    return (MENU_NOTHING);
}

static int
menu_search_next_previous(Menu *menu, int from, int reverse)
{
    int found;
    static int (*routine[2])(Menu *, char *, int) = {
	menuForwardSearch, menuBackwardSearch
    };

    if (menuSearchRoutine == NULL) {
	disp_message("No previous regular expression", TRUE);
	return -1;
    }
    if (reverse != 0)
	reverse = 1;
    if (menuSearchRoutine == menuBackwardSearch)
	reverse ^= 1;
    from += reverse ? -1 : 1;
    found = (*routine[reverse])(menu, SearchString, from);
    if (WrapSearch && found == -1)
        found = (*routine[reverse])(menu, SearchString, reverse * menu->nitem);
    if (found >= 0)
        return found;
    disp_message("Not found", TRUE);
    return -1;
}

int
mSrchN(int c)
{
    int mselect;
    mselect = menu_search_next_previous(CurrentMenu, CurrentMenu->select, 0);
    if (mselect >= 0)
	goto_menu(CurrentMenu, mselect, 1);
    return (MENU_NOTHING);
}

int
mSrchP(int c)
{
    int mselect;
    mselect = menu_search_next_previous(CurrentMenu, CurrentMenu->select, 1);
    if (mselect >= 0)
	goto_menu(CurrentMenu, mselect, -1);
    return (MENU_NOTHING);
}

#ifdef USE_MOUSE
int
mPositional(int key)
{
  int x, y, mselect;
  Menu *menu;

  if (TargetX < 0 || TargetY < 0)
    return (MENU_NOTHING);

  x = TargetX;
  y = TargetY;
  menu = CurrentMenu;

  if (x < menu->x - FRAME_WIDTH ||
      x >= menu->x + menu->width + FRAME_WIDTH ||
      y < menu->y - 1 ||
      y >= menu->y + menu->height + 1) {
    return (MENU_CANCEL);
  }
  else if ((x >= menu->x - FRAME_WIDTH &&
	    x < menu->x) ||
	   (x >= menu->x + menu->width &&
	    x < menu->x + menu->width + FRAME_WIDTH)) {
    return (MENU_NOTHING);
  }
  else if (y == menu->y - 1) {
    mPrev(key);
    return (MENU_NOTHING);
  }
  else if (y == menu->y + menu->height) {
    mNext(key);
    return (MENU_NOTHING);
  }
  else {
    mselect = y - menu->y + menu->offset;
    if (menu->item[mselect].type == MENU_NOP ||
	menu->item[mselect].type == MENU_MFUNC)
      return (MENU_NOTHING);
    return (select_menu(menu, mselect));
  }
}

void
process_menu_mouse(int key, void *arg)
{
  int *p_select;
  FuncList *fl;

  p_select = arg;
  *p_select = MENU_NOTHING;
  key = K_GEN(0, key);

  if ((fl = lookupKey(key, w3mMenuKeyTabList)) &&
      ((Currentbuf && Currentbuf != NO_BUFFER) ||
       (fl - w3mFuncList) == FUNCNAME_quitfm || (fl - w3mFuncList) == FUNCNAME_qquitfm))
    *p_select = fl->func.menu_func(key);
}

int
mMouse(int key)
{
    int btn, x, y, mselect = MENU_NOTHING;

    btn = (unsigned char)getch() - 32;
    x = (unsigned char)getch() - 33;
    if (x < 0)
	x += 0x100;
    y = (unsigned char)getch() - 33;
    if (y < 0)
	y += 0x100;

    if (x >= 0 && x < COLS && y >= 0 && y < LASTLINE)
	process_mouse(btn, x, y, process_menu_mouse, &mselect);

    return mselect;
}
#else				/* not USE_MOUSE */
int
mMouse(int c)
{
    return (MENU_NOTHING);
}
#endif				/* not USE_MOUSE */

/* --- MenuFunctions (END) --- */

/* --- MainMenu --- */

void
popupMenu(short x, short y, Menu *menu)
{
    initSelectMenu();

    menu->cursorX = Currentbuf->rootX + Currentbuf->cursorX;
    menu->cursorY = Currentbuf->rootY + Currentbuf->cursorY;
    menu->x = x + FRAME_WIDTH + 1;
    menu->y = y + 2;

    popup_menu(NULL, menu);
}

void
mainMenu(short x, short y)
{
    popupMenu(x, y, &MainMenu);
}

void
mainMn(void)
{
    Menu *menu = &MainMenu;
    char *data;
    int n;

    if (TargetX >= 0 && TargetY >= 0) {
      cursorXY(Currentbuf, TargetX, TargetY);
      displayBuffer(Currentbuf, B_NORMAL);
    }

    data = searchKeyData();
    if (data != NULL) {
	n = getMenuN(w3mMenuList, data);
	if (n < 0)
	    return;
	menu = w3mMenuList[n].menu;
    }
    popupMenu(Currentbuf->rootX + Currentbuf->cursorX, Currentbuf->rootY + Currentbuf->cursorY, menu);
}

/* --- MainMenu (END) --- */

/* --- SelectMenu --- */

void
selMn(void)
{
    if (TargetX >= 0 && TargetY >= 0) {
      cursorXY(Currentbuf, TargetX, TargetY);
      displayBuffer(Currentbuf, B_NORMAL);
    }

    popupMenu(Currentbuf->rootX + Currentbuf->cursorX, Currentbuf->rootY + Currentbuf->cursorY, &SelectMenu);
}

static void
initSelectMenu(void)
{
    int i, nitem, len = 0;
    Buffer *buf;
    Str str;
    char **label;

    SelectV = -1;
    for (i = 0, buf = Firstbuf; buf != NULL; i++, buf = buf->nextBuffer) {
	if (buf == Currentbuf)
	    SelectV = i;
    }
    nitem = i;

    label = New_N(char *, nitem + 1);
    for (i = 0, buf = Firstbuf; i < nitem; i++, buf = buf->nextBuffer) {
	str = Sprintf("<%s>", buf->buffername);
	if (buf->filename != NULL) {
	    switch (buf->currentURL.scheme) {
	    case SCM_LOCAL:
	    case SCM_LOCAL_CGI:
		if (strcmp(buf->currentURL.file, "-")) {
		    Strcat_char(str, ' ');
		    Strcat_charp(str,
				 conv_from_system(buf->currentURL.real_file));
		}
		break;
	    case SCM_MISSING:
		break;
	    default:
		Strcat_char(str, ' ');
		Strcat(str, parsedURL2Str(&buf->currentURL));
		break;
	    }
	}
	label[i] = str->ptr;
	if (len < str->length)
	    len = str->length;
    }
    label[nitem] = NULL;

    new_option_menu(&SelectMenu, label, &SelectV, smChBuf, SelectMenuItem);
    SelectMenu.initial = SelectV;
    SelectMenu.cursorX = Currentbuf->rootX + Currentbuf->cursorX;
    SelectMenu.cursorY = Currentbuf->rootY + Currentbuf->cursorY;
    SelectMenu.item[nitem].type = MENU_NOP;
}

static void
smChBuf(void)
{
    int i;
    Buffer *buf;

    if (SelectV < 0 || SelectV >= SelectMenu.nitem)
	return;
    for (i = 0, buf = Firstbuf; i < SelectV; i++, buf = buf->nextBuffer)
	;
    Currentbuf = buf;
    for (buf = Firstbuf; buf != NULL; buf = buf->nextBuffer) {
	if (buf == Currentbuf)
	    continue;
	if (clear_buffer)
	    tmpClearBuffer(buf);
    }
}

void
reshapeMenu(void)
{
    int x, y, mselect;

    x = CurrentMenu->x;
    y = CurrentMenu->y;
    mselect = CurrentMenu->select;

    initSelectMenu();

    CurrentMenu->x = x;
    CurrentMenu->y = y;

    geom_menu(CurrentMenu, x, y, 0);

    CurrentMenu->select = (mselect <= CurrentMenu->nitem - 2) ? mselect
	: (CurrentMenu->nitem - 2);
}

int
smDelBuf(int c)
{
    int i;
    Buffer *buf;

    if (CurrentMenu != &SelectMenu ||
	CurrentMenu->select < 0 || CurrentMenu->select >= SelectMenu.nitem)
	return (MENU_NOTHING);
    for (i = 0, buf = Firstbuf; i < CurrentMenu->select; i++, buf = buf->nextBuffer)
	;
    if (Currentbuf == buf)
	Currentbuf = buf->nextBuffer;
    Firstbuf = deleteBuffer(Firstbuf, buf);
    if (!Currentbuf)
	Currentbuf = nthBuffer(Firstbuf, i - 1);
    if (Firstbuf == NULL) {
	Firstbuf = Currentbuf = nullBuffer();
    }

    reshapeMenu();
    displayBuffer(Currentbuf, B_FORCE_REDRAW);
    draw_all_menu(CurrentMenu);
    select_menu(CurrentMenu, CurrentMenu->select);
    return (MENU_NOTHING);
}

/* --- SelectMenu (END) --- */

/* --- OptionMenu --- */

void
optionMenu(int x, int y, char **label, int *variable, int initial,
	   void (*func) ())
{
    Menu menu;

    new_option_menu(&menu, label, variable, func, NULL);
    menu.cursorX = COLS - 1;
    menu.cursorY = LASTLINE;
    menu.x = x;
    menu.y = y;
    menu.initial = initial;

    popup_menu(NULL, &menu);
}

/* --- OptionMenu (END) --- */

/* --- InitMenu --- */

static void
initMenuStart(void)
{
    w3mMenuList = New_N(MenuList, 3);
    w3mMenuList[0].id = "Main";
    w3mMenuList[0].menu = &MainMenu;
    w3mMenuList[0].item = MainMenuItem;
    w3mMenuList[1].id = "Select";
    w3mMenuList[1].menu = &SelectMenu;
    w3mMenuList[1].item = SelectMenuDefaultItem;
    w3mMenuList[2].id = NULL;
}

static void
initMenuFromFile(FILE *mf)
{
    Str line;
    char *p, *s;
    int in_menu, nmenu = 0, nitem = 0, type;
    MenuItem *item = NULL;

    in_menu = 0;
    while (!feof(mf)) {
#ifdef MANY_CHARSET
	line = conv_Str2mbStr(Strfgets(mf), NULL, "!", &tty_mb_r_setup);
#else
	line = Strfgets(mf);
#endif
	Strchop(line);
	Strremovefirstspaces(line);
	if (line->length == 0)
	    continue;
	p = line->ptr;
	s = getWord(&p);
	if (*s == '#')		/* comment */
	    continue;
	if (in_menu) {
	    type = setMenuItem(&item[nitem], s, p);
	    if (type == -1)
		continue;	/* error */
	    if (type == MENU_END)
		in_menu = 0;
	    else {
		nitem++;
		item = New_Reuse(MenuItem, item, (nitem + 1));
		w3mMenuList[nmenu].item = item;
		item[nitem].type = MENU_END;
	    }
	}
	else {
	    if (strcmp(s, "menu"))	/* error */
		continue;
	    s = getQWord(&p);
	    if (*s == '\0')	/* error */
		continue;
	    in_menu = 1;
	    if ((nmenu = getMenuN(w3mMenuList, s)) != -1)
		w3mMenuList[nmenu].item = New(MenuItem);
	    else
		nmenu = addMenuList(&w3mMenuList, s);
	    item = w3mMenuList[nmenu].item;
	    nitem = 0;
	    item[nitem].type = MENU_END;
	}
    }
}

static void
initMenuEnd(void)
{
    MenuList *list;

    SelectMenuItem = w3mMenuList[1].item;
    for (list = w3mMenuList; list->id != NULL; list++) {
	if (list->item == NULL)
	    continue;
	new_menu(list->menu, list->item);
    }
}

void
initMenu(void)
{
  FILE *f;

  initMenuStart();
#ifdef MANY_CHARSET
  try_cfg_files(W3MMENU, initMenuFromFile, etcFile);
#else
  if ((f = fopen(etcFile(W3MMENU), "rt"))) {
    initMenuFromFile(f);
    fclose(f);
  }
#endif
  if ((f = fopen(rcFile(MENU_FILE), "rt"))) {
    initMenuFromFile(f);
    fclose(f);
  }
  initMenuEnd();
  CurrentCmdData = NULL;
}

int
setMenuItem(MenuItem *item, char *type, char *line)
{
    char *label, *func, *popup, *keys, *data;
    int n;

    if (type == NULL || *type == '\0')	/* error */
	return -1;
    if (strcmp(type, "end") == 0) {
	item->type = MENU_END;
	return MENU_END;
    }
    else if (strcmp(type, "nop") == 0) {
	item->type = MENU_NOP;
	item->label = getQWord(&line);
	return MENU_NOP;
    }
    else if (strcmp(type, "func") == 0) {
	FuncList *fl;

	label = getQWord(&line);
	func = getWord(&line);
	keys = getQWord(&line);
	data = getQWord(&line);
	if (*func == '\0')	/* error */
	    return -1;
	item->type = MENU_FUNC;
	item->label = label;
	if (btri_fast_ci_search_str(func, w3mFuncTab, (void **)&fl) != bt_failure) {
	  if (!strncasecmp(func, "M:", sizeof("M:") - 1)) {
	    item->type = MENU_MFUNC;
	    item->value = fl - w3mFuncList;
	  }
	  else
	    item->func = fl->func.main_func;
	}
	else
	  item->func = w3mFuncList[FUNCNAME_nulcmd].func.main_func;
	item->keys = keys;
	item->data = data;
	return item->type;
    }
    else if (strcmp(type, "popup") == 0) {
	label = getQWord(&line);
	popup = getQWord(&line);
	keys = getQWord(&line);
	if (*popup == '\0')	/* error */
	    return -1;
	item->type = MENU_POPUP;
	item->label = label;
	if ((n = getMenuN(w3mMenuList, popup)) == -1)
	    n = addMenuList(&w3mMenuList, popup);
	item->popup = w3mMenuList[n].menu;
	item->keys = keys;
	return MENU_POPUP;
    }
    return -1;			/* error */
}

int
addMenuList(MenuList ** mlist, char *id)
{
    int n;
    MenuList *list = *mlist;

    for (n = 0; list->id != NULL; list++, n++)
	;
    *mlist = New_Reuse(MenuList, *mlist, (n + 2));
    list = *mlist + n;
    list->id = id;
    list->menu = New(Menu);
    list->item = New(MenuItem);
    (list + 1)->id = NULL;
    return n;
}

int
getMenuN(MenuList *list, char *id)
{
    int n;

    for (n = 0; list->id != NULL; list++, n++) {
	if (strcmp(id, list->id) == 0)
	    return n;
    }
    return -1;
}

/* --- InitMenu (END) --- */

#endif				/* USE_MENU */

/* Local Variables:    */
/* c-basic-offset: 4   */
/* tab-width: 8        */
/* End:                */
