#include <GL/gl.h>

#include "global.h"
#include "texture.h"
#include "images.h"	/* Image */

#ifndef VRENGD


static TextureCacheEntry *texture_cache_first;
static Image *default_image;
static int texture_quality;
static Vpthread_mutex_t texture_mutex;
static int texture_number;


void initTextureCache(int quality)
{
  u_int8 *pixmap;
  int i;

  texture_cache_first = NULL;
  default_image = newImage(TEXTURE_SIZE, TEXTURE_SIZE, IMAGE_RGB, IMAGE_FIX);
  pixmap = default_image->pixmap;
  for (i=0; i < TEXTURE_SIZE*TEXTURE_SIZE; i++) {
    pixmap[3*i+0] = pixmap[3*i+1] = pixmap[3*i+2] = 0x80;
  }
  texture_quality = quality;
  initMutex(&texture_mutex);
}

void closeTextureCache(void)
{
  /* TODO: free textures */
  freeImage(default_image);
}

/*
 * called par Render()
 */
void updateTextures(void)
{
  TextureCacheEntry *tc = NULL;
  Image *image1 = NULL;

  lockMutex(&texture_mutex);

  for (tc = texture_cache_first; tc; tc = tc->next) {
    if (tc->image != NULL) {
      glBindTexture(GL_TEXTURE_2D, tc->num);

      if (tc->image->type != IMAGE_ANIM &&
         (tc->image->xsize != TEXTURE_SIZE || tc->image->ysize != TEXTURE_SIZE)) {
        /* image to resize */
        if (tc->image->format != IMAGE_RGB && tc->image->format != IMAGE_RGBA) {
          trace(DBG_FORCE, "updateTextures: n=%d f=%d w=%d h=%d t=%d p=%p u=%s", tc->num, tc->image->format, tc->image->xsize, tc->image->ysize, tc->image->type, tc->image, tc->url);
          continue;
        }
        if (tc->image->pixmap == NULL) {
          trace(DBG_FORCE, "updateTextures pixmap=NULL: n=%d w=%d h=%d u=%s", tc->num, tc->image->xsize, tc->image->ysize, tc->url);
          continue;
        }

        if ((image1 = resizeImage(TEXTURE_SIZE, TEXTURE_SIZE, tc->image->format, tc->image->type, tc->image)) == NULL)
          continue;
        glTexImage2D(GL_TEXTURE_2D, 0, VRENG_TEXTURE_TYPE,
                     image1->xsize, image1->ysize, 0,
                     GL_RGB, GL_UNSIGNED_BYTE, image1->pixmap);
        freeImage(image1);
      }
      else {
        /* image well sized */
        glTexImage2D(GL_TEXTURE_2D, 0, VRENG_TEXTURE_TYPE,
                     tc->image->xsize, tc->image->ysize, 0,
                     GL_RGB, GL_UNSIGNED_BYTE, tc->image->pixmap);
      }

      //DAX if (tc->loaded && tc->image->type == IMAGE_FIX) {
      if (tc->loaded) {
        freeImage(tc->image);
        tc->image = NULL;
      }
    }
  }
  unlockMutex(&texture_mutex);
}

static
int readImage(void *tc1, char *buf, int length)
{
  TextureCacheEntry *tc = (TextureCacheEntry *) tc1;

  return httpRead(tc->handle, buf, length);	/* return read length */
}

static
void displayImage(void *tc1, Image *image)
{
  TextureCacheEntry *tc = (TextureCacheEntry *) tc1;

  lockMutex(&texture_mutex);
  tc->image = image;
  unlockMutex(&texture_mutex);
}

static
void readTexture(void *tc1, void *handle)
{
  TextureCacheEntry *tc = (TextureCacheEntry *) tc1;
  Image *image = NULL;

  trace(DBG_ZV, "texture: handle=%X", (int) handle);

  if (handle == NULL)
    return ; /* error while opening */
  tc->handle = handle;

  trace(DBG_HTTP, "texture: mime=%s url=%s", tc->mime, tc->url);

#if 0
  switch (getImageTypeByExt(tc->url)) {
#endif
  switch (getImageTypeByMime(tc->mime)) {
  case TYPE_PPM:
    image = loadPPM(tc, readImage, displayImage);
    break;
  case TYPE_GIF:
    image = loadGIF(tc, readImage, displayImage);
    break;
  case TYPE_SGI:
    trace(DBG_HTTP, "texture: mime=%s url=%s", tc->mime, tc->url);
    image = loadSGI(tc, readImage, displayImage);
    break;
  case TYPE_PNG:
    trace(DBG_HTTP, "texture: mime=%s url=%s", tc->mime, tc->url);
    image = loadPNG(tc, readImage, displayImage);
    break;
  case TYPE_JPG:
    image = loadJPG(tc, readImage, displayImage);
    break;
  case TYPE_PCX:
    image = loadPCX(tc, readImage, displayImage);
    break;
  case TYPE_PLAIN:
#if 0
    image = loadGIF(tc, readImage, displayImage);
#endif
    break;
  default:
    warning("texture: unknown type mime=%s url=%s", tc->mime, tc->url);
    image = NULL;
    break;
  }
  if (image == NULL)
    return ; /* 0 */
  lockMutex(&texture_mutex);
  tc->image = image;
  tc->loaded = 1;
  unlockMutex(&texture_mutex);
  return ; /* 1 */
}

int getTextureFromCache(const char *url)
{
  TextureCacheEntry *tc;

  lockMutex(&texture_mutex);

  for (tc = texture_cache_first; tc != NULL; tc = tc->next) {
    if (!strcmp(tc->url, url)) {
      /* texture is in the cache */
      unlockMutex(&texture_mutex);
      return texture_number = tc->num;
    }
  }

  /* we must download the texture now */
  /* new entry in cache */
  if ((tc = (TextureCacheEntry *) calloc(1, sizeof(TextureCacheEntry))) == NULL) {
    warning("getTextureFromCache: can't alloc tc");
    unlockMutex(&texture_mutex);
    return 0;
  }
  tc->next = texture_cache_first;
  texture_cache_first = tc;
  tc->image = NULL;
  tc->loaded = 0;
  strcpy(tc->url, url);

  glGenTextures(1, (GLuint *) &tc->num);
  glBindTexture(GL_TEXTURE_2D, tc->num);
  glTexImage2D(GL_TEXTURE_2D, 0, VRENG_TEXTURE_TYPE,
	       default_image->xsize, default_image->ysize, 0,
	       GL_RGB, GL_UNSIGNED_BYTE, default_image->pixmap);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
  
  if (texture_quality) {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  } else {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  }
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

  unlockMutex(&texture_mutex);

  trace(DBG_ZV, "texture: number=%d loading %s", tc->num, url);
  httpOpen(tc->url, readTexture, tc, THREAD_NO_NONBLOCK); 

  texture_number = tc->num;	/* keep it */
  return tc->num;
}

TextureCacheEntry *
getTextureEntryByUrl(const char *url)
{
  TextureCacheEntry *tc;

  for (tc = texture_cache_first; tc != NULL; tc = tc->next) {
    if (!strcmp(tc->url, url)) {
      return tc; /* texture is in the cache */
    }
  }
  /* we must dowdload the texture latter */
  return NULL;
}

TextureCacheEntry *
getTextureEntryByNumber(int num)
{
  TextureCacheEntry *tc;

  for (tc = texture_cache_first; tc != NULL; tc = tc->next) {
    if (tc->num == num) {
      return tc; /* texture is in the cache */
    }
  }
  return NULL;
}

int getCurrentTextureNumber(void)
{
  return texture_number;
}

#endif /* !VRENGD */
