/* GiMd2Viewer - Quake2 model viewer
 * Copyright (C) 1998  Lionel ULMER <bbrox@mygale.org>
 *
 * Based on code Copyright (C) 1997 Trey Harrison <trey@crack.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <GL/gl.h>

#include "global.h"
#include "zv.h"
#include "md2.h"

#ifndef VRENGD


static
void DrawModelMd2(Md2 *md2, int32 frame, int32 nextfrm, float ratio, float scale)
{
  int32 nb_vert;
  int32 *glcmd;
  Frame *frame1, *frame2;
  
  /* Keeps the frame value valid */
  if (md2->numframes == 0) {
    trace(DBG_FORCE, "DrawModelMd2: numframes == 0");
    return;
  }
  frame %= md2->numframes;
  nextfrm %= md2->numframes;
  
  /* get the frames information */
  frame1 = md2->frames + frame;
  frame2 = md2->frames + nextfrm;
  
  /* do the gl commands */
  glcmd = md2->glcmds;
  nb_vert = 0;

  while (*glcmd != 0) {
    int32 num_verts;
    
    /* determine the command to draw the triangles */
    if (*glcmd > 0) { /* triangle strip */
      num_verts = *glcmd;
      glBegin(GL_TRIANGLE_STRIP);
    }
    else { /* triangle fan */
      num_verts = -(*glcmd);
      glBegin(GL_TRIANGLE_FAN);
    }
    glcmd++;
    
    for (int i = 0; i < num_verts; i++) {
      Vec3 p;  /* Interpolated point */
      int32 vert_index;

      /* Grab the vertex index */
      vert_index = *glcmd; glcmd++;
      
      /* Interpolate */
      p.x = frame1->vert_table[vert_index].x +
	(frame2->vert_table[vert_index].x -
	 frame1->vert_table[vert_index].x) * ratio;
      p.y = frame1->vert_table[vert_index].y +
	(frame2->vert_table[vert_index].y -
	 frame1->vert_table[vert_index].y) * ratio;
      p.z = frame1->vert_table[vert_index].z +
	(frame2->vert_table[vert_index].z -
	 frame1->vert_table[vert_index].z) * ratio;
      
      glTexCoord2f(md2->texinfo[nb_vert].s, md2->texinfo[nb_vert].t);
      zvVertex3f(p.x * scale, p.y * scale, p.z * scale);
      nb_vert++;
    }
    glEnd();
  }
}

static
int getFrames(Md2Header *md2_hdr, Md2 *md2, void *handle)
{
  int32 offset = 0;

  trace(DBG_WMGT, "getFrames: numframes=%d", md2->numframes);
  
  /* converts the FrameInfos to Frames */
  md2->frames = (Frame *) calloc(1, sizeof(Frame) * md2->numframes);

  for (int frame = 0; frame < md2->numframes; frame++) {
    Vec3 scale, origin;

    scale.x = httpGetFloat(handle); offset += 4;
    scale.y = httpGetFloat(handle); offset += 4;
    scale.z = httpGetFloat(handle); offset += 4;
    origin.x = httpGetFloat(handle); offset += 4;
    origin.y = httpGetFloat(handle); offset += 4;
    origin.z = httpGetFloat(handle); offset += 4;
    
    httpGetBuf(handle, (char *) &(md2->frames[frame].name), 16);
    offset += 16;
    md2->frames[frame].vert_table = (Vertex *) calloc(1, sizeof(Vertex) *
						      md2_hdr->num_xyz);
    /* loads the vertices */
    for (int i = 0; i < md2_hdr->num_xyz; i++) {
      Trivertex cur_vert;
      Vertex *p = (md2->frames[frame].vert_table) + i;
      
      cur_vert.x = httpGetByte(handle); offset += 1;
      cur_vert.y = httpGetByte(handle); offset += 1;
      cur_vert.z = httpGetByte(handle); offset += 1;
      cur_vert.lightnormalindex = httpGetByte(handle); offset += 1;
      p->x = ((cur_vert.x * scale.x) + origin.x);
      p->y = ((cur_vert.y * scale.y) + origin.y);
      p->z = ((cur_vert.z * scale.z) + origin.z);
    }
  }
  return offset;
}

static
int32 getGLCmds(Md2Header *md2_hdr, Md2 *md2, void *handle)
{
  int32 glcmd, *glcmd_copy;
  TexInfo *texinfo;
  int32 num_vertices;
  int32 offset = 0;

  trace(DBG_ZV, "getGLCmds");
  
  /* we keep only the commands and the index in the glcommands
   * we 'pre-parse' the texture coordinates
   * do not ask me how I found this formula :-))
   */
  num_vertices = ((md2_hdr->num_tris + 2 * md2_hdr->num_glcmds - 2) / 7);
  md2->texinfo = (TexInfo *) calloc(1, sizeof(TexInfo) * num_vertices);
  md2->glcmds = (int32 *)
    calloc(1, sizeof(int32) * (md2_hdr->num_glcmds - 2 * num_vertices));
  
  /* now transform the GL commands */
  glcmd_copy = md2->glcmds;
  texinfo = md2->texinfo;

  while (1) {
    int32 nb_verts;

    glcmd = httpGetInt(handle); offset += 4;
    if (glcmd == 0)
      break;
    
    /* determine the command to draw the triangles */
    if (glcmd > 0) /* triangle strip */
      nb_verts = glcmd;
    else /* triangle fan */
      nb_verts = -glcmd;
    *(glcmd_copy++) = glcmd;
    
    for (int i = 0; i < nb_verts; i++) {
      /* gets the texture information */
      texinfo->s = httpGetFloat(handle); offset += 4;
      texinfo->t = httpGetFloat(handle); offset += 4;
      texinfo++;
      
      /* we keep the vertex index */
      glcmd = httpGetInt(handle); offset += 4;
      *(glcmd_copy++) = glcmd;
    }
  }
  /* do not forget to copy the zero :-) */
  *(glcmd_copy++) = glcmd;
  return offset;
}

static
void readModelMd2(void *amd2, void *handle)
{
  Md2Header md2_hdr;
  int32 offset = 0;
  Md2 **md2 = (Md2 **) amd2;
  
  if (handle == NULL) {
    *md2 = NULL;
    return;
  }
  httpClearBuf();

  /* Read the header */
  httpGetBuf(handle, (char *) &(md2_hdr.ident), 4); offset += 4;
  md2_hdr.version = httpGetInt(handle); offset += 4;
  md2_hdr.skinwidth = httpGetInt(handle); offset += 4;
  md2_hdr.skinheight = httpGetInt(handle); offset += 4;
  md2_hdr.framesize = httpGetInt(handle); offset += 4;
  md2_hdr.num_skins = httpGetInt(handle); offset += 4;
  md2_hdr.num_xyz = httpGetInt(handle); offset += 4;
  md2_hdr.num_st = httpGetInt(handle); offset += 4;
  md2_hdr.num_tris = httpGetInt(handle); offset += 4;
  md2_hdr.num_glcmds = httpGetInt(handle); offset += 4;
  md2_hdr.num_frames = httpGetInt(handle); offset += 4;
  md2_hdr.ofs_skins = httpGetInt(handle); offset += 4;
  md2_hdr.ofs_st = httpGetInt(handle); offset += 4;
  md2_hdr.ofs_tris = httpGetInt(handle); offset += 4;
  md2_hdr.ofs_frames = httpGetInt(handle); offset += 4;
  md2_hdr.ofs_glcmds = httpGetInt(handle); offset += 4;
  md2_hdr.ofs_end = httpGetInt(handle); offset += 4;
  
  /* check if this is really a .MD2 file :-) */
  if (strncmp(md2_hdr.ident, "IDP2", 4)) {
    *md2 = NULL;
    return;
  }
  
  /* create the model */
  *md2 = (Md2 *) calloc(1, sizeof(Md2));
  
  /* we do not need all the info from the header, just some of it */
  (*md2)->numframes = md2_hdr.num_frames;
  (*md2)->skinwidth = md2_hdr.skinwidth;
  (*md2)->skinheight = md2_hdr.skinheight;

  /* check which info is first */
  if (md2_hdr.ofs_frames > md2_hdr.ofs_glcmds) {
    httpSkip(handle, md2_hdr.ofs_glcmds - offset);
    offset = md2_hdr.ofs_glcmds;
    offset += getGLCmds(&md2_hdr, *md2, handle);
    httpSkip(handle, md2_hdr.ofs_frames - offset);
    offset = md2_hdr.ofs_frames;
    offset += getFrames(&md2_hdr, *md2, handle);
  }
  else {
    httpSkip(handle, md2_hdr.ofs_frames - offset);
    offset = md2_hdr.ofs_frames;
    offset += getFrames(&md2_hdr, *md2, handle);
    httpSkip(handle, md2_hdr.ofs_glcmds - offset);
    offset = md2_hdr.ofs_glcmds;
    offset += getGLCmds(&md2_hdr, *md2, handle);
  }
  return;
}

/* Free the memory used by a model */
void freeModelMd2(Md2 *md2)
{
  for (int frame = 0; frame < md2->numframes; frame++)
    free(md2->frames[frame].vert_table);
  free(md2->frames);
  free(md2->glcmds);
  free(md2->texinfo);
  free(md2);
}

int getDlistModelMd2(Md2 *md2, int frame, int nextfrm, float inter, float scale)
{
  if (md2 == NULL)
    return -1;

  int num = glGenLists(1);
  glNewList(num, GL_COMPILE);
  glEnable(GL_CULL_FACE);
  glCullFace(GL_FRONT);
  DrawModelMd2(md2, frame, nextfrm, inter, scale);
  glDisable(GL_CULL_FACE);
  glEndList();
  return num;
}

Md2 *loadModelMd2(const char *url)
{
  Md2 *md2 = NULL;	/* filled by readModelMd2 */
  
  httpOpen(url, readModelMd2, &md2, THREAD_NO_BLOCK);
  return md2;
}

#endif /* !VRENGD */
